# coding: utf-8
# create by tongshiwei on 2019/7/1

import contextlib
import random
from longling import config_logging
from copy import deepcopy
from EduSim.Envs.Learner import Learner

__all__ = ["RandomGraphAgent", "Agent"]


class Agent(object):
    def __init__(self, logger=None, *args, **kwargs):
        # components
        self._logger = logger if logger is not None else config_logging(logger="Agent", console_log_level="info")

        # following variables is episode variables
        # the learning path composed by learning items is generated by the `step` function
        self._path = None
        # interactions refers to the observation sequence
        self._interactions = None
        self._learner = None
        self._learner_learning_history = None
        self._learner_exercise_history = None
        self._episode_reward = None

    @property
    def path(self) -> list:
        return self._path

    @contextlib.contextmanager
    def episode(self, learner: Learner, *args, **kwargs):
        yield self.begin_episode(learner, *args, **kwargs)
        self.end_episode(*args, **kwargs)

    def begin_episode(self, learner: Learner, *args, **kwargs) -> None:
        """invoked at the most beginning of an episode (initialize state)"""
        self._learner = learner
        self._learner_learning_history = deepcopy(learner.learning_history)
        self._learner_exercise_history = deepcopy(learner.exercise_history)
        self._path = []
        self._interactions = []

    def observe(self, *args, **kwargs) -> None:
        """change the agent state based on the observation on environment output"""
        raise NotImplementedError

    def step(self, *args, **kwargs) -> int:
        """recommend/generate an learning item"""
        raise NotImplementedError

    def n_step(self, n, key=lambda x: x, *args, **kwargs) -> list:
        """recommend/generate n learning items"""
        return [key(self.step(*args, **kwargs)) for _ in range(n)]

    def _reset_episode(self):
        self._path = None
        self._interactions = None
        self._learner = None
        self._learner_learning_history = None
        self._learner_exercise_history = None
        self._episode_reward = None

    def summary_episode(self) -> dict:
        episode_summary = {
            "learner_id": self._learner.id,
            "learning_history": self._learner_learning_history,
            "exercise_history": self._learner_exercise_history,
            "path": self._path,
            "interactions": self._interactions,
            "episode_reward": self._episode_reward
        }
        self._logger.info(episode_summary)
        return episode_summary

    def end_episode(self, *args, **kwargs) -> dict:
        """invoked at the end of an episode (clear buffer, trigger tuning or training)"""
        episode_summary = self.summary_episode()
        self._reset_episode()
        return episode_summary

    def tune(self, *args, **kwargs) -> None:
        """api for training: update parameters"""
        raise NotImplementedError

    def step_reward(self, *args, **kwargs) -> None:
        """api for receiving rewards at each step from environment"""
        raise NotImplementedError

    def episode_reward(self, episode_reward, *args, **kwargs) -> None:
        """api for receiving rewards at the end of episode from environment"""
        raise NotImplementedError


class GraphAgent(Agent):
    def __init__(self, graph, logger=None, *args, **kwargs):
        """

        Parameters
        ----------
        graph: Graph
        """
        super(GraphAgent, self).__init__(logger=logger)
        self.graph = graph


class RandomGraphAgent(GraphAgent):
    def step(self, *args, **kwargs) -> int:
        self._path.append(random.choice(self.graph()[0]))
        return self._path[-1]

    def observe(self, exercise, correct, *args, **kwargs) -> None:
        self._interactions.append([exercise, correct])

    def tune(self, *args, **kwargs):
        random.seed(10)

    def step_reward(self, *args, **kwargs):
        pass

    def episode_reward(self, episode_reward, *args, **kwargs):
        self._episode_reward = episode_reward
