# MysqldbPythonWrapper

**mysqldb_wrapper** is a python module that will help you easily communicate with your database and provides encryption with [Fernet](https://cryptography.io/en/latest/fernet/) using the [cryptography](https://cryptography.io/en/latest/) module. This module is inspired by [SQLAlchemy](https://www.sqlalchemy.org/) but is even simpler to use and has less limitations (but is also less complete).

**IMPORTANT:** This module is only made to use with simple mysql queries. You can still use it with more complex queries, but no interface is provided for those.

## Installation

This module is available via pip:

```sh
pip install mysqldb_wrapper
```

## Usage

### The table

First, you need to create a `table`. Here's an example:
```python
from mysqldb_wrapper import Base, Id

class Test(Base):
    __tablename__ = "test"

    id = Id()
    another_table_id = Id()
    hashed = bytes()
    number = int(1)
    string = str("string")
    boolean = bool(True)
```

To explain in details:
- `id = Id()` is, as its name implies, the database object `id`. **Always put one Id**.
- `another_table_id = Id()` is the `id` of another table in the database.
- `hashed = bytes()` will be a hashed string, always declare it empty.
- `number = int(1)` will be a int with a default value `1`.
- `string = str("string")` will be a string with a default value `string`.
- `boolean = bool(True)` will be a boolean with a default value `boolean`.

Easy, right ?

**Note:** This implementation does not support complex types, so keep your table simple ;)

### The database

You need to create your database by hand Although this module will not create the database for you, it will create the tables automatically !

**Note:** keep in mind that if your table change (remove fields, change field types, ...), you will need to remove/update them by hand. 

To initialize a session with the database, use:
```python
from mysqldb_wrapper import Session

session = Session(DB_USERNAME, DB_PASSWORD, DB_TEST, FERNET_KEY)
```
So here we have:
- `DB_USERNAME` is the username of the mysql user
- `DB_PASSWORD` is the password of the mysql user
- `DB_TEST` is the name of the database
- `FERNET_KEY` is the encryption key. You can create one with `Fernet.generate_key()`, but keep in mind to always use the same one. (So generate one by hand then keep it somewhere safe)

And now we have all we need to use the mysql database.

### Operations

#### Adding an object

Remember your table ? Just create a new object from it and use the `add` method of the session with it. Also, the table objects supports `**kwargs`.
```python
obj = Test(hashed="abcd", string="word")
session.add(obj)
```
And `obj` now automatically contains the id generated by the database. Also, this method returns the object in case you want to chain methods.

#### Updating an object

Using the `obj` from before:
```python
obj.number = 2553166
obj.string = "a sentence"
obj.boolean = True
session.update(obj)
```
This method also returns the object.

#### Deleting an object

And again:
```python
session.delete(obj)
```
`obj` now is not in the database anymore but you can still continue to use it, just remember it doesn't have an `id` (set to 0). Like the others, this method returns the object.

#### Querying objects

We have more to talk about in this section.

```python
obj = session.query(Test).first()
```
This one is easy. You put the table you want in `query()` and you want to get the first result.

```python
obj_list = session.query(Test).all()
```
Same than before, except this time you retrieve all the results. It is guaranteed to return a list, even if empty.

Now the tricky part:
```python
obj = session.query(Test).where(Test.id == 2).first()
```
This time, we are still querying in the Test table, but we want a specific result, the one where the id is equal to 2. You don't have to worry about doing anything more, everything is handled by the `Base` class your `Test` table inherited from.

Remember we had a hash in our table ? Don't worry, it is handled as well. How does it work ? Like the others field. =)
```python
obj = session.query(Test).where(Test.hashed == "abcd").first()
```

You can also chain the `where` methods, use variables and query all objects:
```python
a_string = "something"
list_obj = session.query(Test).where(Test.another_table_id == 2).where(Test.hashed == a_string).all()
```

**IMPORTANT: The `where` method only works with equality as of now. Also, you can only query by Id and hashes.**

