import numpy as np

from libensemble.tools.alloc_support import avail_worker_ids, sim_work, gen_work, count_gens, all_returned


def give_sim_work_first(W, H, sim_specs, gen_specs, alloc_specs, persis_info):
    """
    Decide what should be given to workers. This allocation function gives any
    available simulation work first, and only when all simulations are
    completed or running does it start (at most ``alloc_specs['user']['num_active_gens']``)
    generator instances.

    Allows for a ``alloc_specs['user']['batch_mode']`` where no generation
    work is given out unless all entries in ``H`` are returned.

    Allows for ``blocking`` of workers that are not active, for example, so
    their resources can be used for a different simulation evaluation.

    Can give points in highest priority, if ``'priority'`` is a field in ``H``.
    If gen_specs['user']['give_all_with_same_priority'] is set to True, then
    all points with the same priority value are given as a batch to the sim.


    This is the default allocation function if one is not defined.

    .. seealso::
        `test_uniform_sampling.py <https://github.com/Libensemble/libensemble/blob/develop/libensemble/tests/regression_tests/test_uniform_sampling.py>`_ # noqa
    """

    Work = {}
    gen_count = count_gens(W)
    avail_set = set(W['worker_id'][np.logical_and(~W['blocked'],
                                                  W['active'] == 0)])

    task_avail = ~H['given'] & ~H['cancel_requested']
    for i in avail_worker_ids(W):

        if i not in avail_set:
            continue

        if np.any(task_avail):
            # Pick all high priority, oldest high priority, or just oldest point
            if 'priority' in H.dtype.fields:
                priorities = H['priority'][task_avail]
                if gen_specs['user'].get('give_all_with_same_priority'):
                    q_inds = (priorities == np.max(priorities))
                else:
                    q_inds = np.argmax(priorities)
            else:
                q_inds = 0

            # Get sim ids (indices) and check resources needed
            sim_ids_to_send = np.nonzero(task_avail)[0][q_inds]  # oldest point(s)
            sim_ids_to_send = np.atleast_1d(sim_ids_to_send)
            nodes_needed = (np.max(H[sim_ids_to_send]['num_nodes'])
                            if 'num_nodes' in H.dtype.names else 1)
            if nodes_needed > len(avail_set):
                break

            # Assign points to worker and remove from task_avail list.
            sim_work(Work, i, sim_specs['in'], sim_ids_to_send, persis_info.get(i))
            task_avail[sim_ids_to_send] = False

            # Update resource records
            avail_set.remove(i)
            if nodes_needed > 1:
                workers_to_block = list(avail_set)[:nodes_needed-1]
                avail_set.difference_update(workers_to_block)
                Work[i]['libE_info']['blocking'] = workers_to_block

        else:

            # Allow at most num_active_gens active generator instances
            if gen_count >= alloc_specs['user'].get('num_active_gens', gen_count+1):
                break

            # Do not start gen instances in batch mode if workers still working
            if alloc_specs['user'].get('batch_mode') and not all_returned(H):
                break

            # Give gen work
            gen_count += 1
            gen_in = gen_specs.get('in', [])
            return_rows = range(len(H)) if gen_in else []
            gen_work(Work, i, gen_in, return_rows, persis_info.get(i))

    return Work, persis_info
