import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Function as LambdaFunction, FunctionProps, EnvironmentOptions } from './function';
import { FunctionBase } from './function-base';
import { Version } from './lambda-version';
import { ILayerVersion } from './layers';
import { Permission } from './permission';
import { Runtime } from './runtime';
/**
 * Properties for a newly created singleton Lambda.
 *
 * @stability stable
 */
export interface SingletonFunctionProps extends FunctionProps {
    /**
     * A unique identifier to identify this lambda.
     *
     * The identifier should be unique across all custom resource providers.
     * We recommend generating a UUID per provider.
     *
     * @stability stable
     */
    readonly uuid: string;
    /**
     * A descriptive name for the purpose of this Lambda.
     *
     * If the Lambda does not have a physical name, this string will be
     * reflected its generated name. The combination of lambdaPurpose
     * and uuid must be unique.
     *
     * @default SingletonLambda
     * @stability stable
     */
    readonly lambdaPurpose?: string;
}
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @stability stable
 * @resource AWS::Lambda::Function
 */
export declare class SingletonFunction extends FunctionBase {
    /**
     * The principal this Lambda Function is running as.
     *
     * @stability stable
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The name of the function.
     *
     * @stability stable
     */
    readonly functionName: string;
    /**
     * The ARN fo the function.
     *
     * @stability stable
     */
    readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The construct node where permissions are attached.
     *
     * @stability stable
     */
    readonly permissionsNode: cdk.ConstructNode;
    /**
     * The runtime environment for the Lambda function.
     *
     * @stability stable
     */
    readonly runtime: Runtime;
    /**
     * Whether the addPermission() call adds any permissions.
     *
     * True for new Lambdas, false for version $LATEST and imported Lambdas
     * from different accounts.
     *
     * @stability stable
     */
    protected readonly canCreatePermissions: boolean;
    private lambdaFunction;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SingletonFunctionProps);
    /**
     * Whether or not this Lambda function was bound to a VPC.
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     *
     * @stability stable
     * @inheritdoc true
     */
    get isBoundToVpc(): boolean;
    /**
     * Access the Connections object.
     *
     * Will fail if not a VPC-enabled Lambda Function
     *
     * @stability stable
     * @inheritdoc true
     */
    get connections(): ec2.Connections;
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     *
     * @stability stable
     */
    get logGroup(): logs.ILogGroup;
    /**
     * Returns a `lambda.Version` which represents the current version of this singleton Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     *
     * @stability stable
     */
    get currentVersion(): Version;
    /**
     * Adds an environment variable to this Lambda function.
     *
     * If this is a ref to a Lambda function, this operation results in a no-op.
     *
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     * @stability stable
     */
    addEnvironment(key: string, value: string, options?: EnvironmentOptions): LambdaFunction;
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     * @stability stable
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers: ILayerVersion[]): void;
    /**
     * Adds a permission to the Lambda resource policy.
     *
     * @stability stable
     */
    addPermission(name: string, permission: Permission): void;
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled as a singleton across the stack. Use this method instead to declare dependencies.
     *
     * @stability stable
     */
    addDependency(...up: cdk.IDependable[]): void;
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     *
     * @stability stable
     */
    dependOn(down: cdk.IConstruct): void;
    /** @internal */
    _checkEdgeCompatibility(): void;
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    protected _functionNode(): cdk.ConstructNode;
    private ensureLambda;
}
