"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    /**
     * Add permission for this layer version to specific entities.
     *
     * Usage within
     * the same account where the layer is defined is always allowed and does not
     * require calling this method. Note that the principal that creates the
     * Lambda function using the layer (for example, a CloudFormation changeset
     * execution role) also needs to have the ``lambda:GetLayerVersion``
     * permission on the layer version.
     *
     * @stability stable
     */
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 *
 * @stability stable
 */
class LayerVersion extends LayerVersionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_LayerVersionProps(props);
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            compatibleArchitectures: (_b = props.compatibleArchitectures) === null || _b === void 0 ? void 0 : _b.map(a => a.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN.
     *
     * Assumes it is compatible with all Lambda runtimes.
     *
     * @stability stable
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     * @stability stable
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_LayerVersionAttributes(attrs);
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
_a = JSII_RTTI_SYMBOL_1;
LayerVersion[_a] = { fqn: "@aws-cdk/aws-lambda.LayerVersion", version: "1.135.0" };
//# sourceMappingURL=data:application/json;base64,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