"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends constructs_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.getPutRecordTemplate(props.putRecordRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordModel(props.putRecordRequestModel) }
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.getPutRecordsTemplate(props.putRecordsRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordsModel(props.putRecordsRequestModel) }
        });
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
    getPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    getPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    getPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    getPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToKinesisStreams[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-kinesisstreams.ApiGatewayToKinesisStreams", version: "2.0.0-rc.1" };
//# sourceMappingURL=data:application/json;base64,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