"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * (experimental) A Cache Policy configuration.
 *
 * @experimental
 * @resource AWS::CloudFront::CachePolicy
 */
class CachePolicy extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        const cachePolicyName = (_a = props.cachePolicyName) !== null && _a !== void 0 ? _a : this.node.uniqueId;
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.cachePolicyName}'`);
        }
        const minTtl = ((_b = props.minTtl) !== null && _b !== void 0 ? _b : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_c = props.defaultTtl) !== null && _c !== void 0 ? _c : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_d = props.maxTtl) !== null && _d !== void 0 ? _d : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /**
     * (experimental) Imports a Cache Policy from its id.
     *
     * @experimental
     */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _a, _b, _c, _d;
        const cookies = (_a = props.cookieBehavior) !== null && _a !== void 0 ? _a : CacheCookieBehavior.none();
        const headers = (_b = props.headerBehavior) !== null && _b !== void 0 ? _b : CacheHeaderBehavior.none();
        const queryStrings = (_c = props.queryStringBehavior) !== null && _c !== void 0 ? _c : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_d = props.enableAcceptEncodingGzip) !== null && _d !== void 0 ? _d : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
/**
 * (experimental) Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 *
 * @experimental
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * (experimental) Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 *
 * @experimental
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/**
 * (experimental) Disables caching.
 *
 * This policy is useful for dynamic content and for requests that are not cacheable.
 *
 * @experimental
 */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/**
 * (experimental) Designed for use with an origin that is an AWS Elemental MediaPackage endpoint.
 *
 * @experimental
 */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * (experimental) Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * (experimental) Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * (experimental) All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * (experimental) Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * (experimental) All cookies except the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
/**
 * (experimental) Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * (experimental) HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static none() { return new CacheHeaderBehavior('none'); }
    /**
     * (experimental) Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
/**
 * (experimental) Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * (experimental) Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * (experimental) All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * (experimental) Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * (experimental) All query strings except the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,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