"use strict";
// Copyright (c) Martin Renou
// Distributed under the terms of the Modified BSD License.
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const base_1 = require("@jupyter-widgets/base");
const version_1 = require("./version");
const utils_1 = require("./utils");
function getContext(canvas) {
    const context = canvas.getContext("2d");
    if (context === null) {
        throw 'Could not create 2d context.';
    }
    return context;
}
function serializeImageData(array) {
    return new DataView(array.buffer.slice(0));
}
function deserializeImageData(dataview) {
    if (dataview === null) {
        return null;
    }
    return new Uint8ClampedArray(dataview.buffer);
}
class CanvasModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: CanvasModel.model_name, _model_module: CanvasModel.model_module, _model_module_version: CanvasModel.model_module_version, _view_name: CanvasModel.view_name, _view_module: CanvasModel.view_module, _view_module_version: CanvasModel.view_module_version, width: 700, height: 500, sync_image_data: false, image_data: null });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.canvas = document.createElement('canvas');
        this.ctx = getContext(this.canvas);
        this.resizeCanvas();
        this.drawImageData();
        this.on_some_change(['width', 'height'], this.resizeCanvas, this);
        this.on('change:sync_image_data', this.syncImageData.bind(this));
        this.on('msg:custom', this.onCommand.bind(this));
        this.send({ event: 'client_ready' }, {});
    }
    drawImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.get('image_data') !== null) {
                const img = yield utils_1.fromBytes(this.get('image_data'));
                this.ctx.drawImage(img, 0, 0);
                this.trigger('new-frame');
            }
        });
    }
    onCommand(command, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.processCommand(command, buffers);
            this.forEachView((view) => {
                view.updateCanvas();
            });
            this.trigger('new-frame');
            this.syncImageData();
        });
    }
    processCommand(command, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            if (command instanceof Array) {
                let remainingBuffers = buffers;
                for (const subcommand of command) {
                    let subbuffers = [];
                    if (subcommand.n_buffers) {
                        subbuffers = remainingBuffers.slice(0, subcommand.n_buffers);
                        remainingBuffers = remainingBuffers.slice(subcommand.n_buffers);
                    }
                    yield this.processCommand(subcommand, subbuffers);
                }
                return;
            }
            switch (command.name) {
                case 'fillRects':
                    this.drawRects(command.args, buffers, 'fillRect');
                    break;
                case 'strokeRects':
                    this.drawRects(command.args, buffers, 'strokeRect');
                    break;
                case 'fillArc':
                    this.fillArc(command.args, buffers);
                    break;
                case 'strokeArc':
                    this.strokeArc(command.args, buffers);
                    break;
                case 'fillArcs':
                    this.drawArcs(command.args, buffers, 'fill');
                    break;
                case 'strokeArcs':
                    this.drawArcs(command.args, buffers, 'stroke');
                    break;
                case 'drawImage':
                    yield this.drawImage(command.args, buffers);
                    break;
                case 'putImageData':
                    this.putImageData(command.args, buffers);
                    break;
                case 'set':
                    this.setAttr(command.attr, command.value);
                    break;
                case 'clear':
                    this.clearCanvas();
                    break;
                default:
                    this.executeCommand(command.name, command.args);
                    break;
            }
        });
    }
    drawRects(args, buffers, commandName) {
        const x = utils_1.getArg(args[0], buffers);
        const y = utils_1.getArg(args[1], buffers);
        const width = utils_1.getArg(args[2], buffers);
        const height = utils_1.getArg(args[3], buffers);
        const numberRects = Math.min(x.length, y.length, width.length, height.length);
        for (let idx = 0; idx < numberRects; ++idx) {
            this.executeCommand(commandName, [x.getItem(idx), y.getItem(idx), width.getItem(idx), height.getItem(idx)]);
        }
    }
    fillArc(args, buffers) {
        this.ctx.save();
        this.ctx.beginPath();
        this.executeCommand('arc', args);
        this.ctx.closePath();
        this.ctx.fill();
        this.ctx.restore();
    }
    strokeArc(args, buffers) {
        this.ctx.save();
        this.ctx.beginPath();
        this.executeCommand('arc', args);
        this.ctx.closePath();
        this.ctx.stroke();
        this.ctx.restore();
    }
    drawArcs(args, buffers, commandName) {
        const x = utils_1.getArg(args[0], buffers);
        const y = utils_1.getArg(args[1], buffers);
        const radius = utils_1.getArg(args[2], buffers);
        const startAngle = utils_1.getArg(args[3], buffers);
        const endAngle = utils_1.getArg(args[4], buffers);
        const anticlockwise = utils_1.getArg(args[5], buffers);
        const numberArcs = Math.min(x.length, y.length, radius.length, startAngle.length, endAngle.length);
        this.ctx.save();
        for (let idx = 0; idx < numberArcs; ++idx) {
            this.ctx.beginPath();
            this.ctx.arc(x.getItem(idx), y.getItem(idx), radius.getItem(idx), startAngle.getItem(idx), endAngle.getItem(idx), anticlockwise.getItem(idx));
            this.ctx.closePath();
            this.executeCommand(commandName);
        }
        this.ctx.restore();
    }
    drawImage(args, buffers) {
        return __awaiter(this, void 0, void 0, function* () {
            const [serializedImage, x, y, width, height] = args;
            const image = yield base_1.unpack_models(serializedImage, this.widget_manager);
            if (image instanceof CanvasModel) {
                this._drawImage(image.canvas, x, y, width, height);
                return;
            }
            if (image.get('_model_name') == 'ImageModel') {
                // Create the image manually instead of creating an ImageView
                let url;
                const format = image.get('format');
                const value = image.get('value');
                if (format !== 'url') {
                    const blob = new Blob([value], { type: `image/${format}` });
                    url = URL.createObjectURL(blob);
                }
                else {
                    url = (new TextDecoder('utf-8')).decode(value.buffer);
                }
                const img = new Image();
                return new Promise((resolve) => {
                    img.onload = () => {
                        this._drawImage(img, x, y, width, height);
                        resolve();
                    };
                    img.src = url;
                });
            }
        });
    }
    _drawImage(image, x, y, width, height) {
        if (width === undefined || height === undefined) {
            this.ctx.drawImage(image, x, y);
        }
        else {
            this.ctx.drawImage(image, x, y, width, height);
        }
    }
    putImageData(args, buffers) {
        const [bufferMetadata, dx, dy] = args;
        const width = bufferMetadata.shape[1];
        const height = bufferMetadata.shape[0];
        const data = new Uint8ClampedArray(buffers[0].buffer);
        const imageData = new ImageData(data, width, height);
        // Draw on a temporary off-screen canvas. This is a workaround for `putImageData` to support transparency.
        const offscreenCanvas = document.createElement('canvas');
        offscreenCanvas.width = width;
        offscreenCanvas.height = height;
        getContext(offscreenCanvas).putImageData(imageData, 0, 0);
        this.ctx.drawImage(offscreenCanvas, dx, dy);
    }
    setAttr(attr, value) {
        this.ctx[attr] = value;
    }
    clearCanvas() {
        this.forEachView((view) => {
            view.clear();
        });
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }
    executeCommand(name, args = []) {
        this.ctx[name](...args);
    }
    forEachView(callback) {
        for (const view_id in this.views) {
            this.views[view_id].then((view) => {
                callback(view);
            });
        }
    }
    resizeCanvas() {
        this.canvas.setAttribute('width', this.get('width'));
        this.canvas.setAttribute('height', this.get('height'));
    }
    syncImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.get('sync_image_data')) {
                return;
            }
            const bytes = yield utils_1.toBytes(this.canvas);
            this.set('image_data', bytes);
            this.save_changes();
        });
    }
}
exports.CanvasModel = CanvasModel;
CanvasModel.serializers = Object.assign(Object.assign({}, base_1.DOMWidgetModel.serializers), { image_data: {
        serialize: serializeImageData,
        deserialize: deserializeImageData
    } });
CanvasModel.model_name = 'CanvasModel';
CanvasModel.model_module = version_1.MODULE_NAME;
CanvasModel.model_module_version = version_1.MODULE_VERSION;
CanvasModel.view_name = 'CanvasView';
CanvasModel.view_module = version_1.MODULE_NAME;
CanvasModel.view_module_version = version_1.MODULE_VERSION;
class CanvasView extends base_1.DOMWidgetView {
    render() {
        this.canvas = document.createElement('canvas');
        this.el.appendChild(this.canvas);
        this.ctx = getContext(this.canvas);
        this.resizeCanvas();
        this.model.on_some_change(['width', 'height'], this.resizeCanvas, this);
        this.canvas.addEventListener('mousemove', { handleEvent: this.onMouseMove.bind(this) });
        this.canvas.addEventListener('mousedown', { handleEvent: this.onMouseDown.bind(this) });
        this.canvas.addEventListener('mouseup', { handleEvent: this.onMouseUp.bind(this) });
        this.canvas.addEventListener('mouseout', { handleEvent: this.onMouseOut.bind(this) });
        this.canvas.addEventListener('touchstart', { handleEvent: this.onTouchStart.bind(this) });
        this.canvas.addEventListener('touchend', { handleEvent: this.onTouchEnd.bind(this) });
        this.canvas.addEventListener('touchmove', { handleEvent: this.onTouchMove.bind(this) });
        this.canvas.addEventListener('touchcancel', { handleEvent: this.onTouchCancel.bind(this) });
        this.updateCanvas();
    }
    clear() {
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }
    updateCanvas() {
        this.clear();
        this.ctx.drawImage(this.model.canvas, 0, 0);
    }
    resizeCanvas() {
        this.canvas.setAttribute('width', this.model.get('width'));
        this.canvas.setAttribute('height', this.model.get('height'));
    }
    onMouseMove(event) {
        this.model.send(Object.assign({ event: 'mouse_move' }, this.getCoordinates(event)), {});
    }
    onMouseDown(event) {
        this.model.send(Object.assign({ event: 'mouse_down' }, this.getCoordinates(event)), {});
    }
    onMouseUp(event) {
        this.model.send(Object.assign({ event: 'mouse_up' }, this.getCoordinates(event)), {});
    }
    onMouseOut(event) {
        this.model.send(Object.assign({ event: 'mouse_out' }, this.getCoordinates(event)), {});
    }
    onTouchStart(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_start', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchEnd(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_end', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchMove(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_move', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    onTouchCancel(event) {
        const touches = Array.from(event.touches);
        this.model.send({ event: 'touch_cancel', touches: touches.map(this.getCoordinates.bind(this)) }, {});
    }
    getCoordinates(event) {
        const rect = this.canvas.getBoundingClientRect();
        const x = event.clientX - rect.left;
        const y = event.clientY - rect.top;
        return { x, y };
    }
}
exports.CanvasView = CanvasView;
class MultiCanvasModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: MultiCanvasModel.model_name, _model_module: MultiCanvasModel.model_module, _model_module_version: MultiCanvasModel.model_module_version, _view_name: MultiCanvasModel.view_name, _view_module: MultiCanvasModel.view_module, _view_module_version: MultiCanvasModel.view_module_version, _canvases: [], sync_image_data: false, image_data: null });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.on('change:_canvases', this.updateListeners.bind(this));
        this.on('change:sync_image_data', this.syncImageData.bind(this));
        this.updateListeners();
    }
    updateListeners() {
        // TODO: Remove old listeners
        for (const canvasModel of this.get('_canvases')) {
            canvasModel.on('new-frame', this.syncImageData, this);
        }
    }
    syncImageData() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.get('sync_image_data')) {
                return;
            }
            // Draw on a temporary off-screen canvas.
            const offscreenCanvas = document.createElement('canvas');
            offscreenCanvas.width = this.get('width');
            offscreenCanvas.height = this.get('height');
            const ctx = getContext(offscreenCanvas);
            for (const canvasModel of this.get('_canvases')) {
                ctx.drawImage(canvasModel.canvas, 0, 0);
                // Also update the sub-canvas image-data
                const bytes = yield utils_1.toBytes(canvasModel.canvas);
                canvasModel.set('image_data', bytes);
                canvasModel.save_changes();
            }
            const bytes = yield utils_1.toBytes(offscreenCanvas);
            this.set('image_data', bytes);
            this.save_changes();
        });
    }
}
exports.MultiCanvasModel = MultiCanvasModel;
MultiCanvasModel.serializers = Object.assign(Object.assign({}, base_1.DOMWidgetModel.serializers), { _canvases: { deserialize: base_1.unpack_models }, image_data: { serialize: (bytes) => {
            return new DataView(bytes.buffer.slice(0));
        } } });
MultiCanvasModel.model_name = 'MultiCanvasModel';
MultiCanvasModel.model_module = version_1.MODULE_NAME;
MultiCanvasModel.model_module_version = version_1.MODULE_VERSION;
MultiCanvasModel.view_name = 'MultiCanvasView';
MultiCanvasModel.view_module = version_1.MODULE_NAME;
MultiCanvasModel.view_module_version = version_1.MODULE_VERSION;
class MultiCanvasView extends base_1.DOMWidgetView {
    render() {
        this.container = document.createElement('div');
        this.container.style.position = 'relative';
        this.el.appendChild(this.container);
        this.canvas_views = new base_1.ViewList(this.createCanvasView, this.removeCanvasView, this);
        this.updateCanvasViews();
        this.model.on('change:_canvases', this.updateCanvasViews.bind(this));
    }
    updateCanvasViews() {
        this.canvas_views.update(this.model.get('_canvases'));
    }
    createCanvasView(canvasModel, index) {
        // The following ts-ignore is needed due to ipywidgets's implementation
        // @ts-ignore
        return this.create_child_view(canvasModel).then((canvasView) => {
            canvasView.el.style.zIndex = index;
            if (index == 0) {
                // This will enforce the container to respect the children size.
                canvasView.el.style.position = 'relative';
                canvasView.el.style.float = 'left';
            }
            else {
                canvasView.el.style.position = 'absolute';
            }
            this.container.appendChild(canvasView.el);
            return canvasView;
        });
    }
    removeCanvasView(canvasView) {
        this.container.removeChild(canvasView.el);
    }
}
exports.MultiCanvasView = MultiCanvasView;
//# sourceMappingURL=widget.js.map