# SGX DCAP quote verify for Python

This package provides a Python binding to the [SGX Quote Verification Library](https://github.com/intel/SGXDataCenterAttestationPrimitives/tree/master/QuoteVerification/QVL) (QVL), which is the reference implementation of ECDSA-based SGX quote verification. It allows you to verify ECDSA-based quotes generated by the Intel provided Quoting Enclave in Python.

## Usage

```py
import sgx_dcap_quote_verify

from pathlib import Path
from datetime import datetime

# You can download the trusted root CA in PEM format directly from Intel at :
# <https://certificates.trustedservices.intel.com/Intel_SGX_Provisioning_Certification_RootCA.pem>
trusted_root_ca_certificate = Path("path/to/root_ca_certificate.pem").read_text()

# Get the quote and the collateral from the service you want to attest
pck_certificate = Path("path/to/pck_certificate.pem").read_text()
pck_signing_chain = Path("path/to/pck_signing_chain.pem").read_text()
root_ca_crl = Path("/path/to/root_ca_crl.pem").read_text()
intermediate_ca_crl = Path("/path/to/intermediate_ca_crl.pem").read_text()
tcb_info = Path("/path/to/tcb_info.json").read_text()
tcb_signing_chain = Path("/path/to/tcb_signing_chain.pem").read_text()
quote = Path("/path/to/quote.dat").read_bytes()
qe_identity = Path("/path/to/qe_identity.json").read_text()

# Set the date used to check if the collateral (certificates,CRLs...) is still valid
# Except for test purposes it should be set to the current time as is done below
expiration_date = datetime.now()

# Use the package to check the validity of the quote
attestation_result = sgx_dcap_quote_verify.verify(
    trusted_root_ca_certificate,
    pck_certificate,
    pck_signing_chain,
    root_ca_crl,
    intermediate_ca_crl,
    tcb_info,
    tcb_signing_chain,
    quote,
    qe_identity,
    expiration_date,
)

assert attestation_result.ok
assert (
    attestation_result.pck_certificate_status
    == sgx_dcap_quote_verify.VerificationStatus.STATUS_OK
)
assert (
    attestation_result.tcb_info_status
    == sgx_dcap_quote_verify.VerificationStatus.STATUS_OK
)
assert (
    attestation_result.qe_identity_status
    == sgx_dcap_quote_verify.VerificationStatus.STATUS_OK
)
assert (
    attestation_result.quote_status
    == sgx_dcap_quote_verify.VerificationStatus.STATUS_OK
)

# The attestation result contains the report data, which includes the MR_ENCLAVE
print("mr_enclave =", attestation_result.enclave_report.mr_enclave)
```

**Disclaimer** : This package is not endorsed by Intel Corporation. It is provided as is, use it at your own risk.

## License

The source code of the binding is provided under Apache-2.0 license.

This software also uses the SGX Quote Verification Library, which is licensed under [BSD license](https://github.com/mithril-security/SGXDataCenterAttestationPrimitives/blob/master/License.txt).
Distribution of the software as a whole, including the external library, may be subject to the terms of the external library's license.
