"""A minimal example module for demonstrating pdoc-generated documentation.

This module provides a simple greeting function and a CLI entry point.
The main library function is `print_hi()`, while the CLI interface
is available through the `cli` command.

Example:
    As a library:
        >>> from augint_library.cli import print_hi
        >>> print_hi("Alice")
        Hi Alice

    As a CLI:
        $ ai-test-script Alice
        Hi Alice
"""

import click

# Control what pdoc shows - CLI command typically not imported
__all__ = ["print_hi"]


def print_hi(name: str) -> None:
    """Print a friendly greeting to the given name.

    This is the main library function that can be imported and used
    in other Python code.

    Args:
        name: The name of the person to greet.

    Example:
        >>> print_hi("Alice")
        Hi Alice
    """
    print(f"Hi {name}")


@click.command()
@click.argument("name", default="there", type=str)
def cli(name: str) -> None:
    """CLI entry point for the greeting command.

    This function is the Click command that provides the CLI interface.
    It's typically not imported directly but used via the installed
    script entry point.

    Args:
        name: The name to greet (default: "there").

    Note:
        This function is excluded from the public API via __all__.
        Use the installed CLI command instead of importing this directly.
    """
    print_hi(name)


if __name__ == "__main__":
    cli()
