"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains do not allow for unsigned requests or anonymous access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const accessPolicies = getPolicyFromCR() ?? aws_cdk_lib_1.Stack.of(node).resolve(node.accessPolicies);
        if (accessPolicies == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const statements = accessPolicies?.Statement;
        if (statements === undefined || statements.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        for (const statement of statements) {
            if (statement.Effect == 'Allow') {
                const awsString = statement.Principal
                    ? JSON.stringify(statement.Principal)
                    : '';
                if (awsString.includes('*')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    function getPolicyFromCR() {
        const crPath = node.node.path.split('/Resource').slice(0, -1).join('/Resource') +
            (node instanceof aws_opensearchservice_1.CfnDomain ? '/AccessPolicy' : '/ESAccessPolicy') +
            '/Resource/Default';
        const cr = aws_cdk_lib_1.Stack.of(node)
            .node.findAll()
            .find((c) => c.node.path === crPath);
        if (cr) {
            const props = aws_cdk_lib_1.Stack.of(cr).resolve(cr._cfnProperties);
            const update = props?.Update?.['Fn::Join']?.[1];
            return JSON.parse(JSON.parse(update.join('')).parameters.AccessPolicies);
        }
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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