"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const nag_rules_1 = require("../../nag-rules");
/**
 * Glue crawlers and jobs have CloudWatch Log encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_glue_1.CfnCrawler || node instanceof aws_glue_1.CfnJob) {
        let securityConfigurationId;
        if (node instanceof aws_glue_1.CfnCrawler) {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.crawlerSecurityConfiguration);
        }
        else {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.securityConfiguration);
        }
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_glue_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration encrypts CloudWatch Logs
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration encrypts CloudWatch Logs
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const encryptionConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(node.encryptionConfiguration);
        const cloudWatchEncryption = encryptionConfiguration.cloudWatchEncryption;
        if (cloudWatchEncryption !== undefined) {
            const cloudWatchEncryptionMode = nag_rules_1.NagRules.resolveIfPrimitive(node, cloudWatchEncryption.cloudWatchEncryptionMode);
            if (cloudWatchEncryptionMode === 'SSE-KMS') {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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