"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const nag_rules_1 = require("../../nag-rules");
/**
 * CodeBuild projects do not store AWS credentials as plaintext environment variables
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_codebuild_1.CfnProject) {
        //Check for the presence of OAUTH
        const environment = aws_cdk_lib_1.Stack.of(node).resolve(node.environment);
        const environmentVars = aws_cdk_lib_1.Stack.of(node).resolve(environment.environmentVariables);
        if (environmentVars != undefined) {
            //For each envvar, check if its a sensitive credential being stored
            for (const envVar of environmentVars) {
                const resolvedEnvVar = aws_cdk_lib_1.Stack.of(node).resolve(envVar);
                const name = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedEnvVar.name);
                const type = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedEnvVar.type);
                if (name == 'AWS_ACCESS_KEY_ID' || name == 'AWS_SECRET_ACCESS_KEY') {
                    //is this credential being stored as plaintext?
                    if (type == undefined || type == 'PLAINTEXT') {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQ29kZUJ1aWxkUHJvamVjdEVudlZhckF3c0NyZWQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcnVsZXMvY29kZWJ1aWxkL0NvZGVCdWlsZFByb2plY3RFbnZWYXJBd3NDcmVkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLDZDQUFpRDtBQUNqRCw2REFBdUQ7QUFDdkQsK0NBQThEO0FBRTlEOzs7R0FHRztBQUNILGtCQUFlLE1BQU0sQ0FBQyxjQUFjLENBQ2xDLENBQUMsSUFBaUIsRUFBcUIsRUFBRTtJQUN2QyxJQUFJLElBQUksWUFBWSwwQkFBVSxFQUFFO1FBQzlCLGlDQUFpQztRQUNqQyxNQUFNLFdBQVcsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzdELE1BQU0sZUFBZSxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDNUMsV0FBVyxDQUFDLG9CQUFvQixDQUNqQyxDQUFDO1FBQ0YsSUFBSSxlQUFlLElBQUksU0FBUyxFQUFFO1lBQ2hDLG1FQUFtRTtZQUNuRSxLQUFLLE1BQU0sTUFBTSxJQUFJLGVBQWUsRUFBRTtnQkFDcEMsTUFBTSxjQUFjLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUN0RCxNQUFNLElBQUksR0FBRyxvQkFBUSxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3BFLE1BQU0sSUFBSSxHQUFHLG9CQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDcEUsSUFBSSxJQUFJLElBQUksbUJBQW1CLElBQUksSUFBSSxJQUFJLHVCQUF1QixFQUFFO29CQUNsRSwrQ0FBK0M7b0JBQy9DLElBQUksSUFBSSxJQUFJLFNBQVMsSUFBSSxJQUFJLElBQUksV0FBVyxFQUFFO3dCQUM1QyxPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztxQkFDeEM7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0QsT0FBTyw2QkFBaUIsQ0FBQyxTQUFTLENBQUM7S0FDcEM7U0FBTTtRQUNMLE9BQU8sNkJBQWlCLENBQUMsY0FBYyxDQUFDO0tBQ3pDO0FBQ0gsQ0FBQyxFQUNELE1BQU0sRUFDTixFQUFFLEtBQUssRUFBRSxZQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQ2xDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IHBhcnNlIH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5Qcm9qZWN0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNvZGVidWlsZCc7XG5pbXBvcnQgeyBOYWdSdWxlQ29tcGxpYW5jZSwgTmFnUnVsZXMgfSBmcm9tICcuLi8uLi9uYWctcnVsZXMnO1xuXG4vKipcbiAqIENvZGVCdWlsZCBwcm9qZWN0cyBkbyBub3Qgc3RvcmUgQVdTIGNyZWRlbnRpYWxzIGFzIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXNcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBPYmplY3QuZGVmaW5lUHJvcGVydHkoXG4gIChub2RlOiBDZm5SZXNvdXJjZSk6IE5hZ1J1bGVDb21wbGlhbmNlID0+IHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmblByb2plY3QpIHtcbiAgICAgIC8vQ2hlY2sgZm9yIHRoZSBwcmVzZW5jZSBvZiBPQVVUSFxuICAgICAgY29uc3QgZW52aXJvbm1lbnQgPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUuZW52aXJvbm1lbnQpO1xuICAgICAgY29uc3QgZW52aXJvbm1lbnRWYXJzID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgZW52aXJvbm1lbnQuZW52aXJvbm1lbnRWYXJpYWJsZXNcbiAgICAgICk7XG4gICAgICBpZiAoZW52aXJvbm1lbnRWYXJzICE9IHVuZGVmaW5lZCkge1xuICAgICAgICAvL0ZvciBlYWNoIGVudnZhciwgY2hlY2sgaWYgaXRzIGEgc2Vuc2l0aXZlIGNyZWRlbnRpYWwgYmVpbmcgc3RvcmVkXG4gICAgICAgIGZvciAoY29uc3QgZW52VmFyIG9mIGVudmlyb25tZW50VmFycykge1xuICAgICAgICAgIGNvbnN0IHJlc29sdmVkRW52VmFyID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShlbnZWYXIpO1xuICAgICAgICAgIGNvbnN0IG5hbWUgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUobm9kZSwgcmVzb2x2ZWRFbnZWYXIubmFtZSk7XG4gICAgICAgICAgY29uc3QgdHlwZSA9IE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShub2RlLCByZXNvbHZlZEVudlZhci50eXBlKTtcbiAgICAgICAgICBpZiAobmFtZSA9PSAnQVdTX0FDQ0VTU19LRVlfSUQnIHx8IG5hbWUgPT0gJ0FXU19TRUNSRVRfQUNDRVNTX0tFWScpIHtcbiAgICAgICAgICAgIC8vaXMgdGhpcyBjcmVkZW50aWFsIGJlaW5nIHN0b3JlZCBhcyBwbGFpbnRleHQ/XG4gICAgICAgICAgICBpZiAodHlwZSA9PSB1bmRlZmluZWQgfHwgdHlwZSA9PSAnUExBSU5URVhUJykge1xuICAgICAgICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuIl19