#
# TextLength.py: A utility for computing lengths of text
# 
# Copyright (c) 2021-2022 Vincent A Cicirello
# https://www.cicirello.org/
#
# MIT License
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#


# NOTE: This file originated with https://github.com/cicirello/user-statistician/

def calculateTextLength(s, size, pixels, fontWeight) :
    """Calculates the length of a string in DejaVu Sans for
    a specified font size.

    Keyword arguments:
    s - The string.
    size - The font size.
    pixels - If True, the size is in px, otherwise it is in pt.
    fontWeight - The weight of the font (e.g., 400 for normal, 600 for bold, etc)
    """
    if pixels :
        size *= 0.75
    weightMultiplier = 1
    if fontWeight != 400 :
        weightMultiplier = fontWeight / 400
    return weightMultiplier * size * calculateTextLength110(s) / 110

def calculateTextLength110Weighted(s, fontWeight) :
    """Calculates the length of a string in DejaVu Sans 110pt font,
    factoring in font weight.

    Keyword arguments:
    s - The string.
    fontWeight - The weight of the font (e.g., 400 for normal, 600 for bold, etc)
    """
    weightMultiplier = 1
    if fontWeight != 400 :
        weightMultiplier = fontWeight / 400
    return weightMultiplier * calculateTextLength110(s)

def calculateTextLength110(s) :
    """Calculates the length of a string in DejaVu Sans 110pt font.

    Keyword arguments:
    s - The string.
    """
    if s==None or len(s) == 0 :
        return 0
    total = sum(defaultWidths["character-lengths"][c] if c in defaultWidths["character-lengths"] else defaultWidths["mean-character-length"] for c in s)
    for i in range(1,len(s)) :
        pair = s[i-1:i+1]
        if pair in defaultWidths["kerning-pairs"] :
            total -= defaultWidths["kerning-pairs"][pair]
    return total

########################################
# The dict that follows is derived from
# default-widths.json from
# https://github.com/google/pybadges,
# which is licensed under Apache-2.0.
########################################

defaultWidths = {'character-lengths': {' ': 35,
'!': 44,
'"': 51,
'#': 92,
'$': 70,
'%': 105,
'&': 86,
"'": 30,
'(': 43,
')': 43,
'*': 55,
'+': 92,
',': 35,
'-': 40,
'.': 35,
'/': 38,
'0': 70,
'1': 70,
'2': 70,
'3': 70,
'4': 70,
'5': 70,
'6': 70,
'7': 70,
'8': 70,
'9': 70,
':': 37,
';': 37,
'<': 92,
'=': 92,
'>': 92,
'?': 58,
'@': 110,
'A': 75,
'B': 75,
'C': 77,
'D': 85,
'E': 70,
'F': 63,
'G': 85,
'H': 83,
'I': 32,
'J': 38,
'K': 75,
'L': 61,
'M': 95,
'N': 82,
'O': 87,
'P': 66,
'Q': 87,
'R': 76,
'S': 70,
'T': 69,
'U': 81,
'V': 75,
'W': 109,
'X': 75,
'Y': 69,
'Z': 75,
'[': 43,
'\\': 38,
']': 43,
'^': 92,
'_': 59,
'`': 55,
'a': 67,
'b': 70,
'c': 60,
'd': 70,
'e': 68,
'f': 41,
'g': 70,
'h': 70,
'i': 31,
'j': 33,
'k': 64,
'l': 31,
'm': 107,
'n': 70,
'o': 67,
'p': 70,
'q': 70,
'r': 46,
's': 57,
't': 43,
'u': 70,
'v': 65,
'w': 90,
'x': 65,
'y': 65,
'z': 58,
'{': 70,
'|': 37,
'}': 70,
'~': 92,
'\xa0': 35,
'¡': 44,
'¢': 70,
'£': 70,
'¤': 70,
'¥': 70,
'¦': 37,
'§': 55,
'¨': 55,
'©': 110,
'ª': 52,
'«': 67,
'¬': 92,
'\xad': 40,
'®': 110,
'¯': 55,
'°': 55,
'±': 92,
'²': 44,
'³': 44,
'´': 55,
'µ': 70,
'¶': 70,
'·': 35,
'¸': 55,
'¹': 44,
'º': 52,
'»': 67,
'¼': 107,
'½': 107,
'¾': 107,
'¿': 58,
'À': 75,
'Á': 75,
'Â': 75,
'Ã': 75,
'Ä': 75,
'Å': 75,
'Æ': 107,
'Ç': 77,
'È': 70,
'É': 70,
'Ê': 70,
'Ë': 70,
'Ì': 32,
'Í': 32,
'Î': 34,
'Ï': 33,
'Ð': 85,
'Ñ': 82,
'Ò': 87,
'Ó': 87,
'Ô': 87,
'Õ': 87,
'Ö': 87,
'×': 92,
'Ø': 87,
'Ù': 81,
'Ú': 81,
'Û': 81,
'Ü': 81,
'Ý': 69,
'Þ': 67,
'ß': 69,
'à': 67,
'á': 67,
'â': 67,
'ã': 67,
'ä': 67,
'å': 67,
'æ': 108,
'ç': 60,
'è': 68,
'é': 68,
'ê': 68,
'ë': 68,
'ì': 35,
'í': 34,
'î': 35,
'ï': 33,
'ð': 67,
'ñ': 70,
'ò': 67,
'ó': 67,
'ô': 67,
'õ': 67,
'ö': 67,
'÷': 92,
'ø': 67,
'ù': 70,
'ú': 70,
'û': 70,
'ü': 70,
'ý': 65,
'þ': 70,
'ÿ': 65,
'Ā': 75,
'ā': 67,
'Ă': 75,
'ă': 67,
'Ą': 78,
'ą': 67,
'Ć': 77,
'ć': 60,
'Ĉ': 77,
'ĉ': 60,
'Ċ': 77,
'ċ': 60,
'Č': 77,
'č': 60,
'Ď': 85,
'ď': 81,
'Đ': 85,
'đ': 70,
'Ē': 70,
'ē': 68,
'Ĕ': 70,
'ĕ': 68,
'Ė': 70,
'ė': 68,
'Ę': 70,
'ę': 68,
'Ě': 70,
'ě': 68,
'Ĝ': 85,
'ĝ': 70,
'Ğ': 85,
'ğ': 70,
'Ġ': 85,
'ġ': 70,
'Ģ': 85,
'ģ': 70,
'Ĥ': 83,
'ĥ': 72,
'Ħ': 101,
'ħ': 76,
'Ĩ': 36,
'ĩ': 37,
'Ī': 33,
'ī': 33,
'Ĭ': 35,
'ĭ': 35,
'Į': 32,
'į': 31,
'İ': 32,
'ı': 31,
'Ĳ': 65,
'ĳ': 61,
'Ĵ': 39,
'ĵ': 35,
'Ķ': 75,
'ķ': 64,
'ĸ': 64,
'Ĺ': 61,
'ĺ': 32,
'Ļ': 61,
'ļ': 31,
'Ľ': 61,
'ľ': 42,
'Ŀ': 61,
'ŀ': 38,
'Ł': 63,
'ł': 32,
'Ń': 82,
'ń': 70,
'Ņ': 82,
'ņ': 70,
'Ň': 82,
'ň': 70,
'ŉ': 89,
'Ŋ': 82,
'ŋ': 70,
'Ō': 87,
'ō': 67,
'Ŏ': 87,
'ŏ': 67,
'Ő': 87,
'ő': 67,
'Œ': 118,
'œ': 113,
'Ŕ': 76,
'ŕ': 50,
'Ŗ': 76,
'ŗ': 46,
'Ř': 76,
'ř': 47,
'Ś': 70,
'ś': 57,
'Ŝ': 70,
'ŝ': 57,
'Ş': 70,
'ş': 57,
'Š': 70,
'š': 57,
'Ţ': 69,
'ţ': 43,
'Ť': 69,
'ť': 43,
'Ŧ': 69,
'ŧ': 43,
'Ũ': 81,
'ũ': 70,
'Ū': 81,
'ū': 70,
'Ŭ': 81,
'ŭ': 70,
'Ů': 81,
'ů': 70,
'Ű': 81,
'ű': 70,
'Ų': 81,
'ų': 70,
'Ŵ': 109,
'ŵ': 90,
'Ŷ': 69,
'ŷ': 65,
'Ÿ': 69,
'Ź': 75,
'ź': 58,
'Ż': 75,
'ż': 58,
'Ž': 75,
'ž': 58,
'ſ': 41,
'ƀ': 70,
'Ɓ': 87,
'Ƃ': 75,
'ƃ': 70,
'Ƅ': 75,
'ƅ': 70,
'Ɔ': 77,
'Ƈ': 88,
'ƈ': 66,
'Ɖ': 85,
'Ɗ': 96,
'Ƌ': 75,
'ƌ': 70,
'ƍ': 67,
'Ǝ': 70,
'Ə': 87,
'Ɛ': 68,
'Ƒ': 69,
'ƒ': 48,
'Ɠ': 91,
'Ɣ': 76,
'ƕ': 108,
'Ɩ': 39,
'Ɨ': 32,
'Ƙ': 83,
'ƙ': 64,
'ƚ': 31,
'ƛ': 65,
'Ɯ': 107,
'Ɲ': 88,
'ƞ': 70,
'Ɵ': 87,
'Ơ': 100,
'ơ': 67,
'Ƣ': 104,
'ƣ': 84,
'Ƥ': 78,
'ƥ': 70,
'Ʀ': 76,
'Ƨ': 70,
'ƨ': 57,
'Ʃ': 70,
'ƪ': 55,
'ƫ': 43,
'Ƭ': 68,
'ƭ': 43,
'Ʈ': 69,
'Ư': 94,
'ư': 75,
'Ʊ': 84,
'Ʋ': 79,
'Ƴ': 83,
'ƴ': 81,
'Ƶ': 75,
'ƶ': 58,
'Ʒ': 73,
'Ƹ': 73,
'ƹ': 64,
'ƺ': 58,
'ƻ': 70,
'Ƽ': 73,
'ƽ': 64,
'ƾ': 56,
'ƿ': 70,
'ǀ': 32,
'ǁ': 54,
'ǂ': 50,
'ǃ': 33,
'Ǆ': 156,
'ǅ': 143,
'ǆ': 127,
'Ǉ': 92,
'ǈ': 87,
'ǉ': 50,
'Ǌ': 102,
'ǋ': 102,
'ǌ': 88,
'Ǎ': 75,
'ǎ': 67,
'Ǐ': 34,
'ǐ': 35,
'Ǒ': 87,
'ǒ': 67,
'Ǔ': 81,
'ǔ': 70,
'Ǖ': 81,
'ǖ': 70,
'Ǘ': 81,
'ǘ': 70,
'Ǚ': 81,
'ǚ': 70,
'Ǜ': 81,
'ǜ': 70,
'ǝ': 68,
'Ǟ': 75,
'ǟ': 67,
'Ǡ': 75,
'ǡ': 67,
'Ǣ': 107,
'ǣ': 108,
'Ǥ': 85,
'ǥ': 70,
'Ǧ': 85,
'ǧ': 70,
'Ǩ': 75,
'ǩ': 66,
'Ǫ': 87,
'ǫ': 67,
'Ǭ': 87,
'ǭ': 67,
'Ǯ': 73,
'ǯ': 64,
'ǰ': 35,
'Ǳ': 156,
'ǲ': 143,
'ǳ': 127,
'Ǵ': 85,
'ǵ': 70,
'Ƕ': 122,
'Ƿ': 75,
'Ǹ': 82,
'ǹ': 70,
'Ǻ': 75,
'ǻ': 67,
'Ǽ': 107,
'ǽ': 108,
'Ǿ': 87,
'ǿ': 67,
'Ȁ': 75,
'ȁ': 67,
'Ȃ': 75,
'ȃ': 67,
'Ȅ': 70,
'ȅ': 68,
'Ȇ': 70,
'ȇ': 68,
'Ȉ': 39,
'ȉ': 39,
'Ȋ': 34,
'ȋ': 35,
'Ȍ': 87,
'ȍ': 67,
'Ȏ': 87,
'ȏ': 67,
'Ȑ': 76,
'ȑ': 46,
'Ȓ': 76,
'ȓ': 47,
'Ȕ': 81,
'ȕ': 70,
'Ȗ': 81,
'ȗ': 70,
'Ș': 70,
'ș': 57,
'Ț': 69,
'ț': 43,
'Ȝ': 69,
'ȝ': 57,
'Ȟ': 83,
'ȟ': 71,
'Ƞ': 81,
'ȡ': 92,
'Ȣ': 77,
'ȣ': 67,
'Ȥ': 75,
'ȥ': 58,
'Ȧ': 75,
'ȧ': 67,
'Ȩ': 70,
'ȩ': 68,
'Ȫ': 87,
'ȫ': 67,
'Ȭ': 87,
'ȭ': 67,
'Ȯ': 87,
'ȯ': 67,
'Ȱ': 87,
'ȱ': 67,
'Ȳ': 69,
'ȳ': 65,
'ȴ': 52,
'ȵ': 93,
'ȶ': 52,
'ȷ': 33,
'ȸ': 110,
'ȹ': 110,
'Ⱥ': 77,
'Ȼ': 77,
'ȼ': 60,
'Ƚ': 61,
'Ⱦ': 77,
'ȿ': 57,
'ɀ': 58,
'Ɂ': 66,
'ɂ': 53,
'Ƀ': 75,
'Ʉ': 81,
'Ʌ': 75,
'Ɇ': 70,
'ɇ': 68,
'Ɉ': 38,
'ɉ': 33,
'Ɋ': 92,
'ɋ': 73,
'Ɍ': 76,
'ɍ': 46,
'Ɏ': 69,
'ɏ': 65,
'ɐ': 66,
'ɑ': 70,
'ɒ': 70,
'ɓ': 70,
'ɔ': 60,
'ɕ': 60,
'ɖ': 73,
'ɗ': 79,
'ɘ': 68,
'ə': 68,
'ɚ': 90,
'ɛ': 59,
'ɜ': 58,
'ɝ': 85,
'ɞ': 73,
'ɟ': 33,
'ɠ': 79,
'ɡ': 70,
'ɢ': 69,
'ɣ': 66,
'ɤ': 66,
'ɥ': 70,
'ɦ': 70,
'ɧ': 70,
'ɨ': 31,
'ɩ': 37,
'ɪ': 41,
'ɫ': 44,
'ɬ': 54,
'ɭ': 33,
'ɮ': 78,
'ɯ': 107,
'ɰ': 107,
'ɱ': 107,
'ɲ': 73,
'ɳ': 73,
'ɴ': 70,
'ɵ': 67,
'ɶ': 94,
'ɷ': 80,
'ɸ': 73,
'ɹ': 46,
'ɺ': 46,
'ɻ': 48,
'ɼ': 46,
'ɽ': 46,
'ɾ': 58,
'ɿ': 58,
'ʀ': 66,
'ʁ': 66,
'ʂ': 57,
'ʃ': 43,
'ʄ': 43,
'ʅ': 51,
'ʆ': 55,
'ʇ': 43,
'ʈ': 43,
'ʉ': 70,
'ʊ': 68,
'ʋ': 66,
'ʌ': 65,
'ʍ': 90,
'ʎ': 65,
'ʏ': 67,
'ʐ': 66,
'ʑ': 58,
'ʒ': 64,
'ʓ': 64,
'ʔ': 56,
'ʕ': 56,
'ʖ': 56,
'ʗ': 56,
'ʘ': 87,
'ʙ': 64,
'ʚ': 73,
'ʛ': 80,
'ʜ': 72,
'ʝ': 47,
'ʞ': 73,
'ʟ': 56,
'ʠ': 83,
'ʡ': 56,
'ʢ': 56,
'ʣ': 112,
'ʤ': 116,
'ʥ': 111,
'ʦ': 91,
'ʧ': 70,
'ʨ': 86,
'ʩ': 93,
'ʪ': 78,
'ʫ': 72,
'ʬ': 57,
'ʭ': 57,
'ʮ': 73,
'ʯ': 76,
'ʰ': 44,
'ʱ': 44,
'ʲ': 21,
'ʳ': 29,
'ʴ': 33,
'ʵ': 34,
'ʶ': 42,
'ʷ': 57,
'ʸ': 41,
'ʹ': 31,
'ʺ': 51,
'ʻ': 35,
'ʼ': 35,
'ʽ': 35,
'ʾ': 34,
'ʿ': 34,
'ˀ': 41,
'ˁ': 41,
'˂': 55,
'˃': 55,
'˄': 55,
'˅': 55,
'ˆ': 55,
'ˇ': 55,
'ˈ': 30,
'ˉ': 55,
'ˊ': 55,
'ˋ': 55,
'ˌ': 30,
'ˍ': 55,
'ˎ': 55,
'ˏ': 55,
'ː': 37,
'ˑ': 37,
'˒': 34,
'˓': 34,
'˔': 55,
'˕': 55,
'˖': 43,
'˗': 35,
'˘': 55,
'˙': 55,
'˚': 55,
'˛': 55,
'˜': 55,
'˝': 55,
'˞': 38,
'˟': 55,
'ˠ': 47,
'ˡ': 18,
'ˢ': 41,
'ˣ': 49,
'ˤ': 41,
'˥': 54,
'˦': 54,
'˧': 54,
'˨': 54,
'˩': 54,
'ˬ': 55,
'˭': 55,
'ˮ': 57,
'˳': 55,
'˷': 55,
'̀': 46,
'́': 36,
'̂': 45,
'̃': 46,
'̄': 44,
'̅': 59,
'̆': 45,
'̇': 33,
'̈': 44,
'̉': 39,
'̊': 43,
'̋': 42,
'̌': 45,
'̍': 32,
'̎': 43,
'̏': 50,
'̐': 45,
'̑': 45,
'̒': 26,
'̓': 34,
'̔': 34,
'̕': 16,
'̖': 46,
'̗': 36,
'̘': 40,
'̙': 31,
'̚': 27,
'̛': 22,
'̜': 35,
'̝': 41,
'̞': 41,
'̟': 40,
'̠': 41,
'̡': 35,
'̢': 35,
'̣': 33,
'̤': 44,
'̥': 40,
'̦': 36,
'̧': 40,
'̨': 38,
'̩': 32,
'̪': 43,
'̫': 50,
'̬': 45,
'̭': 45,
'̮': 45,
'̯': 45,
'̰': 46,
'̱': 44,
'̲': 59,
'̳': 59,
'̴': 62,
'̵': 35,
'̶': 70,
'̷': 64,
'̸': 82,
'̹': 33,
'̺': 43,
'̻': 40,
'̼': 50,
'̽': 39,
'̾': 28,
'̿': 59,
'̀': 46,
'́': 36,
'͂': 46,
'̓': 34,
'̈́': 43,
'ͅ': 31,
'͆': 44,
'͇': 40,
'͈': 41,
'͉': 40,
'͊': 46,
'͋': 46,
'͌': 46,
'͍': 50,
'͎': 39,
'͏': 0,
'͑': 35,
'͒': 45,
'͓': 39,
'͗': 35,
'͘': 12,
'͚': 48,
'͜': 98,
'͝': 98,
'͞': 98,
'͟': 98,
'͠': 78,
'͡': 98,
'͢': 99,
'Ͱ': 72,
'ͱ': 62,
'Ͳ': 95,
'ͳ': 71,
'ʹ': 31,
'͵': 31,
'Ͷ': 82,
'ͷ': 71,
'ͺ': 55,
'ͻ': 60,
'ͼ': 60,
'ͽ': 60,
';': 37,
'΄': 55,
'΅': 55,
'Ά': 76,
'·': 35,
'Έ': 84,
'Ή': 97,
'Ί': 47,
'Ό': 90,
'Ύ': 93,
'Ώ': 93,
'ΐ': 37,
'Α': 75,
'Β': 75,
'Γ': 61,
'Δ': 75,
'Ε': 70,
'Ζ': 75,
'Η': 83,
'Θ': 87,
'Ι': 32,
'Κ': 75,
'Λ': 75,
'Μ': 95,
'Ν': 82,
'Ξ': 70,
'Ο': 87,
'Π': 83,
'Ρ': 66,
'Σ': 70,
'Τ': 69,
'Υ': 69,
'Φ': 87,
'Χ': 75,
'Ψ': 87,
'Ω': 84,
'Ϊ': 33,
'Ϋ': 69,
'ά': 73,
'έ': 59,
'ή': 70,
'ί': 37,
'ΰ': 64,
'α': 73,
'β': 70,
'γ': 65,
'δ': 67,
'ε': 59,
'ζ': 60,
'η': 70,
'θ': 67,
'ι': 37,
'κ': 65,
'λ': 65,
'μ': 70,
'ν': 61,
'ξ': 61,
'ο': 67,
'π': 66,
'ρ': 70,
'ς': 65,
'σ': 70,
'τ': 66,
'υ': 64,
'φ': 73,
'χ': 64,
'ψ': 73,
'ω': 92,
'ϊ': 37,
'ϋ': 64,
'ό': 67,
'ύ': 64,
'ώ': 92,
'Ϗ': 75,
'ϐ': 68,
'ϑ': 68,
'ϒ': 77,
'ϓ': 95,
'ϔ': 77,
'ϕ': 73,
'ϖ': 92,
'ϗ': 73,
'Ϙ': 87,
'ϙ': 67,
'Ϛ': 71,
'ϛ': 65,
'Ϝ': 63,
'ϝ': 61,
'Ϟ': 73,
'ϟ': 73,
'Ϡ': 95,
'ϡ': 69,
'Ϣ': 103,
'ϣ': 92,
'Ϥ': 83,
'ϥ': 73,
'Ϧ': 87,
'ϧ': 68,
'Ϩ': 76,
'ϩ': 67,
'Ϫ': 84,
'ϫ': 69,
'Ϭ': 77,
'ϭ': 67,
'Ϯ': 67,
'ϯ': 59,
'ϰ': 73,
'ϱ': 70,
'ϲ': 60,
'ϳ': 33,
'ϴ': 87,
'ϵ': 68,
'϶': 68,
'Ϸ': 67,
'ϸ': 70,
'Ϲ': 77,
'Ϻ': 95,
'ϻ': 72,
'ϼ': 70,
'Ͻ': 77,
'Ͼ': 77,
'Ͽ': 77,
'Ѐ': 70,
'Ё': 70,
'Ђ': 87,
'Ѓ': 67,
'Є': 77,
'Ѕ': 70,
'І': 32,
'Ї': 33,
'Ј': 38,
'Љ': 120,
'Њ': 115,
'Ћ': 87,
'Ќ': 78,
'Ѝ': 82,
'Ў': 67,
'Џ': 83,
'А': 75,
'Б': 75,
'В': 75,
'Г': 67,
'Д': 86,
'Е': 70,
'Ж': 118,
'З': 71,
'И': 82,
'Й': 82,
'К': 78,
'Л': 83,
'М': 95,
'Н': 83,
'О': 87,
'П': 83,
'Р': 66,
'С': 77,
'Т': 69,
'У': 67,
'Ф': 95,
'Х': 75,
'Ц': 85,
'Ч': 75,
'Ш': 118,
'Щ': 120,
'Ъ': 92,
'Ы': 97,
'Ь': 75,
'Э': 77,
'Ю': 119,
'Я': 76,
'а': 67,
'б': 68,
'в': 65,
'г': 58,
'д': 76,
'е': 68,
'ж': 99,
'з': 58,
'и': 71,
'й': 71,
'к': 66,
'л': 70,
'м': 83,
'н': 72,
'о': 67,
'п': 72,
'р': 70,
'с': 60,
'т': 64,
'у': 65,
'ф': 94,
'х': 65,
'ц': 75,
'ч': 65,
'ш': 101,
'щ': 104,
'ъ': 78,
'ы': 87,
'ь': 65,
'э': 60,
'ю': 93,
'я': 66,
'ѐ': 68,
'ё': 68,
'ђ': 69,
'ѓ': 58,
'є': 60,
'ѕ': 57,
'і': 31,
'ї': 33,
'ј': 33,
'љ': 99,
'њ': 99,
'ћ': 72,
'ќ': 66,
'ѝ': 71,
'ў': 65,
'џ': 72,
'Ѡ': 103,
'ѡ': 92,
'Ѣ': 85,
'ѣ': 74,
'Ѥ': 104,
'ѥ': 82,
'Ѧ': 97,
'ѧ': 86,
'Ѩ': 128,
'ѩ': 110,
'Ѫ': 87,
'ѫ': 67,
'Ѭ': 113,
'ѭ': 91,
'Ѯ': 70,
'ѯ': 59,
'Ѱ': 94,
'ѱ': 96,
'Ѳ': 87,
'ѳ': 67,
'Ѵ': 86,
'ѵ': 73,
'Ѷ': 86,
'ѷ': 73,
'Ѹ': 109,
'ѹ': 99,
'Ѻ': 105,
'ѻ': 83,
'Ѽ': 130,
'ѽ': 113,
'Ѿ': 103,
'ѿ': 92,
'Ҁ': 77,
'ҁ': 60,
'҂': 55,
'҃': 58,
'҄': 42,
'҅': 32,
'҆': 32,
'҇': 87,
'҈': 159,
'҉': 152,
'Ҋ': 85,
'ҋ': 74,
'Ҍ': 75,
'ҍ': 65,
'Ҏ': 68,
'ҏ': 70,
'Ґ': 67,
'ґ': 58,
'Ғ': 74,
'ғ': 65,
'Ҕ': 69,
'ҕ': 58,
'Җ': 118,
'җ': 99,
'Ҙ': 71,
'ҙ': 58,
'Қ': 79,
'қ': 66,
'Ҝ': 78,
'ҝ': 66,
'Ҟ': 78,
'ҟ': 66,
'Ҡ': 94,
'ҡ': 91,
'Ң': 83,
'ң': 73,
'Ҥ': 112,
'ҥ': 96,
'Ҧ': 119,
'ҧ': 101,
'Ҩ': 97,
'ҩ': 76,
'Ҫ': 77,
'ҫ': 60,
'Ҭ': 69,
'ҭ': 64,
'Ү': 69,
'ү': 65,
'Ұ': 69,
'ұ': 65,
'Ҳ': 75,
'ҳ': 65,
'Ҵ': 104,
'ҵ': 89,
'Ҷ': 76,
'ҷ': 65,
'Ҹ': 75,
'ҹ': 65,
'Һ': 75,
'һ': 70,
'Ҽ': 104,
'ҽ': 80,
'Ҿ': 104,
'ҿ': 80,
'Ӏ': 32,
'Ӂ': 118,
'ӂ': 99,
'Ӄ': 72,
'ӄ': 66,
'Ӆ': 85,
'ӆ': 74,
'Ӈ': 83,
'ӈ': 73,
'Ӊ': 85,
'ӊ': 75,
'Ӌ': 75,
'ӌ': 65,
'Ӎ': 98,
'ӎ': 85,
'ӏ': 31,
'Ӑ': 75,
'ӑ': 67,
'Ӓ': 75,
'ӓ': 67,
'Ӕ': 107,
'ӕ': 108,
'Ӗ': 70,
'ӗ': 68,
'Ә': 87,
'ә': 68,
'Ӛ': 87,
'ӛ': 68,
'Ӝ': 118,
'ӝ': 99,
'Ӟ': 71,
'ӟ': 58,
'Ӡ': 73,
'ӡ': 64,
'Ӣ': 82,
'ӣ': 71,
'Ӥ': 82,
'ӥ': 71,
'Ӧ': 87,
'ӧ': 67,
'Ө': 87,
'ө': 67,
'Ӫ': 87,
'ӫ': 67,
'Ӭ': 77,
'ӭ': 60,
'Ӯ': 67,
'ӯ': 65,
'Ӱ': 67,
'ӱ': 65,
'Ӳ': 67,
'ӳ': 65,
'Ӵ': 75,
'ӵ': 65,
'Ӷ': 67,
'ӷ': 58,
'Ӹ': 97,
'ӹ': 87,
'Ӻ': 74,
'ӻ': 65,
'Ӽ': 75,
'ӽ': 65,
'Ӿ': 75,
'ӿ': 65,
'Ԁ': 75,
'ԁ': 65,
'Ԃ': 111,
'ԃ': 99,
'Ԅ': 107,
'ԅ': 96,
'Ԇ': 75,
'ԇ': 65,
'Ԉ': 118,
'ԉ': 105,
'Ԋ': 122,
'ԋ': 106,
'Ԍ': 85,
'ԍ': 73,
'Ԏ': 86,
'ԏ': 78,
'Ԑ': 68,
'ԑ': 59,
'Ԓ': 83,
'ԓ': 70,
'Ԕ': 129,
'ԕ': 109,
'Ԗ': 98,
'ԗ': 95,
'Ԙ': 113,
'ԙ': 108,
'Ԛ': 87,
'ԛ': 70,
'Ԝ': 109,
'ԝ': 90,
'Ԟ': 78,
'ԟ': 66,
'Ԡ': 119,
'ԡ': 100,
'Ԣ': 119,
'ԣ': 100,
'Ԥ': 87,
'ԥ': 75,
'Ա': 84,
'Բ': 81,
'Գ': 83,
'Դ': 83,
'Ե': 81,
'Զ': 85,
'Է': 70,
'Ը': 81,
'Թ': 95,
'Ժ': 83,
'Ի': 76,
'Լ': 59,
'Խ': 101,
'Ծ': 95,
'Կ': 81,
'Հ': 79,
'Ձ': 84,
'Ղ': 83,
'Ճ': 84,
'Մ': 87,
'Յ': 80,
'Ն': 80,
'Շ': 83,
'Ո': 81,
'Չ': 78,
'Պ': 88,
'Ջ': 84,
'Ռ': 87,
'Ս': 81,
'Վ': 83,
'Տ': 78,
'Ր': 76,
'Ց': 82,
'Ւ': 60,
'Փ': 89,
'Ք': 83,
'Օ': 87,
'Ֆ': 87,
'ՙ': 34,
'՚': 35,
'՛': 26,
'՜': 40,
'՝': 27,
'՞': 45,
'՟': 55,
'ա': 107,
'բ': 70,
'գ': 72,
'դ': 73,
'ե': 70,
'զ': 72,
'է': 57,
'ը': 70,
'թ': 82,
'ժ': 72,
'ի': 70,
'լ': 34,
'խ': 108,
'ծ': 68,
'կ': 70,
'հ': 70,
'ձ': 67,
'ղ': 72,
'ճ': 69,
'մ': 72,
'յ': 33,
'ն': 73,
'շ': 55,
'ո': 70,
'չ': 44,
'պ': 107,
'ջ': 62,
'ռ': 71,
'ս': 70,
'վ': 72,
'տ': 107,
'ր': 70,
'ց': 70,
'ւ': 50,
'փ': 107,
'ք': 70,
'օ': 67,
'ֆ': 89,
'և': 90,
'։': 37,
'֊': 40,
'ְ': 40,
'ֱ': 49,
'ֲ': 50,
'ֳ': 50,
'ִ': 40,
'ֵ': 46,
'ֶ': 46,
'ַ': 52,
'ָ': 52,
'ֹ': 9,
'ֺ': 9,
'ֻ': 52,
'ּ': 40,
'ֽ': 40,
'־': 40,
'ֿ': 52,
'׀': 32,
'ׁ': 79,
'ׂ': 19,
'׃': 32,
'׆': 49,
'ׇ': 52,
'א': 74,
'ב': 64,
'ג': 45,
'ד': 60,
'ה': 72,
'ו': 30,
'ז': 38,
'ח': 72,
'ט': 71,
'י': 25,
'ך': 59,
'כ': 58,
'ל': 63,
'ם': 73,
'מ': 75,
'ן': 30,
'נ': 44,
'ס': 71,
'ע': 69,
'ף': 70,
'פ': 69,
'ץ': 59,
'צ': 65,
'ק': 78,
'ר': 62,
'ש': 78,
'ת': 72,
'װ': 52,
'ױ': 47,
'ײ': 36,
'׳': 46,
'״': 71,
'؆': 70,
'؇': 70,
'؉': 83,
'؊': 107,
'،': 36,
'ؕ': 42,
'؛': 35,
'؟': 58,
'ء': 52,
'آ': 40,
'أ': 31,
'ؤ': 58,
'إ': 31,
'ئ': 86,
'ا': 31,
'ب': 104,
'ة': 58,
'ت': 104,
'ث': 104,
'ج': 71,
'ح': 71,
'خ': 71,
'د': 49,
'ذ': 49,
'ر': 58,
'ز': 58,
'س': 134,
'ش': 134,
'ص': 133,
'ض': 133,
'ط': 102,
'ظ': 102,
'ع': 66,
'غ': 66,
'ـ': 36,
'ف': 114,
'ق': 85,
'ك': 91,
'ل': 80,
'م': 68,
'ن': 81,
'ه': 58,
'و': 58,
'ى': 86,
'ي': 86,
'ً': 44,
'ٌ': 44,
'ٍ': 44,
'َ': 44,
'ُ': 44,
'ِ': 44,
'ّ': 45,
'ْ': 43,
'ٓ': 47,
'ٔ': 37,
'ٕ': 37,
'ٗ': 44,
'ٚ': 55,
'٠': 59,
'١': 59,
'٢': 59,
'٣': 59,
'٤': 59,
'٥': 59,
'٦': 59,
'٧': 59,
'٨': 59,
'٩': 59,
'٪': 59,
'٫': 36,
'٬': 35,
'٭': 60,
'ٮ': 104,
'ٯ': 85,
'ٰ': 31,
'ٴ': 32,
'ٹ': 104,
'ٺ': 104,
'ٻ': 104,
'ټ': 104,
'ٽ': 104,
'پ': 104,
'ٿ': 104,
'ڀ': 104,
'ځ': 71,
'ڂ': 71,
'ڃ': 71,
'ڄ': 71,
'څ': 71,
'چ': 71,
'ڇ': 71,
'ڈ': 49,
'ډ': 49,
'ڊ': 49,
'ڋ': 49,
'ڌ': 49,
'ڍ': 49,
'ڎ': 49,
'ڏ': 49,
'ڐ': 49,
'ڑ': 58,
'ڒ': 58,
'ړ': 61,
'ڔ': 63,
'ڕ': 75,
'ږ': 63,
'ڗ': 58,
'ژ': 58,
'ڙ': 58,
'ښ': 134,
'ڛ': 134,
'ڜ': 134,
'ڝ': 133,
'ڞ': 133,
'ڟ': 102,
'ڠ': 66,
'ڡ': 114,
'ڢ': 114,
'ڣ': 114,
'ڤ': 114,
'ڥ': 114,
'ڦ': 114,
'ڧ': 85,
'ڨ': 85,
'ک': 99,
'ڪ': 116,
'ګ': 99,
'ڬ': 91,
'ڭ': 91,
'ڮ': 91,
'گ': 99,
'ڰ': 99,
'ڱ': 99,
'ڲ': 99,
'ڳ': 99,
'ڴ': 99,
'ڵ': 80,
'ڶ': 80,
'ڷ': 80,
'ڸ': 80,
'ڹ': 81,
'ں': 81,
'ڻ': 81,
'ڼ': 81,
'ڽ': 81,
'ھ': 77,
'ڿ': 71,
'ۆ': 58,
'ۇ': 58,
'ۈ': 58,
'ۋ': 58,
'ی': 86,
'ێ': 86,
'ې': 86,
'ە': 58,
'۰': 59,
'۱': 59,
'۲': 59,
'۳': 59,
'۴': 59,
'۵': 59,
'۶': 59,
'۷': 59,
'۸': 59,
'۹': 59,
'߀': 70,
'߁': 70,
'߂': 70,
'߃': 70,
'߄': 70,
'߅': 70,
'߆': 70,
'߇': 70,
'߈': 70,
'߉': 70,
'ߊ': 31,
'ߋ': 63,
'ߌ': 47,
'ߍ': 65,
'ߎ': 72,
'ߏ': 72,
'ߐ': 65,
'ߑ': 72,
'ߒ': 91,
'ߓ': 48,
'ߔ': 48,
'ߕ': 61,
'ߖ': 67,
'ߗ': 39,
'ߘ': 105,
'ߙ': 52,
'ߚ': 86,
'ߛ': 72,
'ߜ': 69,
'ߝ': 81,
'ߞ': 58,
'ߟ': 80,
'ߠ': 52,
'ߡ': 69,
'ߢ': 65,
'ߣ': 58,
'ߤ': 58,
'ߥ': 57,
'ߦ': 65,
'ߧ': 65,
'߫': 44,
'߬': 52,
'߭': 34,
'߮': 45,
'߯': 51,
'߰': 52,
'߱': 51,
'߲': 33,
'߳': 44,
'ߴ': 34,
'ߵ': 34,
'߸': 62,
'߹': 62,
'ߺ': 43,
'฿': 70,
'ກ': 74,
'ຂ': 76,
'ຄ': 76,
'ງ': 53,
'ຈ': 69,
'ຊ': 76,
'ຍ': 76,
'ດ': 74,
'ຕ': 71,
'ຖ': 72,
'ທ': 72,
'ນ': 72,
'ບ': 69,
'ປ': 69,
'ຜ': 82,
'ຝ': 84,
'ພ': 76,
'ຟ': 76,
'ມ': 77,
'ຢ': 76,
'ຣ': 75,
'ລ': 71,
'ວ': 70,
'ສ': 77,
'ຫ': 90,
'ອ': 70,
'ຮ': 76,
'ຯ': 87,
'ະ': 70,
'ັ': 64,
'າ': 59,
'ຳ': 106,
'ິ': 66,
'ີ': 66,
'ຶ': 66,
'ື': 66,
'ຸ': 42,
'ູ': 47,
'ົ': 64,
'ຼ': 69,
'ຽ': 73,
'ເ': 41,
'ແ': 72,
'ໂ': 54,
'ໃ': 60,
'ໄ': 54,
'ໆ': 74,
'່': 41,
'້': 62,
'໊': 69,
'໋': 51,
'໌': 69,
'ໍ': 47,
'໐': 70,
'໑': 70,
'໒': 70,
'໓': 77,
'໔': 69,
'໕': 69,
'໖': 82,
'໗': 74,
'໘': 75,
'໙': 74,
'ໜ': 113,
'ໝ': 113,
'Ⴀ': 96,
'Ⴁ': 81,
'Ⴂ': 75,
'Ⴃ': 92,
'Ⴄ': 68,
'Ⴅ': 84,
'Ⴆ': 83,
'Ⴇ': 101,
'Ⴈ': 50,
'Ⴉ': 68,
'Ⴊ': 93,
'Ⴋ': 97,
'Ⴌ': 69,
'Ⴍ': 94,
'Ⴎ': 86,
'Ⴏ': 69,
'Ⴐ': 100,
'Ⴑ': 68,
'Ⴒ': 68,
'Ⴓ': 94,
'Ⴔ': 95,
'Ⴕ': 80,
'Ⴖ': 69,
'Ⴗ': 68,
'Ⴘ': 69,
'Ⴙ': 68,
'Ⴚ': 90,
'Ⴛ': 96,
'Ⴜ': 68,
'Ⴝ': 69,
'Ⴞ': 69,
'Ⴟ': 80,
'Ⴠ': 93,
'Ⴡ': 66,
'Ⴢ': 76,
'Ⴣ': 66,
'Ⴤ': 65,
'Ⴥ': 81,
'ა': 56,
'ბ': 57,
'გ': 64,
'დ': 90,
'ე': 56,
'ვ': 56,
'ზ': 55,
'თ': 88,
'ი': 57,
'კ': 56,
'ლ': 117,
'მ': 57,
'ნ': 57,
'ო': 86,
'პ': 56,
'ჟ': 57,
'რ': 88,
'ს': 57,
'ტ': 72,
'უ': 57,
'ფ': 91,
'ქ': 56,
'ღ': 86,
'ყ': 57,
'შ': 57,
'ჩ': 57,
'ც': 63,
'ძ': 57,
'წ': 57,
'ჭ': 57,
'ხ': 57,
'ჯ': 50,
'ჰ': 56,
'ჱ': 57,
'ჲ': 56,
'ჳ': 56,
'ჴ': 57,
'ჵ': 61,
'ჶ': 91,
'ჷ': 61,
'ჸ': 56,
'ჹ': 63,
'ჺ': 56,
'჻': 49,
'ჼ': 36,
'ᐁ': 75,
'ᐂ': 75,
'ᐃ': 75,
'ᐄ': 75,
'ᐅ': 85,
'ᐆ': 85,
'ᐇ': 85,
'ᐉ': 85,
'ᐊ': 85,
'ᐋ': 85,
'ᐌ': 92,
'ᐍ': 92,
'ᐎ': 92,
'ᐏ': 92,
'ᐐ': 92,
'ᐑ': 92,
'ᐒ': 106,
'ᐓ': 111,
'ᐔ': 106,
'ᐕ': 111,
'ᐖ': 85,
'ᐗ': 106,
'ᐘ': 111,
'ᐙ': 106,
'ᐚ': 111,
'ᐛ': 85,
'ᐝ': 28,
'ᐞ': 60,
'ᐟ': 47,
'ᐠ': 47,
'ᐡ': 43,
'ᐢ': 43,
'ᐣ': 43,
'ᐤ': 43,
'ᐥ': 51,
'ᐦ': 42,
'ᐧ': 28,
'ᐨ': 43,
'ᐩ': 43,
'ᐪ': 43,
'ᐫ': 120,
'ᐬ': 100,
'ᐭ': 105,
'ᐮ': 123,
'ᐯ': 75,
'ᐰ': 75,
'ᐱ': 75,
'ᐲ': 75,
'ᐳ': 80,
'ᐴ': 80,
'ᐵ': 80,
'ᐷ': 80,
'ᐸ': 80,
'ᐹ': 80,
'ᐺ': 92,
'ᐻ': 86,
'ᐼ': 92,
'ᐽ': 92,
'ᐾ': 92,
'ᐿ': 92,
'ᑀ': 106,
'ᑁ': 111,
'ᑂ': 106,
'ᑃ': 111,
'ᑄ': 106,
'ᑅ': 111,
'ᑆ': 106,
'ᑇ': 111,
'ᑈ': 80,
'ᑉ': 56,
'ᑊ': 21,
'ᑌ': 81,
'ᑍ': 81,
'ᑎ': 81,
'ᑏ': 81,
'ᑐ': 80,
'ᑑ': 80,
'ᑒ': 80,
'ᑔ': 80,
'ᑕ': 80,
'ᑖ': 80,
'ᑗ': 101,
'ᑘ': 98,
'ᑙ': 101,
'ᑚ': 98,
'ᑛ': 101,
'ᑜ': 98,
'ᑝ': 102,
'ᑞ': 99,
'ᑟ': 102,
'ᑠ': 99,
'ᑡ': 104,
'ᑢ': 99,
'ᑣ': 104,
'ᑤ': 99,
'ᑥ': 104,
'ᑦ': 48,
'ᑧ': 97,
'ᑨ': 97,
'ᑩ': 95,
'ᑪ': 98,
'ᑫ': 69,
'ᑬ': 69,
'ᑭ': 69,
'ᑮ': 69,
'ᑯ': 69,
'ᑰ': 69,
'ᑱ': 69,
'ᑲ': 69,
'ᑳ': 69,
'ᑴ': 95,
'ᑵ': 85,
'ᑶ': 90,
'ᑷ': 90,
'ᑸ': 90,
'ᑹ': 90,
'ᑺ': 95,
'ᑻ': 85,
'ᑼ': 95,
'ᑽ': 85,
'ᑾ': 90,
'ᑿ': 90,
'ᒀ': 90,
'ᒁ': 90,
'ᒂ': 90,
'ᒃ': 45,
'ᒄ': 45,
'ᒅ': 83,
'ᒆ': 85,
'ᒇ': 83,
'ᒈ': 85,
'ᒉ': 69,
'ᒊ': 69,
'ᒋ': 69,
'ᒌ': 69,
'ᒍ': 69,
'ᒎ': 69,
'ᒏ': 69,
'ᒐ': 69,
'ᒑ': 69,
'ᒒ': 95,
'ᒓ': 85,
'ᒔ': 90,
'ᒕ': 90,
'ᒖ': 90,
'ᒗ': 90,
'ᒘ': 95,
'ᒙ': 85,
'ᒚ': 95,
'ᒛ': 85,
'ᒜ': 90,
'ᒝ': 90,
'ᒞ': 90,
'ᒟ': 90,
'ᒠ': 90,
'ᒡ': 48,
'ᒢ': 48,
'ᒣ': 67,
'ᒤ': 61,
'ᒥ': 61,
'ᒦ': 61,
'ᒧ': 67,
'ᒨ': 67,
'ᒩ': 67,
'ᒪ': 61,
'ᒫ': 61,
'ᒬ': 82,
'ᒭ': 85,
'ᒮ': 82,
'ᒯ': 84,
'ᒰ': 82,
'ᒱ': 84,
'ᒲ': 82,
'ᒳ': 85,
'ᒴ': 82,
'ᒵ': 85,
'ᒶ': 82,
'ᒷ': 84,
'ᒸ': 82,
'ᒹ': 84,
'ᒺ': 82,
'ᒻ': 42,
'ᒼ': 56,
'ᒽ': 42,
'ᓀ': 94,
'ᓁ': 94,
'ᓂ': 94,
'ᓃ': 94,
'ᓄ': 94,
'ᓅ': 94,
'ᓆ': 94,
'ᓇ': 94,
'ᓈ': 94,
'ᓉ': 118,
'ᓊ': 114,
'ᓋ': 117,
'ᓌ': 94,
'ᓍ': 117,
'ᓎ': 94,
'ᓏ': 94,
'ᓐ': 66,
'ᓑ': 50,
'ᓒ': 66,
'ᓓ': 94,
'ᓔ': 94,
'ᓕ': 94,
'ᓖ': 94,
'ᓗ': 94,
'ᓘ': 94,
'ᓙ': 94,
'ᓚ': 94,
'ᓛ': 94,
'ᓜ': 118,
'ᓝ': 114,
'ᓞ': 117,
'ᓟ': 113,
'ᓠ': 117,
'ᓡ': 113,
'ᓢ': 118,
'ᓣ': 114,
'ᓤ': 118,
'ᓥ': 114,
'ᓦ': 119,
'ᓧ': 113,
'ᓨ': 119,
'ᓩ': 113,
'ᓪ': 66,
'ᓬ': 80,
'ᓭ': 66,
'ᓮ': 72,
'ᓯ': 66,
'ᓰ': 66,
'ᓱ': 66,
'ᓲ': 66,
'ᓳ': 66,
'ᓴ': 66,
'ᓵ': 66,
'ᓶ': 92,
'ᓷ': 83,
'ᓸ': 87,
'ᓹ': 85,
'ᓺ': 87,
'ᓻ': 85,
'ᓼ': 92,
'ᓽ': 83,
'ᓾ': 92,
'ᓿ': 83,
'ᔀ': 87,
'ᔁ': 85,
'ᔂ': 87,
'ᔃ': 85,
'ᔄ': 87,
'ᔅ': 46,
'ᔆ': 46,
'ᔇ': 46,
'ᔐ': 78,
'ᔑ': 78,
'ᔒ': 78,
'ᔓ': 98,
'ᔔ': 98,
'ᔕ': 98,
'ᔖ': 98,
'ᔗ': 100,
'ᔘ': 96,
'ᔙ': 100,
'ᔚ': 96,
'ᔛ': 100,
'ᔜ': 96,
'ᔝ': 125,
'ᔞ': 121,
'ᔟ': 125,
'ᔠ': 121,
'ᔡ': 125,
'ᔢ': 121,
'ᔣ': 125,
'ᔤ': 121,
'ᔥ': 70,
'ᔦ': 69,
'ᔧ': 69,
'ᔨ': 69,
'ᔩ': 69,
'ᔪ': 69,
'ᔫ': 69,
'ᔬ': 69,
'ᔭ': 69,
'ᔮ': 69,
'ᔯ': 93,
'ᔰ': 86,
'ᔱ': 90,
'ᔲ': 90,
'ᔳ': 90,
'ᔴ': 90,
'ᔵ': 93,
'ᔶ': 86,
'ᔷ': 93,
'ᔸ': 86,
'ᔹ': 90,
'ᔺ': 90,
'ᔻ': 90,
'ᔼ': 90,
'ᔽ': 90,
'ᔾ': 46,
'ᕀ': 43,
'ᕁ': 53,
'ᕂ': 101,
'ᕃ': 101,
'ᕄ': 101,
'ᕅ': 101,
'ᕆ': 101,
'ᕇ': 101,
'ᕈ': 66,
'ᕉ': 66,
'ᕊ': 66,
'ᕋ': 66,
'ᕌ': 66,
'ᕍ': 66,
'ᕎ': 92,
'ᕏ': 83,
'ᕐ': 46,
'ᕒ': 80,
'ᕓ': 75,
'ᕔ': 75,
'ᕕ': 75,
'ᕖ': 75,
'ᕗ': 80,
'ᕘ': 80,
'ᕙ': 80,
'ᕚ': 80,
'ᕛ': 102,
'ᕜ': 111,
'ᕝ': 56,
'ᕞ': 81,
'ᕟ': 81,
'ᕠ': 81,
'ᕡ': 81,
'ᕢ': 81,
'ᕣ': 81,
'ᕤ': 80,
'ᕥ': 80,
'ᕦ': 80,
'ᕧ': 80,
'ᕨ': 104,
'ᕩ': 99,
'ᕪ': 56,
'ᕴ': 91,
'ᕵ': 91,
'ᕶ': 91,
'ᕷ': 91,
'ᕸ': 91,
'ᕹ': 91,
'ᕺ': 91,
'ᕻ': 62,
'ᕼ': 83,
'ᕽ': 53,
'ᕾ': 115,
'ᕿ': 115,
'ᖀ': 115,
'ᖁ': 115,
'ᖂ': 115,
'ᖃ': 115,
'ᖄ': 115,
'ᖅ': 91,
'ᖊ': 91,
'ᖋ': 91,
'ᖌ': 91,
'ᖍ': 91,
'ᖎ': 139,
'ᖏ': 139,
'ᖐ': 139,
'ᖑ': 110,
'ᖒ': 110,
'ᖓ': 139,
'ᖔ': 139,
'ᖕ': 77,
'ᖖ': 118,
'ᖠ': 94,
'ᖡ': 94,
'ᖢ': 94,
'ᖣ': 94,
'ᖤ': 94,
'ᖥ': 94,
'ᖦ': 66,
'ᖧ': 71,
'ᖨ': 71,
'ᖩ': 71,
'ᖪ': 71,
'ᖫ': 71,
'ᖬ': 71,
'ᖭ': 71,
'ᖮ': 46,
'ᖯ': 69,
'ᗞ': 85,
'ᗡ': 84,
'ᙆ': 52,
'ᙇ': 52,
'᙮': 49,
'ᙯ': 115,
'ᙰ': 144,
'ᙱ': 180,
'ᙲ': 180,
'ᙳ': 151,
'ᙴ': 151,
'ᙵ': 180,
'ᙶ': 180,
'\u1680': 56,
'ᚁ': 58,
'ᚂ': 82,
'ᚃ': 106,
'ᚄ': 130,
'ᚅ': 154,
'ᚆ': 58,
'ᚇ': 82,
'ᚈ': 106,
'ᚉ': 130,
'ᚊ': 154,
'ᚋ': 58,
'ᚌ': 83,
'ᚍ': 107,
'ᚎ': 131,
'ᚏ': 155,
'ᚐ': 58,
'ᚑ': 82,
'ᚒ': 106,
'ᚓ': 130,
'ᚔ': 154,
'ᚕ': 58,
'ᚖ': 86,
'ᚗ': 90,
'ᚘ': 136,
'ᚙ': 130,
'ᚚ': 79,
'᚛': 57,
'᚜': 58,
'ᴀ': 65,
'ᴁ': 79,
'ᴂ': 108,
'ᴃ': 64,
'ᴄ': 60,
'ᴅ': 67,
'ᴆ': 67,
'ᴇ': 54,
'ᴈ': 59,
'ᴉ': 31,
'ᴊ': 43,
'ᴋ': 64,
'ᴌ': 64,
'ᴍ': 83,
'ᴎ': 71,
'ᴏ': 67,
'ᴐ': 60,
'ᴑ': 75,
'ᴒ': 75,
'ᴓ': 75,
'ᴔ': 113,
'ᴖ': 67,
'ᴗ': 67,
'ᴘ': 58,
'ᴙ': 66,
'ᴚ': 66,
'ᴛ': 64,
'ᴜ': 63,
'ᴝ': 81,
'ᴞ': 104,
'ᴟ': 70,
'ᴠ': 65,
'ᴡ': 90,
'ᴢ': 58,
'ᴣ': 58,
'ᴦ': 64,
'ᴧ': 65,
'ᴨ': 62,
'ᴩ': 58,
'ᴪ': 65,
'ᴫ': 70,
'ᴬ': 47,
'ᴭ': 67,
'ᴮ': 48,
'ᴰ': 53,
'ᴱ': 44,
'ᴲ': 44,
'ᴳ': 54,
'ᴴ': 52,
'ᴵ': 20,
'ᴶ': 24,
'ᴷ': 47,
'ᴸ': 39,
'ᴹ': 60,
'ᴺ': 52,
'ᴻ': 52,
'ᴼ': 55,
'ᴽ': 48,
'ᴾ': 42,
'ᴿ': 48,
'ᵀ': 44,
'ᵁ': 51,
'ᵂ': 68,
'ᵃ': 43,
'ᵄ': 43,
'ᵅ': 45,
'ᵆ': 71,
'ᵇ': 47,
'ᵈ': 45,
'ᵉ': 46,
'ᵊ': 46,
'ᵋ': 40,
'ᵌ': 40,
'ᵍ': 45,
'ᵎ': 20,
'ᵏ': 47,
'ᵐ': 69,
'ᵑ': 45,
'ᵒ': 46,
'ᵓ': 41,
'ᵔ': 46,
'ᵕ': 46,
'ᵖ': 47,
'ᵗ': 32,
'ᵘ': 45,
'ᵙ': 52,
'ᵚ': 69,
'ᵛ': 46,
'ᵝ': 44,
'ᵞ': 41,
'ᵟ': 42,
'ᵠ': 46,
'ᵡ': 40,
'ᵢ': 20,
'ᵣ': 29,
'ᵤ': 45,
'ᵥ': 46,
'ᵦ': 44,
'ᵧ': 41,
'ᵨ': 45,
'ᵩ': 46,
'ᵪ': 40,
'ᵷ': 70,
'ᵸ': 52,
'ᵻ': 41,
'ᵽ': 73,
'ᶅ': 31,
'ᶛ': 45,
'ᶜ': 41,
'ᶝ': 41,
'ᶞ': 46,
'ᶟ': 40,
'ᶠ': 33,
'ᶡ': 28,
'ᶢ': 45,
'ᶣ': 45,
'ᶤ': 29,
'ᶥ': 27,
'ᶦ': 29,
'ᶧ': 29,
'ᶨ': 36,
'ᶩ': 27,
'ᶪ': 26,
'ᶫ': 41,
'ᶬ': 69,
'ᶭ': 69,
'ᶮ': 47,
'ᶯ': 53,
'ᶰ': 45,
'ᶱ': 46,
'ᶲ': 46,
'ᶳ': 40,
'ᶴ': 34,
'ᶵ': 32,
'ᶶ': 56,
'ᶷ': 46,
'ᶸ': 40,
'ᶹ': 45,
'ᶺ': 46,
'ᶻ': 40,
'ᶼ': 48,
'ᶽ': 40,
'ᶾ': 43,
'ᶿ': 46,
'᷄': 51,
'᷅': 51,
'᷆': 51,
'᷇': 51,
'᷈': 52,
'᷉': 52,
'Ḁ': 75,
'ḁ': 67,
'Ḃ': 75,
'ḃ': 70,
'Ḅ': 75,
'ḅ': 70,
'Ḇ': 75,
'ḇ': 70,
'Ḉ': 77,
'ḉ': 60,
'Ḋ': 85,
'ḋ': 70,
'Ḍ': 85,
'ḍ': 70,
'Ḏ': 85,
'ḏ': 70,
'Ḑ': 85,
'ḑ': 70,
'Ḓ': 85,
'ḓ': 70,
'Ḕ': 70,
'ḕ': 68,
'Ḗ': 70,
'ḗ': 68,
'Ḙ': 70,
'ḙ': 68,
'Ḛ': 70,
'ḛ': 68,
'Ḝ': 70,
'ḝ': 68,
'Ḟ': 63,
'ḟ': 41,
'Ḡ': 85,
'ḡ': 70,
'Ḣ': 83,
'ḣ': 70,
'Ḥ': 83,
'ḥ': 70,
'Ḧ': 83,
'ḧ': 72,
'Ḩ': 83,
'ḩ': 70,
'Ḫ': 83,
'ḫ': 70,
'Ḭ': 36,
'ḭ': 37,
'Ḯ': 33,
'ḯ': 33,
'Ḱ': 75,
'ḱ': 64,
'Ḳ': 75,
'ḳ': 64,
'Ḵ': 75,
'ḵ': 64,
'Ḷ': 61,
'ḷ': 32,
'Ḹ': 61,
'ḹ': 33,
'Ḻ': 61,
'ḻ': 33,
'Ḽ': 61,
'ḽ': 35,
'Ḿ': 95,
'ḿ': 107,
'Ṁ': 95,
'ṁ': 107,
'Ṃ': 95,
'ṃ': 107,
'Ṅ': 82,
'ṅ': 70,
'Ṇ': 82,
'ṇ': 70,
'Ṉ': 82,
'ṉ': 70,
'Ṋ': 82,
'ṋ': 70,
'Ṍ': 87,
'ṍ': 67,
'Ṏ': 87,
'ṏ': 67,
'Ṑ': 87,
'ṑ': 67,
'Ṓ': 87,
'ṓ': 67,
'Ṕ': 66,
'ṕ': 70,
'Ṗ': 66,
'ṗ': 70,
'Ṙ': 76,
'ṙ': 46,
'Ṛ': 76,
'ṛ': 46,
'Ṝ': 76,
'ṝ': 46,
'Ṟ': 76,
'ṟ': 46,
'Ṡ': 70,
'ṡ': 57,
'Ṣ': 70,
'ṣ': 57,
'Ṥ': 70,
'ṥ': 57,
'Ṧ': 70,
'ṧ': 57,
'Ṩ': 70,
'ṩ': 57,
'Ṫ': 69,
'ṫ': 43,
'Ṭ': 69,
'ṭ': 43,
'Ṯ': 69,
'ṯ': 43,
'Ṱ': 69,
'ṱ': 44,
'Ṳ': 81,
'ṳ': 70,
'Ṵ': 81,
'ṵ': 70,
'Ṷ': 81,
'ṷ': 70,
'Ṹ': 81,
'ṹ': 70,
'Ṻ': 81,
'ṻ': 70,
'Ṽ': 75,
'ṽ': 65,
'Ṿ': 75,
'ṿ': 65,
'Ẁ': 109,
'ẁ': 90,
'Ẃ': 109,
'ẃ': 90,
'Ẅ': 109,
'ẅ': 90,
'Ẇ': 109,
'ẇ': 90,
'Ẉ': 109,
'ẉ': 90,
'Ẋ': 75,
'ẋ': 65,
'Ẍ': 75,
'ẍ': 65,
'Ẏ': 69,
'ẏ': 65,
'Ẑ': 75,
'ẑ': 58,
'Ẓ': 75,
'ẓ': 58,
'Ẕ': 75,
'ẕ': 58,
'ẖ': 70,
'ẗ': 43,
'ẘ': 90,
'ẙ': 65,
'ẚ': 74,
'ẛ': 41,
'ẜ': 41,
'ẝ': 41,
'ẞ': 85,
'ẟ': 67,
'Ạ': 75,
'ạ': 67,
'Ả': 75,
'ả': 67,
'Ấ': 75,
'ấ': 67,
'Ầ': 75,
'ầ': 67,
'Ẩ': 75,
'ẩ': 67,
'Ẫ': 75,
'ẫ': 67,
'Ậ': 75,
'ậ': 67,
'Ắ': 75,
'ắ': 67,
'Ằ': 75,
'ằ': 67,
'Ẳ': 75,
'ẳ': 67,
'Ẵ': 75,
'ẵ': 67,
'Ặ': 75,
'ặ': 67,
'Ẹ': 70,
'ẹ': 68,
'Ẻ': 70,
'ẻ': 68,
'Ẽ': 70,
'ẽ': 68,
'Ế': 71,
'ế': 68,
'Ề': 70,
'ề': 68,
'Ể': 70,
'ể': 68,
'Ễ': 70,
'ễ': 68,
'Ệ': 70,
'ệ': 68,
'Ỉ': 32,
'ỉ': 31,
'Ị': 32,
'ị': 31,
'Ọ': 87,
'ọ': 67,
'Ỏ': 87,
'ỏ': 67,
'Ố': 87,
'ố': 67,
'Ồ': 87,
'ồ': 67,
'Ổ': 87,
'ổ': 67,
'Ỗ': 87,
'ỗ': 67,
'Ộ': 87,
'ộ': 67,
'Ớ': 100,
'ớ': 67,
'Ờ': 100,
'ờ': 67,
'Ở': 100,
'ở': 67,
'Ỡ': 100,
'ỡ': 67,
'Ợ': 100,
'ợ': 67,
'Ụ': 81,
'ụ': 70,
'Ủ': 81,
'ủ': 70,
'Ứ': 94,
'ứ': 75,
'Ừ': 94,
'ừ': 75,
'Ử': 94,
'ử': 75,
'Ữ': 94,
'ữ': 75,
'Ự': 94,
'ự': 75,
'Ỳ': 69,
'ỳ': 65,
'Ỵ': 69,
'ỵ': 65,
'Ỷ': 69,
'ỷ': 65,
'Ỹ': 69,
'ỹ': 65,
'Ỻ': 85,
'ỻ': 52,
'ἀ': 73,
'ἁ': 73,
'ἂ': 73,
'ἃ': 73,
'ἄ': 73,
'ἅ': 73,
'ἆ': 73,
'ἇ': 73,
'Ἀ': 75,
'Ἁ': 75,
'Ἂ': 97,
'Ἃ': 97,
'Ἄ': 85,
'Ἅ': 88,
'Ἆ': 78,
'Ἇ': 82,
'ἐ': 59,
'ἑ': 59,
'ἒ': 59,
'ἓ': 59,
'ἔ': 59,
'ἕ': 59,
'Ἐ': 78,
'Ἑ': 78,
'Ἒ': 106,
'Ἓ': 107,
'Ἔ': 99,
'Ἕ': 102,
'ἠ': 70,
'ἡ': 70,
'ἢ': 70,
'ἣ': 70,
'ἤ': 70,
'ἥ': 70,
'ἦ': 70,
'ἧ': 70,
'Ἠ': 92,
'Ἡ': 92,
'Ἢ': 119,
'Ἣ': 120,
'Ἤ': 113,
'Ἥ': 116,
'Ἦ': 103,
'Ἧ': 104,
'ἰ': 37,
'ἱ': 37,
'ἲ': 43,
'ἳ': 43,
'ἴ': 40,
'ἵ': 44,
'ἶ': 40,
'ἷ': 41,
'Ἰ': 42,
'Ἱ': 41,
'Ἲ': 70,
'Ἳ': 70,
'Ἴ': 63,
'Ἵ': 66,
'Ἶ': 54,
'Ἷ': 54,
'ὀ': 67,
'ὁ': 67,
'ὂ': 67,
'ὃ': 67,
'ὄ': 67,
'ὅ': 67,
'Ὀ': 88,
'Ὁ': 93,
'Ὂ': 120,
'Ὃ': 121,
'Ὄ': 103,
'Ὅ': 107,
'ὐ': 64,
'ὑ': 64,
'ὒ': 64,
'ὓ': 64,
'ὔ': 64,
'ὕ': 64,
'ὖ': 64,
'ὗ': 64,
'Ὑ': 87,
'Ὓ': 110,
'Ὕ': 112,
'Ὗ': 99,
'ὠ': 92,
'ὡ': 92,
'ὢ': 92,
'ὣ': 92,
'ὤ': 92,
'ὥ': 92,
'ὦ': 92,
'ὧ': 92,
'Ὠ': 88,
'Ὡ': 93,
'Ὢ': 120,
'Ὣ': 120,
'Ὤ': 104,
'Ὥ': 107,
'Ὦ': 101,
'Ὧ': 105,
'ὰ': 73,
'ά': 73,
'ὲ': 59,
'έ': 60,
'ὴ': 70,
'ή': 72,
'ὶ': 44,
'ί': 37,
'ὸ': 67,
'ό': 67,
'ὺ': 64,
'ύ': 64,
'ὼ': 92,
'ώ': 92,
'ᾀ': 73,
'ᾁ': 73,
'ᾂ': 73,
'ᾃ': 73,
'ᾄ': 73,
'ᾅ': 73,
'ᾆ': 73,
'ᾇ': 73,
'ᾈ': 75,
'ᾉ': 75,
'ᾊ': 97,
'ᾋ': 97,
'ᾌ': 85,
'ᾍ': 88,
'ᾎ': 78,
'ᾏ': 82,
'ᾐ': 70,
'ᾑ': 70,
'ᾒ': 70,
'ᾓ': 70,
'ᾔ': 70,
'ᾕ': 70,
'ᾖ': 70,
'ᾗ': 70,
'ᾘ': 92,
'ᾙ': 92,
'ᾚ': 119,
'ᾛ': 120,
'ᾜ': 113,
'ᾝ': 116,
'ᾞ': 103,
'ᾟ': 104,
'ᾠ': 92,
'ᾡ': 92,
'ᾢ': 92,
'ᾣ': 92,
'ᾤ': 92,
'ᾥ': 92,
'ᾦ': 92,
'ᾧ': 92,
'ᾨ': 88,
'ᾩ': 93,
'ᾪ': 120,
'ᾫ': 120,
'ᾬ': 104,
'ᾭ': 107,
'ᾮ': 101,
'ᾯ': 105,
'ᾰ': 73,
'ᾱ': 73,
'ᾲ': 73,
'ᾳ': 73,
'ᾴ': 73,
'ᾶ': 73,
'ᾷ': 73,
'Ᾰ': 75,
'Ᾱ': 75,
'Ὰ': 80,
'Ά': 76,
'ᾼ': 75,
'᾽': 55,
'ι': 55,
'᾿': 55,
'῀': 55,
'῁': 55,
'ῂ': 70,
'ῃ': 70,
'ῄ': 72,
'ῆ': 70,
'ῇ': 70,
'Ὲ': 90,
'Έ': 84,
'Ὴ': 103,
'Ή': 97,
'ῌ': 83,
'῍': 55,
'῎': 55,
'῏': 55,
'ῐ': 39,
'ῑ': 39,
'ῒ': 40,
'ΐ': 37,
'ῖ': 39,
'ῗ': 39,
'Ῐ': 35,
'Ῑ': 33,
'Ὶ': 53,
'Ί': 47,
'῝': 55,
'῞': 55,
'῟': 55,
'ῠ': 64,
'ῡ': 64,
'ῢ': 64,
'ΰ': 64,
'ῤ': 70,
'ῥ': 70,
'ῦ': 64,
'ῧ': 64,
'Ῠ': 69,
'Ῡ': 69,
'Ὺ': 95,
'Ύ': 93,
'Ῥ': 75,
'῭': 55,
'΅': 55,
'`': 55,
'ῲ': 92,
'ῳ': 92,
'ῴ': 92,
'ῶ': 92,
'ῷ': 92,
'Ὸ': 105,
'Ό': 90,
'Ὼ': 102,
'Ώ': 93,
'ῼ': 84,
'´': 55,
'῾': 55,
'\u2000': 55,
'\u2001': 110,
'\u2002': 55,
'\u2003': 110,
'\u2004': 36,
'\u2005': 28,
'\u2006': 18,
'\u2007': 70,
'\u2008': 35,
'\u2009': 22,
'\u200a': 11,
'\u200b': 0,
'\u200c': 0,
'\u200d': 0,
'\u200e': 0,
'\u200f': 0,
'‐': 40,
'‑': 40,
'‒': 70,
'–': 55,
'—': 110,
'―': 110,
'‖': 55,
'‗': 59,
'‘': 35,
'’': 35,
'‚': 35,
'‛': 35,
'“': 57,
'”': 57,
'„': 57,
'‟': 57,
'†': 55,
'‡': 55,
'•': 65,
'‣': 65,
'․': 37,
'‥': 73,
'…': 110,
'‧': 35,
'\u2028': 0,
'\u2029': 0,
'\u202a': 0,
'\u202b': 0,
'\u202c': 0,
'\u202d': 0,
'\u202e': 0,
'\u202f': 22,
'‰': 148,
'‱': 191,
'′': 25,
'″': 41,
'‴': 57,
'‵': 25,
'‶': 41,
'‷': 57,
'‸': 37,
'‹': 44,
'›': 44,
'※': 92,
'‼': 53,
'‽': 58,
'‾': 59,
'‿': 99,
'⁀': 99,
'⁁': 37,
'⁂': 110,
'⁃': 55,
'⁄': 60,
'⁅': 43,
'⁆': 43,
'⁇': 101,
'⁈': 81,
'⁉': 81,
'⁊': 55,
'⁋': 70,
'⁌': 55,
'⁍': 55,
'⁎': 55,
'⁏': 37,
'⁐': 99,
'⁑': 55,
'⁒': 49,
'⁓': 110,
'⁔': 99,
'⁕': 92,
'⁖': 64,
'⁗': 73,
'⁘': 92,
'⁙': 92,
'⁚': 35,
'⁛': 88,
'⁜': 92,
'⁝': 35,
'⁞': 35,
'\u205f': 24,
'\u2060': 0,
'\u2061': 0,
'\u2062': 0,
'\u2063': 0,
'\u2064': 0,
'\u206a': 0,
'\u206b': 0,
'\u206c': 0,
'\u206d': 0,
'\u206e': 0,
'\u206f': 0,
'⁰': 44,
'ⁱ': 20,
'⁴': 44,
'⁵': 44,
'⁶': 44,
'⁷': 44,
'⁸': 44,
'⁹': 44,
'⁺': 58,
'⁻': 58,
'⁼': 58,
'⁽': 27,
'⁾': 27,
'ⁿ': 44,
'₀': 44,
'₁': 44,
'₂': 44,
'₃': 44,
'₄': 44,
'₅': 44,
'₆': 44,
'₇': 44,
'₈': 44,
'₉': 44,
'₊': 58,
'₋': 58,
'₌': 58,
'₍': 27,
'₎': 27,
'ₐ': 43,
'ₑ': 46,
'ₒ': 46,
'ₓ': 49,
'ₔ': 46,
'ₕ': 44,
'ₖ': 47,
'ₗ': 18,
'ₘ': 69,
'ₙ': 44,
'ₚ': 47,
'ₛ': 41,
'ₜ': 32,
'₠': 96,
'₡': 70,
'₢': 70,
'₣': 70,
'₤': 70,
'₥': 107,
'₦': 70,
'₧': 140,
'₨': 118,
'₩': 109,
'₪': 86,
'₫': 70,
'€': 70,
'₭': 70,
'₮': 70,
'₯': 140,
'₰': 70,
'₱': 70,
'₲': 70,
'₳': 70,
'₴': 85,
'₵': 70,
'₸': 70,
'₹': 70,
'₺': 70,
'₽': 70,
'⃐': 54,
'⃑': 52,
'⃖': 52,
'⃗': 52,
'⃛': 54,
'⃜': 75,
'⃡': 52,
'℀': 112,
'℁': 112,
'ℂ': 77,
'℃': 124,
'℄': 74,
'℅': 112,
'℆': 117,
'ℇ': 68,
'℈': 77,
'℉': 105,
'ℋ': 109,
'ℌ': 83,
'ℍ': 93,
'ℎ': 70,
'ℏ': 70,
'ℐ': 52,
'ℑ': 77,
'ℒ': 79,
'ℓ': 47,
'℔': 90,
'ℕ': 88,
'№': 114,
'℗': 110,
'℘': 77,
'ℙ': 77,
'ℚ': 87,
'ℛ': 88,
'ℜ': 90,
'ℝ': 87,
'℞': 99,
'℟': 75,
'℠': 112,
'℡': 119,
'™': 110,
'℣': 75,
'ℤ': 82,
'℥': 64,
'Ω': 84,
'℧': 84,
'ℨ': 68,
'℩': 37,
'K': 75,
'Å': 75,
'ℬ': 86,
'ℭ': 77,
'℮': 94,
'ℯ': 65,
'ℰ': 67,
'ℱ': 86,
'Ⅎ': 63,
'ℳ': 118,
'ℴ': 51,
'ℵ': 82,
'ℶ': 75,
'ℷ': 51,
'ℸ': 71,
'ℹ': 42,
'℺': 102,
'℻': 131,
'ℼ': 77,
'ℽ': 80,
'ℾ': 72,
'ℿ': 93,
'⅀': 89,
'⅁': 85,
'⅂': 61,
'⅃': 61,
'⅄': 69,
'ⅅ': 90,
'ⅆ': 79,
'ⅇ': 68,
'ⅈ': 39,
'ⅉ': 52,
'⅋': 86,
'ⅎ': 58,
'⅐': 107,
'⅑': 107,
'⅒': 151,
'⅓': 107,
'⅔': 107,
'⅕': 107,
'⅖': 107,
'⅗': 107,
'⅘': 107,
'⅙': 107,
'⅚': 107,
'⅛': 107,
'⅜': 107,
'⅝': 107,
'⅞': 107,
'⅟': 83,
'Ⅰ': 32,
'Ⅱ': 54,
'Ⅲ': 76,
'Ⅳ': 102,
'Ⅴ': 75,
'Ⅵ': 101,
'Ⅶ': 123,
'Ⅷ': 145,
'Ⅸ': 101,
'Ⅹ': 75,
'Ⅺ': 103,
'Ⅻ': 124,
'Ⅼ': 61,
'Ⅽ': 77,
'Ⅾ': 85,
'Ⅿ': 95,
'ⅰ': 31,
'ⅱ': 50,
'ⅲ': 70,
'ⅳ': 89,
'ⅴ': 65,
'ⅵ': 89,
'ⅶ': 109,
'ⅷ': 129,
'ⅸ': 90,
'ⅹ': 65,
'ⅺ': 90,
'ⅻ': 110,
'ⅼ': 31,
'ⅽ': 60,
'ⅾ': 70,
'ⅿ': 107,
'ↀ': 137,
'ↁ': 85,
'ↂ': 137,
'Ↄ': 77,
'ↄ': 60,
'ↅ': 77,
'↉': 107,
'←': 92,
'↑': 92,
'→': 92,
'↓': 92,
'↔': 92,
'↕': 92,
'↖': 92,
'↗': 92,
'↘': 92,
'↙': 92,
'↚': 92,
'↛': 92,
'↜': 92,
'↝': 92,
'↞': 92,
'↟': 92,
'↠': 92,
'↡': 92,
'↢': 92,
'↣': 92,
'↤': 92,
'↥': 92,
'↦': 92,
'↧': 92,
'↨': 92,
'↩': 92,
'↪': 92,
'↫': 92,
'↬': 92,
'↭': 92,
'↮': 92,
'↯': 92,
'↰': 92,
'↱': 92,
'↲': 92,
'↳': 92,
'↴': 92,
'↵': 92,
'↶': 92,
'↷': 92,
'↸': 92,
'↹': 92,
'↺': 92,
'↻': 92,
'↼': 92,
'↽': 92,
'↾': 92,
'↿': 92,
'⇀': 92,
'⇁': 92,
'⇂': 92,
'⇃': 92,
'⇄': 92,
'⇅': 92,
'⇆': 92,
'⇇': 92,
'⇈': 92,
'⇉': 92,
'⇊': 92,
'⇋': 92,
'⇌': 92,
'⇍': 92,
'⇎': 92,
'⇏': 92,
'⇐': 92,
'⇑': 92,
'⇒': 92,
'⇓': 92,
'⇔': 92,
'⇕': 92,
'⇖': 92,
'⇗': 92,
'⇘': 92,
'⇙': 92,
'⇚': 92,
'⇛': 92,
'⇜': 92,
'⇝': 92,
'⇞': 92,
'⇟': 92,
'⇠': 92,
'⇡': 92,
'⇢': 92,
'⇣': 92,
'⇤': 92,
'⇥': 92,
'⇦': 92,
'⇧': 92,
'⇨': 92,
'⇩': 92,
'⇪': 92,
'⇫': 92,
'⇬': 92,
'⇭': 92,
'⇮': 92,
'⇯': 92,
'⇰': 92,
'⇱': 92,
'⇲': 92,
'⇳': 92,
'⇴': 92,
'⇵': 92,
'⇶': 92,
'⇷': 92,
'⇸': 92,
'⇹': 92,
'⇺': 92,
'⇻': 92,
'⇼': 92,
'⇽': 92,
'⇾': 92,
'⇿': 92,
'∀': 75,
'∁': 70,
'∂': 57,
'∃': 70,
'∄': 70,
'∅': 96,
'∆': 75,
'∇': 75,
'∈': 96,
'∉': 96,
'∊': 79,
'∋': 96,
'∌': 96,
'∍': 79,
'∎': 70,
'∏': 83,
'∐': 83,
'∑': 74,
'−': 92,
'∓': 92,
'∔': 92,
'∕': 38,
'∖': 70,
'∗': 92,
'∘': 69,
'∙': 69,
'√': 71,
'∛': 71,
'∜': 71,
'∝': 79,
'∞': 92,
'∟': 92,
'∠': 99,
'∡': 99,
'∢': 92,
'∣': 55,
'∤': 55,
'∥': 55,
'∦': 55,
'∧': 81,
'∨': 81,
'∩': 81,
'∪': 81,
'∫': 57,
'∬': 87,
'∭': 116,
'∮': 57,
'∯': 87,
'∰': 116,
'∱': 58,
'∲': 57,
'∳': 57,
'∴': 70,
'∵': 70,
'∶': 29,
'∷': 70,
'∸': 92,
'∹': 92,
'∺': 92,
'∻': 92,
'∼': 92,
'∽': 92,
'∾': 92,
'∿': 92,
'≀': 41,
'≁': 92,
'≂': 92,
'≃': 92,
'≄': 92,
'≅': 92,
'≆': 92,
'≇': 92,
'≈': 92,
'≉': 92,
'≊': 92,
'≋': 92,
'≌': 92,
'≍': 92,
'≎': 92,
'≏': 92,
'≐': 92,
'≑': 92,
'≒': 92,
'≓': 92,
'≔': 110,
'≕': 110,
'≖': 92,
'≗': 92,
'≘': 92,
'≙': 92,
'≚': 92,
'≛': 92,
'≜': 92,
'≝': 92,
'≞': 92,
'≟': 92,
'≠': 92,
'≡': 92,
'≢': 92,
'≣': 92,
'≤': 92,
'≥': 92,
'≦': 92,
'≧': 92,
'≨': 92,
'≩': 92,
'≪': 115,
'≫': 115,
'≬': 51,
'≭': 92,
'≮': 92,
'≯': 92,
'≰': 92,
'≱': 92,
'≲': 92,
'≳': 92,
'≴': 92,
'≵': 92,
'≶': 92,
'≷': 92,
'≸': 92,
'≹': 92,
'≺': 92,
'≻': 92,
'≼': 92,
'≽': 92,
'≾': 92,
'≿': 92,
'⊀': 92,
'⊁': 92,
'⊂': 92,
'⊃': 92,
'⊄': 92,
'⊅': 92,
'⊆': 92,
'⊇': 92,
'⊈': 92,
'⊉': 92,
'⊊': 92,
'⊋': 92,
'⊌': 81,
'⊍': 81,
'⊎': 81,
'⊏': 92,
'⊐': 92,
'⊑': 92,
'⊒': 92,
'⊓': 86,
'⊔': 86,
'⊕': 92,
'⊖': 92,
'⊗': 92,
'⊘': 92,
'⊙': 92,
'⊚': 92,
'⊛': 92,
'⊜': 92,
'⊝': 92,
'⊞': 92,
'⊟': 92,
'⊠': 92,
'⊡': 92,
'⊢': 96,
'⊣': 96,
'⊤': 96,
'⊥': 96,
'⊦': 57,
'⊧': 57,
'⊨': 96,
'⊩': 96,
'⊪': 96,
'⊫': 96,
'⊬': 96,
'⊭': 96,
'⊮': 96,
'⊯': 96,
'⊰': 92,
'⊱': 92,
'⊲': 92,
'⊳': 92,
'⊴': 92,
'⊵': 92,
'⊶': 110,
'⊷': 110,
'⊸': 92,
'⊹': 92,
'⊺': 57,
'⊻': 81,
'⊼': 81,
'⊽': 81,
'⊾': 92,
'⊿': 92,
'⋀': 92,
'⋁': 92,
'⋂': 90,
'⋃': 90,
'⋄': 69,
'⋅': 35,
'⋆': 69,
'⋇': 92,
'⋈': 110,
'⋉': 110,
'⋊': 110,
'⋋': 110,
'⋌': 110,
'⋍': 92,
'⋎': 81,
'⋏': 81,
'⋐': 92,
'⋑': 92,
'⋒': 92,
'⋓': 92,
'⋔': 92,
'⋕': 92,
'⋖': 92,
'⋗': 92,
'⋘': 156,
'⋙': 156,
'⋚': 92,
'⋛': 92,
'⋜': 92,
'⋝': 92,
'⋞': 92,
'⋟': 92,
'⋠': 92,
'⋡': 92,
'⋢': 92,
'⋣': 92,
'⋤': 92,
'⋥': 92,
'⋦': 92,
'⋧': 92,
'⋨': 92,
'⋩': 92,
'⋪': 92,
'⋫': 92,
'⋬': 92,
'⋭': 92,
'⋮': 110,
'⋯': 110,
'⋰': 110,
'⋱': 110,
'⋲': 110,
'⋳': 96,
'⋴': 79,
'⋵': 96,
'⋶': 96,
'⋷': 79,
'⋸': 96,
'⋹': 96,
'⋺': 110,
'⋻': 96,
'⋼': 79,
'⋽': 96,
'⋾': 79,
'⋿': 96,
'⌀': 66,
'⌁': 66,
'⌂': 70,
'⌃': 92,
'⌄': 92,
'⌅': 92,
'⌆': 92,
'⌇': 54,
'⌈': 43,
'⌉': 43,
'⌊': 43,
'⌋': 43,
'⌌': 89,
'⌍': 89,
'⌎': 89,
'⌏': 89,
'⌐': 92,
'⌑': 57,
'⌘': 110,
'⌙': 92,
'⌜': 52,
'⌝': 52,
'⌞': 52,
'⌟': 52,
'⌠': 57,
'⌡': 57,
'⌤': 127,
'⌥': 127,
'⌦': 156,
'⌧': 127,
'⌨': 159,
'⌫': 156,
'⌬': 96,
'⍳': 37,
'⍴': 70,
'⍵': 92,
'⍺': 73,
'⍽': 83,
'⎇': 127,
'⎔': 96,
'⎛': 55,
'⎜': 55,
'⎝': 55,
'⎞': 55,
'⎟': 55,
'⎠': 55,
'⎡': 55,
'⎢': 55,
'⎣': 55,
'⎤': 55,
'⎥': 55,
'⎦': 55,
'⎧': 83,
'⎨': 83,
'⎩': 83,
'⎪': 83,
'⎫': 83,
'⎬': 83,
'⎭': 83,
'⎮': 57,
'⏎': 92,
'⏏': 104,
'⏣': 96,
'⏥': 85,
'⏨': 70,
'␢': 77,
'␣': 70,
'①': 99,
'②': 99,
'③': 99,
'④': 99,
'⑤': 99,
'⑥': 99,
'⑦': 99,
'⑧': 99,
'⑨': 99,
'⑩': 99,
'─': 70,
'━': 70,
'│': 66,
'┃': 66,
'┄': 66,
'┅': 66,
'┆': 66,
'┇': 66,
'┈': 66,
'┉': 66,
'┊': 66,
'┋': 66,
'┌': 68,
'┍': 68,
'┎': 68,
'┏': 68,
'┐': 68,
'┑': 68,
'┒': 68,
'┓': 68,
'└': 68,
'┕': 68,
'┖': 68,
'┗': 68,
'┘': 68,
'┙': 68,
'┚': 68,
'┛': 68,
'├': 68,
'┝': 68,
'┞': 68,
'┟': 68,
'┠': 68,
'┡': 68,
'┢': 68,
'┣': 68,
'┤': 68,
'┥': 68,
'┦': 68,
'┧': 68,
'┨': 68,
'┩': 68,
'┪': 68,
'┫': 68,
'┬': 70,
'┭': 70,
'┮': 70,
'┯': 70,
'┰': 70,
'┱': 70,
'┲': 70,
'┳': 70,
'┴': 70,
'┵': 70,
'┶': 70,
'┷': 70,
'┸': 70,
'┹': 70,
'┺': 70,
'┻': 70,
'┼': 70,
'┽': 70,
'┾': 70,
'┿': 70,
'╀': 70,
'╁': 70,
'╂': 70,
'╃': 70,
'╄': 70,
'╅': 70,
'╆': 70,
'╇': 70,
'╈': 70,
'╉': 70,
'╊': 70,
'╋': 70,
'╌': 66,
'╍': 66,
'╎': 66,
'╏': 66,
'═': 70,
'║': 66,
'╒': 68,
'╓': 68,
'╔': 68,
'╕': 68,
'╖': 68,
'╗': 68,
'╘': 68,
'╙': 68,
'╚': 68,
'╛': 68,
'╜': 68,
'╝': 68,
'╞': 68,
'╟': 68,
'╠': 68,
'╡': 68,
'╢': 68,
'╣': 68,
'╤': 70,
'╥': 70,
'╦': 70,
'╧': 70,
'╨': 70,
'╩': 70,
'╪': 70,
'╫': 70,
'╬': 70,
'╭': 68,
'╮': 68,
'╯': 68,
'╰': 68,
'╱': 76,
'╲': 76,
'╳': 76,
'╴': 68,
'╵': 66,
'╶': 68,
'╷': 66,
'╸': 68,
'╹': 66,
'╺': 68,
'╻': 66,
'╼': 70,
'╽': 66,
'╾': 70,
'╿': 66,
'▀': 88,
'▁': 88,
'▂': 88,
'▃': 88,
'▄': 88,
'▅': 88,
'▆': 88,
'▇': 88,
'█': 88,
'▉': 87,
'▊': 87,
'▋': 87,
'▌': 87,
'▍': 87,
'▎': 87,
'▏': 87,
'▐': 86,
'░': 87,
'▒': 88,
'▓': 88,
'▔': 88,
'▕': 86,
'▖': 87,
'▗': 86,
'▘': 87,
'▙': 88,
'▚': 88,
'▛': 88,
'▜': 88,
'▝': 86,
'▞': 88,
'▟': 88,
'■': 104,
'□': 104,
'▢': 104,
'▣': 104,
'▤': 104,
'▥': 104,
'▦': 104,
'▧': 104,
'▨': 104,
'▩': 104,
'▪': 75,
'▫': 75,
'▬': 104,
'▭': 104,
'▮': 61,
'▯': 61,
'▰': 85,
'▱': 85,
'▲': 85,
'△': 85,
'▴': 55,
'▵': 55,
'▶': 85,
'▷': 85,
'▸': 55,
'▹': 55,
'►': 85,
'▻': 85,
'▼': 85,
'▽': 85,
'▾': 55,
'▿': 55,
'◀': 85,
'◁': 85,
'◂': 55,
'◃': 55,
'◄': 85,
'◅': 85,
'◆': 85,
'◇': 85,
'◈': 85,
'◉': 96,
'◊': 55,
'○': 96,
'◌': 96,
'◍': 96,
'◎': 96,
'●': 96,
'◐': 96,
'◑': 96,
'◒': 96,
'◓': 96,
'◔': 96,
'◕': 96,
'◖': 58,
'◗': 58,
'◘': 87,
'◙': 107,
'◚': 107,
'◛': 107,
'◜': 43,
'◝': 43,
'◞': 43,
'◟': 43,
'◠': 96,
'◡': 96,
'◢': 85,
'◣': 85,
'◤': 85,
'◥': 85,
'◦': 65,
'◧': 104,
'◨': 104,
'◩': 104,
'◪': 104,
'◫': 104,
'◬': 85,
'◭': 85,
'◮': 85,
'◯': 123,
'◰': 104,
'◱': 104,
'◲': 104,
'◳': 104,
'◴': 96,
'◵': 96,
'◶': 96,
'◷': 96,
'◸': 85,
'◹': 85,
'◺': 85,
'◻': 91,
'◼': 91,
'◽': 81,
'◾': 81,
'◿': 85,
'☀': 99,
'☁': 110,
'☂': 99,
'☃': 99,
'☄': 99,
'★': 99,
'☆': 99,
'☇': 63,
'☈': 99,
'☉': 99,
'☊': 98,
'☋': 98,
'☌': 74,
'☍': 111,
'☎': 137,
'☏': 138,
'☐': 99,
'☑': 99,
'☒': 99,
'☓': 59,
'☔': 99,
'☕': 99,
'☖': 99,
'☗': 99,
'☘': 99,
'☙': 99,
'☚': 99,
'☛': 99,
'☜': 99,
'☝': 67,
'☞': 99,
'☟': 67,
'☠': 99,
'☡': 99,
'☢': 99,
'☣': 99,
'☤': 74,
'☥': 82,
'☦': 71,
'☧': 86,
'☨': 60,
'☩': 99,
'☪': 99,
'☫': 99,
'☬': 78,
'☭': 99,
'☮': 99,
'☯': 99,
'☰': 99,
'☱': 99,
'☲': 99,
'☳': 99,
'☴': 99,
'☵': 99,
'☶': 99,
'☷': 99,
'☸': 99,
'☹': 115,
'☺': 115,
'☻': 115,
'☼': 99,
'☽': 99,
'☾': 99,
'☿': 68,
'♀': 81,
'♁': 81,
'♂': 99,
'♃': 99,
'♄': 99,
'♅': 99,
'♆': 99,
'♇': 99,
'♈': 99,
'♉': 99,
'♊': 99,
'♋': 99,
'♌': 99,
'♍': 99,
'♎': 99,
'♏': 99,
'♐': 99,
'♑': 99,
'♒': 99,
'♓': 99,
'♔': 99,
'♕': 99,
'♖': 99,
'♗': 99,
'♘': 99,
'♙': 99,
'♚': 99,
'♛': 99,
'♜': 99,
'♝': 99,
'♞': 99,
'♟': 99,
'♠': 99,
'♡': 99,
'♢': 99,
'♣': 99,
'♤': 99,
'♥': 99,
'♦': 99,
'♧': 99,
'♨': 99,
'♩': 52,
'♪': 70,
'♫': 99,
'♬': 99,
'♭': 52,
'♮': 39,
'♯': 53,
'♰': 82,
'♱': 84,
'♲': 99,
'♳': 99,
'♴': 99,
'♵': 99,
'♶': 99,
'♷': 99,
'♸': 99,
'♹': 99,
'♺': 99,
'♻': 99,
'♼': 99,
'♽': 99,
'♾': 99,
'♿': 99,
'⚀': 96,
'⚁': 96,
'⚂': 96,
'⚃': 96,
'⚄': 96,
'⚅': 96,
'⚆': 99,
'⚇': 99,
'⚈': 99,
'⚉': 99,
'⚊': 99,
'⚋': 99,
'⚌': 99,
'⚍': 99,
'⚎': 99,
'⚏': 99,
'⚐': 99,
'⚑': 99,
'⚒': 99,
'⚓': 99,
'⚔': 99,
'⚕': 60,
'⚖': 99,
'⚗': 99,
'⚘': 99,
'⚙': 99,
'⚚': 99,
'⚛': 99,
'⚜': 99,
'⚞': 99,
'⚟': 99,
'⚠': 99,
'⚡': 77,
'⚢': 110,
'⚣': 120,
'⚤': 129,
'⚥': 99,
'⚦': 92,
'⚧': 92,
'⚨': 92,
'⚩': 92,
'⚪': 92,
'⚫': 92,
'⚬': 92,
'⚭': 92,
'⚮': 92,
'⚯': 92,
'⚰': 93,
'⚱': 92,
'⚲': 81,
'⚳': 81,
'⚴': 81,
'⚵': 81,
'⚶': 93,
'⚷': 81,
'⚸': 81,
'⛀': 92,
'⛁': 92,
'⛂': 92,
'⛃': 92,
'⛢': 81,
'✁': 92,
'✂': 92,
'✃': 92,
'✄': 92,
'✆': 92,
'✇': 92,
'✈': 92,
'✉': 92,
'✌': 92,
'✍': 92,
'✎': 92,
'✏': 92,
'✐': 92,
'✑': 92,
'✒': 92,
'✓': 92,
'✔': 92,
'✕': 92,
'✖': 92,
'✗': 92,
'✘': 92,
'✙': 92,
'✚': 92,
'✛': 92,
'✜': 92,
'✝': 92,
'✞': 92,
'✟': 92,
'✠': 92,
'✡': 92,
'✢': 92,
'✣': 92,
'✤': 92,
'✥': 92,
'✦': 92,
'✧': 92,
'✩': 92,
'✪': 92,
'✫': 92,
'✬': 92,
'✭': 92,
'✮': 92,
'✯': 92,
'✰': 92,
'✱': 92,
'✲': 92,
'✳': 92,
'✴': 92,
'✵': 92,
'✶': 92,
'✷': 92,
'✸': 92,
'✹': 92,
'✺': 92,
'✻': 92,
'✼': 92,
'✽': 92,
'✾': 92,
'✿': 92,
'❀': 92,
'❁': 92,
'❂': 92,
'❃': 92,
'❄': 92,
'❅': 92,
'❆': 92,
'❇': 92,
'❈': 92,
'❉': 92,
'❊': 92,
'❋': 92,
'❍': 99,
'❏': 99,
'❐': 99,
'❑': 99,
'❒': 99,
'❖': 99,
'❘': 92,
'❙': 92,
'❚': 92,
'❛': 35,
'❜': 35,
'❝': 59,
'❞': 59,
'❡': 92,
'❢': 92,
'❣': 92,
'❤': 92,
'❥': 92,
'❦': 92,
'❧': 92,
'❨': 92,
'❩': 92,
'❪': 92,
'❫': 92,
'❬': 92,
'❭': 92,
'❮': 92,
'❯': 92,
'❰': 92,
'❱': 92,
'❲': 92,
'❳': 92,
'❴': 92,
'❵': 92,
'❶': 99,
'❷': 99,
'❸': 99,
'❹': 99,
'❺': 99,
'❻': 99,
'❼': 99,
'❽': 99,
'❾': 99,
'❿': 99,
'➀': 92,
'➁': 92,
'➂': 92,
'➃': 92,
'➄': 92,
'➅': 92,
'➆': 92,
'➇': 92,
'➈': 92,
'➉': 92,
'➊': 92,
'➋': 92,
'➌': 92,
'➍': 92,
'➎': 92,
'➏': 92,
'➐': 92,
'➑': 92,
'➒': 92,
'➓': 92,
'➔': 92,
'➘': 92,
'➙': 92,
'➚': 92,
'➛': 92,
'➜': 92,
'➝': 92,
'➞': 92,
'➟': 92,
'➠': 92,
'➡': 92,
'➢': 92,
'➣': 92,
'➤': 92,
'➥': 92,
'➦': 92,
'➧': 92,
'➨': 92,
'➩': 92,
'➪': 92,
'➫': 92,
'➬': 92,
'➭': 92,
'➮': 92,
'➯': 92,
'➱': 92,
'➲': 92,
'➳': 92,
'➴': 92,
'➵': 92,
'➶': 92,
'➷': 92,
'➸': 92,
'➹': 92,
'➺': 92,
'➻': 92,
'➼': 92,
'➽': 92,
'➾': 92,
'⟅': 43,
'⟆': 43,
'⟠': 55,
'⟦': 54,
'⟧': 54,
'⟨': 43,
'⟩': 43,
'⟪': 61,
'⟫': 61,
'⟰': 92,
'⟱': 92,
'⟲': 92,
'⟳': 92,
'⟴': 127,
'⟵': 158,
'⟶': 158,
'⟷': 158,
'⟸': 158,
'⟹': 158,
'⟺': 158,
'⟻': 158,
'⟼': 158,
'⟽': 158,
'⟾': 158,
'⟿': 158,
'⠀': 81,
'⠁': 81,
'⠂': 81,
'⠃': 81,
'⠄': 81,
'⠅': 81,
'⠆': 81,
'⠇': 81,
'⠈': 81,
'⠉': 81,
'⠊': 81,
'⠋': 81,
'⠌': 81,
'⠍': 81,
'⠎': 81,
'⠏': 81,
'⠐': 81,
'⠑': 81,
'⠒': 81,
'⠓': 81,
'⠔': 81,
'⠕': 81,
'⠖': 81,
'⠗': 81,
'⠘': 81,
'⠙': 81,
'⠚': 81,
'⠛': 81,
'⠜': 81,
'⠝': 81,
'⠞': 81,
'⠟': 81,
'⠠': 81,
'⠡': 81,
'⠢': 81,
'⠣': 81,
'⠤': 81,
'⠥': 81,
'⠦': 81,
'⠧': 81,
'⠨': 81,
'⠩': 81,
'⠪': 81,
'⠫': 81,
'⠬': 81,
'⠭': 81,
'⠮': 81,
'⠯': 81,
'⠰': 81,
'⠱': 81,
'⠲': 81,
'⠳': 81,
'⠴': 81,
'⠵': 81,
'⠶': 81,
'⠷': 81,
'⠸': 81,
'⠹': 81,
'⠺': 81,
'⠻': 81,
'⠼': 81,
'⠽': 81,
'⠾': 81,
'⠿': 81,
'⡀': 81,
'⡁': 81,
'⡂': 81,
'⡃': 81,
'⡄': 81,
'⡅': 81,
'⡆': 81,
'⡇': 81,
'⡈': 81,
'⡉': 81,
'⡊': 81,
'⡋': 81,
'⡌': 81,
'⡍': 81,
'⡎': 81,
'⡏': 81,
'⡐': 81,
'⡑': 81,
'⡒': 81,
'⡓': 81,
'⡔': 81,
'⡕': 81,
'⡖': 81,
'⡗': 81,
'⡘': 81,
'⡙': 81,
'⡚': 81,
'⡛': 81,
'⡜': 81,
'⡝': 81,
'⡞': 81,
'⡟': 81,
'⡠': 81,
'⡡': 81,
'⡢': 81,
'⡣': 81,
'⡤': 81,
'⡥': 81,
'⡦': 81,
'⡧': 81,
'⡨': 81,
'⡩': 81,
'⡪': 81,
'⡫': 81,
'⡬': 81,
'⡭': 81,
'⡮': 81,
'⡯': 81,
'⡰': 81,
'⡱': 81,
'⡲': 81,
'⡳': 81,
'⡴': 81,
'⡵': 81,
'⡶': 81,
'⡷': 81,
'⡸': 81,
'⡹': 81,
'⡺': 81,
'⡻': 81,
'⡼': 81,
'⡽': 81,
'⡾': 81,
'⡿': 81,
'⢀': 81,
'⢁': 81,
'⢂': 81,
'⢃': 81,
'⢄': 81,
'⢅': 81,
'⢆': 81,
'⢇': 81,
'⢈': 81,
'⢉': 81,
'⢊': 81,
'⢋': 81,
'⢌': 81,
'⢍': 81,
'⢎': 81,
'⢏': 81,
'⢐': 81,
'⢑': 81,
'⢒': 81,
'⢓': 81,
'⢔': 81,
'⢕': 81,
'⢖': 81,
'⢗': 81,
'⢘': 81,
'⢙': 81,
'⢚': 81,
'⢛': 81,
'⢜': 81,
'⢝': 81,
'⢞': 81,
'⢟': 81,
'⢠': 81,
'⢡': 81,
'⢢': 81,
'⢣': 81,
'⢤': 81,
'⢥': 81,
'⢦': 81,
'⢧': 81,
'⢨': 81,
'⢩': 81,
'⢪': 81,
'⢫': 81,
'⢬': 81,
'⢭': 81,
'⢮': 81,
'⢯': 81,
'⢰': 81,
'⢱': 81,
'⢲': 81,
'⢳': 81,
'⢴': 81,
'⢵': 81,
'⢶': 81,
'⢷': 81,
'⢸': 81,
'⢹': 81,
'⢺': 81,
'⢻': 81,
'⢼': 81,
'⢽': 81,
'⢾': 81,
'⢿': 81,
'⣀': 81,
'⣁': 81,
'⣂': 81,
'⣃': 81,
'⣄': 81,
'⣅': 81,
'⣆': 81,
'⣇': 81,
'⣈': 81,
'⣉': 81,
'⣊': 81,
'⣋': 81,
'⣌': 81,
'⣍': 81,
'⣎': 81,
'⣏': 81,
'⣐': 81,
'⣑': 81,
'⣒': 81,
'⣓': 81,
'⣔': 81,
'⣕': 81,
'⣖': 81,
'⣗': 81,
'⣘': 81,
'⣙': 81,
'⣚': 81,
'⣛': 81,
'⣜': 81,
'⣝': 81,
'⣞': 81,
'⣟': 81,
'⣠': 81,
'⣡': 81,
'⣢': 81,
'⣣': 81,
'⣤': 81,
'⣥': 81,
'⣦': 81,
'⣧': 81,
'⣨': 81,
'⣩': 81,
'⣪': 81,
'⣫': 81,
'⣬': 81,
'⣭': 81,
'⣮': 81,
'⣯': 81,
'⣰': 81,
'⣱': 81,
'⣲': 81,
'⣳': 81,
'⣴': 81,
'⣵': 81,
'⣶': 81,
'⣷': 81,
'⣸': 81,
'⣹': 81,
'⣺': 81,
'⣻': 81,
'⣼': 81,
'⣽': 81,
'⣾': 81,
'⣿': 81,
'⤆': 92,
'⤇': 92,
'⤊': 92,
'⤋': 92,
'⥀': 75,
'⥁': 75,
'⦃': 81,
'⦄': 81,
'⧎': 92,
'⧏': 110,
'⧐': 110,
'⧑': 110,
'⧒': 110,
'⧓': 110,
'⧔': 110,
'⧕': 110,
'⧫': 55,
'⧺': 92,
'⧻': 92,
'⨀': 110,
'⨁': 110,
'⨂': 110,
'⨌': 146,
'⨍': 57,
'⨎': 57,
'⨏': 57,
'⨐': 57,
'⨑': 58,
'⨒': 57,
'⨓': 57,
'⨔': 62,
'⨕': 57,
'⨖': 57,
'⨗': 65,
'⨘': 57,
'⨙': 57,
'⨚': 57,
'⨛': 57,
'⨜': 57,
'⨯': 92,
'⩪': 92,
'⩫': 92,
'⩽': 92,
'⩾': 92,
'⩿': 92,
'⪀': 92,
'⪁': 92,
'⪂': 92,
'⪃': 92,
'⪄': 92,
'⪅': 92,
'⪆': 92,
'⪇': 92,
'⪈': 92,
'⪉': 92,
'⪊': 92,
'⪋': 92,
'⪌': 92,
'⪍': 92,
'⪎': 92,
'⪏': 92,
'⪐': 92,
'⪑': 92,
'⪒': 92,
'⪓': 92,
'⪔': 92,
'⪕': 92,
'⪖': 92,
'⪗': 92,
'⪘': 92,
'⪙': 92,
'⪚': 92,
'⪛': 92,
'⪜': 92,
'⪝': 92,
'⪞': 92,
'⪟': 92,
'⪠': 92,
'⪮': 92,
'⪯': 92,
'⪰': 92,
'⪱': 92,
'⪲': 92,
'⪳': 92,
'⪴': 92,
'⪵': 92,
'⪶': 92,
'⪷': 92,
'⪸': 92,
'⪹': 92,
'⪺': 92,
'⫹': 92,
'⫺': 92,
'⬀': 92,
'⬁': 92,
'⬂': 92,
'⬃': 92,
'⬄': 92,
'⬅': 92,
'⬆': 92,
'⬇': 92,
'⬈': 92,
'⬉': 92,
'⬊': 92,
'⬋': 92,
'⬌': 92,
'⬍': 92,
'⬎': 92,
'⬏': 92,
'⬐': 92,
'⬑': 92,
'⬒': 104,
'⬓': 104,
'⬔': 104,
'⬕': 104,
'⬖': 85,
'⬗': 85,
'⬘': 85,
'⬙': 85,
'⬚': 104,
'⬟': 96,
'⬠': 96,
'⬡': 96,
'⬢': 96,
'⬣': 96,
'⬤': 123,
'⭓': 96,
'⭔': 96,
'Ⱡ': 61,
'ⱡ': 31,
'Ɫ': 64,
'Ᵽ': 66,
'Ɽ': 76,
'ⱥ': 67,
'ⱦ': 45,
'Ⱨ': 83,
'ⱨ': 71,
'Ⱪ': 75,
'ⱪ': 64,
'Ⱬ': 82,
'ⱬ': 63,
'Ɑ': 86,
'Ɱ': 95,
'Ɐ': 75,
'Ɒ': 86,
'ⱱ': 81,
'Ⱳ': 125,
'ⱳ': 106,
'ⱴ': 65,
'Ⱶ': 72,
'ⱶ': 62,
'ⱷ': 73,
'ⱹ': 46,
'ⱺ': 67,
'ⱻ': 54,
'ⱼ': 21,
'ⱽ': 47,
'Ȿ': 70,
'Ɀ': 75,
'ⴀ': 65,
'ⴁ': 65,
'ⴂ': 62,
'ⴃ': 66,
'ⴄ': 65,
'ⴅ': 100,
'ⴆ': 69,
'ⴇ': 105,
'ⴈ': 65,
'ⴉ': 67,
'ⴊ': 105,
'ⴋ': 68,
'ⴌ': 65,
'ⴍ': 102,
'ⴎ': 65,
'ⴏ': 89,
'ⴐ': 102,
'ⴑ': 64,
'ⴒ': 65,
'ⴓ': 102,
'ⴔ': 105,
'ⴕ': 91,
'ⴖ': 66,
'ⴗ': 65,
'ⴘ': 65,
'ⴙ': 65,
'ⴚ': 65,
'ⴛ': 68,
'ⴜ': 101,
'ⴝ': 65,
'ⴞ': 65,
'ⴟ': 64,
'ⴠ': 101,
'ⴡ': 66,
'ⴢ': 65,
'ⴣ': 65,
'ⴤ': 71,
'ⴥ': 99,
'ⴰ': 71,
'ⴱ': 98,
'ⴲ': 98,
'ⴳ': 75,
'ⴴ': 75,
'ⴵ': 70,
'ⴶ': 62,
'ⴷ': 75,
'ⴸ': 75,
'ⴹ': 70,
'ⴺ': 70,
'ⴻ': 75,
'ⴼ': 96,
'ⴽ': 75,
'ⴾ': 54,
'ⴿ': 75,
'ⵀ': 98,
'ⵁ': 98,
'ⵂ': 33,
'ⵃ': 69,
'ⵄ': 83,
'ⵅ': 72,
'ⵆ': 58,
'ⵇ': 75,
'ⵈ': 71,
'ⵉ': 70,
'ⵊ': 55,
'ⵋ': 105,
'ⵌ': 86,
'ⵍ': 82,
'ⵎ': 68,
'ⵏ': 32,
'ⵐ': 86,
'ⵑ': 32,
'ⵒ': 83,
'ⵓ': 70,
'ⵔ': 98,
'ⵕ': 98,
'ⵖ': 83,
'ⵗ': 35,
'ⵘ': 82,
'ⵙ': 98,
'ⵚ': 98,
'ⵛ': 77,
'ⵜ': 84,
'ⵝ': 75,
'ⵞ': 77,
'ⵟ': 68,
'ⵠ': 75,
'ⵡ': 83,
'ⵢ': 70,
'ⵣ': 87,
'ⵤ': 62,
'ⵥ': 87,
'ⵯ': 57,
'⸘': 58,
'⸟': 92,
'⸢': 43,
'⸣': 43,
'⸤': 43,
'⸥': 43,
'⸮': 58,
'䷀': 99,
'䷁': 99,
'䷂': 99,
'䷃': 99,
'䷄': 99,
'䷅': 99,
'䷆': 99,
'䷇': 99,
'䷈': 99,
'䷉': 99,
'䷊': 99,
'䷋': 99,
'䷌': 99,
'䷍': 99,
'䷎': 99,
'䷏': 99,
'䷐': 99,
'䷑': 99,
'䷒': 99,
'䷓': 99,
'䷔': 99,
'䷕': 99,
'䷖': 99,
'䷗': 99,
'䷘': 99,
'䷙': 99,
'䷚': 99,
'䷛': 99,
'䷜': 99,
'䷝': 99,
'䷞': 99,
'䷟': 99,
'䷠': 99,
'䷡': 99,
'䷢': 99,
'䷣': 99,
'䷤': 99,
'䷥': 99,
'䷦': 99,
'䷧': 99,
'䷨': 99,
'䷩': 99,
'䷪': 99,
'䷫': 99,
'䷬': 99,
'䷭': 99,
'䷮': 99,
'䷯': 99,
'䷰': 99,
'䷱': 99,
'䷲': 99,
'䷳': 99,
'䷴': 99,
'䷵': 99,
'䷶': 99,
'䷷': 99,
'䷸': 99,
'䷹': 99,
'䷺': 99,
'䷻': 99,
'䷼': 99,
'䷽': 99,
'䷾': 99,
'䷿': 99,
'ꓐ': 75,
'ꓑ': 66,
'ꓒ': 66,
'ꓓ': 85,
'ꓔ': 69,
'ꓕ': 69,
'ꓖ': 85,
'ꓗ': 75,
'ꓘ': 75,
'ꓙ': 56,
'ꓚ': 77,
'ꓛ': 77,
'ꓜ': 75,
'ꓝ': 63,
'ꓞ': 63,
'ꓟ': 95,
'ꓠ': 82,
'ꓡ': 61,
'ꓢ': 70,
'ꓣ': 76,
'ꓤ': 76,
'ꓥ': 75,
'ꓦ': 75,
'ꓧ': 83,
'ꓨ': 85,
'ꓩ': 57,
'ꓪ': 109,
'ꓫ': 75,
'ꓬ': 69,
'ꓭ': 75,
'ꓮ': 75,
'ꓯ': 75,
'ꓰ': 70,
'ꓱ': 70,
'ꓲ': 32,
'ꓳ': 87,
'ꓴ': 81,
'ꓵ': 81,
'ꓶ': 61,
'ꓷ': 84,
'ꓸ': 33,
'ꓹ': 33,
'ꓺ': 66,
'ꓻ': 66,
'ꓼ': 33,
'ꓽ': 33,
'꓾': 65,
'꓿': 65,
'Ꙅ': 70,
'ꙅ': 57,
'Ꙇ': 39,
'ꙇ': 37,
'Ꙍ': 130,
'ꙍ': 113,
'Ꙑ': 113,
'ꙑ': 100,
'Ꙕ': 119,
'ꙕ': 93,
'Ꙗ': 107,
'ꙗ': 93,
'Ꙣ': 117,
'ꙣ': 100,
'Ꙥ': 117,
'ꙥ': 99,
'Ꙧ': 130,
'ꙧ': 111,
'Ꙩ': 87,
'ꙩ': 67,
'Ꙫ': 94,
'ꙫ': 78,
'Ꙭ': 149,
'ꙭ': 112,
'ꙮ': 97,
'Ꚋ': 87,
'ꚋ': 75,
'Ꚍ': 69,
'ꚍ': 64,
'Ꚕ': 75,
'ꚕ': 70,
'꜈': 54,
'꜉': 54,
'꜊': 54,
'꜋': 54,
'꜌': 54,
'꜍': 54,
'꜎': 54,
'꜏': 54,
'꜐': 54,
'꜑': 54,
'꜒': 54,
'꜓': 54,
'꜔': 54,
'꜕': 54,
'꜖': 54,
'ꜛ': 41,
'ꜜ': 41,
'ꜝ': 28,
'ꜞ': 28,
'ꜟ': 28,
'Ꜣ': 42,
'ꜣ': 39,
'Ꜥ': 52,
'ꜥ': 52,
'Ꜧ': 83,
'ꜧ': 70,
'Ꜩ': 98,
'ꜩ': 78,
'Ꜫ': 68,
'ꜫ': 59,
'ꜰ': 54,
'ꜱ': 57,
'Ꜳ': 137,
'ꜳ': 108,
'Ꜵ': 132,
'ꜵ': 109,
'Ꜷ': 126,
'ꜷ': 108,
'Ꜹ': 107,
'ꜹ': 90,
'Ꜻ': 107,
'ꜻ': 90,
'Ꜽ': 105,
'ꜽ': 90,
'Ꜿ': 77,
'ꜿ': 60,
'Ꝁ': 75,
'ꝁ': 64,
'Ꝇ': 75,
'ꝇ': 43,
'Ꝉ': 64,
'ꝉ': 47,
'Ꝋ': 89,
'ꝋ': 77,
'Ꝏ': 149,
'ꝏ': 112,
'Ꝑ': 66,
'ꝑ': 71,
'Ꝓ': 81,
'ꝓ': 85,
'Ꝗ': 87,
'ꝗ': 71,
'Ꝥ': 67,
'ꝥ': 71,
'Ꝧ': 67,
'ꝧ': 71,
'Ꞁ': 61,
'ꞁ': 31,
'Ꞃ': 84,
'ꞃ': 71,
'꞉': 37,
'꞊': 41,
'Ꞌ': 44,
'ꞌ': 30,
'Ɥ': 75,
'ꞎ': 54,
'Ꞑ': 85,
'ꞑ': 73,
'Ꞡ': 86,
'ꞡ': 70,
'Ꞣ': 75,
'ꞣ': 64,
'Ꞥ': 83,
'ꞥ': 70,
'Ꞧ': 77,
'ꞧ': 46,
'Ꞩ': 70,
'ꞩ': 58,
'Ɦ': 94,
'ꟸ': 63,
'ꟹ': 71,
'ꟺ': 101,
'ꟻ': 63,
'ꟼ': 66,
'ꟽ': 95,
'ꟾ': 32,
'ꟿ': 132,
'\uef00': 36,
'\uef01': 38,
'\uef02': 40,
'\uef03': 41,
'\uef04': 41,
'\uef05': 38,
'\uef06': 36,
'\uef07': 38,
'\uef08': 40,
'\uef09': 41,
'\uef0a': 40,
'\uef0b': 38,
'\uef0c': 36,
'\uef0d': 38,
'\uef0e': 40,
'\uef0f': 41,
'\uef10': 40,
'\uef11': 38,
'\uef12': 36,
'\uef13': 38,
'\uef14': 41,
'\uef15': 41,
'\uef16': 40,
'\uef17': 38,
'\uef18': 36,
'\uef19': 30,
'\uf000': 107,
'\uf001': 107,
'\uf002': 107,
'\uf003': 107,
'\uf400': 64,
'\uf401': 64,
'\uf402': 69,
'\uf403': 98,
'\uf404': 64,
'\uf405': 64,
'\uf406': 72,
'\uf407': 97,
'\uf408': 61,
'\uf409': 64,
'\uf40a': 129,
'\uf40b': 65,
'\uf40c': 65,
'\uf40d': 96,
'\uf40e': 64,
'\uf40f': 65,
'\uf410': 101,
'\uf411': 65,
'\uf412': 80,
'\uf413': 64,
'\uf414': 96,
'\uf415': 65,
'\uf416': 98,
'\uf417': 65,
'\uf418': 65,
'\uf419': 65,
'\uf41a': 71,
'\uf41b': 65,
'\uf41c': 65,
'\uf41d': 66,
'\uf41e': 65,
'\uf41f': 57,
'\uf420': 64,
'\uf421': 64,
'\uf422': 64,
'\uf423': 64,
'\uf424': 64,
'\uf425': 70,
'\uf426': 105,
'\uf428': 102,
'\uf429': 89,
'\uf42a': 56,
'\uf42b': 56,
'\uf42c': 56,
'\uf42d': 56,
'\uf42e': 56,
'\uf42f': 56,
'\uf430': 56,
'\uf431': 56,
'\uf432': 56,
'\uf433': 56,
'\uf434': 57,
'\uf435': 57,
'\uf436': 57,
'\uf437': 87,
'\uf438': 87,
'\uf439': 87,
'\uf43a': 87,
'\uf43b': 87,
'\uf43c': 60,
'\uf43d': 60,
'\uf43e': 60,
'\uf43f': 60,
'\uf440': 60,
'\uf441': 60,
'\uf6c5': 67,
'ﬀ': 78,
'ﬁ': 69,
'ﬂ': 69,
'ﬃ': 106,
'ﬄ': 106,
'ﬅ': 75,
'ﬆ': 95,
'ﬓ': 132,
'ﬔ': 132,
'ﬕ': 132,
'ﬖ': 130,
'ﬗ': 168,
'יִ': 25,
'ﬞ': 52,
'ײַ': 37,
'ﬠ': 70,
'ﬡ': 94,
'ﬢ': 85,
'ﬣ': 100,
'ﬤ': 85,
'ﬥ': 93,
'ﬦ': 94,
'ﬧ': 89,
'ﬨ': 96,
'﬩': 92,
'שׁ': 78,
'שׂ': 78,
'שּׁ': 78,
'שּׂ': 78,
'אַ': 74,
'אָ': 74,
'אּ': 74,
'בּ': 64,
'גּ': 45,
'דּ': 60,
'הּ': 72,
'וּ': 39,
'זּ': 45,
'טּ': 71,
'יּ': 36,
'ךּ': 59,
'כּ': 58,
'לּ': 63,
'מּ': 75,
'נּ': 44,
'סּ': 71,
'ףּ': 70,
'פּ': 69,
'צּ': 65,
'קּ': 78,
'רּ': 62,
'שּ': 78,
'תּ': 72,
'וֹ': 30,
'בֿ': 64,
'כֿ': 58,
'פֿ': 69,
'ﭏ': 69,
'ﭒ': 104,
'ﭓ': 110,
'ﭔ': 33,
'ﭕ': 37,
'ﭖ': 104,
'ﭗ': 110,
'ﭘ': 33,
'ﭙ': 37,
'ﭚ': 104,
'ﭛ': 110,
'ﭜ': 33,
'ﭝ': 37,
'ﭞ': 104,
'ﭟ': 110,
'ﭠ': 33,
'ﭡ': 37,
'ﭢ': 104,
'ﭣ': 110,
'ﭤ': 33,
'ﭥ': 37,
'ﭦ': 104,
'ﭧ': 110,
'ﭨ': 33,
'ﭩ': 37,
'ﭪ': 114,
'ﭫ': 115,
'ﭬ': 55,
'ﭭ': 59,
'ﭮ': 114,
'ﭯ': 115,
'ﭰ': 55,
'ﭱ': 59,
'ﭲ': 71,
'ﭳ': 73,
'ﭴ': 70,
'ﭵ': 75,
'ﭶ': 71,
'ﭷ': 73,
'ﭸ': 70,
'ﭹ': 75,
'ﭺ': 71,
'ﭻ': 73,
'ﭼ': 70,
'ﭽ': 75,
'ﭾ': 71,
'ﭿ': 73,
'ﮀ': 70,
'ﮁ': 75,
'ﮂ': 49,
'ﮃ': 59,
'ﮄ': 49,
'ﮅ': 59,
'ﮆ': 49,
'ﮇ': 59,
'ﮈ': 49,
'ﮉ': 59,
'ﮊ': 58,
'ﮋ': 67,
'ﮌ': 58,
'ﮍ': 67,
'ﮎ': 99,
'ﮏ': 108,
'ﮐ': 55,
'ﮑ': 64,
'ﮒ': 99,
'ﮓ': 108,
'ﮔ': 55,
'ﮕ': 64,
'ﮖ': 99,
'ﮗ': 108,
'ﮘ': 55,
'ﮙ': 64,
'ﮚ': 99,
'ﮛ': 108,
'ﮜ': 55,
'ﮝ': 64,
'ﮞ': 81,
'ﮟ': 85,
'ﮠ': 81,
'ﮡ': 85,
'ﮢ': 33,
'ﮣ': 37,
'ﮪ': 77,
'ﮫ': 71,
'ﮬ': 60,
'ﮭ': 54,
'ﯓ': 91,
'ﯔ': 94,
'ﯕ': 55,
'ﯖ': 64,
'ﯗ': 58,
'ﯘ': 63,
'ﯙ': 58,
'ﯚ': 63,
'ﯛ': 58,
'ﯜ': 63,
'ﯞ': 58,
'ﯟ': 63,
'ﯤ': 86,
'ﯥ': 93,
'ﯦ': 33,
'ﯧ': 37,
'ﯨ': 33,
'ﯩ': 37,
'ﯼ': 86,
'ﯽ': 93,
'ﯾ': 33,
'ﯿ': 37,
'︀': 0,
'︁': 0,
'︂': 0,
'︃': 0,
'︄': 0,
'︅': 0,
'︆': 0,
'︇': 0,
'︈': 0,
'︉': 0,
'︊': 0,
'︋': 0,
'︌': 0,
'︍': 0,
'︎': 0,
'️': 0,
'︠': 49,
'︡': 49,
'︢': 39,
'︣': 39,
'ﹰ': 32,
'ﹱ': 36,
'ﹲ': 32,
'ﹳ': 30,
'ﹴ': 32,
'ﹶ': 32,
'ﹷ': 36,
'ﹸ': 32,
'ﹹ': 36,
'ﹺ': 32,
'ﹻ': 36,
'ﹼ': 34,
'ﹽ': 36,
'ﹾ': 32,
'ﹿ': 36,
'ﺀ': 52,
'ﺁ': 40,
'ﺂ': 40,
'ﺃ': 31,
'ﺄ': 35,
'ﺅ': 58,
'ﺆ': 63,
'ﺇ': 31,
'ﺈ': 35,
'ﺉ': 86,
'ﺊ': 93,
'ﺋ': 33,
'ﺌ': 37,
'ﺍ': 31,
'ﺎ': 35,
'ﺏ': 104,
'ﺐ': 110,
'ﺑ': 33,
'ﺒ': 37,
'ﺓ': 58,
'ﺔ': 61,
'ﺕ': 104,
'ﺖ': 110,
'ﺗ': 33,
'ﺘ': 37,
'ﺙ': 104,
'ﺚ': 110,
'ﺛ': 33,
'ﺜ': 37,
'ﺝ': 71,
'ﺞ': 73,
'ﺟ': 70,
'ﺠ': 75,
'ﺡ': 71,
'ﺢ': 73,
'ﺣ': 70,
'ﺤ': 75,
'ﺥ': 71,
'ﺦ': 73,
'ﺧ': 70,
'ﺨ': 75,
'ﺩ': 49,
'ﺪ': 59,
'ﺫ': 49,
'ﺬ': 59,
'ﺭ': 58,
'ﺮ': 67,
'ﺯ': 58,
'ﺰ': 67,
'ﺱ': 134,
'ﺲ': 142,
'ﺳ': 94,
'ﺴ': 102,
'ﺵ': 134,
'ﺶ': 142,
'ﺷ': 94,
'ﺸ': 102,
'ﺹ': 133,
'ﺺ': 136,
'ﺻ': 95,
'ﺼ': 99,
'ﺽ': 133,
'ﺾ': 136,
'ﺿ': 95,
'ﻀ': 99,
'ﻁ': 102,
'ﻂ': 106,
'ﻃ': 90,
'ﻄ': 94,
'ﻅ': 102,
'ﻆ': 106,
'ﻇ': 90,
'ﻈ': 94,
'ﻉ': 66,
'ﻊ': 65,
'ﻋ': 68,
'ﻌ': 57,
'ﻍ': 66,
'ﻎ': 65,
'ﻏ': 60,
'ﻐ': 57,
'ﻑ': 114,
'ﻒ': 115,
'ﻓ': 55,
'ﻔ': 59,
'ﻕ': 85,
'ﻖ': 93,
'ﻗ': 55,
'ﻘ': 59,
'ﻙ': 91,
'ﻚ': 94,
'ﻛ': 55,
'ﻜ': 64,
'ﻝ': 80,
'ﻞ': 85,
'ﻟ': 36,
'ﻠ': 40,
'ﻡ': 68,
'ﻢ': 75,
'ﻣ': 61,
'ﻤ': 67,
'ﻥ': 81,
'ﻦ': 85,
'ﻧ': 33,
'ﻨ': 37,
'ﻩ': 58,
'ﻪ': 61,
'ﻫ': 60,
'ﻬ': 54,
'ﻭ': 58,
'ﻮ': 63,
'ﻯ': 86,
'ﻰ': 93,
'ﻱ': 86,
'ﻲ': 93,
'ﻳ': 33,
'ﻴ': 37,
'ﻵ': 75,
'ﻶ': 79,
'ﻷ': 65,
'ﻸ': 69,
'ﻹ': 63,
'ﻺ': 67,
'ﻻ': 63,
'ﻼ': 67,
'\ufeff': 0,
'\ufff9': 0,
'\ufffa': 0,
'\ufffb': 0,
'￼': 0,
'�': 113,
'𐌀': 83,
'𐌁': 67,
'𐌂': 62,
'𐌃': 66,
'𐌄': 60,
'𐌅': 60,
'𐌆': 48,
'𐌇': 69,
'𐌈': 99,
'𐌉': 31,
'𐌊': 70,
'𐌋': 60,
'𐌌': 157,
'𐌍': 97,
'𐌎': 102,
'𐌏': 99,
'𐌐': 75,
'𐌑': 96,
'𐌒': 64,
'𐌓': 67,
'𐌔': 49,
'𐌕': 77,
'𐌖': 70,
'𐌗': 75,
'𐌘': 75,
'𐌙': 89,
'𐌚': 58,
'𐌛': 61,
'𐌜': 61,
'𐌝': 48,
'𐌞': 83,
'𐌠': 31,
'𐌡': 83,
'𐌢': 75,
'𐌣': 90,
'𝌀': 99,
'𝌁': 99,
'𝌂': 99,
'𝌃': 99,
'𝌄': 99,
'𝌅': 99,
'𝌆': 99,
'𝌇': 99,
'𝌈': 99,
'𝌉': 99,
'𝌊': 99,
'𝌋': 99,
'𝌌': 99,
'𝌍': 99,
'𝌎': 99,
'𝌏': 99,
'𝌐': 99,
'𝌑': 99,
'𝌒': 99,
'𝌓': 99,
'𝌔': 99,
'𝌕': 99,
'𝌖': 99,
'𝌗': 99,
'𝌘': 99,
'𝌙': 99,
'𝌚': 99,
'𝌛': 99,
'𝌜': 99,
'𝌝': 99,
'𝌞': 99,
'𝌟': 99,
'𝌠': 99,
'𝌡': 99,
'𝌢': 99,
'𝌣': 99,
'𝌤': 99,
'𝌥': 99,
'𝌦': 99,
'𝌧': 99,
'𝌨': 99,
'𝌩': 99,
'𝌪': 99,
'𝌫': 99,
'𝌬': 99,
'𝌭': 99,
'𝌮': 99,
'𝌯': 99,
'𝌰': 99,
'𝌱': 99,
'𝌲': 99,
'𝌳': 99,
'𝌴': 99,
'𝌵': 99,
'𝌶': 99,
'𝌷': 99,
'𝌸': 99,
'𝌹': 99,
'𝌺': 99,
'𝌻': 99,
'𝌼': 99,
'𝌽': 99,
'𝌾': 99,
'𝌿': 99,
'𝍀': 99,
'𝍁': 99,
'𝍂': 99,
'𝍃': 99,
'𝍄': 99,
'𝍅': 99,
'𝍆': 99,
'𝍇': 99,
'𝍈': 99,
'𝍉': 99,
'𝍊': 99,
'𝍋': 99,
'𝍌': 99,
'𝍍': 99,
'𝍎': 99,
'𝍏': 99,
'𝍐': 99,
'𝍑': 99,
'𝍒': 99,
'𝍓': 99,
'𝍔': 99,
'𝍕': 99,
'𝍖': 99,
'𝔸': 81,
'𝔹': 80,
'𝔻': 90,
'𝔼': 80,
'𝔽': 74,
'𝔾': 85,
'𝕀': 43,
'𝕁': 49,
'𝕂': 86,
'𝕃': 72,
'𝕄': 113,
'𝕆': 87,
'𝕊': 70,
'𝕋': 80,
'𝕌': 91,
'𝕍': 80,
'𝕎': 122,
'𝕏': 89,
'𝕐': 79,
'𝕒': 73,
'𝕓': 78,
'𝕔': 60,
'𝕕': 78,
'𝕖': 68,
'𝕗': 54,
'𝕘': 78,
'𝕙': 81,
'𝕚': 39,
'𝕛': 42,
'𝕜': 72,
'𝕝': 41,
'𝕞': 126,
'𝕟': 81,
'𝕠': 67,
'𝕡': 78,
'𝕢': 78,
'𝕣': 54,
'𝕤': 57,
'𝕥': 51,
'𝕦': 81,
'𝕧': 67,
'𝕨': 100,
'𝕩': 74,
'𝕪': 69,
'𝕫': 65,
'𝖠': 75,
'𝖡': 75,
'𝖢': 77,
'𝖣': 85,
'𝖤': 70,
'𝖥': 63,
'𝖦': 85,
'𝖧': 83,
'𝖨': 49,
'𝖩': 38,
'𝖪': 75,
'𝖫': 61,
'𝖬': 95,
'𝖭': 82,
'𝖮': 87,
'𝖯': 66,
'𝖰': 87,
'𝖱': 76,
'𝖲': 70,
'𝖳': 69,
'𝖴': 81,
'𝖵': 75,
'𝖶': 109,
'𝖷': 75,
'𝖸': 69,
'𝖹': 75,
'𝖺': 67,
'𝖻': 70,
'𝖼': 60,
'𝖽': 70,
'𝖾': 68,
'𝖿': 41,
'𝗀': 70,
'𝗁': 70,
'𝗂': 31,
'𝗃': 33,
'𝗄': 64,
'𝗅': 31,
'𝗆': 107,
'𝗇': 70,
'𝗈': 67,
'𝗉': 70,
'𝗊': 70,
'𝗋': 46,
'𝗌': 57,
'𝗍': 43,
'𝗎': 70,
'𝗏': 65,
'𝗐': 90,
'𝗑': 65,
'𝗒': 65,
'𝗓': 58,
'𝟘': 70,
'𝟙': 70,
'𝟚': 70,
'𝟛': 70,
'𝟜': 70,
'𝟝': 70,
'𝟞': 70,
'𝟟': 70,
'𝟠': 70,
'𝟡': 70,
'𝟢': 70,
'𝟣': 70,
'𝟤': 70,
'𝟥': 70,
'𝟦': 70,
'𝟧': 70,
'𝟨': 70,
'𝟩': 70,
'𝟪': 70,
'𝟫': 70,
'𞸀': 31,
'𞸁': 104,
'𞸂': 71,
'𞸃': 49,
'𞸅': 58,
'𞸆': 58,
'𞸇': 71,
'𞸈': 102,
'𞸉': 86,
'𞸊': 91,
'𞸋': 80,
'𞸌': 68,
'𞸍': 81,
'𞸎': 134,
'𞸏': 66,
'𞸐': 114,
'𞸑': 133,
'𞸒': 85,
'𞸓': 58,
'𞸔': 134,
'𞸕': 104,
'𞸖': 104,
'𞸗': 71,
'𞸘': 49,
'𞸙': 133,
'𞸚': 102,
'𞸛': 66,
'𞸜': 104,
'𞸝': 81,
'𞸞': 114,
'𞸟': 85,
'𞸡': 46,
'𞸢': 75,
'𞸤': 68,
'𞸧': 75,
'𞸩': 46,
'𞸪': 68,
'𞸫': 49,
'𞸬': 69,
'𞸭': 46,
'𞸮': 100,
'𞸯': 68,
'𞸰': 55,
'𞸱': 101,
'𞸲': 55,
'𞸴': 100,
'𞸵': 46,
'𞸶': 46,
'𞸷': 75,
'𞸹': 101,
'𞸻': 60,
'𞹡': 72,
'𞹢': 102,
'𞹤': 92,
'𞹧': 102,
'𞹨': 121,
'𞹩': 72,
'𞹪': 86,
'𞹬': 92,
'𞹭': 72,
'𞹮': 126,
'𞹯': 99,
'𞹰': 86,
'𞹱': 127,
'𞹲': 86,
'𞹴': 126,
'𞹵': 72,
'𞹶': 72,
'𞹷': 102,
'𞹹': 127,
'𞹺': 121,
'𞹻': 99,
'𞹼': 72,
'𞹾': 86,
'🀰': 150,
'🀱': 150,
'🀲': 150,
'🀳': 150,
'🀴': 150,
'🀵': 150,
'🀶': 150,
'🀷': 150,
'🀸': 150,
'🀹': 150,
'🀺': 150,
'🀻': 150,
'🀼': 150,
'🀽': 150,
'🀾': 150,
'🀿': 150,
'🁀': 150,
'🁁': 150,
'🁂': 150,
'🁃': 150,
'🁄': 150,
'🁅': 150,
'🁆': 150,
'🁇': 150,
'🁈': 150,
'🁉': 150,
'🁊': 150,
'🁋': 150,
'🁌': 150,
'🁍': 150,
'🁎': 150,
'🁏': 150,
'🁐': 150,
'🁑': 150,
'🁒': 150,
'🁓': 150,
'🁔': 150,
'🁕': 150,
'🁖': 150,
'🁗': 150,
'🁘': 150,
'🁙': 150,
'🁚': 150,
'🁛': 150,
'🁜': 150,
'🁝': 150,
'🁞': 150,
'🁟': 150,
'🁠': 150,
'🁡': 150,
'🁢': 90,
'🁣': 90,
'🁤': 90,
'🁥': 90,
'🁦': 90,
'🁧': 90,
'🁨': 90,
'🁩': 90,
'🁪': 90,
'🁫': 90,
'🁬': 90,
'🁭': 90,
'🁮': 90,
'🁯': 90,
'🁰': 90,
'🁱': 90,
'🁲': 90,
'🁳': 90,
'🁴': 90,
'🁵': 90,
'🁶': 90,
'🁷': 90,
'🁸': 90,
'🁹': 90,
'🁺': 90,
'🁻': 90,
'🁼': 90,
'🁽': 90,
'🁾': 90,
'🁿': 90,
'🂀': 90,
'🂁': 90,
'🂂': 90,
'🂃': 90,
'🂄': 90,
'🂅': 90,
'🂆': 90,
'🂇': 90,
'🂈': 90,
'🂉': 90,
'🂊': 90,
'🂋': 90,
'🂌': 90,
'🂍': 90,
'🂎': 90,
'🂏': 90,
'🂐': 90,
'🂑': 90,
'🂒': 90,
'🂓': 90,
'🂠': 113,
'🂡': 113,
'🂢': 113,
'🂣': 113,
'🂤': 113,
'🂥': 113,
'🂦': 113,
'🂧': 113,
'🂨': 113,
'🂩': 113,
'🂪': 113,
'🂫': 113,
'🂬': 113,
'🂭': 113,
'🂮': 113,
'🂱': 113,
'🂲': 113,
'🂳': 113,
'🂴': 113,
'🂵': 113,
'🂶': 113,
'🂷': 113,
'🂸': 113,
'🂹': 113,
'🂺': 113,
'🂻': 113,
'🂼': 113,
'🂽': 113,
'🂾': 113,
'🃁': 113,
'🃂': 113,
'🃃': 113,
'🃄': 113,
'🃅': 113,
'🃆': 113,
'🃇': 113,
'🃈': 113,
'🃉': 113,
'🃊': 113,
'🃋': 113,
'🃌': 113,
'🃍': 113,
'🃎': 113,
'🃏': 113,
'🃑': 113,
'🃒': 113,
'🃓': 113,
'🃔': 113,
'🃕': 113,
'🃖': 113,
'🃗': 113,
'🃘': 113,
'🃙': 113,
'🃚': 113,
'🃛': 113,
'🃜': 113,
'🃝': 113,
'🃞': 113,
'🃟': 113,
'🐭': 115,
'🐮': 130,
'🐱': 115,
'🐵': 127,
'😀': 115,
'😁': 115,
'😂': 129,
'😃': 115,
'😄': 115,
'😅': 115,
'😆': 115,
'😇': 115,
'😈': 115,
'😉': 115,
'😊': 115,
'😋': 115,
'😌': 115,
'😍': 115,
'😎': 115,
'😏': 115,
'😐': 115,
'😑': 115,
'😒': 115,
'😓': 115,
'😔': 115,
'😕': 115,
'😖': 115,
'😗': 115,
'😘': 115,
'😙': 115,
'😚': 115,
'😛': 115,
'😜': 115,
'😝': 115,
'😞': 115,
'😟': 115,
'😠': 115,
'😡': 115,
'😢': 115,
'😣': 115,
'😥': 115,
'😦': 115,
'😧': 115,
'😨': 115,
'😩': 115,
'😪': 115,
'😫': 115,
'😭': 129,
'😮': 115,
'😯': 115,
'😰': 115,
'😱': 115,
'😲': 115,
'😳': 115,
'😴': 176,
'😵': 115,
'😶': 115,
'😷': 115,
'😸': 115,
'😹': 129,
'😺': 115,
'😻': 115,
'😼': 115,
'😽': 115,
'😾': 115,
'😿': 115,
'🙀': 115},
'kerning-characters': ' '
 '!"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\xa0'
 '¡¢£¤¥¦§¨©ª«¬\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿŒœŠšŸŽžƒˆ˜–—‘’‚“”„†‡•…‰‹›€™ '
 '!"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\xa0'
 '¡¢£¤¥¦§¨©ª«¬\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿŒœŠšŸŽžƒˆ˜–—‘’‚“”„†‡•…‰‹›€™ '
 '!"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\xa0'
 '¡¢£¤¥¦§¨©ª«¬\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿŒœŠšŸŽžƒˆ˜–—‘’‚“”„†‡•…‰‹›€™ '
 '!"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\xa0'
 '¡¢£¤¥¦§¨©ª«¬\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿŒœŠšŸŽžƒˆ˜–—‘’‚“”„†‡•…‰‹›€™',
'kerning-pairs': {' J': 6,
' T': 1,
' Y': 1,
' _': 2,
' j': 2,
' Î': 1,
' Ý': 1,
' ì': 4,
' î': 2,
' ï': 1,
' Ÿ': 1,
' ƒ': 7,
'!J': 6,
'!T': 1,
'!Y': 1,
'!_': 2,
'!j': 2,
'!Î': 1,
'!Ý': 1,
'!ì': 4,
'!î': 2,
'!ï': 1,
'!Ÿ': 1,
'!ƒ': 7,
'"J': 6,
'"T': 1,
'"Y': 1,
'"_': 2,
'"j': 2,
'"Î': 1,
'"Ý': 1,
'"ì': 4,
'"î': 2,
'"ï': 1,
'"Ÿ': 1,
'"ƒ': 7,
'#J': 6,
'#T': 1,
'#Y': 1,
'#_': 2,
'#j': 2,
'#Î': 1,
'#Ý': 1,
'#ì': 4,
'#î': 2,
'#ï': 1,
'#Ÿ': 1,
'#ƒ': 7,
'$J': 6,
'$T': 1,
'$Y': 1,
'$_': 2,
'$j': 2,
'$Î': 1,
'$Ý': 1,
'$ì': 4,
'$î': 2,
'$ï': 1,
'$Ÿ': 1,
'$ƒ': 7,
'%J': 6,
'%T': 1,
'%Y': 1,
'%_': 2,
'%j': 2,
'%Î': 1,
'%Ý': 1,
'%ì': 4,
'%î': 2,
'%ï': 1,
'%Ÿ': 1,
'%ƒ': 7,
'&J': 6,
'&T': 1,
'&Y': 1,
'&_': 2,
'&j': 2,
'&Î': 1,
'&Ý': 1,
'&ì': 4,
'&î': 2,
'&ï': 1,
'&Ÿ': 1,
'&ƒ': 7,
"'J": 6,
"'T": 1,
"'Y": 1,
"'_": 2,
"'j": 2,
"'Î": 1,
"'Ý": 1,
"'ì": 4,
"'î": 2,
"'ï": 1,
"'Ÿ": 1,
"'ƒ": 7,
'(J': 6,
'(T': 1,
'(Y': 1,
'(_': 2,
'(j': 2,
'(Î': 1,
'(Ý': 1,
'(ì': 4,
'(î': 2,
'(ï': 1,
'(Ÿ': 1,
'(ƒ': 7,
')J': 6,
')T': 1,
')Y': 1,
')_': 2,
')j': 2,
')Î': 1,
')Ý': 1,
')ì': 4,
')î': 2,
')ï': 1,
')Ÿ': 1,
')ƒ': 7,
'*J': 6,
'*T': 1,
'*Y': 1,
'*_': 2,
'*j': 2,
'*Î': 1,
'*Ý': 1,
'*ì': 4,
'*î': 2,
'*ï': 1,
'*Ÿ': 1,
'*ƒ': 7,
'+J': 6,
'+T': 1,
'+Y': 1,
'+_': 2,
'+j': 2,
'+Î': 1,
'+Ý': 1,
'+ì': 4,
'+î': 2,
'+ï': 1,
'+Ÿ': 1,
'+ƒ': 7,
',J': 6,
',T': 1,
',Y': 1,
',_': 2,
',j': 2,
',Î': 1,
',Ý': 1,
',ì': 4,
',î': 2,
',ï': 1,
',Ÿ': 1,
',ƒ': 7,
'-G': -1,
'-J': 5,
'-O': -1,
'-Q': -1,
'-T': 1,
'-Y': 1,
'-_': 2,
'-j': 2,
'-o': -1,
'-Î': 1,
'-Ò': -1,
'-Ó': -1,
'-Ô': -1,
'-Õ': -1,
'-Ö': -1,
'-Ý': 1,
'-ì': 4,
'-î': 2,
'-ï': 1,
'-ò': -1,
'-ó': -1,
'-ô': -1,
'-õ': -1,
'-ö': -1,
'-Ÿ': 1,
'-ƒ': 7,
'.J': 6,
'.T': 1,
'.Y': 1,
'._': 2,
'.j': 2,
'.Î': 1,
'.Ý': 1,
'.ì': 4,
'.î': 2,
'.ï': 1,
'.Ÿ': 1,
'.ƒ': 7,
'/ ': 1,
'/!': 1,
'/"': 1,
'/#': 1,
'/$': 1,
'/%': 1,
'/&': 1,
"/'": 1,
'/(': 1,
'/)': 1,
'/*': 1,
'/+': 1,
'/,': 1,
'/-': 1,
'/.': 1,
'//': 1,
'/0': 1,
'/1': 1,
'/2': 1,
'/3': 1,
'/4': 1,
'/5': 1,
'/6': 1,
'/7': 1,
'/8': 1,
'/9': 1,
'/:': 1,
'/;': 1,
'/<': 1,
'/=': 1,
'/>': 1,
'/?': 1,
'/@': 1,
'/A': 1,
'/B': 1,
'/C': 1,
'/D': 1,
'/E': 1,
'/F': 1,
'/G': 1,
'/H': 1,
'/I': 1,
'/J': 7,
'/K': 1,
'/L': 1,
'/M': 1,
'/N': 1,
'/O': 1,
'/P': 1,
'/Q': 1,
'/R': 1,
'/S': 1,
'/T': 2,
'/U': 1,
'/V': 1,
'/W': 1,
'/X': 1,
'/Y': 2,
'/Z': 1,
'/[': 1,
'/\\': 1,
'/]': 1,
'/^': 1,
'/_': 3,
'/`': 1,
'/a': 1,
'/b': 1,
'/c': 1,
'/d': 1,
'/e': 1,
'/f': 1,
'/g': 1,
'/h': 1,
'/i': 1,
'/j': 3,
'/k': 1,
'/l': 1,
'/m': 1,
'/n': 1,
'/o': 1,
'/p': 1,
'/q': 1,
'/r': 1,
'/s': 1,
'/t': 1,
'/u': 1,
'/v': 1,
'/w': 1,
'/x': 1,
'/y': 1,
'/z': 1,
'/{': 1,
'/|': 1,
'/}': 1,
'/~': 1,
'/\xa0': 1,
'/¡': 1,
'/¢': 1,
'/£': 1,
'/¤': 1,
'/¥': 1,
'/¦': 1,
'/§': 1,
'/¨': 1,
'/©': 1,
'/ª': 1,
'/«': 1,
'/¬': 1,
'/\xad': 1,
'/®': 1,
'/¯': 1,
'/°': 1,
'/±': 1,
'/²': 1,
'/³': 1,
'/´': 1,
'/µ': 1,
'/¶': 1,
'/·': 1,
'/¸': 1,
'/¹': 1,
'/º': 1,
'/»': 1,
'/¼': 1,
'/½': 1,
'/¾': 1,
'/¿': 1,
'/À': 1,
'/Á': 1,
'/Â': 1,
'/Ã': 1,
'/Ä': 1,
'/Å': 1,
'/Æ': 1,
'/Ç': 1,
'/È': 1,
'/É': 1,
'/Ê': 1,
'/Ë': 1,
'/Ì': 1,
'/Í': 1,
'/Î': 2,
'/Ï': 1,
'/Ð': 1,
'/Ñ': 1,
'/Ò': 1,
'/Ó': 1,
'/Ô': 1,
'/Õ': 1,
'/Ö': 1,
'/×': 1,
'/Ø': 1,
'/Ù': 1,
'/Ú': 1,
'/Û': 1,
'/Ü': 1,
'/Ý': 2,
'/Þ': 1,
'/ß': 1,
'/à': 1,
'/á': 1,
'/â': 1,
'/ã': 1,
'/ä': 1,
'/å': 1,
'/æ': 1,
'/ç': 1,
'/è': 1,
'/é': 1,
'/ê': 1,
'/ë': 1,
'/ì': 5,
'/í': 1,
'/î': 3,
'/ï': 2,
'/ð': 1,
'/ñ': 1,
'/ò': 1,
'/ó': 1,
'/ô': 1,
'/õ': 1,
'/ö': 1,
'/÷': 1,
'/ø': 1,
'/ù': 1,
'/ú': 1,
'/û': 1,
'/ü': 1,
'/ý': 1,
'/þ': 1,
'/ÿ': 1,
'/Œ': 1,
'/œ': 1,
'/Š': 1,
'/š': 1,
'/Ÿ': 2,
'/Ž': 1,
'/ž': 1,
'/ƒ': 8,
'/ˆ': 1,
'/˜': 1,
'/–': 1,
'/—': 1,
'/‘': 1,
'/’': 1,
'/‚': 1,
'/“': 1,
'/”': 1,
'/„': 1,
'/†': 1,
'/‡': 1,
'/•': 1,
'/…': 1,
'/‰': 1,
'/‹': 1,
'/›': 1,
'/€': 1,
'/™': 1,
'0J': 6,
'0T': 1,
'0Y': 1,
'0_': 2,
'0j': 2,
'0Î': 1,
'0Ý': 1,
'0ì': 4,
'0î': 2,
'0ï': 1,
'0Ÿ': 1,
'0ƒ': 7,
'1J': 6,
'1T': 1,
'1Y': 1,
'1_': 2,
'1j': 2,
'1Î': 1,
'1Ý': 1,
'1ì': 4,
'1î': 2,
'1ï': 1,
'1Ÿ': 1,
'1ƒ': 7,
'2J': 6,
'2T': 1,
'2Y': 1,
'2_': 2,
'2j': 2,
'2Î': 1,
'2Ý': 1,
'2ì': 4,
'2î': 2,
'2ï': 1,
'2Ÿ': 1,
'2ƒ': 7,
'3J': 6,
'3T': 1,
'3Y': 1,
'3_': 2,
'3j': 2,
'3Î': 1,
'3Ý': 1,
'3ì': 4,
'3î': 2,
'3ï': 1,
'3Ÿ': 1,
'3ƒ': 7,
'4J': 6,
'4T': 1,
'4Y': 1,
'4_': 2,
'4j': 2,
'4Î': 1,
'4Ý': 1,
'4ì': 4,
'4î': 2,
'4ï': 1,
'4Ÿ': 1,
'4ƒ': 7,
'5J': 6,
'5T': 1,
'5Y': 1,
'5_': 2,
'5j': 2,
'5Î': 1,
'5Ý': 1,
'5ì': 4,
'5î': 2,
'5ï': 1,
'5Ÿ': 1,
'5ƒ': 7,
'6J': 6,
'6T': 1,
'6Y': 1,
'6_': 2,
'6j': 2,
'6Î': 1,
'6Ý': 1,
'6ì': 4,
'6î': 2,
'6ï': 1,
'6Ÿ': 1,
'6ƒ': 7,
'7J': 6,
'7T': 1,
'7Y': 1,
'7_': 2,
'7j': 2,
'7Î': 1,
'7Ý': 1,
'7ì': 4,
'7î': 2,
'7ï': 1,
'7Ÿ': 1,
'7ƒ': 7,
'8J': 6,
'8T': 1,
'8Y': 1,
'8_': 2,
'8j': 2,
'8Î': 1,
'8Ý': 1,
'8ì': 4,
'8î': 2,
'8ï': 1,
'8Ÿ': 1,
'8ƒ': 7,
'9J': 6,
'9T': 1,
'9Y': 1,
'9_': 2,
'9j': 2,
'9Î': 1,
'9Ý': 1,
'9ì': 4,
'9î': 2,
'9ï': 1,
'9Ÿ': 1,
'9ƒ': 7,
':J': 6,
':T': 1,
':Y': 1,
':_': 2,
':j': 2,
':Î': 1,
':Ý': 1,
':ì': 4,
':î': 2,
':ï': 1,
':Ÿ': 1,
':ƒ': 7,
';J': 6,
';T': 1,
';Y': 1,
';_': 2,
';j': 2,
';Î': 1,
';Ý': 1,
';ì': 4,
';î': 2,
';ï': 1,
';Ÿ': 1,
';ƒ': 7,
'<J': 6,
'<T': 1,
'<Y': 1,
'<_': 2,
'<j': 2,
'<Î': 1,
'<Ý': 1,
'<ì': 4,
'<î': 2,
'<ï': 1,
'<Ÿ': 1,
'<ƒ': 7,
'=J': 6,
'=T': 1,
'=Y': 1,
'=_': 2,
'=j': 2,
'=Î': 1,
'=Ý': 1,
'=ì': 4,
'=î': 2,
'=ï': 1,
'=Ÿ': 1,
'=ƒ': 7,
'>J': 6,
'>T': 1,
'>Y': 1,
'>_': 2,
'>j': 2,
'>Î': 1,
'>Ý': 1,
'>ì': 4,
'>î': 2,
'>ï': 1,
'>Ÿ': 1,
'>ƒ': 7,
'?J': 6,
'?T': 1,
'?Y': 1,
'?_': 2,
'?j': 2,
'?Î': 1,
'?Ý': 1,
'?ì': 4,
'?î': 2,
'?ï': 1,
'?Ÿ': 1,
'?ƒ': 7,
'@J': 6,
'@T': 1,
'@Y': 1,
'@_': 2,
'@j': 2,
'@Î': 1,
'@Ý': 1,
'@ì': 4,
'@î': 2,
'@ï': 1,
'@Ÿ': 1,
'@ƒ': 7,
'AA': -1,
'AJ': 6,
'AT': 1,
'AY': 1,
'A_': 2,
'Aj': 2,
'AÀ': -1,
'AÁ': -1,
'AÂ': -1,
'AÃ': -1,
'AÄ': -1,
'AÎ': 1,
'AÝ': 1,
'Aì': 4,
'Aî': 2,
'Aï': 1,
'AŸ': 1,
'Aƒ': 7,
'A„': -1,
'BJ': 6,
'BT': 1,
'BY': 1,
'B_': 2,
'Bj': 2,
'BÎ': 1,
'BÝ': 1,
'Bì': 4,
'Bî': 2,
'Bï': 1,
'BŸ': 1,
'Bƒ': 7,
'CJ': 6,
'CT': 1,
'CY': 1,
'C_': 2,
'Cj': 2,
'CÎ': 1,
'CÝ': 1,
'Cì': 4,
'Cî': 2,
'Cï': 1,
'CŸ': 1,
'Cƒ': 7,
'C”': -1,
'DJ': 6,
'DT': 1,
'DY': 1,
'D_': 2,
'Dj': 2,
'DÎ': 1,
'DÝ': 1,
'Dì': 4,
'Dî': 2,
'Dï': 1,
'DŸ': 1,
'Dƒ': 7,
'EJ': 6,
'ET': 1,
'EY': 1,
'E_': 2,
'Ej': 2,
'EÎ': 1,
'EÝ': 1,
'Eì': 4,
'Eî': 2,
'Eï': 1,
'EŸ': 1,
'Eƒ': 7,
'FJ': 6,
'FT': 1,
'FY': 1,
'F_': 2,
'Fj': 2,
'FÎ': 1,
'FÝ': 1,
'Fì': 4,
'Fî': 2,
'Fï': 1,
'FŸ': 1,
'Fƒ': 7,
'GJ': 6,
'GT': 1,
'GY': 1,
'G_': 2,
'Gj': 2,
'GÎ': 1,
'GÝ': 1,
'Gì': 4,
'Gî': 2,
'Gï': 1,
'GŸ': 1,
'Gƒ': 7,
'HJ': 6,
'HT': 1,
'HY': 1,
'H_': 2,
'Hj': 2,
'HÎ': 1,
'HÝ': 1,
'Hì': 4,
'Hî': 2,
'Hï': 1,
'HŸ': 1,
'Hƒ': 7,
'IJ': 6,
'IT': 1,
'IY': 1,
'I_': 2,
'Ij': 2,
'IÎ': 1,
'IÝ': 1,
'Iì': 4,
'Iî': 2,
'Iï': 1,
'IŸ': 1,
'Iƒ': 7,
'JJ': 6,
'JT': 1,
'JY': 1,
'J_': 2,
'Jj': 2,
'JÎ': 1,
'JÝ': 1,
'Jì': 4,
'Jî': 2,
'Jï': 1,
'JŸ': 1,
'Jƒ': 7,
'K ': 3,
'K!': 3,
'K"': 3,
'K#': 3,
'K$': 3,
'K%': 3,
'K&': 3,
"K'": 3,
'K(': 3,
'K)': 3,
'K*': 3,
'K+': 3,
'K,': 3,
'K-': 3,
'K.': 3,
'K/': 3,
'K0': 3,
'K1': 3,
'K2': 3,
'K3': 3,
'K4': 3,
'K5': 3,
'K6': 3,
'K7': 3,
'K8': 3,
'K9': 3,
'K:': 3,
'K;': 3,
'K<': 3,
'K=': 3,
'K>': 3,
'K?': 3,
'K@': 3,
'KA': 3,
'KB': 3,
'KC': 3,
'KD': 3,
'KE': 3,
'KF': 3,
'KG': 3,
'KH': 3,
'KI': 3,
'KJ': 9,
'KK': 3,
'KL': 3,
'KM': 3,
'KN': 3,
'KO': 3,
'KP': 3,
'KQ': 3,
'KR': 3,
'KS': 3,
'KT': 4,
'KU': 3,
'KV': 3,
'KW': 3,
'KX': 3,
'KY': 4,
'KZ': 3,
'K[': 3,
'K\\': 3,
'K]': 3,
'K^': 3,
'K_': 5,
'K`': 3,
'Ka': 3,
'Kb': 3,
'Kc': 3,
'Kd': 3,
'Ke': 3,
'Kf': 3,
'Kg': 3,
'Kh': 3,
'Ki': 3,
'Kj': 5,
'Kk': 3,
'Kl': 3,
'Km': 3,
'Kn': 3,
'Ko': 3,
'Kp': 3,
'Kq': 3,
'Kr': 3,
'Ks': 3,
'Kt': 3,
'Ku': 3,
'Kv': 3,
'Kw': 3,
'Kx': 3,
'Ky': 3,
'Kz': 3,
'K{': 3,
'K|': 3,
'K}': 3,
'K~': 3,
'K\xa0': 3,
'K¡': 3,
'K¢': 3,
'K£': 3,
'K¤': 3,
'K¥': 3,
'K¦': 3,
'K§': 3,
'K¨': 3,
'K©': 3,
'Kª': 3,
'K«': 3,
'K¬': 3,
'K\xad': 3,
'K®': 3,
'K¯': 3,
'K°': 3,
'K±': 3,
'K²': 3,
'K³': 3,
'K´': 3,
'Kµ': 3,
'K¶': 3,
'K·': 3,
'K¸': 3,
'K¹': 3,
'Kº': 3,
'K»': 3,
'K¼': 3,
'K½': 3,
'K¾': 3,
'K¿': 3,
'KÀ': 3,
'KÁ': 3,
'KÂ': 3,
'KÃ': 3,
'KÄ': 3,
'KÅ': 3,
'KÆ': 3,
'KÇ': 3,
'KÈ': 3,
'KÉ': 3,
'KÊ': 3,
'KË': 3,
'KÌ': 3,
'KÍ': 3,
'KÎ': 4,
'KÏ': 3,
'KÐ': 3,
'KÑ': 3,
'KÒ': 3,
'KÓ': 3,
'KÔ': 3,
'KÕ': 3,
'KÖ': 3,
'K×': 3,
'KØ': 3,
'KÙ': 3,
'KÚ': 3,
'KÛ': 3,
'KÜ': 3,
'KÝ': 4,
'KÞ': 3,
'Kß': 3,
'Kà': 3,
'Ká': 3,
'Kâ': 3,
'Kã': 3,
'Kä': 3,
'Kå': 3,
'Kæ': 3,
'Kç': 3,
'Kè': 3,
'Ké': 3,
'Kê': 3,
'Kë': 3,
'Kì': 7,
'Kí': 3,
'Kî': 5,
'Kï': 4,
'Kð': 3,
'Kñ': 3,
'Kò': 3,
'Kó': 3,
'Kô': 3,
'Kõ': 3,
'Kö': 3,
'K÷': 3,
'Kø': 3,
'Kù': 3,
'Kú': 3,
'Kû': 3,
'Kü': 3,
'Ký': 3,
'Kþ': 3,
'Kÿ': 3,
'KŒ': 3,
'Kœ': 3,
'KŠ': 3,
'Kš': 3,
'KŸ': 4,
'KŽ': 3,
'Kž': 3,
'Kƒ': 10,
'Kˆ': 3,
'K˜': 3,
'K–': 3,
'K—': 3,
'K‘': 3,
'K’': 3,
'K‚': 3,
'K“': 3,
'K”': 3,
'K„': 3,
'K†': 3,
'K‡': 3,
'K•': 3,
'K…': 3,
'K‰': 3,
'K‹': 3,
'K›': 3,
'K€': 3,
'K™': 3,
'LA': -1,
'LJ': 6,
'LT': 1,
'LY': 1,
'L_': 2,
'Lj': 2,
'LÀ': -1,
'LÁ': -1,
'LÂ': -1,
'LÃ': -1,
'LÄ': -1,
'LÎ': 1,
'LÝ': 1,
'Lì': 4,
'Lî': 2,
'Lï': 1,
'LŸ': 1,
'Lƒ': 7,
'MJ': 6,
'MT': 1,
'MY': 1,
'M_': 2,
'Mj': 2,
'MÎ': 1,
'MÝ': 1,
'Mì': 4,
'Mî': 2,
'Mï': 1,
'MŸ': 1,
'Mƒ': 7,
'NJ': 6,
'NT': 1,
'NY': 1,
'N_': 2,
'Nj': 2,
'NÎ': 1,
'NÝ': 1,
'Nì': 4,
'Nî': 2,
'Nï': 1,
'NŸ': 1,
'Nƒ': 7,
'O-': -1,
'OJ': 6,
'OT': 1,
'OY': 1,
'O_': 2,
'Oj': 2,
'OÎ': 1,
'OÝ': 1,
'Oì': 4,
'Oî': 2,
'Oï': 1,
'OŸ': 1,
'Oƒ': 7,
'PJ': 6,
'PT': 1,
'PY': 1,
'P_': 2,
'Pj': 2,
'PÎ': 1,
'PÝ': 1,
'Pì': 4,
'Pî': 2,
'Pï': 1,
'PŸ': 1,
'Pƒ': 7,
'P“': -1,
'P”': -1,
'Q-': -1,
'QJ': 6,
'QT': 1,
'QY': 1,
'Q_': 2,
'Qj': 2,
'QÎ': 1,
'QÝ': 1,
'Qì': 4,
'Qî': 2,
'Qï': 1,
'QŸ': 1,
'Qƒ': 7,
'RJ': 6,
'RT': 1,
'RY': 1,
'R_': 2,
'Rj': 2,
'RÎ': 1,
'RÝ': 1,
'Rì': 4,
'Rî': 2,
'Rï': 1,
'RŸ': 1,
'Rƒ': 7,
'SA': -1,
'SJ': 6,
'ST': 1,
'SY': 1,
'S_': 2,
'Sj': 2,
'SÀ': -1,
'SÁ': -1,
'SÂ': -1,
'SÃ': -1,
'SÄ': -1,
'SÎ': 1,
'SÝ': 1,
'Sì': 4,
'Sî': 2,
'Sï': 1,
'SŸ': 1,
'Sƒ': 7,
'T ': 1,
'T!': 1,
'T"': 1,
'T#': 1,
'T$': 1,
'T%': 1,
'T&': 1,
"T'": 1,
'T(': 1,
'T)': 1,
'T*': 1,
'T+': 1,
'T,': 1,
'T-': 1,
'T.': 1,
'T/': 1,
'T0': 1,
'T1': 1,
'T2': 1,
'T3': 1,
'T4': 1,
'T5': 1,
'T6': 1,
'T7': 1,
'T8': 1,
'T9': 1,
'T:': 1,
'T;': 1,
'T<': 1,
'T=': 1,
'T>': 1,
'T?': 1,
'T@': 1,
'TA': 1,
'TB': 1,
'TC': 1,
'TD': 1,
'TE': 1,
'TF': 1,
'TG': 1,
'TH': 1,
'TI': 1,
'TJ': 7,
'TK': 1,
'TL': 1,
'TM': 1,
'TN': 1,
'TO': 1,
'TP': 1,
'TQ': 1,
'TR': 1,
'TS': 1,
'TT': 2,
'TU': 1,
'TV': 1,
'TW': 1,
'TX': 1,
'TY': 2,
'TZ': 1,
'T[': 1,
'T\\': 1,
'T]': 1,
'T^': 1,
'T_': 3,
'T`': 1,
'Ta': 1,
'Tb': 1,
'Tc': 1,
'Td': 1,
'Te': 1,
'Tf': 1,
'Tg': 1,
'Th': 1,
'Ti': 1,
'Tj': 3,
'Tk': 1,
'Tl': 1,
'Tm': 1,
'Tn': 1,
'To': 1,
'Tp': 1,
'Tq': 1,
'Tr': 1,
'Ts': 1,
'Tt': 1,
'Tu': 1,
'Tv': 1,
'Tw': 1,
'Tx': 1,
'Ty': 1,
'Tz': 1,
'T{': 1,
'T|': 1,
'T}': 1,
'T~': 1,
'T\xa0': 1,
'T¡': 1,
'T¢': 1,
'T£': 1,
'T¤': 1,
'T¥': 1,
'T¦': 1,
'T§': 1,
'T¨': 1,
'T©': 1,
'Tª': 1,
'T«': 1,
'T¬': 1,
'T\xad': 1,
'T®': 1,
'T¯': 1,
'T°': 1,
'T±': 1,
'T²': 1,
'T³': 1,
'T´': 1,
'Tµ': 1,
'T¶': 1,
'T·': 1,
'T¸': 1,
'T¹': 1,
'Tº': 1,
'T»': 1,
'T¼': 1,
'T½': 1,
'T¾': 1,
'T¿': 1,
'TÀ': 1,
'TÁ': 1,
'TÂ': 1,
'TÃ': 1,
'TÄ': 1,
'TÅ': 1,
'TÆ': 1,
'TÇ': 1,
'TÈ': 1,
'TÉ': 1,
'TÊ': 1,
'TË': 1,
'TÌ': 1,
'TÍ': 1,
'TÎ': 2,
'TÏ': 1,
'TÐ': 1,
'TÑ': 1,
'TÒ': 1,
'TÓ': 1,
'TÔ': 1,
'TÕ': 1,
'TÖ': 1,
'T×': 1,
'TØ': 1,
'TÙ': 1,
'TÚ': 1,
'TÛ': 1,
'TÜ': 1,
'TÝ': 2,
'TÞ': 1,
'Tß': 1,
'Tà': 1,
'Tá': 1,
'Tâ': 1,
'Tã': 1,
'Tä': 1,
'Tå': 1,
'Tæ': 1,
'Tç': 1,
'Tè': 1,
'Té': 1,
'Tê': 1,
'Të': 1,
'Tì': 5,
'Tí': 1,
'Tî': 3,
'Tï': 2,
'Tð': 1,
'Tñ': 1,
'Tò': 1,
'Tó': 1,
'Tô': 1,
'Tõ': 1,
'Tö': 1,
'T÷': 1,
'Tø': 1,
'Tù': 1,
'Tú': 1,
'Tû': 1,
'Tü': 1,
'Tý': 1,
'Tþ': 1,
'Tÿ': 1,
'TŒ': 1,
'Tœ': 1,
'TŠ': 1,
'Tš': 1,
'TŸ': 2,
'TŽ': 1,
'Tž': 1,
'Tƒ': 8,
'Tˆ': 1,
'T˜': 1,
'T–': 1,
'T—': 1,
'T‘': 1,
'T’': 1,
'T‚': 1,
'T“': 1,
'T”': 1,
'T„': 1,
'T†': 1,
'T‡': 1,
'T•': 1,
'T…': 1,
'T‰': 1,
'T‹': 1,
'T›': 1,
'T€': 1,
'T™': 1,
'UJ': 6,
'UT': 1,
'UY': 1,
'U_': 2,
'Uj': 2,
'UÎ': 1,
'UÝ': 1,
'Uì': 4,
'Uî': 2,
'Uï': 1,
'UŸ': 1,
'Uƒ': 7,
'VJ': 6,
'VT': 1,
'VY': 1,
'V_': 2,
'Vj': 2,
'VÎ': 1,
'VÝ': 1,
'Vì': 4,
'Vî': 2,
'Vï': 1,
'VŸ': 1,
'Vƒ': 7,
'WJ': 6,
'WT': 1,
'WY': 1,
'W_': 2,
'Wj': 2,
'WÎ': 1,
'WÝ': 1,
'Wì': 4,
'Wî': 2,
'Wï': 1,
'WŸ': 1,
'Wƒ': 7,
'XJ': 6,
'XT': 1,
'XY': 1,
'X_': 2,
'Xj': 2,
'XÎ': 1,
'XÝ': 1,
'Xì': 4,
'Xî': 2,
'Xï': 1,
'XŸ': 1,
'Xƒ': 7,
'Y ': 1,
'Y!': 1,
'Y"': 1,
'Y#': 1,
'Y$': 1,
'Y%': 1,
'Y&': 1,
"Y'": 1,
'Y(': 1,
'Y)': 1,
'Y*': 1,
'Y+': 1,
'Y,': 1,
'Y-': 1,
'Y.': 1,
'Y/': 1,
'Y0': 1,
'Y1': 1,
'Y2': 1,
'Y3': 1,
'Y4': 1,
'Y5': 1,
'Y6': 1,
'Y7': 1,
'Y8': 1,
'Y9': 1,
'Y:': 1,
'Y;': 1,
'Y<': 1,
'Y=': 1,
'Y>': 1,
'Y?': 1,
'Y@': 1,
'YA': 1,
'YB': 1,
'YC': 1,
'YD': 1,
'YE': 1,
'YF': 1,
'YG': 1,
'YH': 1,
'YI': 1,
'YJ': 7,
'YK': 1,
'YL': 1,
'YM': 1,
'YN': 1,
'YO': 1,
'YP': 1,
'YQ': 1,
'YR': 1,
'YS': 1,
'YT': 2,
'YU': 1,
'YV': 1,
'YW': 1,
'YX': 1,
'YY': 2,
'YZ': 1,
'Y[': 1,
'Y\\': 1,
'Y]': 1,
'Y^': 1,
'Y_': 3,
'Y`': 1,
'Ya': 1,
'Yb': 1,
'Yc': 1,
'Yd': 1,
'Ye': 1,
'Yf': 1,
'Yg': 1,
'Yh': 1,
'Yi': 1,
'Yj': 3,
'Yk': 1,
'Yl': 1,
'Ym': 1,
'Yn': 1,
'Yo': 1,
'Yp': 1,
'Yq': 1,
'Yr': 1,
'Ys': 1,
'Yt': 1,
'Yu': 1,
'Yv': 1,
'Yw': 1,
'Yx': 1,
'Yy': 1,
'Yz': 1,
'Y{': 1,
'Y|': 1,
'Y}': 1,
'Y~': 1,
'Y\xa0': 1,
'Y¡': 1,
'Y¢': 1,
'Y£': 1,
'Y¤': 1,
'Y¥': 1,
'Y¦': 1,
'Y§': 1,
'Y¨': 1,
'Y©': 1,
'Yª': 1,
'Y«': 1,
'Y¬': 1,
'Y\xad': 1,
'Y®': 1,
'Y¯': 1,
'Y°': 1,
'Y±': 1,
'Y²': 1,
'Y³': 1,
'Y´': 1,
'Yµ': 1,
'Y¶': 1,
'Y·': 1,
'Y¸': 1,
'Y¹': 1,
'Yº': 1,
'Y»': 1,
'Y¼': 1,
'Y½': 1,
'Y¾': 1,
'Y¿': 1,
'YÀ': 1,
'YÁ': 1,
'YÂ': 1,
'YÃ': 1,
'YÄ': 1,
'YÅ': 1,
'YÆ': 1,
'YÇ': 1,
'YÈ': 1,
'YÉ': 1,
'YÊ': 1,
'YË': 1,
'YÌ': 1,
'YÍ': 1,
'YÎ': 2,
'YÏ': 1,
'YÐ': 1,
'YÑ': 1,
'YÒ': 1,
'YÓ': 1,
'YÔ': 1,
'YÕ': 1,
'YÖ': 1,
'Y×': 1,
'YØ': 1,
'YÙ': 1,
'YÚ': 1,
'YÛ': 1,
'YÜ': 1,
'YÝ': 2,
'YÞ': 1,
'Yß': 1,
'Yà': 1,
'Yá': 1,
'Yâ': 1,
'Yã': 1,
'Yä': 1,
'Yå': 1,
'Yæ': 1,
'Yç': 1,
'Yè': 1,
'Yé': 1,
'Yê': 1,
'Yë': 1,
'Yì': 5,
'Yí': 1,
'Yî': 3,
'Yï': 2,
'Yð': 1,
'Yñ': 1,
'Yò': 1,
'Yó': 1,
'Yô': 1,
'Yõ': 1,
'Yö': 1,
'Y÷': 1,
'Yø': 1,
'Yù': 1,
'Yú': 1,
'Yû': 1,
'Yü': 1,
'Yý': 1,
'Yþ': 1,
'Yÿ': 1,
'YŒ': 1,
'Yœ': 1,
'YŠ': 1,
'Yš': 1,
'YŸ': 2,
'YŽ': 1,
'Yž': 1,
'Yƒ': 8,
'Yˆ': 1,
'Y˜': 1,
'Y–': 1,
'Y—': 1,
'Y‘': 1,
'Y’': 1,
'Y‚': 1,
'Y“': 1,
'Y”': 1,
'Y„': 1,
'Y†': 1,
'Y‡': 1,
'Y•': 1,
'Y…': 1,
'Y‰': 1,
'Y‹': 1,
'Y›': 1,
'Y€': 1,
'Y™': 1,
'ZJ': 6,
'ZT': 1,
'ZY': 1,
'Z_': 2,
'Zj': 2,
'ZÎ': 1,
'ZÝ': 1,
'Zì': 4,
'Zî': 2,
'Zï': 1,
'ZŸ': 1,
'Zƒ': 7,
'[J': 6,
'[T': 1,
'[Y': 1,
'[_': 2,
'[j': 2,
'[Î': 1,
'[Ý': 1,
'[ì': 4,
'[î': 2,
'[ï': 1,
'[Ÿ': 1,
'[ƒ': 7,
'\\ ': 1,
'\\!': 1,
'\\"': 1,
'\\#': 1,
'\\$': 1,
'\\%': 1,
'\\&': 1,
"\\'": 1,
'\\(': 1,
'\\)': 1,
'\\*': 1,
'\\+': 1,
'\\,': 1,
'\\-': 1,
'\\.': 1,
'\\/': 1,
'\\0': 1,
'\\1': 1,
'\\2': 1,
'\\3': 1,
'\\4': 1,
'\\5': 1,
'\\6': 1,
'\\7': 1,
'\\8': 1,
'\\9': 1,
'\\:': 1,
'\\;': 1,
'\\<': 1,
'\\=': 1,
'\\>': 1,
'\\?': 1,
'\\@': 1,
'\\A': 1,
'\\B': 1,
'\\C': 1,
'\\D': 1,
'\\E': 1,
'\\F': 1,
'\\G': 1,
'\\H': 1,
'\\I': 1,
'\\J': 7,
'\\K': 1,
'\\L': 1,
'\\M': 1,
'\\N': 1,
'\\O': 1,
'\\P': 1,
'\\Q': 1,
'\\R': 1,
'\\S': 1,
'\\T': 2,
'\\U': 1,
'\\V': 1,
'\\W': 1,
'\\X': 1,
'\\Y': 2,
'\\Z': 1,
'\\[': 1,
'\\\\': 1,
'\\]': 1,
'\\^': 1,
'\\_': 3,
'\\`': 1,
'\\a': 1,
'\\b': 1,
'\\c': 1,
'\\d': 1,
'\\e': 1,
'\\f': 1,
'\\g': 1,
'\\h': 1,
'\\i': 1,
'\\j': 3,
'\\k': 1,
'\\l': 1,
'\\m': 1,
'\\n': 1,
'\\o': 1,
'\\p': 1,
'\\q': 1,
'\\r': 1,
'\\s': 1,
'\\t': 1,
'\\u': 1,
'\\v': 1,
'\\w': 1,
'\\x': 1,
'\\y': 1,
'\\z': 1,
'\\{': 1,
'\\|': 1,
'\\}': 1,
'\\~': 1,
'\\\xa0': 1,
'\\¡': 1,
'\\¢': 1,
'\\£': 1,
'\\¤': 1,
'\\¥': 1,
'\\¦': 1,
'\\§': 1,
'\\¨': 1,
'\\©': 1,
'\\ª': 1,
'\\«': 1,
'\\¬': 1,
'\\\xad': 1,
'\\®': 1,
'\\¯': 1,
'\\°': 1,
'\\±': 1,
'\\²': 1,
'\\³': 1,
'\\´': 1,
'\\µ': 1,
'\\¶': 1,
'\\·': 1,
'\\¸': 1,
'\\¹': 1,
'\\º': 1,
'\\»': 1,
'\\¼': 1,
'\\½': 1,
'\\¾': 1,
'\\¿': 1,
'\\À': 1,
'\\Á': 1,
'\\Â': 1,
'\\Ã': 1,
'\\Ä': 1,
'\\Å': 1,
'\\Æ': 1,
'\\Ç': 1,
'\\È': 1,
'\\É': 1,
'\\Ê': 1,
'\\Ë': 1,
'\\Ì': 1,
'\\Í': 1,
'\\Î': 2,
'\\Ï': 1,
'\\Ð': 1,
'\\Ñ': 1,
'\\Ò': 1,
'\\Ó': 1,
'\\Ô': 1,
'\\Õ': 1,
'\\Ö': 1,
'\\×': 1,
'\\Ø': 1,
'\\Ù': 1,
'\\Ú': 1,
'\\Û': 1,
'\\Ü': 1,
'\\Ý': 2,
'\\Þ': 1,
'\\ß': 1,
'\\à': 1,
'\\á': 1,
'\\â': 1,
'\\ã': 1,
'\\ä': 1,
'\\å': 1,
'\\æ': 1,
'\\ç': 1,
'\\è': 1,
'\\é': 1,
'\\ê': 1,
'\\ë': 1,
'\\ì': 5,
'\\í': 1,
'\\î': 3,
'\\ï': 2,
'\\ð': 1,
'\\ñ': 1,
'\\ò': 1,
'\\ó': 1,
'\\ô': 1,
'\\õ': 1,
'\\ö': 1,
'\\÷': 1,
'\\ø': 1,
'\\ù': 1,
'\\ú': 1,
'\\û': 1,
'\\ü': 1,
'\\ý': 1,
'\\þ': 1,
'\\ÿ': 1,
'\\Œ': 1,
'\\œ': 1,
'\\Š': 1,
'\\š': 1,
'\\Ÿ': 2,
'\\Ž': 1,
'\\ž': 1,
'\\ƒ': 8,
'\\ˆ': 1,
'\\˜': 1,
'\\–': 1,
'\\—': 1,
'\\‘': 1,
'\\’': 1,
'\\‚': 1,
'\\“': 1,
'\\”': 1,
'\\„': 1,
'\\†': 1,
'\\‡': 1,
'\\•': 1,
'\\…': 1,
'\\‰': 1,
'\\‹': 1,
'\\›': 1,
'\\€': 1,
'\\™': 1,
']J': 6,
']T': 1,
']Y': 1,
']_': 2,
']j': 2,
']Î': 1,
']Ý': 1,
']ì': 4,
']î': 2,
']ï': 1,
']Ÿ': 1,
']ƒ': 7,
'^J': 6,
'^T': 1,
'^Y': 1,
'^_': 2,
'^j': 2,
'^Î': 1,
'^Ý': 1,
'^ì': 4,
'^î': 2,
'^ï': 1,
'^Ÿ': 1,
'^ƒ': 7,
'_ ': 2,
'_!': 2,
'_"': 2,
'_#': 2,
'_$': 2,
'_%': 2,
'_&': 2,
"_'": 2,
'_(': 2,
'_)': 2,
'_*': 2,
'_+': 2,
'_,': 2,
'_-': 2,
'_.': 2,
'_/': 2,
'_0': 2,
'_1': 2,
'_2': 2,
'_3': 2,
'_4': 2,
'_5': 2,
'_6': 2,
'_7': 2,
'_8': 2,
'_9': 2,
'_:': 2,
'_;': 2,
'_<': 2,
'_=': 2,
'_>': 2,
'_?': 2,
'_@': 2,
'_A': 2,
'_B': 2,
'_C': 2,
'_D': 2,
'_E': 2,
'_F': 2,
'_G': 2,
'_H': 2,
'_I': 2,
'_J': 8,
'_K': 2,
'_L': 2,
'_M': 2,
'_N': 2,
'_O': 2,
'_P': 2,
'_Q': 2,
'_R': 2,
'_S': 2,
'_T': 3,
'_U': 2,
'_V': 2,
'_W': 2,
'_X': 2,
'_Y': 3,
'_Z': 2,
'_[': 2,
'_\\': 2,
'_]': 2,
'_^': 2,
'__': 4,
'_`': 2,
'_a': 2,
'_b': 2,
'_c': 2,
'_d': 2,
'_e': 2,
'_f': 2,
'_g': 2,
'_h': 2,
'_i': 2,
'_j': 4,
'_k': 2,
'_l': 2,
'_m': 2,
'_n': 2,
'_o': 2,
'_p': 2,
'_q': 2,
'_r': 2,
'_s': 2,
'_t': 2,
'_u': 2,
'_v': 2,
'_w': 2,
'_x': 2,
'_y': 2,
'_z': 2,
'_{': 2,
'_|': 2,
'_}': 2,
'_~': 2,
'_\xa0': 2,
'_¡': 2,
'_¢': 2,
'_£': 2,
'_¤': 2,
'_¥': 2,
'_¦': 2,
'_§': 2,
'_¨': 2,
'_©': 2,
'_ª': 2,
'_«': 2,
'_¬': 2,
'_\xad': 2,
'_®': 2,
'_¯': 2,
'_°': 2,
'_±': 2,
'_²': 2,
'_³': 2,
'_´': 2,
'_µ': 2,
'_¶': 2,
'_·': 2,
'_¸': 2,
'_¹': 2,
'_º': 2,
'_»': 2,
'_¼': 2,
'_½': 2,
'_¾': 2,
'_¿': 2,
'_À': 2,
'_Á': 2,
'_Â': 2,
'_Ã': 2,
'_Ä': 2,
'_Å': 2,
'_Æ': 2,
'_Ç': 2,
'_È': 2,
'_É': 2,
'_Ê': 2,
'_Ë': 2,
'_Ì': 2,
'_Í': 2,
'_Î': 3,
'_Ï': 2,
'_Ð': 2,
'_Ñ': 2,
'_Ò': 2,
'_Ó': 2,
'_Ô': 2,
'_Õ': 2,
'_Ö': 2,
'_×': 2,
'_Ø': 2,
'_Ù': 2,
'_Ú': 2,
'_Û': 2,
'_Ü': 2,
'_Ý': 3,
'_Þ': 2,
'_ß': 2,
'_à': 2,
'_á': 2,
'_â': 2,
'_ã': 2,
'_ä': 2,
'_å': 2,
'_æ': 2,
'_ç': 2,
'_è': 2,
'_é': 2,
'_ê': 2,
'_ë': 2,
'_ì': 6,
'_í': 2,
'_î': 4,
'_ï': 3,
'_ð': 2,
'_ñ': 2,
'_ò': 2,
'_ó': 2,
'_ô': 2,
'_õ': 2,
'_ö': 2,
'_÷': 2,
'_ø': 2,
'_ù': 2,
'_ú': 2,
'_û': 2,
'_ü': 2,
'_ý': 2,
'_þ': 2,
'_ÿ': 2,
'_Œ': 2,
'_œ': 2,
'_Š': 2,
'_š': 2,
'_Ÿ': 3,
'_Ž': 2,
'_ž': 2,
'_ƒ': 9,
'_ˆ': 2,
'_˜': 2,
'_–': 2,
'_—': 2,
'_‘': 2,
'_’': 2,
'_‚': 2,
'_“': 2,
'_”': 2,
'_„': 2,
'_†': 2,
'_‡': 2,
'_•': 2,
'_…': 2,
'_‰': 2,
'_‹': 2,
'_›': 2,
'_€': 2,
'_™': 2,
'`J': 6,
'`T': 1,
'`Y': 1,
'`_': 2,
'`j': 2,
'`Î': 1,
'`Ý': 1,
'`ì': 4,
'`î': 2,
'`ï': 1,
'`Ÿ': 1,
'`ƒ': 7,
'aJ': 6,
'aT': 1,
'aY': 1,
'a_': 2,
'aj': 2,
'aÎ': 1,
'aÝ': 1,
'aì': 4,
'aî': 2,
'aï': 1,
'aŸ': 1,
'aƒ': 7,
'bJ': 6,
'bT': 1,
'bY': 1,
'b_': 2,
'bj': 2,
'bÎ': 1,
'bÝ': 1,
'bì': 4,
'bî': 2,
'bï': 1,
'bŸ': 1,
'bƒ': 7,
'cJ': 6,
'cT': 1,
'cY': 1,
'c_': 2,
'cj': 2,
'cÎ': 1,
'cÝ': 1,
'cì': 4,
'cî': 2,
'cï': 1,
'cŸ': 1,
'cƒ': 7,
'dJ': 6,
'dT': 1,
'dY': 1,
'd_': 2,
'dj': 2,
'dÎ': 1,
'dÝ': 1,
'dì': 4,
'dî': 2,
'dï': 1,
'dŸ': 1,
'dƒ': 7,
'eJ': 6,
'eT': 1,
'eY': 1,
'e_': 2,
'ej': 2,
'eÎ': 1,
'eÝ': 1,
'eì': 4,
'eî': 2,
'eï': 1,
'eŸ': 1,
'eƒ': 7,
'f ': 2,
'f!': 2,
'f"': 2,
'f#': 2,
'f$': 2,
'f%': 2,
'f&': 2,
"f'": 2,
'f(': 2,
'f)': 2,
'f*': 2,
'f+': 2,
'f,': 2,
'f-': 2,
'f.': 2,
'f/': 2,
'f0': 2,
'f1': 2,
'f2': 2,
'f3': 2,
'f4': 2,
'f5': 2,
'f6': 2,
'f7': 2,
'f8': 2,
'f9': 2,
'f:': 2,
'f;': 2,
'f<': 2,
'f=': 2,
'f>': 2,
'f?': 2,
'f@': 2,
'fA': 2,
'fB': 2,
'fC': 2,
'fD': 2,
'fE': 2,
'fF': 2,
'fG': 2,
'fH': 2,
'fI': 2,
'fJ': 8,
'fK': 2,
'fL': 2,
'fM': 2,
'fN': 2,
'fO': 2,
'fP': 2,
'fQ': 2,
'fR': 2,
'fS': 2,
'fT': 3,
'fU': 2,
'fV': 2,
'fW': 2,
'fX': 2,
'fY': 3,
'fZ': 2,
'f[': 2,
'f\\': 2,
'f]': 2,
'f^': 2,
'f_': 4,
'f`': 2,
'fa': 2,
'fb': 2,
'fc': 2,
'fd': 2,
'fe': 2,
'ff': 2,
'fg': 2,
'fh': 2,
'fi': 2,
'fj': 4,
'fk': 2,
'fl': 2,
'fm': 2,
'fn': 2,
'fo': 2,
'fp': 2,
'fq': 2,
'fr': 2,
'fs': 2,
'ft': 2,
'fu': 2,
'fv': 2,
'fw': 2,
'fx': 2,
'fy': 2,
'fz': 2,
'f{': 2,
'f|': 2,
'f}': 2,
'f~': 2,
'f\xa0': 2,
'f¡': 2,
'f¢': 2,
'f£': 2,
'f¤': 2,
'f¥': 2,
'f¦': 2,
'f§': 2,
'f¨': 2,
'f©': 2,
'fª': 2,
'f«': 2,
'f¬': 2,
'f\xad': 2,
'f®': 2,
'f¯': 2,
'f°': 2,
'f±': 2,
'f²': 2,
'f³': 2,
'f´': 2,
'fµ': 2,
'f¶': 2,
'f·': 2,
'f¸': 2,
'f¹': 2,
'fº': 2,
'f»': 2,
'f¼': 2,
'f½': 2,
'f¾': 2,
'f¿': 2,
'fÀ': 2,
'fÁ': 2,
'fÂ': 2,
'fÃ': 2,
'fÄ': 2,
'fÅ': 2,
'fÆ': 2,
'fÇ': 2,
'fÈ': 2,
'fÉ': 2,
'fÊ': 2,
'fË': 2,
'fÌ': 2,
'fÍ': 2,
'fÎ': 3,
'fÏ': 2,
'fÐ': 2,
'fÑ': 2,
'fÒ': 2,
'fÓ': 2,
'fÔ': 2,
'fÕ': 2,
'fÖ': 2,
'f×': 2,
'fØ': 2,
'fÙ': 2,
'fÚ': 2,
'fÛ': 2,
'fÜ': 2,
'fÝ': 3,
'fÞ': 2,
'fß': 2,
'fà': 2,
'fá': 2,
'fâ': 2,
'fã': 2,
'fä': 2,
'få': 2,
'fæ': 2,
'fç': 2,
'fè': 2,
'fé': 2,
'fê': 2,
'fë': 2,
'fì': 6,
'fí': 2,
'fî': 4,
'fï': 3,
'fð': 2,
'fñ': 2,
'fò': 2,
'fó': 2,
'fô': 2,
'fõ': 2,
'fö': 2,
'f÷': 2,
'fø': 2,
'fù': 2,
'fú': 2,
'fû': 2,
'fü': 2,
'fý': 2,
'fþ': 2,
'fÿ': 2,
'fŒ': 2,
'fœ': 2,
'fŠ': 2,
'fš': 2,
'fŸ': 3,
'fŽ': 2,
'fž': 2,
'fƒ': 9,
'fˆ': 2,
'f˜': 2,
'f–': 2,
'f—': 2,
'f‘': 2,
'f’': 2,
'f‚': 2,
'f“': 1,
'f”': 2,
'f„': 2,
'f†': 2,
'f‡': 2,
'f•': 2,
'f…': 2,
'f‰': 2,
'f‹': 2,
'f›': 2,
'f€': 2,
'f™': 2,
'gJ': 6,
'gT': 1,
'gY': 1,
'g_': 2,
'gj': 2,
'gÎ': 1,
'gÝ': 1,
'gì': 4,
'gî': 2,
'gï': 1,
'gŸ': 1,
'gƒ': 7,
'hJ': 6,
'hT': 1,
'hY': 1,
'h_': 2,
'hj': 2,
'hÎ': 1,
'hÝ': 1,
'hì': 4,
'hî': 2,
'hï': 1,
'hŸ': 1,
'hƒ': 7,
'iJ': 6,
'iT': 1,
'iY': 1,
'i_': 2,
'ij': 2,
'iÎ': 1,
'iÝ': 1,
'iì': 4,
'iî': 2,
'iï': 1,
'iŸ': 1,
'iƒ': 7,
'jJ': 6,
'jT': 1,
'jY': 1,
'j_': 2,
'jj': 2,
'jÎ': 1,
'jÝ': 1,
'jì': 4,
'jî': 2,
'jï': 1,
'jŸ': 1,
'jƒ': 7,
'kJ': 6,
'kT': 1,
'kY': 1,
'k_': 2,
'kj': 2,
'kÎ': 1,
'kÝ': 1,
'kì': 4,
'kî': 2,
'kï': 1,
'kŸ': 1,
'kƒ': 7,
'lJ': 6,
'lT': 1,
'lY': 1,
'l_': 2,
'lj': 2,
'lÎ': 1,
'lÝ': 1,
'lì': 4,
'lî': 2,
'lï': 1,
'lŸ': 1,
'lƒ': 7,
'mJ': 6,
'mT': 1,
'mY': 1,
'm_': 2,
'mj': 2,
'mÎ': 1,
'mÝ': 1,
'mì': 4,
'mî': 2,
'mï': 1,
'mŸ': 1,
'mƒ': 7,
'nJ': 6,
'nT': 1,
'nY': 1,
'n_': 2,
'nj': 2,
'nÎ': 1,
'nÝ': 1,
'nì': 4,
'nî': 2,
'nï': 1,
'nŸ': 1,
'nƒ': 7,
'o-': -1,
'oJ': 6,
'oT': 1,
'oY': 1,
'o_': 2,
'oj': 2,
'oÎ': 1,
'oÝ': 1,
'oì': 4,
'oî': 2,
'oï': 1,
'oŸ': 1,
'oƒ': 7,
'pJ': 6,
'pT': 1,
'pY': 1,
'p_': 2,
'pj': 2,
'pÎ': 1,
'pÝ': 1,
'pì': 4,
'pî': 2,
'pï': 1,
'pŸ': 1,
'pƒ': 7,
'qJ': 6,
'qT': 1,
'qY': 1,
'q_': 2,
'qj': 2,
'qÎ': 1,
'qÝ': 1,
'qì': 4,
'qî': 2,
'qï': 1,
'qŸ': 1,
'qƒ': 7,
'r ': 1,
'r!': 1,
'r"': 1,
'r#': 1,
'r$': 1,
'r%': 1,
'r&': 1,
"r'": 1,
'r(': 1,
'r)': 1,
'r*': 1,
'r+': 1,
'r,': 1,
'r-': 1,
'r.': 1,
'r/': 1,
'r0': 1,
'r1': 1,
'r2': 1,
'r3': 1,
'r4': 1,
'r5': 1,
'r6': 1,
'r7': 1,
'r8': 1,
'r9': 1,
'r:': 1,
'r;': 1,
'r<': 1,
'r=': 1,
'r>': 1,
'r?': 1,
'r@': 1,
'rA': 1,
'rB': 1,
'rC': 1,
'rD': 1,
'rE': 1,
'rF': 1,
'rG': 1,
'rH': 1,
'rI': 1,
'rJ': 7,
'rK': 1,
'rL': 1,
'rM': 1,
'rN': 1,
'rO': 1,
'rP': 1,
'rQ': 1,
'rR': 1,
'rS': 1,
'rT': 2,
'rU': 1,
'rV': 1,
'rW': 1,
'rX': 1,
'rY': 2,
'rZ': 1,
'r[': 1,
'r\\': 1,
'r]': 1,
'r^': 1,
'r_': 3,
'r`': 1,
'ra': 1,
'rb': 1,
'rc': 1,
'rd': 1,
're': 1,
'rf': 1,
'rg': 1,
'rh': 1,
'ri': 1,
'rj': 3,
'rk': 1,
'rl': 1,
'rm': 1,
'rn': 1,
'ro': 1,
'rp': 1,
'rq': 1,
'rr': 1,
'rs': 1,
'rt': 1,
'ru': 1,
'rv': 1,
'rw': 1,
'rx': 1,
'ry': 1,
'rz': 1,
'r{': 1,
'r|': 1,
'r}': 1,
'r~': 1,
'r\xa0': 1,
'r¡': 1,
'r¢': 1,
'r£': 1,
'r¤': 1,
'r¥': 1,
'r¦': 1,
'r§': 1,
'r¨': 1,
'r©': 1,
'rª': 1,
'r«': 1,
'r¬': 1,
'r\xad': 1,
'r®': 1,
'r¯': 1,
'r°': 1,
'r±': 1,
'r²': 1,
'r³': 1,
'r´': 1,
'rµ': 1,
'r¶': 1,
'r·': 1,
'r¸': 1,
'r¹': 1,
'rº': 1,
'r»': 1,
'r¼': 1,
'r½': 1,
'r¾': 1,
'r¿': 1,
'rÀ': 1,
'rÁ': 1,
'rÂ': 1,
'rÃ': 1,
'rÄ': 1,
'rÅ': 1,
'rÆ': 1,
'rÇ': 1,
'rÈ': 1,
'rÉ': 1,
'rÊ': 1,
'rË': 1,
'rÌ': 1,
'rÍ': 1,
'rÎ': 2,
'rÏ': 1,
'rÐ': 1,
'rÑ': 1,
'rÒ': 1,
'rÓ': 1,
'rÔ': 1,
'rÕ': 1,
'rÖ': 1,
'r×': 1,
'rØ': 1,
'rÙ': 1,
'rÚ': 1,
'rÛ': 1,
'rÜ': 1,
'rÝ': 2,
'rÞ': 1,
'rß': 1,
'rà': 1,
'rá': 1,
'râ': 1,
'rã': 1,
'rä': 1,
'rå': 1,
'ræ': 1,
'rç': 1,
'rè': 1,
'ré': 1,
'rê': 1,
'rë': 1,
'rì': 5,
'rí': 1,
'rî': 3,
'rï': 2,
'rð': 1,
'rñ': 1,
'rò': 1,
'ró': 1,
'rô': 1,
'rõ': 1,
'rö': 1,
'r÷': 1,
'rø': 1,
'rù': 1,
'rú': 1,
'rû': 1,
'rü': 1,
'rý': 1,
'rþ': 1,
'rÿ': 1,
'rŒ': 1,
'rœ': 1,
'rŠ': 1,
'rš': 1,
'rŸ': 2,
'rŽ': 1,
'rž': 1,
'rƒ': 8,
'rˆ': 1,
'r˜': 1,
'r–': 1,
'r—': 1,
'r‘': 1,
'r’': 1,
'r‚': 1,
'r“': 1,
'r„': 1,
'r†': 1,
'r‡': 1,
'r•': 1,
'r…': 1,
'r‰': 1,
'r‹': 1,
'r›': 1,
'r€': 1,
'r™': 1,
'sJ': 6,
'sT': 1,
'sY': 1,
's_': 2,
'sj': 2,
'sÎ': 1,
'sÝ': 1,
'sì': 4,
'sî': 2,
'sï': 1,
'sŸ': 1,
'sƒ': 7,
'tJ': 6,
'tT': 1,
'tY': 1,
't_': 2,
'tj': 2,
'tÎ': 1,
'tÝ': 1,
'tì': 4,
'tî': 2,
'tï': 1,
'tŸ': 1,
'tƒ': 7,
'uJ': 6,
'uT': 1,
'uY': 1,
'u_': 2,
'uj': 2,
'uÎ': 1,
'uÝ': 1,
'uì': 4,
'uî': 2,
'uï': 1,
'uŸ': 1,
'uƒ': 7,
'vJ': 6,
'vT': 1,
'vY': 1,
'v_': 2,
'vj': 2,
'vÎ': 1,
'vÝ': 1,
'vì': 4,
'vî': 2,
'vï': 1,
'vŸ': 1,
'vƒ': 7,
'wJ': 6,
'wT': 1,
'wY': 1,
'w_': 2,
'wj': 2,
'wÎ': 1,
'wÝ': 1,
'wì': 4,
'wî': 2,
'wï': 1,
'wŸ': 1,
'wƒ': 7,
'xJ': 6,
'xT': 1,
'xY': 1,
'x_': 2,
'xj': 2,
'xÎ': 1,
'xÝ': 1,
'xì': 4,
'xî': 2,
'xï': 1,
'xŸ': 1,
'xƒ': 7,
'yJ': 6,
'yT': 1,
'yY': 1,
'y_': 2,
'yj': 2,
'yÎ': 1,
'yÝ': 1,
'yì': 4,
'yî': 2,
'yï': 1,
'yŸ': 1,
'yƒ': 7,
'zJ': 6,
'zT': 1,
'zY': 1,
'z_': 2,
'zj': 2,
'zÎ': 1,
'zÝ': 1,
'zì': 4,
'zî': 2,
'zï': 1,
'zŸ': 1,
'zƒ': 7,
'{J': 6,
'{T': 1,
'{Y': 1,
'{_': 2,
'{j': 2,
'{Î': 1,
'{Ý': 1,
'{ì': 4,
'{î': 2,
'{ï': 1,
'{Ÿ': 1,
'{ƒ': 7,
'|J': 6,
'|T': 1,
'|Y': 1,
'|_': 2,
'|j': 2,
'|Î': 1,
'|Ý': 1,
'|ì': 4,
'|î': 2,
'|ï': 1,
'|Ÿ': 1,
'|ƒ': 7,
'}J': 6,
'}T': 1,
'}Y': 1,
'}_': 2,
'}j': 2,
'}Î': 1,
'}Ý': 1,
'}ì': 4,
'}î': 2,
'}ï': 1,
'}Ÿ': 1,
'}ƒ': 7,
'~J': 6,
'~T': 1,
'~Y': 1,
'~_': 2,
'~j': 2,
'~Î': 1,
'~Ý': 1,
'~ì': 4,
'~î': 2,
'~ï': 1,
'~Ÿ': 1,
'~ƒ': 7,
'\xa0J': 6,
'\xa0T': 1,
'\xa0Y': 1,
'\xa0_': 2,
'\xa0j': 2,
'\xa0Î': 1,
'\xa0Ý': 1,
'\xa0ì': 4,
'\xa0î': 2,
'\xa0ï': 1,
'\xa0Ÿ': 1,
'\xa0ƒ': 7,
'¡J': 6,
'¡T': 1,
'¡Y': 1,
'¡_': 2,
'¡j': 2,
'¡Î': 1,
'¡Ý': 1,
'¡ì': 4,
'¡î': 2,
'¡ï': 1,
'¡Ÿ': 1,
'¡ƒ': 7,
'¢J': 6,
'¢T': 1,
'¢Y': 1,
'¢_': 2,
'¢j': 2,
'¢Î': 1,
'¢Ý': 1,
'¢ì': 4,
'¢î': 2,
'¢ï': 1,
'¢Ÿ': 1,
'¢ƒ': 7,
'£J': 6,
'£T': 1,
'£Y': 1,
'£_': 2,
'£j': 2,
'£Î': 1,
'£Ý': 1,
'£ì': 4,
'£î': 2,
'£ï': 1,
'£Ÿ': 1,
'£ƒ': 7,
'¤J': 6,
'¤T': 1,
'¤Y': 1,
'¤_': 2,
'¤j': 2,
'¤Î': 1,
'¤Ý': 1,
'¤ì': 4,
'¤î': 2,
'¤ï': 1,
'¤Ÿ': 1,
'¤ƒ': 7,
'¥J': 6,
'¥T': 1,
'¥Y': 1,
'¥_': 2,
'¥j': 2,
'¥Î': 1,
'¥Ý': 1,
'¥ì': 4,
'¥î': 2,
'¥ï': 1,
'¥Ÿ': 1,
'¥ƒ': 7,
'¦J': 6,
'¦T': 1,
'¦Y': 1,
'¦_': 2,
'¦j': 2,
'¦Î': 1,
'¦Ý': 1,
'¦ì': 4,
'¦î': 2,
'¦ï': 1,
'¦Ÿ': 1,
'¦ƒ': 7,
'§J': 6,
'§T': 1,
'§Y': 1,
'§_': 2,
'§j': 2,
'§Î': 1,
'§Ý': 1,
'§ì': 4,
'§î': 2,
'§ï': 1,
'§Ÿ': 1,
'§ƒ': 7,
'¨J': 6,
'¨T': 1,
'¨Y': 1,
'¨_': 2,
'¨j': 2,
'¨Î': 1,
'¨Ý': 1,
'¨ì': 4,
'¨î': 2,
'¨ï': 1,
'¨Ÿ': 1,
'¨ƒ': 7,
'©J': 6,
'©T': 1,
'©Y': 1,
'©_': 2,
'©j': 2,
'©Î': 1,
'©Ý': 1,
'©ì': 4,
'©î': 2,
'©ï': 1,
'©Ÿ': 1,
'©ƒ': 7,
'ªJ': 6,
'ªT': 1,
'ªY': 1,
'ª_': 2,
'ªj': 2,
'ªÎ': 1,
'ªÝ': 1,
'ªì': 4,
'ªî': 2,
'ªï': 1,
'ªŸ': 1,
'ªƒ': 7,
'«J': 6,
'«T': 1,
'«Y': 1,
'«_': 2,
'«j': 2,
'«Æ': -1,
'«Î': 1,
'«Ý': 1,
'«ì': 4,
'«î': 2,
'«ï': 1,
'«Ÿ': 1,
'«ƒ': 7,
'¬J': 6,
'¬T': 1,
'¬Y': 1,
'¬_': 2,
'¬j': 2,
'¬Î': 1,
'¬Ý': 1,
'¬ì': 4,
'¬î': 2,
'¬ï': 1,
'¬Ÿ': 1,
'¬ƒ': 7,
'\xadJ': 6,
'\xadT': 1,
'\xadY': 1,
'\xad_': 2,
'\xadj': 2,
'\xadÎ': 1,
'\xadÝ': 1,
'\xadì': 4,
'\xadî': 2,
'\xadï': 1,
'\xadŸ': 1,
'\xadƒ': 7,
'®J': 6,
'®T': 1,
'®Y': 1,
'®_': 2,
'®j': 2,
'®Î': 1,
'®Ý': 1,
'®ì': 4,
'®î': 2,
'®ï': 1,
'®Ÿ': 1,
'®ƒ': 7,
'¯J': 6,
'¯T': 1,
'¯Y': 1,
'¯_': 2,
'¯j': 2,
'¯Î': 1,
'¯Ý': 1,
'¯ì': 4,
'¯î': 2,
'¯ï': 1,
'¯Ÿ': 1,
'¯ƒ': 7,
'°J': 6,
'°T': 1,
'°Y': 1,
'°_': 2,
'°j': 2,
'°Î': 1,
'°Ý': 1,
'°ì': 4,
'°î': 2,
'°ï': 1,
'°Ÿ': 1,
'°ƒ': 7,
'±J': 6,
'±T': 1,
'±Y': 1,
'±_': 2,
'±j': 2,
'±Î': 1,
'±Ý': 1,
'±ì': 4,
'±î': 2,
'±ï': 1,
'±Ÿ': 1,
'±ƒ': 7,
'²J': 6,
'²T': 1,
'²Y': 1,
'²_': 2,
'²j': 2,
'²Î': 1,
'²Ý': 1,
'²ì': 4,
'²î': 2,
'²ï': 1,
'²Ÿ': 1,
'²ƒ': 7,
'³J': 6,
'³T': 1,
'³Y': 1,
'³_': 2,
'³j': 2,
'³Î': 1,
'³Ý': 1,
'³ì': 4,
'³î': 2,
'³ï': 1,
'³Ÿ': 1,
'³ƒ': 7,
'´J': 6,
'´T': 1,
'´Y': 1,
'´_': 2,
'´j': 2,
'´Î': 1,
'´Ý': 1,
'´ì': 4,
'´î': 2,
'´ï': 1,
'´Ÿ': 1,
'´ƒ': 7,
'µJ': 6,
'µT': 1,
'µY': 1,
'µ_': 2,
'µj': 2,
'µÎ': 1,
'µÝ': 1,
'µì': 4,
'µî': 2,
'µï': 1,
'µŸ': 1,
'µƒ': 7,
'¶J': 6,
'¶T': 1,
'¶Y': 1,
'¶_': 2,
'¶j': 2,
'¶Î': 1,
'¶Ý': 1,
'¶ì': 4,
'¶î': 2,
'¶ï': 1,
'¶Ÿ': 1,
'¶ƒ': 7,
'·J': 6,
'·T': 1,
'·Y': 1,
'·_': 2,
'·j': 2,
'·Î': 1,
'·Ý': 1,
'·ì': 4,
'·î': 2,
'·ï': 1,
'·Ÿ': 1,
'·ƒ': 7,
'¸J': 6,
'¸T': 1,
'¸Y': 1,
'¸_': 2,
'¸j': 2,
'¸Î': 1,
'¸Ý': 1,
'¸ì': 4,
'¸î': 2,
'¸ï': 1,
'¸Ÿ': 1,
'¸ƒ': 7,
'¹J': 6,
'¹T': 1,
'¹Y': 1,
'¹_': 2,
'¹j': 2,
'¹Î': 1,
'¹Ý': 1,
'¹ì': 4,
'¹î': 2,
'¹ï': 1,
'¹Ÿ': 1,
'¹ƒ': 7,
'ºJ': 6,
'ºT': 1,
'ºY': 1,
'º_': 2,
'ºj': 2,
'ºÎ': 1,
'ºÝ': 1,
'ºì': 4,
'ºî': 2,
'ºï': 1,
'ºŸ': 1,
'ºƒ': 7,
'»J': 6,
'»T': 1,
'»Y': 1,
'»_': 2,
'»j': 2,
'»Î': 1,
'»Ý': 1,
'»ì': 4,
'»î': 2,
'»ï': 1,
'»Ÿ': 1,
'»ƒ': 7,
'¼J': 6,
'¼T': 1,
'¼Y': 1,
'¼_': 2,
'¼j': 2,
'¼Î': 1,
'¼Ý': 1,
'¼ì': 4,
'¼î': 2,
'¼ï': 1,
'¼Ÿ': 1,
'¼ƒ': 7,
'½J': 6,
'½T': 1,
'½Y': 1,
'½_': 2,
'½j': 2,
'½Î': 1,
'½Ý': 1,
'½ì': 4,
'½î': 2,
'½ï': 1,
'½Ÿ': 1,
'½ƒ': 7,
'¾J': 6,
'¾T': 1,
'¾Y': 1,
'¾_': 2,
'¾j': 2,
'¾Î': 1,
'¾Ý': 1,
'¾ì': 4,
'¾î': 2,
'¾ï': 1,
'¾Ÿ': 1,
'¾ƒ': 7,
'¿J': 6,
'¿T': 1,
'¿Y': 1,
'¿_': 2,
'¿j': 2,
'¿Î': 1,
'¿Ý': 1,
'¿ì': 4,
'¿î': 2,
'¿ï': 1,
'¿Ÿ': 1,
'¿ƒ': 7,
'ÀA': -1,
'ÀJ': 6,
'ÀT': 1,
'ÀY': 1,
'À_': 2,
'Àj': 2,
'ÀÀ': -1,
'ÀÁ': -1,
'ÀÂ': -1,
'ÀÃ': -1,
'ÀÄ': -1,
'ÀÎ': 1,
'ÀÝ': 1,
'Àì': 4,
'Àî': 2,
'Àï': 1,
'ÀŸ': 1,
'Àƒ': 7,
'À„': -1,
'ÁA': -1,
'ÁJ': 6,
'ÁT': 1,
'ÁY': 1,
'Á_': 2,
'Áj': 2,
'ÁÀ': -1,
'ÁÁ': -1,
'ÁÂ': -1,
'ÁÃ': -1,
'ÁÄ': -1,
'ÁÎ': 1,
'ÁÝ': 1,
'Áì': 4,
'Áî': 2,
'Áï': 1,
'ÁŸ': 1,
'Áƒ': 7,
'Á„': -1,
'ÂA': -1,
'ÂJ': 6,
'ÂT': 1,
'ÂY': 1,
'Â_': 2,
'Âj': 2,
'ÂÀ': -1,
'ÂÁ': -1,
'ÂÂ': -1,
'ÂÃ': -1,
'ÂÄ': -1,
'ÂÎ': 1,
'ÂÝ': 1,
'Âì': 4,
'Âî': 2,
'Âï': 1,
'ÂŸ': 1,
'Âƒ': 7,
'Â„': -1,
'ÃA': -1,
'ÃJ': 6,
'ÃT': 1,
'ÃY': 1,
'Ã_': 2,
'Ãj': 2,
'ÃÀ': -1,
'ÃÁ': -1,
'ÃÂ': -1,
'ÃÃ': -1,
'ÃÄ': -1,
'ÃÎ': 1,
'ÃÝ': 1,
'Ãì': 4,
'Ãî': 2,
'Ãï': 1,
'ÃŸ': 1,
'Ãƒ': 7,
'Ã„': -1,
'ÄA': -1,
'ÄJ': 6,
'ÄT': 1,
'ÄY': 1,
'Ä_': 2,
'Äj': 2,
'ÄÀ': -1,
'ÄÁ': -1,
'ÄÂ': -1,
'ÄÃ': -1,
'ÄÄ': -1,
'ÄÎ': 1,
'ÄÝ': 1,
'Äì': 4,
'Äî': 2,
'Äï': 1,
'ÄŸ': 1,
'Äƒ': 7,
'Ä„': -1,
'ÅJ': 6,
'ÅT': 1,
'ÅY': 1,
'Å_': 2,
'Åj': 2,
'ÅÎ': 1,
'ÅÝ': 1,
'Åì': 4,
'Åî': 2,
'Åï': 1,
'ÅŸ': 1,
'Åƒ': 7,
'ÆJ': 6,
'ÆT': 1,
'ÆY': 1,
'Æ_': 2,
'Æj': 2,
'ÆÎ': 1,
'ÆÝ': 1,
'Æì': 4,
'Æî': 2,
'Æï': 1,
'ÆŸ': 1,
'Æƒ': 7,
'ÇJ': 6,
'ÇT': 1,
'ÇY': 1,
'Ç_': 2,
'Çj': 2,
'ÇÎ': 1,
'ÇÝ': 1,
'Çì': 4,
'Çî': 2,
'Çï': 1,
'ÇŸ': 1,
'Çƒ': 7,
'Ç”': -1,
'ÈJ': 6,
'ÈT': 1,
'ÈY': 1,
'È_': 2,
'Èj': 2,
'ÈÎ': 1,
'ÈÝ': 1,
'Èì': 4,
'Èî': 2,
'Èï': 1,
'ÈŸ': 1,
'Èƒ': 7,
'ÉJ': 6,
'ÉT': 1,
'ÉY': 1,
'É_': 2,
'Éj': 2,
'ÉÎ': 1,
'ÉÝ': 1,
'Éì': 4,
'Éî': 2,
'Éï': 1,
'ÉŸ': 1,
'Éƒ': 7,
'ÊJ': 6,
'ÊT': 1,
'ÊY': 1,
'Ê_': 2,
'Êj': 2,
'ÊÎ': 1,
'ÊÝ': 1,
'Êì': 4,
'Êî': 2,
'Êï': 1,
'ÊŸ': 1,
'Êƒ': 7,
'ËJ': 6,
'ËT': 1,
'ËY': 1,
'Ë_': 2,
'Ëj': 2,
'ËÎ': 1,
'ËÝ': 1,
'Ëì': 4,
'Ëî': 2,
'Ëï': 1,
'ËŸ': 1,
'Ëƒ': 7,
'ÌJ': 6,
'ÌT': 1,
'ÌY': 1,
'Ì_': 2,
'Ìj': 2,
'ÌÎ': 1,
'ÌÝ': 1,
'Ìì': 4,
'Ìî': 2,
'Ìï': 1,
'ÌŸ': 1,
'Ìƒ': 7,
'ÍJ': 6,
'ÍT': 1,
'ÍY': 1,
'Í_': 2,
'Íj': 2,
'ÍÎ': 1,
'ÍÝ': 1,
'Íì': 4,
'Íî': 2,
'Íï': 1,
'ÍŸ': 1,
'Íƒ': 7,
'Î ': 1,
'Î!': 1,
'Î"': 1,
'Î#': 1,
'Î$': 1,
'Î%': 1,
'Î&': 1,
"Î'": 1,
'Î(': 1,
'Î)': 1,
'Î*': 1,
'Î+': 1,
'Î,': 1,
'Î-': 1,
'Î.': 1,
'Î/': 1,
'Î0': 1,
'Î1': 1,
'Î2': 1,
'Î3': 1,
'Î4': 1,
'Î5': 1,
'Î6': 1,
'Î7': 1,
'Î8': 1,
'Î9': 1,
'Î:': 1,
'Î;': 1,
'Î<': 1,
'Î=': 1,
'Î>': 1,
'Î?': 1,
'Î@': 1,
'ÎA': 1,
'ÎB': 1,
'ÎC': 1,
'ÎD': 1,
'ÎE': 1,
'ÎF': 1,
'ÎG': 1,
'ÎH': 1,
'ÎI': 1,
'ÎJ': 7,
'ÎK': 1,
'ÎL': 1,
'ÎM': 1,
'ÎN': 1,
'ÎO': 1,
'ÎP': 1,
'ÎQ': 1,
'ÎR': 1,
'ÎS': 1,
'ÎT': 2,
'ÎU': 1,
'ÎV': 1,
'ÎW': 1,
'ÎX': 1,
'ÎY': 2,
'ÎZ': 1,
'Î[': 1,
'Î\\': 1,
'Î]': 1,
'Î^': 1,
'Î_': 3,
'Î`': 1,
'Îa': 1,
'Îb': 1,
'Îc': 1,
'Îd': 1,
'Îe': 1,
'Îf': 1,
'Îg': 1,
'Îh': 1,
'Îi': 1,
'Îj': 3,
'Îk': 1,
'Îl': 1,
'Îm': 1,
'În': 1,
'Îo': 1,
'Îp': 1,
'Îq': 1,
'Îr': 1,
'Îs': 1,
'Ît': 1,
'Îu': 1,
'Îv': 1,
'Îw': 1,
'Îx': 1,
'Îy': 1,
'Îz': 1,
'Î{': 1,
'Î|': 1,
'Î}': 1,
'Î~': 1,
'Î\xa0': 1,
'Î¡': 1,
'Î¢': 1,
'Î£': 1,
'Î¤': 1,
'Î¥': 1,
'Î¦': 1,
'Î§': 1,
'Î¨': 1,
'Î©': 1,
'Îª': 1,
'Î«': 1,
'Î¬': 1,
'Î\xad': 1,
'Î®': 1,
'Î¯': 1,
'Î°': 1,
'Î±': 1,
'Î²': 1,
'Î³': 1,
'Î´': 1,
'Îµ': 1,
'Î¶': 1,
'Î·': 1,
'Î¸': 1,
'Î¹': 1,
'Îº': 1,
'Î»': 1,
'Î¼': 1,
'Î½': 1,
'Î¾': 1,
'Î¿': 1,
'ÎÀ': 1,
'ÎÁ': 1,
'ÎÂ': 1,
'ÎÃ': 1,
'ÎÄ': 1,
'ÎÅ': 1,
'ÎÆ': 1,
'ÎÇ': 1,
'ÎÈ': 1,
'ÎÉ': 1,
'ÎÊ': 1,
'ÎË': 1,
'ÎÌ': 1,
'ÎÍ': 1,
'ÎÎ': 2,
'ÎÏ': 1,
'ÎÐ': 1,
'ÎÑ': 1,
'ÎÒ': 1,
'ÎÓ': 1,
'ÎÔ': 1,
'ÎÕ': 1,
'ÎÖ': 1,
'Î×': 1,
'ÎØ': 1,
'ÎÙ': 1,
'ÎÚ': 1,
'ÎÛ': 1,
'ÎÜ': 1,
'ÎÝ': 2,
'ÎÞ': 1,
'Îß': 1,
'Îà': 1,
'Îá': 1,
'Îâ': 1,
'Îã': 1,
'Îä': 1,
'Îå': 1,
'Îæ': 1,
'Îç': 1,
'Îè': 1,
'Îé': 1,
'Îê': 1,
'Îë': 1,
'Îì': 5,
'Îí': 1,
'Îî': 3,
'Îï': 2,
'Îð': 1,
'Îñ': 1,
'Îò': 1,
'Îó': 1,
'Îô': 1,
'Îõ': 1,
'Îö': 1,
'Î÷': 1,
'Îø': 1,
'Îù': 1,
'Îú': 1,
'Îû': 1,
'Îü': 1,
'Îý': 1,
'Îþ': 1,
'Îÿ': 1,
'ÎŒ': 1,
'Îœ': 1,
'ÎŠ': 1,
'Îš': 1,
'ÎŸ': 2,
'ÎŽ': 1,
'Îž': 1,
'Îƒ': 8,
'Îˆ': 1,
'Î˜': 1,
'Î–': 1,
'Î—': 1,
'Î‘': 1,
'Î’': 1,
'Î‚': 1,
'Î“': 1,
'Î”': 1,
'Î„': 1,
'Î†': 1,
'Î‡': 1,
'Î•': 1,
'Î…': 1,
'Î‰': 1,
'Î‹': 1,
'Î›': 1,
'Î€': 1,
'Î™': 1,
'Ï ': 1,
'Ï!': 1,
'Ï"': 1,
'Ï#': 1,
'Ï$': 1,
'Ï%': 1,
'Ï&': 1,
"Ï'": 1,
'Ï(': 1,
'Ï)': 1,
'Ï*': 1,
'Ï+': 1,
'Ï,': 1,
'Ï-': 1,
'Ï.': 1,
'Ï/': 1,
'Ï0': 1,
'Ï1': 1,
'Ï2': 1,
'Ï3': 1,
'Ï4': 1,
'Ï5': 1,
'Ï6': 1,
'Ï7': 1,
'Ï8': 1,
'Ï9': 1,
'Ï:': 1,
'Ï;': 1,
'Ï<': 1,
'Ï=': 1,
'Ï>': 1,
'Ï?': 1,
'Ï@': 1,
'ÏA': 1,
'ÏB': 1,
'ÏC': 1,
'ÏD': 1,
'ÏE': 1,
'ÏF': 1,
'ÏG': 1,
'ÏH': 1,
'ÏI': 1,
'ÏJ': 7,
'ÏK': 1,
'ÏL': 1,
'ÏM': 1,
'ÏN': 1,
'ÏO': 1,
'ÏP': 1,
'ÏQ': 1,
'ÏR': 1,
'ÏS': 1,
'ÏT': 2,
'ÏU': 1,
'ÏV': 1,
'ÏW': 1,
'ÏX': 1,
'ÏY': 2,
'ÏZ': 1,
'Ï[': 1,
'Ï\\': 1,
'Ï]': 1,
'Ï^': 1,
'Ï_': 3,
'Ï`': 1,
'Ïa': 1,
'Ïb': 1,
'Ïc': 1,
'Ïd': 1,
'Ïe': 1,
'Ïf': 1,
'Ïg': 1,
'Ïh': 1,
'Ïi': 1,
'Ïj': 3,
'Ïk': 1,
'Ïl': 1,
'Ïm': 1,
'Ïn': 1,
'Ïo': 1,
'Ïp': 1,
'Ïq': 1,
'Ïr': 1,
'Ïs': 1,
'Ït': 1,
'Ïu': 1,
'Ïv': 1,
'Ïw': 1,
'Ïx': 1,
'Ïy': 1,
'Ïz': 1,
'Ï{': 1,
'Ï|': 1,
'Ï}': 1,
'Ï~': 1,
'Ï\xa0': 1,
'Ï¡': 1,
'Ï¢': 1,
'Ï£': 1,
'Ï¤': 1,
'Ï¥': 1,
'Ï¦': 1,
'Ï§': 1,
'Ï¨': 1,
'Ï©': 1,
'Ïª': 1,
'Ï«': 1,
'Ï¬': 1,
'Ï\xad': 1,
'Ï®': 1,
'Ï¯': 1,
'Ï°': 1,
'Ï±': 1,
'Ï²': 1,
'Ï³': 1,
'Ï´': 1,
'Ïµ': 1,
'Ï¶': 1,
'Ï·': 1,
'Ï¸': 1,
'Ï¹': 1,
'Ïº': 1,
'Ï»': 1,
'Ï¼': 1,
'Ï½': 1,
'Ï¾': 1,
'Ï¿': 1,
'ÏÀ': 1,
'ÏÁ': 1,
'ÏÂ': 1,
'ÏÃ': 1,
'ÏÄ': 1,
'ÏÅ': 1,
'ÏÆ': 1,
'ÏÇ': 1,
'ÏÈ': 1,
'ÏÉ': 1,
'ÏÊ': 1,
'ÏË': 1,
'ÏÌ': 1,
'ÏÍ': 1,
'ÏÎ': 2,
'ÏÏ': 1,
'ÏÐ': 1,
'ÏÑ': 1,
'ÏÒ': 1,
'ÏÓ': 1,
'ÏÔ': 1,
'ÏÕ': 1,
'ÏÖ': 1,
'Ï×': 1,
'ÏØ': 1,
'ÏÙ': 1,
'ÏÚ': 1,
'ÏÛ': 1,
'ÏÜ': 1,
'ÏÝ': 2,
'ÏÞ': 1,
'Ïß': 1,
'Ïà': 1,
'Ïá': 1,
'Ïâ': 1,
'Ïã': 1,
'Ïä': 1,
'Ïå': 1,
'Ïæ': 1,
'Ïç': 1,
'Ïè': 1,
'Ïé': 1,
'Ïê': 1,
'Ïë': 1,
'Ïì': 5,
'Ïí': 1,
'Ïî': 3,
'Ïï': 2,
'Ïð': 1,
'Ïñ': 1,
'Ïò': 1,
'Ïó': 1,
'Ïô': 1,
'Ïõ': 1,
'Ïö': 1,
'Ï÷': 1,
'Ïø': 1,
'Ïù': 1,
'Ïú': 1,
'Ïû': 1,
'Ïü': 1,
'Ïý': 1,
'Ïþ': 1,
'Ïÿ': 1,
'ÏŒ': 1,
'Ïœ': 1,
'ÏŠ': 1,
'Ïš': 1,
'ÏŸ': 2,
'ÏŽ': 1,
'Ïž': 1,
'Ïƒ': 8,
'Ïˆ': 1,
'Ï˜': 1,
'Ï–': 1,
'Ï—': 1,
'Ï‘': 1,
'Ï’': 1,
'Ï‚': 1,
'Ï“': 1,
'Ï”': 1,
'Ï„': 1,
'Ï†': 1,
'Ï‡': 1,
'Ï•': 1,
'Ï…': 1,
'Ï‰': 1,
'Ï‹': 1,
'Ï›': 1,
'Ï€': 1,
'Ï™': 1,
'ÐJ': 6,
'ÐT': 1,
'ÐY': 1,
'Ð_': 2,
'Ðj': 2,
'ÐÎ': 1,
'ÐÝ': 1,
'Ðì': 4,
'Ðî': 2,
'Ðï': 1,
'ÐŸ': 1,
'Ðƒ': 7,
'ÑJ': 6,
'ÑT': 1,
'ÑY': 1,
'Ñ_': 2,
'Ñj': 2,
'ÑÎ': 1,
'ÑÝ': 1,
'Ñì': 4,
'Ñî': 2,
'Ñï': 1,
'ÑŸ': 1,
'Ñƒ': 7,
'Ò-': -1,
'ÒJ': 6,
'ÒT': 1,
'ÒY': 1,
'Ò_': 2,
'Òj': 2,
'ÒÎ': 1,
'ÒÝ': 1,
'Òì': 4,
'Òî': 2,
'Òï': 1,
'ÒŸ': 1,
'Òƒ': 7,
'Ó-': -1,
'ÓJ': 6,
'ÓT': 1,
'ÓY': 1,
'Ó_': 2,
'Ój': 2,
'ÓÎ': 1,
'ÓÝ': 1,
'Óì': 4,
'Óî': 2,
'Óï': 1,
'ÓŸ': 1,
'Óƒ': 7,
'Ô-': -1,
'ÔJ': 6,
'ÔT': 1,
'ÔY': 1,
'Ô_': 2,
'Ôj': 2,
'ÔÎ': 1,
'ÔÝ': 1,
'Ôì': 4,
'Ôî': 2,
'Ôï': 1,
'ÔŸ': 1,
'Ôƒ': 7,
'Õ-': -1,
'ÕJ': 6,
'ÕT': 1,
'ÕY': 1,
'Õ_': 2,
'Õj': 2,
'ÕÎ': 1,
'ÕÝ': 1,
'Õì': 4,
'Õî': 2,
'Õï': 1,
'ÕŸ': 1,
'Õƒ': 7,
'Ö-': -1,
'ÖJ': 6,
'ÖT': 1,
'ÖY': 1,
'Ö_': 2,
'Öj': 2,
'ÖÎ': 1,
'ÖÝ': 1,
'Öì': 4,
'Öî': 2,
'Öï': 1,
'ÖŸ': 1,
'Öƒ': 7,
'×J': 6,
'×T': 1,
'×Y': 1,
'×_': 2,
'×j': 2,
'×Î': 1,
'×Ý': 1,
'×ì': 4,
'×î': 2,
'×ï': 1,
'×Ÿ': 1,
'×ƒ': 7,
'ØJ': 6,
'ØT': 1,
'ØY': 1,
'Ø_': 2,
'Øj': 2,
'ØÎ': 1,
'ØÝ': 1,
'Øì': 4,
'Øî': 2,
'Øï': 1,
'ØŸ': 1,
'Øƒ': 7,
'ÙJ': 6,
'ÙT': 1,
'ÙY': 1,
'Ù_': 2,
'Ùj': 2,
'ÙÎ': 1,
'ÙÝ': 1,
'Ùì': 4,
'Ùî': 2,
'Ùï': 1,
'ÙŸ': 1,
'Ùƒ': 7,
'ÚJ': 6,
'ÚT': 1,
'ÚY': 1,
'Ú_': 2,
'Új': 2,
'ÚÎ': 1,
'ÚÝ': 1,
'Úì': 4,
'Úî': 2,
'Úï': 1,
'ÚŸ': 1,
'Úƒ': 7,
'ÛJ': 6,
'ÛT': 1,
'ÛY': 1,
'Û_': 2,
'Ûj': 2,
'ÛÎ': 1,
'ÛÝ': 1,
'Ûì': 4,
'Ûî': 2,
'Ûï': 1,
'ÛŸ': 1,
'Ûƒ': 7,
'ÜJ': 6,
'ÜT': 1,
'ÜY': 1,
'Ü_': 2,
'Üj': 2,
'ÜÎ': 1,
'ÜÝ': 1,
'Üì': 4,
'Üî': 2,
'Üï': 1,
'ÜŸ': 1,
'Üƒ': 7,
'Ý ': 1,
'Ý!': 1,
'Ý"': 1,
'Ý#': 1,
'Ý$': 1,
'Ý%': 1,
'Ý&': 1,
"Ý'": 1,
'Ý(': 1,
'Ý)': 1,
'Ý*': 1,
'Ý+': 1,
'Ý,': 1,
'Ý-': 1,
'Ý.': 1,
'Ý/': 1,
'Ý0': 1,
'Ý1': 1,
'Ý2': 1,
'Ý3': 1,
'Ý4': 1,
'Ý5': 1,
'Ý6': 1,
'Ý7': 1,
'Ý8': 1,
'Ý9': 1,
'Ý:': 1,
'Ý;': 1,
'Ý<': 1,
'Ý=': 1,
'Ý>': 1,
'Ý?': 1,
'Ý@': 1,
'ÝA': 1,
'ÝB': 1,
'ÝC': 1,
'ÝD': 1,
'ÝE': 1,
'ÝF': 1,
'ÝG': 1,
'ÝH': 1,
'ÝI': 1,
'ÝJ': 7,
'ÝK': 1,
'ÝL': 1,
'ÝM': 1,
'ÝN': 1,
'ÝO': 1,
'ÝP': 1,
'ÝQ': 1,
'ÝR': 1,
'ÝS': 1,
'ÝT': 2,
'ÝU': 1,
'ÝV': 1,
'ÝW': 1,
'ÝX': 1,
'ÝY': 2,
'ÝZ': 1,
'Ý[': 1,
'Ý\\': 1,
'Ý]': 1,
'Ý^': 1,
'Ý_': 3,
'Ý`': 1,
'Ýa': 1,
'Ýb': 1,
'Ýc': 1,
'Ýd': 1,
'Ýe': 1,
'Ýf': 1,
'Ýg': 1,
'Ýh': 1,
'Ýi': 1,
'Ýj': 3,
'Ýk': 1,
'Ýl': 1,
'Ým': 1,
'Ýn': 1,
'Ýo': 1,
'Ýp': 1,
'Ýq': 1,
'Ýr': 1,
'Ýs': 1,
'Ýt': 1,
'Ýu': 1,
'Ýv': 1,
'Ýw': 1,
'Ýx': 1,
'Ýy': 1,
'Ýz': 1,
'Ý{': 1,
'Ý|': 1,
'Ý}': 1,
'Ý~': 1,
'Ý\xa0': 1,
'Ý¡': 1,
'Ý¢': 1,
'Ý£': 1,
'Ý¤': 1,
'Ý¥': 1,
'Ý¦': 1,
'Ý§': 1,
'Ý¨': 1,
'Ý©': 1,
'Ýª': 1,
'Ý«': 1,
'Ý¬': 1,
'Ý\xad': 1,
'Ý®': 1,
'Ý¯': 1,
'Ý°': 1,
'Ý±': 1,
'Ý²': 1,
'Ý³': 1,
'Ý´': 1,
'Ýµ': 1,
'Ý¶': 1,
'Ý·': 1,
'Ý¸': 1,
'Ý¹': 1,
'Ýº': 1,
'Ý»': 1,
'Ý¼': 1,
'Ý½': 1,
'Ý¾': 1,
'Ý¿': 1,
'ÝÀ': 1,
'ÝÁ': 1,
'ÝÂ': 1,
'ÝÃ': 1,
'ÝÄ': 1,
'ÝÅ': 1,
'ÝÆ': 1,
'ÝÇ': 1,
'ÝÈ': 1,
'ÝÉ': 1,
'ÝÊ': 1,
'ÝË': 1,
'ÝÌ': 1,
'ÝÍ': 1,
'ÝÎ': 2,
'ÝÏ': 1,
'ÝÐ': 1,
'ÝÑ': 1,
'ÝÒ': 1,
'ÝÓ': 1,
'ÝÔ': 1,
'ÝÕ': 1,
'ÝÖ': 1,
'Ý×': 1,
'ÝØ': 1,
'ÝÙ': 1,
'ÝÚ': 1,
'ÝÛ': 1,
'ÝÜ': 1,
'ÝÝ': 2,
'ÝÞ': 1,
'Ýß': 1,
'Ýà': 1,
'Ýá': 1,
'Ýâ': 1,
'Ýã': 1,
'Ýä': 1,
'Ýå': 1,
'Ýæ': 1,
'Ýç': 1,
'Ýè': 1,
'Ýé': 1,
'Ýê': 1,
'Ýë': 1,
'Ýì': 5,
'Ýí': 1,
'Ýî': 3,
'Ýï': 2,
'Ýð': 1,
'Ýñ': 1,
'Ýò': 1,
'Ýó': 1,
'Ýô': 1,
'Ýõ': 1,
'Ýö': 1,
'Ý÷': 1,
'Ýø': 1,
'Ýù': 1,
'Ýú': 1,
'Ýû': 1,
'Ýü': 1,
'Ýý': 1,
'Ýþ': 1,
'Ýÿ': 1,
'ÝŒ': 1,
'Ýœ': 1,
'ÝŠ': 1,
'Ýš': 1,
'ÝŸ': 2,
'ÝŽ': 1,
'Ýž': 1,
'Ýƒ': 8,
'Ýˆ': 1,
'Ý˜': 1,
'Ý–': 1,
'Ý—': 1,
'Ý‘': 1,
'Ý’': 1,
'Ý‚': 1,
'Ý“': 1,
'Ý”': 1,
'Ý„': 1,
'Ý†': 1,
'Ý‡': 1,
'Ý•': 1,
'Ý…': 1,
'Ý‰': 1,
'Ý‹': 1,
'Ý›': 1,
'Ý€': 1,
'Ý™': 1,
'ÞJ': 6,
'ÞT': 1,
'ÞY': 1,
'Þ_': 2,
'Þj': 2,
'ÞÎ': 1,
'ÞÝ': 1,
'Þì': 4,
'Þî': 2,
'Þï': 1,
'ÞŸ': 1,
'Þƒ': 7,
'ß-': -1,
'ßJ': 6,
'ßT': 1,
'ßY': 1,
'ß_': 2,
'ßj': 2,
'ßÎ': 1,
'ßÝ': 1,
'ßì': 4,
'ßî': 2,
'ßï': 1,
'ßŸ': 1,
'ßƒ': 7,
'àJ': 6,
'àT': 1,
'àY': 1,
'à_': 2,
'àj': 2,
'àÎ': 1,
'àÝ': 1,
'àì': 4,
'àî': 2,
'àï': 1,
'àŸ': 1,
'àƒ': 7,
'áJ': 6,
'áT': 1,
'áY': 1,
'á_': 2,
'áj': 2,
'áÎ': 1,
'áÝ': 1,
'áì': 4,
'áî': 2,
'áï': 1,
'áŸ': 1,
'áƒ': 7,
'âJ': 6,
'âT': 1,
'âY': 1,
'â_': 2,
'âj': 2,
'âÎ': 1,
'âÝ': 1,
'âì': 4,
'âî': 2,
'âï': 1,
'âŸ': 1,
'âƒ': 7,
'ãJ': 6,
'ãT': 1,
'ãY': 1,
'ã_': 2,
'ãj': 2,
'ãÎ': 1,
'ãÝ': 1,
'ãì': 4,
'ãî': 2,
'ãï': 1,
'ãŸ': 1,
'ãƒ': 7,
'äJ': 6,
'äT': 1,
'äY': 1,
'ä_': 2,
'äj': 2,
'äÎ': 1,
'äÝ': 1,
'äì': 4,
'äî': 2,
'äï': 1,
'äŸ': 1,
'äƒ': 7,
'åJ': 6,
'åT': 1,
'åY': 1,
'å_': 2,
'åj': 2,
'åÎ': 1,
'åÝ': 1,
'åì': 4,
'åî': 2,
'åï': 1,
'åŸ': 1,
'åƒ': 7,
'æJ': 6,
'æT': 1,
'æY': 1,
'æ_': 2,
'æj': 2,
'æÎ': 1,
'æÝ': 1,
'æì': 4,
'æî': 2,
'æï': 1,
'æŸ': 1,
'æƒ': 7,
'çJ': 6,
'çT': 1,
'çY': 1,
'ç_': 2,
'çj': 2,
'çÎ': 1,
'çÝ': 1,
'çì': 4,
'çî': 2,
'çï': 1,
'çŸ': 1,
'çƒ': 7,
'èJ': 6,
'èT': 1,
'èY': 1,
'è_': 2,
'èj': 2,
'èÎ': 1,
'èÝ': 1,
'èì': 4,
'èî': 2,
'èï': 1,
'èŸ': 1,
'èƒ': 7,
'éJ': 6,
'éT': 1,
'éY': 1,
'é_': 2,
'éj': 2,
'éÎ': 1,
'éÝ': 1,
'éì': 4,
'éî': 2,
'éï': 1,
'éŸ': 1,
'éƒ': 7,
'êJ': 6,
'êT': 1,
'êY': 1,
'ê_': 2,
'êj': 2,
'êÎ': 1,
'êÝ': 1,
'êì': 4,
'êî': 2,
'êï': 1,
'êŸ': 1,
'êƒ': 7,
'ëJ': 6,
'ëT': 1,
'ëY': 1,
'ë_': 2,
'ëj': 2,
'ëÎ': 1,
'ëÝ': 1,
'ëì': 4,
'ëî': 2,
'ëï': 1,
'ëŸ': 1,
'ëƒ': 7,
'ìJ': 6,
'ìT': 1,
'ìY': 1,
'ì_': 2,
'ìj': 2,
'ìÎ': 1,
'ìÝ': 1,
'ìì': 4,
'ìî': 2,
'ìï': 1,
'ìŸ': 1,
'ìƒ': 7,
'í ': 3,
'í!': 3,
'í"': 3,
'í#': 3,
'í$': 3,
'í%': 3,
'í&': 3,
"í'": 3,
'í(': 3,
'í)': 3,
'í*': 3,
'í+': 3,
'í,': 3,
'í-': 3,
'í.': 3,
'í/': 3,
'í0': 3,
'í1': 3,
'í2': 3,
'í3': 3,
'í4': 3,
'í5': 3,
'í6': 3,
'í7': 3,
'í8': 3,
'í9': 3,
'í:': 3,
'í;': 3,
'í<': 3,
'í=': 3,
'í>': 3,
'í?': 3,
'í@': 3,
'íA': 3,
'íB': 3,
'íC': 3,
'íD': 3,
'íE': 3,
'íF': 3,
'íG': 3,
'íH': 3,
'íI': 3,
'íJ': 9,
'íK': 3,
'íL': 3,
'íM': 3,
'íN': 3,
'íO': 3,
'íP': 3,
'íQ': 3,
'íR': 3,
'íS': 3,
'íT': 4,
'íU': 3,
'íV': 3,
'íW': 3,
'íX': 3,
'íY': 4,
'íZ': 3,
'í[': 3,
'í\\': 3,
'í]': 3,
'í^': 3,
'í_': 5,
'í`': 3,
'ía': 3,
'íb': 3,
'íc': 3,
'íd': 3,
'íe': 3,
'íf': 3,
'íg': 3,
'íh': 3,
'íi': 3,
'íj': 5,
'ík': 3,
'íl': 3,
'ím': 3,
'ín': 3,
'ío': 3,
'íp': 3,
'íq': 3,
'ír': 3,
'ís': 3,
'ít': 3,
'íu': 3,
'ív': 3,
'íw': 3,
'íx': 3,
'íy': 3,
'íz': 3,
'í{': 3,
'í|': 3,
'í}': 3,
'í~': 3,
'í\xa0': 3,
'í¡': 3,
'í¢': 3,
'í£': 3,
'í¤': 3,
'í¥': 3,
'í¦': 3,
'í§': 3,
'í¨': 3,
'í©': 3,
'íª': 3,
'í«': 3,
'í¬': 3,
'í\xad': 3,
'í®': 3,
'í¯': 3,
'í°': 3,
'í±': 3,
'í²': 3,
'í³': 3,
'í´': 3,
'íµ': 3,
'í¶': 3,
'í·': 3,
'í¸': 3,
'í¹': 3,
'íº': 3,
'í»': 3,
'í¼': 3,
'í½': 3,
'í¾': 3,
'í¿': 3,
'íÀ': 3,
'íÁ': 3,
'íÂ': 3,
'íÃ': 3,
'íÄ': 3,
'íÅ': 3,
'íÆ': 3,
'íÇ': 3,
'íÈ': 3,
'íÉ': 3,
'íÊ': 3,
'íË': 3,
'íÌ': 3,
'íÍ': 3,
'íÎ': 4,
'íÏ': 3,
'íÐ': 3,
'íÑ': 3,
'íÒ': 3,
'íÓ': 3,
'íÔ': 3,
'íÕ': 3,
'íÖ': 3,
'í×': 3,
'íØ': 3,
'íÙ': 3,
'íÚ': 3,
'íÛ': 3,
'íÜ': 3,
'íÝ': 4,
'íÞ': 3,
'íß': 3,
'íà': 3,
'íá': 3,
'íâ': 3,
'íã': 3,
'íä': 3,
'íå': 3,
'íæ': 3,
'íç': 3,
'íè': 3,
'íé': 3,
'íê': 3,
'íë': 3,
'íì': 7,
'íí': 3,
'íî': 5,
'íï': 4,
'íð': 3,
'íñ': 3,
'íò': 3,
'íó': 3,
'íô': 3,
'íõ': 3,
'íö': 3,
'í÷': 3,
'íø': 3,
'íù': 3,
'íú': 3,
'íû': 3,
'íü': 3,
'íý': 3,
'íþ': 3,
'íÿ': 3,
'íŒ': 3,
'íœ': 3,
'íŠ': 3,
'íš': 3,
'íŸ': 4,
'íŽ': 3,
'íž': 3,
'íƒ': 10,
'íˆ': 3,
'í˜': 3,
'í–': 3,
'í—': 3,
'í‘': 3,
'í’': 3,
'í‚': 3,
'í“': 3,
'í”': 3,
'í„': 3,
'í†': 3,
'í‡': 3,
'í•': 3,
'í…': 3,
'í‰': 3,
'í‹': 3,
'í›': 3,
'í€': 3,
'í™': 3,
'î ': 2,
'î!': 2,
'î"': 2,
'î#': 2,
'î$': 2,
'î%': 2,
'î&': 2,
"î'": 2,
'î(': 2,
'î)': 2,
'î*': 2,
'î+': 2,
'î,': 2,
'î-': 2,
'î.': 2,
'î/': 2,
'î0': 2,
'î1': 2,
'î2': 2,
'î3': 2,
'î4': 2,
'î5': 2,
'î6': 2,
'î7': 2,
'î8': 2,
'î9': 2,
'î:': 2,
'î;': 2,
'î<': 2,
'î=': 2,
'î>': 2,
'î?': 2,
'î@': 2,
'îA': 2,
'îB': 2,
'îC': 2,
'îD': 2,
'îE': 2,
'îF': 2,
'îG': 2,
'îH': 2,
'îI': 2,
'îJ': 8,
'îK': 2,
'îL': 2,
'îM': 2,
'îN': 2,
'îO': 2,
'îP': 2,
'îQ': 2,
'îR': 2,
'îS': 2,
'îT': 3,
'îU': 2,
'îV': 2,
'îW': 2,
'îX': 2,
'îY': 3,
'îZ': 2,
'î[': 2,
'î\\': 2,
'î]': 2,
'î^': 2,
'î_': 4,
'î`': 2,
'îa': 2,
'îb': 2,
'îc': 2,
'îd': 2,
'îe': 2,
'îf': 2,
'îg': 2,
'îh': 2,
'îi': 2,
'îj': 4,
'îk': 2,
'îl': 2,
'îm': 2,
'în': 2,
'îo': 2,
'îp': 2,
'îq': 2,
'îr': 2,
'îs': 2,
'ît': 2,
'îu': 2,
'îv': 2,
'îw': 2,
'îx': 2,
'îy': 2,
'îz': 2,
'î{': 2,
'î|': 2,
'î}': 2,
'î~': 2,
'î\xa0': 2,
'î¡': 2,
'î¢': 2,
'î£': 2,
'î¤': 2,
'î¥': 2,
'î¦': 2,
'î§': 2,
'î¨': 2,
'î©': 2,
'îª': 2,
'î«': 2,
'î¬': 2,
'î\xad': 2,
'î®': 2,
'î¯': 2,
'î°': 2,
'î±': 2,
'î²': 2,
'î³': 2,
'î´': 2,
'îµ': 2,
'î¶': 2,
'î·': 2,
'î¸': 2,
'î¹': 2,
'îº': 2,
'î»': 2,
'î¼': 2,
'î½': 2,
'î¾': 2,
'î¿': 2,
'îÀ': 2,
'îÁ': 2,
'îÂ': 2,
'îÃ': 2,
'îÄ': 2,
'îÅ': 2,
'îÆ': 2,
'îÇ': 2,
'îÈ': 2,
'îÉ': 2,
'îÊ': 2,
'îË': 2,
'îÌ': 2,
'îÍ': 2,
'îÎ': 3,
'îÏ': 2,
'îÐ': 2,
'îÑ': 2,
'îÒ': 2,
'îÓ': 2,
'îÔ': 2,
'îÕ': 2,
'îÖ': 2,
'î×': 2,
'îØ': 2,
'îÙ': 2,
'îÚ': 2,
'îÛ': 2,
'îÜ': 2,
'îÝ': 3,
'îÞ': 2,
'îß': 2,
'îà': 2,
'îá': 2,
'îâ': 2,
'îã': 2,
'îä': 2,
'îå': 2,
'îæ': 2,
'îç': 2,
'îè': 2,
'îé': 2,
'îê': 2,
'îë': 2,
'îì': 6,
'îí': 2,
'îî': 4,
'îï': 3,
'îð': 2,
'îñ': 2,
'îò': 2,
'îó': 2,
'îô': 2,
'îõ': 2,
'îö': 2,
'î÷': 2,
'îø': 2,
'îù': 2,
'îú': 2,
'îû': 2,
'îü': 2,
'îý': 2,
'îþ': 2,
'îÿ': 2,
'îŒ': 2,
'îœ': 2,
'îŠ': 2,
'îš': 2,
'îŸ': 3,
'îŽ': 2,
'îž': 2,
'îƒ': 9,
'îˆ': 2,
'î˜': 2,
'î–': 2,
'î—': 2,
'î‘': 2,
'î’': 2,
'î‚': 2,
'î“': 2,
'î”': 2,
'î„': 2,
'î†': 2,
'î‡': 2,
'î•': 2,
'î…': 2,
'î‰': 2,
'î‹': 2,
'î›': 2,
'î€': 2,
'î™': 2,
'ï ': 1,
'ï!': 1,
'ï"': 1,
'ï#': 1,
'ï$': 1,
'ï%': 1,
'ï&': 1,
"ï'": 1,
'ï(': 1,
'ï)': 1,
'ï*': 1,
'ï+': 1,
'ï,': 1,
'ï-': 1,
'ï.': 1,
'ï/': 1,
'ï0': 1,
'ï1': 1,
'ï2': 1,
'ï3': 1,
'ï4': 1,
'ï5': 1,
'ï6': 1,
'ï7': 1,
'ï8': 1,
'ï9': 1,
'ï:': 1,
'ï;': 1,
'ï<': 1,
'ï=': 1,
'ï>': 1,
'ï?': 1,
'ï@': 1,
'ïA': 1,
'ïB': 1,
'ïC': 1,
'ïD': 1,
'ïE': 1,
'ïF': 1,
'ïG': 1,
'ïH': 1,
'ïI': 1,
'ïJ': 7,
'ïK': 1,
'ïL': 1,
'ïM': 1,
'ïN': 1,
'ïO': 1,
'ïP': 1,
'ïQ': 1,
'ïR': 1,
'ïS': 1,
'ïT': 2,
'ïU': 1,
'ïV': 1,
'ïW': 1,
'ïX': 1,
'ïY': 2,
'ïZ': 1,
'ï[': 1,
'ï\\': 1,
'ï]': 1,
'ï^': 1,
'ï_': 3,
'ï`': 1,
'ïa': 1,
'ïb': 1,
'ïc': 1,
'ïd': 1,
'ïe': 1,
'ïf': 1,
'ïg': 1,
'ïh': 1,
'ïi': 1,
'ïj': 3,
'ïk': 1,
'ïl': 1,
'ïm': 1,
'ïn': 1,
'ïo': 1,
'ïp': 1,
'ïq': 1,
'ïr': 1,
'ïs': 1,
'ït': 1,
'ïu': 1,
'ïv': 1,
'ïw': 1,
'ïx': 1,
'ïy': 1,
'ïz': 1,
'ï{': 1,
'ï|': 1,
'ï}': 1,
'ï~': 1,
'ï\xa0': 1,
'ï¡': 1,
'ï¢': 1,
'ï£': 1,
'ï¤': 1,
'ï¥': 1,
'ï¦': 1,
'ï§': 1,
'ï¨': 1,
'ï©': 1,
'ïª': 1,
'ï«': 1,
'ï¬': 1,
'ï\xad': 1,
'ï®': 1,
'ï¯': 1,
'ï°': 1,
'ï±': 1,
'ï²': 1,
'ï³': 1,
'ï´': 1,
'ïµ': 1,
'ï¶': 1,
'ï·': 1,
'ï¸': 1,
'ï¹': 1,
'ïº': 1,
'ï»': 1,
'ï¼': 1,
'ï½': 1,
'ï¾': 1,
'ï¿': 1,
'ïÀ': 1,
'ïÁ': 1,
'ïÂ': 1,
'ïÃ': 1,
'ïÄ': 1,
'ïÅ': 1,
'ïÆ': 1,
'ïÇ': 1,
'ïÈ': 1,
'ïÉ': 1,
'ïÊ': 1,
'ïË': 1,
'ïÌ': 1,
'ïÍ': 1,
'ïÎ': 2,
'ïÏ': 1,
'ïÐ': 1,
'ïÑ': 1,
'ïÒ': 1,
'ïÓ': 1,
'ïÔ': 1,
'ïÕ': 1,
'ïÖ': 1,
'ï×': 1,
'ïØ': 1,
'ïÙ': 1,
'ïÚ': 1,
'ïÛ': 1,
'ïÜ': 1,
'ïÝ': 2,
'ïÞ': 1,
'ïß': 1,
'ïà': 1,
'ïá': 1,
'ïâ': 1,
'ïã': 1,
'ïä': 1,
'ïå': 1,
'ïæ': 1,
'ïç': 1,
'ïè': 1,
'ïé': 1,
'ïê': 1,
'ïë': 1,
'ïì': 5,
'ïí': 1,
'ïî': 3,
'ïï': 2,
'ïð': 1,
'ïñ': 1,
'ïò': 1,
'ïó': 1,
'ïô': 1,
'ïõ': 1,
'ïö': 1,
'ï÷': 1,
'ïø': 1,
'ïù': 1,
'ïú': 1,
'ïû': 1,
'ïü': 1,
'ïý': 1,
'ïþ': 1,
'ïÿ': 1,
'ïŒ': 1,
'ïœ': 1,
'ïŠ': 1,
'ïš': 1,
'ïŸ': 2,
'ïŽ': 1,
'ïž': 1,
'ïƒ': 8,
'ïˆ': 1,
'ï˜': 1,
'ï–': 1,
'ï—': 1,
'ï‘': 1,
'ï’': 1,
'ï‚': 1,
'ï“': 1,
'ï”': 1,
'ï„': 1,
'ï†': 1,
'ï‡': 1,
'ï•': 1,
'ï…': 1,
'ï‰': 1,
'ï‹': 1,
'ï›': 1,
'ï€': 1,
'ï™': 1,
'ðJ': 6,
'ðT': 1,
'ðY': 1,
'ð_': 2,
'ðj': 2,
'ðÎ': 1,
'ðÝ': 1,
'ðì': 4,
'ðî': 2,
'ðï': 1,
'ðŸ': 1,
'ðƒ': 7,
'ñJ': 6,
'ñT': 1,
'ñY': 1,
'ñ_': 2,
'ñj': 2,
'ñÎ': 1,
'ñÝ': 1,
'ñì': 4,
'ñî': 2,
'ñï': 1,
'ñŸ': 1,
'ñƒ': 7,
'ò-': -1,
'òJ': 6,
'òT': 1,
'òY': 1,
'ò_': 2,
'òj': 2,
'òÎ': 1,
'òÝ': 1,
'òì': 4,
'òî': 2,
'òï': 1,
'òŸ': 1,
'òƒ': 7,
'ó-': -1,
'óJ': 6,
'óT': 1,
'óY': 1,
'ó_': 2,
'ój': 2,
'óÎ': 1,
'óÝ': 1,
'óì': 4,
'óî': 2,
'óï': 1,
'óŸ': 1,
'óƒ': 7,
'ô-': -1,
'ôJ': 6,
'ôT': 1,
'ôY': 1,
'ô_': 2,
'ôj': 2,
'ôÎ': 1,
'ôÝ': 1,
'ôì': 4,
'ôî': 2,
'ôï': 1,
'ôŸ': 1,
'ôƒ': 7,
'õ-': -1,
'õJ': 6,
'õT': 1,
'õY': 1,
'õ_': 2,
'õj': 2,
'õÎ': 1,
'õÝ': 1,
'õì': 4,
'õî': 2,
'õï': 1,
'õŸ': 1,
'õƒ': 7,
'ö-': -1,
'öJ': 6,
'öT': 1,
'öY': 1,
'ö_': 2,
'öj': 2,
'öÎ': 1,
'öÝ': 1,
'öì': 4,
'öî': 2,
'öï': 1,
'öŸ': 1,
'öƒ': 7,
'÷J': 6,
'÷T': 1,
'÷Y': 1,
'÷_': 2,
'÷j': 2,
'÷Î': 1,
'÷Ý': 1,
'÷ì': 4,
'÷î': 2,
'÷ï': 1,
'÷Ÿ': 1,
'÷ƒ': 7,
'øJ': 6,
'øT': 1,
'øY': 1,
'ø_': 2,
'øj': 2,
'øÎ': 1,
'øÝ': 1,
'øì': 4,
'øî': 2,
'øï': 1,
'øŸ': 1,
'øƒ': 7,
'ùJ': 6,
'ùT': 1,
'ùY': 1,
'ù_': 2,
'ùj': 2,
'ùÎ': 1,
'ùÝ': 1,
'ùì': 4,
'ùî': 2,
'ùï': 1,
'ùŸ': 1,
'ùƒ': 7,
'úJ': 6,
'úT': 1,
'úY': 1,
'ú_': 2,
'új': 2,
'úÎ': 1,
'úÝ': 1,
'úì': 4,
'úî': 2,
'úï': 1,
'úŸ': 1,
'úƒ': 7,
'ûJ': 6,
'ûT': 1,
'ûY': 1,
'û_': 2,
'ûj': 2,
'ûÎ': 1,
'ûÝ': 1,
'ûì': 4,
'ûî': 2,
'ûï': 1,
'ûŸ': 1,
'ûƒ': 7,
'üJ': 6,
'üT': 1,
'üY': 1,
'ü_': 2,
'üj': 2,
'üÎ': 1,
'üÝ': 1,
'üì': 4,
'üî': 2,
'üï': 1,
'üŸ': 1,
'üƒ': 7,
'ýJ': 6,
'ýT': 1,
'ýY': 1,
'ý_': 2,
'ýj': 2,
'ýÎ': 1,
'ýÝ': 1,
'ýì': 4,
'ýî': 2,
'ýï': 1,
'ýŸ': 1,
'ýƒ': 7,
'þJ': 6,
'þT': 1,
'þY': 1,
'þ_': 2,
'þj': 2,
'þÎ': 1,
'þÝ': 1,
'þì': 4,
'þî': 2,
'þï': 1,
'þŸ': 1,
'þƒ': 7,
'ÿJ': 6,
'ÿT': 1,
'ÿY': 1,
'ÿ_': 2,
'ÿj': 2,
'ÿÎ': 1,
'ÿÝ': 1,
'ÿì': 4,
'ÿî': 2,
'ÿï': 1,
'ÿŸ': 1,
'ÿƒ': 7,
'ŒJ': 6,
'ŒT': 1,
'ŒY': 1,
'Œ_': 2,
'Œj': 2,
'ŒÎ': 1,
'ŒÝ': 1,
'Œì': 4,
'Œî': 2,
'Œï': 1,
'ŒŸ': 1,
'Œƒ': 7,
'œJ': 6,
'œT': 1,
'œY': 1,
'œ_': 2,
'œj': 2,
'œÎ': 1,
'œÝ': 1,
'œì': 4,
'œî': 2,
'œï': 1,
'œŸ': 1,
'œƒ': 7,
'ŠA': -1,
'ŠJ': 6,
'ŠT': 1,
'ŠY': 1,
'Š_': 2,
'Šj': 2,
'ŠÀ': -1,
'ŠÁ': -1,
'ŠÂ': -1,
'ŠÃ': -1,
'ŠÄ': -1,
'ŠÎ': 1,
'ŠÝ': 1,
'Šì': 4,
'Šî': 2,
'Šï': 1,
'ŠŸ': 1,
'Šƒ': 7,
'šJ': 6,
'šT': 1,
'šY': 1,
'š_': 2,
'šj': 2,
'šÎ': 1,
'šÝ': 1,
'šì': 4,
'šî': 2,
'šï': 1,
'šŸ': 1,
'šƒ': 7,
'Ÿ ': 1,
'Ÿ!': 1,
'Ÿ"': 1,
'Ÿ#': 1,
'Ÿ$': 1,
'Ÿ%': 1,
'Ÿ&': 1,
"Ÿ'": 1,
'Ÿ(': 1,
'Ÿ)': 1,
'Ÿ*': 1,
'Ÿ+': 1,
'Ÿ,': 1,
'Ÿ-': 1,
'Ÿ.': 1,
'Ÿ/': 1,
'Ÿ0': 1,
'Ÿ1': 1,
'Ÿ2': 1,
'Ÿ3': 1,
'Ÿ4': 1,
'Ÿ5': 1,
'Ÿ6': 1,
'Ÿ7': 1,
'Ÿ8': 1,
'Ÿ9': 1,
'Ÿ:': 1,
'Ÿ;': 1,
'Ÿ<': 1,
'Ÿ=': 1,
'Ÿ>': 1,
'Ÿ?': 1,
'Ÿ@': 1,
'ŸA': 1,
'ŸB': 1,
'ŸC': 1,
'ŸD': 1,
'ŸE': 1,
'ŸF': 1,
'ŸG': 1,
'ŸH': 1,
'ŸI': 1,
'ŸJ': 7,
'ŸK': 1,
'ŸL': 1,
'ŸM': 1,
'ŸN': 1,
'ŸO': 1,
'ŸP': 1,
'ŸQ': 1,
'ŸR': 1,
'ŸS': 1,
'ŸT': 2,
'ŸU': 1,
'ŸV': 1,
'ŸW': 1,
'ŸX': 1,
'ŸY': 2,
'ŸZ': 1,
'Ÿ[': 1,
'Ÿ\\': 1,
'Ÿ]': 1,
'Ÿ^': 1,
'Ÿ_': 3,
'Ÿ`': 1,
'Ÿa': 1,
'Ÿb': 1,
'Ÿc': 1,
'Ÿd': 1,
'Ÿe': 1,
'Ÿf': 1,
'Ÿg': 1,
'Ÿh': 1,
'Ÿi': 1,
'Ÿj': 3,
'Ÿk': 1,
'Ÿl': 1,
'Ÿm': 1,
'Ÿn': 1,
'Ÿo': 1,
'Ÿp': 1,
'Ÿq': 1,
'Ÿr': 1,
'Ÿs': 1,
'Ÿt': 1,
'Ÿu': 1,
'Ÿv': 1,
'Ÿw': 1,
'Ÿx': 1,
'Ÿy': 1,
'Ÿz': 1,
'Ÿ{': 1,
'Ÿ|': 1,
'Ÿ}': 1,
'Ÿ~': 1,
'Ÿ\xa0': 1,
'Ÿ¡': 1,
'Ÿ¢': 1,
'Ÿ£': 1,
'Ÿ¤': 1,
'Ÿ¥': 1,
'Ÿ¦': 1,
'Ÿ§': 1,
'Ÿ¨': 1,
'Ÿ©': 1,
'Ÿª': 1,
'Ÿ«': 1,
'Ÿ¬': 1,
'Ÿ\xad': 1,
'Ÿ®': 1,
'Ÿ¯': 1,
'Ÿ°': 1,
'Ÿ±': 1,
'Ÿ²': 1,
'Ÿ³': 1,
'Ÿ´': 1,
'Ÿµ': 1,
'Ÿ¶': 1,
'Ÿ·': 1,
'Ÿ¸': 1,
'Ÿ¹': 1,
'Ÿº': 1,
'Ÿ»': 1,
'Ÿ¼': 1,
'Ÿ½': 1,
'Ÿ¾': 1,
'Ÿ¿': 1,
'ŸÀ': 1,
'ŸÁ': 1,
'ŸÂ': 1,
'ŸÃ': 1,
'ŸÄ': 1,
'ŸÅ': 1,
'ŸÆ': 1,
'ŸÇ': 1,
'ŸÈ': 1,
'ŸÉ': 1,
'ŸÊ': 1,
'ŸË': 1,
'ŸÌ': 1,
'ŸÍ': 1,
'ŸÎ': 2,
'ŸÏ': 1,
'ŸÐ': 1,
'ŸÑ': 1,
'ŸÒ': 1,
'ŸÓ': 1,
'ŸÔ': 1,
'ŸÕ': 1,
'ŸÖ': 1,
'Ÿ×': 1,
'ŸØ': 1,
'ŸÙ': 1,
'ŸÚ': 1,
'ŸÛ': 1,
'ŸÜ': 1,
'ŸÝ': 2,
'ŸÞ': 1,
'Ÿß': 1,
'Ÿà': 1,
'Ÿá': 1,
'Ÿâ': 1,
'Ÿã': 1,
'Ÿä': 1,
'Ÿå': 1,
'Ÿæ': 1,
'Ÿç': 1,
'Ÿè': 1,
'Ÿé': 1,
'Ÿê': 1,
'Ÿë': 1,
'Ÿì': 5,
'Ÿí': 1,
'Ÿî': 3,
'Ÿï': 2,
'Ÿð': 1,
'Ÿñ': 1,
'Ÿò': 1,
'Ÿó': 1,
'Ÿô': 1,
'Ÿõ': 1,
'Ÿö': 1,
'Ÿ÷': 1,
'Ÿø': 1,
'Ÿù': 1,
'Ÿú': 1,
'Ÿû': 1,
'Ÿü': 1,
'Ÿý': 1,
'Ÿþ': 1,
'Ÿÿ': 1,
'ŸŒ': 1,
'Ÿœ': 1,
'ŸŠ': 1,
'Ÿš': 1,
'ŸŸ': 2,
'ŸŽ': 1,
'Ÿž': 1,
'Ÿƒ': 8,
'Ÿˆ': 1,
'Ÿ˜': 1,
'Ÿ–': 1,
'Ÿ—': 1,
'Ÿ‘': 1,
'Ÿ’': 1,
'Ÿ‚': 1,
'Ÿ“': 1,
'Ÿ”': 1,
'Ÿ„': 1,
'Ÿ†': 1,
'Ÿ‡': 1,
'Ÿ•': 1,
'Ÿ…': 1,
'Ÿ‰': 1,
'Ÿ‹': 1,
'Ÿ›': 1,
'Ÿ€': 1,
'Ÿ™': 1,
'ŽJ': 6,
'ŽT': 1,
'ŽY': 1,
'Ž_': 2,
'Žj': 2,
'ŽÎ': 1,
'ŽÝ': 1,
'Žì': 4,
'Žî': 2,
'Žï': 1,
'ŽŸ': 1,
'Žƒ': 7,
'žJ': 6,
'žT': 1,
'žY': 1,
'ž_': 2,
'žj': 2,
'žÎ': 1,
'žÝ': 1,
'žì': 4,
'žî': 2,
'žï': 1,
'žŸ': 1,
'žƒ': 7,
'ƒ ': 2,
'ƒ!': 2,
'ƒ"': 2,
'ƒ#': 2,
'ƒ$': 2,
'ƒ%': 2,
'ƒ&': 2,
"ƒ'": 2,
'ƒ(': 2,
'ƒ)': 2,
'ƒ*': 2,
'ƒ+': 2,
'ƒ,': 2,
'ƒ-': 2,
'ƒ.': 2,
'ƒ/': 2,
'ƒ0': 2,
'ƒ1': 2,
'ƒ2': 2,
'ƒ3': 2,
'ƒ4': 2,
'ƒ5': 2,
'ƒ6': 2,
'ƒ7': 2,
'ƒ8': 2,
'ƒ9': 2,
'ƒ:': 2,
'ƒ;': 2,
'ƒ<': 2,
'ƒ=': 2,
'ƒ>': 2,
'ƒ?': 2,
'ƒ@': 2,
'ƒA': 2,
'ƒB': 2,
'ƒC': 2,
'ƒD': 2,
'ƒE': 2,
'ƒF': 2,
'ƒG': 2,
'ƒH': 2,
'ƒI': 2,
'ƒJ': 8,
'ƒK': 2,
'ƒL': 2,
'ƒM': 2,
'ƒN': 2,
'ƒO': 2,
'ƒP': 2,
'ƒQ': 2,
'ƒR': 2,
'ƒS': 2,
'ƒT': 3,
'ƒU': 2,
'ƒV': 2,
'ƒW': 2,
'ƒX': 2,
'ƒY': 3,
'ƒZ': 2,
'ƒ[': 2,
'ƒ\\': 2,
'ƒ]': 2,
'ƒ^': 2,
'ƒ_': 4,
'ƒ`': 2,
'ƒa': 2,
'ƒb': 2,
'ƒc': 2,
'ƒd': 2,
'ƒe': 2,
'ƒf': 2,
'ƒg': 2,
'ƒh': 2,
'ƒi': 2,
'ƒj': 4,
'ƒk': 2,
'ƒl': 2,
'ƒm': 2,
'ƒn': 2,
'ƒo': 2,
'ƒp': 2,
'ƒq': 2,
'ƒr': 2,
'ƒs': 2,
'ƒt': 2,
'ƒu': 2,
'ƒv': 2,
'ƒw': 2,
'ƒx': 2,
'ƒy': 2,
'ƒz': 2,
'ƒ{': 2,
'ƒ|': 2,
'ƒ}': 2,
'ƒ~': 2,
'ƒ\xa0': 2,
'ƒ¡': 2,
'ƒ¢': 2,
'ƒ£': 2,
'ƒ¤': 2,
'ƒ¥': 2,
'ƒ¦': 2,
'ƒ§': 2,
'ƒ¨': 2,
'ƒ©': 2,
'ƒª': 2,
'ƒ«': 2,
'ƒ¬': 2,
'ƒ\xad': 2,
'ƒ®': 2,
'ƒ¯': 2,
'ƒ°': 2,
'ƒ±': 2,
'ƒ²': 2,
'ƒ³': 2,
'ƒ´': 2,
'ƒµ': 2,
'ƒ¶': 2,
'ƒ·': 2,
'ƒ¸': 2,
'ƒ¹': 2,
'ƒº': 2,
'ƒ»': 2,
'ƒ¼': 2,
'ƒ½': 2,
'ƒ¾': 2,
'ƒ¿': 2,
'ƒÀ': 2,
'ƒÁ': 2,
'ƒÂ': 2,
'ƒÃ': 2,
'ƒÄ': 2,
'ƒÅ': 2,
'ƒÆ': 2,
'ƒÇ': 2,
'ƒÈ': 2,
'ƒÉ': 2,
'ƒÊ': 2,
'ƒË': 2,
'ƒÌ': 2,
'ƒÍ': 2,
'ƒÎ': 3,
'ƒÏ': 2,
'ƒÐ': 2,
'ƒÑ': 2,
'ƒÒ': 2,
'ƒÓ': 2,
'ƒÔ': 2,
'ƒÕ': 2,
'ƒÖ': 2,
'ƒ×': 2,
'ƒØ': 2,
'ƒÙ': 2,
'ƒÚ': 2,
'ƒÛ': 2,
'ƒÜ': 2,
'ƒÝ': 3,
'ƒÞ': 2,
'ƒß': 2,
'ƒà': 2,
'ƒá': 2,
'ƒâ': 2,
'ƒã': 2,
'ƒä': 2,
'ƒå': 2,
'ƒæ': 2,
'ƒç': 2,
'ƒè': 2,
'ƒé': 2,
'ƒê': 2,
'ƒë': 2,
'ƒì': 6,
'ƒí': 2,
'ƒî': 4,
'ƒï': 3,
'ƒð': 2,
'ƒñ': 2,
'ƒò': 2,
'ƒó': 2,
'ƒô': 2,
'ƒõ': 2,
'ƒö': 2,
'ƒ÷': 2,
'ƒø': 2,
'ƒù': 2,
'ƒú': 2,
'ƒû': 2,
'ƒü': 2,
'ƒý': 2,
'ƒþ': 2,
'ƒÿ': 2,
'ƒŒ': 2,
'ƒœ': 2,
'ƒŠ': 2,
'ƒš': 2,
'ƒŸ': 3,
'ƒŽ': 2,
'ƒž': 2,
'ƒƒ': 9,
'ƒˆ': 2,
'ƒ˜': 2,
'ƒ–': 2,
'ƒ—': 2,
'ƒ‘': 2,
'ƒ’': 2,
'ƒ‚': 2,
'ƒ“': 2,
'ƒ”': 2,
'ƒ„': 2,
'ƒ†': 2,
'ƒ‡': 2,
'ƒ•': 2,
'ƒ…': 2,
'ƒ‰': 2,
'ƒ‹': 2,
'ƒ›': 2,
'ƒ€': 2,
'ƒ™': 2,
'ˆJ': 6,
'ˆT': 1,
'ˆY': 1,
'ˆ_': 2,
'ˆj': 2,
'ˆÎ': 1,
'ˆÝ': 1,
'ˆì': 4,
'ˆî': 2,
'ˆï': 1,
'ˆŸ': 1,
'ˆƒ': 7,
'˜J': 6,
'˜T': 1,
'˜Y': 1,
'˜_': 2,
'˜j': 2,
'˜Î': 1,
'˜Ý': 1,
'˜ì': 4,
'˜î': 2,
'˜ï': 1,
'˜Ÿ': 1,
'˜ƒ': 7,
'–J': 6,
'–T': 1,
'–Y': 1,
'–_': 2,
'–j': 2,
'–Î': 1,
'–Ý': 1,
'–ì': 4,
'–î': 2,
'–ï': 1,
'–Ÿ': 1,
'–ƒ': 7,
'—J': 6,
'—T': 1,
'—Y': 1,
'—_': 2,
'—j': 2,
'—Î': 1,
'—Ý': 1,
'—ì': 4,
'—î': 2,
'—ï': 1,
'—Ÿ': 1,
'—ƒ': 7,
'‘J': 6,
'‘T': 1,
'‘Y': 1,
'‘_': 2,
'‘j': 2,
'‘Î': 1,
'‘Ý': 1,
'‘ì': 4,
'‘î': 2,
'‘ï': 1,
'‘Ÿ': 1,
'‘ƒ': 7,
'’J': 6,
'’T': 1,
'’Y': 1,
'’_': 2,
'’j': 2,
'’Î': 1,
'’Ý': 1,
'’ì': 4,
'’î': 2,
'’ï': 1,
'’Ÿ': 1,
'’ƒ': 7,
'‚J': 6,
'‚T': 1,
'‚Y': 1,
'‚_': 2,
'‚j': 2,
'‚Î': 1,
'‚Ý': 1,
'‚ì': 4,
'‚î': 2,
'‚ï': 1,
'‚Ÿ': 1,
'‚ƒ': 7,
'“J': 6,
'“T': 1,
'“Y': 1,
'“_': 2,
'“j': 2,
'“Î': 1,
'“Ý': 1,
'“ì': 4,
'“î': 2,
'“ï': 1,
'“Ÿ': 1,
'“ƒ': 7,
'”J': 6,
'”T': 1,
'”Y': 1,
'”_': 2,
'”j': 2,
'”Î': 1,
'”Ý': 1,
'”ì': 4,
'”î': 2,
'”ï': 1,
'”Ÿ': 1,
'”ƒ': 7,
'„A': -1,
'„J': 5,
'„T': 1,
'„Y': 1,
'„_': 2,
'„j': 2,
'„À': -1,
'„Á': -1,
'„Â': -1,
'„Ã': -1,
'„Ä': -1,
'„Æ': -1,
'„Î': 1,
'„Ý': 1,
'„ì': 4,
'„î': 2,
'„ï': 1,
'„Ÿ': 1,
'„ƒ': 7,
'†J': 6,
'†T': 1,
'†Y': 1,
'†_': 2,
'†j': 2,
'†Î': 1,
'†Ý': 1,
'†ì': 4,
'†î': 2,
'†ï': 1,
'†Ÿ': 1,
'†ƒ': 7,
'‡J': 6,
'‡T': 1,
'‡Y': 1,
'‡_': 2,
'‡j': 2,
'‡Î': 1,
'‡Ý': 1,
'‡ì': 4,
'‡î': 2,
'‡ï': 1,
'‡Ÿ': 1,
'‡ƒ': 7,
'•J': 6,
'•T': 1,
'•Y': 1,
'•_': 2,
'•j': 2,
'•Î': 1,
'•Ý': 1,
'•ì': 4,
'•î': 2,
'•ï': 1,
'•Ÿ': 1,
'•ƒ': 7,
'…J': 6,
'…T': 1,
'…Y': 1,
'…_': 2,
'…j': 2,
'…Î': 1,
'…Ý': 1,
'…ì': 4,
'…î': 2,
'…ï': 1,
'…Ÿ': 1,
'…ƒ': 7,
'‰J': 6,
'‰T': 1,
'‰Y': 1,
'‰_': 2,
'‰j': 2,
'‰Î': 1,
'‰Ý': 1,
'‰ì': 4,
'‰î': 2,
'‰ï': 1,
'‰Ÿ': 1,
'‰ƒ': 7,
'‹J': 6,
'‹T': 1,
'‹Y': 1,
'‹_': 2,
'‹j': 2,
'‹Î': 1,
'‹Ý': 1,
'‹ì': 4,
'‹î': 2,
'‹ï': 1,
'‹Ÿ': 1,
'‹ƒ': 7,
'›J': 6,
'›T': 1,
'›Y': 1,
'›_': 2,
'›j': 2,
'›Î': 1,
'›Ý': 1,
'›ì': 4,
'›î': 2,
'›ï': 1,
'›Ÿ': 1,
'›ƒ': 7,
'€J': 6,
'€T': 1,
'€Y': 1,
'€_': 2,
'€j': 2,
'€Î': 1,
'€Ý': 1,
'€ì': 4,
'€î': 2,
'€ï': 1,
'€Ÿ': 1,
'€ƒ': 7,
'™J': 6,
'™T': 1,
'™Y': 1,
'™_': 2,
'™j': 2,
'™Î': 1,
'™Ý': 1,
'™ì': 4,
'™î': 2,
'™ï': 1,
'™Ÿ': 1,
'™ƒ': 7},
'mean-character-length': 77.47223162885201}
