# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bblocks',
 'bblocks.analysis_tools',
 'bblocks.cleaning_tools',
 'bblocks.dataframe_tools',
 'bblocks.import_tools',
 'bblocks.import_tools.debt',
 'bblocks.other_tools']

package_data = \
{'': ['*'], 'bblocks': ['.raw_data/*'], 'bblocks.import_tools': ['settings/*']}

install_requires = \
['PyPDF2>=2,<3',
 'beautifulsoup4>=4.11.1,<5.0.0',
 'camelot-py>=0.10.1,<0.11.0',
 'country-converter>=0.8.0,<0.9.0',
 'numpy>=1.24.1,<2.0.0',
 'opencv-python>=4.7.0,<5.0.0',
 'openpyxl>=3.0.10,<4.0.0',
 'pandas>=1.5.3,<2.0.0',
 'pyarrow>=11.0.0,<12.0.0',
 'pyjstat>=2.3.0,<3.0.0',
 'requests>=2.28.2,<3.0.0',
 'wbgapi<1.1',
 'weo>=0.7.4,<0.8.0']

setup_kwargs = {
    'name': 'bblocks',
    'version': '1.0.2',
    'description': 'A package with tools to download and analyse international development data. These tools are meant to be the building blocks of further analysis.',
    'long_description': '[![pypi](https://img.shields.io/pypi/v/bblocks.svg)](https://pypi.org/project/bblocks/)\n[![python](https://img.shields.io/pypi/pyversions/bblocks.svg)](https://pypi.org/project/bblocks/)\n[![codecov](https://codecov.io/gh/ONEcampaign/bblocks/branch/main/graph/badge.svg?token=YN8S1719NH)](https://codecov.io/gh/ONEcampaign/bblocks)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n# The bblocks package\n\n**bblocks** is a python package with tools to download and analyse\ndevelopment data. These tools are meant to be the *building blocks* of\nfurther analysis.\n\nWe have built **bblocks** to support our work at ONE, but we hope that\nit will be useful to others working with development data. We welcome feedback,\nfeature requests, and collaboration.\n\n**bblocks** is organised around the following main features:\n\n- *Import tools* to help with import data from:\n    - The World Bank (building on the wbgapi package)\n    - The IMF World Economic outlook (building on the weo package)\n    - The IMF data on Special Drawing Rights\n    - The World Food Programme (WFP) data on food security and inflation\n    - The FAO (notably the price index)\n    - The UNDP Human Development Report data\n    - UNAIDS\n    - The WHO Government Health Expenditure data\n\n- *Cleaning tools* to help with:\n    - Cleaning numbers/numeric series\n    - Transforming country identifiers (ISO2, ISO3, WB, UN, etc., building on the country_converter package)\n    - Transforming text to datetime objects, and datetime objects to text\n    - Formatting numbers as text (percentages, millions, billions, etc.)\n\n- *Analysis tools* to help with:\n    - Calculating period averages\n    - Calculating the change from one period to another\n\n- *DataFrame tools* to help with:\n    - Adding a population column to a DataFrame\n    - Adding a "share of population" / "per capita" column to a DataFrame\n    - Adding a population density column to a DataFrame\n    - Adding a GDP column to a DataFrame\n    - Adding a "share of GDP" column to a DataFrame\n    - Adding a poverty ratio column to a DataFrame\n    - Adding a government expenditure column to a DataFrame\n    - Adding a "share of government expenditure" column to a DataFrame\n    - Adding a "World Bank income level" column to a DataFrame\n    - Adding a column with short country names to a DataFrame\n    - Adding a column with ISO3 codes to a DataFrame\n    - Adding the median observation of a group\n    - Adding a column with geojson geometries to a DataFrame\n\n- *Other tools* like:\n    - Dictionaries mapping ISO3 codes (and vice-versa) to\n        - OECD DAC codes\n        - WB income groups\n        - geojson geometries\n        - G7, EU27, G20 countries\n        - Income levels\n        - Life expectancy\n        - Population\n\nMore information is available:\n\n- Documentation: https://bblocks.readthedocs.io/\n- GitHub: https://github.com/ONECampaign/bblocks\n- PyPI: https://pypi.org/project/bblocks/\n\n## Installation\n\nbblocks can be installed from using pip\n\n```bash\npip install bblocks --upgrade\n\n```\n\nThe package is compatible with Python 3.10 and above.\n\n## Basic usage\n\nTo get started, import the package. It is strongly recommended that you specify\nthe path to the folder where you want to store the data.\n\nYou only have to do this once per file/notebook.\n\n```python\nfrom bblocks import set_bblocks_data_path\n\n# Set to the folder you want\nset_bblocks_data_path("path/to/data/folder")\n```\n\nAll the examples below assume that you have done this.\n\n### Importing data from the World Bank\n\n```python\nfrom bblocks import WorldBankData\n\n# create a WorldBankData object. This object will allow you\n# to download indicators from the World Bank and get them as DataFrames\nwb = WorldBankData()\n\n# For example to get "primary completion rate" (SE.PRM.CMPT.ZS) from 2010 to 2020.\n# If the data is not already in your data folder, it will be downloaded\nwb.load_data(\n    indicator="SE.PRM.CMPT.ZS",\n    start_year=2010,\n    end_year=2020\n)\n\n# Get the data as a DataFrame\ndf = wb.get_data()\n\n# Print a sample of 10 rows\nprint(df.sample(10))\n```\n\nThe above would return a DataFrame like this:\n\n| date       | iso\\_code | indicator\\_code | value     |\n|:-----------|:----------|:----------------|:----------|\n| 2010-01-01 | LMC       | SE.PRM.CMPT.ZS  | 87.753189 |\n| 2012-01-01 | SWZ       | SE.PRM.CMPT.ZS  | 84.697472 |\n| 2013-01-01 | NAM       | SE.PRM.CMPT.ZS  | 93.020042 |\n| 2012-01-01 | PAK       | SE.PRM.CMPT.ZS  | 63.486210 |\n| 2015-01-01 | LIC       | SE.PRM.CMPT.ZS  | 63.463470 |\n| 2016-01-01 | BGD       | SE.PRM.CMPT.ZS  | NaN       |\n| 2019-01-01 | SYR       | SE.PRM.CMPT.ZS  | NaN       |\n| 2013-01-01 | NAC       | SE.PRM.CMPT.ZS  | 99.025703 |\n| 2011-01-01 | AND       | SE.PRM.CMPT.ZS  | NaN       |\n| 2013-01-01 | GRL       | SE.PRM.CMPT.ZS  | NaN       |\n\nYou can also get the latest data (most recent non-empty observation) for one or more indicators:\n\n```python\nfrom bblocks import WorldBankData\n\n# create a WorldBankData object.\nwb_data = WorldBankData()\n\n# Load the indicators. If they are not downloaded, they will be\nwb_data.load_data(\n    indicator=["SH.XPD.CHEX.PC.CD", "SH.XPD.CHEX.GD.ZS"],\n    most_recent_only=True\n)\n\n# Get the data as a DataFrame\ndf = wb_data.get_data(indicators="all")\n\n# Print a sample of the data\nprint(df.sample(10))\n```\n\nThis would return a DataFrame like this:\n\n| date       | iso\\_code | indicator\\_code   | value       |\n|:-----------|:----------|:------------------|:------------|\n| 2019-01-01 | HRV       | SH.XPD.CHEX.PC.CD | 1040.085693 |\n| 2019-01-01 | ERI       | SH.XPD.CHEX.GD.ZS | 4.458767    |\n| 2019-01-01 | JAM       | SH.XPD.CHEX.PC.CD | 327.403534  |\n| 2019-01-01 | MYS       | SH.XPD.CHEX.PC.CD | 436.612030  |\n| 2019-01-01 | BHS       | SH.XPD.CHEX.GD.ZS | 5.749775    |\n| 2015-01-01 | YEM       | SH.XPD.CHEX.PC.CD | 73.176743   |\n| 2019-01-01 | PER       | SH.XPD.CHEX.PC.CD | 370.109955  |\n| 2019-01-01 | IDA       | SH.XPD.CHEX.PC.CD | 52.076285   |\n| 2019-01-01 | ERI       | SH.XPD.CHEX.PC.CD | 25.267935   |\n| 2019-01-01 | WLD       | SH.XPD.CHEX.PC.CD | 1115.008730 |\n\nIn all cases, if you had already downloaded the data and you want to update it\nyou can call `.update_data()` after loading the data in order to refresh it.\n\n```python\nwb_data.update_data(reload_data=True)\n```\n\n### Importing data from UNAIDS\n\n```python\nfrom bblocks import Aids\n\n# create an Aids object. This object will allow you\n# to download indicators from UNAIDS and get them as DataFrames\naids = Aids()\n\n# To view all the indicators that can be downloaded using this tool\n# you can use the `.available_indicators` property\naids.available_indicators\n```\n\nHer are the first 10 indicators, but over 50 are available:\n\n|     | indicator                                       | category                    |\n|:----|:------------------------------------------------|:----------------------------|\n| 0   | Trend of new HIV infections                     | Epidemic transition metrics |\n| 1   | Trend of AIDS-related deaths                    | Epidemic transition metrics |\n| 2   | Incidence:prevalence ratio                      | Epidemic transition metrics |\n| 3   | Incidence:mortality ratio                       | Epidemic transition metrics |\n| 4   | People living with HIV - All ages               | People living with HIV      |\n| 5   | People living with HIV - Children \\(0-14\\)      | People living with HIV      |\n| 6   | People living with HIV - Adolescents \\(10-19\\)  | People living with HIV      |\n| 7   | People living with HIV - Young people \\(15-24\\) | People living with HIV      |\n| 8   | People living with HIV - Adults \\(15+\\)         | People living with HIV      |\n| 9   | People living with HIV - Adults \\(15-49\\)       | People living with HIV      |\n\n```python\n# to load/download indicators, you can use the `.load_data` method\n# you can also specify whether to download "country", "region", or "all"\naids.load_data(\n    indicator="Trend of AIDS-related deaths",\n    area_grouping="region"\n)\n\n# get the data as a DataFrame\ndf = aids.get_data()\n\n# print a sample of 10 rows\nprint(df.sample(10))\n```\n\n| area\\_name                                 | area\\_id | year | indicator                    | dimension               | value        |\n|:-------------------------------------------|:---------|:-----|:-----------------------------|:------------------------|:-------------|\n| Global                                     | 03M49WLD | 2013 | Trend of AIDS-related deaths | All ages estimate       | 1.061395e+06 |\n| Latin America                              | UNALA    | 2021 | Trend of AIDS-related deaths | All ages estimate       | 2.916500e+04 |\n| Middle East and North Africa               | UNAMENA  | 2018 | Trend of AIDS-related deaths | All ages lower estimate | 4.089657e+03 |\n| Western & Central Europe and North America | UNAWCENA | 2019 | Trend of AIDS-related deaths | All ages estimate       | 1.305140e+04 |\n| Caribbean                                  | UNACAR   | 2021 | Trend of AIDS-related deaths | All ages lower estimate | 4.213485e+03 |\n| Middle East and North Africa               | UNAMENA  | 2021 | Trend of AIDS-related deaths | All ages upper estimate | 6.867407e+03 |\n| Western & Central Europe and North America | UNAWCENA | 2016 | Trend of AIDS-related deaths | All ages upper estimate | 1.771698e+04 |\n| Western & Central Europe and North America | UNAWCENA | 2020 | Trend of AIDS-related deaths | All ages upper estimate | 1.632782e+04 |\n| Eastern Europe and Central Asia            | UNAEECA  | 2017 | Trend of AIDS-related deaths | All ages upper estimate | 4.553729e+04 |\n| Latin America                              | UNALA    | 2020 | Trend of AIDS-related deaths | All ages upper estimate | 4.577862e+04 |\n\nAs with other bblocks tools, you can also get multiple indicators at once\n(see the WorldBank example).\n\nIn all cases, if you had already downloaded the data and you want to update it\nyou can call `.update_data()` after loading the data in order to refresh it.\n\n```python\naids.update_data(reload_data=True)\n```\n\n### Importing SDR data from the IMF\n\n```python\n# Import the SDR object from the sdr module of "import_tools"\nfrom bblocks.import_tools.sdr import SDR\n\n# Create an SDR object\nsdr = SDR()\n\n# To view the latest date for which data is available,\n# call the `.latest_date()` method\nsdr.latest_date()\n\n# To download the latest data\nsdr.load_data(date="latest")\n\n# To get the data as a DataFrame. You can specify getting a \n# specific indicator by using \'indicator\'. In this case,\n# we\'ll get holdings (allocations are also available)\ndf = sdr.get_data(indicator="holdings")\n\n# Print a sample of 10 rows\nprint(df.sample(10))\n```\n\n| entity                             | indicator | value        | date       |\n|:-----------------------------------|:----------|:-------------|:-----------|\n| Samoa                              | holdings  | 1.584296e+07 | 2023-01-31 |\n| Iraq                               | holdings  | 3.301367e+07 | 2023-01-31 |\n| Lao People\\\\\'s Democratic Republic | holdings  | 5.870183e+07 | 2023-01-31 |\n| Haiti                              | holdings  | 9.169516e+07 | 2023-01-31 |\n| Bahamas, The                       | holdings  | 1.245326e+08 | 2023-01-31 |\n| Total                              | holdings  | 6.606989e+11 | 2023-01-31 |\n| Libya                              | holdings  | 3.187335e+09 | 2023-01-31 |\n| Namibia                            | holdings  | 1.783556e+08 | 2023-01-31 |\n| Tajikistan, Republic of            | holdings  | 1.891507e+08 | 2023-01-31 |\n| Malta                              | holdings  | 2.499760e+08 | 2023-01-31 |\n\nIn all cases, if you had already downloaded the data and you want to update it\nyou can call `.update_data()` after loading the data in order to refresh it.\n\n```python\nsdr.update_data(reload_data=True)\n```\n\n### Adding World Bank income levels to a DataFrame\n\nFor this example, we will continue using the SDR data as above.\n\n```python\nfrom bblocks import add_income_level_column\n\n# We can add the column by passing the dataframe to the function\n\ndf = add_income_level_column(\n    df=df,\n    id_column="entity",\n    id_type="regex",  # so the text can be matched to the right country\n)\n```\n\nWhich adds the income level column:\n\n| entity                    | indicator | value        | date       | income\\_level       |\n|:--------------------------|:----------|:-------------|:-----------|:--------------------|\n| Montenegro, Republic of   | holdings  | 7.404593e+07 | 2023-01-31 | Upper middle income |\n| Gambia, The               | holdings  | 5.857020e+07 | 2023-01-31 | Low income          |\n| Suriname                  | holdings  | 1.211070e+08 | 2023-01-31 | Upper middle income |\n| Syrian Arab Republic      | holdings  | 5.636629e+08 | 2023-01-31 | Low income          |\n| Iran, Islamic Republic of | holdings  | 4.976198e+09 | 2023-01-31 | Lower middle income |\n| Uruguay                   | holdings  | 6.330507e+08 | 2023-01-31 | High income         |\n| South Africa              | holdings  | 4.424154e+09 | 2023-01-31 | Upper middle income |\n| Nigeria                   | holdings  | 3.755370e+09 | 2023-01-31 | Lower middle income |\n| Dominican Republic        | holdings  | 4.498683e+08 | 2023-01-31 | Upper middle income |\n| Trinidad and Tobago       | holdings  | 7.722810e+08 | 2023-01-31 | High income         |\n\nAn optional argument can be passed to the function to redownload the income classification\ndata from the World Bank.\n\n```python\ndf = add_income_level_column(\n    df=df,\n    id_column="entity",\n    id_type="regex",\n    update_data=True,\n)\n\n```\n\n### Adding a GDP share column to a DataFrame\n\nFor this example, we will continue working with data on military expenditure downloaded using\nthe World Bank tool.\n\n```python\n# First import the function from the `add` module of `dataframe_tools`\nfrom bblocks.dataframe_tools.add import add_gdp_share_column\nfrom bblocks import WorldBankData\n\n# this data is in local currency units\ndf = WorldBankData().load_data(indicator="MS.MIL.XPND.CN", most_recent_only=True).get_data()\n\n```\n\n| date       | iso\\_code | indicator\\_code | value        |\n|:-----------|:----------|:----------------|:-------------|\n| 2021-01-01 | BDI       | MS.MIL.XPND.CN  | 1.351000e+11 |\n| 2014-01-01 | YEM       | MS.MIL.XPND.CN  | 3.685000e+11 |\n| 2021-01-01 | AFG       | MS.MIL.XPND.CN  | 2.304000e+10 |\n| 2021-01-01 | PER       | MS.MIL.XPND.CN  | 9.086000e+09 |\n| 2021-01-01 | AUS       | MS.MIL.XPND.CN  | 4.229595e+10 |\n\n```python\n\n# Then call the function, passing the DataFrame and the column name\ndf = add_gdp_share_column(\n    df=df,\n    id_column="iso_code",\n    id_type="ISO3",\n    date_column="date",  # to match the gdp values with the year of the data\n    value_column="value",\n    decimals=1,\n    usd=False,  # since the data is in local currency units\n    include_estimates=True,  # to include official data and IMF estimates for GDP\n)\n\nprint(df.sample(10))\n```\n\nWhich returns a dataframe with an extra column "gdp_share".\n\n| date       | iso\\_code | indicator\\_code | value        | gdp\\_share |\n|:-----------|:----------|:----------------|:-------------|:-----------|\n| 2021-01-01 | GIN       | MS.MIL.XPND.CN  | 2.406750e+12 | 1.5        |\n| 2014-01-01 | ARE       | MS.MIL.XPND.CN  | 8.356800e+10 | 5.6        |\n| 2021-01-01 | NGA       | MS.MIL.XPND.CN  | 1.783120e+12 | 1.0        |\n| 2021-01-01 | GNQ       | MS.MIL.XPND.CN  | 9.439700e+10 | 1.4        |\n| 2021-01-01 | ISL       | MS.MIL.XPND.CN  | 0.000000e+00 | 0.0        |\n| 2021-01-01 | ESP       | MS.MIL.XPND.CN  | 1.652680e+10 | 1.4        |\n| 2021-01-01 | BHR       | MS.MIL.XPND.CN  | 5.194000e+08 | 3.6        |\n| 2021-01-01 | GEO       | MS.MIL.XPND.CN  | 9.723000e+08 | 1.6        |\n| 2021-01-01 | MDA       | MS.MIL.XPND.CN  | 9.144000e+08 | 0.4        |\n| 2013-01-01 | LAO       | MS.MIL.XPND.CN  | 1.782500e+11 | 0.2        |\n\n### Cleaning a numeric series which contains numbers with text\n\nSometimes dataframes contain columns which don\'t have clean text. For example,\nsomething like\n\n|     | iso\\_code | value   |\n|:----|:----------|:--------|\n| 0   | USA       | 10%     |\n| 1   | GBR       | +12%    |\n| 2   | FRA       | 13.4%   |\n| 3   | DEU       | %14.3   |\n| 4   | ITA       | 15.3  % |\n| 5   | ESP       | 16%     |\n| 6   | CAN       | 17%     |\n| 7   | JPN       | 18%     |\n| 8   | AUS       | 19%     |\n| 9   | CHN       | 20%     |\n\nbblocks can help clean that data.\n\n```python\nfrom bblocks import clean_numeric_series\n\ndf[\'value\'] = clean_numeric_series(\n    data=df[\'value\'],\n    to=float  # or if dealing with integers, use to=int\n)\n\n```\n\nReturns a clean version of the data\n\n|     | iso\\_code | value |\n|:----|:----------|:------|\n| 0   | USA       | 10.0  |\n| 1   | GBR       | 12.0  |\n| 2   | FRA       | 13.4  |\n| 3   | DEU       | 14.3  |\n| 4   | ITA       | 15.3  |\n| 5   | ESP       | 16.0  |\n| 6   | CAN       | 17.0  |\n| 7   | JPN       | 18.0  |\n| 8   | AUS       | 19.0  |\n| 9   | CHN       | 20.0  |\n\n\n# Contributing\nInterested in contributing to the package? Please reach out.',
    'author': 'The ONE Campaign',
    'author_email': 'data@one.org',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
