"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.YamlFile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const YAML = require("yaml");
const json_patch_1 = require("./json-patch");
/**
 * Represents a Yaml File.
 */
class YamlFile {
    constructor(filePath, options = {}) {
        this.filePath = filePath;
        this.obj = options.obj ?? {};
        this.patchOperations = [];
    }
    /**
     * Update the output object.
     */
    update(obj) {
        this.obj = obj;
    }
    /**
     * Applies an RFC 6902 JSON-patch to the synthesized object file.
     * See https://datatracker.ietf.org/doc/html/rfc6902 for more information.
     *
     * For example, with the following yaml file
     * ```yaml
     * name: deploy
     * on:
     *   push:
     *     branches:
     *       - main
     *   workflow_dispatch: {}
     * ...
     * ```
     *
     * modified in the following way:
     *
     * ```ts
     * pipeline.workflowFile.patch(JsonPatch.add("/on/workflow_call", "{}"));
     * pipeline.workflowFile.patch(JsonPatch.remove("/on/workflow_dispatch"));
     * ```
     *
     * would result in the following yaml file:
     *
     * ```yaml
     * name: deploy
     * on:
     *   push:
     *     branches:
     *       - main
     *   workflow_call: {}
     * ...
     * ```
     *
     * @param patches - The patch operations to apply
     */
    patch(...patches) {
        this.patchOperations.push(...patches);
    }
    /**
     * Returns the patched yaml file.
     */
    toYaml() {
        const patched = json_patch_1.JsonPatch.apply(this.obj, ...this.patchOperations);
        return YAML.stringify(patched, {
            indent: 2,
        });
    }
    /**
     * Write the patched yaml file to the specified location.
     */
    writeFile() {
        fs_1.writeFileSync(this.filePath, this.toYaml());
    }
}
exports.YamlFile = YamlFile;
_a = JSII_RTTI_SYMBOL_1;
YamlFile[_a] = { fqn: "cdk-pipelines-github.YamlFile", version: "0.2.196" };
//# sourceMappingURL=data:application/json;base64,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