"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Snapshot matching
// --------------------------------------------------------------
test('Test for default Params snapshot match', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-snapshot', props);
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for ApiGateway params
// --------------------------------------------------------------
test('Test for default Params construct props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    const construct = new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    // Assertion
    expect(construct.apiGateway).not.toBeNull();
    expect(construct.apiGatewayCloudWatchRole).not.toBeNull();
    expect(construct.apiGatewayLogGroup).not.toBeNull();
    expect(construct.apiGatewayRole).not.toBeNull();
});
// --------------------------------------------------------------
// Check for Default IAM Role
// --------------------------------------------------------------
test('Test for default IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "awsapigatewayiotpolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Request Validator
// --------------------------------------------------------------
test('Test for default Params request validator', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-request-validator', props);
    // Assertion
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RequestValidator", {
        ValidateRequestBody: false,
        ValidateRequestParameters: true,
    });
});
// --------------------------------------------------------------
// Check for Integ Props and Method Props
// --------------------------------------------------------------
test('Test for default Params Integ Props and Method Props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-integpros-methodprops', props);
    // Assertion for {topic-level-7} to ensure all Integration Request Params, Integration Responses,
    // Method Request Params and Method Reponses are intact
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "5\\d{2}",
                    StatusCode: "500"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    StatusCode: "403"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.path.topic-level-1": "method.request.path.topic-level-1",
                "integration.request.path.topic-level-2": "method.request.path.topic-level-2",
                "integration.request.path.topic-level-3": "method.request.path.topic-level-3",
                "integration.request.path.topic-level-4": "method.request.path.topic-level-4",
                "integration.request.path.topic-level-5": "method.request.path.topic-level-5",
                "integration.request.path.topic-level-6": "method.request.path.topic-level-6",
                "integration.request.path.topic-level-7": "method.request.path.topic-level-7"
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:${props.iotEndpoint}.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}/{topic-level-4}/{topic-level-5}/{topic-level-6}/{topic-level-7}`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200"
            },
            {
                StatusCode: "500"
            },
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true,
            "method.request.path.topic-level-2": true,
            "method.request.path.topic-level-3": true,
            "method.request.path.topic-level-4": true,
            "method.request.path.topic-level-5": true,
            "method.request.path.topic-level-6": true,
            "method.request.path.topic-level-7": true
        }
    });
});
// --------------------------------------------------------------
// Check for valid IoT Endpoint
// --------------------------------------------------------------
test('Test for valid iot enpoint', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: ' '
    };
    const app = () => {
        new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-no-endpoint', props);
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Check for binaryMediaTypes
// --------------------------------------------------------------
test('Test for Binary Media types', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-binaryMediaTypes', {
        iotEndpoint: 'a1234567890123-ats'
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        BinaryMediaTypes: [
            "application/octet-stream",
        ],
    });
});
// --------------------------------------------------------------
// Check for multiple constructs
// --------------------------------------------------------------
test('Test for multiple constructs usage', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params-1', props);
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for ApiGateway Overriden Props Snapshot match
// --------------------------------------------------------------
test('Test for overriden props snapshot', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api',
        endpointConfiguration: { types: [api.EndpointType.REGIONAL] },
        apiKeySourceType: api.ApiKeySourceType.HEADER,
        defaultMethodOptions: {
            authorizationType: api.AuthorizationType.NONE,
        }
    };
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    // Api gateway setup
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true,
        apiGatewayExecutionRole,
        apiGatewayProps
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-params', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for Api Name and Desc
// --------------------------------------------------------------
test('Test for Api Name and Desc', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api'
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-name-desc', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        Name: 'RestApi-Regional',
        Description: 'Description for the Regional Rest Api'
    });
});
// --------------------------------------------------------------
// Check for Overriden IAM Role
// --------------------------------------------------------------
test('Test for overriden IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/mything1`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/topic-abc`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayExecutionRole,
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/mything1"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/topic-abc"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "testPolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Api Key Source
// --------------------------------------------------------------
test('Test for APi Key Source', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        apiKeySourceType: api.ApiKeySourceType.AUTHORIZER,
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key-source', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        ApiKeySourceType: "AUTHORIZER"
    });
});
// --------------------------------------------------------------
// Check for Api Key Creation
// --------------------------------------------------------------
test('Test for Api Key Creation', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key', props);
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Properties: {
            ApiKeyRequired: true
        },
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W59"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------
// Test deployment for ApiGateway endPointCongiurationOverride
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-auth-none', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.REGIONAL]
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        EndpointConfiguration: {
            Types: ["REGIONAL"]
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for fully qualified iotEndpoint name
// -----------------------------------------------------------------
test('Test for handling fully qualified iotEndpoint', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats.iot.ap-south-1.amazonaws.com',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Integration: {
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        ":a1234567890123-ats.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}"
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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