"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const type_1 = require("./private/type");
const ABSENT = '{{ABSENT}}';
/**
 * (experimental) Partial and special matching during template assertions.
 *
 * @experimental
 */
class Match {
    /**
     * (experimental) Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @experimental
     */
    static absentProperty() {
        return ABSENT;
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * (experimental) Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * (experimental) Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @experimental
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * (experimental) Matches any string-encoded JSON and applies the specified pattern after parsing it.
     *
     * @param pattern the pattern to match after parsing the encoded JSON.
     * @experimental
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * (experimental) Matches any non-null value at the target.
     *
     * @experimental
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.126.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.push(this, [], `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`);
            return result;
        }
        if (this.pattern === ABSENT) {
            throw new Error('absentProperty() can only be used in an object matcher');
        }
        if (actual !== this.pattern) {
            result.push(this, [], `Expected ${this.pattern} but received ${actual}`);
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type array but received ${type_1.getType(actual)}`);
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected array of length ${this.pattern.length} but received ${actual.length}`);
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            if (this.subsequence && matcher instanceof AnyMatch) {
                // array subsequence matcher is not compatible with anyValue() matcher. They don't make sense to be used together.
                throw new Error('The Matcher anyValue() cannot be nested within arrayWith()');
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.push(this, [], `Missing element${element}at pattern index ${patternIdx}`);
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type object but received ${type_1.getType(actual)}`);
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.push(this, [`/${a}`], 'Unexpected key');
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (patternVal === ABSENT) {
                if (patternKey in actual) {
                    result.push(this, [`/${patternKey}`], 'Key should be absent');
                }
                continue;
            }
            if (!(patternKey in actual)) {
                result.push(this, [`/${patternKey}`], 'Missing key');
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.push(this, [], `Expected JSON as a string but found ${type_1.getType(actual)}`);
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.push(this, [], `Invalid JSON string: ${actual}`);
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.push(this, [], `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`);
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.push(this, [], 'Expected a value but found none');
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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