import { IBucket, Location } from '@aws-cdk/aws-s3';
import { AssetOptions } from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
export declare abstract class Seeds {
    /**
     * @returns `S3Seeds` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket: IBucket, key: string, objectVersion?: string): S3Seeds;
    /**
     * @returns `InlineSeeds` with inline seeds.
     * @param seeds The actual json code (limited to 4KiB)
     */
    static fromInline(seeds: Record<string, unknown>[]): InlineSeeds;
    /**
     * Loads the seeds from a local disk path and uploads it to s3.
     *
     * @returns `JsonFileSeeds` associated with the specified S3 object.
     * @param path Path to json seeds file.
     */
    static fromJsonFile(path: string, options?: AssetOptions): JsonFileSeeds;
    /**
     * Called when the seeder is initialized to allow this object to bind
     * to the stack.
     *
     * @param scope The binding scope.
     */
    abstract bind(scope: Construct): SeedsConfig;
}
export interface SeedsConfig {
    /**
     * The location of the seeds in S3.
     */
    readonly s3Location?: Location;
    /**
     * Inline seeds.
     */
    readonly inlineSeeds?: string;
}
/**
 * Seeds from an S3 archive.
 */
export declare class S3Seeds extends Seeds {
    private key;
    private objectVersion?;
    private bucketName;
    constructor(bucket: IBucket, key: string, objectVersion?: string | undefined);
    bind(_scope: Construct): SeedsConfig;
}
/**
 * Seeds from an inline json object (limited to 4KiB).
 */
export declare class InlineSeeds extends Seeds {
    private seeds;
    constructor(seeds: string);
    bind(_scope: Construct): SeedsConfig;
}
/**
 * Seeds from a local json file.
 */
export declare class JsonFileSeeds extends Seeds {
    readonly path: string;
    private readonly options;
    private asset?;
    constructor(path: string, options?: AssetOptions);
    bind(scope: Construct): SeedsConfig;
}
