"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = exports.WordPress = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const efs = require("@aws-cdk/aws-efs");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const cdk = require("@aws-cdk/core");
const patterns = require("./index");
class WordPress extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : getOrCreateVpc(this);
        this.db = this.addDatabase({
            vpc: this.vpc,
            databaseSubnets: props.databaseSubnets,
            instanceType: props.databaseInstanceType,
            instanceEngine: props.instanceEngine,
            clusterEngine: props.clusterEngine,
            auroraServerless: props.auroraServerless,
            singleDbInstance: props.singleDbInstance,
            backupRetention: props.backupRetention,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const task = new ecs.FargateTaskDefinition(this, 'Task', {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        // bootstrap container that creates the database if not exist
        const bootstrap = task.addContainer('bootstrap', {
            essential: false,
            image: ecs.ContainerImage.fromRegistry('public.ecr.aws/ubuntu/mysql:latest'),
            environment: {
                DB_NAME: 'wordpress',
            },
            secrets: {
                WORDPRESS_DB_HOST: ecs.Secret.fromSecretsManager(this.db.secret, 'host'),
                WORDPRESS_DB_USER: ecs.Secret.fromSecretsManager(this.db.secret, 'username'),
                WORDPRESS_DB_PASSWORD: ecs.Secret.fromSecretsManager(this.db.secret, 'password'),
            },
            command: [
                'sh', '-c',
                'mysql -u$WORDPRESS_DB_USER -p$WORDPRESS_DB_PASSWORD -h$WORDPRESS_DB_HOST -e "CREATE DATABASE IF NOT EXISTS $DB_NAME"',
            ],
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'bootstrap',
                logGroup,
            }),
        });
        const wp = task.addContainer('wordpress', {
            image: ecs.ContainerImage.fromRegistry('wordpress:latest'),
            portMappings: [{ containerPort: 80 }],
            environment: {
                WORDPRESS_DB_NAME: 'wordpress',
            },
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'wordpress-fargate',
                logGroup,
            }),
            secrets: {
                WORDPRESS_DB_HOST: ecs.Secret.fromSecretsManager(this.db.secret, 'host'),
                WORDPRESS_DB_USER: ecs.Secret.fromSecretsManager(this.db.secret, 'username'),
                WORDPRESS_DB_PASSWORD: ecs.Secret.fromSecretsManager(this.db.secret, 'password'),
            },
        });
        wp.addContainerDependencies({
            container: bootstrap,
            condition: ecs.ContainerDependencyCondition.SUCCESS,
        });
        this.svc = new patterns.DualAlbFargateService(this, 'ALBFargateService', {
            spot: true,
            enableExecuteCommand: true,
            tasks: [
                {
                    listenerPort: 80,
                    accessibility: patterns.LoadBalancerAccessibility.EXTERNAL_ONLY,
                    task,
                    healthCheck: {
                        path: '/wp-includes/images/blank.gif',
                        interval: cdk.Duration.minutes(1),
                    },
                },
            ],
            route53Ops: {
                enableLoadBalancerAlias: false,
            },
        });
        // EFS volume
        const filesystem = new efs.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
        });
        const volumeName = 'efs';
        this.svc.service[0].taskDefinition.addVolume({
            name: volumeName,
            efsVolumeConfiguration: {
                fileSystemId: filesystem.fileSystemId,
            },
        });
        (_b = this.svc.service[0].taskDefinition.defaultContainer) === null || _b === void 0 ? void 0 : _b.addMountPoints({
            containerPath: '/var/www/html',
            readOnly: false,
            sourceVolume: volumeName,
        });
        filesystem.connections.allowFrom(this.svc.service[0], ec2.Port.tcp(2049));
    }
    addDatabase(props) {
        return new Database(this, 'Database', {
            ...props,
        });
    }
}
exports.WordPress = WordPress;
/**
 * Represents the database instance or database cluster
 */
class Database extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mysqlListenerPort = 3306;
        this.vpc = props.vpc;
        const config = props.auroraServerless ? this._createServerlessCluster(props)
            : props.singleDbInstance ? this._createRdsInstance(props) : this._createRdsCluster(props);
        this.secret = config.secret;
        // allow internally from the same security group
        config.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the whole vpc cidr
        config.connections.allowFrom((_a = props.allowFrom) !== null && _a !== void 0 ? _a : ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.clusterEndpointHostname = config.endpoint;
        this.clusterIdentifier = config.identifier;
        this.connections = config.connections;
        printOutput(this, 'DBSecretArn', config.secret.secretArn);
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
    }
    _createRdsInstance(props) {
        var _a, _b, _c;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            engine: (_a = props.instanceEngine) !== null && _a !== void 0 ? _a : rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_21,
            }),
            storageEncrypted: true,
            backupRetention: (_b = props.backupRetention) !== null && _b !== void 0 ? _b : cdk.Duration.days(7),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_c = props.instanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbInstance.connections,
            endpoint: dbInstance.dbInstanceEndpointAddress,
            identifier: dbInstance.instanceIdentifier,
            secret: dbInstance.secret,
        };
    }
    // create a RDS for MySQL DB cluster
    _createRdsCluster(props) {
        var _a, _b, _c;
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: (_a = props.clusterEngine) !== null && _a !== void 0 ? _a : rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_09_1,
            }),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceProps: {
                vpc: props.vpc,
                vpcSubnets: props.databaseSubnets,
                instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType('r5.large'),
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            backup: {
                retention: (_c = props.backupRetention) !== null && _c !== void 0 ? _c : cdk.Duration.days(7),
            },
            storageEncrypted: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
    _createServerlessCluster(props) {
        var _a;
        const dbCluster = new rds.ServerlessCluster(this, 'AuroraServerlessCluster', {
            engine: rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            backupRetention: (_a = props.backupRetention) !== null && _a !== void 0 ? _a : cdk.Duration.days(7),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
}
exports.Database = Database;
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,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