"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const main_1 = require("./main");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel service
const laravelTask = new ecs.FargateTaskDefinition(stack, 'laravelTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelTask.addContainer('laravel', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            listenerPort: 80,
            accessibility: main_1.LoadBalancerAccessibility.EXTERNAL_ONLY,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY,
            listenerPort: 8080,
            task: customerTask,
            desiredCount: 1,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        { listenerPort: 9090, task: productTask, desiredCount: 1, accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY },
        // The nginx service(external only)
        { listenerPort: 9091, task: nginxTask, desiredCount: 1, accessibility: main_1.LoadBalancerAccessibility.EXTERNAL_ONLY },
        // The nginx-php-fpm service(external/internal)
        { listenerPort: 9092, task: phpTask, desiredCount: 1 },
        // The NuxtJS service(external/internal)
        { listenerPort: 9093, task: nuxtTask, desiredCount: 1 },
        // The node service(external/internal)
        { listenerPort: 9094, task: nodeTask, desiredCount: 1 },
        // The laravel service(external/internal)
        { listenerPort: 9095, task: laravelTask, desiredCount: 1 },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZGVmYXVsdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9pbnRlZy5kZWZhdWx0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNkJBQTZCO0FBQzdCLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMscUNBQXFDO0FBQ3JDLG1DQUFnRDtBQUNoRCxpQ0FBbUQ7QUFHbkQsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7QUFFMUIsTUFBTSxHQUFHLEdBQUc7SUFDVixNQUFNLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0I7SUFDdEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUJBQW1CO0NBQ3pDLENBQUM7QUFFRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7QUFFekQsTUFBTSxRQUFRLEdBQUcsV0FBVyxDQUFDO0FBQzdCLE1BQU0scUJBQXFCLEdBQUcsVUFBVSxDQUFDO0FBQ3pDLE1BQU0scUJBQXFCLEdBQUcsVUFBVSxDQUFDO0FBQ3pDLE1BQU0sbUJBQW1CLEdBQUcsVUFBVSxxQkFBcUIsSUFBSSxRQUFRLEVBQUUsQ0FBQztBQUUxRSxnQkFBZ0I7QUFDaEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtJQUNsRSxHQUFHLEVBQUUsR0FBRztJQUNSLGNBQWMsRUFBRSxHQUFHO0NBQ3BCLENBQUMsQ0FBQztBQUVILFNBQVMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO0lBQzlCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO0lBQ3JGLFlBQVksRUFBRTtRQUNaLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRTtLQUN4QjtJQUNELFdBQVcsRUFBRTtRQUNYLGVBQWUsRUFBRSxHQUFHLG1CQUFtQixPQUFPO1FBQzlDLGdCQUFnQixFQUFFLEdBQUcsbUJBQW1CLE9BQU87UUFDL0MsV0FBVyxFQUFFLE9BQU87UUFDcEIsVUFBVSxFQUFFLEtBQUs7S0FDbEI7Q0FDRixDQUFDLENBQUM7QUFFSCxtQkFBbUI7QUFDbkIsTUFBTSxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtJQUN4RSxHQUFHLEVBQUUsR0FBRztJQUNSLGNBQWMsRUFBRSxHQUFHO0NBQ3BCLENBQUMsQ0FBQztBQUVILFlBQVksQ0FBQyxZQUFZLENBQUMsVUFBVSxFQUFFO0lBQ3BDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO0lBQ3RGLFlBQVksRUFBRTtRQUNaLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRTtLQUN4QjtJQUNELFdBQVcsRUFBRTtRQUNYLGVBQWUsRUFBRSxHQUFHLG1CQUFtQixPQUFPO1FBQzlDLGdCQUFnQixFQUFFLEdBQUcsbUJBQW1CLE9BQU87UUFDL0MsV0FBVyxFQUFFLFVBQVU7UUFDdkIsVUFBVSxFQUFFLEtBQUs7S0FDbEI7Q0FDRixDQUFDLENBQUM7QUFFSCxrQkFBa0I7QUFDbEIsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtJQUN0RSxHQUFHLEVBQUUsR0FBRztJQUNSLGNBQWMsRUFBRSxHQUFHO0NBQ3BCLENBQUMsQ0FBQztBQUVILFdBQVcsQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFO0lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO0lBQ3RGLFlBQVksRUFBRTtRQUNaLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRTtLQUN4QjtJQUNELFdBQVcsRUFBRTtRQUNYLGVBQWUsRUFBRSxHQUFHLG1CQUFtQixPQUFPO1FBQzlDLGdCQUFnQixFQUFFLEdBQUcsbUJBQW1CLE9BQU87UUFDL0MsV0FBVyxFQUFFLFNBQVM7UUFDdEIsVUFBVSxFQUFFLEtBQUs7S0FDbEI7Q0FDRixDQUFDLENBQUM7QUFFSCxnQkFBZ0I7QUFDaEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtJQUNsRSxHQUFHLEVBQUUsR0FBRztJQUNSLGNBQWMsRUFBRSxHQUFHO0NBQ3BCLENBQUMsQ0FBQztBQUVILFNBQVMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO0lBQzlCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUM7SUFDdEQsWUFBWSxFQUFFO1FBQ1osRUFBRSxhQUFhLEVBQUUsRUFBRSxFQUFFO0tBQ3RCO0NBQ0YsQ0FBQyxDQUFDO0FBRUgsY0FBYztBQUNkLE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7SUFDOUQsR0FBRyxFQUFFLEdBQUc7SUFDUixjQUFjLEVBQUUsR0FBRztDQUNwQixDQUFDLENBQUM7QUFFSCxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRTtJQUMxQixLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztJQUNsRixZQUFZLEVBQUU7UUFDWixFQUFFLGFBQWEsRUFBRSxFQUFFLEVBQUU7S0FDdEI7Q0FDRixDQUFDLENBQUM7QUFFSCxrQkFBa0I7QUFDbEIsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtJQUN0RSxHQUFHLEVBQUUsR0FBRztJQUNSLGNBQWMsRUFBRSxHQUFHO0NBQ3BCLENBQUMsQ0FBQztBQUVILFdBQVcsQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFO0lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO0lBQ2hGLFlBQVksRUFBRTtRQUNaLEVBQUUsYUFBYSxFQUFFLEVBQUUsRUFBRTtLQUN0QjtDQUNGLENBQUMsQ0FBQztBQUVILGlCQUFpQjtBQUNqQixNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO0lBQ2hFLEdBQUcsRUFBRSxHQUFHO0lBQ1IsY0FBYyxFQUFFLEdBQUc7Q0FDcEIsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7SUFDNUIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFDN0UsWUFBWSxFQUFFO1FBQ1osRUFBRSxhQUFhLEVBQUUsRUFBRSxFQUFFO0tBQ3RCO0NBQ0YsQ0FBQyxDQUFDO0FBRUgsa0JBQWtCO0FBQ2xCLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7SUFDaEUsR0FBRyxFQUFFLEdBQUc7SUFDUixjQUFjLEVBQUUsR0FBRztDQUNwQixDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRTtJQUM1QixLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztJQUM3RSxZQUFZLEVBQUU7UUFDWixFQUFFLGFBQWEsRUFBRSxFQUFFLEVBQUU7S0FDdEI7Q0FDRixDQUFDLENBQUM7QUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLDZCQUFxQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7SUFDdEQsSUFBSSxFQUFFLElBQUk7SUFDVixvQkFBb0IsRUFBRSxJQUFJO0lBQzFCLEtBQUssRUFBRTtRQUNMLHVEQUF1RDtRQUN2RDtZQUNFLFlBQVksRUFBRSxFQUFFO1lBQ2hCLGFBQWEsRUFBRSxnQ0FBeUIsQ0FBQyxhQUFhO1lBQ3RELElBQUksRUFBRSxTQUFTO1lBQ2YsWUFBWSxFQUFFLENBQUM7WUFDZiwyQ0FBMkM7WUFDM0MsYUFBYSxFQUFFO2dCQUNiLFdBQVcsRUFBRSxFQUFFO2dCQUNmLGdCQUFnQixFQUFFLElBQUk7Z0JBQ3RCLG9CQUFvQixFQUFFLEVBQUU7YUFDekI7U0FDRjtRQUNEO1lBQ0Usc0NBQXNDO1lBQ3RDLGFBQWEsRUFBRSxnQ0FBeUIsQ0FBQyxhQUFhO1lBQ3RELFlBQVksRUFBRSxJQUFJO1lBQ2xCLElBQUksRUFBRSxZQUFZO1lBQ2xCLFlBQVksRUFBRSxDQUFDO1lBQ2Ysd0JBQXdCLEVBQUU7Z0JBQ3hCO29CQUNFLGdCQUFnQixFQUFFLFNBQVM7b0JBQzNCLElBQUksRUFBRSxDQUFDO29CQUNQLE1BQU0sRUFBRSxDQUFDO2lCQUNWO2dCQUNEO29CQUNFLGdCQUFnQixFQUFFLGNBQWM7b0JBQ2hDLElBQUksRUFBRSxDQUFDO29CQUNQLE1BQU0sRUFBRSxDQUFDO2lCQUNWO2FBQ0Y7U0FDRjtRQUNELHFDQUFxQztRQUNyQyxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLFdBQVcsRUFBRSxZQUFZLEVBQUUsQ0FBQyxFQUFFLGFBQWEsRUFBRSxnQ0FBeUIsQ0FBQyxhQUFhLEVBQUU7UUFDbEgsbUNBQW1DO1FBQ25DLEVBQUUsWUFBWSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxDQUFDLEVBQUUsYUFBYSxFQUFFLGdDQUF5QixDQUFDLGFBQWEsRUFBRTtRQUNoSCwrQ0FBK0M7UUFDL0MsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsWUFBWSxFQUFFLENBQUMsRUFBRTtRQUN0RCx3Q0FBd0M7UUFDeEMsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsWUFBWSxFQUFFLENBQUMsRUFBRTtRQUN2RCxzQ0FBc0M7UUFDdEMsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsWUFBWSxFQUFFLENBQUMsRUFBRTtRQUN2RCx5Q0FBeUM7UUFDekMsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsWUFBWSxFQUFFLENBQUMsRUFBRTtLQUMzRDtJQUNELFVBQVUsRUFBRTtRQUNWLFFBQVE7UUFDUixxQkFBcUI7UUFDckIscUJBQXFCO0tBQ3RCO0NBQ0YsQ0FBQyxDQUFDO0FBR0gsb0JBQW9CO0FBQ3BCLE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO0lBQ3RELEdBQUcsRUFBRSxHQUFHLENBQUMsR0FBRztDQUNiLENBQUMsQ0FBQztBQUVILHNEQUFzRDtBQUN0RCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7SUFDM0MsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7Q0FDdEUiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgRHVhbEFsYkZhcmdhdGVTZXJ2aWNlIH0gZnJvbSAnLi9pbmRleCc7XG5pbXBvcnQgeyBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5IH0gZnJvbSAnLi9tYWluJztcblxuXG5jb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuXG5jb25zdCBlbnYgPSB7XG4gIHJlZ2lvbjogcHJvY2Vzcy5lbnYuQ0RLX0RFRkFVTFRfUkVHSU9OLFxuICBhY2NvdW50OiBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5ULFxufTtcblxuY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ2RlbW8tc3RhY2syJywgeyBlbnYgfSk7XG5cbmNvbnN0IHpvbmVOYW1lID0gJ3N2Yy5sb2NhbCc7XG5jb25zdCBpbnRlcm5hbEFsYlJlY29yZE5hbWUgPSAnaW50ZXJuYWwnO1xuY29uc3QgZXh0ZXJuYWxBbGJSZWNvcmROYW1lID0gJ2V4dGVybmFsJztcbmNvbnN0IGludGVybmFsQUxCRW5kcG9pbnQgPSBgaHR0cDovLyR7aW50ZXJuYWxBbGJSZWNvcmROYW1lfS4ke3pvbmVOYW1lfWA7XG5cbi8vIG9yZGVyIHNlcnZpY2VcbmNvbnN0IG9yZGVyVGFzayA9IG5ldyBlY3MuRmFyZ2F0ZVRhc2tEZWZpbml0aW9uKHN0YWNrLCAnb3JkZXJUYXNrJywge1xuICBjcHU6IDI1NixcbiAgbWVtb3J5TGltaXRNaUI6IDUxMixcbn0pO1xuXG5vcmRlclRhc2suYWRkQ29udGFpbmVyKCdvcmRlcicsIHtcbiAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NlcnZpY2VzL09yZGVyU2VydmljZScpKSxcbiAgcG9ydE1hcHBpbmdzOiBbXG4gICAgeyBjb250YWluZXJQb3J0OiA4MDgwIH0sXG4gIF0sXG4gIGVudmlyb25tZW50OiB7XG4gICAgUFJPRFVDVF9TVkNfVVJMOiBgJHtpbnRlcm5hbEFMQkVuZHBvaW50fTo5MDkwYCxcbiAgICBDVVNUT01FUl9TVkNfVVJMOiBgJHtpbnRlcm5hbEFMQkVuZHBvaW50fTo4MDgwYCxcbiAgICBzZXJ2aWNlTmFtZTogJ29yZGVyJyxcbiAgICB2ZXJzaW9uTnVtOiAnMS4wJyxcbiAgfSxcbn0pO1xuXG4vLyBjdXN0b21lciBzZXJ2aWNlXG5jb25zdCBjdXN0b21lclRhc2sgPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ2N1c3RvbWVyVGFzaycsIHtcbiAgY3B1OiAyNTYsXG4gIG1lbW9yeUxpbWl0TWlCOiA1MTIsXG59KTtcblxuY3VzdG9tZXJUYXNrLmFkZENvbnRhaW5lcignY3VzdG9tZXInLCB7XG4gIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zZXJ2aWNlcy9Db21tb25TZXJ2aWNlJykpLFxuICBwb3J0TWFwcGluZ3M6IFtcbiAgICB7IGNvbnRhaW5lclBvcnQ6IDgwODAgfSxcbiAgXSxcbiAgZW52aXJvbm1lbnQ6IHtcbiAgICBQUk9EVUNUX1NWQ19VUkw6IGAke2ludGVybmFsQUxCRW5kcG9pbnR9OjkwOTBgLFxuICAgIENVU1RPTUVSX1NWQ19VUkw6IGAke2ludGVybmFsQUxCRW5kcG9pbnR9OjgwODBgLFxuICAgIHNlcnZpY2VOYW1lOiAnY3VzdG9tZXInLFxuICAgIHZlcnNpb25OdW06ICcxLjAnLFxuICB9LFxufSk7XG5cbi8vIHByb2R1Y3Qgc2VydmljZVxuY29uc3QgcHJvZHVjdFRhc2sgPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ3Byb2R1Y3RUYXNrJywge1xuICBjcHU6IDI1NixcbiAgbWVtb3J5TGltaXRNaUI6IDUxMixcbn0pO1xuXG5wcm9kdWN0VGFzay5hZGRDb250YWluZXIoJ3Byb2R1Y3QnLCB7XG4gIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zZXJ2aWNlcy9Db21tb25TZXJ2aWNlJykpLFxuICBwb3J0TWFwcGluZ3M6IFtcbiAgICB7IGNvbnRhaW5lclBvcnQ6IDgwODAgfSxcbiAgXSxcbiAgZW52aXJvbm1lbnQ6IHtcbiAgICBQUk9EVUNUX1NWQ19VUkw6IGAke2ludGVybmFsQUxCRW5kcG9pbnR9OjkwOTBgLFxuICAgIENVU1RPTUVSX1NWQ19VUkw6IGAke2ludGVybmFsQUxCRW5kcG9pbnR9OjgwODBgLFxuICAgIHNlcnZpY2VOYW1lOiAncHJvZHVjdCcsXG4gICAgdmVyc2lvbk51bTogJzEuMCcsXG4gIH0sXG59KTtcblxuLy8gbmdpbnggc2VydmljZVxuY29uc3QgbmdpbnhUYXNrID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICduZ2lueFRhc2snLCB7XG4gIGNwdTogMjU2LFxuICBtZW1vcnlMaW1pdE1pQjogNTEyLFxufSk7XG5cbm5naW54VGFzay5hZGRDb250YWluZXIoJ25naW54Jywge1xuICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnbmdpbng6bGF0ZXN0JyksXG4gIHBvcnRNYXBwaW5nczogW1xuICAgIHsgY29udGFpbmVyUG9ydDogODAgfSxcbiAgXSxcbn0pO1xuXG4vLyBwaHAgc2VydmljZVxuY29uc3QgcGhwVGFzayA9IG5ldyBlY3MuRmFyZ2F0ZVRhc2tEZWZpbml0aW9uKHN0YWNrLCAncGhwVGFzaycsIHtcbiAgY3B1OiAyNTYsXG4gIG1lbW9yeUxpbWl0TWlCOiA1MTIsXG59KTtcblxucGhwVGFzay5hZGRDb250YWluZXIoJ3BocCcsIHtcbiAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NlcnZpY2VzL25naW54LXBocCcpKSxcbiAgcG9ydE1hcHBpbmdzOiBbXG4gICAgeyBjb250YWluZXJQb3J0OiA4MCB9LFxuICBdLFxufSk7XG5cbi8vIGxhcmF2ZWwgc2VydmljZVxuY29uc3QgbGFyYXZlbFRhc2sgPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ2xhcmF2ZWxUYXNrJywge1xuICBjcHU6IDI1NixcbiAgbWVtb3J5TGltaXRNaUI6IDUxMixcbn0pO1xuXG5sYXJhdmVsVGFzay5hZGRDb250YWluZXIoJ2xhcmF2ZWwnLCB7XG4gIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zZXJ2aWNlcy9sYXJhdmVsJykpLFxuICBwb3J0TWFwcGluZ3M6IFtcbiAgICB7IGNvbnRhaW5lclBvcnQ6IDgwIH0sXG4gIF0sXG59KTtcblxuLy8gTnV4dEpTIHNlcnZpY2VcbmNvbnN0IG51eHRUYXNrID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdudXh0VGFzaycsIHtcbiAgY3B1OiAyNTYsXG4gIG1lbW9yeUxpbWl0TWlCOiA1MTIsXG59KTtcblxubnV4dFRhc2suYWRkQ29udGFpbmVyKCdudXh0Jywge1xuICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vc2VydmljZXMvbnV4dCcpKSxcbiAgcG9ydE1hcHBpbmdzOiBbXG4gICAgeyBjb250YWluZXJQb3J0OiA4MCB9LFxuICBdLFxufSk7XG5cbi8vIE5vZGUuanMgc2VydmljZVxuY29uc3Qgbm9kZVRhc2sgPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ25vZGVUYXNrJywge1xuICBjcHU6IDI1NixcbiAgbWVtb3J5TGltaXRNaUI6IDUxMixcbn0pO1xuXG5ub2RlVGFzay5hZGRDb250YWluZXIoJ25vZGUnLCB7XG4gIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zZXJ2aWNlcy9ub2RlJykpLFxuICBwb3J0TWFwcGluZ3M6IFtcbiAgICB7IGNvbnRhaW5lclBvcnQ6IDgwIH0sXG4gIF0sXG59KTtcblxuY29uc3Qgc3ZjID0gbmV3IER1YWxBbGJGYXJnYXRlU2VydmljZShzdGFjaywgJ1NlcnZpY2UnLCB7XG4gIHNwb3Q6IHRydWUsIC8vIEZBUkdBVEVfU1BPVCBvbmx5IGNsdXN0ZXJcbiAgZW5hYmxlRXhlY3V0ZUNvbW1hbmQ6IHRydWUsXG4gIHRhc2tzOiBbXG4gICAgLy8gVGhlIG9yZGVyIHNlcnZpY2Ugd2l0aCBib3RoIGV4dGVybmFsL2ludGVybmFsIGFjY2Vzc1xuICAgIHtcbiAgICAgIGxpc3RlbmVyUG9ydDogODAsXG4gICAgICBhY2Nlc3NpYmlsaXR5OiBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5LkVYVEVSTkFMX09OTFksXG4gICAgICB0YXNrOiBvcmRlclRhc2ssXG4gICAgICBkZXNpcmVkQ291bnQ6IDIsXG4gICAgICAvLyBjdXN0b21pemUgdGhlIHNlcnZpY2UgYXV0b3NjYWxpbmcgcG9saWN5XG4gICAgICBzY2FsaW5nUG9saWN5OiB7XG4gICAgICAgIG1heENhcGFjaXR5OiAyMCxcbiAgICAgICAgcmVxdWVzdFBlclRhcmdldDogMTAwMCxcbiAgICAgICAgdGFyZ2V0Q3B1VXRpbGl6YXRpb246IDUwLFxuICAgICAgfSxcbiAgICB9LFxuICAgIHtcbiAgICAgIC8vIFRoZSBjdXN0b21lciBzZXJ2aWNlKGludGVybmFsIG9ubHkpXG4gICAgICBhY2Nlc3NpYmlsaXR5OiBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5LklOVEVSTkFMX09OTFksXG4gICAgICBsaXN0ZW5lclBvcnQ6IDgwODAsXG4gICAgICB0YXNrOiBjdXN0b21lclRhc2ssXG4gICAgICBkZXNpcmVkQ291bnQ6IDEsXG4gICAgICBjYXBhY2l0eVByb3ZpZGVyU3RyYXRlZ3k6IFtcbiAgICAgICAge1xuICAgICAgICAgIGNhcGFjaXR5UHJvdmlkZXI6ICdGQVJHQVRFJyxcbiAgICAgICAgICBiYXNlOiAxLFxuICAgICAgICAgIHdlaWdodDogMSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIGNhcGFjaXR5UHJvdmlkZXI6ICdGQVJHQVRFX1NQT1QnLFxuICAgICAgICAgIGJhc2U6IDAsXG4gICAgICAgICAgd2VpZ2h0OiAzLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9LFxuICAgIC8vIFRoZSBwcm9kdWNlIHNlcnZpY2UoaW50ZXJuYWwgb25seSlcbiAgICB7IGxpc3RlbmVyUG9ydDogOTA5MCwgdGFzazogcHJvZHVjdFRhc2ssIGRlc2lyZWRDb3VudDogMSwgYWNjZXNzaWJpbGl0eTogTG9hZEJhbGFuY2VyQWNjZXNzaWJpbGl0eS5JTlRFUk5BTF9PTkxZIH0sXG4gICAgLy8gVGhlIG5naW54IHNlcnZpY2UoZXh0ZXJuYWwgb25seSlcbiAgICB7IGxpc3RlbmVyUG9ydDogOTA5MSwgdGFzazogbmdpbnhUYXNrLCBkZXNpcmVkQ291bnQ6IDEsIGFjY2Vzc2liaWxpdHk6IExvYWRCYWxhbmNlckFjY2Vzc2liaWxpdHkuRVhURVJOQUxfT05MWSB9LFxuICAgIC8vIFRoZSBuZ2lueC1waHAtZnBtIHNlcnZpY2UoZXh0ZXJuYWwvaW50ZXJuYWwpXG4gICAgeyBsaXN0ZW5lclBvcnQ6IDkwOTIsIHRhc2s6IHBocFRhc2ssIGRlc2lyZWRDb3VudDogMSB9LFxuICAgIC8vIFRoZSBOdXh0SlMgc2VydmljZShleHRlcm5hbC9pbnRlcm5hbClcbiAgICB7IGxpc3RlbmVyUG9ydDogOTA5MywgdGFzazogbnV4dFRhc2ssIGRlc2lyZWRDb3VudDogMSB9LFxuICAgIC8vIFRoZSBub2RlIHNlcnZpY2UoZXh0ZXJuYWwvaW50ZXJuYWwpXG4gICAgeyBsaXN0ZW5lclBvcnQ6IDkwOTQsIHRhc2s6IG5vZGVUYXNrLCBkZXNpcmVkQ291bnQ6IDEgfSxcbiAgICAvLyBUaGUgbGFyYXZlbCBzZXJ2aWNlKGV4dGVybmFsL2ludGVybmFsKVxuICAgIHsgbGlzdGVuZXJQb3J0OiA5MDk1LCB0YXNrOiBsYXJhdmVsVGFzaywgZGVzaXJlZENvdW50OiAxIH0sXG4gIF0sXG4gIHJvdXRlNTNPcHM6IHtcbiAgICB6b25lTmFtZSwgLy8gc3ZjLmxvY2FsXG4gICAgZXh0ZXJuYWxBbGJSZWNvcmROYW1lLCAvLyBleHRlcm5hbC5zdmMubG9jYWxcbiAgICBpbnRlcm5hbEFsYlJlY29yZE5hbWUsIC8vIGludGVybmFsLnN2Yy5sb2NhbFxuICB9LFxufSk7XG5cblxuLy8gY3JlYXRlIGEgZHVtbXkgc2dcbmNvbnN0IGR1bW15U2cgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoc3RhY2ssICdEdW1teVNHJywge1xuICB2cGM6IHN2Yy52cGMsXG59KTtcblxuLy8gYWxsb3cgYWxsIHRyYWZmaWMgZnJvbSBkdW1teSBzZyB0byBhbGwgdGhlIHNlcnZpY2VzXG5mb3IgKGxldCBpID0gMDsgaSA8IHN2Yy5zZXJ2aWNlLmxlbmd0aDsgaSsrKSB7XG4gIHN2Yy5zZXJ2aWNlW2ldLmNvbm5lY3Rpb25zLmFsbG93RnJvbShkdW1teVNnLCBlYzIuUG9ydC5hbGxUcmFmZmljKCkpO1xufVxuIl19