# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['proc_wrapper']

package_data = \
{'': ['*']}

extras_require = \
{'docs': ['Sphinx>=3.3.0,<4.0.0',
          'sphinx-rtd-theme>=0.5.0,<0.6.0',
          'myst-parser>=0.13.0,<0.14.0']}

setup_kwargs = {
    'name': 'cloudreactor-procwrapper',
    'version': '3.0.0a1',
    'description': 'Wraps the execution of processes so that a service API endpoint (CloudReactor) can monitor and manage them. Also implements retries, timeouts, and secret injection from AWS into the environment.',
    'long_description': '# cloudreactor-procwrapper\n\n<p align="center">\n  <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/actions?query=workflow%3ACI">\n    <img src="https://img.shields.io/github/workflow/status/CloudReactor/cloudreactor-procwrapper/CI/main?label=CI&logo=github&style=flat-square" alt="CI Status" >\n  </a>\n  <a href="https://cloudreactor-procwrapper.readthedocs.io">\n    <img src="https://img.shields.io/readthedocs/cloudreactor-procwrapper.svg?logo=read-the-docs&logoColor=fff&style=flat-square" alt="Documentation Status">\n  </a>\n  <a href="https://codecov.io/gh/CloudReactor/cloudreactor-procwrapper">\n    <img src="https://img.shields.io/codecov/c/github/CloudReactor/cloudreactor-procwrapper.svg?logo=codecov&logoColor=fff&style=flat-square" alt="Test coverage percentage">\n  </a>\n</p>\n<p align="center">\n  <a href="https://python-poetry.org/">\n    <img src="https://img.shields.io/badge/packaging-poetry-299bd7?style=flat-square&logo=data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAA4AAAASCAYAAABrXO8xAAAACXBIWXMAAAsTAAALEwEAmpwYAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAJJSURBVHgBfZLPa1NBEMe/s7tNXoxW1KJQKaUHkXhQvHgW6UHQQ09CBS/6V3hKc/AP8CqCrUcpmop3Cx48eDB4yEECjVQrlZb80CRN8t6OM/teagVxYZi38+Yz853dJbzoMV3MM8cJUcLMSUKIE8AzQ2PieZzFxEJOHMOgMQQ+dUgSAckNXhapU/NMhDSWLs1B24A8sO1xrN4NECkcAC9ASkiIJc6k5TRiUDPhnyMMdhKc+Zx19l6SgyeW76BEONY9exVQMzKExGKwwPsCzza7KGSSWRWEQhyEaDXp6ZHEr416ygbiKYOd7TEWvvcQIeusHYMJGhTwF9y7sGnSwaWyFAiyoxzqW0PM/RjghPxF2pWReAowTEXnDh0xgcLs8l2YQmOrj3N7ByiqEoH0cARs4u78WgAVkoEDIDoOi3AkcLOHU60RIg5wC4ZuTC7FaHKQm8Hq1fQuSOBvX/sodmNJSB5geaF5CPIkUeecdMxieoRO5jz9bheL6/tXjrwCyX/UYBUcjCaWHljx1xiX6z9xEjkYAzbGVnB8pvLmyXm9ep+W8CmsSHQQY77Zx1zboxAV0w7ybMhQmfqdmmw3nEp1I0Z+FGO6M8LZdoyZnuzzBdjISicKRnpxzI9fPb+0oYXsNdyi+d3h9bm9MWYHFtPeIZfLwzmFDKy1ai3p+PDls1Llz4yyFpferxjnyjJDSEy9CaCx5m2cJPerq6Xm34eTrZt3PqxYO1XOwDYZrFlH1fWnpU38Y9HRze3lj0vOujZcXKuuXm3jP+s3KbZVra7y2EAAAAAASUVORK5CYII=" alt="Poetry">\n  </a>\n  <a href="https://github.com/pre-commit/pre-commit">\n    <img src="https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white&style=flat-square" alt="pre-commit">\n  </a>\n</p>\n<p align="center">\n  <a href="https://pypi.org/project/cloudreactor-procwrapper/">\n    <img src="https://img.shields.io/pypi/v/cloudreactor-procwrapper.svg?logo=python&logoColor=fff&style=flat-square" alt="PyPI Version">\n  </a>\n  <img src="https://img.shields.io/pypi/pyversions/cloudreactor-procwrapper.svg?style=flat-square&logo=python&amp;logoColor=fff" alt="Supported Python versions">\n  <img src="https://img.shields.io/pypi/l/cloudreactor-procwrapper.svg?style=flat-square" alt="License">\n</p>\n\nWraps the execution of processes so that an API server\n([CloudReactor](https://cloudreactor.io/))\ncan monitor and manage them.\nAvailable as a standalone executable or as a python module.\n\n## Features\n\n* Runs either processes started with a command line or a python function you\nsupply\n* Implements retries and time limits on wrapped processes\n* Injects secrets from AWS Secrets Manager and extracts them into the\nprocess environment\n* When used with the CloudReactor service:\n  * Sends heartbeats, optionally with status information like the number of\n  items processed\n  * Prevents too many concurrent executions\n  * Stops execution when manually stopped in the CloudReactor dashboard\n\n## How it works\n\nBefore your code runs, the module requests the API server to create a\nTask Execution associated with the Task name or UUID which you pass to the\nmodule.\nThe API server may reject the request if too many instances of the Task are\ncurrently running, but otherwise records that a Task Execution has started.\nThe module then passes control to your code.\n\nWhile your code is running, it may report progress to the API server,\nand the API server may signal that your Task stop execution (due to\nuser manually stopping the Task Execution), in which\ncase the module terminates your code and exits.\n\nAfter your code finishes, the module informs the API server of\nthe exit code or result. CloudReactor monitors Tasks to ensure they\nare still responsive, and keeps a history of the Executions of Tasks,\nallowing you to view failures and run durations in the past.\n\n### Auto-created Tasks\n\nBefore your Task is run (including this module),\nthe [AWS ECS CloudReactor Deployer](https://github.com/CloudReactor/aws-ecs-cloudreactor-deployer)\ncan be used to set it up in AWS ECS,\nand inform CloudReactor of details of your Task.\nThat way CloudReactor can start and schedule your Task, and setup your\nTask as a service.\nSee [CloudReactor python ECS QuickStart](https://github.com/CloudReactor/cloudreactor-python-ecs-quickstart)\nfor an example.\n\nHowever, it may not be possible or desired to change your deployment process.\nInstead, you may configure the Task to be *auto-created*.\n\nAuto-created Tasks are created the first time your Task runs.\nThis means there is no need to inform the API server of the Task details\n(during deployment) before it runs.\nInstead, each time the module runs, it informs the API server of the\nTask details at the same time as it requests the creation of a Task Execution.\nOne disadvantage of auto-created Tasks is that they are not available\nin the CloudReactor dashboard until the first time they run.\n\nWhen configuring a Task to be auto-created, you must specify the name\nor UUID of the Run Environment in CloudReactor that the Task is\nassociated with. The Run Environment must be created ahead of time,\neither by the Cloudreactor AWS Setup Wizard,\nor manually in the CloudReactor dashboard.\n\nYou can also specify more Task properties, such as Alert Methods and\nexternal links in the dashboard, by setting the environment variable\n`PROC_WRAPPER_AUTO_CREATE_TASK_PROPS` set to a JSON-encoded object that has the\n[CloudReactor Task schema](https://apidocs.cloudreactor.io/#operation/api_v1_tasks_create).\n\n### Execution Methods\n\nCloudReactor currently supports two Execution Methods:\n\n1) [AWS ECS (in Fargate)](https://aws.amazon.com/fargate/)\n2) Unknown\n\nIf a Task is running in AWS ECS, CloudReactor is able to run additional\nTask Executions, provided the details of running the Task is provided\nduring deployment with the AWS ECS CloudReactor Deployer, or if the\nTask is configured to be auto-created, and this module is run. In the\nsecond case, this module uses the ECS Metadata endpoint to detect\nthe ECS Task settings, and sends them to the API server. CloudReactor\ncan also schedule Tasks or setup long-running services using Tasks,\nprovided they are run in AWS ECS.\n\nHowever, a Task may use the Unknown execution method if it is not running\nin AWS ECS. If that is the case, CloudReactor won\'t be able to\nstart the Task in the dashboard or as part of a Workflow,\nschedule the Task, or setup a service with the Task. But the advantage is\nthat the Task code can be executed by any method available to you,\nsuch as bare metal servers, VM\'s, Docker, AWS Lambda, or Kubernetes.\nAll Tasks in CloudReactor, regardless of execution method, have their\nhistory kept and are monitored.\n\nThis module detects which of the two Execution Methods your Task is\nrunning with and sends that information to the API server, provided\nyou configure your Task to be auto-created.\n\n### Passive Tasks\n\nPassive Tasks are Tasks that CloudReactor does not manage. This means\nscheduling and service setup must be handled by other means\n(cron jobs, [supervisord](http://supervisord.org/), etc).\nHowever, Tasks marked as services or that have a schedule will still be\nmonitored by CloudReactor, which will send notifications if\na service Task goes down or a Task does not run on schedule.\n\nThe module reports to the API server that auto-created Tasks are passive,\nunless you specify the `--force-task-passive` commmand-line option or\nset the environment variable `PROC_WRAPPER_TASK_IS_PASSIVE` to `FALSE`.\nIf a Task uses the Unknown Execution Method, it must be marked as passive,\nbecause CloudReactor does not know how to manage it.\n\n## Pre-requisites\n\nIf you just want to use this module to retry processes, limit execution time,\nor fetch secrets, you can use offline mode, in which case no CloudReactor API\nkey is required. But CloudReactor offers a free tier so we hope you\n[sign up](https://dash.cloudreactor.io/signup)\nor a free account to enable monitoring and/or management.\n\nIf you want CloudReactor to be able to start your Tasks, you should use the\n[Cloudreactor AWS Setup Wizard](https://github.com/CloudReactor/cloudreactor-aws-setup-wizard)\nto configure your AWS environment to run Tasks in ECS Fargate.\nYou can skip this step if running in passive mode is OK for you.\n\nIf you want to use CloudReactor to manage or just monitor your Tasks,\nyou need to create a Run Environment and an API key in the CloudReactor\ndashboard. The API key can be scoped to the Run Environment if you\nwish. The key must have at least the Task access level, but for\nan auto-created Task, it must have at least the Developer access level.\n\n## Installation\n\n### In a Linux/AMD64 or Windows 64 environment\n\nStandalone executables for 64-bit Linux and Windows are available,\nlocated in `pyinstaller_build/platforms`. These executables bundle python\nso you don\'t need to have python installed on your machine. They also bundle\nall optional library dependencies so you can fetch secrets from AWS\nSecrets Manager and extract them with jsonpath-ng, for example.\n\nOn a debian buster machine, the following packages (with known supported versions)\nmust be installed:\n\n      openssl=1.1.1d-0+deb10u5\n      libexpat1=2.2.6-2+deb10u1\n      ca-certificates=20200601~deb10u2\n\nSee the example [Dockerfile](tests/integration/standalone_executable/docker_context_linux_amd64/Dockerfile) for a known working\nenvironment.\n\nSpecial thanks to [PyInstaller](https://www.pyinstaller.org/),\n[wine](https://www.winehq.org/), and\n[PyInstaller Docker Images](https://github.com/cdrx/docker-pyinstaller)\nfor making this possible!\n\n### When python is available\n\nInstall this module via pip (or your favourite package manager):\n\n`pip install cloudreactor-procwrapper`\n\nFetching secrets from AWS Secrets Manager requires that\n[boto3](https://boto3.amazonaws.com/v1/documentation/api/latest/index.html) is available to import in your python environment.\n\nJSON Path transformation requires that [jsonpath-ng](https://github.com/h2non/jsonpath-ng)\nbe available to import in your python environment.\n\nYou can get the tested versions of both dependencies in\n[proc_wrapper-requirements.in](https://github.com/CloudReactor/cloudreactor-procwrapper/blob/main/proc_wrapper-requirements.in)\n(suitable for use by [https://github.com/jazzband/pip-tools/](pip-tools)) or the resolved requirements in\n[proc_wrapper-requirements.txt](https://github.com/CloudReactor/cloudreactor-procwrapper/blob/main/proc_wrapper-requirements.txt).\n\n## Usage\n\nThere are two ways of using the module: wrapped mode and embedded mode.\n\n### Wrapped mode\n\nIn wrapped mode, you pass a command line to the module which it\nexecutes in a child process. The command can be implemented in whatever\nprogramming language the running machine supports.\n\nInstead of running\n\n    somecommand --somearg x\n\nyou would run\n\n    ./proc_wrapper somecommand --somearg x\n\nassuming that are using a standalone executable, and that\nyou configure the program using environment variables.\n\nOr, if you have python installed:\n\n    python -m proc_wrapper somecommand --somearg x\n\nHere are all the options:\n\n    usage: proc_wrapper [-h] [-v] [-n TASK_NAME] [--task-uuid TASK_UUID] [-a]\n                        [--auto-create-task-run-environment-name AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME]\n                        [--auto-create-task-run-environment-uuid AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID]\n                        [--auto-create-task-props AUTO_CREATE_TASK_PROPS]\n                        [--force-task-active]\n                        [--task-execution-uuid TASK_EXECUTION_UUID]\n                        [--task-version-number TASK_VERSION_NUMBER]\n                        [--task-version-text TASK_VERSION_TEXT]\n                        [--task-version-signature TASK_VERSION_SIGNATURE]\n                        [--execution-method-props EXECUTION_METHOD_PROPS]\n                        [--task-instance-metadata TASK_INSTANCE_METADATA]\n                        [--api-base-url API_BASE_URL] [-k API_KEY]\n                        [--api-heartbeat-interval API_HEARTBEAT_INTERVAL]\n                        [--api-error-timeout API_ERROR_TIMEOUT]\n                        [--api-final-update-timeout API_FINAL_UPDATE_TIMEOUT]\n                        [--api-retry-delay API_RETRY_DELAY]\n                        [--api-resume-delay API_RESUME_DELAY]\n                        [--api-task-execution-creation-error-timeout API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT]\n                        [--api-task-execution-creation-conflict-timeout API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT]\n                        [--api-task-execution-creation-conflict-retry-delay API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY]\n                        [--api-request-timeout API_REQUEST_TIMEOUT] [-o] [-s]\n                        [--max-concurrency MAX_CONCURRENCY]\n                        [--max-conflicting-age MAX_CONFLICTING_AGE] [-p]\n                        [-l LOG_LEVEL] [--log-secrets] [-w WORK_DIR]\n                        [-t PROCESS_TIMEOUT] [-r PROCESS_MAX_RETRIES]\n                        [--process-retry-delay PROCESS_RETRY_DELAY]\n                        [--process-check-interval PROCESS_CHECK_INTERVAL]\n                        [--process-termination-grace-period PROCESS_TERMINATION_GRACE_PERIOD]\n                        [--enable-status-update-listener]\n                        [--status-update-socket-port STATUS_UPDATE_SOCKET_PORT]\n                        [--status-update-message-max-bytes STATUS_UPDATE_MESSAGE_MAX_BYTES]\n                        [--status-update-interval STATUS_UPDATE_INTERVAL]\n                        [--send-pid] [--send-hostname]\n                        [--no-send-runtime-metadata] [-d DEPLOYMENT]\n                        [--schedule SCHEDULE]\n                        [--resolved-env-ttl RESOLVED_ENV_TTL]\n                        [--rollbar-access-token ROLLBAR_ACCESS_TOKEN]\n                        [--rollbar-retries ROLLBAR_RETRIES]\n                        [--rollbar-retry-delay ROLLBAR_RETRY_DELAY]\n                        [--rollbar-timeout ROLLBAR_TIMEOUT]\n                        ...\n\n    Wraps the execution of processes so that a service API endpoint (CloudReactor)\n    is optionally informed of the progress. Also implements retries, timeouts, and\n    secret injection into the environment.\n\n    positional arguments:\n      command\n\n    optional arguments:\n      -h, --help            show this help message and exit\n      -v, --version         Print the version and exit\n      -n TASK_NAME, --task-name TASK_NAME\n                            Name of Task (either the Task Name or the Task UUID\n                            must be specified\n      --task-uuid TASK_UUID\n                            UUID of Task (either the Task Name or the Task UUID\n                            must be specified)\n      -a, --auto-create-task\n                            Create the Task even if not known by the API server\n      --auto-create-task-run-environment-name AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME\n                            Name of the Run Environment to use if auto-creating\n                            the Task (either the name or UUID of the Run\n                            Environment must be specified if auto-creating the\n                            Task). Defaults to the deployment name if the Run\n                            Environment UUID is not specified.\n      --auto-create-task-run-environment-uuid AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID\n                            UUID of the Run Environment to use if auto-creating\n                            the Task (either the name or UUID of the Run\n                            Environment must be specified if auto-creating the\n                            Task)\n      --auto-create-task-props AUTO_CREATE_TASK_PROPS\n                            Additional properties of the auto-created Task, in\n                            JSON format. See https://apidocs.cloudreactor.io/#oper\n                            ation/api_v1_tasks_create for the schema.\n      --force-task-active   Indicates that the auto-created Task should be\n                            scheduled and made a service by the API server, if\n                            applicable. Otherwise, auto-created Tasks are marked\n                            passive.\n      --task-execution-uuid TASK_EXECUTION_UUID\n                            UUID of Task Execution to attach to\n      --task-version-number TASK_VERSION_NUMBER\n                            Numeric version of the Task\'s source code\n      --task-version-text TASK_VERSION_TEXT\n                            Human readable version of the Task\'s source code\n      --task-version-signature TASK_VERSION_SIGNATURE\n                            Version signature of the Task\'s source code\n      --execution-method-props EXECUTION_METHOD_PROPS\n                            Additional properties of the execution method, in JSON\n                            format\n      --task-instance-metadata TASK_INSTANCE_METADATA\n                            Additional metadata about the Task instance, in JSON\n                            format\n      --api-base-url API_BASE_URL\n                            Base URL of API server. Defaults to\n                            https://api.cloudreactor.io\n      -k API_KEY, --api-key API_KEY\n                            API key. Must have at least the Task access level, or\n                            Developer access level for auto-created Tasks.\n      --api-heartbeat-interval API_HEARTBEAT_INTERVAL\n                            Number of seconds to wait between sending heartbeats\n                            to the API server. -1 means to not send heartbeats.\n                            Defaults to 30 for concurrency limited services, 300\n                            otherwise.\n      --api-error-timeout API_ERROR_TIMEOUT\n                            Number of seconds to wait while receiving recoverable\n                            errors from the API server. Defaults to 300.\n      --api-final-update-timeout API_FINAL_UPDATE_TIMEOUT\n                            Number of seconds to wait while receiving recoverable\n                            errors from the API server when sending the final\n                            update before exiting. Defaults to 1800.\n      --api-retry-delay API_RETRY_DELAY\n                            Number of seconds to wait before retrying an API\n                            request. Defaults to 120.\n      --api-resume-delay API_RESUME_DELAY\n                            Number of seconds to wait before resuming API\n                            requests, after retries are exhausted. Defaults to\n                            600. -1 means no resumption.\n      --api-task-execution-creation-error-timeout API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT\n                            Number of seconds to keep retrying Task Execution\n                            creation while receiving error responses from the API\n                            server. -1 means to keep trying indefinitely. Defaults\n                            to 300.\n      --api-task-execution-creation-conflict-timeout API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT\n                            Number of seconds to keep retrying Task Execution\n                            creation while conflict is detected by the API server.\n                            -1 means to keep trying indefinitely. Defaults to 1800\n                            for concurrency limited services, 0 otherwise.\n      --api-task-execution-creation-conflict-retry-delay API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY\n                            Number of seconds between attempts to retry Task\n                            Execution creation after conflict is detected.\n                            Defaults to 60 for concurrency-limited services, 120\n                            otherwise.\n      --api-request-timeout API_REQUEST_TIMEOUT\n                            Timeout for contacting API server, in seconds.\n                            Defaults to 30.\n      -o, --offline-mode    Do not communicate with or rely on an API server\n      -s, --service         Indicate that this is a Task that should run\n                            indefinitely\n      --max-concurrency MAX_CONCURRENCY\n                            Maximum number of concurrent Task Executions allowed\n                            with the same Task UUID. Defaults to 1.\n      --max-conflicting-age MAX_CONFLICTING_AGE\n                            Maximum age of conflicting Tasks to consider, in\n                            seconds. -1 means no limit. Defaults to the heartbeat\n                            interval, plus 60 seconds for services that send\n                            heartbeats. Otherwise, defaults to no limit.\n      -p, --prevent-offline-execution\n                            Do not start processes if the API server is\n                            unavailable.\n      -l LOG_LEVEL, --log-level LOG_LEVEL\n                            Log level (DEBUG, INFO, WARNING, ERROR, CRITICAL).\n                            Defaults to WARNING.\n      --log-secrets         Log sensitive information\n      -w WORK_DIR, --work-dir WORK_DIR\n                            Working directory. Defaults to the current directory.\n      -t PROCESS_TIMEOUT, --process-timeout PROCESS_TIMEOUT\n                            Timeout for process, in seconds. Defaults to None for\n                            non-services, infinite for services. -1 means no\n                            timeout.\n      -r PROCESS_MAX_RETRIES, --process-max-retries PROCESS_MAX_RETRIES\n                            Maximum number of times to retry failed processes. -1\n                            means to retry forever. Defaults to 0.\n      --process-retry-delay PROCESS_RETRY_DELAY\n                            Number of seconds to wait before retrying a process.\n                            Defaults to 60.\n      --process-check-interval PROCESS_CHECK_INTERVAL\n                            Number of seconds to wait between checking the status\n                            of processes. Defaults to 10.\n      --process-termination-grace-period PROCESS_TERMINATION_GRACE_PERIOD\n                            Number of seconds to wait after sending SIGTERM to a\n                            process, but before killing it with SIGKILL. Defaults\n                            to 30.\n      --enable-status-update-listener\n                            Listen for status updates from the process, sent on\n                            the status socket port via UDP. If not specified,\n                            status update messages will not be read.\n      --status-update-socket-port STATUS_UPDATE_SOCKET_PORT\n                            The port used to receive status updates from the\n                            process. Defaults to 2373.\n      --status-update-message-max-bytes STATUS_UPDATE_MESSAGE_MAX_BYTES\n                            The maximum number of bytes status update messages can\n                            be. Defaults to 65536.\n      --status-update-interval STATUS_UPDATE_INTERVAL\n                            Minimum of seconds to wait between sending status\n                            updates to the API server. -1 means to not send status\n                            updates except with heartbeats. Defaults to -1.\n      --send-pid            Send the process ID to the API server\n      --send-hostname       Send the hostname to the API server\n      --no-send-runtime-metadata\n                            Do not send metadata about the runtime environment\n      -d DEPLOYMENT, --deployment DEPLOYMENT\n                            Deployment name (production, staging, etc.)\n      --schedule SCHEDULE   Run schedule reported to the API server\n      --resolved-env-ttl RESOLVED_ENV_TTL\n                            Number of seconds to cache resolved environment\n                            variables instead of refreshing them when a process\n                            restarts. -1 means to never refresh. Defaults to -1.\n      --rollbar-access-token ROLLBAR_ACCESS_TOKEN\n                            Access token for Rollbar (used to report error when\n                            communicating with API server)\n      --rollbar-retries ROLLBAR_RETRIES\n                            Number of retries per Rollbar request. Defaults to 2.\n      --rollbar-retry-delay ROLLBAR_RETRY_DELAY\n                            Number of seconds to wait before retrying a Rollbar\n                            request. Defaults to 120.\n      --rollbar-timeout ROLLBAR_TIMEOUT\n                            Timeout for contacting Rollbar server, in seconds.\n                            Defaults to 30.\n\nThese environment variables take precedence over command-line arguments:\n\n* PROC_WRAPPER_TASK_NAME\n* PROC_WRAPPER_TASK_UUID\n* PROC_WRAPPER_TASK_EXECUTION_UUID\n* PROC_WRAPPER_AUTO_CREATE_TASK (TRUE or FALSE)\n* PROC_WRAPPER_AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME\n* PROC_WRAPPER_AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID\n* PROC_WRAPPER_AUTO_CREATE_TASK_PROPS (JSON encoded property map)\n* PROC_WRAPPER_TASK_IS_PASSIVE (TRUE OR FALSE)\n* PROC_WRAPPER_TASK_IS_SERVICE (TRUE or FALSE)\n* PROC_WRAPPER_EXECUTION_METHOD_PROPS (JSON encoded property map)\n* PROC_WRAPPER_TASK_MAX_CONCURRENCY (can be set to -1 to indicate no limit)\n* PROC_WRAPPER_PREVENT_OFFLINE_EXECUTION (TRUE or FALSE)\n* PROC_WRAPPER_TASK_VERSION_NUMBER\n* PROC_WRAPPER_TASK_VERSION_TEXT\n* PROC_WRAPPER_TASK_VERSION_SIGNATURE\n* PROC_WRAPPER_TASK_INSTANCE_METADATA\n* PROC_WRAPPER_LOG_LEVEL\n* PROC_WRAPPER_DEPLOYMENT\n* PROC_WRAPPER_API_BASE_URL\n* PROC_WRAPPER_API_KEY\n* PROC_WRAPPER_API_HEARTBEAT_INTERVAL_SECONDS\n* PROC_WRAPPER_API_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_FINAL_UPDATE_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_REQUEST_TIMEOUT_SECONDS\n* PROC_WRAPPER_SEND_PID (TRUE or FALSE)\n* PROC_WRAPPER_SEND_HOSTNAME (TRUE or FALSE)\n* PROC_WRAPPER_SEND_RUNTIME_METADATA (TRUE or FALSE)\n* PROC_WRAPPER_ROLLBAR_ACCESS_TOKEN\n* PROC_WRAPPER_ROLLBAR_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_RETRIES\n* PROC_WRAPPER_ROLLBAR_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_MAX_CONFLICTING_AGE_SECONDS\n* PROC_WRAPPER_WORK_DIR\n* PROC_WRAPPER_PROCESS_MAX_RETRIES\n* PROC_WRAPPER_PROCESS_TIMEOUT_SECONDS\n* PROC_WRAPPER_PROCESS_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_PROCESS_CHECK_INTERVAL_SECONDS\n* PROC_WRAPPER_PROCESS_TERMINATION_GRACE_PERIOD_SECONDS\n* PROC_WRAPPER_STATUS_UPDATE_SOCKET_PORT\n* PROC_WRAPPER_STATUS_UPDATE_MESSAGE_MAX_BYTES\n\nThe command is executed with the same environment that the wrapper script gets,\nexcept that these properties are copied/overridden:\n\n* PROC_WRAPPER_DEPLOYMENT\n* PROC_WRAPPER_API_BASE_URL\n* PROC_WRAPPER_API_KEY\n* PROC_WRAPPER_API_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_API_REQUEST_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_ACCESS_TOKEN\n* PROC_WRAPPER_ROLLBAR_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_RETRIES\n* PROC_WRAPPER_ROLLBAR_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_ROLLBAR_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_TASK_EXECUTION_UUID\n* PROC_WRAPPER_TASK_UUID\n* PROC_WRAPPER_TASK_NAME\n* PROC_WRAPPER_TASK_VERSION_NUMBER\n* PROC_WRAPPER_TASK_VERSION_TEXT\n* PROC_WRAPPER_TASK_VERSION_SIGNATURE\n* PROC_WRAPPER_TASK_INSTANCE_METADATA\n* PROC_WRAPPER_SCHEDULE\n* PROC_WRAPPER_PROCESS_TIMEOUT_SECONDS\n* PROC_WRAPPER_TASK_MAX_CONCURRENCY\n* PROC_WRAPPER_PREVENT_OFFLINE_EXECUTION\n* PROC_WRAPPER_PROCESS_TERMINATION_GRACE_PERIOD_SECONDS\n* PROC_WRAPPER_ENABLE_STATUS_UPDATE_LISTENER\n* PROC_WRAPPER_STATUS_UPDATE_SOCKET_PORT\n* PROC_WRAPPER_STATUS_UPDATE_INTERVAL_SECONDS\n* PROC_WRAPPER_STATUS_UPDATE_MESSAGE_MAX_BYTES\n\nWrapped mode is suitable for running in a shell on your own (virtual) machine\nor in a Docker container. It requires multi-process support, as the module\nruns at the same time as the command it wraps.\n\n### Embedded mode\n\nEmbedded mode works for executing python code in the same process.\nYou include this package in your python project\'s\ndependencies. To run a task you want to be monitored:\n\n    from typing import Any, Dict, Mapping\n\n    import proc_wrapper\n    from proc_wrapper import ProcWrapper, ProcWrapperParams\n\n\n    def fun(wrapper: ProcWrapper, cbdata: Dict[str, int],\n            config: Mapping[str, Any]) -> int:\n        print(cbdata)\n        return cbdata[\'a\']\n\n    params = ProcWrapperParams()\n    params.auto_create_task = True\n    params.auto_create_run_environment_name = \'production\'\n    params.task_name = \'embedded_test\'\n    params.api_key = \'YOUR_CLOUDREACTOR_API_KEY\'\n    proc_wrapper = ProcWrapper(params=params)\n    x = proc_wrapper.managed_call(fun, {\'a\': 1, \'b\': 2})\n    # Should print 1\n    print(x)\n\n\nThis is suitable for running in single-threaded environments like\nAWS Lambda, or as part of a larger process that executes\nsub-routines that should be monitored.\n\nCurrently, Tasks running as Lambdas must be marked as\npassive Tasks, as the execution method is Unknown. In the near future,\nCloudReactor will support running and managing Tasks that run as\nLambdas.\n\n## Secrets Resolution\n\n### Fetching from AWS Secrets Manager\n\nBoth usage modes can fetch secrets from\n[AWS Secrets Manager](https://aws.amazon.com/secrets-manager/),\noptionally extract embedded data, then inject them into the environment\n(in the case of wrapped mode)\nor a configuration dictionary (in the case of embedded mode).\n\nTo enable secret resolution, set environment variable `PROC_WRAPPER_RESOLVE_SECRETS`\nto `TRUE`.\n\nThen to resolve the target environment variable `MY_SECRET`\nby fetching from AWS Secrets Manager, define the environment variable\n`AWS_SM_MY_SECRET_FOR_PROC_WRAPPER_TO_RESOLVE`\nset to the ARN of the secret, for example:\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:config-PPrpY\n\nThen when the wrapped process is run, it will see the environment variable\nMY_SECRET resolved to the value of the secret in Secrets Manager. Or, if\nrunning in embedded mode, the `config` dict argument will have the key\n`MY_SECRET` mapped to the value of the secret.\n\nIf the secret was stored in Secrets Manager as binary, the\ncorresponding environment variable will be set to the Base-64 encoded value.\n\nboto3 is used to fetch secrets. It will try to access to AWS Secrets Manager\nusing environment variables AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY if they\nare set, or use the EC2 instance role, ECS task role, or Lambda execution role\nif available.\n\nYou can also use "partial ARNs" (without the hyphened suffix) as keys.\nIn the example above\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:config\n\ncould be used to fetch the same secret, provided there are no conflicting secret ARNs.\n\n### Secret Tranformation\n\nFetching secrets can be relatively expensive and it makes sense to group related\nsecrets together. Therefore it is common to store JSON values as secrets.\nTo facilitate this, pieces of JSON values can be extracted to individual\nenvironment variables using [jsonpath-ng](https://github.com/h2non/jsonpath-ng).\nTo specify that a variable be extracted from a JSON value using\na JSON Path expression, append "|JP:" followed by the JSON Path expression\nin the environment variable value. For example, if the AWS Secrets Manager\nARN is\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:dbconfig-PPrpY\n\nand the value is\n\n    {\n      "username": "postgres",\n      "password": "badpassword"\n    }\n\nThen you can populate the environment variable `DB_USERNAME` by setting the\nenvironment variable\n\n    AWS_SM_DB_USERNAME_FOR_PROC_WRAPPER_TO_RESOLVE\n\nto\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:dbconfig-PPrpY|JP:$.username\n\nIf you do something similar to get the password from the same JSON value, proc_wrapper is\nsmart enough to cache the JSON value, so that the secret is only fetched once.\n\nSince JSON path expressions yield a list of results, we implement the following rules to\ntransform the list to the environment variable value:\n\n1. If the list of results has a single value, that value is used as the environment variable value,\nunless `[*]` is appended to the JSON path expression. If the value is boolean, the value\nwill be converted to either "TRUE" or "FALSE". If the value is a string or\nnumber, it will be simply left/converted to a string. Otherwise, the value is\nserialized to a JSON string and set to the environment variable value.\n2. Otherwise, the list of results is serialized to a JSON string and set to the\nenvironment variable value.\n\n### Fetching from another environment variable\n\nIn some deployment scenarios, multiple secrets can be injected into a\nsingle environment variable as a JSON encoded object. In that case,\nthe module can extract secrets using the *ENV* secret source. For example,\nyou may have arranged to have the environment variable DB_CONFIG injected\nwith the JSON encoded value:\n\n    { "username": "postgres", "password": "nohackme" }\n\nThen to extract the username to the environment variable DB_USERNAME you\nyou would add the environment variable ENV_DB_USER_FOR_PROC_WRAPPER_TO_RESOLVE\nset to\n\n    DB_CONFIG|JP:$.username\n\n### Secrets Refreshing\n\nYou can set a Time to Live (TTL) on the duration that secrets are cached,\nusing the --resolved-env-ttl command argument or PROC_WRAPPER_RESOLVED_ENV_TTL_SECONDS environment variable.\n\nIf your process exits, you have configured the script to retry, and the TTL has expired since the last fetch, proc_wrapper will re-fetch the secrets\nand resolve them again, for the environment passed to the next invocation of\nyour process.\n\n## Status Updates\n\n### Status Updates in Wrapped Mode\n\nAs your process or function runs, you can send status updates to\nCloudReactor by using the StatusUpdater class. Status updates are shown in\nthe CloudReactor dashboard and allow you to track the current progress of a\nTask and also how many items are being processed in multiple executions\nover time.\n\nIn wrapped mode, your application code would send updates to the\nproc_wrapper program via UDP port 2373 (configurable with the PROC_WRAPPER_STATUS_UPDATE_PORT environment variable).\nIf your application code is in python, you can use the provided\nStatusUpdater class to do this:\n\n    from proc_wrapper import StatusUpdater\n\n    with StatusUpdater() as updater:\n        updater.send_update(last_status_message=\'Starting ...\')\n        success_count = 0\n\n        for i in range(100):\n            try:\n                do_work()\n                success_count += 1\n                updater.send_update(success_count=success_count)\n            except Exception:\n                failed_count += 1\n                updater.send_update(failed_count=failed_count)\n\n        updater.send_update(last_status_message=\'Finished!\')\n\n### Status Updates in Embedded Mode\n\nIn embedded mode, your callback in python code can use the wrapper instance to send updates:\n\n    from typing import Any, Dict, Mapping\n\n    import proc_wrapper\n    from proc_wrapper import ProcWrapper\n\n    def fun(wrapper: ProcWrapper, cbdata: Dict[str, int],\n            config: Mapping[str, Any]) -> int:\n        wrapper.send_update(status_message=\'Starting the fun ...\')\n\n        for i in range(100):\n            try:\n                do_work()\n                success_count += 1\n            except Exception:\n                failed_count += 1\n\n            # Coalesce updates to avoid using too much bandwidth / API credits\n            if (success_count + failed_count) % 10 == 0:\n                wrapper.send_update(success_count=success_count,\n                        failed_count=failed_count)\n\n        wrapper.send_update(status_message=\'The fun is over.\')\n\n        return cbdata[\'a\']\n\n    params = ProcWrapperParams()\n    params.auto_create_task = True\n    params.auto_create_run_environment_name = \'production\'\n    params.task_name = \'embedded_test\'\n    params.api_key = \'YOUR_CLOUDREACTOR_API_KEY\'\n    proc_wrapper = ProcWrapper(params=params)\n    proc_wrapper.managed_call(fun, {\'a\': 1, \'b\': 2})\n\n## Example Project\n\nThe [cloudreactor-python-ecs-quickstart](https://github.com/CloudReactor/cloudreactor-python-ecs-quickstart)\nproject uses this library to deploy some sample tasks, written in python,\nto CloudReactor, running using AWS ECS Fargate.\n\n## License\n\nThis software is dual-licensed under open source (MPL 2.0) and commercial licenses. See `LICENSE` for details.\n\n## Contributors ✨\n\nThanks goes to these wonderful people ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="https://github.com/jtsay362"><img src="https://avatars0.githubusercontent.com/u/1079646?s=460&v=4?s=80" width="80px;" alt=""/><br /><sub><b>Jeff Tsay</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=jtsay362" title="Code">💻</a> <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=jtsay362" title="Documentation">📖</a> <a href="#infra-jtsay362" title="Infrastructure (Hosting, Build-Tools, etc)">🚇</a> <a href="#maintenance-jtsay362" title="Maintenance">🚧</a></td>\n    <td align="center"><a href="https://github.com/mwaldne"><img src="https://avatars0.githubusercontent.com/u/40419?s=460&u=3a5266861feeb27db392622371ecc57ebca09f32&v=4?s=80" width="80px;" alt=""/><br /><sub><b>Mike Waldner</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=mwaldne" title="Code">💻</a></td>\n    <td align="center"><a href="https://browniebroke.com/"><img src="https://avatars.githubusercontent.com/u/861044?v=4?s=80" width="80px;" alt=""/><br /><sub><b>Bruno Alla</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=browniebroke" title="Code">💻</a> <a href="#ideas-browniebroke" title="Ideas, Planning, & Feedback">🤔</a> <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=browniebroke" title="Documentation">📖</a></td>\n  </tr>\n</table>\n\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors) specification. Contributions of any kind welcome!\n\n## Credits\n\nThis package was created with\n[Cookiecutter](https://github.com/audreyr/cookiecutter) and the\n[browniebroke/cookiecutter-pypackage](https://github.com/browniebroke/cookiecutter-pypackage)\nproject template.\n',
    'author': 'Jeff Tsay',
    'author_email': 'jeff@cloudreactor.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://cloudreactor.io',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
