from medspacy.ner import TargetRule

concept_tag_rules = {
    "coronavirus": [
        TargetRule(
            literal="coronavirus",
            category="COVID-19",
            pattern=[{"LOWER": {"REGEX": "coronavirus|hcov|ncov$"}}],
        ),
        TargetRule(
            literal="covid",
            category="COVID-19",
            pattern=[{"LOWER": {"REGEX": "^covid"}}],
        ),
        TargetRule(literal="Novel Coronavirus (COVID-19)", category="COVID-19"),
        TargetRule(literal="novel coronavirus", category="COVID-19"),
        TargetRule(
            literal="[{'LOWER': {'REGEX': '^covid-19'}}]",
            category="COVID-19",
            pattern=[{"LOWER": {"REGEX": "^covid-19"}}],
        ),
        TargetRule(
            literal="[{'LOWER': 'sars'}, {'LOWER': '-', 'OP': '?'}, {'LOWER': 'cov-2'}]",
            category="COVID-19",
            pattern=[{"LOWER": "sars"}, {"LOWER": "-", "OP": "?"}, {"LOWER": "cov-2"}],
        ),
        TargetRule(literal="2019-cov", category="COVID-19"),
        TargetRule(literal="cov2", category="COVID-19"),
        TargetRule(literal="novel coronavirus 2019", category="COVID-19"),
        TargetRule(literal="novel corona", category="COVID-19"),
        TargetRule(literal="covid-10", category="COVID-19"),
        TargetRule(literal="corona 2019", category="COVID-19"),
        TargetRule(literal="coronavirus 19", category="COVID-19"),
        TargetRule(literal="covd-19", category="COVID-19"),
        TargetRule(literal="COVID-19", category="COVID-19"),
        TargetRule(
            literal="COVID- [\s]+19",
            category="COVID-19",
            pattern=[
                {"LOWER": "covid-"},
                {"IS_SPACE": True, "OP": "+"},
                {"LOWER": {"REGEX": "19"}},
            ],
        ),
        TargetRule(literal="covd 19", category="COVID-19"),
        TargetRule(literal="covid", category="COVID-19"),
        TargetRule(literal="SARS-CoV-2", category="COVID-19"),
        TargetRule(literal="SARS-CoVID-2", category="COVID-19"),
        TargetRule(literal="SARS CoV", category="COVID-19"),
        TargetRule(literal="SARS-CoV-19", category="COVID-19"),
        TargetRule(literal="no-cov", category="COVID-19"),
        TargetRule(
            literal="coivid",
            category="COVID-19",
            pattern=[{"LOWER": {"REGEX": "^coivid"}}],
        ),
    ],
    "positive": [
        TargetRule("+", "positive", pattern=[{"LOWER": {"REGEX": "\+$"}}]),
        TargetRule("(+)", "positive"),
        TargetRule("+ve", "positive"),
        TargetRule("+ ve", "positive"),
        TargetRule("positive", "positive"),
        TargetRule("active", "positive"),
        TargetRule("confirmed", "positive"),
        TargetRule(
            "results positive",
            "positive",
            pattern=[
                {"LOWER": "results"},
                {"LEMMA": "be", "OP": "?"},
                {"LOWER": {"IN": ["pos", "positive"]}},
            ],
        ),
    ],
    "associated_diagnosis": [
        TargetRule(
            literal="pneumonia",
            category="associated_diagnosis",
            pattern=[{"LOWER": {"IN": ["pneumonia", "pneum", "pna"]}}],
        ),
        TargetRule(literal="ards", category="associated_diagnosis"),
        TargetRule(
            literal="ards",
            category="associated_diagnosis",
            pattern=[
                {"LOWER": "ards"},
                {"LOWER": "(", "OP": "?"},
                {"LOWER": {"REGEX": "[12]/2"}},
                {"LOWER": ")", "OP": "?"},
            ],
        ),
        # Taking this for out now as it may be too imprecise
        # TargetRule(literal="infection", category="associated_diagnosis"),
        # TargetRule(literal="illness", category="associated_diagnosis"),
        TargetRule(
            literal="respiratory failure",
            category="associated_diagnosis",
            pattern=[{"LOWER": {"REGEX": "resp"}}, {"LOWER": "failure"},],
        ),
        TargetRule(
            "respiratory failure 2/2",
            "associated_diagnosis",
            pattern=[
                {"LOWER": {"IN": ["hypoxemic", "acute", "severe"]}, "OP": "+"},
                {"LOWER": {"REGEX": "resp"}},
                {"LOWER": "failure"},
                {"LOWER": "(", "OP": "?"},
                {"LOWER": {"REGEX": "[12]/2"}},
                {"LOWER": ")", "OP": "?"},
            ],
        ),
        TargetRule("hypoxia", "associated_diagnosis"),
        TargetRule("septic shock", "associated_diagnosis"),
        # TargetRule("sepsis", "associated_diagnosis"),
    ],
    "diagnosis": [
        TargetRule(
            "diagnosis",
            "diagnosis",
            pattern=[
                {"LOWER": {"IN": ["diagnosis", "dx", "dx."]}},
                {"LOWER": "of", "OP": "?"},
            ],
        ),
        TargetRule(
            "diagnosed with",
            "diagnosis",
            pattern=[
                {"LOWER": {"IN": ["diagnosed", "dx", "dx.", "dx'd"]}},
                {"LOWER": "with"},
            ],
        ),
    ],
    "screening": [
        TargetRule("screen", "screening", pattern=[{"LOWER": {"REGEX": "^screen"}}]),
    ],
    "patient": [
        TargetRule(
            "patient",
            category="patient",
            pattern=[{"LOWER": {"IN": ["patient", "pt"]}}],
        ),
        TargetRule(
            "veteran",
            category="patient",
            pattern=[{"LOWER": {"IN": ["veteran", "vet"]}}],
        ),
    ],
    # These rules are meant to capture mentions of other family members,
    # Sometimes this will be referring to a family member who tested positive
    "family": [
        TargetRule(
            "family member",
            category="family",
            pattern=[
                {
                    "POS": {"IN": ["NOUN", "PROPN", "PRON"]},
                    "LOWER": {
                        "IN": [
                            "wife",
                            "husband",
                            "spouse",
                            "family",
                            "member",
                            "girlfriend",
                            "boyfriend",
                            "mother",
                            "father",
                            "nephew",
                            "niece",
                            "grandparent",
                            "grandparents",
                            "granddaughter",
                            "grandchild",
                            "grandson",
                            "cousin",
                            "grandmother",
                            "grandfather",
                            "parent",
                            "son",
                            "daughter",
                            "mom",
                            "dad",
                            "brother",
                            "sister",
                            "aunt",
                            "uncle",
                            "child",
                            "children",
                            "sibling",
                            "siblings",
                            "relative",
                            "relatives",
                            "caregiver",
                        ]
                    },
                }
            ],
        )
    ],
    "timex": [
        TargetRule(
            "<NUM> <TIME> <AGO>",
            category="timex",
            pattern=[
                {"LIKE_NUM": True},
                {"LOWER": {"IN": ["days", "day", "weeks", "week", "month", "months"]}},
                {"LOWER": {"IN": ["ago", "prior"]}},
            ],
        ),
    ],
    # These rules are meant capture mentions of COVID-19 + individuals
    # other than the patient and family members.
    "other_experiencer": [
        TargetRule(
            "other experiencer",
            category="other_experiencer",
            pattern=[
                {
                    "POS": {"IN": ["NOUN", "PROPN", "PRON", "ADJ"]},
                    "LOWER": {
                        "IN": [
                            "someone",
                            "somebody",
                            "person",
                            "anyone",
                            "anybody",
                            "people",
                            "individual",
                            "individuals",
                            "teacher",
                            "anybody",
                            "employees",
                            "employer",
                            "customer",
                            "client",
                            "residents",
                            "resident(s",
                            "pts",
                            "patients",
                            "coworker",
                            "coworkers",
                            "workers",
                            "colleague",
                            "captain",
                            "captains",
                            "pilot",
                            "pilots",
                            "wife",
                            "husband",
                            "spouse",
                            "family",
                            "member",
                            "girlfriend",
                            "boyfriend",
                            "persons",
                            "person(s",
                            "church",
                            "convention",
                            "guest",
                            "party",
                            "attendee",
                            "conference",
                            "roommate",
                            "friend",
                            "friends",
                            "coach",
                            "player",
                            "neighbor",
                            "manager",
                            "boss",
                            "cashier",
                            "landlord",
                            "worked",
                            "works",
                            "nobody",
                            # "mate",
                            "mates",
                            "housemate",
                            "housemates",
                            "hotel",
                            "soldier",
                            "airport",
                            "tsa",
                            "lady",
                            "ladies",
                            "lobby",
                            "staffer",
                            "staffers",  # "staff",
                            "sailor",
                            "sailors",
                            "meeting",
                        ]
                    },
                }
            ],
        ),
        TargetRule(
            "the women",
            "other_experiencer",
            pattern=[{"LOWER": {"IN": ["the", "a"]}}, {"LEMMA": "woman"}],
        ),
        TargetRule(
            "the men",
            "other_experiencer",
            pattern=[{"LOWER": {"IN": ["the", "a"]}}, {"LEMMA": "man"}],
        ),
        TargetRule("in contact with", "other_experiencer"),
        TargetRule("any one", "other_experiencer"),
        TargetRule("co-worker", "other_experiencer"),
        TargetRule("at work", "other_experiencer"),
        TargetRule(
            "another patient",
            "other_experiencer",
            pattern=[{"LOWER": "another"}, {"LOWER": {"IN": ["pt", "patient", "pt."]}}],
        ),
        TargetRule(
            "a patient",
            "other_experiencer",
            pattern=[{"LOWER": "a"}, {"LOWER": {"IN": ["pt", "patient", "pt."]}}],
        ),
    ],
}
