"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MathExpression = exports.Metric = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    constructor(props) {
        var _c;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MetricProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        this.dimensions = this.validateDimensions((_c = props.dimensionsMap) !== null && _c !== void 0 ? _c : props.dimensions);
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        var _c, _d;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.with);
            }
            throw error;
        }
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.dimensionsMap === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensionsMap: (_d = (_c = props.dimensionsMap) !== null && _c !== void 0 ? _c : props.dimensions) !== null && _d !== void 0 ? _d : this.dimensions,
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * Attach the metric object to the given construct scope
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    /** @deprecated use toMetricConfig() */
    toAlarmConfig() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.Metric#toAlarmConfig", "use toMetricConfig()");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toAlarmConfig);
            }
            throw error;
        }
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        var _c, _d, _e, _f;
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.Metric#toGraphConfig", "use toMetricConfig()");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toGraphConfig);
            }
            throw error;
        }
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
                label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_e = metricConfig.renderingProperties) === null || _e === void 0 ? void 0 : _e.color),
            label: asString((_f = metricConfig.renderingProperties) === null || _f === void 0 ? void 0 : _f.label),
            unit: this.unit,
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_CreateAlarmOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.createAlarm);
            }
            throw error;
        }
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
    validateDimensions(dims) {
        if (!dims) {
            return dims;
        }
        var dimsArray = Object.keys(dims);
        if ((dimsArray === null || dimsArray === void 0 ? void 0 : dimsArray.length) > 10) {
            throw new Error(`The maximum number of dimensions is 10, received ${dimsArray.length}`);
        }
        dimsArray.map(key => {
            if (dims[key] === undefined || dims[key] === null) {
                throw new Error(`Dimension value of '${dims[key]}' is invalid`);
            }
            ;
            if (key.length < 1 || key.length > 255) {
                throw new Error(`Dimension name must be at least 1 and no more than 255 characters; received ${key}`);
            }
            ;
            if (dims[key].length < 1 || dims[key].length > 255) {
                throw new Error(`Dimension value must be at least 1 and no more than 255 characters; received ${dims[key]}`);
            }
            ;
        });
        return dims;
    }
}
exports.Metric = Metric;
_a = JSII_RTTI_SYMBOL_1;
Metric[_a] = { fqn: "@aws-cdk/aws-cloudwatch.Metric", version: "1.153.1" };
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * A math expression built with metric(s) emitted by a service
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * MathExpression can also be used for search expressions. In this case,
 * it also optionally accepts a searchRegion and searchAccount property for cross-environment
 * search expressions.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class MathExpression {
    constructor(props) {
        var _c, _d;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MathExpressionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods((_c = props.usingMetrics) !== null && _c !== void 0 ? _c : {}, this.period);
        this.label = props.label;
        this.color = props.color;
        this.searchAccount = props.searchAccount;
        this.searchRegion = props.searchRegion;
        const invalidVariableNames = Object.keys(this.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
        // Check that all IDs used in the expression are also in the `usingMetrics` map. We
        // can't throw on this anymore since we didn't use to do this validation from the start
        // and now there will be loads of people who are violating the expected contract, but
        // we can add warnings.
        const missingIdentifiers = allIdentifiersInExpression(this.expression).filter(i => !this.usingMetrics[i]);
        const warnings = [];
        if (missingIdentifiers.length > 0) {
            warnings.push(`Math expression '${this.expression}' references unknown identifiers: ${missingIdentifiers.join(', ')}. Please add them to the 'usingMetrics' map.`);
        }
        // Also copy warnings from deeper levels so graphs, alarms only have to inspect the top-level objects
        for (const m of Object.values(this.usingMetrics)) {
            warnings.push(...(_d = m.warnings) !== null && _d !== void 0 ? _d : []);
        }
        if (warnings.length > 0) {
            this.warnings = warnings;
        }
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MathExpressionOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.with);
            }
            throw error;
        }
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())
            && (props.searchAccount === undefined || props.searchAccount === this.searchAccount)
            && (props.searchRegion === undefined || props.searchRegion === this.searchRegion)) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
            searchAccount: ifUndefined(props.searchAccount, this.searchAccount),
            searchRegion: ifUndefined(props.searchRegion, this.searchRegion),
        });
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toAlarmConfig() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.MathExpression#toAlarmConfig", "use toMetricConfig()");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toAlarmConfig);
            }
            throw error;
        }
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.MathExpression#toGraphConfig", "use toMetricConfig()");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toGraphConfig);
            }
            throw error;
        }
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
                searchAccount: this.searchAccount,
                searchRegion: this.searchRegion,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_CreateAlarmOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.createAlarm);
            }
            throw error;
        }
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
_b = JSII_RTTI_SYMBOL_1;
MathExpression[_b] = { fqn: "@aws-cdk/aws-cloudwatch.MathExpression", version: "1.153.1" };
/**
 * Pattern for a variable name. Alphanum starting with lowercase.
 */
const VARIABLE_PAT = '[a-z][a-zA-Z0-9_]*';
const VALID_VARIABLE = new RegExp(`^${VARIABLE_PAT}$`);
const FIND_VARIABLE = new RegExp(VARIABLE_PAT, 'g');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
/**
 * Return all variable names used in an expression
 */
function allIdentifiersInExpression(x) {
    return Array.from(matchAll(x, FIND_VARIABLE)).map(m => m[0]);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
// Polyfill for string.matchAll(regexp)
function matchAll(x, re) {
    const ret = new Array();
    let m;
    while (m = re.exec(x)) {
        ret.push(m);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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