# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['trycast']
install_requires = \
['coverage<6.3']

setup_kwargs = {
    'name': 'trycast',
    'version': '0.4.0',
    'description': '',
    'long_description': "# trycast\n\ntrycast parses JSON-like values whose shape is defined by\n[typed dictionaries](https://www.python.org/dev/peps/pep-0589/#abstract)\n(TypedDicts) and other standard Python type hints.\n\nHere is an example of parsing a `Point2D` object defined as a `TypedDict`:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import TypedDict\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n    name: str\n\n@route('/draw_point')\ndef draw_point_endpoint() -> HTTPResponse:\n    request_json = request.json  # type: object\n    if (point := trycast(Point2D, request_json)) is not None:\n        draw_point(point)  # type is narrowed to Point2D\n        return HTTPResponse(status=200)\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n\ndef draw_point(point: Point2D) -> None:\n    ...\n```\n\nIn this example the `trycast` function is asked to parse a `request_json`\ninto a `Point2D` object, returning the original object (with its type narrowed\nappropriately) if parsing was successful.\n\nMore complex types can be parsed as well, such as the `Shape` in the following\nexample, which is a tagged union that can be either a `Circle` or `Rect` value:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import Literal, TypedDict, Union\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n\nclass Circle(TypedDict):\n    type: Literal['circle']\n    center: Point2D  # a nested TypedDict!\n    radius: float\n\nclass Rect(TypedDict):\n    type: Literal['rect']\n    x: float\n    y: float\n    width: float\n    height: float\n\nShape = Union[Circle, Rect]  # a Tagged Union!\n\n@route('/draw_shape')\ndef draw_shape_endpoint() -> HTTPResponse:\n    request_json = request.json  # type: object\n    if (shape := trycast(Shape, request_json)) is not None:\n        draw_shape(shape)  # type is narrowed to Shape\n        return HTTPResponse(status=200)\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n```\n\n> **Important:** Current limitations in the mypy typechecker require that you\n> add an extra `cast(Optional[Shape], ...)` around the call to `trycast`\n> in the example so that it is accepted by the typechecker without complaining:\n> \n> ```python\n> shape = cast(Optional[Shape], trycast(Shape, request_json))\n> if shape is not None:\n>     ...\n> ```\n> \n> These limitations are in the process of being resolved by\n> [introducing TypeForm support to mypy](https://github.com/python/mypy/issues/9773).\n\n\n## Motivation & Alternatives\n\nWhy use typed dictionaries to represent data structures instead of classes,\nnamed tuples, or other formats?\n\nTyped dictionaries are the natural form that JSON data comes in over the wire.\nThey can be trivially serialized and deserialized without any additional logic.\nFor applications that use a lot of JSON data - such as web applications - \nusing typed dictionaries is very convenient for representing data structures.\n\nOther alternatives for representing data structures in Python include\n[dataclasses], [named tuples], [attrs], and plain classes.\n\n[dataclasses]: https://www.python.org/dev/peps/pep-0557/#abstract\n[named tuples]: https://docs.python.org/3/library/typing.html#typing.NamedTuple\n[attrs]: https://www.attrs.org/en/stable/\n\n\n## Recommendations while using trycast\n\n- So that `trycast()` can recognize TypedDicts with mixed required and\n  optional keys correctly:\n    * Use Python 3.9+ if possible.\n    * Prefer using `typing.TypedDict`, unless you must use Python 3.8.\n      In Python 3.8 prefer `typing_extensions.TypedDict` instead.\n    * Avoid using `mypy_extensions.TypedDict` in general.\n\n\n# License\n\n[MIT](LICENSE.md)\n\n\n## Changelog\n\n### Future\n\n* See the [Roadmap](https://github.com/davidfstr/trycast/wiki/Roadmap).\n\n### master\n\n* (nothing new yet!)\n\n### v0.4.0b\n\n* Upgrade development status from Alpha to Beta:\n    * trycast is thoroughly tested.\n    * trycast has high code coverage (92% on Python 3.9).\n    * trycast has been in production use for over a year\n      at [at least one company] without issues\n* Add support for Python 3.10.\n* Setup continuous integration with GitHub Actions, against Python 3.6 - 3.10.\n* Migrate to the Black code style.\n* Introduce Black and isort code formatters.\n* Introduce flake8 linter.\n* Introduce coverage.py code coverage reports.\n\n[at least one company]: https://dafoster.net/projects/techsmart-platform/\n\n### v0.3.0\n\n* TypedDict improvements & fixes:\n    * Fix `trycast()` to recognize custom Mapping subclasses as TypedDicts.\n* Extend `trycast()` to recognize more JSON-like values:\n    * Extend `trycast()` to recognize `Mapping` and `MutableMapping` values.\n    * Extend `trycast()` to recognize `tuple[T, ...]` and `Tuple[T, ...]` values.\n    * Extend `trycast()` to recognize `Sequence` and `MutableSequence` values.\n* Extend `trycast()` to recognize `tuple[T1, T2, etc]` and `Tuple[T1, T2, etc]` values.\n* Documentation improvements:\n    * Improve introduction.\n    * Outline motivation to use trycast and note alternatives.\n\n### v0.2.0\n\n* TypedDict improvements & fixes:\n    * Fix `trycast()` to recognize TypedDicts from `mypy_extensions`.\n    * Extend `trycast()` to recognize TypedDicts that contain forward-references\n      to other types.\n        - Unfortunately there appears to be no easy way to support arbitrary kinds\n          of types that contain forward-references.\n        - In particular {Union, Optional} types and collection types (List, Dict)\n          with forward-references remain unsupported by `trycast()`.\n    * Recognize TypedDicts that have mixed required and optional keys correctly.\n        - Exception: Does not work for mypy_extensions.TypedDict or\n          Python 3.8's typing.TypedDict due to insufficient runtime\n          type annotation information.\n    * Fix recognition of a total=False TypedDict so that extra keys are disallowed.\n* Alter `typing_extensions` to be an optional dependency of `trycast`.\n\n### v0.1.0\n\n* Add support for Python 3.6, 3.7, and 3.9, in addition to 3.8.\n\n### v0.0.2\n\n* Fix README to appear on PyPI.\n* Add other package metadata, such as the supported Python versions.\n\n### v0.0.1a\n\n* Initial release.\n* Supports typechecking all types found in JSON.\n",
    'author': 'David Foster',
    'author_email': 'david@dafoster.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/davidfstr/trycast',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
