# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fastapi_cloudauth']

package_data = \
{'': ['*']}

install_requires = \
['fastapi>=0.60.1,<1.0',
 'python-jose[cryptography]>=3.3.0,<4.0.0',
 'requests>=2.24.0,<3.0.0']

setup_kwargs = {
    'name': 'fastapi-cloudauth',
    'version': '0.4.3',
    'description': 'fastapi-cloudauth supports simple integration between FastAPI and cloud authentication services (AWS Cognito, Auth0, Firebase Authentication).',
    'long_description': '# FastAPI Cloud Auth\n\n![Tests](https://github.com/tokusumi/fastapi-cloudauth/workflows/Tests/badge.svg)\n[![codecov](https://codecov.io/gh/tokusumi/fastapi-cloudauth/branch/master/graph/badge.svg)](https://codecov.io/gh/tokusumi/fastapi-cloudauth)\n[![PyPI version](https://badge.fury.io/py/fastapi-cloudauth.svg)](https://badge.fury.io/py/fastapi-cloudauth)\n\nfastapi-cloudauth standardizes and simplifies the integration between FastAPI and cloud authentication services (AWS Cognito, Auth0, Firebase Authentication).\n\n## Features\n\n* [X] Verify access/id token: standard JWT validation (signature, expiration), token audience claims, etc.\n* [X] Verify permissions based on scope (or groups) within access token and extract user info \n* [X] Get the detail of login user info (name, email, etc.) within ID token\n* [X] Dependency injection for verification/getting user, powered by [FastAPI](https://github.com/tiangolo/fastapi)\n* [X] Support for:\n    * [X] [AWS Cognito](https://aws.amazon.com/jp/cognito/)\n    * [X] [Auth0](https://auth0.com/jp/)\n    * [x] [Firebase Auth](https://firebase.google.com/docs/auth) (Only ID token)\n\n## Requirements\n\nPython 3.6+\n\n## Install\n\n```console\n$ pip install fastapi-cloudauth\n```\n\n## Example (AWS Cognito)\n\n### Pre-requirements\n\n* Check `region`, `userPoolID` and `AppClientID` of AWS Cognito that you manage to\n* Create a user\'s assigned `read:users` permission in AWS Cognito \n* Get Access/ID token for the created user\n\nNOTE: access token is valid for verification, scope-based authentication, and getting user info (optional). ID token is valid for verification and getting full user info from claims.\n\n### Create it\n\nCreate a *main.py* file with the following content:\n\n```python3\nimport os\nfrom pydantic import BaseModel\nfrom fastapi import FastAPI, Depends\nfrom fastapi_cloudauth.cognito import Cognito, CognitoCurrentUser, CognitoClaims\n\napp = FastAPI()\nauth = Cognito(\n    region=os.environ["REGION"], \n    userPoolId=os.environ["USERPOOLID"],\n    client_id=os.environ["APPCLIENTID"]\n)\n\n@app.get("/", dependencies=[Depends(auth.scope(["read:users"]))])\ndef secure():\n    # access token is valid\n    return "Hello"\n\n\nclass AccessUser(BaseModel):\n    sub: str\n\n\n@app.get("/access/")\ndef secure_access(current_user: AccessUser = Depends(auth.claim(AccessUser))):\n    # access token is valid and getting user info from access token\n    return f"Hello", {current_user.sub}\n\n\nget_current_user = CognitoCurrentUser(\n    region=os.environ["REGION"], \n    userPoolId=os.environ["USERPOOLID"],\n    client_id=os.environ["APPCLIENTID"]\n)\n\n\n@app.get("/user/")\ndef secure_user(current_user: CognitoClaims = Depends(get_current_user)):\n    # ID token is valid and getting user info from ID token\n    return f"Hello, {current_user.username}"\n```\n\nRun the server with:\n\n```console\n$ uvicorn main:app\n\nINFO:     Started server process [15332]\nINFO:     Waiting for application startup.\nINFO:     Application startup complete.\nINFO:     Uvicorn running on http://127.0.0.1:8000 (Press CTRL+C to quit)\n```\n\n### Interactive API Doc\n\nGo to http://127.0.0.1:8000/docs.\n\nYou will see the automatic interactive API documentation (provided by Swagger UI).\n\n`Authorize` :unlock: button can be available at the endpoint\'s injected dependency.\n\nYou can supply a token and try the endpoint interactively.\n\n![Swagger UI](https://raw.githubusercontent.com/tokusumi/fastapi-cloudauth/master/docs/src/authorize_in_doc.jpg)\n\n\n## Example (Auth0)\n\n### Pre-requirement\n\n* Check `domain`, `customAPI` (Audience) and `ClientID` of Auth0 that you manage to\n* Create a user assigned `read:users` permission in Auth0 \n* Get Access/ID token for the created user\n\n### Create it\n\nCreate a file main.py with:\n\n```python3\nimport os\nfrom pydantic import BaseModel\nfrom fastapi import FastAPI, Depends\nfrom fastapi_cloudauth.auth0 import Auth0, Auth0CurrentUser, Auth0Claims\n\napp = FastAPI()\n\nauth = Auth0(domain=os.environ["DOMAIN"], customAPI=os.environ["CUSTOMAPI"])\n\n\n@app.get("/", dependencies=[Depends(auth.scope(["read:users"]))])\ndef secure():\n    # access token is valid\n    return "Hello"\n\n\nclass AccessUser(BaseModel):\n    sub: str\n\n\n@app.get("/access/")\ndef secure_access(current_user: AccessUser = Depends(auth.claim(AccessUser))):\n    # access token is valid and getting user info from access token\n    return f"Hello", {current_user.sub}\n\n\nget_current_user = Auth0CurrentUser(\n    domain=os.environ["DOMAIN"],\n    client_id=os.environ["CLIENTID"]\n)\n\n\n@app.get("/user/")\ndef secure_user(current_user: Auth0Claims = Depends(get_current_user)):\n    # ID token is valid and getting user info from ID token\n    return f"Hello, {current_user.username}"\n```\n\nTry to run the server and see interactive UI in the same way.\n\n\n## Example (Firebase Authentication)\n\n### Pre-requirement\n\n* Create a user in Firebase Authentication and get `project ID`\n* Get ID token for the created user\n\n### Create it\n\nCreate a file main.py with:\n\n```python3\nfrom fastapi import FastAPI, Depends\nfrom fastapi_cloudauth.firebase import FirebaseCurrentUser, FirebaseClaims\n\napp = FastAPI()\n\nget_current_user = FirebaseCurrentUser(\n    project_id=os.environ["PROJECT_ID"]\n)\n\n\n@app.get("/user/")\ndef secure_user(current_user: FirebaseClaims = Depends(get_current_user)):\n    # ID token is valid and getting user info from ID token\n    return f"Hello, {current_user.user_id}"\n```\n\nTry to run the server and see the interactive UI in the same way.\n\n## Additional User Information\n\nWe can get values for the current user from access/ID token by writing a few lines.\n\n### Custom Claims\n\nFor Auth0, the ID token contains the following extra values (Ref at [Auth0 official doc](https://auth0.com/docs/tokens)):\n\n```json\n{\n  "iss": "http://YOUR_DOMAIN/",\n  "sub": "auth0|123456",\n  "aud": "YOUR_CLIENT_ID",\n  "exp": 1311281970,\n  "iat": 1311280970,\n  "name": "Jane Doe",\n  "given_name": "Jane",\n  "family_name": "Doe",\n  "gender": "female",\n  "birthdate": "0000-10-31",\n  "email": "janedoe@example.com",\n  "picture": "http://example.com/janedoe/me.jpg"\n}\n```\n\nBy default, `Auth0CurrentUser` gives `pydantic.BaseModel` object, which has `username` (name) and `email` fields.\n\nHere is sample code for extracting extra user information (adding `user_id`) from ID token:\n\n```python3\nfrom pydantic import Field\nfrom fastapi_cloudauth.auth0 import Auth0Claims  # base current user info model (inheriting `pydantic`).\n\n# extend current user info model by `pydantic`.\nclass CustomAuth0Claims(Auth0Claims):\n    user_id: str = Field(alias="sub")\n\nget_current_user = Auth0CurrentUser(domain=DOMAIN, client_id=CLIENTID)\nget_current_user.user_info = CustomAuth0Claims  # override user info model with a custom one.\n```\n\nOr, we can set new custom claims as follows:\n\n```python3\nget_user_detail = get_current_user.claim(CustomAuth0Claims)\n\n@app.get("/new/")\nasync def detail(user: CustomAuth0Claims = Depends(get_user_detail)):\n    return f"Hello, {user.user_id}"\n```\n\n### Raw payload\n\nIf you don\'t require `pydantic` data serialization (validation), `FastAPI-CloudAuth` has an option to extract the raw payload.\n\nAll you need is:\n\n```python3\nget_raw_info = get_current_user.claim(None)\n\n@app.get("/new/")\nasync def raw_detail(user = Depends(get_raw_info)):\n    # user has all items (ex. iss, sub, aud, exp, ... it depends on passed token) \n    return f"Hello, {user.get(\'sub\')}"\n```\n\n## Additional scopes\n\nAdvanced user-SCOPE verification to protect your API.\n\nSupports:\n\n- all (default): required all scopes you set\n- any: At least one of the configured scopes is required\n\nUse as (`auth` is this instanse and `app` is fastapi.FastAPI instanse):\n\n```python3\nfrom fastapi import Depends\nfrom fastapi_cloudauth import Operator\n\n@app.get("/", dependencies=[Depends(auth.scope(["allowned", "scopes"]))])\ndef api_all_scope():\n    return "user has \'allowned\' and \'scopes\' scopes"\n\n@app.get("/", dependencies=[Depends(auth.scope(["allowned", "scopes"], op=Operator._any))])\ndef api_any_scope():\n    return "user has at least one of scopes (allowned, scopes)"\n```\n\n## Development - Contributing\n\nPlease read [CONTRIBUTING](./CONTRIBUTING.md) for how to set up the development environment and testing.\n',
    'author': 'tokusumi',
    'author_email': 'tksmtoms@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tokusumi/fastapi-cloudauth',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.2,<4.0',
}


setup(**setup_kwargs)
