#!/usr/bin/env python3

import re
import sys
import logging
import operator
import numpy as np
import pandas as pd
import itertools
from collections import OrderedDict

#-------------------------------------------
# References and Points of Interest
#-------------------------------------------

# 1. Grimont PA, Weill FX. Antigenic Formulae of the Salmonella Serovars. 9th. Paris, France: WHO Collaborating Center for Reference and Research on Salmonella, Institut Pasteur; 2007.
# 2. Guibourdenche M, Roggentin P, Mikoleit M, Fields PI, Bockemühl J, Grimont PA, Weill FX. Supplement 2003-2007 (No. 47) to the White-Kauffmann-Le Minor scheme. Res Microbiol. 2010 Jan-Feb;161(1):26-9.
# 3. Issenhuth-Jeanjean S, Roggentin P, Mikoleit M, Guibourdenche M, de Pinna E, Nair S, Fields PI, Weill FX. Supplement 2008-2010 (no. 48) to the White-Kauffmann-Le Minor scheme. Res Microbiol. 2014 Sep;165(7):526-30.
# 4. Bugarel M, den Bakker HC, Nightingale KK, Brichta-Harhay DM, Edrington TS, Loneragan GH. Two Draft Genome Sequences of a New Serovar of Salmonella enterica, Serovar Lubbock. Genome Announc. 2015 Apr 16;3(2). 

# Note - phage conversion factors denoted with underlining in [1] are here denoted as optional '[]'
#        with the exception of the exclusive factors ({15} and {15,34}).

# Note - Montevideo is listed twice in [1]:  O:7 I 6,7,[14]:g,m,[p],s:[1,2,7] and O:54 I {6,7,[14]}{54}:g,m,s:-.
# The profile from the Alphabetical List p. 137 will be used here - I 6,7,[14],[54]:g,m,[p],s:[1,2,7].

# Note - As in [1], although S. bongori is not a subsp of S. enterica, 
#        symbol 'V' was retained in order to avoid confusion. 


#-------------------------------------------
# WKLM Repository and Defaults
#-------------------------------------------


###########################
#### WKLM dictionaries ####
###########################

# Dictionaries include serovars [1-4] and named subsp. enterica variants [1]
# Keys are lowercase for matching purposes
# Values are properly formatted for output
wklm_name_to_formula={'paratyphi a': 'I [1],2,12:a:[1,5]','nitra': 'I 2,12:g,m:–','kiel': 'I [1],2,12:g,p:–','koessen': 'I 2,12:l,v:1,5','kisangani': 'I [1],4,[5],12:a:1,2','hessarek': 'I 4,12,[27]:a:1,5','fulica': 'I 4,[5],12:a:[1,5]','arechavaleta': 'I 4,[5],12:a:1,7','bispebjerg': 'I [1],4,[5],12:a:e,n,x','tinda': 'I [1],4,12,27:a:e,n,z15','ii 1,4,5,12,27:a:e,n,x': 'II [1],4,[5],12,[27]:a:e,n,x','huettwilen': 'I [1],4,12:a:l,w','nakuru': 'I [1],4,12,27:a:z6','ii 1,4,12,27:a:z39': 'II [1],4,12,[27]:a:z39','paratyphi b': 'I [1],4,[5],12:b:1,2:[z5],[z33]','limete': 'I [1],4,[5],12,[27]:b:1,5','ii 4,12:b:1,5': 'II 4,12:b:1,5','canada': 'I 4,12,[27]:b:1,6','uppsala': 'I [1],4,12,27:b:1,7','abony': 'I [1],4,[5],12,[27]:b:e,n,x','ii 1,4,5,12,27:b:e,n,x': 'II [1],4,[5],12,[27]:b:[e,n,x]','wagenia': 'I [1],4,12,27:b:e,n,z15','wien': 'I [1],4,12,[27]:b:l,w','tripoli': 'I [1],4,12,27:b:z6','schleissheim': 'I 4,12,27:b:–','legon': 'I [1],4,12,[27]:c:1,5','abortusovis': 'I 4,12:c:1,6','altendorf': 'I 4,12,[27]:c:1,7','bissau': 'I 4,12:c:e,n,x','jericho': 'I [1],4,12,27:c:e,n,z15','hallfold': 'I [1],4,12,27:c:l,w','bury': 'I 4,12,27:c:z6','stanley': 'I [1],4,[5],12,[27]:d:1,2','eppendorf': 'I [1],4,12,[27]:d:1,5','brezany': 'I [1],4,12,27:d:1,6','schwarzengrund': 'I [1],4,12,27:d:1,7','ii 4,12:d:e,n,x': 'II 4,12:d:e,n,x','sarajane': 'I [1],4,[5],12,[27]:d:e,n,x','duisburg': 'I [1],4,12,[27]:d:e,n,z15:[e,h]','mons': 'I [1],4,12,27:d:l,w','ayinde': 'I [1],4,12,27:d:z6','chennai': 'I 4,12:d:z35','saintpaul': 'I [1],4,[5],12:e,h:1,2','reading': 'I [1],4,[5],12:e,h:1,5:[R1…]','eko': 'I 4,12:e,h:1,6','kaapstad': 'I 4,12:e,h:1,7','chester': 'I [1],4,[5],12:e,h:e,n,x','sandiego': 'I [1],4,[5],12:e,h:e,n,z15','chartres': 'I [1],4,12:e,h:l,w','ii 4,12:e,n,x:1,2,7': 'II 4,12:e,n,x:1,2,7','ii 1,4,12,27:e,n,x:1,5,7': 'II [1],4,12,[27]:e,n,x:1,[5],7','derby': 'I [1],4,[5],12:f,g:[1,2]','agona': 'I [1],4,[5],12:f,g,s:[1,2]:[z27],[z45]','ii 1,4,5,12:f,g,t:z6:z42': 'II [1],4,[5],12:f,g,t:z6:z42','essen': 'I 4,12:g,m:–','hato': 'I [1],4,[5],12:g,m,s:[1,2]','ii 1,4,12,27:g,m,s,t:e,n,x': 'II [1],4,12,[27]:g,[m],[s],t:e,n,x','ii 1,4,12,27:g,m,t:1,5': 'II [1],4,12,[27]:g,[m],t:[1,5]','ii 4,12:g,m,t:z39': 'II 4,12:g,m,t:z39','california': 'I 4,12:g,m,t:[z67]','kingston': 'I [1],4,[5],12,[27]:g,s,t:[1,2]:z43','budapest': 'I [1],4,12,[27]:g,t:–','travis': 'I 4,[5],12:g,z51:1,7','tennyson': 'I 4,[5],12:g,z51:e,n,z15','ii 4,12:g,z62:–': 'II 4,12:g,z62:–','banana': 'I [1],4,[5],12:m,t:[1,5]','madras': 'I 4,[5],12:m,t:e,n,z15','typhimurium': 'I [1],4,[5],12:i:1,2','lagos': 'I [1],4,[5],12:i:1,5','agama': 'I 4,12:i:1,6','farsta': 'I 4,12:i:e,n,x','tsevie': 'I [1],4,12:i:e,n,z15','gloucester': 'I [1],4,12,27:i:l,w','tumodi': 'I [1],4,12:i:z6','ii 4,12,27:i:z35': 'II 4,12,27:i:z35','massenya': 'I [1],4,12,27:k:1,5','neumuenster': 'I [1],4,12,27:k:1,6','ii 1,4,12,27:k:1,6': 'II [1],4,12,27:k:1,6','ljubljana': 'I 4,12,27:k:e,n,x','texas': 'I 4,[5],12:k:e,n,z15','fyris': 'I 4,[5],12:l,v:1,2','azteca': 'I [1],4,[5],12,[27]:l,v:1,5','clackamas': 'I 4,12:l,v:1,6','bredeney': 'I [1],4,12,27:l,v:1,7:[z40]','kimuenza': 'I [1],4,12,27:l,v:e,n,x','ii 1,4,12,27:l,v:e,n,x': 'II [1],4,12,27:l,v:e,n,x','brandenburg': 'I 4,[5],12:l,v:e,n,z15','ii 1,4,12,27:l,v:z39': 'II [1],4,12,27:l,v:z39','mono': 'I 4,12:l,w:1,5','togo': 'I 4,12:l,w:1,6','ii 4,12:l,w:e,n,x': 'II 4,12:l,w:e,n,x','blancmesnil': 'I 4,12:l,w:e,n,z15','ayton': 'I [1],4,12,27:l,w:z6','kunduchi': 'I [1],4,[5],12,[27]:l,[z13],[z28]:1,2','tyresoe': 'I [1],4,12,[27]:l,[z13],z28:1,5','haduna': 'I 4,12:l,z13,[z28]:1,6','kubacha': 'I [1],4,12,27:l,z13,z28:1,7','kano': 'I [1],4,12,27:l,z13,z28:e,n,x','vom': 'I [1],4,12,27:l,z13,z28:e,n,z15','reinickendorf': 'I 4,12:l,z28:e,n,x','ii 1,4,12:l,z28:e,n,x': 'II [1],4,12:l,z28:[e,n,x]','heidelberg': 'I [1],4,[5],12:r:1,2','bradford': 'I 4,12,[27]:r:1,5','winneba': 'I 4,12:r:1,6','remo': 'I [1],4,12,27:r:1,7','bochum': 'I [1],4,[5],12:r:l,w','southampton': 'I 4,12,27:r:z6','drogana': 'I [1],4,12,27:r,[i]:e,n,z15','africana': 'I 4,12:r,i:l,w','coeln': 'I [1],4,[5],12:y:1,2','trachau': 'I 4,12,27:y:1,5','finaghy': 'I 4,12:y:1,6','teddington': 'I [1],4,12,27:y:1,7','ball': 'I [1],4,[5],12,[27]:y:e,n,x','jos': 'I [1],4,12,27:y:e,n,z15','kamoru': 'I [1],4,12,27:y:z6','shubra': 'I 4,[5],12:z:1,2','kiambu': 'I [1],4,12:z:1,5','ii 1,4,12,27:z:1,5': 'II [1],4,12,27:z:1,5','loubomo': 'I 4,12:z:1,6','indiana': 'I [1],4,12:z:1,7','ii 4,12:z:1,7': 'II 4,12:z:1,7','neftenbach': 'I 4,12:z:e,n,x','ii 1,4,12,27:z:e,n,x': 'II [1],4,12,27:z:e,n,x','koenigstuhl': 'I [1],4,[5],12:z:e,n,z15','preston': 'I [1],4,12:z:l,w','entebbe': 'I [1],4,12,27:z:z6','ii 4,12:z:z39': 'II 4,12:z:z39','stanleyville': 'I [1],4,[5],12,[27]:z4,z23:[1,2]','perrosguirec': 'I 4,12:z4,z23:e,n,z15','vuadens': 'I 4,12,27:z4,z23:z6','kalamu': 'I [1],4,[5],12:z4,z24:[1,5]','haifa': 'I [1],4,[5],12:z10:1,2','ituri': 'I [1],4,12:z10:1,5','tudu': 'I 4,12:z10:1,6','harrow': 'I 4,12:z10:1,7','albert': 'I 4,12:z10:e,n,x','tokoin': 'I 4,12:z10:e,n,z15','mura': 'I [1],4,12:z10:l,w','fortune': 'I [1],4,12,[27]:z10:z6','vellore': 'I [1],4,12,27:z10:z35','brancaster': 'I [1],4,12,27:z29:–','ii 1,4,12:z29:e,n,x': 'II [1],4,12:z29:e,n,x','pasing': 'I 4,12:z35:1,5','tafo': 'I [1],4,12,27:z35:1,7','sloterdijk': 'I [1],4,12,27:z35:z6','yaounde': 'I [1],4,12,27:z35:e,n,z15','tejas': 'I 4,12:z36:–','yverdon': 'I [1],4,12:z38:1,5','wilhelmsburg': 'I [1],4,[5],12,[27]:z38:[e,n,z15]','ii 1,4,12,27:z39:1,5,7': 'II [1],4,12,[27]:z39:1,[5],7','thayngen': 'I [1],4,12,27:z41:1,(2),5','maska': 'I [1],4,12,27:z41:e,n,z15','abortusequi': 'I 4,12:–:e,n,x','mygdal': 'I 4,12:z91:–','sanjuan': 'I 6,7:a:1,5','ii 6,7,14:a:1,5': 'II 6,7,[14]:a:1,5','umhlali': 'I 6,7:a:1,6','austin': 'I 6,7:a:1,7','oslo': 'I 6,7,[14]:a:e,n,x','denver': 'I 6,7:a:e,n,z15','coleypark': 'I 6,7,[14]:a:l,w','damman': 'I 6,7,[14]:a:z6','ii 6,7:a:z6': 'II 6,7:a:z6','ii 6,7:a:z42': 'II 6,7:a:z42','brazzaville': 'I 6,7:b:1,2','ii 6,7:b:1,5': 'II 6,7:b:1,5','edinburg': 'I 6,7,[14]:b:1,5','adime': 'I 6,7:b:1,6','koumra': 'I 6,7:b:1,7','lockleaze': 'I 6,7,[14]:b:e,n,x','georgia': 'I 6,7:b:e,n,z15','ii 6,7:b:e,n,x:z42': 'II 6,7:b:[e,n,x]:z42','ohio': 'I 6,7,[14]:b:l,w:[z59]','leopoldville': 'I 6,7,[14]:b:z6','kotte': 'I 6,7:b:z35','ii 6,7:b:z39': 'II 6,7:b:z39','hissar': 'I 6,7,[14]:c:1,2','paratyphi c': 'I 6,7,[Vi]:c:1,5','choleraesuis': 'I 6,7:c:1,5','typhisuis': 'I 6,7:c:1,5','birkenhead': 'I 6,7:c:1,6','schwabach': 'I 6,7:c:1,7','cotonou': 'I 6,7:c:z6','namibia': 'I 6,7:c:e,n,x','kaduna': 'I 6,7,[14]:c:e,n,z15','kisii': 'I 6,7:d:1,2','isangi': 'I 6,7,[14]:d:1,5','kivu': 'I 6,7:d:1,6','kambole': 'I 6,7:d:1,[2],7','amersfoort': 'I 6,7,[14]:d:e,n,x','gombe': 'I 6,7,[14]:d:e,n,z15','livingstone': 'I 6,7,[14]:d:l,w','wil': 'I 6,7:d:l,z13,z28','nieukerk': 'I 6,7,[14]:d:z6','ii 6,7:d:z42': 'II 6,7:d:z42','larochelle': 'I 6,7:e,h:1,2','lomita': 'I 6,7:e,h:1,5','norwich': 'I 6,7:e,h:1,6','nola': 'I 6,7:e,h:1,7','braenderup': 'I 6,7,[14]:e,h:e,n,z15','ii 6,7:e,n,x:1,6:z42': 'II 6,7:e,n,x:1,6:z42','kastrup': 'I 6,7:e,n,z15:1,6','rissen': 'I 6,7,[14]:f,g:–','eingedi': 'I 6,7:f,g,t:1,2,7','afula': 'I 6,7:f,g,t:e,n,x','montevideo': 'I 6,7,[14],[54]:g,m,[p],s:[1,2,7]','lubbock': 'I 6,7:g,m,s:e,n,z15','ii 6,7:g,m,s,t:e,n,x': 'II 6,7:g,m,[s],t:e,n,x','ii 6,7:g,m,s,t:1,5': 'II 6,7:(g),m,[s],t:1,5','ii 6,7:g,m,s,t:z39': 'II 6,7:g,m,s,t:z39','ii 6,7:g,m,s,t:z42': 'II 6,7:g,[m],s,t:[z42]','othmarschen': 'I 6,7,[14]:g,m,[t]:–','plumaugat': 'I 6,7:g,s,q:–','menston': 'I 6,7:g,s,[t]:[1,6]','ii 6,7:g,t:e,n,x:z42': 'II 6,7:g,t:[e,n,x]:z42','riggil': 'I 6,7:g,(t):–','alamo': 'I 6,7:g,z51:1,5','larose': 'I 6,7:g,z51:e,n,z15','iv 6,7:g,z51:–': 'IV 6,7:g,z51:–','haelsingborg': 'I 6,7:m,p,t,[u]:–','winston': 'I 6,7:m,t:1,6','oakey': 'I 6,7:m,t:z64','ii 6,7:m,t:–': 'II 6,7:m,t:–','oranienburg': 'I 6,7,[14]:m,t:[z57]','augustenborg': 'I 6,7,[14]:i:1,2','oritamerin': 'I 6,7:i:1,5','garoli': 'I 6,7:i:1,6','lika': 'I 6,7:i:1,7','athinai': 'I 6,7:i:e,n,z15','norton': 'I 6,7:i:l,w','stuttgart': 'I 6,7,[14]:i:z6','galiema': 'I 6,7,[14]:k:1,2','thompson': 'I 6,7,[14]:k:1,5:[R1…]','daytona': 'I 6,7:k:1,6','baiboukoum': 'I 6,7:k:1,7','singapore': 'I 6,7:k:e,n,x','escanaba': 'I 6,7:k:e,n,z15','iiib 6,7:k:z:z55': 'IIIb 6,7:(k):z:[z55]','ii 6,7:k:z6': 'II 6,7:k:[z6]','ashford': 'I 6,7:k:z35','concord': 'I 6,7:l,v:1,2','irumu': 'I 6,7:l,v:1,5','iiib 6,7:l,v:1,5,7': 'IIIb 6,7:l,v:1,5,7','mkamba': 'I 6,7:l,v:1,6','kortrijk': 'I 6,7:l,v:1,7','bonn': 'I 6,7:l,v:e,n,x','potsdam': 'I 6,7,[14]:l,v:e,n,z15','gdansk': 'I 6,7,[14]:l,v:z6','coromandel': 'I 6,7:l,v:z35','iiib 6,7:l,v:z53': 'IIIb 6,7:l,v:z53','gabon': 'I 6,7:l,w:1,2','colorado': 'I 6,7:l,w:1,5','ii 6,7:l,w:1,5,7': 'II 6,7:l,w:1,5,7','langeveld': 'I 6,7:l,w:e,n,z15','ii 6,7:l,w:z42': 'II 6,7:l,w:z42','nessziona': 'I 6,7:l,z13:1,5','kenya': 'I 6,7:l,z13:e,n,x','neukoelln': 'I 6,7:l,z13,[z28]:e,n,z15','makiso': 'I 6,7:l,z13,z28:z6','strathcona': 'I 6,7:l,z13,z28:1,7','ii 6,7:l,z28:1,5:z42': 'II 6,7:l,z28:1,5:[z42]','ii 6,7:l,z28:e,n,x': 'II 6,7:l,z28:e,n,x','ii 6,7:l,z28:z6': 'II 6,7:l,z28:z6','virchow': 'I 6,7,[14]:r:1,2','infantis': 'I 6,7,[14]:r:1,5:[R1…],[z37],[z45],[z49]','nigeria': 'I 6,7:r:1,6','colindale': 'I 6,7:r:1,7','papuana': 'I 6,7:r:e,n,z15','grampian': 'I 6,7:r:l,w','richmond': 'I 6,7:y:1,2','bareilly': 'I 6,7,[14]:y:1,5','oyonnax': 'I 6,7:y:1,6','gatow': 'I 6,7:y:1,7','hartford': 'I 6,7:y:e,n,x:[z67]','mikawasima': 'I 6,7,[14]:y:e,n,z15:[z47],[z50]','chile': 'I 6,7:z:1,2','poitiers': 'I 6,7:z:1,5','ii 6,7:z:1,5': 'II 6,7:z:1,5','oakland': 'I 6,7:z:1,6,[7]','cayar': 'I 6,7:z:e,n,x','ii 6,7:z:e,n,x': 'II 6,7:z:e,n,x','ii 6,7:z:e,n,x,z15': 'II 6,7:z:e,n,x,z15','businga': 'I 6,7:z:e,n,z15','bruck': 'I 6,7:z:l,w','ii 6,7:z:z6': 'II 6,7:z:z6','ii 6,7:z:z39': 'II 6,7:z:z39','ii 6,7:z:z42': 'II 6,7:z:z42','obogu': 'I 6,7:z4,z23:1,5','planckendael': 'I 6,7:z4,z23:1,6','aequatoria': 'I 6,7:z4,z23:e,n,z15','goma': 'I 6,7:z4,z23:z6','ii 6,7:z4,z23:–': 'II 6,7:z4,z23:–','iv 6,7:z4,z23:–': 'IV 6,7:z4,z23:–','zollikerberg': 'I 6,7:z4,z24:1,5','ii 6,7:z4,z24:z42': 'II 6,7:z4,z24:z42','somone': 'I 6,7:z4,z24:–','iv 6,7:z4,z24:–': 'IV 6,7:z4,z24:–','ii 6,7:z6:1,7': 'II 6,7:z6:1,7','menden': 'I 6,7:z10:1,2','inganda': 'I 6,7:z10:1,5','eschweiler': 'I 6,7:z10:1,6','ngili': 'I 6,7:z10:1,7','djugu': 'I 6,7:z10:e,n,x','mbandaka': 'I 6,7,[14]:z10:e,n,z15:[z37],[z45]','jerusalem': 'I 6,7,[14]:z10:l,w','redba': 'I 6,7:z10:z6','omuna': 'I 6,7:z10:z35','tennessee': 'I 6,7,[14]:z29:[1,2,7]','ii 6,7:z29:z42': 'II 6,7:z29:[z42]','tienba': 'I 6,7:z35:1,6','palime': 'I 6,7:z35:e,n,z15','tampico': 'I 6,7:z36:e,n,z15','ii 6,7:z36:z42': 'II 6,7:z36:z42','iv 6,7:z36:–': 'IV 6,7:z36:–','rumford': 'I 6,7:z38:1,2:[z82]','ii 6,7:z38:e,n,x': 'II 6,7:z38:e,n,x','lille': 'I 6,7,[14]:z38:–:[z82]','iiib 6,7,14:z39:1,2': 'IIIb 6,7,[14]:z39:1,2','ii 6,7:z39:1,5,7': 'II 6,7:z39:1,5,7','vi 6,7:z41:1,7': 'VI 6,7:z41:1,7','hillsborough': 'I 6,7:z41:l,w','tamilnadu': 'I 6,7:z41:z35','ii 6,7:z42:1,5,7': 'II 6,7:z42:1,[5],7','bulovka': 'I 6,7:z44:–','ii 6,7:–:1,6': 'II 6,7:–:1,6','be': 'I 8,[20]:a:[z6]','valdosta': 'I 6,8:a:1,2','doncaster': 'I 6,8:a:1,5','curacao': 'I 6,8:a:1,6','nordufer': 'I 6,8:a:1,7','narashino': 'I 6,8:a:e,n,x','ii 6,8:a:e,n,x': 'II 6,8:a:e,n,x','leith': 'I 6,8:a:e,n,z15','ii 6,8:a:z39': 'II 6,8:a:z39','ii 6,8:a:z52': 'II 6,8:a:z52','djelfa': 'I 8:b:1,2','skansen': 'I 6,8:b:1,2','korbol': 'I 8,[20]:b:1,5','nagoya': 'I 6,8:b:1,5','ii 6,8:b:1,5': 'II 6,8:b:1,5','stourbridge': 'I 6,8:b:1,6','sanga': 'I 8:b:1,7','eboko': 'I 6,8:b:1,7','konstanz': 'I 8:b:e,n,x','gatuni': 'I 6,8:b:e,n,x','shipley': 'I 8,[20]:b:e,n,z15','presov': 'I 6,8:b:e,n,z15','bukuru': 'I 6,8:b:l,w','heistopdenberg': 'I 8,[20]:b:l,w','tounouma': 'I 8,[20]:b:z6','banalia': 'I 6,8:b:z6','wingrove': 'I 6,8:c:1,2','gaillac': 'I 8,[20]:c:1,5','utah': 'I 6,8:c:1,5','bronx': 'I 6,8:c:1,6','belfast': 'I 6,8:c:1,7','alexanderpolder': 'I 8:c:l,w','santiago': 'I 8,[20]:c:e,n,x','belem': 'I 6,8:c:e,n,x','quiniela': 'I 6,8:c:e,n,z15','tado': 'I 8,[20]:c:z6','virginia': 'I 8:d:1,2','muenchen': 'I 6,8:d:1,2:[z67]','yovokome': 'I 8,[20]:d:1,5','manhattan': 'I 6,8:d:1,5:[z58]','portanigra': 'I 8,[20]:d:1,7','dunkwa': 'I 6,8:d:1,7','sterrenbos': 'I 6,8:d:e,n,x','herston': 'I 6,8:d:e,n,z15','labadi': 'I 8,[20]:d:z6','ii 6,8:d:z6:z42': 'II 6,8:d:z6:z42','bardo': 'I 8:e,h:1,2','newport': 'I 6,8,[20]:e,h:1,2:[z67],[z78]','ferruch': 'I 8:e,h:1,5','kottbus': 'I 6,8:e,h:1,5','cremieu': 'I 6,8:e,h:1,6:[R1…]','atakpame': 'I 8,[20]:e,h:1,7','fillmore': 'I 6,8:e,h:e,n,x','tshiongwe': 'I 6,8:e,h:e,n,z15','rechovot': 'I 8,[20]:e,h:z6','sandow': 'I 6,8:f,g:e,n,z15','ii 6,8:f,g,m,t:e,n,x': 'II 6,8:f,g,m,t:[e,n,x]','emek': 'I 8,[20]:g,m,s:–','chincol': 'I 6,8:g,m,[s]:[e,n,x]','ii 6,8:g,m,t:1,7': 'II 6,8:g,m,t:1,7','reubeuss': 'I 8,[20]:g,m,t:–','alminko': 'I 8,[20]:g,s,t:–','nanergou': 'I 6,8:g,s,t:–','yokoe': 'I 8,[20]:m,t:–','ii 6,8:m,t:1,5': 'II 6,8:m,t:1,5','ii 6,8:m,t:e,n,x': 'II 6,8:m,t:e,n,x','bassa': 'I 6,8:m,t:–','lindenburg': 'I 6,8:i:1,2','bargny': 'I 8,[20]:i:1,5','takoradi': 'I 6,8:i:1,5','warnow': 'I 6,8:i:1,6','malmoe': 'I 6,8:i:1,7','bonariensis': 'I 6,8:i:e,n,x','aba': 'I 6,8:i:e,n,z15','magherafelt': 'I 8,[20]:i:l,w','cyprus': 'I 6,8:i:l,w','kentucky': 'I 8,[20]:i:z6','kallo': 'I 6,8:k:1,2','haardt': 'I 8:k:1,5','blockley': 'I 6,8:k:1,5:[z58]','sandbanks': 'I 6,8:k:1,7','schwerin': 'I 6,8:k:e,n,x','charlottenburg': 'I 6,8:k:e,n,z15','iiib 8:k:z35': 'IIIb 8:(k):z35','pakistan': 'I 8:l,v:1,2','litchfield': 'I 6,8:l,v:1,2','loanda': 'I 6,8:l,v:1,5','amherstiana': 'I 8:l,v:1,6','manchester': 'I 6,8:l,v:1,7','holcomb': 'I 6,8:l,v:e,n,x','ii 6,8:l,v:e,n,x': 'II 6,8:l,v:e,n,x','edmonton': 'I 6,8:l,v:e,n,z15','lund': 'I 6,8:l,v:z6','fayed': 'I 6,8:l,w:1,2','ii 6,8:l,w:z6:z42': 'II 6,8:l,w:z6:z42','hiduddify': 'I 6,8:l,z13,z28:1,5','breukelen': 'I 6,8:l,z13,[z28]:e,n,z15','ii 6,8:l,z28:e,n,x': 'II 6,8:l,z28:e,n,x','bsilla': 'I 6,8:r:1,2','hindmarsh': 'I 8,[20]:r:1,5','bovismorbificans': 'I 6,8,[20]:r,[i]:1,5:[R1…]','colchester': 'I 8,[20]:r:1,6','noya': 'I 8:r:1,7','akanji': 'I 6,8:r:1,7','cocody': 'I 8,[20]:r,i:e,n,z15','hidalgo': 'I 6,8:r,[i]:e,n,z15','brikama': 'I 8,[20]:r,[i]:l,w','goldcoast': 'I 6,8:r:l,w','iiib 8:r:z': 'IIIb 8:r:z','altona': 'I 8,[20]:r,[i]:z6','giza': 'I 8,[20]:y:1,2','lamphun': 'I 6,8:y:1,2','brunei': 'I 8,[20]:y:1,5','tananarive': 'I 6,8:y:1,5','bulgaria': 'I 6,8:y:1,6','ii 6,8:y:1,6:z42': 'II 6,8:y:1,6:z42','alagbon': 'I 8,[20]:y:1,7','inchpark': 'I 6,8:y:1,7','sunnycove': 'I 8:y:e,n,x','daarle': 'I 6,8:y:e,n,x','praha': 'I 6,8:y:e,n,z15','kralingen': 'I 8,[20]:y:z6','benue': 'I 6,8:y:l,w','sindelfingen': 'I 8,[20]:y:l,w','mowanjum': 'I 6,8:z:1,5','ii 6,8:z:1,5': 'II 6,8:z:1,5','marmande': 'I 6,8:z:1,7','phaliron': 'I 8:z:e,n,z15','kalumburu': 'I 6,8:z:e,n,z15','kuru': 'I 6,8:z:l,w','daula': 'I 8,[20]:z:z6','bellevue': 'I 8:z4,z23:1,7','lezennes': 'I 6,8:z4,z23:1,7','breda': 'I 6,8:z4,z23:e,n,x','chailey': 'I 6,8:z4,z23:[e,n,z15]','dabou': 'I 8,[20]:z4,z23:l,w','corvallis': 'I 8,[20]:z4,z23:[z6]','nuorikkala': 'I 8:z4,z24:e,n,x','albany': 'I 8,[20]:z4,z24:–:[z45]','duesseldorf': 'I 6,8:z4,z24:–','tallahassee': 'I 6,8:z4,z32:–','bazenheid': 'I 8,[20]:z10:1,2','zerifin': 'I 6,8:z10:1,2','paris': 'I 8,[20]:z10:1,5','mapo': 'I 6,8:z10:1,5','cleveland': 'I 6,8:z10:1,7','istanbul': 'I 8:z10:e,n,x','hadar': 'I 6,8:z10:e,n,x','chomedey': 'I 8,[20]:z10:e,n,z15','glostrup': 'I 6,8:z10:e,n,z15','remiremont': 'I 8,[20]:z10:l,w','molade': 'I 8,[20]:z10:z6','wippra': 'I 6,8:z10:z6','ii 6,8:z29:1,5': 'II 6,8:z29:1,5','ii 6,8:z29:e,n,x:z42': 'II 6,8:z29:e,n,x:z42','tamale': 'I 8,[20]:z29:[e,n,z15]','uno': 'I 6,8:z29:[e,n,z15]','ii 6,8:z29:e,n,x': 'II 6,8:z29:e,n,x','kolda': 'I 8,[20]:z35:1,2','yarm': 'I 6,8:z35:1,2','angers': 'I 8,[20]:z35:z6','apeyeme': 'I 8,[20]:z38:–','diogoye': 'I 8,[20]:z41:z6','aesch': 'I 6,8:z60:1,2','sendai': 'I [1],9,12:a:1,5','miami': 'I [1],9,12:a:1,5','ii 9,12:a:1,5': 'II 9,12:a:1,5','os': 'I 9,12:a:1,6','saarbruecken': 'I [1],9,12:a:1,7','lomalinda': 'I [1],9,12:a:e,n,x','ii 1,9,12:a:e,n,x': 'II [1],9,12:a:e,n,x','durban': 'I [1],9,12:a:e,n,z15','ii 9,12:a:z39': 'II 9,12:a:z39','ii 1,9,12:a:z42': 'II [1],9,12:a:z42','onarimon': 'I [1],9,12:b:1,2','frintrop': 'I [1],9,12:b:1,5','bata': 'I 9,12:b:1,7','ii 1,9,12:b:e,n,x': 'II [1],9,12:b:e,n,x','mana': 'I 9,12:b:e,n,z15','ii 1,9,12:b:z6': 'II [1],9,12:b:z6','ii 1,9,12:b:z39': 'II [1],9,12:b:z39','goeteborg': 'I 9,12:c:1,5','ipeko': 'I 9,12:c:1,6','elokate': 'I 9,12:c:1,7','alabama': 'I 9,12:c:e,n,z15','ridge': 'I 9,12:c:z6','ndolo': 'I [1],9,12:d:1,5','tarshyne': 'I 9,12:d:1,6','eschberg': 'I 9,12:d:1,7','ii 1,9,12:d:e,n,x': 'II [1],9,12:d:e,n,x','viikki': 'I [1],9,12:d:e,n,x','bangui': 'I 9,12:d:e,n,z15','zega': 'I 9,12:d:z6','jaffna': 'I [1],9,12:d:z35','ii 9,12:d:z39': 'II 9,12:d:z39','typhi': 'I 9,12,[Vi]:d:–:[z66]','bournemouth': 'I 9,12:e,h:1,2','eastbourne': 'I [1],9,12:e,h:1,5','westafrica': 'I 9,12:e,h:1,7','israel': 'I 9,12:e,h:e,n,z15','ii 9,12:e,n,x:1,5,7': 'II 9,12:e,n,x:1,[5],7','ii 9,12:e,n,x:1,6': 'II 9,12:e,n,x:1,6','berta': 'I [1],9,12:[f],g,[t]:–','enteritidis': 'I [1],9,12:g,m:–','gueuletapee': 'I 9,12:g,m,s:–','blegdam': 'I 9,12:g,m,q:–','ii 1,9,12:g,m,s,t:1,5,7:z42': 'II [1],9,12:g,m,[s],t:[1,5,7]:[z42]','ii 1,9,12:g,m,s,t:e,n,x': 'II [1],9,12:g,m,s,t:e,n,x','ii 9,12:g,m,t:z39': 'II 9,12:g,m,t:z39','dublin': 'I [1],9,12,[Vi]:g,p:–','naestved': 'I [1],9,12:g,p,s:–','rostock': 'I [1],9,12:g,p,u:–','moscow': 'I [1],9,12:g,q:–','ii 9,12:g,s,t:e,n,x': 'II 9,12:g,s,t:e,n,x','newmexico': 'I 9,12:g,z51:1,5','ii 1,9,12:g,z62:e,n,x': 'II [1],9,12:g,z62:[e,n,x]','antarctica': 'I 9,12:g,z63:–','rosenberg': 'I 9,12:g,z85:–','ii 9,12:m,t:e,n,x': 'II 9,12:m,t:e,n,x','pensacola': 'I [1],9,12:m,t:[1,2]','ii 1,9,12:m,t:1,5': 'II [1],9,12:m,t:1,5','ii 1,9,12:m,t:z39': 'II [1],9,12:m,t:z39','seremban': 'I 9,12:i:1,5','claibornei': 'I [1],9,12:k:1,5','goverdhan': 'I 9,12:k:1,6','mendoza': 'I 9,12:l,v:1,2','panama': 'I [1],9,12:l,v:1,5:[R1…]','houston': 'I 9,12:l,v:1,5:d','kapemba': 'I 9,12:l,v:1,7:[z40]','zaiman': 'I 9,12:l,v:e,n,x','ii 9,12:l,v:e,n,x': 'II 9,12:l,v:e,n,x','goettingen': 'I 9,12:l,v:e,n,z15','ii 9,12:l,v:z39': 'II 9,12:l,v:z39','victoria': 'I [1],9,12:l,w:1,5','ii 1,9,12:l,w:e,n,x': 'II [1],9,12:l,w:e,n,x','itami': 'I 9,12:l,z13:1,5','miyazaki': 'I 9,12:l,z13:1,7','napoli': 'I [1],9,12:l,z13:e,n,x','javiana': 'I [1],9,12:l,z28:1,5:[R1…]','kotu': 'I 9,12:l,z28:1,6','ii 9,12:l,z28:1,5:z42': 'II 9,12:l,z28:1,5:z42','ii 9,12:l,z28:e,n,x': 'II 9,12:l,z28:e,n,x','york': 'I 9,12:l,z28:e,n,z15','jamaica': 'I 9,12:r:1,5','camberwell': 'I 9,12:r:1,7','yellowknife': 'I 9,12:r:e,n,x','campinense': 'I 9,12:r:e,n,z15','lome': 'I 9,12:r:z6','powell': 'I 9,12:y:1,7','ii 1,9,12:y:z39': 'II [1],9,12:y:z39','mulhouse': 'I [1],9,12:z:1,2','lawndale': 'I [1],9,12:z:1,5','kimpese': 'I 9,12:z:1,6','ii 1,9,12:z:1,7': 'II [1],9,12:z:1,7','aurelianis': 'I 9,12:z:e,n,z15','ii 1,9,12:z:z6': 'II [1],9,12:z:z6','ii 9,12:z:z39': 'II 9,12:z:z39','wangata': 'I [1],9,12:z4,z23:[1,7]','natal': 'I 9,12:z4,z24:–','franken': 'I 9,12:z6:z67:[R1…]','portland': 'I 9,12:z10:1,5','treguier': 'I 9,12:z10:z6','ruanda': 'I 9,12:z10:e,n,z15','ii 9,12:z29:1,5': 'II 9,12:z29:1,5','ii 1,9,12:z29:e,n,x': 'II [1],9,12:z29:e,n,x','penarth': 'I 9,12:z35:z6','elomrane': 'I [1],9,12:z38:–','ii 1,9,12:z39:1,7': 'II [1],9,12:z39:1,7','ottawa': 'I [1],9,12:z41:1,5','ii 1,9,12:z42:1,5,7': 'II [1],9,12:z42:1,[5],7','gallinarum': 'I [1],9,12:–:–','detmold': 'I 9,46:a:1,2','poole': 'I 9,46:a:1,6','baildon': 'I 9,46:a:e,n,x','doba': 'I 9,46:a:e,n,z15','montaigu': 'I 9,46:b:1,2','cheltenham': 'I 9,46:b:1,5','zadar': 'I 9,46:b:1,6','worb': 'I 9,46:b:e,n,x','ii 9,46:b:e,n,x': 'II 9,46:b:e,n,x','bamboye': 'I 9,46:b:l,w','linguere': 'I 9,46:b:z6','kolar': 'I 9,46:b:z35','argenteuil': 'I [1],9,46:c:1,7','itutaba': 'I 9,46:c:z6','ontario': 'I 9,46:d:1,5','quentin': 'I 9,46:d:1,6','strasbourg': 'I 9,46:d:1,7','olten': 'I 9,46:d:e,n,z15','plymouth': 'I 9,46:d:z6','sontheim': 'I 9,46:d:z35','bergedorf': 'I 9,46:e,h:1,2','waedenswil': 'I 9,46:e,h:1,5','guerin': 'I 9,46:e,h:z6','ii 9,46:e,n,x:1,5,7': 'II 9,46:e,n,x:1,5,7','wernigerode': 'I 9,46:f,g:–','hillingdon': 'I 9,46:g,m:–','macclesfield': 'I 9,46:g,m,s:1,2,7','ii 9,46:g,m,s,t:e,n,x': 'II 9,46:g,[m],[s],t:[e,n,x]','gateshead': 'I 9,46:g,s,t:–','ii 9,46:g,z62:–': 'II 9,46:g,z62:–','ii 9,46:m,t:e,n,x': 'II 9,46:m,t:e,n,x','sangalkam': 'I 9,46:m,t:–','mathura': 'I 9,46:i:e,n,z15','potto': 'I 9,46:i:z6','marylebone': 'I 9,46:k:1,2','cochin': 'I 9,46:k:1,5','clontarf': 'I 9,46:k:1,6','ceyco': 'I 9,46:k:z35','india': 'I 9,46:l,v:1,5','geraldton': 'I 9,46:l,v:1,6','toronto': 'I 9,46:l,v:e,n,x','ackwepe': 'I 9,46:l,w:–','ii 9,46:l,w:e,n,x': 'II 9,46:l,w:e,n,x','nordrhein': 'I 9,46:l,z13,z28:e,n,z15','deckstein': 'I 9,46:r:1,7','shoreditch': 'I 9,46:r:e,n,z15','sokode': 'I 9,46:r:z6','cork': 'I 9,46:y:1,2','benin': 'I 9,46:y:1,7','irchel': 'I 9,46:y:e,n,x','nantes': 'I 9,46:y:l,w','mayday': 'I 9,46:y:z6','ii 9,46:z:1,5': 'II 9,46:z:1,5','ii 9,46:z:e,n,x': 'II 9,46:z:e,n,x','bambylor': 'I 9,46:z:e,n,z15','ii 9,46:z:z39': 'II 9,46:z:z39','ekotedo': 'I 9,46:z4,z23:–','ii 9,46:z4,z24:z39:z42': 'II 9,46:z4,z24:z39:z42','ngaparou': 'I 9,46:z4,z24:–','lishabi': 'I 9,46:z10:1,7','inglis': 'I 9,46:z10:e,n,x','mahina': 'I 9,46:z10:e,n,z15','louisiana': 'I 9,46:z10:z6','ii 9,46:z10:z6': 'II 9,46:z10:z6','ii 9,46:z10:z39': 'II 9,46:z10:z39','ouakam': 'I 9,46:z29:–:[z45]','hillegersberg': 'I 9,46:z35:1,5','basingstoke': 'I 9,46:z35:e,n,z15','trimdon': 'I 9,46:z35:z6','fresno': 'I 9,46:z38:–','ii 9,46:z39:1,7': 'II 9,46:z39:1,7','wuppertal': 'I 9,46:z41:–','ii 1,9,12,46,27:a:z6': 'II 1,9,12,46,27:a:z6','ii 1,9,12,46,27:b:z39': 'II 1,9,12,46,27:b:z39','ii 1,9,12,46,27:c:z39': 'II 1,9,12,46,27:c:z39','ii 9,12,46,27:g,t:e,n,x': 'II 9,12,46,27:g,t:e,n,x','ii 1,9,12,46,27:l,z13,z28:z39': 'II 1,9,12,46,27:l,z13,z28:z39','ii 1,9,12,46,27:y:z39': 'II 1,9,12,46,27:y:z39','ii 1,9,12,46,27:z4,z24:1,5': 'II 1,9,12,46,27:z4,z24:1,5','ii 1,9,12,46,27:z10:1,5': 'II 1,9,12,46,27:z10:1,5','ii 1,9,12,46,27:z10:e,n,x': 'II 1,9,12,46,27:z10:e,n,x','ii 1,9,12,46,27:z10:z39': 'II 1,9,12,46,27:z10:z39','aminatu': 'I 3,10:a:1,2','goelzau': 'I 3,{10}{15}:a:1,5','oxford': 'I 3,{10}{15}{15,34}:a:1,7','masembe': 'I 3,10:a:e,n,x','ii 3,10:a:e,n,x': 'II 3,10:a:e,n,x','galil': 'I 3,10:a:e,n,z15','ii 3,10:a:l,v': 'II 3,10:a:l,v','ii 3,10:a:z39': 'II 3,10:a:z39','ii 3,10:a:z87': 'II 3,10:a:z87','kalina': 'I 3,10:b:1,2','butantan': 'I 3,{10}{15}{15,34}:b:1,5','allerton': 'I 3,10:b:1,6','huvudsta': 'I 3,{10}{15,34}:b:1,7','benfica': 'I 3,10:b:e,n,x','ii 3,10:b:e,n,x': 'II 3,10:b:e,n,x','yaba': 'I 3,{10}{15}:b:e,n,z15','epicrates': 'I 3,10:b:l,w','wilmington': 'I 3,10:b:z6','westminster': 'I 3,{10}{15}:b:z35','ii 3,10:b:z39': 'II 3,10:b:z39','asylanta': 'I 3,10:c:1,2','gbadago': 'I 3,{10}{15}:c:1,5','ikayi': 'I 3,{10}{15}:c:1,6','pramiso': 'I 3,10:c:1,7','agege': 'I 3,10:c:e,n,z15','anderlecht': 'I 3,10:c:l,w','okefoko': 'I 3,10:c:z6','stormont': 'I 3,10:d:1,2','shangani': 'I 3,{10}{15}:d:1,5','lekke': 'I 3,10:d:1,6','onireke': 'I 3,10:d:1,7','souza': 'I 3,{10}{15}:d:e,n,x','ii 3,10:d:e,n,x': 'II 3,10:d:e,n,x','madjorio': 'I 3,10:d:e,n,z15','birmingham': 'I 3,{10}{15}:d:l,w','weybridge': 'I 3,10:d:z6','maron': 'I 3,10:d:z35','vejle': 'I 3,{10}{15}:e,h:1,2:[z27]','muenster': 'I 3,{10}{15}{15,34}:e,h:1,5:[z48]','anatum': 'I 3,{10}{15}{15,34}:e,h:1,6:[z64]','nyborg': 'I 3,{10}{15}:e,h:1,7','newlands': 'I 3,{10}{15,34}:e,h:e,n,x','lamberhurst': 'I 3,10:e,h:e,n,z15','meleagridis': 'I 3,{10}{15}{15,34}:e,h:l,w','sekondi': 'I 3,10:e,h:z6','ii 3,10:e,n,x:1,7': 'II 3,10:e,n,x:1,7','regent': 'I 3,10:f,g,[s]:[1,6]','alfort': 'I 3,10:f,g:e,n,x','suberu': 'I 3,10:g,m:–','amsterdam': 'I 3,{10}{15}{15,34}:g,m,s:–','ii 3,10,15:g,m,s,t:1,5': 'II 3,{10}{15}:g,m,s,t:[1,5]','westhampton': 'I 3,{10}{15}{15,34}:g,s,t:–:[z37]','bloomsbury': 'I 3,10:g,t:1,5','ii 3,10:g,t:–': 'II 3,10:g,t:–','ii 3,10:m,t:1,5': 'II 3,10:m,t:1,5','southbank': 'I 3,{10}{15}{15,34}:m,t:[1,6]','ii 3,10:m,t:e,n,x': 'II 3,10:m,t:e,n,x','cuckmere': 'I 3,10:i:1,2','amounderness': 'I 3,10:i:1,5','tibati': 'I 3,10:i:1,6','truro': 'I 3,10:i:1,7','bessi': 'I 3,10:i:e,n,x','falkensee': 'I 3,{10}{15}:i:e,n,z15','hoboken': 'I 3,10:i:l,w','yeerongpilly': 'I 3,10:i:z6','wimborne': 'I 3,10:k:1,2','zanzibar': 'I 3,{10}{15}:k:1,5','serrekunda': 'I 3,10:k:1,7','yundum': 'I 3,10:k:e,n,x','marienthal': 'I 3,10:k:e,n,z15','newrochelle': 'I 3,10:k:l,w','nchanga': 'I 3,{10}{15}:l,v:1,2','sinstorf': 'I 3,10:l,v:1,5','london': 'I 3,{10}{15}:l,v:1,6','give': 'I 3,{10}{15}{15,34}:l,v:1,7:[d],[z77]','ii 3,10:l,v:e,n,x': 'II 3,10:l,v:e,n,x','ruzizi': 'I 3,10:l,v:e,n,z15','ii 3,10:l,v:z6': 'II 3,10:l,v:z6','sinchew': 'I 3,10:l,v:z35','assinie': 'I 3,10:l,w:z6:[z45]','freiburg': 'I 3,10:l,z13:1,2','uganda': 'I 3,{10}{15}:l,z13:1,5','fallowfield': 'I 3,10:l,z13,z28:e,n,z15','hoghton': 'I 3,10:l,z13,z28:z6','ii 3,10:l,z28:1,5': 'II 3,10:l,z28:1,5','torbay': 'I 3,10:l,z28:1,5','joal': 'I 3,10:l,z28:1,7','lamin': 'I 3,10:l,z28:e,n,x','ii 3,10:l,z28:e,n,x': 'II 3,10:l,z28:e,n,x','ii 3,10:l,z28:z39': 'II 3,10:l,z28:z39','ughelli': 'I 3,10:r:1,5','elisabethville': 'I 3,{10}{15}:r:1,7','simi': 'I 3,10:r:e,n,z15','weltevreden': 'I 3,{10}{15}:r:z6','seegefeld': 'I 3,10:r,i:1,2','dumfries': 'I 3,10:r,i:1,6','amager': 'I 3,{10}{15}:y:1,2:[z45]','orion': 'I 3,{10}{15}{15,34}:y:1,5','mokola': 'I 3,10:y:1,7','ohlstedt': 'I 3,{10}{15}:y:e,n,x','bolton': 'I 3,10:y:e,n,z15','langensalza': 'I 3,10:y:l,w','stockholm': 'I 3,{10}{15}:y:z6','fufu': 'I 3,10:z:1,5','ii 3,10:z:1,5': 'II 3,10:z:1,5','harleystreet': 'I 3,10:z:1,6','huddinge': 'I 3,10:z:1,7','ii 3,10:z:e,n,x': 'II 3,10:z:e,n,x','kingslynn': 'I 3,10:z:e,n,z15','clerkenwell': 'I 3,10:z:l,w','landwasser': 'I 3,10:z:z6','ii 3,10:z:z39': 'II 3,10:z:z39','adabraka': 'I 3,10:z4,z23:[1,7]','wagadugu': 'I 3,10:z4,z23:z6','florian': 'I 3,{10}{15}:z4,z24:–','ii 3,10:z4,z24:–': 'II 3,10:z4,z24:–','okerara': 'I 3,10:z10:1,2','lexington': 'I 3,{10}{15}{15,34}:z10:1,5:[z49]','harrisonburg': 'I 3,{10}{15}{15,34}:z10:1,6','coquilhatville': 'I 3,10:z10:1,7','podiensis': 'I 3,10:z10:e,n,x','kristianstad': 'I 3,10:z10:e,n,z15','biafra': 'I 3,10:z10:z6','everleigh': 'I 3,10:z29:e,n,x','ii 3,10:z29:e,n,x': 'II 3,10:z29:[e,n,x]','jedburgh': 'I 3,{10}{15}:z29:–','santaclara': 'I 3,10:z35:1,5','ratchaburi': 'I 3,10:z35:1,6','zongo': 'I 3,10:z35:1,7','ii 3,10:z35:e,n,x,z15': 'II 3,10:z35:e,n,x,z15','shannon': 'I 3,10:z35:l,w','cairina': 'I 3,10:z35:z6','macallen': 'I 3,10:z36:–','sandaga': 'I 3,10:z38:1,2','albertslund': 'I 3,10:z38:1,6','bolombo': 'I 3,10:z38:[z6]','ii 3,10:z38:z42': 'II 3,10:z38:z42','ii 3,10:z39:1,5,7': 'II 3,10:z39:1,[5],7','dortmund': 'I 3,10:z41:1,[2],5','pietersburg': 'I 3,{10}{15,34}:z69:1,7','ii 3,15:z87:e,n,x,z15': 'II 3,15:z87:e,n,x,z15','capetown': 'I 3,15,34:y:1,6','niumi': 'I 1,3,19:a:1,5','juba': 'I 1,3,19:a:1,7','gwoza': 'I 1,3,19:a:e,n,z15','alkmaar': 'I 1,3,19:a:l,w','gnesta': 'I 1,3,19:b:1,5:[z37]','visby': 'I 1,3,19:b:1,6','tambacounda': 'I 1,3,19:b:e,n,x','kande': 'I 1,3,19:b:e,n,z15','broughton': 'I 1,3,19:b:l,w','accra': 'I 1,3,19:b:z6','eastglam': 'I 1,3,19:c:1,5','bida': 'I 1,3,19:c:1,6','madiago': 'I 1,3,19:c:1,7','umbadah': 'I 1,3,19:d:1,2','ahmadi': 'I 1,3,19:d:1,5','wanatah': 'I 1,3,19:d:1,7','liverpool': 'I 1,3,19:d:e,n,z15','tilburg': 'I 1,3,19:d:l,w:[z49]','niloese': 'I 1,3,19:d:z6','vilvoorde': 'I 1,3,19:e,h:1,5','hayindogo': 'I 1,3,19:e,h:1,6','sanktmarx': 'I 1,3,19:e,h:1,7','sao': 'I 1,3,19:e,h:e,n,z15','calabar': 'I 1,3,19:e,h:l,w','rideau': 'I 1,3,19:f,g:–','petahtikve': 'I 1,3,19:f,g,t:1,7','maiduguri': 'I 1,3,19:f,g,t:e,n,z15','kouka': 'I 1,3,19:g,m,[t]:–','senftenberg': 'I 1,3,19:g,[s],t:–:[z27],[z34],[z37],[z43],[z45],[z46],[z82]','cannstatt': 'I 1,3,19:m,t:–','stratford': 'I 1,3,19:i:1,2','ouagadougou': 'I 1,3,19:i:1,5','chichester': 'I 1,3,19:i:1,6','machaga': 'I 1,3,19:i:e,n,x','avonmouth': 'I 1,3,19:i:e,n,z15','zuilen': 'I 1,3,19:i:l,w','taksony': 'I 1,3,19:i:z6','oesterbro': 'I 1,3,19:k:1,5','bethune': 'I 1,3,19:k:1,7','ngor': 'I 1,3,19:l,v:1,5','parkroyal': 'I 1,3,19:l,v:1,7','svedvi': 'I 1,3,19:l,v:e,n,z15','fulda': 'I 1,3,19:l,w:1,5','westerstede': 'I 1,3,19:l,z13:1,2','winterthur': 'I 1,3,19:l,z13:1,6','lokstedt': 'I 1,3,19:l,z13,z28:1,2','stuivenberg': 'I 1,3,19:l,[z13],z28:1,5','bedford': 'I 1,3,19:l,z13,z28:e,n,z15','tomelilla': 'I 1,3,19:l,z28:1,7','kindia': 'I 1,3,19:l,z28:e,n,x','yalding': 'I 1,3,19:r:e,n,z15','fareham': 'I 1,3,19:r,i:l,w','gatineau': 'I 1,3,19:y:1,5','thies': 'I 1,3,19:y:1,7','slade': 'I 1,3,19:y:e,n,z15','kinson': 'I 1,3,19:y:e,n,x','krefeld': 'I 1,3,19:y:l,w','korlebu': 'I 1,3,19:z:1,5','kainji': 'I 1,3,19:z:1,6','lerum': 'I 1,3,19:z:1,7','schoeneberg': 'I 1,3,19:z:e,n,z15','carno': 'I 1,3,19:z:l,w','hongkong': 'I 1,3,19:z:z6','miromesnil': 'I 1,3,19:z4,z23:–','sambre': 'I 1,3,19:z4,z24:–','yenne': 'I 1,3,19:z10:1,5','dallgow': 'I 1,3,19:z10:e,n,z15','llandoff': 'I 1,3,19:z29:[z6]:[z37]','catumagos': 'I 1,3,19:z35:1,5','ochiogu': 'I 1,3,19:z38:[e,n,z15]','fortbend': 'I 1,3,19:z41:e,n,z15','chittagong': 'I 1,3,10,19:b:z35','bilu': 'I 1,3,10,19:f,g,t:1,(2),7','ilugun': 'I 1,3,10,19:z4,z23:z6','dessau': 'I 1,3,[15],19:g,s,t:–','cannonhill': 'I 1,3,{10}{15},19:y:e,n,x','gallen': 'I 11:a:1,2','marseille': 'I 11:a:1,5','vi 11:a:1,5': 'VI 11:a:1,5','massilia': 'I 11:a:1,6','toowong': 'I 11:a:1,7','luciana': 'I 11:a:e,n,z15','ii 11:a:e,n,z15:d': 'II 11:a:e,n,z15:d','epinay': 'I 11:a:l,z13,z28','ii 11:a:z6:z42': 'II 11:a:z6:z42','atento': 'I 11:b:1,2','leeuwarden': 'I 11:b:1,5','wohlen': 'I 11:b:1,6','vi 11:b:1,7': 'VI 11:b:1,7','vi 11:b:e,n,x': 'VI 11:b:e,n,x','pharr': 'I 11:b:e,n,z15','erfurt': 'I 11:b:z6','chiredzi': 'I 11:c:1,5','brindisi': 'I 11:c:1,6','ii 11:c:e,n,z15': 'II 11:c:e,n,z15','woodinville': 'I 11:c:e,n,x','ati': 'I 11:d:1,2','gustavia': 'I 11:d:1,5','chandans': 'I 11:d:[e,n,x]:[r]','findorff': 'I 11:d:z6','chingola': 'I 11:e,h:1,2','adamstua': 'I 11:e,h:1,6','redhill': 'I 11:e,h:l,z13,z28','abuja': 'I 11:g,m:1,5','missouri': 'I 11:g,s,t:–','ii 11:g,m,s,t:z39': 'II 11:g,[m],s,t:z39','iv 11:g,z51:–': 'IV 11:g,z51:–','moers': 'I 11:m,t:–','ii 11:m,t:e,n,x': 'II 11:m,t:e,n,x','aberdeen': 'I 11:i:1,2','brijbhumi': 'I 11:i:1,5','heerlen': 'I 11:i:1,6','veneziana': 'I 11:i:e,n,x','iiib 11:i:z': 'IIIb 11:i:z','pretoria': 'I 11:k:1,2','abaetetuba': 'I 11:k:1,5','sharon': 'I 11:k:1,6','colobane': 'I 11:k:1,7','kisarawe': 'I 11:k:e,n,x,[z15]','mannheim': 'I 11:k:l,w','amba': 'I 11:k:l,z13,z28','iiib 11:k:z53': 'IIIb 11:k:z53','stendal': 'I 11:l,v:1,2','maracaibo': 'I 11:l,v:1,5','fann': 'I 11:l,v:e,n,x','bullbay': 'I 11:l,v:e,n,z15','iiib 11:l,v:z:z56': 'IIIb 11:l,v:z:[z56]','iiib 11:l,v:z53': 'IIIb 11:l,v:z53','glidji': 'I 11:l,w:1,5','tours': 'I 11:l,z13:1,2','connecticut': 'I 11:l,z13,z28:1,5','osnabrueck': 'I 11:l,z13,z28:e,n,x','welwyn': 'I 11:l,z13,z28:e,n,z15','ii 11:l,z28:e,n,x': 'II 11:l,z28:e,n,x','senegal': 'I 11:r:1,5','rubislaw': 'I 11:r:e,n,x','clanvillian': 'I 11:r:e,n,z15','euston': 'I 11:r,i:e,n,x,z15','volta': 'I 11:r:l,z13,z28','solt': 'I 11:y:1,5','jalisco': 'I 11:y:1,7','herzliya': 'I 11:y:e,n,x','woumbou': 'I 11:y:e,n,x,z15','crewe': 'I 11:z:1,5','maroua': 'I 11:z:1,7','ii 11:z:e,n,x': 'II 11:z:e,n,x','maybush': 'I 11:z:e,n,z15','nyanza': 'I 11:z:z6:[z83]','ii 11:z:z39': 'II 11:z:z39','remete': 'I 11:z4,z23:1,6','etterbeek': 'I 11:z4,z23:e,n,z15','iiia 11:z4,z23:–': 'IIIa 11:z4,z23:–','iv 11:z4,z23:–': 'IV 11:z4,z23:–','yehuda': 'I 11:z4,z24:–','iv 11:z4,z32:–': 'IV 11:z4,z32:–','wentworth': 'I 11:z10:1,2','straengnaes': 'I 11:z10:1,5','telhashomer': 'I 11:z10:e,n,x','lene': 'I 11:z38:–','maastricht': 'I 11:z41:1,2','ii 11:z41:e,n,x': 'II 11:z41:e,n,x','ii 11:–:1,5': 'II 11:–:1,5','chagoua': 'I [1],13,23:a:1,5','ii 1,13,23:a:1,5': 'II [1],13,23:a:1,5','mim': 'I 13,22:a:1,6','ii 13,22:a:e,n,x': 'II 13,22:a:e,n,x','wyldegreen': 'I [1],13,23:a:l,w','marshall': 'I 13,22:a:l,z13,z28','ii 1,13,23:a:z42': 'II [1],13,23:a:z42','ibadan': 'I 13,22:b:1,5','mississippi': 'I [1],13,23:b:1,5','oudwijk': 'I 13,22:b:1,6','ii 1,13,23:b:1,5:z42': 'II [1],13,23:b:[1,5]:z42','bracknell': 'I 13,23:b:1,6','rottnest': 'I [1],13,22:b:1,7','vaertan': 'I 13,22:b:e,n,x','ullevi': 'I [1],13,23:b:e,n,x','bahati': 'I 13,22:b:e,n,z15','durham': 'I 13,23:b:e,n,z15','sanktjohann': 'I 13,23:b:l,w','ii 1,13,22:b:z42': 'II [1],13,22:b:z42','haouaria': 'I 13,22:c:e,n,x,z15','boston': 'I 13,23:c:e,n,z15','handen': 'I [1],13,23:d:1,2','ii 13,22:d:1,5': 'II 13,22:d:1,5','mishmarhaemek': 'I [1],13,23:d:1,5','friedenau': 'I 13,22:d:1,6','wichita': 'I [1],13,23:d:1,6:[z37]','grumpensis': 'I [1],13,23:d:1,7','ii 13,23:d:e,n,x': 'II 13,23:d:e,n,x','saintes': 'I 13,23:d:e,n,x','diguel': 'I [1],13,22:d:e,n,z15','telelkebir': 'I 13,23:d:e,n,z15','putten': 'I [1],13,23:d:l,w','isuge': 'I 13,23:d:z6','tschangu': 'I [1],13,23:e,h:1,5','willemstad': 'I [1],13,22:e,h:1,6','vridi': 'I [1],13,23:e,h:l,w','ii 1,13,23:e,n,x:1,5,7': 'II [1],13,23:e,n,x:1,[5],7','raus': 'I 13,22:f,g:e,n,x','havana': 'I [1],13,23:f,g,[s]:–:[z79]','bron': 'I 13,22:g,m:[e,n,z15]','iiib 13,22:g,m,s:z': 'IIIb 13,22:g,m,s:z','agbeni': 'I [1],13,23:g,m,[s],[t]:–','ii 1,13,22:g,m,t:1,5': 'II [1],13,22:g,m,t:[1,5]','ii 13,22:g,m,t:z42': 'II 13,22:g,m,t:z42','ii 1,13,23:g,m,s,t:1,5': 'II [1],13,23:g,m,s,t:1,5','ii 1,13,23:g,m,s,t:e,n,x': 'II [1],13,23:g,m,[s],t:[e,n,x]','ii 1,13,23:g,m,s,t:z42': 'II [1],13,23:g,m,s,t:z42','newyork': 'I 13,22:g,s,t:–','ii 13,23:g,s,t:e,n,x': 'II 13,23:g,s,t:e,n,x','okatie': 'I 13,23:g,[s],t:–','ii 1,13,22:g,t:1,5': 'II [1],13,22:g,t:[1,5]','ii 13,22:g,t:z6': 'II 13,22:g,t:z6','ii 1,13,23:g,t:1,5': 'II [1],13,23:g,t:1,5','ii 13,23:g,t:e,n,x': 'II 13,23:g,t:e,n,x','ii 1,13,23:g,s,t:z42': 'II [1],13,23:g,[s],t:z42','iiia 1,13,23:g,z51:–': 'IIIa [1],13,23:g,z51:–','washington': 'I 13,22:m,t:–','ii 1,13,23:m,t:1,5': 'II [1],13,23:m,t:1,5','ii 1,13,23:m,t:e,n,x': 'II [1],13,23:m,t:e,n,x','ii 13,22:m,t:z42:z39': 'II 13,22:m,t:z42:z39','ii 1,13,23:m,t:z42': 'II [1],13,23:m,t:z42','kintambo': 'I [1],13,23:m,t:–','v 1,13,22:i:–': 'V [1],13,22:i:–','idikan': 'I [1],13,23:i:1,5','myrria': 'I 13,23:i:1,7','jukestown': 'I 13,23:i:e,n,z15','kedougou': 'I [1],13,23:i:l,w','chapuri': 'I 13,23:i:z6','ii 13,22:k:1,5:z42': 'II 13,22:k:1,5:z42','noho': 'I 13,23:k:1,6','marburg': 'I 13,23:k:–','galway': 'I 13,23:k:e,n,z15','ii 13,23:k:z41': 'II 13,23:k:z41','lovelace': 'I 13,22:l,v:1,5','iiib 13,22:l,v:1,5,7': 'IIIb 13,22:l,v:1,5,7','borbeck': 'I 13,22:l,v:1,6','nanga': 'I [1],13,23:l,v:e,n,z15','ii 13,23:l,w:e,n,x': 'II 13,23:l,w:e,n,x','taiping': 'I 13,22:l,z13:e,n,z15','ii 13,22:l,z28:1,5': 'II 13,22:l,z28:1,5','ii 13,23:l,z28:1,5': 'II 13,23:l,z28:1,5','ii 13,23:l,z28:z6': 'II 13,23:l,z28:z6','ii 1,13,23:l,z28:z42': 'II [1],13,23:l,z28:z42','v 13,22:r:–': 'V 13,22:r:–','adjame': 'I 13,23:r:1,6','linton': 'I 13,23:r:e,n,z15','tanger': 'I [1],13,22:y:1,6','yarrabah': 'I 13,23:y:1,7','ordonez': 'I [1],13,23:y:l,w','tunis': 'I [1],13,23:y:z6','winslow': 'I 13,22:z:1,5','ii 1,13,23:z:1,5': 'II [1],13,23:z:1,5','serenli': 'I [1],13,23:z:1,5','iiib 13,23:z:1,5': 'IIIb 13,23:z:1,5','poona': 'I [1],13,22:z:1,6:[z44],[z59]','farmsen': 'I 13,23:z:1,6','bristol': 'I 13,22:z:1,7','durance': 'I 13,22:z:z6','ivrysurseine': 'I [1],13,23:z:z6','tanzania': 'I [1],13,22:z:e,n,z15','worthington': 'I [1],13,23:z:l,w:[z43]','ii 1,13,23:z:z42': 'II [1],13,23:z:z42','ii 13,22:z:–': 'II 13,22:z:–','ried': 'I [1],13,22:z4,z23:[e,n,z15]','iiia 13,22:z4,z23:–': 'IIIa 13,22:z4,z23:–','ajiobo': 'I 13,23:z4,z23:–','iiia 13,23:z4,z23,z32:–': 'IIIa 13,23:z4,z23,[z32]:–','romanby': 'I [1],13,23:z4,z24:–','iiia 13,23:z4,z24:–': 'IIIa 13,23:z4,z24:–','roodepoort': 'I [1],13,22:z10:1,5','ii 1,13,22:z10:z6': 'II [1],13,22:z10:z6','sapele': 'I 13,23:z10:e,n,z15','demerara': 'I 13,23:z10:l,w','ii 13,22:z29:1,5': 'II 13,22:z29:1,5','ii 13,22:z29:e,n,x': 'II 13,22:z29:e,n,x','ii 1,13,23:z29:1,5': 'II [1],13,23:z29:1,5','ii 1,13,23:z29:e,n,x': 'II [1],13,23:z29:e,n,x','agoueve': 'I 13,22:z29:–','cubana': 'I [1],13,23:z29:–:[z37],[z43]','mampong': 'I 13,22:z35:1,6','nimes': 'I 13,22:z35:e,n,z15','picpus': 'I 13,23:z35:1,6','anna': 'I 13,23:z35:e,n,z15','leiden': 'I 13,22:z38:–','fanti': 'I 13,23:z38:–','v 13,22:z39:–': 'V 13,22:z39:–','ii 13,22:z39:1,7': 'II 13,22:z39:1,7','ii 1,13,23:z39:1,5,7': 'II [1],13,23:z39:1,5,7','pajala': 'I 13,23:z41:e,n,z15','ii 1,13,23:z42:1,5,7': 'II [1],13,23:[z42]:1,[5],7','ii 13,23:–:1,6': 'II 13,23:–:1,6','garba': 'I 1,6,14,25:a:1,5','vi 1,6,14:a:1,5': 'VI [1],6,14:a:1,5','vi 1,6,14,25:a:e,n,x': 'VI [1],6,14,[25]:a:e,n,x','banjul': 'I 1,6,14,25:a:e,n,z15','ndjamena': 'I 1,6,14,25:b:1,2','kuntair': 'I 1,6,14,25:b:1,5','tucson': 'I [1],6,14,[25]:b:1,7','iiib 6,14:b:e,n,x': 'IIIb (6),14:b:e,n,x','blijdorp': 'I 1,6,14,25:c:1,5','kassberg': 'I 1,6,14,25:c:1,6','runby': 'I 1,6,14,25:c:e,n,x','minna': 'I 1,6,14,25:c:l,w','martonos': 'I 6,14,24:d:1,5','finkenwerder': 'I [1],6,14,[25]:d:1,5','woodhull': 'I 1,6,14,25:d:1,6','midway': 'I 6,14,24:d:1,7','florida': 'I [1],6,14,[25]:d:1,7','lindern': 'I 6,14,[24]:d:e,n,x','charity': 'I [1],6,14,[25]:d:e,n,x','teko': 'I [1],6,14,[25]:d:e,n,z15','encino': 'I 1,6,14,25:d:l,z13,z28','albuquerque': 'I 1,6,14,24:d:z6','bahrenfeld': 'I 6,14,[24]:e,h:1,5','onderstepoort': 'I 1,6,14,[25]:e,h:1,5','magumeri': 'I 1,6,14,25:e,h:1,6','beaudesert': 'I [1],6,14,[25]:e,h:1,7','v 6,14:e,n,z15:–': 'V 6,14:e,n,z15:–','warragul': 'I [1],6,14,[25]:g,m:–','caracas': 'I [1],6,14,[25]:g,m,s:–','sylvania': 'I [1],6,14,[25]:g,p:–','catanzaro': 'I 6,14:g,s,t:–','ii 1,6,14:m,t:1,5': 'II 1,6,14:m,t:1,5','ii 6,14:m,t:e,n,x': 'II 6,14:m,t:e,n,x','kaitaan': 'I 1,6,14,25:m,t:–','mampeza': 'I 1,6,14,25:i:1,5','buzu': 'I [1],6,14,[25]:i:1,7','schalkwijk': 'I 6,14,[24]:i:e,n,z15','moussoro': 'I 1,6,14,25:i:e,n,z15','harburg': 'I [1],6,14,[25]:k:1,5','ii 6,14,24:k:1,6': 'II 6,14,[24]:k:1,6','ii 6,14:k:e,n,x': 'II 6,14:k:e,n,x','trumersee': 'I 1,6,14,25:k:e,n,x','iiib 6,14:k:z': 'IIIb (6),14:k:z','ii 1,6,14:k:z6:z42': 'II 1,6,14:k:z6:z42','iiib 6,14:k:z53': 'IIIb (6),14:k:z53','amberg': 'I 6,14,24:l,v:1,7','boecker': 'I [1],6,14,[25]:l,v:1,7','horsham': 'I 1,6,14,[25]:l,v:e,n,x','alpenquai': 'I 6,14:l,v:e,n,z15','iiib 6,14:l,v:z': 'IIIb (6),14:l,v:z','iiib 6,14:l,v:z35': 'IIIb (6),14:l,v:z35','iiib 6,14:l,v:z53': 'IIIb (6),14:l,v:z53','vi 6,14:l,v:z88': 'VI 6,14:l,v:z88','owerri': 'I 6,14,25:l,z13,z28:1,6','aflao': 'I 1,6,14,25:l,z28:e,n,x','istoria': 'I 1,6,14,25:r,i:1,5','iiib 6,14:r:z': 'IIIb (6),14:r:z','surat': 'I [1],6,14,[25]:r,[i]:e,n,z15','carrau': 'I 6,14,[24]:y:1,7','madelia': 'I 1,6,14,25:y:1,7','fischerkietz': 'I 1,6,14,25:y:e,n,x','mornington': 'I 1,6,14,25:y:e,n,z15','homosassa': 'I 1,6,14,25:z:1,5','kanifing': 'I 1,6,14,25:z:1,6','soahanina': 'I 6,14,24:z:e,n,x','sundsvall': 'I [1],6,14,[25]:z:e,n,x','royan': 'I 1,6,14,25:z:e,n,z15','stafford': 'I 6,14,24:z:l,z13,z28','poano': 'I [1],6,14,[25]:z:l,z13,z28','arapahoe': 'I 6,14:z4,z23:1,5','bousso': 'I 1,6,14,25:z4,z23:[e,n,z15]','iv 6,14:z4,z23:–': 'IV 6,14:z4,z23:–','chichiri': 'I 6,14,24:z4,z24:–','uzaramo': 'I 1,6,14,25:z4,z24:–','nessa': 'I 1,6,14,25:z10:1,2','vi 1,6,14,25:z10:1,2,7': 'VI 1,6,14,25:z10:1,(2),7','ii 1,6,14:z10:1,5': 'II 1,6,14:z10:1,5','laredo': 'I 1,6,14,25:z10:1,6','iiib 6,14:z10:e,n,x,z15': 'IIIb (6),14:z10:e,n,x,z15','iiib 6,14:z10:z:z56,z90': 'IIIb (6),14:z10:z:[z56],[z90]','ulm': 'I 1,6,14,25:z10:z6','ii 1,6,14:z10:z6:z42': 'II 1,6,14:z10:z6:z42','iiib 6,14:z10:z53': 'IIIb 6,14:z10:z53','potosi': 'I 6,14:z36:1,5','ii 6,14:z36:–': 'II 6,14:z36:–','sara': 'I 1,6,14,25:z38:[e,n,x]','ii 1,6,14:z42:1,6': 'II 1,6,14:z42:1,6','iiib 6,14:z52:e,n,x,z15': 'IIIb 6,14:z52:e,n,x,z15','iiib 1,6,14,25:z52:z35': 'IIIb [1],6,14,[25]:z52:z35','hannover': 'I 16:a:1,2','brazil': 'I 16:a:1,5','amunigun': 'I 16:a:1,6','nyeko': 'I 16:a:1,7','togba': 'I 16:a:e,n,x','fischerhuette': 'I 16:a:e,n,z15','heron': 'I 16:a:z6','hull': 'I 16:b:1,2','melaka': 'I 16:b:1,2,5','wa': 'I 16:b:1,5','glasgow': 'I 16:b:1,6','hvittingfoss': 'I 16:b:e,n,x','ii 16:b:e,n,x': 'II 16:b:e,n,x','sangera': 'I 16:b:e,n,z15','vegesack': 'I 16:b:l,w','malstatt': 'I 16:b:z6','ii 16:b:z39': 'II 16:b:z39','ii 16:b:z42': 'II 16:b:z42','vancouver': 'I 16:c:1,5','gafsa': 'I 16:c:1,6','shamba': 'I 16:c:e,n,x','hithergreen': 'I 16:c:e,n,z15','yoruba': 'I 16:c:l,w','oldenburg': 'I 16:d:1,2','sculcoates': 'I 16:d:1,5','ii 16:d:1,5': 'II 16:d:1,5','sherbrooke': 'I 16:d:1,6','gaminara': 'I 16:d:1,7','barranquilla': 'I 16:d:e,n,x','ii 16:d:e,n,x': 'II 16:d:e,n,x','nottingham': 'I 16:d:e,n,z15','caen': 'I 16:d:l,w:[z82]','barmbek': 'I 16:d:z6','malakal': 'I 16:e,h:1,2','saboya': 'I 16:e,h:1,5','astridplein': 'I 16:e,h:1,6','rhydyfelin': 'I 16:e,h:e,n,x','moabit': 'I 16:e,h:l,w','weston': 'I 16:e,h:z6','ii 16:e,n,x:1,5,7': 'II 16:e,n,x:1,(5),7','ii 16:e,n,x:1,6:z42': 'II 16:e,n,x:1,6:z42','tees': 'I 16:f,g:–:[z37]','adeoyo': 'I 16:g,m,[t]:–','nikolaifleet': 'I 16:g,m,s:–','ii 16:g,m,s,t:1,5:z42': 'II 16:g,[m],[s],t:[1,5]:[z42]','ii 16:g,m,s,t:e,n,x': 'II 16:g,[m],[s],t:[e,n,x]','cardoner': 'I 16:g,s,t:–','ii 16:m,t:e,n,x': 'II 16:m,t:e,n,x','morbihan': 'I 16:m,t:e,n,z15','ii 16:m,t:z42': 'II 16:m,t:[z42]','mpouto': 'I 16:m,t:–','amina': 'I 16:i:1,5','agbara': 'I 16:i:1,6','wisbech': 'I 16:i:1,7','frankfurt': 'I 16:i:e,n,z15','pisa': 'I 16:i:l,w','abobo': 'I 16:i:z6','iiib 16:i:z35': 'IIIb 16:i:z35','szentes': 'I 16:k:1,2','maumee': 'I 16:k:1,6','nuatja': 'I 16:k:e,n,x','orientalis': 'I 16:k:e,n,z15','iiib 16:k:e,n,x,z15': 'IIIb 16:(k):e,n,x,z15','iiib 16:k:z': 'IIIb 16:k:z','iiib 16:k:z35': 'IIIb 16:(k):z35','iiib 16:k:z53': 'IIIb 16:k:z53','iiib 16:l,v:1,5,7': 'IIIb 16:l,v:1,5,7','shanghai': 'I 16:l,v:1,6:[z45]','welikade': 'I 16:l,v:1,7','salford': 'I 16:l,v:e,n,x','burgas': 'I 16:l,v:e,n,z15','iiib 16:l,v:z:z61': 'IIIb 16:l,v:z:[z61]','losangeles': 'I 16:l,v:z6','iiib 16:l,v:z35': 'IIIb 16:l,v:z35','iiib 16:l,v:z53': 'IIIb 16:l,v:z53','zigong': 'I 16:l,w:1,5','westeinde': 'I 16:l,w:1,6','brooklyn': 'I 16:l,w:e,n,x','lomnava': 'I 16:l,w:e,n,z15','essingen': 'I 16:l,w:z6','ii 16:l,w:z6': 'II 16:l,w:z6','mandera': 'I 16:l,z13:e,n,z15','enugu': 'I 16:l,[z13],z28:[1,5]','battle': 'I 16:l,z13,z28:1,6','ablogame': 'I 16:l,z13,z28:z6','koblenz': 'I 16:l,z13,z28:e,n,x','ii 16:l,z28:z42': 'II 16:l,z28:z42','rovaniemi': 'I 16:r,i:1,5','ivory': 'I 16:r:1,6','brunflo': 'I 16:r:1,7','lehrte': 'I 16:r:z6','annedal': 'I 16:r,i:e,n,x','zwickau': 'I 16:r,i:e,n,z15','saphra': 'I 16:y:1,5','akuafo': 'I 16:y:1,6','kikoma': 'I 16:y:e,n,x','avignon': 'I 16:y:e,n,z15','gerland': 'I 16:z:1,5','fortlamy': 'I 16:z:1,6','lingwala': 'I 16:z:1,7','kassel': 'I 16:z:e,n,x','ii 16:z:e,n,x': 'II 16:z:e,n,x','brevik': 'I 16:z:e,n,[x],z15','bouake': 'I 16:z:z6','ii 16:z:z42': 'II 16:z:z42','kibi': 'I 16:z4,z23:[1,6]','axim': 'I 16:z4,z23:z6','ii 16:z4,z23:–': 'II 16:z4,z23:–','iv 16:z4,z23:–': 'IV 16:z4,z23:–','kaevlinge': 'I 16:z4,z24:–','ii 16:z4,z24:–': 'II 16:z4,z24:–','iv 16:z4,z24:–': 'IV 16:z4,z24:–','iv 16:z4,z32:–': 'IV 16:z4,z32:–','ii 16:z6:1,6': 'II 16:z6:1,6','badagry': 'I 16:z10:1,5','iiib 16:z10:1,7': 'IIIb 16:z10:1,7','lisboa': 'I 16:z10:1,6','iiib 16:z10:e,n,x,z15': 'IIIb 16:z10:e,n,x,z15','redlands': 'I 16:z10:e,n,z15','angouleme': 'I 16:z10:z6','saloniki': 'I 16:z29:–','ii 16:z29:1,5': 'II 16:z29:1,5','ii 16:z29:e,n,x': 'II 16:z29:e,n,x','neuland': 'I 16:z35:1,5','trier': 'I 16:z35:1,6','dakota': 'I 16:z35:e,n,z15','ii 16:z35:e,n,x': 'II 16:z35:e,n,x','iv 16:z36:–': 'IV 16:z36:–','ii 16:z36:e,n,z15': 'II 16:z36:e,n,z15','naware': 'I 16:z38:–','grancanaria': 'I 16:z39:[1,6]','ii 16:z42:1,5,7': 'II 16:z42:1,(5),7','ii 16:z42:1,6': 'II 16:z42:1,6','iiib 16:z52:z35': 'IIIb 16:z52:z35','bonames': 'I 17:a:1,2','jangwani': 'I 17:a:1,5','kinondoni': 'I 17:a:e,n,x','kirkee': 'I 17:b:1,2','dahra': 'I 17:b:1,5','mattenhof': 'I 17:b:e,n,x','ii 17:b:e,n,x,z15': 'II 17:b:e,n,x,z15','bignona': 'I 17:b:e,n,z15','ii 17:b:z6': 'II 17:b:z6','luedinghausen': 'I 17:c:1,5','victoriaborg': 'I 17:c:1,6','ii 17:c:z39': 'II 17:c:z39','ii 17:d:–': 'II 17:d:–','berlin': 'I 17:d:1,5','karlshamn': 'I 17:d:e,n,z15','niamey': 'I 17:d:l,w','ii 17:d:z6': 'II 17:d:z6','jubilee': 'I 17:e,h:1,2','ii 17:e,n,x,z15:1,6': 'II 17:e,n,x,z15:1,6','ii 17:e,n,x,z15:1,5,7': 'II 17:e,n,x,z15:1,[5],7','wembleypark': 'I 17:g,m:–','ii 17:g,m,s,t:–': 'II 17:g,m,s,t:–','lowestoft': 'I 17:g,s,t:–','ii 17:g,t:e,n,x,z15': 'II 17:g,t:[e,n,x,z15]','ii 17:g,t:z39': 'II 17:g,t:z39','iiia 17:g,z51:–': 'IIIa 17:g,z51:–','bama': 'I 17:m,t:–','ii 17:m,t:–': 'II 17:m,t:–','ahanou': 'I 17:i:1,7','iiib 17:i:z35': 'IIIb 17:i:z35','irenea': 'I 17:k:1,5','bandim': 'I 17:k:1,6','warri': 'I 17:k:1,7','matadi': 'I 17:k:e,n,x','zaria': 'I 17:k:e,n,z15','iiib 17:k:z': 'IIIb 17:k:z','ii 17:k:–': 'II 17:k:–','morotai': 'I 17:l,v:1,2','michigan': 'I 17:l,v:1,5','lancaster': 'I 17:l,v:1,7','carmel': 'I 17:l,v:e,n,x','iiib 17:l,v:e,n,x,z15': 'IIIb 17:l,v:e,n,x,z15','iiib 17:l,v:z35': 'IIIb 17:l,v:z35','granlo': 'I 17:l,z28:e,n,x','lode': 'I 17:r:1,2','iiib 17:r:z': 'IIIb 17:r:z','ii 17:y:–': 'II 17:y:–','tendeba': 'I 17:y:e,n,x','hadejia': 'I 17:y:e,n,z15','lokomo': 'I 17:y:l,w','gori': 'I 17:z:1,2','warengo': 'I 17:z:1,5','dingiri': 'I 17:z:1,6','ii 17:z:1,7': 'II 17:z:1,7','tchamba': 'I 17:z:e,n,z15','ii 17:z:l,w:z42': 'II 17:z:l,w:z42','iiia 17:z4,z23:–': 'IIIa 17:z4,z23:–','iiia 17:z4,z23,z32:–': 'IIIa 17:z4,z23,z32:–','iiia 17:z4,z24:–': 'IIIa 17:z4,z24:–','iiia 17:z4,z32:–': 'IIIa 17:z4,z32:–','djibouti': 'I 17:z10:e,n,x','iiib 17:z10:e,n,x,z15:z56': 'IIIb 17:z10:e,n,x,z15:[z56]','iiib 17:z10:z': 'IIIb 17:z10:z','ii 17:z10:–': 'II 17:z10:–','kandla': 'I 17:z29:–','iiia 17:z29:–': 'IIIa 17:z29:–','iv 17:z29:–': 'IV 17:z29:–','aachen': 'I 17:z35:1,6','iiia 17:z36:–': 'IIIa 17:z36:–','iv 17:z36:–': 'IV 17:z36:–','brazos': 'I [6],[14],18:a:e,n,z15','fluntern': 'I [6],[14],18:b:1,5','cochise': 'I 18:b:1,7','rawash': 'I [6],[14],18:c:e,n,x','groenekan': 'I 18:d:1,5','usumbura': 'I [6],[14],18:d:1,7','pontypridd': 'I 18:g,m:–','eaubonne': 'I 18:g,s,t:–','iiia 18:g,z51:–': 'IIIa 18:g,z51:–','iv 18:g,z51:–': 'IV 18:g,z51:–','ii 18:m,t:1,5': 'II 18:m,t:1,5','langenhorn': 'I 18:m,t:–','memphis': 'I 18:k:1,5','iiib 18:k:z53': 'IIIb 18:(k):z53','iiib 18:k:z54': 'IIIb 18:(k):z54','iiib 18:l,v:e,n,x,z15': 'IIIb 18:l,v:e,n,x,z15','orlando': 'I 18:l,v:e,n,z15','iiib 18:l,v:z:z50': 'IIIb 18:l,v:z:[z50]','iiib 18:l,v:z53': 'IIIb 18:l,v:z53','toulon': 'I 18:l,w:e,n,z15','tennenlohe': 'I 18:r:1,5','iiib 18:r:z': 'IIIb 18:r:z','troy': 'I 18:y:1,7','ii 18:y:e,n,x,z15': 'II 18:y:e,n,x,z15','potengi': 'I 18:z:–','cerro': 'I [6],[14],18:z4,z23:[1,5]:[z45],[z82]','aarhus': 'I 18:z4,z23:z64','ii 18:z4,z23:–': 'II 18:z4,z23:–','iiia 18:z4,z23:–': 'IIIa 18:z4,z23:–','blukwa': 'I [6],[14],18:z4,z24:–','ii 18:z4,z24:–': 'II 18:z4,z24:–','iiia 18:z4,z32:–': 'IIIa 18:z4,z32:–','iiib 18:z10:e,n,x,z15': 'IIIb 18:z10:e,n,x,z15','leer': 'I 18:z10:1,5','carnac': 'I 18:z10:z6','ii 18:z10:z6': 'II 18:z10:z6','ii 18:z36:–': 'II 18:z36:–','iiia 18:z36:–': 'IIIa 18:z36:–','iv 18:z36,z38:–': 'IV 18:z36,z38:–','sinthia': 'I 18:z38:–','delmenhorst': 'I 18:z71:–','cotia': 'I 18:–:1,6','assen': 'I 21:a:[1,5]','ii 21:b:1,5': 'II 21:b:1,5','ghana': 'I 21:b:1,6','minnesota': 'I 21:b:e,n,x:[z33],[z49]','hydra': 'I 21:c:1,6','rhone': 'I 21:c:e,n,x','ii 21:c:e,n,x': 'II 21:c:e,n,x','iiib 21:c:e,n,x,z15': 'IIIb 21:c:e,n,x,z15','spartel': 'I 21:d:1,5','magwa': 'I 21:d:e,n,x','madison': 'I 21:d:z6','good': 'I 21:f,g:e,n,x','ii 21:g,m,s,t:–': 'II 21:g,[m],[s],t:–','iiia 21:g,z51:–': 'IIIa 21:g,z51:–','iv 21:g,z51:–': 'IV 21:g,z51:–','ii 21:m,t:–': 'II 21:m,t:–','iv 21:m,t:–': 'IV 21:m,t:–','diourbel': 'I 21:i:1,2','iiib 21:i:1,5,7': 'IIIb 21:i:1,5,7','iiib 21:i:e,n,x,z15': 'IIIb 21:i:e,n,x,z15','salpetriere': 'I 21:k:1,6','iiib 21:k:e,n,x,z15': 'IIIb 21:k:e,n,x,z15','iiib 21:k:z': 'IIIb 21:k:z','surrey': 'I 21:k:1,(2),5','iiib 21:l,v:z': 'IIIb 21:l,v:z','iiib 21:l,v:z57': 'IIIb 21:l,v:z57','keve': 'I 21:l,w:–','jambur': 'I 21:l,z28:e,n,z15','mountmagnet': 'I 21:r:–','iiib 21:r:z': 'IIIb 21:r:z','ibaragi': 'I 21:y:1,2','ruiru': 'I 21:y:e,n,x','ii 21:z:–': 'II 21:z:–','baguida': 'I 21:z4,z23:–','iiia 21:z4,z23:–': 'IIIa 21:z4,z23:–','iv 21:z4,z23:–': 'IV 21:z4,z23:–','ii 21:z4,z24:–': 'II 21:z4,z24:–','iiia 21:z4,z24:–': 'IIIa 21:z4,z24:–','iv 21:z4,z32:–': 'IV 21:z4,z32:–','iiib 21:z10:e,n,x,z15': 'IIIb 21:z10:e,n,x,z15','iiib 21:z10:z': 'IIIb 21:z10:z','ii 21:z10:z6': 'II 21:z10:[z6]','iiib 21:z10:z53': 'IIIb 21:z10:z53','iiia 21:z29:–': 'IIIa 21:z29:–','gambaga': 'I 21:z35:e,n,z15','iiia 21:z36:–': 'IIIa 21:z36:–','iv 21:z36:–': 'IV 21:z36:–','iv 21:z38:–': 'IV 21:z38:–','iiib 21:z65:e,n,x,z15': 'IIIb 21:z65:e,n,x,z15','solna': 'I 28:a:1,5','dakar': 'I 28:a:1,6','bakau': 'I 28:a:1,7','seattle': 'I 28:a:e,n,x','ii 28:a:e,n,x': 'II 28:a:e,n,x','honelis': 'I 28:a:e,n,z15','dibra': 'I 28:a:z6','moero': 'I 28:b:1,5','ashanti': 'I 28:b:1,6','bokanjac': 'I 28:b:1,7','soumbedioune': 'I 28:b:e,n,x','ii 28:b:e,n,x': 'II 28:b:e,n,x','langford': 'I 28:b:e,n,z15','freefalls': 'I 28:b:l,w','ii 28:b:z6': 'II 28:b:z6','hermannswerder': 'I 28:c:1,5','eberswalde': 'I 28:c:1,6','halle': 'I 28:c:1,7','dresden': 'I 28:c:e,n,x','wedding': 'I 28:c:e,n,z15','parabiago': 'I 28:c:l,w','techimani': 'I 28:c:z6','amoutive': 'I 28:d:1,5','hatfield': 'I 28:d:1,6','mundonobo': 'I 28:d:1,7','mocamedes': 'I 28:d:e,n,x','patience': 'I 28:d:e,n,z15','cullingworth': 'I 28:d:l,w','etobicoke': 'I 28:d:z6','korkeasaari': 'I 28:e,h:1,5','kpeme': 'I 28:e,h:1,7','gozo': 'I 28:e,h:e,n,z15','ii 28:e,n,x:1,7': 'II 28:e,n,x:1,7','ii 28:e,n,z15:z87': 'II 28:e,n,z15:z87','friedrichsfelde': 'I 28:f,g:–','yardley': 'I 28:g,m:1,6','abadina': 'I 28:g,m:[e,n,z15]','ii 28:g,m,s,t:1,5': 'II 28:g,(m),[s],t:1,5','croft': 'I 28:g,m,s:[e,n,z15]','ii 28:g,m,t:e,n,x': 'II 28:g,m,t:e,n,x','ii 28:g,m,t:z39': 'II 28:g,m,t:z39','ii 28:g,s,t:e,n,x': 'II 28:g,s,t:e,n,x','ona': 'I 28:g,s,t:–','ii 28:m,t:e,n,x': 'II 28:m,t:[e,n,x]','vinohrady': 'I 28:m,t:[e,n,z15]','ii 28:m,t:z39': 'II 28:m,t:z39','morillons': 'I 28:m,t:1,6','doorn': 'I 28:i:1,2','cotham': 'I 28:i:1,5','volkmarsdorf': 'I 28:i:1,6','dieuppeul': 'I 28:i:1,7','warnemuende': 'I 28:i:e,n,x','kuessel': 'I 28:i:e,n,z15','douala': 'I 28:i:l,w','guildford': 'I 28:k:1,2','ilala': 'I 28:k:1,5','adamstown': 'I 28:k:1,6','ikeja': 'I 28:k:1,7','iiib 28:k:1,7': 'IIIb 28:k:1,7','taunton': 'I 28:k:e,n,x','ank': 'I 28:k:e,n,z15','leoben': 'I 28:l,v:1,5','vitkin': 'I 28:l,v:e,n,x','nashua': 'I 28:l,v:e,n,z15','ramsey': 'I 28:l,w:1,6','catalunia': 'I 28:l,z13,z28:1,5','penilla': 'I 28:l,z13,z28:e,n,z15','ii 28:l,z28:1,5': 'II 28:l,z28:1,5','fajara': 'I 28:l,z28:e,n,x','ii 28:l,z28:e,n,x': 'II 28:l,z28:e,n,x','bassadji': 'I 28:r:1,6','kibusi': 'I 28:r:e,n,x','ii 28:r:e,n,z15': 'II 28:r:e,n,z15','fairfield': 'I 28:r:l,w','chicago': 'I 28:r,[i]:1,5','banco': 'I 28:r,i:1,7','sanktgeorg': 'I 28:r,[i]:e,n,z15','oskarshamn': 'I 28:y:1,2','nima': 'I 28:y:1,5','pomona': 'I 28:y:1,7:[z80],[z90]','kitenge': 'I 28:y:e,n,x','telaviv': 'I 28:y:e,n,z15','shomolu': 'I 28:y:l,w','selby': 'I 28:y:z6','vanier': 'I 28:z:1,5','ii 28:z:1,5': 'II 28:z:1,5','doel': 'I 28:z:1,6','ezra': 'I 28:z:1,7','brisbane': 'I 28:z:e,n,z15','ii 28:z:z39': 'II 28:z:z39','cannobio': 'I 28:z4,z23:1,5','teltow': 'I 28:z4,z23:1,6','babelsberg': 'I 28:z4,z23:[e,n,z15]','kethiabarny': 'I 28:z4,z24:–','rogy': 'I 28:z10:1,2','farakan': 'I 28:z10:1,5','libreville': 'I 28:z10:1,6','malaysia': 'I 28:z10:1,7','umbilo': 'I 28:z10:e,n,x','luckenwalde': 'I 28:z10:e,n,z15','moroto': 'I 28:z10:l,w','iiib 28:z10:z': 'IIIb 28:z10:z','djermaia': 'I 28:z29:–','ii 28:z29:1,5': 'II 28:z29:1,5','ii 28:z29:e,n,x': 'II 28:z29:e,n,x','konolfingen': 'I 28:z35:1,6','babili': 'I 28:z35:1,7','santander': 'I 28:z35:e,n,z15','aderike': 'I 28:z38:e,n,z15','overvecht': 'I 30:a:1,2','zehlendorf': 'I 30:a:1,5','guarapiranga': 'I 30:a:e,n,x','doulassame': 'I 30:a:e,n,z15','ii 30:a:z39': 'II 30:a:z39','louga': 'I 30:b:1,2','aschersleben': 'I 30:b:1,5','tempe': 'I 30:b:1,7:[z33]','urbana': 'I 30:b:e,n,x','neudorf': 'I 30:b:e,n,z15','coppettswood': 'I 30:b:l,z13,z28','ii 30:b:z6': 'II 30:b:z6','kakikoka': 'I 30:b:z6','zaire': 'I 30:c:1,7','morningside': 'I 30:c:e,n,z15','ii 30:c:z39': 'II 30:c:z39','messina': 'I 30:d:1,5','livulu': 'I 30:e,h:1,2','torhout': 'I 30:e,h:1,5','godesberg': 'I 30:g,m,[t]:–','ii 30:g,m,s:e,n,x': 'II 30:g,m,s:e,n,x','giessen': 'I 30:g,m,s:–','iv 30:g,m,t,z51:–': 'IV 30:g,m,t,z51:–','sternschanze': 'I 30:g,s,t:–:[z59]','ii 30:g,t:–': 'II 30:g,t:–','wayne': 'I 30:g,z51:–','ii 30:m,t:–': 'II 30:m,t:–','landau': 'I 30:i:1,2','morehead': 'I 30:i:1,5','mjordan': 'I 30:i:e,n,z15','soerenga': 'I 30:i:l,w','hilversum': 'I 30:k:1,2','ramatgan': 'I 30:k:1,5','aqua': 'I 30:k:1,6','angoda': 'I 30:k:e,n,x','odozi': 'I 30:k:e,n,[x],z15','ii 30:k:e,n,x,z15': 'II 30:k:e,n,x,z15','scarborough': 'I 30:k:l,z13,z28','ligeo': 'I 30:l,v:1,2','donna': 'I 30:l,v:1,5','ockenheim': 'I 30:l,z13,z28:1,6','morocco': 'I 30:l,z13,z28:e,n,z15','ii 30:l,z28:z6': 'II 30:l,z28:z6','grandhaven': 'I 30:r:1,2','gege': 'I 30:r:1,5','quincy': 'I 30:r:1,6','matopeni': 'I 30:y:1,2','bietri': 'I 30:y:1,5','steinplatz': 'I 30:y:1,6','baguirmi': 'I 30:y:e,n,x','nijmegen': 'I 30:y:e,n,z15','hohentwiel': 'I 30:z:e,n,x,z15','stoneferry': 'I 30:z4,z23:–','bodjonegoro': 'I 30:z4,z24:–','ii 30:z6:1,6': 'II 30:z6:1,6','sada': 'I 30:z10:1,2','senneville': 'I 30:z10:1,5','kumasi': 'I 30:z10:e,n,z15','ii 30:z10:e,n,x,z15': 'II 30:z10:e,n,x,z15','aragua': 'I 30:z29:–','kokoli': 'I 30:z35:1,6','wuiti': 'I 30:z35:e,n,z15','ago': 'I 30:z38:–','ii 30:z39:1,7': 'II 30:z39:1,7','umhlatazana': 'I 35:a:e,n,z15','tchad': 'I 35:b:–','keurmassar': 'I 35:c:1,2','gouloumbo': 'I 35:c:1,5','yolo': 'I 35:c:[e,n,z15]','ii 35:d:1,5': 'II 35:d:1,5','dembe': 'I 35:d:l,w:[z58]','gassi': 'I 35:e,h:z6','adelaide': 'I 35:f,g:–:[z27]','ealing': 'I 35:g,m,s:–','ii 35:g,m,s,t:–': 'II 35:g,m,s,t:–','ebrie': 'I 35:g,m,t:–','anecho': 'I 35:g,s,t:–','ii 35:g,t:1,5': 'II 35:g,t:1,5','ii 35:g,t:z42': 'II 35:g,t:z42','agodi': 'I 35:g,t:–','iiia 35:g,z51:–': 'IIIa 35:g,z51:–','ii 35:g,z62:e,n,x': 'II 35:g,z62:e,n,x','monschaui': 'I 35:m,t:–','ii 35:m,t:–': 'II 35:m,t:–','iiib 35:i:e,n,x,z15': 'IIIb 35:i:e,n,x,z15','gambia': 'I 35:i:e,n,z15','bandia': 'I 35:i:l,w','iiib 35:i:z': 'IIIb 35:i:z','evry': 'I 35:i:z6','iiib 35:i:z35': 'IIIb 35:i:z35','iiib 35:i:z53': 'IIIb 35:i:z53','penzing': 'I 35:k:e,n,z15','iiib 35:k:e,n,x,z15': 'IIIb 35:k:e,n,x,z15','iiib 35:k:z': 'IIIb 35:k:z','iiib 35:k:z35': 'IIIb 35:(k):z35','iiib 35:k:z53:z50': 'IIIb 35:k:z53:[z50]','iiib 35:l,v:1,5,7': 'IIIb 35:l,v:1,5,7','iiib 35:l,v:e,n,x,z15:z50': 'IIIb 35:l,v:e,n,x,z15:[z50]','iiib 35:l,v:z35:z67': 'IIIb 35:l,v:z35:[z67]','ii 35:l,z28:–': 'II 35:l,z28:–','iiib 35:r:e,n,x,z15': 'IIIb 35:r:e,n,x,z15','massakory': 'I 35:r:l,w','iiib 35:r:z': 'IIIb 35:r:z','iiib 35:r:z35': 'IIIb 35:r:z35','iiib 35:r:z61': 'IIIb 35:r:z61','baltimore': 'I 35:y:l,w','alachua': 'I 35:z4,z23:–:[z37],[z45]','iiia 35:z4,z23:–': 'IIIa 35:z4,z23:–','westphalia': 'I 35:z4,z24:–','iiia 35:z4,z24:–': 'IIIa 35:z4,z24:–','iiia 35:z4,z32:–': 'IIIa 35:z4,z32:–','camberene': 'I 35:z10:1,5','enschede': 'I 35:z10:l,w','ligna': 'I 35:z10:z6','iiib 35:z10:z35': 'IIIb 35:z10:z35','ii 35:z29:e,n,x': 'II 35:z29:e,n,x','widemarsh': 'I 35:z29:–','iiia 35:z29:–': 'IIIa 35:z29:–','iiia 35:z36:–': 'IIIa 35:z36:–','haga': 'I 35:z38:–','iiib 35:z52:1,5,7': 'IIIb 35:z52:1,5,7','iiib 35:z52:e,n,x,z15': 'IIIb 35:z52:e,n,x,z15','iiib 35:z52:z': 'IIIb 35:z52:z','iiib 35:z52:z35': 'IIIb 35:z52:z35','oran': 'I 38:a:e,n,z15','ii 38:b:1,2': 'II 38:b:1,2','rittersbach': 'I 38:b:e,n,z15','sheffield': 'I 38:c:1,5','kidderminster': 'I 38:c:1,6:[z58]','willamette': 'I 38:d:1,5','ii 38:d:1,5': 'II 38:d:[1,5]','ii 38:d:z39': 'II 38:d:z39','thiaroye': 'I 38:e,h:1,2','kasenyi': 'I 38:e,h:1,5','korovi': 'I 38:g,m,[s]:–','ii 38:g,t:–': 'II 38:g,t:–','iiia 38:g,z51:–': 'IIIa 38:g,z51:–','iv 38:g,z51:–': 'IV 38:g,z51:–','rothenburgsort': 'I 38:m,t:–','mgulani': 'I 38:i:1,2','lansing': 'I 38:i:1,5','iiib 38:i:z': 'IIIb 38:i:z','iiib 38:i:z53': 'IIIb 38:i:z53','echa': 'I 38:k:1,2','mango': 'I 38:k:1,5','inverness': 'I 38:k:1,6','njala': 'I 38:k:e,n,x','iiib 38:k:e,n,x,z15': 'IIIb 38:k:e,n,x,z15','iiib 38:k:z': 'IIIb 38:k:z','iiib 38:k:z53': 'IIIb 38:k:z53','iiib 38:k:1,5,7': 'IIIb 38:(k):1,5,7','iiib 38:k:z35:z56': 'IIIb 38:(k):z35:[z56]','iiib 38:k:–': 'IIIb 38:(k):–','iiib 38:k:z55': 'IIIb 38:(k):z55','alger': 'I 38:l,v:1,2','kimberley': 'I 38:l,v:1,5','taylor': 'I 38:l,v:e,n,z15','roan': 'I 38:l,v:e,n,x','iiib 38:l,v:z': 'IIIb 38:l,v:z','iiib 38:l,v:z35': 'IIIb 38:l,v:z35','iiib 38:l,v:z53:z54': 'IIIb 38:l,v:[z53]:[z54]','lindi': 'I 38:r:1,5','iiib 38:r:1,5,7': 'IIIb 38:r:1,5,7','emmastad': 'I 38:r:1,6','iiib 38:r:e,n,x,z15': 'IIIb 38:r:e,n,x,z15','iiib 38:r:z:z57': 'IIIb 38:r:z:[z57]','iiib 38:r:z35': 'IIIb 38:r:z35','freetown': 'I 38:y:1,5','colombo': 'I 38:y:1,6','perth': 'I 38:y:e,n,x','stachus': 'I 38:z:–','yoff': 'I 38:z4,z23:1,2','iiia 38:z4,z23:–': 'IIIa 38:z4,z23:–','iv 38:z4,z23:–': 'IV 38:z4,z23:–','bangkok': 'I 38:z4,z24:–','neunkirchen': 'I 38:z10:[1,5]','iiib 38:z10:z': 'IIIb 38:z10:z','iiib 38:z10:z53': 'IIIb 38:z10:z53','carpentras': 'I 38:z35:e,n,z15','klouto': 'I 38:z38:–','iiib 38:z52:z35': 'IIIb 38:z52:z35','iiib 38:z52:z53': 'IIIb 38:z52:z53','iiib 38:z53:–:z47,z50,z76': 'IIIb 38:z53:–:[z47],[z50],[z76]','iiib 38:z61:z53': 'IIIb 38:z61:[z53]','ii 39:a:z39': 'II 39:a:z39','wandsworth': 'I 39:b:1,2','abidjan': 'I 39:b:l,w','ii 39:c:e,n,x': 'II 39:c:e,n,x','logone': 'I 39:d:1,5','bruebach': 'I 39:e,h:1,2','mara': 'I 39:e,h:1,5','ii 39:e,n,x:1,7': 'II 39:e,n,x:1,7','dietrichsdorf': 'I 39:m,t:–','ii 39:g,m,t:e,n,x': 'II 39:[g],m,t:[e,n,x]','hofit': 'I 39:i:1,5','cumberland': 'I 39:i:e,n,x','alma': 'I 39:i:e,n,z15','champaign': 'I 39:k:1,5:[z48]','newjersey': 'I 39:k:e,n,x','ii 39:l,v:1,5': 'II 39:l,v:1,5','kokomlemle': 'I 39:l,v:e,n,x','oerlikon': 'I 39:l,v:e,n,z15','ii 39:l,z28:e,n,x': 'II 39:l,z28:e,n,x','ii 39:l,z28:z39': 'II 39:l,z28:z39','frimley': 'I 39:r:1,5','anfo': 'I 39:y:1,2','windermere': 'I 39:y:1,5','delan': 'I 39:y:e,n,z15','namur': 'I 39:z4,z23:–','hegau': 'I 39:z10:–','ii 39:z10:z6': 'II 39:z10:z6','landeskrone': 'I 39:–:1,6','ii 39:–:1,7': 'II 39:–:1,7','shikmonah': 'I 40:a:1,5','salinas': 'I 40:a:1,7','greiz': 'I 40:a:z6','ii 40:a:z39': 'II 40:a:z39','riogrande': 'I 40:b:1,5','saugus': 'I 40:b:1,7','johannesburg': 'I [1],40:b:e,n,x','duval': 'I [1],40:b:e,n,z15','benguella': 'I 40:b:z6','ii 40:b:–': 'II 40:b:–','ii 1,40:c:e,n,x,z15': 'II [1],40:c:e,n,x,z15','ii 40:c:z6': 'II 40:c:z6','ii 1,40:c:z39': 'II [1],40:c:z39','driffield': 'I [1],40:d:1,5','ii 40:d:–': 'II 40:d:–','tilene': 'I [1],40:e,h:1,2','ii 1,40:e,n,x:1,5,7': 'II [1],40:e,n,x:1,[5],7','ii 1,40:e,n,x,z15:1,6': 'II [1],40:e,n,x,z15:1,6','bijlmer': 'I [1],40:g,m:–','athens': 'I [1],40:g,m,s:e,n,x','ii 1,40:g,m,s,t:e,n,x': 'II [1],40:g,[m],[s],[t]:e,n,x','ii 1,40:g,m,s,t:1,5': 'II [1],40:g,[m],[s],t:[1,5]','ii 1,40:g,t:e,n,x,z15': 'II [1],40:g,t:e,n,x,z15','ii 40:g,t:z39': 'II 40:g,t:z39','iv 1,40:g,t:–': 'IV [1],40:g,t:–','ii 1,40:g,m,s,t:z42': 'II [1],40:g,[m],[s],t:z42','iiia 40:g,z51:–': 'IIIa 40:g,z51:–','iiib 40:g,z51:e,n,x,z15': 'IIIb 40:g,z51:e,n,x,z15','iv 1,40:g,z51:–': 'IV [1],40:g,z51:–','ii 40:m,t:e,n,x': 'II 40:m,t:e,n,x','ii 40:m,t:z39': 'II 40:m,t:z39','ii 1,40:m,t:z42': 'II [1],40:m,t:z42','iv 40:m,t:–': 'IV 40:m,t:–','iiib 40:i:1,5,7': 'IIIb 40:i:1,5,7','goulfey': 'I [1],40:k:1,5','allandale': 'I [1],40:k:1,6','hann': 'I 40:k:e,n,x','ii 1,40:k:e,n,x,z15': 'II [1],40:k:e,n,x,z15','iiib 40:k:z:z57': 'IIIb 40:k:z:z57','ii 40:k:z6': 'II 40:k:z6','iiib 40:k:z53': 'IIIb 40:k:z53','millesi': 'I [1],40:l,v:1,2','canary': 'I 40:l,v:1,6','ii 40:l,v:e,n,x': 'II 40:l,v:e,n,x','iiib 40:l,v:z': 'IIIb 40:l,v:z','iiib 1,40:l,v:z53': 'IIIb [1],40:l,v:z53','overchurch': 'I [1],40:l,w:[1,2]','ii 40:l,z28:e,n,x': 'II 40:l,z28:e,n,x','tiko': 'I [1],40:l,z13,z28:1,2','bukavu': 'I [1],40:l,z28:1,5','ii 1,40:l,z28:1,5:z42': 'II [1],40:l,z28:1,5:z42','santhiaba': 'I 40:l,z28:1,6','ii 1,40:l,z28:z39': 'II [1],40:l,z28:z39','iiib 40:r:z53': 'IIIb 40:r:z53','odienne': 'I 40:y:1,5','ii 1,40:z:1,5': 'II [1],40:z:1,5','casamance': 'I 40:z:e,n,x','nowawes': 'I 40:z:z6','ii 1,40:z:z6': 'II [1],40:z:z6','ii 1,40:z:z39': 'II [1],40:z:z39','ii 40:z:z42': 'II 40:z:z42','iiia 40:z4,z23:–': 'IIIa 40:z4,z23:–','iv 1,40:z4,z23:–': 'IV [1],40:z4,z23:–','ii 40:z4,z24:z39': 'II 40:z4,z24:z39','iiia 40:z4,z24:–': 'IIIa 40:z4,z24:–','iv 1,40:z4,z24:–': 'IV [1],40:z4,z24:–','iiia 40:z4,z32:–': 'IIIa 40:z4,z32:–','iv 1,40:z4,z32:–': 'IV [1],40:z4,z32:–','ii 1,40:z6:1,5': 'II [1],40:z6:1,5','trotha': 'I 40:z10:z6','ii 40:z10:e,n,x': 'II 40:z10:e,n,x','iiib 40:z10:z35': 'IIIb 40:z10:z35','omifisan': 'I [1],40:z29:–','iiia 40:z29:–': 'IIIa 40:z29:–','ii 1,40:z35:e,n,x,z15': 'II [1],40:z35:e,n,x,z15','yekepa': 'I [1],40:z35:e,n,z15','v 1,40:z35:–': 'V [1],40:z35:–','iiia 40:z36:–': 'IIIa 40:z36:–','ii 1,40:z39:1,5:z42': 'II [1],40:z39:1,5:z42','ii 1,40:z39:1,6': 'II [1],40:z39:1,6','iiib 40:z39:1,6': 'IIIb 40:z39:1,6','ii 40:z39:1,7': 'II 40:z39:1,7','karamoja': 'I [1],40:z41:1,2','ii 1,40:z42:1,6': 'II [1],40:z42:1,6','ii 1,40:z42:1,5,7': 'II [1],40:[z42]:1,(5),7','ii 1,40:z81:z6': 'II [1],40:z81:z6','v 1,40:z81:–': 'V [1],40:z81:–','burundi': 'I 41:a:–','ii 41:b:1,5': 'II 41:b:1,5','vaugirard': 'I 41:b:1,6','vi 41:b:1,7': 'VI 41:b:1,7','vietnam': 'I 41:b:z6','sica': 'I 41:b:e,n,z15','lonestar': 'I 41:c:–','iiib 41:c:e,n,x,z15': 'IIIb 41:c:e,n,x,z15','ii 41:c:z6': 'II 41:c:z6','egusi': 'I 41:d:1,5','ii 41:d:z6': 'II 41:d:z6','ii 41:g,m,s,t:z6': 'II 41:g,m,s,t:z6','ii 41:g,t:–': 'II 41:g,t:–','iiia 41:g,z51:–': 'IIIa 41:g,z51:–','leatherhead': 'I 41:m,t:1,6','samaru': 'I 41:i:1,5','verona': 'I 41:i:1,6','ferlo': 'I 41:k:1,6','ii 41:k:1,6': 'II 41:k:1,6','ii 41:k:z6': 'II 41:k:z6','iiib 41:k:z35': 'IIIb 41:(k):z35','ii 41:l,z13,z28:e,n,x,z15': 'II 41:l,z13,z28:e,n,x,z15','lubumbashi': 'I 41:r:1,5','konongo': 'I 41:r:1,7','bruxelles': 'I 41:y:–','ii 41:z:1,5': 'II 41:z:1,5','sally': 'I 41:z:1,6','bofflens': 'I 41:z4,z23:1,7','waycross': 'I 41:z4,z23:[e,n,z15]','iiia 41:z4,z23:–': 'IIIa 41:z4,z23:–','iv 41:z4,z23:–': 'IV 41:z4,z23:–','iiia 41:z4,z23,z32:–': 'IIIa 41:z4,z23,z32:–','ipswich': 'I 41:z4,z24:1,5','iiia 41:z4,z24:–': 'IIIa 41:z4,z24:–','iiia 41:z4,z32:–': 'IIIa 41:z4,z32:–','ii 41:z10:1,2': 'II 41:z10:1,2','leipzig': 'I 41:z10:1,5','landala': 'I 41:z10:1,6','inpraw': 'I 41:z10:e,n,x','ii 41:z10:e,n,x,z15': 'II 41:z10:e,n,x,z15','ii 41:z10:z6': 'II 41:z10:z6','lodz': 'I 41:z29:–','iiia 41:z29:–': 'IIIa 41:z29:–','iv 41:z29:–': 'IV 41:z29:–','ahoutoue': 'I 41:z35:1,6','iiia 41:z36:–': 'IIIa 41:z36:–','iv 41:z36:–': 'IV 41:z36:–','offa': 'I 41:z38:–','iv 41:z52:–': 'IV 41:z52:–','ii 41:–:1,6': 'II 41:–:1,6','faji': 'I [1],42:a:e,n,z15','ii 42:b:1,5': 'II 42:b:1,5','orbe': 'I 42:b:1,6','ii 42:b:e,n,x,z15': 'II 42:b:e,n,x,z15','tomegbe': 'I [1],42:b:e,n,z15','frederiksberg': 'I [1],42:b:l,w','egusitoo': 'I [1],42:b:z6','ii 42:b:z6': 'II 42:b:z6','antwerpen': 'I [1],42:c:e,n,z15','kampala': 'I [1],42:c:z6','gostenhof': 'I [1],42:d:z6','ii 42:d:z6': 'II 42:d:z6','ii 42:e,n,x:1,6': 'II 42:e,n,x:1,6','ii 42:g,t:–': 'II 42:g,t:–','maricopa': 'I [1],42:g,z51:1,5','iiia 42:g,z51:–': 'IIIa 42:g,z51:–','iv 1,42:g,z51:–': 'IV [1],42:g,z51:–','ii 42:m,t:e,n,x,z15': 'II 42:m,t:[e,n,x,z15]','waral': 'I [1],42:m,t:–','kaneshie': 'I [1],42:i:l,w','borromea': 'I 42:i:1,6','middlesbrough': 'I [1],42:i:z6','haferbreite': 'I 42:k:1,6','iiib 42:k:e,n,x,z15': 'IIIb 42:k:e,n,x,z15','iiib 42:k:z': 'IIIb 42:k:z','gwale': 'I [1],42:k:z6','iiib 42:k:z35': 'IIIb 42:(k):z35','iiib 42:l,v:1,5,7:z76': 'IIIb 42:l,v:1,5,7:[z76]','portedeslilas': 'I [1],42:l,v:1,6,7','ii 42:l,v:e,n,x,z15': 'II 42:l,v:e,n,x,z15','iiib 42:l,v:e,n,x,z15': 'IIIb 42:l,v:e,n,x,z15','coogee': 'I 42:l,v:e,n,z15','iiib 42:l,v:z': 'IIIb 42:l,v:z','iiib 42:l,v:z53': 'IIIb 42:l,v:z53','ii 1,42:l,w:e,n,x': 'II [1],42:l,w:e,n,x','parakou': 'I [1],42:l,w:z35','ii 1,42:l,z13,z28:z6': 'II [1],42:l,[z13],z28:[z6]','sipane': 'I [1],42:r:e,n,z15','brive': 'I [1],42:r:l,w','iiib 42:r:z': 'IIIb 42:r:z','iiib 42:r:z53': 'IIIb 42:r:z53','ii 42:r:–': 'II 42:r:–','iiib 42:r:–:z50': 'IIIb 42:r:–:[z50]','spalentor': 'I [1],42:y:e,n,z15','harvestehude': 'I [1],42:y:z6','ii 42:z:1,5': 'II 42:z:1,5','ursenbach': 'I [1],42:z:1,6','ii 42:z:e,n,x,z15': 'II 42:z:e,n,x,z15','melbourne': 'I 42:z:e,n,z15','ii 42:z:z6': 'II 42:z:z6','gera': 'I [1],42:z4,z23:1,6:1,6','broc': 'I 42:z4,z23:e,n,z15','iiia 42:z4,z23:–': 'IIIa 42:z4,z23:–','toricada': 'I [1],42:z4,z24:–','iiia 42:z4,z24:–': 'IIIa 42:z4,z24:–','iv 1,42:z4,z24:–': 'IV [1],42:z4,z24:–','ii 42:z6:1,6': 'II 42:z6:1,6','ii 42:z10:1,2': 'II 42:z10:1,2','ii 42:z10:e,n,x,z15': 'II 42:z10:e,n,x,z15','iiib 42:z10:e,n,x,z15:z60': 'IIIb 42:z10:e,n,x,z15:[z60]','iiib 42:z10:z': 'IIIb 42:z10:z','loenga': 'I [1],42:z10:z6','ii 42:z10:z6': 'II 42:z10:z6','iiib 42:z10:z35': 'IIIb 42:z10:z35','iiib 42:z10:z67': 'IIIb 42:z10:z67','djama': 'I [1],42:z29:[1,5]','ii 42:z29:–': 'II 42:z29:–','kahla': 'I [1],42:z35:1,6','hennekamp': 'I 42:z35:e,n,z15','tema': 'I [1],42:z35:z6','weslaco': 'I 42:z36:–','iv 42:z36:–': 'IV 42:z36:–','vogan': 'I [1],42:z38:z6','taset': 'I [1],42:z41:–','iiib 42:z52:z': 'IIIb 42:z52:z','iiib 42:z57:1,5': 'IIIb 42:z57:1,5','graz': 'I 43:a:1,2','berkeley': 'I 43:a:1,5','ii 43:a:1,5': 'II 43:a:1,5','ii 43:a:z6': 'II 43:a:z6','niederoderwitz': 'I 43:b:–','ede': 'I 43:b:e,n,z15','ii 43:b:z42': 'II 43:b:z42','montreal': 'I 43:c:1,5','orleans': 'I 43:d:1,5','ii 43:d:e,n,x,z15': 'II 43:d:e,n,x,z15','ii 43:d:z39': 'II 43:d:z39','ii 43:d:z42': 'II 43:d:z42','ii 43:e,n,x,z15:1,5,7': 'II 43:e,n,x,z15:1,(5),7','ii 43:e,n,x,z15:1,6': 'II 43:e,n,x,z15:1,6','milwaukee': 'I 43:f,g,[t]:–','ii 43:g,m,s,t:z42': 'II 43:g,m,[s],t:[z42]','ii 43:g,t:1,5': 'II 43:g,t:[1,5]','iiib 43:g,t:–': 'IIIb 43:g,t:–','iiia 43:g,z51:–': 'IIIa 43:g,z51:–','iv 43:g,z51:–': 'IV 43:g,z51:–','ii 43:g,z62:e,n,x': 'II 43:g,z62:e,n,x','mbao': 'I 43:i:1,2','voulte': 'I 43:i:e,n,x','thetford': 'I 43:k:1,2','ahuza': 'I 43:k:1,5','iiib 43:k:z': 'IIIb 43:k:z','iiib 43:l,v:z53:z56': 'IIIb 43:l,v:z53:[z56]','epalinges': 'I 43:l,w:[z44]','sudan': 'I 43:l,z13:–','ii 43:l,z13,z28:1,5': 'II 43:l,z13,z28:1,5','ii 43:m,t:–': 'II 43:m,t:–','iiib 43:r:e,n,x,z15': 'IIIb 43:r:e,n,x,z15','iiib 43:r:z': 'IIIb 43:r:z','iiib 43:r:z53': 'IIIb 43:r:z53','farcha': 'I 43:y:1,2','kingabwa': 'I 43:y:1,5','ogbete': 'I 43:z:1,5','ii 43:z:1,5': 'II 43:z:1,5','arusha': 'I 43:z:e,n,z15','farmingdale': 'I 43:z4,z23:[1,2]','ii 43:z4,z23:–': 'II 43:z4,z23:–','iiia 43:z4,z23:–': 'IIIa 43:z4,z23:–','iv 43:z4,z23:–': 'IV 43:z4,z23:–','iiia 43:z4,z24:–': 'IIIa 43:z4,z24:–','iv 43:z4,z24:–': 'IV 43:z4,z24:–','iv 43:z4,z32:–': 'IV 43:z4,z32:–','adana': 'I 43:z10:1,5','ii 43:z29:e,n,x': 'II 43:z29:e,n,x','ii 43:z29:z42': 'II 43:z29:z42','makiling': 'I 43:z29:–','iiia 43:z29:–': 'IIIa 43:z29:–','iv 43:z29:–': 'IV 43:z29:–','ahepe': 'I 43:z35:1,6','iiia 43:z36:–': 'IIIa 43:z36:–','iv 43:z36,z38:–': 'IV 43:z36,z38:–','irigny': 'I 43:z38:–','darfur': 'I 43:z41:1,2','ii 43:z42:1,5,7': 'II 43:z42:[1,5,7]','iiib 43:z52:z53': 'IIIb 43:z52:z53','iv 44:a:–': 'IV 44:a:–','niakhar': 'I 44:a:1,5','tiergarten': 'I 44:a:e,n,x','niarembe': 'I 44:a:l,w','shahalam': 'I 44:b:1,6','elbeuf': 'I 44:b:e,n,x','sedgwick': 'I 44:b:e,n,z15','madigan': 'I 44:c:1,5','bergerac': 'I 44:c:e,n,x','quebec': 'I 44:c:e,n,z15','bobo': 'I 44:d:1,5','kermel': 'I 44:d:e,n,x','fischerstrasse': 'I 44:d:e,n,z15','palamaner': 'I [1],44:d:z35','ii 1,44:e,n,x:1,6': 'II [1],44:e,n,x:1,6','vleuten': 'I 44:f,g:–','gamaba': 'I [1],44:g,m,[s]:[1,6]','splott': 'I 44:g,s,t:[1,7]','ii 44:g,t:z42': 'II 44:g,t:z42','iiib 1,44:g,t:1,5:z42': 'IIIb [1],44:g,t:1,5:z42','carswell': 'I 44:g,z51:–','iv 44:g,z51:–': 'IV 44:g,z51:–','muguga': 'I 44:m,t:–','ii 1,44:m,t:z42': 'II [1],44:m,t:z42','maritzburg': 'I [1],44:i:e,n,z15','lawra': 'I 44:k:e,n,z15','malika': 'I 44:l,z28:1,5','albertbanjul': 'I 44:r:1,5','brefet': 'I 44:r:e,n,z15','v 44:r:–': 'V 44:r:–','brackenridge': 'I 44:z:1,5','uhlenhorst': 'I 44:z:l,w','bolama': 'I 44:z:e,n,x','goodmayes': 'I 44:z:e,n,z15','kua': 'I 44:z4,z23:–','ploufragan': 'I [1],44:z4,z23:e,n,z15','ii 44:z4,z23:–': 'II 44:z4,z23:–','iiia 44:z4,z23:–': 'IIIa 44:z4,z23:–','iv 44:z4,z23:–': 'IV 44:z4,z23:–','iiia 44:z4,z23,z32:–': 'IIIa 44:z4,z23,z32:–','christiansborg': 'I 44:z4,z24:–','iiia 44:z4,z24:–': 'IIIa 44:z4,z24:–','iv 44:z4,z24:–': 'IV 44:z4,z24:–','iiia 44:z4,z32:–': 'IIIa 44:z4,z32:–','iv 1,44:z4,z32:–': 'IV [1],44:z4,z32:–','guinea': 'I [1],44:z10:1,7','llobregat': 'I [1],44:z10:e,n,x','ii 44:z29:e,n,x:z42': 'II 44:z29:e,n,x:z42','zinder': 'I 44:z29:–','iv 44:z29:–': 'IV 44:z29:–','iv 44:z36,z38:–': 'IV 44:z36,[z38]:–','koketime': 'I 44:z38:–','ii 1,44:z39:e,n,x,z15': 'II [1],44:z39:e,n,x,z15','v 44:z39:–': 'V 44:z39:–','vi 45:a:e,n,x': 'VI 45:a:e,n,x','meekatharra': 'I 45:a:e,n,z15','ii 45:a:z10': 'II 45:a:z10','riverside': 'I 45:b:1,5','leinster': 'I 45:b:1,6','fomeco': 'I 45:b:e,n,z15','deversoir': 'I 45:c:e,n,x','dugbe': 'I 45:d:1,6','karachi': 'I 45:d:e,n,x','warmsen': 'I 45:d:e,n,z15','suelldorf': 'I 45:f,g:–','tornow': 'I 45:g,m,[s],[t]:–','ii 45:g,m,s,t:1,5': 'II 45:g,m,s,t:1,5','ii 45:g,m,s,t:e,n,x': 'II 45:g,m,s,t:e,n,x','ii 45:g,m,t:e,n,x,z15': 'II 45:g,m,t:e,n,x,z15','binningen': 'I 45:g,s,t:–','iiia 45:g,z51:–': 'IIIa 45:g,z51:–','iv 45:g,z51:–': 'IV 45:g,z51:–','ii 45:m,t:1,5': 'II 45:m,t:1,5','apapa': 'I 45:m,t:–','verviers': 'I 45:k:1,5','casablanca': 'I 45:k:1,7','cairns': 'I 45:k:e,n,z15','imo': 'I 45:l,v:[e,n,z15]','kofandoka': 'I 45:r:e,n,z15','ii 45:z:1,5': 'II 45:z:1,5','yopougon': 'I 45:z:e,n,z15','ii 45:z:z39': 'II 45:z:z39','iiia 45:z4,z23:–': 'IIIa 45:z4,z23:–','iv 45:z4,z23:–': 'IV 45:z4,z23:–','transvaal': 'I 45:z4,z24:–','iiia 45:z4,z24:–': 'IIIa 45:z4,z24:–','iiia 45:z4,z32:–': 'IIIa 45:z4,z32:–','aprad': 'I 45:z10:–','jodhpur': 'I 45:z29:–:[z45]','ii 45:z29:1,5': 'II 45:z29:1,5','ii 45:z29:e,n,x': 'II 45:z29:e,n,x','ii 45:z29:z42': 'II 45:z29:z42','iiia 45:z29:–': 'IIIa 45:z29:–','lattenkamp': 'I 45:z35:1,5','balcones': 'I 45:z36:–','iiia 45:z36:–': 'IIIa 45:z36:–','iv 45:z36,z38:–': 'IV 45:z36,z38:–','ii 47:a:1,5': 'II 47:a:1,5','ii 47:a:e,n,x,z15': 'II 47:a:e,n,x,z15','wenatchee': 'I 47:b:1,2','ii 47:b:1,5': 'II 47:b:1,5','ii 47:b:e,n,x,z15': 'II 47:b:e,n,x,z15','sya': 'I 47:b:z6','ii 47:b:z6': 'II 47:b:z6','iiib 47:c:1,5,7': 'IIIb 47:c:1,5,7','kodjovi': 'I 47:c:1,6:[z78]','iiib 47:c:e,n,x,z15:z57': 'IIIb 47:c:e,n,x,z15:[z57]','iiib 47:c:z': 'IIIb 47:c:z','braeswood': 'I 47:c:z6','iiib 47:c:z35': 'IIIb 47:c:z35','ii 47:d:1,5': 'II 47:d:1,5','stellingen': 'I 47:d:e,n,x:[z58]','ii 47:d:e,n,x,z15': 'II 47:d:e,n,x,z15','ii 47:d:z39': 'II 47:d:z39','ii 47:e,n,x,z15:1,6': 'II 47:e,n,x,z15:1,6','sljeme': 'I [1],47:f,g:–','luke': 'I [1],47:g,m:–','carshalton': 'I 47:g,m,s:–','ii 47:g,t:e,n,x': 'II 47:[g,t]:e,n,x','iiia 47:g,z51:–': 'IIIa 47:g,z51:–','mesbit': 'I 47:m,t:e,n,z15','iiib 47:i:e,n,x,z15:z50': 'IIIb 47:i:e,n,x,z15:[z50]','bergen': 'I 47:i:e,n,z15','iiib 47:i:z': 'IIIb 47:i:z','iiib 47:i:z35': 'IIIb 47:i:z35','iiib 47:i:z53:z57,z84': 'IIIb 47:i:z53:[z57],[z84]','staoueli': 'I 47:k:1,2','bootle': 'I 47:k:1,5','iiib 47:k:1,5,7': 'IIIb 47:k:1,5,7','dahomey': 'I 47:k:1,6:[z58]','iiib 47:k:e,n,x,z15': 'IIIb 47:k:e,n,x,z15','lyon': 'I 47:k:e,n,z15','iiib 47:k:z': 'IIIb 47:k:z','iiib 47:k:z35': 'IIIb 47:k:z35','iiib 47:k:z53:z84': 'IIIb 47:k:z53:[z84]','iiib 47:l,v:1,5,7:z50': 'IIIb 47:l,v:1,[5],7:[z50]','drac': 'I 47:l,v:e,n,x','iiib 47:l,v:e,n,x,z15': 'IIIb 47:l,v:e,n,x,z15','iiib 47:l,v:z': 'IIIb 47:l,v:z','iiib 47:l,v:z35': 'IIIb 47:l,v:z35','iiib 47:l,v:z53': 'IIIb 47:l,v:z53','iiib 47:l,v:z57': 'IIIb 47:l,v:z57','iv 47:l,v:–': 'IV 47:l,v:–','teshie': 'I [1],47:l,z13,z28:e,n,z15','iiib 47:r:e,n,x,z15': 'IIIb 47:r:e,n,x,z15','dapango': 'I 47:r:1,2','iiib 47:r:1,5,7': 'IIIb 47:r:1,5,7','iiib 47:r:z': 'IIIb 47:r:z','iiib 47:r,i:z35': 'IIIb 47:r,[i]:z35','iiib 47:r:z53:z70,z74,z77,z90': 'IIIb 47:r:z53:[z70],[z74],[z77],[z90]','moualine': 'I 47:y:1,6','blitta': 'I 47:y:e,n,x','mountpleasant': 'I 47:z:1,5','kaolack': 'I 47:z:1,6','ii 47:z:e,n,x,z15': 'II 47:z:e,n,x,z15','ii 47:z:z6': 'II 47:z:z6','forges': 'I 47:z4,z23:1,2','tabligbo': 'I 47:z4,z23:e,n,z15','fehrbellin': 'I 47:z4,z23:1,6','bere': 'I 47:z4,z23:z6:[z45],[z58]','binche': 'I 47:z4,z23:l,w','iiia 47:z4,z23:–': 'IIIa 47:z4,z23:–','tamberma': 'I 47:z4,z24:–','ii 47:z6:1,6': 'II 47:z6:1,6','iiib 47:z10:1,5,7': 'IIIb 47:z10:1,5,7','namoda': 'I 47:z10:e,n,z15','iiib 47:z10:z': 'IIIb 47:z10:z','iiib 47:z10:z35': 'IIIb 47:z10:z35','ii 47:z29:e,n,x,z15': 'II 47:z29:e,n,x,z15','ekpoui': 'I 47:z29:–','iiia 47:z29:–': 'IIIa 47:z29:–','yombesali': 'I 47:z35:z6','bingerville': 'I 47:z35:e,n,z15','iv 47:z36:–': 'IV 47:z36:–','alexanderplatz': 'I 47:z38:–','quinhon': 'I 47:z44:–','iiib 47:z52:1,5:z54': 'IIIb 47:z52:1,5:z54','iiib 47:z52:1,5,7': 'IIIb 47:z52:1,5,7','iiib 47:z52:e,n,x,z15': 'IIIb 47:z52:e,n,x,z15','iiib 47:z52:z': 'IIIb 47:z52:z','iiib 47:z52:z35': 'IIIb 47:z52:z35','iiib 47:z53:–:z90': 'IIIb 47:z53:–:[z90]','hisingen': 'I 48:a:1,5,7','ii 48:a:z6': 'II 48:a:z6','ii 48:a:z39': 'II 48:a:z39','ii 48:b:z6': 'II 48:b:z6','ii 48:b:e,n,x,z15': 'II 48:b:e,n,x,z15','v 48:b:–': 'V 48:b:–','iiib 48:c:z': 'IIIb 48:c:z','ii 48:d:1,2': 'II 48:d:1,2','ii 48:d:z6': 'II 48:d:z6','buckeye': 'I 48:d:–:[z58]','fitzroy': 'I 48:e,h:1,5','ii 48:e,n,x,z15:z6': 'II 48:e,n,x,z15:z6','ii 48:g,m,t:–': 'II 48:g,m,t:–','iiia 48:g,z51:–': 'IIIa 48:g,z51:–','iv 48:g,z51:–': 'IV 48:g,z51:–','iiib 48:i:z:z72': 'IIIb 48:i:z:[z72]','iiib 48:i:z35:z57': 'IIIb 48:i:z35:[z57]','iiib 48:i:z53': 'IIIb 48:i:z53','iiib 48:i:z61': 'IIIb 48:i:z61','v 48:i:–': 'V 48:i:–','iiib 48:k:1,5,7': 'IIIb 48:k:1,5,(7)','ii 48:k:e,n,x,z15': 'II 48:k:e,n,x,z15','iiib 48:k:e,n,x,z15': 'IIIb 48:k:e,n,x,z15','dahlem': 'I 48:k:e,n,z15','iiib 48:k:z': 'IIIb 48:k:z','iiib 48:k:z35:z75': 'IIIb 48:k:z35:[z75]','ii 48:k:z39': 'II 48:k:z39','iiib 48:k:z53': 'IIIb 48:k:z53','iiib 48:k:z57': 'IIIb 48:k:z57','australia': 'I 48:l,v:1,5','iiib 48:l,v:1,5,7:z47,z50,z89': 'IIIb 48:l,v:1,5,(7):[z47],[z50],[z89]','iiib 48:l,v:z': 'IIIb 48:l,v:z','iiib 48:l,w:1,5,7:z50': 'IIIb 48:l,w:1,5,7:[z50]','iiib 48:r:e,n,x,z15': 'IIIb 48:r:e,n,x,z15','iiib 48:r:z': 'IIIb 48:r:z','toucra': 'I 48:z:1,5:[z58]','ii 48:z:1,5': 'II 48:z:1,5','iiib 48:z:1,5,7': 'IIIb 48:z:1,5,7','iiia 48:z4,z23:–': 'IIIa 48:z4,z23:–','iv 48:z4,z23:–': 'IV 48:z4,z23:–','iiia 48:z4,z23,z32:–': 'IIIa 48:z4,z23,z32:–','djakarta': 'I 48:z4,z24:–','iiia 48:z4,z24:–': 'IIIa 48:z4,z24:–','iiib 48:z4,z24:–': 'IIIb 48:z4,z24:–','iv 48:z4,z24:–': 'IV 48:z4,z24:–','iiia 48:z4,z32:–': 'IIIa 48:z4,z32:–','iv 48:z4,z32:–': 'IV 48:z4,z32:–','ii 48:z10:1,5': 'II 48:z10:[1,5]','vi 48:z10:1,5': 'VI 48:z10:1,5','ii 48:z10:1,6': 'II 48:z10:1,6','isaszeg': 'I 48:z10:e,n,x','iiib 48:z10:e,n,x,z15': 'IIIb 48:z10:e,n,x,z15','iiib 48:z10:z': 'IIIb 48:z10:z','ii 48:z29:–': 'II 48:z29:–','iiia 48:z29:–': 'IIIa 48:z29:–','iv 48:z29:–': 'IV 48:z29:–','iiib 48:z35:z52': 'IIIb 48:z35:z52','v 48:z35:–': 'V 48:z35:–','iiia 48:z36:–': 'IIIa 48:z36:–','iv 48:z36,z38:–': 'IV 48:z36,[z38]:–','ii 48:z39:z81': 'II 48:z39:z81','v 48:z39:–': 'V 48:z39:–','v 48:z41:–': 'V 48:z41:–','iiib 48:z52:e,n,x,z15': 'IIIb 48:z52:e,n,x,z15','iiib 48:z52:z': 'IIIb 48:z52:z','v 48:z65:–': 'V 48:z65:–','v 48:z81:–': 'V 48:z81:–','ii 48:z81:1,5,7': 'II 48:z81:1,5,7','iv 50:a:–': 'IV 50:a:–','rochdale': 'I 50:b:e,n,x','ii 50:b:z6': 'II 50:b:z6','iv 50:b:–': 'IV 50:b:–','hemingford': 'I 50:d:1,5:[z82]','iv 50:d:–': 'IV 50:d:–','ii 50:e,n,x:1,7': 'II 50:e,n,x:1,7','ii 50:g,m,s,t:1,5': 'II 50:g,[m],s,t:[1,5]','iiia 50:g,z51:–': 'IIIa 50:g,z51:–','iv 50:g,z51:–': 'IV 50:g,z51:–','ii 50:g,z62:e,n,x': 'II 50:g,z62:e,n,x','ii 50:m,t:z6:z42': 'II 50:m,t:z6:z42','iiib 50:i:1,5,7': 'IIIb 50:i:1,5,7','iiib 50:i:e,n,x,z15': 'IIIb 50:i:e,n,x,z15','iiib 50:i:z': 'IIIb 50:i:z','iiib 50:i:z53': 'IIIb 50:i:z53','iiib 50:k:1,5,7': 'IIIb 50:k:1,5,7','ii 50:k:e,n,x:z42': 'II 50:k:e,n,x:z42','iiib 50:k:e,n,x,z15': 'IIIb 50:k:e,n,x,z15','iiib 50:k:z:z50,z57,z68,z86': 'IIIb 50:k:z:[z50],[z57],[z68],[z86]','ii 50:k:z6': 'II 50:k:z6','iiib 50:k:z35': 'IIIb 50:k:z35','iiib 50:k:z53': 'IIIb 50:k:z53','fass': 'I 50:l,v:1,2','iiib 50:l,v:e,n,x,z15': 'IIIb 50:l,v:e,n,x,z15','iiib 50:l,v:z': 'IIIb 50:l,v:z','iiib 50:l,v:z35': 'IIIb 50:l,v:z35','iiib 50:l,v:z57': 'IIIb 50:l,v:z57','vi 50:l,v:z67': 'VI 50:l,v:z67','ii 50:l,w:e,n,x,z15:z42': 'II 50:l,w:e,n,x,z15:z42','ii 50:l,z28:z42': 'II 50:l,z28:z42','iiib 50:r:1,5,7': 'IIIb 50:r:1,5,(7)','iiib 50:r:e,n,x,z15': 'IIIb 50:r:e,n,x,z15','iiib 50:r:z:z67': 'IIIb 50:r:z:[z67]','iiib 50:r:z35:z58': 'IIIb 50:r:z35:[z58]','iiib 50:r:z53': 'IIIb 50:r:z53','dougi': 'I 50:y:1,6','ii 50:z:e,n,x': 'II 50:z:e,n,x','iiib 50:z:z52': 'IIIb 50:z:z52','iiia 50:z4,z23:–': 'IIIa 50:z4,z23:–','iv 50:z4,z23:–': 'IV 50:z4,z23:–','iiia 50:z4,z23,z32:–': 'IIIa 50:z4,z23,z32:–','iiia 50:z4,z24:–': 'IIIa 50:z4,z24:–','iv 50:z4,z24:–': 'IV 50:z4,z24:–','iiia 50:z4,z32:–': 'IIIa 50:z4,z32:–','iv 50:z4,z32:–': 'IV 50:z4,z32:–','iiib 50:z10:z:z56': 'IIIb 50:z10:z:[z56]','ii 50:z10:z6:z42': 'II 50:z10:z6:z42','iiib 50:z10:z53': 'IIIb 50:z10:z53','ivorycoast': 'I 50:z29:–','iiia 50:z29:–': 'IIIa 50:z29:–','iiia 50:z36:–': 'IIIa 50:z36:–','ii 50:z42:1,7': 'II 50:z42:1,7','iiib 50:z52:1,5,7': 'IIIb 50:z52:1,5,7','iiib 50:z52:z35': 'IIIb 50:z52:z35','iiib 50:z52:z53': 'IIIb 50:z52:z53','iv 51:a:–': 'IV 51:a:–','windsheim': 'I 51:a:1,2','tione': 'I 51:a:e,n,x','karaya': 'I 51:b:1,5','iv 51:b:–': 'IV 51:b:–','ii 51:c:–': 'II 51:c:–','gokul': 'I [1],51:d:1,5','meskin': 'I 51:e,h:1,2','ii 51:g,s,t:e,n,x': 'II 51:g,s,t:e,n,x','iiia 51:g,z51:–': 'IIIa 51:g,z51:–','djinten': 'I 51:m,t:–','elmdon': 'I 51:i:1,2','kabete': 'I 51:i:1,5','dan': 'I 51:k:e,n,z15','iiib 51:k:z35': 'IIIb 51:k:z35','harcourt': 'I 51:l,v:1,2','overschie': 'I 51:l,v:1,5','dadzie': 'I 51:l,v:e,n,x','iiib 51:l,v:z': 'IIIb 51:l,v:z','moundou': 'I 51:l,z28:1,5','ii 51:l,z28:z6': 'II 51:l,z28:z6','ii 51:l,z28:z39': 'II 51:l,z28:z39','mali': 'I 51:r:e,n,x','lutetia': 'I 51:r,i:l,z13,z28','rosslyn': 'I 51:y:1,2','antsalova': 'I 51:z:1,5','treforest': 'I [1],51:z:1,6','lechler': 'I 51:z:e,n,z15','iiia 51:z4,z23:–': 'IIIa 51:z4,z23:–','iv 51:z4,z23:–': 'IV 51:z4,z23:–','iiia 51:z4,z24:–': 'IIIa 51:z4,z24:–','iiia 51:z4,z32:–': 'IIIa 51:z4,z32:–','bergues': 'I 51:z10:1,5','ii 51:z29:e,n,x,z15': 'II 51:z29:e,n,x,z15','iiia 51:z36:–': 'IIIa 51:z36:–','ii 51:–:1,7': 'II 51:–:1,7','uithof': 'I 52:a:1,5','ord': 'I 52:a:e,n,z15','molesey': 'I 52:b:1,5','flottbek': 'I 52:b:e,n,x','ii 52:c:k': 'II 52:c:k','utrecht': 'I 52:d:1,5','ii 52:d:e,n,x,z15': 'II 52:d:e,n,x,z15','ii 52:d:z39': 'II 52:d:z39','butare': 'I 52:e,h:1,6','derkle': 'I 52:e,h:1,7','saintemarie': 'I 52:g,t:–','ii 52:g,t:–': 'II 52:g,t:–','bordeaux': 'I 52:k:1,5','iiib 52:k:e,n,x,z15': 'IIIb 52:k:e,n,x,z15','iiib 52:k:z35': 'IIIb 52:k:z35','iiib 52:k:z53': 'IIIb 52:k:z53','iiib 52:l,v:z53': 'IIIb 52:l,v:z53','marsabit': 'I 52:l,w:1,5','ii 52:z:z39': 'II 52:z:z39','iiib 52:z:z52': 'IIIb 52:z:z52','ii 52:z39:1,5,7': 'II 52:z39:1,5,7','ii 52:z44:1,5,7': 'II 52:z44:1,5,7','ii 53:c:1,5:r1': 'II 53:c:1,5:[R1…]','ii 53:d:1,5': 'II 53:d:1,5','ii 1,53:d:z39': 'II [1],53:d:z39','ii 53:d:z42': 'II 53:d:z42','iiia 53:g,z51:–': 'IIIa 53:g,z51:–','iv 1,53:g,z51:–': 'IV [1],53:g,z51:–','iiib 53:i:z': 'IIIb 53:i:z','iiib 53:k:e,n,x,z15': 'IIIb 53:k:e,n,x,z15','iiib 53:k:z': 'IIIb 53:k:z','iiib 53:k:z35': 'IIIb 53:(k):z35','iiib 53:k:z53': 'IIIb 53:k:z53','iiib 53:l,v:e,n,x,z15': 'IIIb 53:l,v:e,n,x,z15','iiib 53:l,v:z': 'IIIb 53:l,v:z','iiib 53:l,v:z35': 'IIIb 53:l,v:z35','ii 53:l,z28:e,n,x': 'II 53:l,z28:e,n,x','ii 53:l,z28:z6': 'II 53:l,z28:z6','ii 53:l,z28:z39': 'II 53:l,z28:z39','iiib 53:r:z': 'IIIb 53:r:z','iiib 53:r:z35': 'IIIb 53:r:z35','iiib 53:r:z68': 'IIIb 53:r:z68','ii 53:z:1,5': 'II 53:z:1,5','iiib 53:z:1,5,7': 'IIIb 53:z:1,5,(7)','ii 53:z:z6': 'II 53:z:z6','iiia 53:z4,z23:–': 'IIIa 53:z4,z23:–','iv 53:z4,z23:–': 'IV 53:z4,z23:–','iiia 53:z4,z23,z32:–': 'IIIa 53:z4,z23,z32:–','ii 53:z4,z24:–': 'II 53:z4,z24:–','iiia 53:z4,z24:–': 'IIIa 53:z4,z24:–','iiib 53:z10:z': 'IIIb 53:z10:z','iiib 53:z10:z35': 'IIIb 53:z10:z35','iiia 53:z29:–': 'IIIa 53:z29:–','iv 1,53:z36,z38:–': 'IV [1],53:z36,z38:–','iiib 53:z52:z35': 'IIIb 53:z52:z35','iiib 53:z52:z53': 'IIIb 53:z52:z53','leda': 'I 53:–:1,6','tonev': 'I 21,54:b:e,n,x','winnipeg': 'I 54:e,h:1,5','rossleben': 'I 3,54:e,h:1,6','borreze': 'I 54:f,g,s:–','uccle': 'I 3,54:g,s,t:–','newholland': 'I 4,12,54:m,t:–','poeseldorf': 'I 8,[20],54:i:z6','ochsenwerder': 'I 6,7,54:k:1,5','czernyring': 'I 54:r:1,5','steinwerder': 'I 3,15,54:y:1,5','yerba': 'I 54:z4,z23:–','canton': 'I 54:z10:e,n,x','barry': 'I 54:z10:e,n,z15','mundubbera': 'I 54:z29:–','ii 55:k:z39': 'II 55:k:z39','ii 56:b:1,5': 'II 56:b:[1,5]','ii 56:d:–': 'II 56:d:–','ii 56:e,n,x:1,7': 'II 56:e,n,x:1,7','ii 56:l,v:z39': 'II 56:l,v:z39','ii 56:l,z28:–': 'II 56:l,z28:–','ii 56:z:z6': 'II 56:z:z6','iiia 56:z4,z23:–': 'IIIa 56:z4,z23:–','iiia 56:z4,z23,z32:–': 'IIIa 56:z4,z23,z32:–','ii 56:z10:e,n,x': 'II 56:z10:e,n,x','iiia 56:z29:–': 'IIIa 56:z29:–','antonio': 'I 57:a:z6','ii 57:a:z42': 'II 57:a:z42','maryland': 'I 57:b:1,7','batonrouge': 'I 57:b:e,n,z15','iiib 57:c:e,n,x,z15': 'IIIb 57:c:e,n,x,z15','iiib 57:c:z:z70,z90': 'IIIb 57:c:z:[z70],[z90]','ii 57:d:1,5': 'II 57:d:1,5','ii 57:g,m,s,t:z42': 'II 57:g,[m],s,t:z42','ii 57:g,t:–': 'II 57:g,t:–','iiib 57:i:e,n,x,z15': 'IIIb 57:i:e,n,x,z15','iiib 57:i:z': 'IIIb 57:i:z','iiib 57:k:e,n,x,z15': 'IIIb 57:k:e,n,x,z15','iv 57:z4,z23:–': 'IV 57:z4,z23:–','iiib 57:z10:z': 'IIIb 57:z10:z','ii 57:z29:z42': 'II 57:z29:z42','ii 57:z39:e,n,x,z15': 'II 57:z39:e,n,x,z15','ii 57:z42:1,6:z53': 'II 57:z42:1,6:z53','ii 58:a:z6': 'II 58:a:z6','ii 58:b:1,5': 'II 58:b:1,5','ii 58:c:z6': 'II 58:c:z6','ii 58:d:z6': 'II 58:d:z6','iiib 58:i:e,n,x,z15': 'IIIb 58:i:e,n,x,z15','iiib 58:i:z53': 'IIIb 58:i:z53','iiib 58:k:z': 'IIIb 58:k:z','iiib 58:l,v:e,n,x,z15': 'IIIb 58:l,v:e,n,x,z15','iiib 58:l,v:z35': 'IIIb 58:l,v:z35','ii 58:l,z13,z28:1,5': 'II 58:l,z13,z28:1,5','ii 58:l,z13,z28:z6': 'II 58:l,z13,z28:z6','iiib 58:r:e,n,x,z15': 'IIIb 58:r:e,n,x,z15','iiib 58:r:z': 'IIIb 58:r:z','iiib 58:r:z53:z47,z57,z70,z73': 'IIIb 58:r:z53:[z47],[z57],[z70],[z73]','ii 58:z6:1,6': 'II 58:z6:1,6','ii 58:z10:1,6': 'II 58:z10:1,6','iiib 58:z10:e,n,x,z15': 'IIIb 58:z10:e,n,x,z15','ii 58:z10:z6': 'II 58:z10:z6','iiib 58:z10:z53:z50': 'IIIb 58:z10:z53:[z50]','ii 58:z39:e,n,x,z15': 'II 58:z39:e,n,x,z15','iiib 58:z52:z': 'IIIb 58:z52:z','iiib 58:z52:z35': 'IIIb 58:z52:z35','iiib 59:c:e,n,x,z15': 'IIIb 59:c:e,n,x,z15','iiib 59:i:e,n,x,z15': 'IIIb 59:i:e,n,x,z15','iiib 59:i:z': 'IIIb 59:i:z','iiib 59:i:z35:z84': 'IIIb 59:i:z35:[z84]','iiib 59:k:e,n,x,z15': 'IIIb 59:(k):e,n,x,z15','ii 59:k:z65': 'II 59:k:z65','iiib 59:k:z': 'IIIb 59:(k):z','iiib 59:k:z35': 'IIIb 59:(k):z35','iiib 59:k:z53': 'IIIb 59:k:z53','iiib 59:l,v:z': 'IIIb 59:l,v:z','iiib 59:l,v:z53': 'IIIb 59:l,v:z53','iiib 59:r:z35': 'IIIb 59:r:z35','ii 1,59:z:z6': 'II [1],59:z:z6','iiia 59:z4,z23:–': 'IIIa 59:z4,z23:–','iiib 59:z10:z53': 'IIIb 59:z10:z53','iiib 59:z10:z57': 'IIIb 59:z10:z57','iiia 59:z29:–': 'IIIa 59:z29:–','iiia 59:z36:–': 'IIIa 59:z36:–','vi 59:z36:–': 'VI 59:z36:–','iiib 59:z52:z53': 'IIIb 59:z52:z53','ii 60:b:–:r1': 'II 60:b:–:[R1…]','ii 60:g,m,t:z6': 'II 60:g,m,t:z6','iiib 60:i:e,n,x,z15:z50': 'IIIb 60:i:[e,n,x,z15]:[z50]','iiib 60:i:z:z50': 'IIIb 60:i:[z]:[z50]','iiib 60:i:z35:z50': 'IIIb 60:i:[z35]:[z50]','iiib 60:k:z': 'IIIb 60:k:z','iiib 60:k:z35': 'IIIb 60:k:z35','iiib 60:k:z53': 'IIIb 60:(k):z53','iiib 60:l,v:z': 'IIIb 60:l,v:z','iiib 60:r:e,n,x,z15': 'IIIb 60:r:e,n,x,z15','iiib 60:r:z': 'IIIb 60:r:z','iiib 60:r:z35': 'IIIb 60:r:z35','iiib 60:r:z53': 'IIIb 60:r:z53','ii 60:z:e,n,x': 'II 60:z:e,n,x','iiib 60:z10:z': 'IIIb 60:z10:z','iiib 60:z10:z35': 'IIIb 60:z10:z35','ii 60:z10:z39': 'II 60:z10:z39','iiib 60:z10:z53': 'IIIb 60:z10:z53','ii 60:z29:e,n,x': 'II 60:z29:e,n,x','v 60:z41:–': 'V 60:z41:–','iiib 60:z52:1,5,7': 'IIIb 60:z52:1,5,[7]','iiib 60:z52:z': 'IIIb 60:z52:z','iiib 60:z52:z35': 'IIIb 60:z52:z35','iiib 60:z52:z53': 'IIIb 60:z52:z53','iiib 61:c:1,5,7': 'IIIb 61:c:1,5,(7)','iiib 61:c:z35': 'IIIb 61:c:z35','iiib 61:i:e,n,x,z15': 'IIIb 61:i:e,n,x,z15','iiib 61:i:z': 'IIIb 61:i:z','iiib 61:i:z35': 'IIIb 61:i:z35','iiib 61:i:z53': 'IIIb 61:i:z53','iiib 61:k:1,5,7': 'IIIb 61:k:1,5,(7)','iiib 61:k:z35': 'IIIb 61:k:z35','iiib 61:k:z53': 'IIIb 61:(k):z53','iiib 61:l,v:1,5,7:z57': 'IIIb 61:l,v:1,5,7:[z57]','iiib 61:l,v:z': 'IIIb 61:l,v:z','iiib 61:l,v:z35': 'IIIb 61:l,v:z35','iiib 61:r:1,5,7': 'IIIb 61:r:1,5,7','iiib 61:r:z': 'IIIb 61:r:z','iiib 61:r:z35': 'IIIb 61:r:z35','iiib 61:r:z53:z47,z50': 'IIIb 61:r:z53:[z47],[z50]','iiib 61:z10:z35': 'IIIb 61:z10:z35','v 61:z35:–': 'V 61:z35:–','iiib 61:z52:1,5,7': 'IIIb 61:z52:1,5,7','iiib 61:z52:z': 'IIIb 61:z52:z','iiib 61:z52:z35': 'IIIb 61:z52:z35','iiib 61:z52:z53': 'IIIb 61:z52:z53','iiia 62:g,z51:–': 'IIIa 62:g,z51:–','iiia 62:z4,z23:–': 'IIIa 62:z4,z23:–','iiia 62:z4,z32:–': 'IIIa 62:z4,z32:–','iiia 62:z29:–': 'IIIa 62:z29:–','iiia 62:z36:–': 'IIIa 62:z36:–','iiia 63:g,z51:–': 'IIIa 63:g,z51:–','iiib 63:k:z': 'IIIb 63:(k):z','iiia 63:z4,z23:–': 'IIIa 63:z4,z23:–','iiia 63:z4,z32:–': 'IIIa 63:z4,z32:–','iiia 63:z36:–': 'IIIa 63:z36:–','iiib 65:c:1,5,7': 'IIIb 65:c:1,5,7','iiib 65:c:z': 'IIIb 65:c:z','iiib 65:c:z53': 'IIIb 65:c:z53','ii 65:g,t:–': 'II 65:g,t:–','iiib 65:i:e,n,x,z15': 'IIIb 65:i:e,n,x,z15','iiib 65:i:z': 'IIIb 65:i:z','iiib 65:k:z': 'IIIb 65:(k):z','iiib 65:k:z35': 'IIIb 65:(k):z35','iiib 65:k:z53': 'IIIb 65:(k):z53','iiib 65:l,v:e,n,x,z15': 'IIIb 65:l,v:e,n,x,z15','iiib 65:l,v:z': 'IIIb 65:l,v:z','iiib 65:l,v:z35': 'IIIb 65:l,v:z35','iiib 65:l,v:z53': 'IIIb 65:l,v:z53','iiib 65:r:z35': 'IIIb 65:r:z35','iiib 65:z10:e,n,x,z15': 'IIIb 65:z10:e,n,x,z15','iiib 65:z10:z': 'IIIb 65:z10:z','iiib 65:z52:e,n,x,z15': 'IIIb 65:z52:e,n,x,z15','iiib 65:z52:z': 'IIIb 65:z52:z','iiib 65:z52:z35': 'IIIb 65:z52:z35','iiib 65:z52:z53': 'IIIb 65:z52:z53','ii 65:–:1,6': 'II 65:–:1,6','v 66:z35:–': 'V 66:z35:–','v 66:z39:–': 'V 66:z39:–','v 66:z41:–': 'V 66:z41:–','v 66:z65:–': 'V 66:z65:–','v 66:z81:–': 'V 66:z81:–','choleraesuis or typhisuis': 'I 6,7:c:1,5','miami or sendai': 'I [1],9,12:a:1,5','crossness': 'I 67:r:1,2','amager var. 15+': 'I 3,15:y:1,2:[z45]','amersfoort var. 14+': 'I 6,7,14:d:e,n,x','amsterdam var. 15+': 'I 3,15:g,m,s:–','anatum var. 15+': 'I 3,15:e,h:1,6:[z64]','anatum var. 15+,34+': 'I 3,15,34:e,h:1,6:[z64]','butantan var. 15+,34+': 'I 3,15,34:b:1,5','cerro var. 14+': 'I 6,14,18:z4,z23:[1,5]:[z45],[z82]','gdansk var. 14+': 'I 6,7,14:l,v:z6','give var. 15+': 'I 3,15:l,v:1,7:[d],[z77]','give var. 15+,34+': 'I 3,15,34:l,v:1,7:[d],[z77]','goelzau var. 15+': 'I 3,15:a:1,5','lexington var. 15+': 'I 3,15:z10:1,5:[z49]','lexington var. 15+,34+': 'I 3,15,34:z10:1,5:[z49]','lille var. 14+': 'I 6,7,14:z38:–:[z82]','livingstone var. 14+': 'I 6,7,14:d:l,w','london var. 15+': 'I 3,15:l,v:1,6','meleagridis var. 15+': 'I 3,15:e,h:l,w','meleagridis var. 15+,34+': 'I 3,15,34:e,h:l,w','muenster var. 15+': 'I 3,15:e,h:1,5:[z48]','muenster var. 15+,34+': 'I 3,15,34:e,h:1,5:[z48]','nchanga var. 15+': 'I 3,15:l,v:1,2','nyborg var. 15+': 'I 3,15:e,h:1,7','oranienburg var. 14+': 'I 6,7,14:m,t:[z57]','orion var. 15+': 'I 3,15:y:1,5','orion var. 15+,34+': 'I 3,15,34:y:1,5','oxford var. 15+,34+': 'I 3,15,34:a:1,7','paratyphi b var. l+ tartrate = d–tartrate+': 'I [1],4,[5],12:b:1,2:[z5],[z33]','rissen var. 14+': 'I 6,7,14:f,g:–','shangani var. 15+': 'I 3,15:d:1,5','souza var. 15+': 'I 3,15:d:e,n,x','stanleyville var. 27+': 'I 1,4,[5],12,27:z4,z23:[1,2]','stockholm var. 15+': 'I 3,15:y:z6','uganda var. 15+': 'I 3,15:l,z13:1,5','vejle var. 15+': 'I 3,15:e,h:1,2','vejle var. 15+,rz27': 'I 3,15:e,h:1,2:z27','weltevreden var. 15+': 'I 3,15:r:z6','westhampton var. 15+': 'I 3,15:g,s,t:–:[z37]','westhampton var. 15+,34+': 'I 3,15,34:g,s,t:–:[z37]'}
wklm_formula_to_name={'i 1,2,12:a:1,5': 'Paratyphi A','i 2,12:g,m:–': 'Nitra','i 1,2,12:g,p:–': 'Kiel','i 2,12:l,v:1,5': 'Koessen','i 1,4,5,12:a:1,2': 'Kisangani','i 4,12,27:a:1,5': 'Hessarek','i 4,5,12:a:1,5': 'Fulica','i 4,5,12:a:1,7': 'Arechavaleta','i 1,4,5,12:a:e,n,x': 'Bispebjerg','i 1,4,12,27:a:e,n,z15': 'Tinda','ii 1,4,5,12,27:a:e,n,x': 'II [1],4,[5],12,[27]:a:e,n,x','i 1,4,12:a:l,w': 'Huettwilen','i 1,4,12,27:a:z6': 'Nakuru','ii 1,4,12,27:a:z39': 'II [1],4,12,[27]:a:z39','i 1,4,5,12:b:1,2:z5,z33': 'Paratyphi B','i 1,4,5,12,27:b:1,5': 'Limete','ii 4,12:b:1,5': 'II 4,12:b:1,5','i 4,12,27:b:1,6': 'Canada','i 1,4,12,27:b:1,7': 'Uppsala','i 1,4,5,12,27:b:e,n,x': 'Abony','ii 1,4,5,12,27:b:e,n,x': 'II [1],4,[5],12,[27]:b:[e,n,x]','i 1,4,12,27:b:e,n,z15': 'Wagenia','i 1,4,12,27:b:l,w': 'Wien','i 1,4,12,27:b:z6': 'Tripoli','i 4,12,27:b:–': 'Schleissheim','i 1,4,12,27:c:1,5': 'Legon','i 4,12:c:1,6': 'Abortusovis','i 4,12,27:c:1,7': 'Altendorf','i 4,12:c:e,n,x': 'Bissau','i 1,4,12,27:c:e,n,z15': 'Jericho','i 1,4,12,27:c:l,w': 'Hallfold','i 4,12,27:c:z6': 'Bury','i 1,4,5,12,27:d:1,2': 'Stanley','i 1,4,12,27:d:1,5': 'Eppendorf','i 1,4,12,27:d:1,6': 'Brezany','i 1,4,12,27:d:1,7': 'Schwarzengrund','ii 4,12:d:e,n,x': 'II 4,12:d:e,n,x','i 1,4,5,12,27:d:e,n,x': 'Sarajane','i 1,4,12,27:d:e,n,z15:e,h': 'Duisburg','i 1,4,12,27:d:l,w': 'Mons','i 1,4,12,27:d:z6': 'Ayinde','i 4,12:d:z35': 'Chennai','i 1,4,5,12:e,h:1,2': 'Saintpaul','i 1,4,5,12:e,h:1,5:r1': 'Reading','i 4,12:e,h:1,6': 'Eko','i 4,12:e,h:1,7': 'Kaapstad','i 1,4,5,12:e,h:e,n,x': 'Chester','i 1,4,5,12:e,h:e,n,z15': 'Sandiego','i 1,4,12:e,h:l,w': 'Chartres','ii 4,12:e,n,x:1,2,7': 'II 4,12:e,n,x:1,2,7','ii 1,4,12,27:e,n,x:1,5,7': 'II [1],4,12,[27]:e,n,x:1,[5],7','i 1,4,5,12:f,g:1,2': 'Derby','i 1,4,5,12:f,g,s:1,2:z27,z45': 'Agona','ii 1,4,5,12:f,g,t:z6:z42': 'II [1],4,[5],12:f,g,t:z6:z42','i 4,12:g,m:–': 'Essen','i 1,4,5,12:g,m,s:1,2': 'Hato','ii 1,4,12,27:g,m,s,t:e,n,x': 'II [1],4,12,[27]:g,[m],[s],t:e,n,x','ii 1,4,12,27:g,m,t:1,5': 'II [1],4,12,[27]:g,[m],t:[1,5]','ii 4,12:g,m,t:z39': 'II 4,12:g,m,t:z39','i 4,12:g,m,t:z67': 'California','i 1,4,5,12,27:g,s,t:1,2:z43': 'Kingston','i 1,4,12,27:g,t:–': 'Budapest','i 4,5,12:g,z51:1,7': 'Travis','i 4,5,12:g,z51:e,n,z15': 'Tennyson','ii 4,12:g,z62:–': 'II 4,12:g,z62:–','i 1,4,5,12:m,t:1,5': 'Banana','i 4,5,12:m,t:e,n,z15': 'Madras','i 1,4,5,12:i:1,2': 'Typhimurium','i 1,4,5,12:i:1,5': 'Lagos','i 4,12:i:1,6': 'Agama','i 4,12:i:e,n,x': 'Farsta','i 1,4,12:i:e,n,z15': 'Tsevie','i 1,4,12,27:i:l,w': 'Gloucester','i 1,4,12:i:z6': 'Tumodi','ii 4,12,27:i:z35': 'II 4,12,27:i:z35','i 1,4,12,27:k:1,5': 'Massenya','i 1,4,12,27:k:1,6': 'Neumuenster','ii 1,4,12,27:k:1,6': 'II [1],4,12,27:k:1,6','i 4,12,27:k:e,n,x': 'Ljubljana','i 4,5,12:k:e,n,z15': 'Texas','i 4,5,12:l,v:1,2': 'Fyris','i 1,4,5,12,27:l,v:1,5': 'Azteca','i 4,12:l,v:1,6': 'Clackamas','i 1,4,12,27:l,v:1,7:z40': 'Bredeney','i 1,4,12,27:l,v:e,n,x': 'Kimuenza','ii 1,4,12,27:l,v:e,n,x': 'II [1],4,12,27:l,v:e,n,x','i 4,5,12:l,v:e,n,z15': 'Brandenburg','ii 1,4,12,27:l,v:z39': 'II [1],4,12,27:l,v:z39','i 4,12:l,w:1,5': 'Mono','i 4,12:l,w:1,6': 'Togo','ii 4,12:l,w:e,n,x': 'II 4,12:l,w:e,n,x','i 4,12:l,w:e,n,z15': 'Blancmesnil','i 1,4,12,27:l,w:z6': 'Ayton','i 1,4,5,12,27:l,z13,z28:1,2': 'Kunduchi','i 1,4,12,27:l,z13,z28:1,5': 'Tyresoe','i 4,12:l,z13,z28:1,6': 'Haduna','i 1,4,12,27:l,z13,z28:1,7': 'Kubacha','i 1,4,12,27:l,z13,z28:e,n,x': 'Kano','i 1,4,12,27:l,z13,z28:e,n,z15': 'Vom','i 4,12:l,z28:e,n,x': 'Reinickendorf','ii 1,4,12:l,z28:e,n,x': 'II [1],4,12:l,z28:[e,n,x]','i 1,4,5,12:r:1,2': 'Heidelberg','i 4,12,27:r:1,5': 'Bradford','i 4,12:r:1,6': 'Winneba','i 1,4,12,27:r:1,7': 'Remo','i 1,4,5,12:r:l,w': 'Bochum','i 4,12,27:r:z6': 'Southampton','i 1,4,12,27:r,i:e,n,z15': 'Drogana','i 4,12:r,i:l,w': 'Africana','i 1,4,5,12:y:1,2': 'Coeln','i 4,12,27:y:1,5': 'Trachau','i 4,12:y:1,6': 'Finaghy','i 1,4,12,27:y:1,7': 'Teddington','i 1,4,5,12,27:y:e,n,x': 'Ball','i 1,4,12,27:y:e,n,z15': 'Jos','i 1,4,12,27:y:z6': 'Kamoru','i 4,5,12:z:1,2': 'Shubra','i 1,4,12:z:1,5': 'Kiambu','ii 1,4,12,27:z:1,5': 'II [1],4,12,27:z:1,5','i 4,12:z:1,6': 'Loubomo','i 1,4,12:z:1,7': 'Indiana','ii 4,12:z:1,7': 'II 4,12:z:1,7','i 4,12:z:e,n,x': 'Neftenbach','ii 1,4,12,27:z:e,n,x': 'II [1],4,12,27:z:e,n,x','i 1,4,5,12:z:e,n,z15': 'Koenigstuhl','i 1,4,12:z:l,w': 'Preston','i 1,4,12,27:z:z6': 'Entebbe','ii 4,12:z:z39': 'II 4,12:z:z39','i 1,4,5,12,27:z4,z23:1,2': 'Stanleyville','i 4,12:z4,z23:e,n,z15': 'Perrosguirec','i 4,12,27:z4,z23:z6': 'Vuadens','i 1,4,5,12:z4,z24:1,5': 'Kalamu','i 1,4,5,12:z10:1,2': 'Haifa','i 1,4,12:z10:1,5': 'Ituri','i 4,12:z10:1,6': 'Tudu','i 4,12:z10:1,7': 'Harrow','i 4,12:z10:e,n,x': 'Albert','i 4,12:z10:e,n,z15': 'Tokoin','i 1,4,12:z10:l,w': 'Mura','i 1,4,12,27:z10:z6': 'Fortune','i 1,4,12,27:z10:z35': 'Vellore','i 1,4,12,27:z29:–': 'Brancaster','ii 1,4,12:z29:e,n,x': 'II [1],4,12:z29:e,n,x','i 4,12:z35:1,5': 'Pasing','i 1,4,12,27:z35:1,7': 'Tafo','i 1,4,12,27:z35:z6': 'Sloterdijk','i 1,4,12,27:z35:e,n,z15': 'Yaounde','i 4,12:z36:–': 'Tejas','i 1,4,12:z38:1,5': 'Yverdon','i 1,4,5,12,27:z38:e,n,z15': 'Wilhelmsburg','ii 1,4,12,27:z39:1,5,7': 'II [1],4,12,[27]:z39:1,[5],7','i 1,4,12,27:z41:1,2,5': 'Thayngen','i 1,4,12,27:z41:e,n,z15': 'Maska','i 4,12:–:e,n,x': 'Abortusequi','i 4,12:z91:–': 'Mygdal','i 6,7:a:1,5': 'Sanjuan','ii 6,7,14:a:1,5': 'II 6,7,[14]:a:1,5','i 6,7:a:1,6': 'Umhlali','i 6,7:a:1,7': 'Austin','i 6,7,14:a:e,n,x': 'Oslo','i 6,7:a:e,n,z15': 'Denver','i 6,7,14:a:l,w': 'Coleypark','i 6,7,14:a:z6': 'Damman','ii 6,7:a:z6': 'II 6,7:a:z6','ii 6,7:a:z42': 'II 6,7:a:z42','i 6,7:b:1,2': 'Brazzaville','ii 6,7:b:1,5': 'II 6,7:b:1,5','i 6,7,14:b:1,5': 'Edinburg','i 6,7:b:1,6': 'Adime','i 6,7:b:1,7': 'Koumra','i 6,7,14:b:e,n,x': 'Lockleaze','i 6,7:b:e,n,z15': 'Georgia','ii 6,7:b:e,n,x:z42': 'II 6,7:b:[e,n,x]:z42','i 6,7,14:b:l,w:z59': 'Ohio','i 6,7,14:b:z6': 'Leopoldville','i 6,7:b:z35': 'Kotte','ii 6,7:b:z39': 'II 6,7:b:z39','i 6,7,14:c:1,2': 'Hissar','i 6,7,vi:c:1,5': 'Paratyphi C','i 6,7:c:1,5': 'Choleraesuis or Typhisuis','i 6,7:c:1,6': 'Birkenhead','i 6,7:c:1,7': 'Schwabach','i 6,7:c:z6': 'Cotonou','i 6,7:c:e,n,x': 'Namibia','i 6,7,14:c:e,n,z15': 'Kaduna','i 6,7:d:1,2': 'Kisii','i 6,7,14:d:1,5': 'Isangi','i 6,7:d:1,6': 'Kivu','i 6,7:d:1,2,7': 'Kambole','i 6,7,14:d:e,n,x': 'Amersfoort','i 6,7,14:d:e,n,z15': 'Gombe','i 6,7,14:d:l,w': 'Livingstone','i 6,7:d:l,z13,z28': 'Wil','i 6,7,14:d:z6': 'Nieukerk','ii 6,7:d:z42': 'II 6,7:d:z42','i 6,7:e,h:1,2': 'Larochelle','i 6,7:e,h:1,5': 'Lomita','i 6,7:e,h:1,6': 'Norwich','i 6,7:e,h:1,7': 'Nola','i 6,7,14:e,h:e,n,z15': 'Braenderup','ii 6,7:e,n,x:1,6:z42': 'II 6,7:e,n,x:1,6:z42','i 6,7:e,n,z15:1,6': 'Kastrup','i 6,7,14:f,g:–': 'Rissen','i 6,7:f,g,t:1,2,7': 'Eingedi','i 6,7:f,g,t:e,n,x': 'Afula','i 6,7,14,54:g,m,p,s:1,2,7': 'Montevideo','i 6,7:g,m,s:e,n,z15': 'Lubbock','ii 6,7:g,m,s,t:e,n,x': 'II 6,7:g,m,[s],t:e,n,x','ii 6,7:g,m,s,t:1,5': 'II 6,7:(g),m,[s],t:1,5','ii 6,7:g,m,s,t:z39': 'II 6,7:g,m,s,t:z39','ii 6,7:g,m,s,t:z42': 'II 6,7:g,[m],s,t:[z42]','i 6,7,14:g,m,t:–': 'Othmarschen','i 6,7:g,s,q:–': 'Plumaugat','i 6,7:g,s,t:1,6': 'Menston','ii 6,7:g,t:e,n,x:z42': 'II 6,7:g,t:[e,n,x]:z42','i 6,7:g,t:–': 'Riggil','i 6,7:g,z51:1,5': 'Alamo','i 6,7:g,z51:e,n,z15': 'Larose','iv 6,7:g,z51:–': 'IV 6,7:g,z51:–','i 6,7:m,p,t,u:–': 'Haelsingborg','i 6,7:m,t:1,6': 'Winston','i 6,7:m,t:z64': 'Oakey','ii 6,7:m,t:–': 'II 6,7:m,t:–','i 6,7,14:m,t:z57': 'Oranienburg','i 6,7,14:i:1,2': 'Augustenborg','i 6,7:i:1,5': 'Oritamerin','i 6,7:i:1,6': 'Garoli','i 6,7:i:1,7': 'Lika','i 6,7:i:e,n,z15': 'Athinai','i 6,7:i:l,w': 'Norton','i 6,7,14:i:z6': 'Stuttgart','i 6,7,14:k:1,2': 'Galiema','i 6,7,14:k:1,5:r1': 'Thompson','i 6,7:k:1,6': 'Daytona','i 6,7:k:1,7': 'Baiboukoum','i 6,7:k:e,n,x': 'Singapore','i 6,7:k:e,n,z15': 'Escanaba','iiib 6,7:k:z:z55': 'IIIb 6,7:(k):z:[z55]','ii 6,7:k:z6': 'II 6,7:k:[z6]','i 6,7:k:z35': 'Ashford','i 6,7:l,v:1,2': 'Concord','i 6,7:l,v:1,5': 'Irumu','iiib 6,7:l,v:1,5,7': 'IIIb 6,7:l,v:1,5,7','i 6,7:l,v:1,6': 'Mkamba','i 6,7:l,v:1,7': 'Kortrijk','i 6,7:l,v:e,n,x': 'Bonn','i 6,7,14:l,v:e,n,z15': 'Potsdam','i 6,7,14:l,v:z6': 'Gdansk','i 6,7:l,v:z35': 'Coromandel','iiib 6,7:l,v:z53': 'IIIb 6,7:l,v:z53','i 6,7:l,w:1,2': 'Gabon','i 6,7:l,w:1,5': 'Colorado','ii 6,7:l,w:1,5,7': 'II 6,7:l,w:1,5,7','i 6,7:l,w:e,n,z15': 'Langeveld','ii 6,7:l,w:z42': 'II 6,7:l,w:z42','i 6,7:l,z13:1,5': 'Nessziona','i 6,7:l,z13:e,n,x': 'Kenya','i 6,7:l,z13,z28:e,n,z15': 'Neukoelln','i 6,7:l,z13,z28:z6': 'Makiso','i 6,7:l,z13,z28:1,7': 'Strathcona','ii 6,7:l,z28:1,5:z42': 'II 6,7:l,z28:1,5:[z42]','ii 6,7:l,z28:e,n,x': 'II 6,7:l,z28:e,n,x','ii 6,7:l,z28:z6': 'II 6,7:l,z28:z6','i 6,7,14:r:1,2': 'Virchow','i 6,7,14:r:1,5:r1,z37,z45,z49': 'Infantis','i 6,7:r:1,6': 'Nigeria','i 6,7:r:1,7': 'Colindale','i 6,7:r:e,n,z15': 'Papuana','i 6,7:r:l,w': 'Grampian','i 6,7:y:1,2': 'Richmond','i 6,7,14:y:1,5': 'Bareilly','i 6,7:y:1,6': 'Oyonnax','i 6,7:y:1,7': 'Gatow','i 6,7:y:e,n,x:z67': 'Hartford','i 6,7,14:y:e,n,z15:z47,z50': 'Mikawasima','i 6,7:z:1,2': 'Chile','i 6,7:z:1,5': 'Poitiers','ii 6,7:z:1,5': 'II 6,7:z:1,5','i 6,7:z:1,6,7': 'Oakland','i 6,7:z:e,n,x': 'Cayar','ii 6,7:z:e,n,x': 'II 6,7:z:e,n,x','ii 6,7:z:e,n,x,z15': 'II 6,7:z:e,n,x,z15','i 6,7:z:e,n,z15': 'Businga','i 6,7:z:l,w': 'Bruck','ii 6,7:z:z6': 'II 6,7:z:z6','ii 6,7:z:z39': 'II 6,7:z:z39','ii 6,7:z:z42': 'II 6,7:z:z42','i 6,7:z4,z23:1,5': 'Obogu','i 6,7:z4,z23:1,6': 'Planckendael','i 6,7:z4,z23:e,n,z15': 'Aequatoria','i 6,7:z4,z23:z6': 'Goma','ii 6,7:z4,z23:–': 'II 6,7:z4,z23:–','iv 6,7:z4,z23:–': 'IV 6,7:z4,z23:–','i 6,7:z4,z24:1,5': 'Zollikerberg','ii 6,7:z4,z24:z42': 'II 6,7:z4,z24:z42','i 6,7:z4,z24:–': 'Somone','iv 6,7:z4,z24:–': 'IV 6,7:z4,z24:–','ii 6,7:z6:1,7': 'II 6,7:z6:1,7','i 6,7:z10:1,2': 'Menden','i 6,7:z10:1,5': 'Inganda','i 6,7:z10:1,6': 'Eschweiler','i 6,7:z10:1,7': 'Ngili','i 6,7:z10:e,n,x': 'Djugu','i 6,7,14:z10:e,n,z15:z37,z45': 'Mbandaka','i 6,7,14:z10:l,w': 'Jerusalem','i 6,7:z10:z6': 'Redba','i 6,7:z10:z35': 'Omuna','i 6,7,14:z29:1,2,7': 'Tennessee','ii 6,7:z29:z42': 'II 6,7:z29:[z42]','i 6,7:z35:1,6': 'Tienba','i 6,7:z35:e,n,z15': 'Palime','i 6,7:z36:e,n,z15': 'Tampico','ii 6,7:z36:z42': 'II 6,7:z36:z42','iv 6,7:z36:–': 'IV 6,7:z36:–','i 6,7:z38:1,2:z82': 'Rumford','ii 6,7:z38:e,n,x': 'II 6,7:z38:e,n,x','i 6,7,14:z38:–:z82': 'Lille','iiib 6,7,14:z39:1,2': 'IIIb 6,7,[14]:z39:1,2','ii 6,7:z39:1,5,7': 'II 6,7:z39:1,5,7','vi 6,7:z41:1,7': 'VI 6,7:z41:1,7','i 6,7:z41:l,w': 'Hillsborough','i 6,7:z41:z35': 'Tamilnadu','ii 6,7:z42:1,5,7': 'II 6,7:z42:1,[5],7','i 6,7:z44:–': 'Bulovka','ii 6,7:–:1,6': 'II 6,7:–:1,6','i 8,20:a:z6': 'Be','i 6,8:a:1,2': 'Valdosta','i 6,8:a:1,5': 'Doncaster','i 6,8:a:1,6': 'Curacao','i 6,8:a:1,7': 'Nordufer','i 6,8:a:e,n,x': 'Narashino','ii 6,8:a:e,n,x': 'II 6,8:a:e,n,x','i 6,8:a:e,n,z15': 'Leith','ii 6,8:a:z39': 'II 6,8:a:z39','ii 6,8:a:z52': 'II 6,8:a:z52','i 8:b:1,2': 'Djelfa','i 6,8:b:1,2': 'Skansen','i 8,20:b:1,5': 'Korbol','i 6,8:b:1,5': 'Nagoya','ii 6,8:b:1,5': 'II 6,8:b:1,5','i 6,8:b:1,6': 'Stourbridge','i 8:b:1,7': 'Sanga','i 6,8:b:1,7': 'Eboko','i 8:b:e,n,x': 'Konstanz','i 6,8:b:e,n,x': 'Gatuni','i 8,20:b:e,n,z15': 'Shipley','i 6,8:b:e,n,z15': 'Presov','i 6,8:b:l,w': 'Bukuru','i 8,20:b:l,w': 'Heistopdenberg','i 8,20:b:z6': 'Tounouma','i 6,8:b:z6': 'Banalia','i 6,8:c:1,2': 'Wingrove','i 8,20:c:1,5': 'Gaillac','i 6,8:c:1,5': 'Utah','i 6,8:c:1,6': 'Bronx','i 6,8:c:1,7': 'Belfast','i 8:c:l,w': 'Alexanderpolder','i 8,20:c:e,n,x': 'Santiago','i 6,8:c:e,n,x': 'Belem','i 6,8:c:e,n,z15': 'Quiniela','i 8,20:c:z6': 'Tado','i 8:d:1,2': 'Virginia','i 6,8:d:1,2:z67': 'Muenchen','i 8,20:d:1,5': 'Yovokome','i 6,8:d:1,5:z58': 'Manhattan','i 8,20:d:1,7': 'Portanigra','i 6,8:d:1,7': 'Dunkwa','i 6,8:d:e,n,x': 'Sterrenbos','i 6,8:d:e,n,z15': 'Herston','i 8,20:d:z6': 'Labadi','ii 6,8:d:z6:z42': 'II 6,8:d:z6:z42','i 8:e,h:1,2': 'Bardo','i 6,8,20:e,h:1,2:z67,z78': 'Newport','i 8:e,h:1,5': 'Ferruch','i 6,8:e,h:1,5': 'Kottbus','i 6,8:e,h:1,6:r1': 'Cremieu','i 8,20:e,h:1,7': 'Atakpame','i 6,8:e,h:e,n,x': 'Fillmore','i 6,8:e,h:e,n,z15': 'Tshiongwe','i 8,20:e,h:z6': 'Rechovot','i 6,8:f,g:e,n,z15': 'Sandow','ii 6,8:f,g,m,t:e,n,x': 'II 6,8:f,g,m,t:[e,n,x]','i 8,20:g,m,s:–': 'Emek','i 6,8:g,m,s:e,n,x': 'Chincol','ii 6,8:g,m,t:1,7': 'II 6,8:g,m,t:1,7','i 8,20:g,m,t:–': 'Reubeuss','i 8,20:g,s,t:–': 'Alminko','i 6,8:g,s,t:–': 'Nanergou','i 8,20:m,t:–': 'Yokoe','ii 6,8:m,t:1,5': 'II 6,8:m,t:1,5','ii 6,8:m,t:e,n,x': 'II 6,8:m,t:e,n,x','i 6,8:m,t:–': 'Bassa','i 6,8:i:1,2': 'Lindenburg','i 8,20:i:1,5': 'Bargny','i 6,8:i:1,5': 'Takoradi','i 6,8:i:1,6': 'Warnow','i 6,8:i:1,7': 'Malmoe','i 6,8:i:e,n,x': 'Bonariensis','i 6,8:i:e,n,z15': 'Aba','i 8,20:i:l,w': 'Magherafelt','i 6,8:i:l,w': 'Cyprus','i 8,20:i:z6': 'Kentucky','i 6,8:k:1,2': 'Kallo','i 8:k:1,5': 'Haardt','i 6,8:k:1,5:z58': 'Blockley','i 6,8:k:1,7': 'Sandbanks','i 6,8:k:e,n,x': 'Schwerin','i 6,8:k:e,n,z15': 'Charlottenburg','iiib 8:k:z35': 'IIIb 8:(k):z35','i 8:l,v:1,2': 'Pakistan','i 6,8:l,v:1,2': 'Litchfield','i 6,8:l,v:1,5': 'Loanda','i 8:l,v:1,6': 'Amherstiana','i 6,8:l,v:1,7': 'Manchester','i 6,8:l,v:e,n,x': 'Holcomb','ii 6,8:l,v:e,n,x': 'II 6,8:l,v:e,n,x','i 6,8:l,v:e,n,z15': 'Edmonton','i 6,8:l,v:z6': 'Lund','i 6,8:l,w:1,2': 'Fayed','ii 6,8:l,w:z6:z42': 'II 6,8:l,w:z6:z42','i 6,8:l,z13,z28:1,5': 'Hiduddify','i 6,8:l,z13,z28:e,n,z15': 'Breukelen','ii 6,8:l,z28:e,n,x': 'II 6,8:l,z28:e,n,x','i 6,8:r:1,2': 'Bsilla','i 8,20:r:1,5': 'Hindmarsh','i 6,8,20:r,i:1,5:r1': 'Bovismorbificans','i 8,20:r:1,6': 'Colchester','i 8:r:1,7': 'Noya','i 6,8:r:1,7': 'Akanji','i 8,20:r,i:e,n,z15': 'Cocody','i 6,8:r,i:e,n,z15': 'Hidalgo','i 8,20:r,i:l,w': 'Brikama','i 6,8:r:l,w': 'Goldcoast','iiib 8:r:z': 'IIIb 8:r:z','i 8,20:r,i:z6': 'Altona','i 8,20:y:1,2': 'Giza','i 6,8:y:1,2': 'Lamphun','i 8,20:y:1,5': 'Brunei','i 6,8:y:1,5': 'Tananarive','i 6,8:y:1,6': 'Bulgaria','ii 6,8:y:1,6:z42': 'II 6,8:y:1,6:z42','i 8,20:y:1,7': 'Alagbon','i 6,8:y:1,7': 'Inchpark','i 8:y:e,n,x': 'Sunnycove','i 6,8:y:e,n,x': 'Daarle','i 6,8:y:e,n,z15': 'Praha','i 8,20:y:z6': 'Kralingen','i 6,8:y:l,w': 'Benue','i 8,20:y:l,w': 'Sindelfingen','i 6,8:z:1,5': 'Mowanjum','ii 6,8:z:1,5': 'II 6,8:z:1,5','i 6,8:z:1,7': 'Marmande','i 8:z:e,n,z15': 'Phaliron','i 6,8:z:e,n,z15': 'Kalumburu','i 6,8:z:l,w': 'Kuru','i 8,20:z:z6': 'Daula','i 8:z4,z23:1,7': 'Bellevue','i 6,8:z4,z23:1,7': 'Lezennes','i 6,8:z4,z23:e,n,x': 'Breda','i 6,8:z4,z23:e,n,z15': 'Chailey','i 8,20:z4,z23:l,w': 'Dabou','i 8,20:z4,z23:z6': 'Corvallis','i 8:z4,z24:e,n,x': 'Nuorikkala','i 8,20:z4,z24:–:z45': 'Albany','i 6,8:z4,z24:–': 'Duesseldorf','i 6,8:z4,z32:–': 'Tallahassee','i 8,20:z10:1,2': 'Bazenheid','i 6,8:z10:1,2': 'Zerifin','i 8,20:z10:1,5': 'Paris','i 6,8:z10:1,5': 'Mapo','i 6,8:z10:1,7': 'Cleveland','i 8:z10:e,n,x': 'Istanbul','i 6,8:z10:e,n,x': 'Hadar','i 8,20:z10:e,n,z15': 'Chomedey','i 6,8:z10:e,n,z15': 'Glostrup','i 8,20:z10:l,w': 'Remiremont','i 8,20:z10:z6': 'Molade','i 6,8:z10:z6': 'Wippra','ii 6,8:z29:1,5': 'II 6,8:z29:1,5','ii 6,8:z29:e,n,x:z42': 'II 6,8:z29:e,n,x:z42','i 8,20:z29:e,n,z15': 'Tamale','i 6,8:z29:e,n,z15': 'Uno','ii 6,8:z29:e,n,x': 'II 6,8:z29:e,n,x','i 8,20:z35:1,2': 'Kolda','i 6,8:z35:1,2': 'Yarm','i 8,20:z35:z6': 'Angers','i 8,20:z38:–': 'Apeyeme','i 8,20:z41:z6': 'Diogoye','i 6,8:z60:1,2': 'Aesch','i 1,9,12:a:1,5': 'Miami or Sendai','ii 9,12:a:1,5': 'II 9,12:a:1,5','i 9,12:a:1,6': 'Os','i 1,9,12:a:1,7': 'Saarbruecken','i 1,9,12:a:e,n,x': 'Lomalinda','ii 1,9,12:a:e,n,x': 'II [1],9,12:a:e,n,x','i 1,9,12:a:e,n,z15': 'Durban','ii 9,12:a:z39': 'II 9,12:a:z39','ii 1,9,12:a:z42': 'II [1],9,12:a:z42','i 1,9,12:b:1,2': 'Onarimon','i 1,9,12:b:1,5': 'Frintrop','i 9,12:b:1,7': 'Bata','ii 1,9,12:b:e,n,x': 'II [1],9,12:b:e,n,x','i 9,12:b:e,n,z15': 'Mana','ii 1,9,12:b:z6': 'II [1],9,12:b:z6','ii 1,9,12:b:z39': 'II [1],9,12:b:z39','i 9,12:c:1,5': 'Goeteborg','i 9,12:c:1,6': 'Ipeko','i 9,12:c:1,7': 'Elokate','i 9,12:c:e,n,z15': 'Alabama','i 9,12:c:z6': 'Ridge','i 1,9,12:d:1,5': 'Ndolo','i 9,12:d:1,6': 'Tarshyne','i 9,12:d:1,7': 'Eschberg','ii 1,9,12:d:e,n,x': 'II [1],9,12:d:e,n,x','i 1,9,12:d:e,n,x': 'Viikki','i 9,12:d:e,n,z15': 'Bangui','i 9,12:d:z6': 'Zega','i 1,9,12:d:z35': 'Jaffna','ii 9,12:d:z39': 'II 9,12:d:z39','i 9,12,vi:d:–:z66': 'Typhi','i 9,12:e,h:1,2': 'Bournemouth','i 1,9,12:e,h:1,5': 'Eastbourne','i 9,12:e,h:1,7': 'Westafrica','i 9,12:e,h:e,n,z15': 'Israel','ii 9,12:e,n,x:1,5,7': 'II 9,12:e,n,x:1,[5],7','ii 9,12:e,n,x:1,6': 'II 9,12:e,n,x:1,6','i 1,9,12:f,g,t:–': 'Berta','i 1,9,12:g,m:–': 'Enteritidis','i 9,12:g,m,s:–': 'Gueuletapee','i 9,12:g,m,q:–': 'Blegdam','ii 1,9,12:g,m,s,t:1,5,7:z42': 'II [1],9,12:g,m,[s],t:[1,5,7]:[z42]','ii 1,9,12:g,m,s,t:e,n,x': 'II [1],9,12:g,m,s,t:e,n,x','ii 9,12:g,m,t:z39': 'II 9,12:g,m,t:z39','i 1,9,12,vi:g,p:–': 'Dublin','i 1,9,12:g,p,s:–': 'Naestved','i 1,9,12:g,p,u:–': 'Rostock','i 1,9,12:g,q:–': 'Moscow','ii 9,12:g,s,t:e,n,x': 'II 9,12:g,s,t:e,n,x','i 9,12:g,z51:1,5': 'Newmexico','ii 1,9,12:g,z62:e,n,x': 'II [1],9,12:g,z62:[e,n,x]','i 9,12:g,z63:–': 'Antarctica','i 9,12:g,z85:–': 'Rosenberg','ii 9,12:m,t:e,n,x': 'II 9,12:m,t:e,n,x','i 1,9,12:m,t:1,2': 'Pensacola','ii 1,9,12:m,t:1,5': 'II [1],9,12:m,t:1,5','ii 1,9,12:m,t:z39': 'II [1],9,12:m,t:z39','i 9,12:i:1,5': 'Seremban','i 1,9,12:k:1,5': 'Claibornei','i 9,12:k:1,6': 'Goverdhan','i 9,12:l,v:1,2': 'Mendoza','i 1,9,12:l,v:1,5:r1': 'Panama','i 9,12:l,v:1,5:d': 'Houston','i 9,12:l,v:1,7:z40': 'Kapemba','i 9,12:l,v:e,n,x': 'Zaiman','ii 9,12:l,v:e,n,x': 'II 9,12:l,v:e,n,x','i 9,12:l,v:e,n,z15': 'Goettingen','ii 9,12:l,v:z39': 'II 9,12:l,v:z39','i 1,9,12:l,w:1,5': 'Victoria','ii 1,9,12:l,w:e,n,x': 'II [1],9,12:l,w:e,n,x','i 9,12:l,z13:1,5': 'Itami','i 9,12:l,z13:1,7': 'Miyazaki','i 1,9,12:l,z13:e,n,x': 'Napoli','i 1,9,12:l,z28:1,5:r1': 'Javiana','i 9,12:l,z28:1,6': 'Kotu','ii 9,12:l,z28:1,5:z42': 'II 9,12:l,z28:1,5:z42','ii 9,12:l,z28:e,n,x': 'II 9,12:l,z28:e,n,x','i 9,12:l,z28:e,n,z15': 'York','i 9,12:r:1,5': 'Jamaica','i 9,12:r:1,7': 'Camberwell','i 9,12:r:e,n,x': 'Yellowknife','i 9,12:r:e,n,z15': 'Campinense','i 9,12:r:z6': 'Lome','i 9,12:y:1,7': 'Powell','ii 1,9,12:y:z39': 'II [1],9,12:y:z39','i 1,9,12:z:1,2': 'Mulhouse','i 1,9,12:z:1,5': 'Lawndale','i 9,12:z:1,6': 'Kimpese','ii 1,9,12:z:1,7': 'II [1],9,12:z:1,7','i 9,12:z:e,n,z15': 'Aurelianis','ii 1,9,12:z:z6': 'II [1],9,12:z:z6','ii 9,12:z:z39': 'II 9,12:z:z39','i 1,9,12:z4,z23:1,7': 'Wangata','i 9,12:z4,z24:–': 'Natal','i 9,12:z6:z67:r1': 'Franken','i 9,12:z10:1,5': 'Portland','i 9,12:z10:z6': 'Treguier','i 9,12:z10:e,n,z15': 'Ruanda','ii 9,12:z29:1,5': 'II 9,12:z29:1,5','ii 1,9,12:z29:e,n,x': 'II [1],9,12:z29:e,n,x','i 9,12:z35:z6': 'Penarth','i 1,9,12:z38:–': 'Elomrane','ii 1,9,12:z39:1,7': 'II [1],9,12:z39:1,7','i 1,9,12:z41:1,5': 'Ottawa','ii 1,9,12:z42:1,5,7': 'II [1],9,12:z42:1,[5],7','i 1,9,12:–:–': 'Gallinarum','i 9,46:a:1,2': 'Detmold','i 9,46:a:1,6': 'Poole','i 9,46:a:e,n,x': 'Baildon','i 9,46:a:e,n,z15': 'Doba','i 9,46:b:1,2': 'Montaigu','i 9,46:b:1,5': 'Cheltenham','i 9,46:b:1,6': 'Zadar','i 9,46:b:e,n,x': 'Worb','ii 9,46:b:e,n,x': 'II 9,46:b:e,n,x','i 9,46:b:l,w': 'Bamboye','i 9,46:b:z6': 'Linguere','i 9,46:b:z35': 'Kolar','i 1,9,46:c:1,7': 'Argenteuil','i 9,46:c:z6': 'Itutaba','i 9,46:d:1,5': 'Ontario','i 9,46:d:1,6': 'Quentin','i 9,46:d:1,7': 'Strasbourg','i 9,46:d:e,n,z15': 'Olten','i 9,46:d:z6': 'Plymouth','i 9,46:d:z35': 'Sontheim','i 9,46:e,h:1,2': 'Bergedorf','i 9,46:e,h:1,5': 'Waedenswil','i 9,46:e,h:z6': 'Guerin','ii 9,46:e,n,x:1,5,7': 'II 9,46:e,n,x:1,5,7','i 9,46:f,g:–': 'Wernigerode','i 9,46:g,m:–': 'Hillingdon','i 9,46:g,m,s:1,2,7': 'Macclesfield','ii 9,46:g,m,s,t:e,n,x': 'II 9,46:g,[m],[s],t:[e,n,x]','i 9,46:g,s,t:–': 'Gateshead','ii 9,46:g,z62:–': 'II 9,46:g,z62:–','ii 9,46:m,t:e,n,x': 'II 9,46:m,t:e,n,x','i 9,46:m,t:–': 'Sangalkam','i 9,46:i:e,n,z15': 'Mathura','i 9,46:i:z6': 'Potto','i 9,46:k:1,2': 'Marylebone','i 9,46:k:1,5': 'Cochin','i 9,46:k:1,6': 'Clontarf','i 9,46:k:z35': 'Ceyco','i 9,46:l,v:1,5': 'India','i 9,46:l,v:1,6': 'Geraldton','i 9,46:l,v:e,n,x': 'Toronto','i 9,46:l,w:–': 'Ackwepe','ii 9,46:l,w:e,n,x': 'II 9,46:l,w:e,n,x','i 9,46:l,z13,z28:e,n,z15': 'Nordrhein','i 9,46:r:1,7': 'Deckstein','i 9,46:r:e,n,z15': 'Shoreditch','i 9,46:r:z6': 'Sokode','i 9,46:y:1,2': 'Cork','i 9,46:y:1,7': 'Benin','i 9,46:y:e,n,x': 'Irchel','i 9,46:y:l,w': 'Nantes','i 9,46:y:z6': 'Mayday','ii 9,46:z:1,5': 'II 9,46:z:1,5','ii 9,46:z:e,n,x': 'II 9,46:z:e,n,x','i 9,46:z:e,n,z15': 'Bambylor','ii 9,46:z:z39': 'II 9,46:z:z39','i 9,46:z4,z23:–': 'Ekotedo','ii 9,46:z4,z24:z39:z42': 'II 9,46:z4,z24:z39:z42','i 9,46:z4,z24:–': 'Ngaparou','i 9,46:z10:1,7': 'Lishabi','i 9,46:z10:e,n,x': 'Inglis','i 9,46:z10:e,n,z15': 'Mahina','i 9,46:z10:z6': 'Louisiana','ii 9,46:z10:z6': 'II 9,46:z10:z6','ii 9,46:z10:z39': 'II 9,46:z10:z39','i 9,46:z29:–:z45': 'Ouakam','i 9,46:z35:1,5': 'Hillegersberg','i 9,46:z35:e,n,z15': 'Basingstoke','i 9,46:z35:z6': 'Trimdon','i 9,46:z38:–': 'Fresno','ii 9,46:z39:1,7': 'II 9,46:z39:1,7','i 9,46:z41:–': 'Wuppertal','ii 1,9,12,46,27:a:z6': 'II 1,9,12,46,27:a:z6','ii 1,9,12,46,27:b:z39': 'II 1,9,12,46,27:b:z39','ii 1,9,12,46,27:c:z39': 'II 1,9,12,46,27:c:z39','ii 9,12,46,27:g,t:e,n,x': 'II 9,12,46,27:g,t:e,n,x','ii 1,9,12,46,27:l,z13,z28:z39': 'II 1,9,12,46,27:l,z13,z28:z39','ii 1,9,12,46,27:y:z39': 'II 1,9,12,46,27:y:z39','ii 1,9,12,46,27:z4,z24:1,5': 'II 1,9,12,46,27:z4,z24:1,5','ii 1,9,12,46,27:z10:1,5': 'II 1,9,12,46,27:z10:1,5','ii 1,9,12,46,27:z10:e,n,x': 'II 1,9,12,46,27:z10:e,n,x','ii 1,9,12,46,27:z10:z39': 'II 1,9,12,46,27:z10:z39','i 3,10:a:1,2': 'Aminatu','i 3,10,15:a:1,5': 'Goelzau','i 3,10,15,15,34:a:1,7': 'Oxford','i 3,10:a:e,n,x': 'Masembe','ii 3,10:a:e,n,x': 'II 3,10:a:e,n,x','i 3,10:a:e,n,z15': 'Galil','ii 3,10:a:l,v': 'II 3,10:a:l,v','ii 3,10:a:z39': 'II 3,10:a:z39','ii 3,10:a:z87': 'II 3,10:a:z87','i 3,10:b:1,2': 'Kalina','i 3,10,15,15,34:b:1,5': 'Butantan','i 3,10:b:1,6': 'Allerton','i 3,10,15,34:b:1,7': 'Huvudsta','i 3,10:b:e,n,x': 'Benfica','ii 3,10:b:e,n,x': 'II 3,10:b:e,n,x','i 3,10,15:b:e,n,z15': 'Yaba','i 3,10:b:l,w': 'Epicrates','i 3,10:b:z6': 'Wilmington','i 3,10,15:b:z35': 'Westminster','ii 3,10:b:z39': 'II 3,10:b:z39','i 3,10:c:1,2': 'Asylanta','i 3,10,15:c:1,5': 'Gbadago','i 3,10,15:c:1,6': 'Ikayi','i 3,10:c:1,7': 'Pramiso','i 3,10:c:e,n,z15': 'Agege','i 3,10:c:l,w': 'Anderlecht','i 3,10:c:z6': 'Okefoko','i 3,10:d:1,2': 'Stormont','i 3,10,15:d:1,5': 'Shangani','i 3,10:d:1,6': 'Lekke','i 3,10:d:1,7': 'Onireke','i 3,10,15:d:e,n,x': 'Souza','ii 3,10:d:e,n,x': 'II 3,10:d:e,n,x','i 3,10:d:e,n,z15': 'Madjorio','i 3,10,15:d:l,w': 'Birmingham','i 3,10:d:z6': 'Weybridge','i 3,10:d:z35': 'Maron','i 3,10,15:e,h:1,2:z27': 'Vejle','i 3,10,15,15,34:e,h:1,5:z48': 'Muenster','i 3,10,15,15,34:e,h:1,6:z64': 'Anatum','i 3,10,15:e,h:1,7': 'Nyborg','i 3,10,15,34:e,h:e,n,x': 'Newlands','i 3,10:e,h:e,n,z15': 'Lamberhurst','i 3,10,15,15,34:e,h:l,w': 'Meleagridis','i 3,10:e,h:z6': 'Sekondi','ii 3,10:e,n,x:1,7': 'II 3,10:e,n,x:1,7','i 3,10:f,g,s:1,6': 'Regent','i 3,10:f,g:e,n,x': 'Alfort','i 3,10:g,m:–': 'Suberu','i 3,10,15,15,34:g,m,s:–': 'Amsterdam','ii 3,10,15:g,m,s,t:1,5': 'II 3,{10}{15}:g,m,s,t:[1,5]','i 3,10,15,15,34:g,s,t:–:z37': 'Westhampton','i 3,10:g,t:1,5': 'Bloomsbury','ii 3,10:g,t:–': 'II 3,10:g,t:–','ii 3,10:m,t:1,5': 'II 3,10:m,t:1,5','i 3,10,15,15,34:m,t:1,6': 'Southbank','ii 3,10:m,t:e,n,x': 'II 3,10:m,t:e,n,x','i 3,10:i:1,2': 'Cuckmere','i 3,10:i:1,5': 'Amounderness','i 3,10:i:1,6': 'Tibati','i 3,10:i:1,7': 'Truro','i 3,10:i:e,n,x': 'Bessi','i 3,10,15:i:e,n,z15': 'Falkensee','i 3,10:i:l,w': 'Hoboken','i 3,10:i:z6': 'Yeerongpilly','i 3,10:k:1,2': 'Wimborne','i 3,10,15:k:1,5': 'Zanzibar','i 3,10:k:1,7': 'Serrekunda','i 3,10:k:e,n,x': 'Yundum','i 3,10:k:e,n,z15': 'Marienthal','i 3,10:k:l,w': 'Newrochelle','i 3,10,15:l,v:1,2': 'Nchanga','i 3,10:l,v:1,5': 'Sinstorf','i 3,10,15:l,v:1,6': 'London','i 3,10,15,15,34:l,v:1,7:d,z77': 'Give','ii 3,10:l,v:e,n,x': 'II 3,10:l,v:e,n,x','i 3,10:l,v:e,n,z15': 'Ruzizi','ii 3,10:l,v:z6': 'II 3,10:l,v:z6','i 3,10:l,v:z35': 'Sinchew','i 3,10:l,w:z6:z45': 'Assinie','i 3,10:l,z13:1,2': 'Freiburg','i 3,10,15:l,z13:1,5': 'Uganda','i 3,10:l,z13,z28:e,n,z15': 'Fallowfield','i 3,10:l,z13,z28:z6': 'Hoghton','ii 3,10:l,z28:1,5': 'II 3,10:l,z28:1,5','i 3,10:l,z28:1,5': 'Torbay','i 3,10:l,z28:1,7': 'Joal','i 3,10:l,z28:e,n,x': 'Lamin','ii 3,10:l,z28:e,n,x': 'II 3,10:l,z28:e,n,x','ii 3,10:l,z28:z39': 'II 3,10:l,z28:z39','i 3,10:r:1,5': 'Ughelli','i 3,10,15:r:1,7': 'Elisabethville','i 3,10:r:e,n,z15': 'Simi','i 3,10,15:r:z6': 'Weltevreden','i 3,10:r,i:1,2': 'Seegefeld','i 3,10:r,i:1,6': 'Dumfries','i 3,10,15:y:1,2:z45': 'Amager','i 3,10,15,15,34:y:1,5': 'Orion','i 3,10:y:1,7': 'Mokola','i 3,10,15:y:e,n,x': 'Ohlstedt','i 3,10:y:e,n,z15': 'Bolton','i 3,10:y:l,w': 'Langensalza','i 3,10,15:y:z6': 'Stockholm','i 3,10:z:1,5': 'Fufu','ii 3,10:z:1,5': 'II 3,10:z:1,5','i 3,10:z:1,6': 'Harleystreet','i 3,10:z:1,7': 'Huddinge','ii 3,10:z:e,n,x': 'II 3,10:z:e,n,x','i 3,10:z:e,n,z15': 'Kingslynn','i 3,10:z:l,w': 'Clerkenwell','i 3,10:z:z6': 'Landwasser','ii 3,10:z:z39': 'II 3,10:z:z39','i 3,10:z4,z23:1,7': 'Adabraka','i 3,10:z4,z23:z6': 'Wagadugu','i 3,10,15:z4,z24:–': 'Florian','ii 3,10:z4,z24:–': 'II 3,10:z4,z24:–','i 3,10:z10:1,2': 'Okerara','i 3,10,15,15,34:z10:1,5:z49': 'Lexington','i 3,10,15,15,34:z10:1,6': 'Harrisonburg','i 3,10:z10:1,7': 'Coquilhatville','i 3,10:z10:e,n,x': 'Podiensis','i 3,10:z10:e,n,z15': 'Kristianstad','i 3,10:z10:z6': 'Biafra','i 3,10:z29:e,n,x': 'Everleigh','ii 3,10:z29:e,n,x': 'II 3,10:z29:[e,n,x]','i 3,10,15:z29:–': 'Jedburgh','i 3,10:z35:1,5': 'Santaclara','i 3,10:z35:1,6': 'Ratchaburi','i 3,10:z35:1,7': 'Zongo','ii 3,10:z35:e,n,x,z15': 'II 3,10:z35:e,n,x,z15','i 3,10:z35:l,w': 'Shannon','i 3,10:z35:z6': 'Cairina','i 3,10:z36:–': 'Macallen','i 3,10:z38:1,2': 'Sandaga','i 3,10:z38:1,6': 'Albertslund','i 3,10:z38:z6': 'Bolombo','ii 3,10:z38:z42': 'II 3,10:z38:z42','ii 3,10:z39:1,5,7': 'II 3,10:z39:1,[5],7','i 3,10:z41:1,2,5': 'Dortmund','i 3,10,15,34:z69:1,7': 'Pietersburg','ii 3,15:z87:e,n,x,z15': 'II 3,15:z87:e,n,x,z15','i 3,15,34:y:1,6': 'Capetown','i 1,3,19:a:1,5': 'Niumi','i 1,3,19:a:1,7': 'Juba','i 1,3,19:a:e,n,z15': 'Gwoza','i 1,3,19:a:l,w': 'Alkmaar','i 1,3,19:b:1,5:z37': 'Gnesta','i 1,3,19:b:1,6': 'Visby','i 1,3,19:b:e,n,x': 'Tambacounda','i 1,3,19:b:e,n,z15': 'Kande','i 1,3,19:b:l,w': 'Broughton','i 1,3,19:b:z6': 'Accra','i 1,3,19:c:1,5': 'Eastglam','i 1,3,19:c:1,6': 'Bida','i 1,3,19:c:1,7': 'Madiago','i 1,3,19:d:1,2': 'Umbadah','i 1,3,19:d:1,5': 'Ahmadi','i 1,3,19:d:1,7': 'Wanatah','i 1,3,19:d:e,n,z15': 'Liverpool','i 1,3,19:d:l,w:z49': 'Tilburg','i 1,3,19:d:z6': 'Niloese','i 1,3,19:e,h:1,5': 'Vilvoorde','i 1,3,19:e,h:1,6': 'Hayindogo','i 1,3,19:e,h:1,7': 'Sanktmarx','i 1,3,19:e,h:e,n,z15': 'Sao','i 1,3,19:e,h:l,w': 'Calabar','i 1,3,19:f,g:–': 'Rideau','i 1,3,19:f,g,t:1,7': 'Petahtikve','i 1,3,19:f,g,t:e,n,z15': 'Maiduguri','i 1,3,19:g,m,t:–': 'Kouka','i 1,3,19:g,s,t:–:z27,z34,z37,z43,z45,z46,z82': 'Senftenberg','i 1,3,19:m,t:–': 'Cannstatt','i 1,3,19:i:1,2': 'Stratford','i 1,3,19:i:1,5': 'Ouagadougou','i 1,3,19:i:1,6': 'Chichester','i 1,3,19:i:e,n,x': 'Machaga','i 1,3,19:i:e,n,z15': 'Avonmouth','i 1,3,19:i:l,w': 'Zuilen','i 1,3,19:i:z6': 'Taksony','i 1,3,19:k:1,5': 'Oesterbro','i 1,3,19:k:1,7': 'Bethune','i 1,3,19:l,v:1,5': 'Ngor','i 1,3,19:l,v:1,7': 'Parkroyal','i 1,3,19:l,v:e,n,z15': 'Svedvi','i 1,3,19:l,w:1,5': 'Fulda','i 1,3,19:l,z13:1,2': 'Westerstede','i 1,3,19:l,z13:1,6': 'Winterthur','i 1,3,19:l,z13,z28:1,2': 'Lokstedt','i 1,3,19:l,z13,z28:1,5': 'Stuivenberg','i 1,3,19:l,z13,z28:e,n,z15': 'Bedford','i 1,3,19:l,z28:1,7': 'Tomelilla','i 1,3,19:l,z28:e,n,x': 'Kindia','i 1,3,19:r:e,n,z15': 'Yalding','i 1,3,19:r,i:l,w': 'Fareham','i 1,3,19:y:1,5': 'Gatineau','i 1,3,19:y:1,7': 'Thies','i 1,3,19:y:e,n,z15': 'Slade','i 1,3,19:y:e,n,x': 'Kinson','i 1,3,19:y:l,w': 'Krefeld','i 1,3,19:z:1,5': 'Korlebu','i 1,3,19:z:1,6': 'Kainji','i 1,3,19:z:1,7': 'Lerum','i 1,3,19:z:e,n,z15': 'Schoeneberg','i 1,3,19:z:l,w': 'Carno','i 1,3,19:z:z6': 'Hongkong','i 1,3,19:z4,z23:–': 'Miromesnil','i 1,3,19:z4,z24:–': 'Sambre','i 1,3,19:z10:1,5': 'Yenne','i 1,3,19:z10:e,n,z15': 'Dallgow','i 1,3,19:z29:z6:z37': 'Llandoff','i 1,3,19:z35:1,5': 'Catumagos','i 1,3,19:z38:e,n,z15': 'Ochiogu','i 1,3,19:z41:e,n,z15': 'Fortbend','i 1,3,10,19:b:z35': 'Chittagong','i 1,3,10,19:f,g,t:1,2,7': 'Bilu','i 1,3,10,19:z4,z23:z6': 'Ilugun','i 1,3,15,19:g,s,t:–': 'Dessau','i 1,3,10,15,19:y:e,n,x': 'Cannonhill','i 11:a:1,2': 'Gallen','i 11:a:1,5': 'Marseille','vi 11:a:1,5': 'VI 11:a:1,5','i 11:a:1,6': 'Massilia','i 11:a:1,7': 'Toowong','i 11:a:e,n,z15': 'Luciana','ii 11:a:e,n,z15:d': 'II 11:a:e,n,z15:d','i 11:a:l,z13,z28': 'Epinay','ii 11:a:z6:z42': 'II 11:a:z6:z42','i 11:b:1,2': 'Atento','i 11:b:1,5': 'Leeuwarden','i 11:b:1,6': 'Wohlen','vi 11:b:1,7': 'VI 11:b:1,7','vi 11:b:e,n,x': 'VI 11:b:e,n,x','i 11:b:e,n,z15': 'Pharr','i 11:b:z6': 'Erfurt','i 11:c:1,5': 'Chiredzi','i 11:c:1,6': 'Brindisi','ii 11:c:e,n,z15': 'II 11:c:e,n,z15','i 11:c:e,n,x': 'Woodinville','i 11:d:1,2': 'Ati','i 11:d:1,5': 'Gustavia','i 11:d:e,n,x:r': 'Chandans','i 11:d:z6': 'Findorff','i 11:e,h:1,2': 'Chingola','i 11:e,h:1,6': 'Adamstua','i 11:e,h:l,z13,z28': 'Redhill','i 11:g,m:1,5': 'Abuja','i 11:g,s,t:–': 'Missouri','ii 11:g,m,s,t:z39': 'II 11:g,[m],s,t:z39','iv 11:g,z51:–': 'IV 11:g,z51:–','i 11:m,t:–': 'Moers','ii 11:m,t:e,n,x': 'II 11:m,t:e,n,x','i 11:i:1,2': 'Aberdeen','i 11:i:1,5': 'Brijbhumi','i 11:i:1,6': 'Heerlen','i 11:i:e,n,x': 'Veneziana','iiib 11:i:z': 'IIIb 11:i:z','i 11:k:1,2': 'Pretoria','i 11:k:1,5': 'Abaetetuba','i 11:k:1,6': 'Sharon','i 11:k:1,7': 'Colobane','i 11:k:e,n,x,z15': 'Kisarawe','i 11:k:l,w': 'Mannheim','i 11:k:l,z13,z28': 'Amba','iiib 11:k:z53': 'IIIb 11:k:z53','i 11:l,v:1,2': 'Stendal','i 11:l,v:1,5': 'Maracaibo','i 11:l,v:e,n,x': 'Fann','i 11:l,v:e,n,z15': 'Bullbay','iiib 11:l,v:z:z56': 'IIIb 11:l,v:z:[z56]','iiib 11:l,v:z53': 'IIIb 11:l,v:z53','i 11:l,w:1,5': 'Glidji','i 11:l,z13:1,2': 'Tours','i 11:l,z13,z28:1,5': 'Connecticut','i 11:l,z13,z28:e,n,x': 'Osnabrueck','i 11:l,z13,z28:e,n,z15': 'Welwyn','ii 11:l,z28:e,n,x': 'II 11:l,z28:e,n,x','i 11:r:1,5': 'Senegal','i 11:r:e,n,x': 'Rubislaw','i 11:r:e,n,z15': 'Clanvillian','i 11:r,i:e,n,x,z15': 'Euston','i 11:r:l,z13,z28': 'Volta','i 11:y:1,5': 'Solt','i 11:y:1,7': 'Jalisco','i 11:y:e,n,x': 'Herzliya','i 11:y:e,n,x,z15': 'Woumbou','i 11:z:1,5': 'Crewe','i 11:z:1,7': 'Maroua','ii 11:z:e,n,x': 'II 11:z:e,n,x','i 11:z:e,n,z15': 'Maybush','i 11:z:z6:z83': 'Nyanza','ii 11:z:z39': 'II 11:z:z39','i 11:z4,z23:1,6': 'Remete','i 11:z4,z23:e,n,z15': 'Etterbeek','iiia 11:z4,z23:–': 'IIIa 11:z4,z23:–','iv 11:z4,z23:–': 'IV 11:z4,z23:–','i 11:z4,z24:–': 'Yehuda','iv 11:z4,z32:–': 'IV 11:z4,z32:–','i 11:z10:1,2': 'Wentworth','i 11:z10:1,5': 'Straengnaes','i 11:z10:e,n,x': 'Telhashomer','i 11:z38:–': 'Lene','i 11:z41:1,2': 'Maastricht','ii 11:z41:e,n,x': 'II 11:z41:e,n,x','ii 11:–:1,5': 'II 11:–:1,5','i 1,13,23:a:1,5': 'Chagoua','ii 1,13,23:a:1,5': 'II [1],13,23:a:1,5','i 13,22:a:1,6': 'Mim','ii 13,22:a:e,n,x': 'II 13,22:a:e,n,x','i 1,13,23:a:l,w': 'Wyldegreen','i 13,22:a:l,z13,z28': 'Marshall','ii 1,13,23:a:z42': 'II [1],13,23:a:z42','i 13,22:b:1,5': 'Ibadan','i 1,13,23:b:1,5': 'Mississippi','i 13,22:b:1,6': 'Oudwijk','ii 1,13,23:b:1,5:z42': 'II [1],13,23:b:[1,5]:z42','i 13,23:b:1,6': 'Bracknell','i 1,13,22:b:1,7': 'Rottnest','i 13,22:b:e,n,x': 'Vaertan','i 1,13,23:b:e,n,x': 'Ullevi','i 13,22:b:e,n,z15': 'Bahati','i 13,23:b:e,n,z15': 'Durham','i 13,23:b:l,w': 'Sanktjohann','ii 1,13,22:b:z42': 'II [1],13,22:b:z42','i 13,22:c:e,n,x,z15': 'Haouaria','i 13,23:c:e,n,z15': 'Boston','i 1,13,23:d:1,2': 'Handen','ii 13,22:d:1,5': 'II 13,22:d:1,5','i 1,13,23:d:1,5': 'Mishmarhaemek','i 13,22:d:1,6': 'Friedenau','i 1,13,23:d:1,6:z37': 'Wichita','i 1,13,23:d:1,7': 'Grumpensis','ii 13,23:d:e,n,x': 'II 13,23:d:e,n,x','i 13,23:d:e,n,x': 'Saintes','i 1,13,22:d:e,n,z15': 'Diguel','i 13,23:d:e,n,z15': 'Telelkebir','i 1,13,23:d:l,w': 'Putten','i 13,23:d:z6': 'Isuge','i 1,13,23:e,h:1,5': 'Tschangu','i 1,13,22:e,h:1,6': 'Willemstad','i 1,13,23:e,h:l,w': 'Vridi','ii 1,13,23:e,n,x:1,5,7': 'II [1],13,23:e,n,x:1,[5],7','i 13,22:f,g:e,n,x': 'Raus','i 1,13,23:f,g,s:–:z79': 'Havana','i 13,22:g,m:e,n,z15': 'Bron','iiib 13,22:g,m,s:z': 'IIIb 13,22:g,m,s:z','i 1,13,23:g,m,s,t:–': 'Agbeni','ii 1,13,22:g,m,t:1,5': 'II [1],13,22:g,m,t:[1,5]','ii 13,22:g,m,t:z42': 'II 13,22:g,m,t:z42','ii 1,13,23:g,m,s,t:1,5': 'II [1],13,23:g,m,s,t:1,5','ii 1,13,23:g,m,s,t:e,n,x': 'II [1],13,23:g,m,[s],t:[e,n,x]','ii 1,13,23:g,m,s,t:z42': 'II [1],13,23:g,m,s,t:z42','i 13,22:g,s,t:–': 'Newyork','ii 13,23:g,s,t:e,n,x': 'II 13,23:g,s,t:e,n,x','i 13,23:g,s,t:–': 'Okatie','ii 1,13,22:g,t:1,5': 'II [1],13,22:g,t:[1,5]','ii 13,22:g,t:z6': 'II 13,22:g,t:z6','ii 1,13,23:g,t:1,5': 'II [1],13,23:g,t:1,5','ii 13,23:g,t:e,n,x': 'II 13,23:g,t:e,n,x','ii 1,13,23:g,s,t:z42': 'II [1],13,23:g,[s],t:z42','iiia 1,13,23:g,z51:–': 'IIIa [1],13,23:g,z51:–','i 13,22:m,t:–': 'Washington','ii 1,13,23:m,t:1,5': 'II [1],13,23:m,t:1,5','ii 1,13,23:m,t:e,n,x': 'II [1],13,23:m,t:e,n,x','ii 13,22:m,t:z42:z39': 'II 13,22:m,t:z42:z39','ii 1,13,23:m,t:z42': 'II [1],13,23:m,t:z42','i 1,13,23:m,t:–': 'Kintambo','v 1,13,22:i:–': 'V [1],13,22:i:–','i 1,13,23:i:1,5': 'Idikan','i 13,23:i:1,7': 'Myrria','i 13,23:i:e,n,z15': 'Jukestown','i 1,13,23:i:l,w': 'Kedougou','i 13,23:i:z6': 'Chapuri','ii 13,22:k:1,5:z42': 'II 13,22:k:1,5:z42','i 13,23:k:1,6': 'Noho','i 13,23:k:–': 'Marburg','i 13,23:k:e,n,z15': 'Galway','ii 13,23:k:z41': 'II 13,23:k:z41','i 13,22:l,v:1,5': 'Lovelace','iiib 13,22:l,v:1,5,7': 'IIIb 13,22:l,v:1,5,7','i 13,22:l,v:1,6': 'Borbeck','i 1,13,23:l,v:e,n,z15': 'Nanga','ii 13,23:l,w:e,n,x': 'II 13,23:l,w:e,n,x','i 13,22:l,z13:e,n,z15': 'Taiping','ii 13,22:l,z28:1,5': 'II 13,22:l,z28:1,5','ii 13,23:l,z28:1,5': 'II 13,23:l,z28:1,5','ii 13,23:l,z28:z6': 'II 13,23:l,z28:z6','ii 1,13,23:l,z28:z42': 'II [1],13,23:l,z28:z42','v 13,22:r:–': 'V 13,22:r:–','i 13,23:r:1,6': 'Adjame','i 13,23:r:e,n,z15': 'Linton','i 1,13,22:y:1,6': 'Tanger','i 13,23:y:1,7': 'Yarrabah','i 1,13,23:y:l,w': 'Ordonez','i 1,13,23:y:z6': 'Tunis','i 13,22:z:1,5': 'Winslow','ii 1,13,23:z:1,5': 'II [1],13,23:z:1,5','i 1,13,23:z:1,5': 'Serenli','iiib 13,23:z:1,5': 'IIIb 13,23:z:1,5','i 1,13,22:z:1,6:z44,z59': 'Poona','i 13,23:z:1,6': 'Farmsen','i 13,22:z:1,7': 'Bristol','i 13,22:z:z6': 'Durance','i 1,13,23:z:z6': 'Ivrysurseine','i 1,13,22:z:e,n,z15': 'Tanzania','i 1,13,23:z:l,w:z43': 'Worthington','ii 1,13,23:z:z42': 'II [1],13,23:z:z42','ii 13,22:z:–': 'II 13,22:z:–','i 1,13,22:z4,z23:e,n,z15': 'Ried','iiia 13,22:z4,z23:–': 'IIIa 13,22:z4,z23:–','i 13,23:z4,z23:–': 'Ajiobo','iiia 13,23:z4,z23,z32:–': 'IIIa 13,23:z4,z23,[z32]:–','i 1,13,23:z4,z24:–': 'Romanby','iiia 13,23:z4,z24:–': 'IIIa 13,23:z4,z24:–','i 1,13,22:z10:1,5': 'Roodepoort','ii 1,13,22:z10:z6': 'II [1],13,22:z10:z6','i 13,23:z10:e,n,z15': 'Sapele','i 13,23:z10:l,w': 'Demerara','ii 13,22:z29:1,5': 'II 13,22:z29:1,5','ii 13,22:z29:e,n,x': 'II 13,22:z29:e,n,x','ii 1,13,23:z29:1,5': 'II [1],13,23:z29:1,5','ii 1,13,23:z29:e,n,x': 'II [1],13,23:z29:e,n,x','i 13,22:z29:–': 'Agoueve','i 1,13,23:z29:–:z37,z43': 'Cubana','i 13,22:z35:1,6': 'Mampong','i 13,22:z35:e,n,z15': 'Nimes','i 13,23:z35:1,6': 'Picpus','i 13,23:z35:e,n,z15': 'Anna','i 13,22:z38:–': 'Leiden','i 13,23:z38:–': 'Fanti','v 13,22:z39:–': 'V 13,22:z39:–','ii 13,22:z39:1,7': 'II 13,22:z39:1,7','ii 1,13,23:z39:1,5,7': 'II [1],13,23:z39:1,5,7','i 13,23:z41:e,n,z15': 'Pajala','ii 1,13,23:z42:1,5,7': 'II [1],13,23:[z42]:1,[5],7','ii 13,23:–:1,6': 'II 13,23:–:1,6','i 1,6,14,25:a:1,5': 'Garba','vi 1,6,14:a:1,5': 'VI [1],6,14:a:1,5','vi 1,6,14,25:a:e,n,x': 'VI [1],6,14,[25]:a:e,n,x','i 1,6,14,25:a:e,n,z15': 'Banjul','i 1,6,14,25:b:1,2': 'Ndjamena','i 1,6,14,25:b:1,5': 'Kuntair','i 1,6,14,25:b:1,7': 'Tucson','iiib 6,14:b:e,n,x': 'IIIb (6),14:b:e,n,x','i 1,6,14,25:c:1,5': 'Blijdorp','i 1,6,14,25:c:1,6': 'Kassberg','i 1,6,14,25:c:e,n,x': 'Runby','i 1,6,14,25:c:l,w': 'Minna','i 6,14,24:d:1,5': 'Martonos','i 1,6,14,25:d:1,5': 'Finkenwerder','i 1,6,14,25:d:1,6': 'Woodhull','i 6,14,24:d:1,7': 'Midway','i 1,6,14,25:d:1,7': 'Florida','i 6,14,24:d:e,n,x': 'Lindern','i 1,6,14,25:d:e,n,x': 'Charity','i 1,6,14,25:d:e,n,z15': 'Teko','i 1,6,14,25:d:l,z13,z28': 'Encino','i 1,6,14,24:d:z6': 'Albuquerque','i 6,14,24:e,h:1,5': 'Bahrenfeld','i 1,6,14,25:e,h:1,5': 'Onderstepoort','i 1,6,14,25:e,h:1,6': 'Magumeri','i 1,6,14,25:e,h:1,7': 'Beaudesert','v 6,14:e,n,z15:–': 'V 6,14:e,n,z15:–','i 1,6,14,25:g,m:–': 'Warragul','i 1,6,14,25:g,m,s:–': 'Caracas','i 1,6,14,25:g,p:–': 'Sylvania','i 6,14:g,s,t:–': 'Catanzaro','ii 1,6,14:m,t:1,5': 'II 1,6,14:m,t:1,5','ii 6,14:m,t:e,n,x': 'II 6,14:m,t:e,n,x','i 1,6,14,25:m,t:–': 'Kaitaan','i 1,6,14,25:i:1,5': 'Mampeza','i 1,6,14,25:i:1,7': 'Buzu','i 6,14,24:i:e,n,z15': 'Schalkwijk','i 1,6,14,25:i:e,n,z15': 'Moussoro','i 1,6,14,25:k:1,5': 'Harburg','ii 6,14,24:k:1,6': 'II 6,14,[24]:k:1,6','ii 6,14:k:e,n,x': 'II 6,14:k:e,n,x','i 1,6,14,25:k:e,n,x': 'Trumersee','iiib 6,14:k:z': 'IIIb (6),14:k:z','ii 1,6,14:k:z6:z42': 'II 1,6,14:k:z6:z42','iiib 6,14:k:z53': 'IIIb (6),14:k:z53','i 6,14,24:l,v:1,7': 'Amberg','i 1,6,14,25:l,v:1,7': 'Boecker','i 1,6,14,25:l,v:e,n,x': 'Horsham','i 6,14:l,v:e,n,z15': 'Alpenquai','iiib 6,14:l,v:z': 'IIIb (6),14:l,v:z','iiib 6,14:l,v:z35': 'IIIb (6),14:l,v:z35','iiib 6,14:l,v:z53': 'IIIb (6),14:l,v:z53','vi 6,14:l,v:z88': 'VI 6,14:l,v:z88','i 6,14,25:l,z13,z28:1,6': 'Owerri','i 1,6,14,25:l,z28:e,n,x': 'Aflao','i 1,6,14,25:r,i:1,5': 'Istoria','iiib 6,14:r:z': 'IIIb (6),14:r:z','i 1,6,14,25:r,i:e,n,z15': 'Surat','i 6,14,24:y:1,7': 'Carrau','i 1,6,14,25:y:1,7': 'Madelia','i 1,6,14,25:y:e,n,x': 'Fischerkietz','i 1,6,14,25:y:e,n,z15': 'Mornington','i 1,6,14,25:z:1,5': 'Homosassa','i 1,6,14,25:z:1,6': 'Kanifing','i 6,14,24:z:e,n,x': 'Soahanina','i 1,6,14,25:z:e,n,x': 'Sundsvall','i 1,6,14,25:z:e,n,z15': 'Royan','i 6,14,24:z:l,z13,z28': 'Stafford','i 1,6,14,25:z:l,z13,z28': 'Poano','i 6,14:z4,z23:1,5': 'Arapahoe','i 1,6,14,25:z4,z23:e,n,z15': 'Bousso','iv 6,14:z4,z23:–': 'IV 6,14:z4,z23:–','i 6,14,24:z4,z24:–': 'Chichiri','i 1,6,14,25:z4,z24:–': 'Uzaramo','i 1,6,14,25:z10:1,2': 'Nessa','vi 1,6,14,25:z10:1,2,7': 'VI 1,6,14,25:z10:1,(2),7','ii 1,6,14:z10:1,5': 'II 1,6,14:z10:1,5','i 1,6,14,25:z10:1,6': 'Laredo','iiib 6,14:z10:e,n,x,z15': 'IIIb (6),14:z10:e,n,x,z15','iiib 6,14:z10:z:z56,z90': 'IIIb (6),14:z10:z:[z56],[z90]','i 1,6,14,25:z10:z6': 'Ulm','ii 1,6,14:z10:z6:z42': 'II 1,6,14:z10:z6:z42','iiib 6,14:z10:z53': 'IIIb 6,14:z10:z53','i 6,14:z36:1,5': 'Potosi','ii 6,14:z36:–': 'II 6,14:z36:–','i 1,6,14,25:z38:e,n,x': 'Sara','ii 1,6,14:z42:1,6': 'II 1,6,14:z42:1,6','iiib 6,14:z52:e,n,x,z15': 'IIIb 6,14:z52:e,n,x,z15','iiib 1,6,14,25:z52:z35': 'IIIb [1],6,14,[25]:z52:z35','i 16:a:1,2': 'Hannover','i 16:a:1,5': 'Brazil','i 16:a:1,6': 'Amunigun','i 16:a:1,7': 'Nyeko','i 16:a:e,n,x': 'Togba','i 16:a:e,n,z15': 'Fischerhuette','i 16:a:z6': 'Heron','i 16:b:1,2': 'Hull','i 16:b:1,2,5': 'Melaka','i 16:b:1,5': 'Wa','i 16:b:1,6': 'Glasgow','i 16:b:e,n,x': 'Hvittingfoss','ii 16:b:e,n,x': 'II 16:b:e,n,x','i 16:b:e,n,z15': 'Sangera','i 16:b:l,w': 'Vegesack','i 16:b:z6': 'Malstatt','ii 16:b:z39': 'II 16:b:z39','ii 16:b:z42': 'II 16:b:z42','i 16:c:1,5': 'Vancouver','i 16:c:1,6': 'Gafsa','i 16:c:e,n,x': 'Shamba','i 16:c:e,n,z15': 'Hithergreen','i 16:c:l,w': 'Yoruba','i 16:d:1,2': 'Oldenburg','i 16:d:1,5': 'Sculcoates','ii 16:d:1,5': 'II 16:d:1,5','i 16:d:1,6': 'Sherbrooke','i 16:d:1,7': 'Gaminara','i 16:d:e,n,x': 'Barranquilla','ii 16:d:e,n,x': 'II 16:d:e,n,x','i 16:d:e,n,z15': 'Nottingham','i 16:d:l,w:z82': 'Caen','i 16:d:z6': 'Barmbek','i 16:e,h:1,2': 'Malakal','i 16:e,h:1,5': 'Saboya','i 16:e,h:1,6': 'Astridplein','i 16:e,h:e,n,x': 'Rhydyfelin','i 16:e,h:l,w': 'Moabit','i 16:e,h:z6': 'Weston','ii 16:e,n,x:1,5,7': 'II 16:e,n,x:1,(5),7','ii 16:e,n,x:1,6:z42': 'II 16:e,n,x:1,6:z42','i 16:f,g:–:z37': 'Tees','i 16:g,m,t:–': 'Adeoyo','i 16:g,m,s:–': 'Nikolaifleet','ii 16:g,m,s,t:1,5:z42': 'II 16:g,[m],[s],t:[1,5]:[z42]','ii 16:g,m,s,t:e,n,x': 'II 16:g,[m],[s],t:[e,n,x]','i 16:g,s,t:–': 'Cardoner','ii 16:m,t:e,n,x': 'II 16:m,t:e,n,x','i 16:m,t:e,n,z15': 'Morbihan','ii 16:m,t:z42': 'II 16:m,t:[z42]','i 16:m,t:–': 'Mpouto','i 16:i:1,5': 'Amina','i 16:i:1,6': 'Agbara','i 16:i:1,7': 'Wisbech','i 16:i:e,n,z15': 'Frankfurt','i 16:i:l,w': 'Pisa','i 16:i:z6': 'Abobo','iiib 16:i:z35': 'IIIb 16:i:z35','i 16:k:1,2': 'Szentes','i 16:k:1,6': 'Maumee','i 16:k:e,n,x': 'Nuatja','i 16:k:e,n,z15': 'Orientalis','iiib 16:k:e,n,x,z15': 'IIIb 16:(k):e,n,x,z15','iiib 16:k:z': 'IIIb 16:k:z','iiib 16:k:z35': 'IIIb 16:(k):z35','iiib 16:k:z53': 'IIIb 16:k:z53','iiib 16:l,v:1,5,7': 'IIIb 16:l,v:1,5,7','i 16:l,v:1,6:z45': 'Shanghai','i 16:l,v:1,7': 'Welikade','i 16:l,v:e,n,x': 'Salford','i 16:l,v:e,n,z15': 'Burgas','iiib 16:l,v:z:z61': 'IIIb 16:l,v:z:[z61]','i 16:l,v:z6': 'Losangeles','iiib 16:l,v:z35': 'IIIb 16:l,v:z35','iiib 16:l,v:z53': 'IIIb 16:l,v:z53','i 16:l,w:1,5': 'Zigong','i 16:l,w:1,6': 'Westeinde','i 16:l,w:e,n,x': 'Brooklyn','i 16:l,w:e,n,z15': 'Lomnava','i 16:l,w:z6': 'Essingen','ii 16:l,w:z6': 'II 16:l,w:z6','i 16:l,z13:e,n,z15': 'Mandera','i 16:l,z13,z28:1,5': 'Enugu','i 16:l,z13,z28:1,6': 'Battle','i 16:l,z13,z28:z6': 'Ablogame','i 16:l,z13,z28:e,n,x': 'Koblenz','ii 16:l,z28:z42': 'II 16:l,z28:z42','i 16:r,i:1,5': 'Rovaniemi','i 16:r:1,6': 'Ivory','i 16:r:1,7': 'Brunflo','i 16:r:z6': 'Lehrte','i 16:r,i:e,n,x': 'Annedal','i 16:r,i:e,n,z15': 'Zwickau','i 16:y:1,5': 'Saphra','i 16:y:1,6': 'Akuafo','i 16:y:e,n,x': 'Kikoma','i 16:y:e,n,z15': 'Avignon','i 16:z:1,5': 'Gerland','i 16:z:1,6': 'Fortlamy','i 16:z:1,7': 'Lingwala','i 16:z:e,n,x': 'Kassel','ii 16:z:e,n,x': 'II 16:z:e,n,x','i 16:z:e,n,x,z15': 'Brevik','i 16:z:z6': 'Bouake','ii 16:z:z42': 'II 16:z:z42','i 16:z4,z23:1,6': 'Kibi','i 16:z4,z23:z6': 'Axim','ii 16:z4,z23:–': 'II 16:z4,z23:–','iv 16:z4,z23:–': 'IV 16:z4,z23:–','i 16:z4,z24:–': 'Kaevlinge','ii 16:z4,z24:–': 'II 16:z4,z24:–','iv 16:z4,z24:–': 'IV 16:z4,z24:–','iv 16:z4,z32:–': 'IV 16:z4,z32:–','ii 16:z6:1,6': 'II 16:z6:1,6','i 16:z10:1,5': 'Badagry','iiib 16:z10:1,7': 'IIIb 16:z10:1,7','i 16:z10:1,6': 'Lisboa','iiib 16:z10:e,n,x,z15': 'IIIb 16:z10:e,n,x,z15','i 16:z10:e,n,z15': 'Redlands','i 16:z10:z6': 'Angouleme','i 16:z29:–': 'Saloniki','ii 16:z29:1,5': 'II 16:z29:1,5','ii 16:z29:e,n,x': 'II 16:z29:e,n,x','i 16:z35:1,5': 'Neuland','i 16:z35:1,6': 'Trier','i 16:z35:e,n,z15': 'Dakota','ii 16:z35:e,n,x': 'II 16:z35:e,n,x','iv 16:z36:–': 'IV 16:z36:–','ii 16:z36:e,n,z15': 'II 16:z36:e,n,z15','i 16:z38:–': 'Naware','i 16:z39:1,6': 'Grancanaria','ii 16:z42:1,5,7': 'II 16:z42:1,(5),7','ii 16:z42:1,6': 'II 16:z42:1,6','iiib 16:z52:z35': 'IIIb 16:z52:z35','i 17:a:1,2': 'Bonames','i 17:a:1,5': 'Jangwani','i 17:a:e,n,x': 'Kinondoni','i 17:b:1,2': 'Kirkee','i 17:b:1,5': 'Dahra','i 17:b:e,n,x': 'Mattenhof','ii 17:b:e,n,x,z15': 'II 17:b:e,n,x,z15','i 17:b:e,n,z15': 'Bignona','ii 17:b:z6': 'II 17:b:z6','i 17:c:1,5': 'Luedinghausen','i 17:c:1,6': 'Victoriaborg','ii 17:c:z39': 'II 17:c:z39','ii 17:d:–': 'II 17:d:–','i 17:d:1,5': 'Berlin','i 17:d:e,n,z15': 'Karlshamn','i 17:d:l,w': 'Niamey','ii 17:d:z6': 'II 17:d:z6','i 17:e,h:1,2': 'Jubilee','ii 17:e,n,x,z15:1,6': 'II 17:e,n,x,z15:1,6','ii 17:e,n,x,z15:1,5,7': 'II 17:e,n,x,z15:1,[5],7','i 17:g,m:–': 'Wembleypark','ii 17:g,m,s,t:–': 'II 17:g,m,s,t:–','i 17:g,s,t:–': 'Lowestoft','ii 17:g,t:e,n,x,z15': 'II 17:g,t:[e,n,x,z15]','ii 17:g,t:z39': 'II 17:g,t:z39','iiia 17:g,z51:–': 'IIIa 17:g,z51:–','i 17:m,t:–': 'Bama','ii 17:m,t:–': 'II 17:m,t:–','i 17:i:1,7': 'Ahanou','iiib 17:i:z35': 'IIIb 17:i:z35','i 17:k:1,5': 'Irenea','i 17:k:1,6': 'Bandim','i 17:k:1,7': 'Warri','i 17:k:e,n,x': 'Matadi','i 17:k:e,n,z15': 'Zaria','iiib 17:k:z': 'IIIb 17:k:z','ii 17:k:–': 'II 17:k:–','i 17:l,v:1,2': 'Morotai','i 17:l,v:1,5': 'Michigan','i 17:l,v:1,7': 'Lancaster','i 17:l,v:e,n,x': 'Carmel','iiib 17:l,v:e,n,x,z15': 'IIIb 17:l,v:e,n,x,z15','iiib 17:l,v:z35': 'IIIb 17:l,v:z35','i 17:l,z28:e,n,x': 'Granlo','i 17:r:1,2': 'Lode','iiib 17:r:z': 'IIIb 17:r:z','ii 17:y:–': 'II 17:y:–','i 17:y:e,n,x': 'Tendeba','i 17:y:e,n,z15': 'Hadejia','i 17:y:l,w': 'Lokomo','i 17:z:1,2': 'Gori','i 17:z:1,5': 'Warengo','i 17:z:1,6': 'Dingiri','ii 17:z:1,7': 'II 17:z:1,7','i 17:z:e,n,z15': 'Tchamba','ii 17:z:l,w:z42': 'II 17:z:l,w:z42','iiia 17:z4,z23:–': 'IIIa 17:z4,z23:–','iiia 17:z4,z23,z32:–': 'IIIa 17:z4,z23,z32:–','iiia 17:z4,z24:–': 'IIIa 17:z4,z24:–','iiia 17:z4,z32:–': 'IIIa 17:z4,z32:–','i 17:z10:e,n,x': 'Djibouti','iiib 17:z10:e,n,x,z15:z56': 'IIIb 17:z10:e,n,x,z15:[z56]','iiib 17:z10:z': 'IIIb 17:z10:z','ii 17:z10:–': 'II 17:z10:–','i 17:z29:–': 'Kandla','iiia 17:z29:–': 'IIIa 17:z29:–','iv 17:z29:–': 'IV 17:z29:–','i 17:z35:1,6': 'Aachen','iiia 17:z36:–': 'IIIa 17:z36:–','iv 17:z36:–': 'IV 17:z36:–','i 6,14,18:a:e,n,z15': 'Brazos','i 6,14,18:b:1,5': 'Fluntern','i 18:b:1,7': 'Cochise','i 6,14,18:c:e,n,x': 'Rawash','i 18:d:1,5': 'Groenekan','i 6,14,18:d:1,7': 'Usumbura','i 18:g,m:–': 'Pontypridd','i 18:g,s,t:–': 'Eaubonne','iiia 18:g,z51:–': 'IIIa 18:g,z51:–','iv 18:g,z51:–': 'IV 18:g,z51:–','ii 18:m,t:1,5': 'II 18:m,t:1,5','i 18:m,t:–': 'Langenhorn','i 18:k:1,5': 'Memphis','iiib 18:k:z53': 'IIIb 18:(k):z53','iiib 18:k:z54': 'IIIb 18:(k):z54','iiib 18:l,v:e,n,x,z15': 'IIIb 18:l,v:e,n,x,z15','i 18:l,v:e,n,z15': 'Orlando','iiib 18:l,v:z:z50': 'IIIb 18:l,v:z:[z50]','iiib 18:l,v:z53': 'IIIb 18:l,v:z53','i 18:l,w:e,n,z15': 'Toulon','i 18:r:1,5': 'Tennenlohe','iiib 18:r:z': 'IIIb 18:r:z','i 18:y:1,7': 'Troy','ii 18:y:e,n,x,z15': 'II 18:y:e,n,x,z15','i 18:z:–': 'Potengi','i 6,14,18:z4,z23:1,5:z45,z82': 'Cerro','i 18:z4,z23:z64': 'Aarhus','ii 18:z4,z23:–': 'II 18:z4,z23:–','iiia 18:z4,z23:–': 'IIIa 18:z4,z23:–','i 6,14,18:z4,z24:–': 'Blukwa','ii 18:z4,z24:–': 'II 18:z4,z24:–','iiia 18:z4,z32:–': 'IIIa 18:z4,z32:–','iiib 18:z10:e,n,x,z15': 'IIIb 18:z10:e,n,x,z15','i 18:z10:1,5': 'Leer','i 18:z10:z6': 'Carnac','ii 18:z10:z6': 'II 18:z10:z6','ii 18:z36:–': 'II 18:z36:–','iiia 18:z36:–': 'IIIa 18:z36:–','iv 18:z36,z38:–': 'IV 18:z36,z38:–','i 18:z38:–': 'Sinthia','i 18:z71:–': 'Delmenhorst','i 18:–:1,6': 'Cotia','i 21:a:1,5': 'Assen','ii 21:b:1,5': 'II 21:b:1,5','i 21:b:1,6': 'Ghana','i 21:b:e,n,x:z33,z49': 'Minnesota','i 21:c:1,6': 'Hydra','i 21:c:e,n,x': 'Rhone','ii 21:c:e,n,x': 'II 21:c:e,n,x','iiib 21:c:e,n,x,z15': 'IIIb 21:c:e,n,x,z15','i 21:d:1,5': 'Spartel','i 21:d:e,n,x': 'Magwa','i 21:d:z6': 'Madison','i 21:f,g:e,n,x': 'Good','ii 21:g,m,s,t:–': 'II 21:g,[m],[s],t:–','iiia 21:g,z51:–': 'IIIa 21:g,z51:–','iv 21:g,z51:–': 'IV 21:g,z51:–','ii 21:m,t:–': 'II 21:m,t:–','iv 21:m,t:–': 'IV 21:m,t:–','i 21:i:1,2': 'Diourbel','iiib 21:i:1,5,7': 'IIIb 21:i:1,5,7','iiib 21:i:e,n,x,z15': 'IIIb 21:i:e,n,x,z15','i 21:k:1,6': 'Salpetriere','iiib 21:k:e,n,x,z15': 'IIIb 21:k:e,n,x,z15','iiib 21:k:z': 'IIIb 21:k:z','i 21:k:1,2,5': 'Surrey','iiib 21:l,v:z': 'IIIb 21:l,v:z','iiib 21:l,v:z57': 'IIIb 21:l,v:z57','i 21:l,w:–': 'Keve','i 21:l,z28:e,n,z15': 'Jambur','i 21:r:–': 'Mountmagnet','iiib 21:r:z': 'IIIb 21:r:z','i 21:y:1,2': 'Ibaragi','i 21:y:e,n,x': 'Ruiru','ii 21:z:–': 'II 21:z:–','i 21:z4,z23:–': 'Baguida','iiia 21:z4,z23:–': 'IIIa 21:z4,z23:–','iv 21:z4,z23:–': 'IV 21:z4,z23:–','ii 21:z4,z24:–': 'II 21:z4,z24:–','iiia 21:z4,z24:–': 'IIIa 21:z4,z24:–','iv 21:z4,z32:–': 'IV 21:z4,z32:–','iiib 21:z10:e,n,x,z15': 'IIIb 21:z10:e,n,x,z15','iiib 21:z10:z': 'IIIb 21:z10:z','ii 21:z10:z6': 'II 21:z10:[z6]','iiib 21:z10:z53': 'IIIb 21:z10:z53','iiia 21:z29:–': 'IIIa 21:z29:–','i 21:z35:e,n,z15': 'Gambaga','iiia 21:z36:–': 'IIIa 21:z36:–','iv 21:z36:–': 'IV 21:z36:–','iv 21:z38:–': 'IV 21:z38:–','iiib 21:z65:e,n,x,z15': 'IIIb 21:z65:e,n,x,z15','i 28:a:1,5': 'Solna','i 28:a:1,6': 'Dakar','i 28:a:1,7': 'Bakau','i 28:a:e,n,x': 'Seattle','ii 28:a:e,n,x': 'II 28:a:e,n,x','i 28:a:e,n,z15': 'Honelis','i 28:a:z6': 'Dibra','i 28:b:1,5': 'Moero','i 28:b:1,6': 'Ashanti','i 28:b:1,7': 'Bokanjac','i 28:b:e,n,x': 'Soumbedioune','ii 28:b:e,n,x': 'II 28:b:e,n,x','i 28:b:e,n,z15': 'Langford','i 28:b:l,w': 'Freefalls','ii 28:b:z6': 'II 28:b:z6','i 28:c:1,5': 'Hermannswerder','i 28:c:1,6': 'Eberswalde','i 28:c:1,7': 'Halle','i 28:c:e,n,x': 'Dresden','i 28:c:e,n,z15': 'Wedding','i 28:c:l,w': 'Parabiago','i 28:c:z6': 'Techimani','i 28:d:1,5': 'Amoutive','i 28:d:1,6': 'Hatfield','i 28:d:1,7': 'Mundonobo','i 28:d:e,n,x': 'Mocamedes','i 28:d:e,n,z15': 'Patience','i 28:d:l,w': 'Cullingworth','i 28:d:z6': 'Etobicoke','i 28:e,h:1,5': 'Korkeasaari','i 28:e,h:1,7': 'Kpeme','i 28:e,h:e,n,z15': 'Gozo','ii 28:e,n,x:1,7': 'II 28:e,n,x:1,7','ii 28:e,n,z15:z87': 'II 28:e,n,z15:z87','i 28:f,g:–': 'Friedrichsfelde','i 28:g,m:1,6': 'Yardley','i 28:g,m:e,n,z15': 'Abadina','ii 28:g,m,s,t:1,5': 'II 28:g,(m),[s],t:1,5','i 28:g,m,s:e,n,z15': 'Croft','ii 28:g,m,t:e,n,x': 'II 28:g,m,t:e,n,x','ii 28:g,m,t:z39': 'II 28:g,m,t:z39','ii 28:g,s,t:e,n,x': 'II 28:g,s,t:e,n,x','i 28:g,s,t:–': 'Ona','ii 28:m,t:e,n,x': 'II 28:m,t:[e,n,x]','i 28:m,t:e,n,z15': 'Vinohrady','ii 28:m,t:z39': 'II 28:m,t:z39','i 28:m,t:1,6': 'Morillons','i 28:i:1,2': 'Doorn','i 28:i:1,5': 'Cotham','i 28:i:1,6': 'Volkmarsdorf','i 28:i:1,7': 'Dieuppeul','i 28:i:e,n,x': 'Warnemuende','i 28:i:e,n,z15': 'Kuessel','i 28:i:l,w': 'Douala','i 28:k:1,2': 'Guildford','i 28:k:1,5': 'Ilala','i 28:k:1,6': 'Adamstown','i 28:k:1,7': 'Ikeja','iiib 28:k:1,7': 'IIIb 28:k:1,7','i 28:k:e,n,x': 'Taunton','i 28:k:e,n,z15': 'Ank','i 28:l,v:1,5': 'Leoben','i 28:l,v:e,n,x': 'Vitkin','i 28:l,v:e,n,z15': 'Nashua','i 28:l,w:1,6': 'Ramsey','i 28:l,z13,z28:1,5': 'Catalunia','i 28:l,z13,z28:e,n,z15': 'Penilla','ii 28:l,z28:1,5': 'II 28:l,z28:1,5','i 28:l,z28:e,n,x': 'Fajara','ii 28:l,z28:e,n,x': 'II 28:l,z28:e,n,x','i 28:r:1,6': 'Bassadji','i 28:r:e,n,x': 'Kibusi','ii 28:r:e,n,z15': 'II 28:r:e,n,z15','i 28:r:l,w': 'Fairfield','i 28:r,i:1,5': 'Chicago','i 28:r,i:1,7': 'Banco','i 28:r,i:e,n,z15': 'Sanktgeorg','i 28:y:1,2': 'Oskarshamn','i 28:y:1,5': 'Nima','i 28:y:1,7:z80,z90': 'Pomona','i 28:y:e,n,x': 'Kitenge','i 28:y:e,n,z15': 'Telaviv','i 28:y:l,w': 'Shomolu','i 28:y:z6': 'Selby','i 28:z:1,5': 'Vanier','ii 28:z:1,5': 'II 28:z:1,5','i 28:z:1,6': 'Doel','i 28:z:1,7': 'Ezra','i 28:z:e,n,z15': 'Brisbane','ii 28:z:z39': 'II 28:z:z39','i 28:z4,z23:1,5': 'Cannobio','i 28:z4,z23:1,6': 'Teltow','i 28:z4,z23:e,n,z15': 'Babelsberg','i 28:z4,z24:–': 'Kethiabarny','i 28:z10:1,2': 'Rogy','i 28:z10:1,5': 'Farakan','i 28:z10:1,6': 'Libreville','i 28:z10:1,7': 'Malaysia','i 28:z10:e,n,x': 'Umbilo','i 28:z10:e,n,z15': 'Luckenwalde','i 28:z10:l,w': 'Moroto','iiib 28:z10:z': 'IIIb 28:z10:z','i 28:z29:–': 'Djermaia','ii 28:z29:1,5': 'II 28:z29:1,5','ii 28:z29:e,n,x': 'II 28:z29:e,n,x','i 28:z35:1,6': 'Konolfingen','i 28:z35:1,7': 'Babili','i 28:z35:e,n,z15': 'Santander','i 28:z38:e,n,z15': 'Aderike','i 30:a:1,2': 'Overvecht','i 30:a:1,5': 'Zehlendorf','i 30:a:e,n,x': 'Guarapiranga','i 30:a:e,n,z15': 'Doulassame','ii 30:a:z39': 'II 30:a:z39','i 30:b:1,2': 'Louga','i 30:b:1,5': 'Aschersleben','i 30:b:1,7:z33': 'Tempe','i 30:b:e,n,x': 'Urbana','i 30:b:e,n,z15': 'Neudorf','i 30:b:l,z13,z28': 'Coppettswood','ii 30:b:z6': 'II 30:b:z6','i 30:b:z6': 'Kakikoka','i 30:c:1,7': 'Zaire','i 30:c:e,n,z15': 'Morningside','ii 30:c:z39': 'II 30:c:z39','i 30:d:1,5': 'Messina','i 30:e,h:1,2': 'Livulu','i 30:e,h:1,5': 'Torhout','i 30:g,m,t:–': 'Godesberg','ii 30:g,m,s:e,n,x': 'II 30:g,m,s:e,n,x','i 30:g,m,s:–': 'Giessen','iv 30:g,m,t,z51:–': 'IV 30:g,m,t,z51:–','i 30:g,s,t:–:z59': 'Sternschanze','ii 30:g,t:–': 'II 30:g,t:–','i 30:g,z51:–': 'Wayne','ii 30:m,t:–': 'II 30:m,t:–','i 30:i:1,2': 'Landau','i 30:i:1,5': 'Morehead','i 30:i:e,n,z15': 'Mjordan','i 30:i:l,w': 'Soerenga','i 30:k:1,2': 'Hilversum','i 30:k:1,5': 'Ramatgan','i 30:k:1,6': 'Aqua','i 30:k:e,n,x': 'Angoda','i 30:k:e,n,x,z15': 'Odozi','ii 30:k:e,n,x,z15': 'II 30:k:e,n,x,z15','i 30:k:l,z13,z28': 'Scarborough','i 30:l,v:1,2': 'Ligeo','i 30:l,v:1,5': 'Donna','i 30:l,z13,z28:1,6': 'Ockenheim','i 30:l,z13,z28:e,n,z15': 'Morocco','ii 30:l,z28:z6': 'II 30:l,z28:z6','i 30:r:1,2': 'Grandhaven','i 30:r:1,5': 'Gege','i 30:r:1,6': 'Quincy','i 30:y:1,2': 'Matopeni','i 30:y:1,5': 'Bietri','i 30:y:1,6': 'Steinplatz','i 30:y:e,n,x': 'Baguirmi','i 30:y:e,n,z15': 'Nijmegen','i 30:z:e,n,x,z15': 'Hohentwiel','i 30:z4,z23:–': 'Stoneferry','i 30:z4,z24:–': 'Bodjonegoro','ii 30:z6:1,6': 'II 30:z6:1,6','i 30:z10:1,2': 'Sada','i 30:z10:1,5': 'Senneville','i 30:z10:e,n,z15': 'Kumasi','ii 30:z10:e,n,x,z15': 'II 30:z10:e,n,x,z15','i 30:z29:–': 'Aragua','i 30:z35:1,6': 'Kokoli','i 30:z35:e,n,z15': 'Wuiti','i 30:z38:–': 'Ago','ii 30:z39:1,7': 'II 30:z39:1,7','i 35:a:e,n,z15': 'Umhlatazana','i 35:b:–': 'Tchad','i 35:c:1,2': 'Keurmassar','i 35:c:1,5': 'Gouloumbo','i 35:c:e,n,z15': 'Yolo','ii 35:d:1,5': 'II 35:d:1,5','i 35:d:l,w:z58': 'Dembe','i 35:e,h:z6': 'Gassi','i 35:f,g:–:z27': 'Adelaide','i 35:g,m,s:–': 'Ealing','ii 35:g,m,s,t:–': 'II 35:g,m,s,t:–','i 35:g,m,t:–': 'Ebrie','i 35:g,s,t:–': 'Anecho','ii 35:g,t:1,5': 'II 35:g,t:1,5','ii 35:g,t:z42': 'II 35:g,t:z42','i 35:g,t:–': 'Agodi','iiia 35:g,z51:–': 'IIIa 35:g,z51:–','ii 35:g,z62:e,n,x': 'II 35:g,z62:e,n,x','i 35:m,t:–': 'Monschaui','ii 35:m,t:–': 'II 35:m,t:–','iiib 35:i:e,n,x,z15': 'IIIb 35:i:e,n,x,z15','i 35:i:e,n,z15': 'Gambia','i 35:i:l,w': 'Bandia','iiib 35:i:z': 'IIIb 35:i:z','i 35:i:z6': 'Evry','iiib 35:i:z35': 'IIIb 35:i:z35','iiib 35:i:z53': 'IIIb 35:i:z53','i 35:k:e,n,z15': 'Penzing','iiib 35:k:e,n,x,z15': 'IIIb 35:k:e,n,x,z15','iiib 35:k:z': 'IIIb 35:k:z','iiib 35:k:z35': 'IIIb 35:(k):z35','iiib 35:k:z53:z50': 'IIIb 35:k:z53:[z50]','iiib 35:l,v:1,5,7': 'IIIb 35:l,v:1,5,7','iiib 35:l,v:e,n,x,z15:z50': 'IIIb 35:l,v:e,n,x,z15:[z50]','iiib 35:l,v:z35:z67': 'IIIb 35:l,v:z35:[z67]','ii 35:l,z28:–': 'II 35:l,z28:–','iiib 35:r:e,n,x,z15': 'IIIb 35:r:e,n,x,z15','i 35:r:l,w': 'Massakory','iiib 35:r:z': 'IIIb 35:r:z','iiib 35:r:z35': 'IIIb 35:r:z35','iiib 35:r:z61': 'IIIb 35:r:z61','i 35:y:l,w': 'Baltimore','i 35:z4,z23:–:z37,z45': 'Alachua','iiia 35:z4,z23:–': 'IIIa 35:z4,z23:–','i 35:z4,z24:–': 'Westphalia','iiia 35:z4,z24:–': 'IIIa 35:z4,z24:–','iiia 35:z4,z32:–': 'IIIa 35:z4,z32:–','i 35:z10:1,5': 'Camberene','i 35:z10:l,w': 'Enschede','i 35:z10:z6': 'Ligna','iiib 35:z10:z35': 'IIIb 35:z10:z35','ii 35:z29:e,n,x': 'II 35:z29:e,n,x','i 35:z29:–': 'Widemarsh','iiia 35:z29:–': 'IIIa 35:z29:–','iiia 35:z36:–': 'IIIa 35:z36:–','i 35:z38:–': 'Haga','iiib 35:z52:1,5,7': 'IIIb 35:z52:1,5,7','iiib 35:z52:e,n,x,z15': 'IIIb 35:z52:e,n,x,z15','iiib 35:z52:z': 'IIIb 35:z52:z','iiib 35:z52:z35': 'IIIb 35:z52:z35','i 38:a:e,n,z15': 'Oran','ii 38:b:1,2': 'II 38:b:1,2','i 38:b:e,n,z15': 'Rittersbach','i 38:c:1,5': 'Sheffield','i 38:c:1,6:z58': 'Kidderminster','i 38:d:1,5': 'Willamette','ii 38:d:1,5': 'II 38:d:[1,5]','ii 38:d:z39': 'II 38:d:z39','i 38:e,h:1,2': 'Thiaroye','i 38:e,h:1,5': 'Kasenyi','i 38:g,m,s:–': 'Korovi','ii 38:g,t:–': 'II 38:g,t:–','iiia 38:g,z51:–': 'IIIa 38:g,z51:–','iv 38:g,z51:–': 'IV 38:g,z51:–','i 38:m,t:–': 'Rothenburgsort','i 38:i:1,2': 'Mgulani','i 38:i:1,5': 'Lansing','iiib 38:i:z': 'IIIb 38:i:z','iiib 38:i:z53': 'IIIb 38:i:z53','i 38:k:1,2': 'Echa','i 38:k:1,5': 'Mango','i 38:k:1,6': 'Inverness','i 38:k:e,n,x': 'Njala','iiib 38:k:e,n,x,z15': 'IIIb 38:k:e,n,x,z15','iiib 38:k:z': 'IIIb 38:k:z','iiib 38:k:z53': 'IIIb 38:k:z53','iiib 38:k:1,5,7': 'IIIb 38:(k):1,5,7','iiib 38:k:z35:z56': 'IIIb 38:(k):z35:[z56]','iiib 38:k:–': 'IIIb 38:(k):–','iiib 38:k:z55': 'IIIb 38:(k):z55','i 38:l,v:1,2': 'Alger','i 38:l,v:1,5': 'Kimberley','i 38:l,v:e,n,z15': 'Taylor','i 38:l,v:e,n,x': 'Roan','iiib 38:l,v:z': 'IIIb 38:l,v:z','iiib 38:l,v:z35': 'IIIb 38:l,v:z35','iiib 38:l,v:z53:z54': 'IIIb 38:l,v:[z53]:[z54]','i 38:r:1,5': 'Lindi','iiib 38:r:1,5,7': 'IIIb 38:r:1,5,7','i 38:r:1,6': 'Emmastad','iiib 38:r:e,n,x,z15': 'IIIb 38:r:e,n,x,z15','iiib 38:r:z:z57': 'IIIb 38:r:z:[z57]','iiib 38:r:z35': 'IIIb 38:r:z35','i 38:y:1,5': 'Freetown','i 38:y:1,6': 'Colombo','i 38:y:e,n,x': 'Perth','i 38:z:–': 'Stachus','i 38:z4,z23:1,2': 'Yoff','iiia 38:z4,z23:–': 'IIIa 38:z4,z23:–','iv 38:z4,z23:–': 'IV 38:z4,z23:–','i 38:z4,z24:–': 'Bangkok','i 38:z10:1,5': 'Neunkirchen','iiib 38:z10:z': 'IIIb 38:z10:z','iiib 38:z10:z53': 'IIIb 38:z10:z53','i 38:z35:e,n,z15': 'Carpentras','i 38:z38:–': 'Klouto','iiib 38:z52:z35': 'IIIb 38:z52:z35','iiib 38:z52:z53': 'IIIb 38:z52:z53','iiib 38:z53:–:z47,z50,z76': 'IIIb 38:z53:–:[z47],[z50],[z76]','iiib 38:z61:z53': 'IIIb 38:z61:[z53]','ii 39:a:z39': 'II 39:a:z39','i 39:b:1,2': 'Wandsworth','i 39:b:l,w': 'Abidjan','ii 39:c:e,n,x': 'II 39:c:e,n,x','i 39:d:1,5': 'Logone','i 39:e,h:1,2': 'Bruebach','i 39:e,h:1,5': 'Mara','ii 39:e,n,x:1,7': 'II 39:e,n,x:1,7','i 39:m,t:–': 'Dietrichsdorf','ii 39:g,m,t:e,n,x': 'II 39:[g],m,t:[e,n,x]','i 39:i:1,5': 'Hofit','i 39:i:e,n,x': 'Cumberland','i 39:i:e,n,z15': 'Alma','i 39:k:1,5:z48': 'Champaign','i 39:k:e,n,x': 'Newjersey','ii 39:l,v:1,5': 'II 39:l,v:1,5','i 39:l,v:e,n,x': 'Kokomlemle','i 39:l,v:e,n,z15': 'Oerlikon','ii 39:l,z28:e,n,x': 'II 39:l,z28:e,n,x','ii 39:l,z28:z39': 'II 39:l,z28:z39','i 39:r:1,5': 'Frimley','i 39:y:1,2': 'Anfo','i 39:y:1,5': 'Windermere','i 39:y:e,n,z15': 'Delan','i 39:z4,z23:–': 'Namur','i 39:z10:–': 'Hegau','ii 39:z10:z6': 'II 39:z10:z6','i 39:–:1,6': 'Landeskrone','ii 39:–:1,7': 'II 39:–:1,7','i 40:a:1,5': 'Shikmonah','i 40:a:1,7': 'Salinas','i 40:a:z6': 'Greiz','ii 40:a:z39': 'II 40:a:z39','i 40:b:1,5': 'Riogrande','i 40:b:1,7': 'Saugus','i 1,40:b:e,n,x': 'Johannesburg','i 1,40:b:e,n,z15': 'Duval','i 40:b:z6': 'Benguella','ii 40:b:–': 'II 40:b:–','ii 1,40:c:e,n,x,z15': 'II [1],40:c:e,n,x,z15','ii 40:c:z6': 'II 40:c:z6','ii 1,40:c:z39': 'II [1],40:c:z39','i 1,40:d:1,5': 'Driffield','ii 40:d:–': 'II 40:d:–','i 1,40:e,h:1,2': 'Tilene','ii 1,40:e,n,x:1,5,7': 'II [1],40:e,n,x:1,[5],7','ii 1,40:e,n,x,z15:1,6': 'II [1],40:e,n,x,z15:1,6','i 1,40:g,m:–': 'Bijlmer','i 1,40:g,m,s:e,n,x': 'Athens','ii 1,40:g,m,s,t:e,n,x': 'II [1],40:g,[m],[s],[t]:e,n,x','ii 1,40:g,m,s,t:1,5': 'II [1],40:g,[m],[s],t:[1,5]','ii 1,40:g,t:e,n,x,z15': 'II [1],40:g,t:e,n,x,z15','ii 40:g,t:z39': 'II 40:g,t:z39','iv 1,40:g,t:–': 'IV [1],40:g,t:–','ii 1,40:g,m,s,t:z42': 'II [1],40:g,[m],[s],t:z42','iiia 40:g,z51:–': 'IIIa 40:g,z51:–','iiib 40:g,z51:e,n,x,z15': 'IIIb 40:g,z51:e,n,x,z15','iv 1,40:g,z51:–': 'IV [1],40:g,z51:–','ii 40:m,t:e,n,x': 'II 40:m,t:e,n,x','ii 40:m,t:z39': 'II 40:m,t:z39','ii 1,40:m,t:z42': 'II [1],40:m,t:z42','iv 40:m,t:–': 'IV 40:m,t:–','iiib 40:i:1,5,7': 'IIIb 40:i:1,5,7','i 1,40:k:1,5': 'Goulfey','i 1,40:k:1,6': 'Allandale','i 40:k:e,n,x': 'Hann','ii 1,40:k:e,n,x,z15': 'II [1],40:k:e,n,x,z15','iiib 40:k:z:z57': 'IIIb 40:k:z:z57','ii 40:k:z6': 'II 40:k:z6','iiib 40:k:z53': 'IIIb 40:k:z53','i 1,40:l,v:1,2': 'Millesi','i 40:l,v:1,6': 'Canary','ii 40:l,v:e,n,x': 'II 40:l,v:e,n,x','iiib 40:l,v:z': 'IIIb 40:l,v:z','iiib 1,40:l,v:z53': 'IIIb [1],40:l,v:z53','i 1,40:l,w:1,2': 'Overchurch','ii 40:l,z28:e,n,x': 'II 40:l,z28:e,n,x','i 1,40:l,z13,z28:1,2': 'Tiko','i 1,40:l,z28:1,5': 'Bukavu','ii 1,40:l,z28:1,5:z42': 'II [1],40:l,z28:1,5:z42','i 40:l,z28:1,6': 'Santhiaba','ii 1,40:l,z28:z39': 'II [1],40:l,z28:z39','iiib 40:r:z53': 'IIIb 40:r:z53','i 40:y:1,5': 'Odienne','ii 1,40:z:1,5': 'II [1],40:z:1,5','i 40:z:e,n,x': 'Casamance','i 40:z:z6': 'Nowawes','ii 1,40:z:z6': 'II [1],40:z:z6','ii 1,40:z:z39': 'II [1],40:z:z39','ii 40:z:z42': 'II 40:z:z42','iiia 40:z4,z23:–': 'IIIa 40:z4,z23:–','iv 1,40:z4,z23:–': 'IV [1],40:z4,z23:–','ii 40:z4,z24:z39': 'II 40:z4,z24:z39','iiia 40:z4,z24:–': 'IIIa 40:z4,z24:–','iv 1,40:z4,z24:–': 'IV [1],40:z4,z24:–','iiia 40:z4,z32:–': 'IIIa 40:z4,z32:–','iv 1,40:z4,z32:–': 'IV [1],40:z4,z32:–','ii 1,40:z6:1,5': 'II [1],40:z6:1,5','i 40:z10:z6': 'Trotha','ii 40:z10:e,n,x': 'II 40:z10:e,n,x','iiib 40:z10:z35': 'IIIb 40:z10:z35','i 1,40:z29:–': 'Omifisan','iiia 40:z29:–': 'IIIa 40:z29:–','ii 1,40:z35:e,n,x,z15': 'II [1],40:z35:e,n,x,z15','i 1,40:z35:e,n,z15': 'Yekepa','v 1,40:z35:–': 'V [1],40:z35:–','iiia 40:z36:–': 'IIIa 40:z36:–','ii 1,40:z39:1,5:z42': 'II [1],40:z39:1,5:z42','ii 1,40:z39:1,6': 'II [1],40:z39:1,6','iiib 40:z39:1,6': 'IIIb 40:z39:1,6','ii 40:z39:1,7': 'II 40:z39:1,7','i 1,40:z41:1,2': 'Karamoja','ii 1,40:z42:1,6': 'II [1],40:z42:1,6','ii 1,40:z42:1,5,7': 'II [1],40:[z42]:1,(5),7','ii 1,40:z81:z6': 'II [1],40:z81:z6','v 1,40:z81:–': 'V [1],40:z81:–','i 41:a:–': 'Burundi','ii 41:b:1,5': 'II 41:b:1,5','i 41:b:1,6': 'Vaugirard','vi 41:b:1,7': 'VI 41:b:1,7','i 41:b:z6': 'Vietnam','i 41:b:e,n,z15': 'Sica','i 41:c:–': 'Lonestar','iiib 41:c:e,n,x,z15': 'IIIb 41:c:e,n,x,z15','ii 41:c:z6': 'II 41:c:z6','i 41:d:1,5': 'Egusi','ii 41:d:z6': 'II 41:d:z6','ii 41:g,m,s,t:z6': 'II 41:g,m,s,t:z6','ii 41:g,t:–': 'II 41:g,t:–','iiia 41:g,z51:–': 'IIIa 41:g,z51:–','i 41:m,t:1,6': 'Leatherhead','i 41:i:1,5': 'Samaru','i 41:i:1,6': 'Verona','i 41:k:1,6': 'Ferlo','ii 41:k:1,6': 'II 41:k:1,6','ii 41:k:z6': 'II 41:k:z6','iiib 41:k:z35': 'IIIb 41:(k):z35','ii 41:l,z13,z28:e,n,x,z15': 'II 41:l,z13,z28:e,n,x,z15','i 41:r:1,5': 'Lubumbashi','i 41:r:1,7': 'Konongo','i 41:y:–': 'Bruxelles','ii 41:z:1,5': 'II 41:z:1,5','i 41:z:1,6': 'Sally','i 41:z4,z23:1,7': 'Bofflens','i 41:z4,z23:e,n,z15': 'Waycross','iiia 41:z4,z23:–': 'IIIa 41:z4,z23:–','iv 41:z4,z23:–': 'IV 41:z4,z23:–','iiia 41:z4,z23,z32:–': 'IIIa 41:z4,z23,z32:–','i 41:z4,z24:1,5': 'Ipswich','iiia 41:z4,z24:–': 'IIIa 41:z4,z24:–','iiia 41:z4,z32:–': 'IIIa 41:z4,z32:–','ii 41:z10:1,2': 'II 41:z10:1,2','i 41:z10:1,5': 'Leipzig','i 41:z10:1,6': 'Landala','i 41:z10:e,n,x': 'Inpraw','ii 41:z10:e,n,x,z15': 'II 41:z10:e,n,x,z15','ii 41:z10:z6': 'II 41:z10:z6','i 41:z29:–': 'Lodz','iiia 41:z29:–': 'IIIa 41:z29:–','iv 41:z29:–': 'IV 41:z29:–','i 41:z35:1,6': 'Ahoutoue','iiia 41:z36:–': 'IIIa 41:z36:–','iv 41:z36:–': 'IV 41:z36:–','i 41:z38:–': 'Offa','iv 41:z52:–': 'IV 41:z52:–','ii 41:–:1,6': 'II 41:–:1,6','i 1,42:a:e,n,z15': 'Faji','ii 42:b:1,5': 'II 42:b:1,5','i 42:b:1,6': 'Orbe','ii 42:b:e,n,x,z15': 'II 42:b:e,n,x,z15','i 1,42:b:e,n,z15': 'Tomegbe','i 1,42:b:l,w': 'Frederiksberg','i 1,42:b:z6': 'Egusitoo','ii 42:b:z6': 'II 42:b:z6','i 1,42:c:e,n,z15': 'Antwerpen','i 1,42:c:z6': 'Kampala','i 1,42:d:z6': 'Gostenhof','ii 42:d:z6': 'II 42:d:z6','ii 42:e,n,x:1,6': 'II 42:e,n,x:1,6','ii 42:g,t:–': 'II 42:g,t:–','i 1,42:g,z51:1,5': 'Maricopa','iiia 42:g,z51:–': 'IIIa 42:g,z51:–','iv 1,42:g,z51:–': 'IV [1],42:g,z51:–','ii 42:m,t:e,n,x,z15': 'II 42:m,t:[e,n,x,z15]','i 1,42:m,t:–': 'Waral','i 1,42:i:l,w': 'Kaneshie','i 42:i:1,6': 'Borromea','i 1,42:i:z6': 'Middlesbrough','i 42:k:1,6': 'Haferbreite','iiib 42:k:e,n,x,z15': 'IIIb 42:k:e,n,x,z15','iiib 42:k:z': 'IIIb 42:k:z','i 1,42:k:z6': 'Gwale','iiib 42:k:z35': 'IIIb 42:(k):z35','iiib 42:l,v:1,5,7:z76': 'IIIb 42:l,v:1,5,7:[z76]','i 1,42:l,v:1,6,7': 'Portedeslilas','ii 42:l,v:e,n,x,z15': 'II 42:l,v:e,n,x,z15','iiib 42:l,v:e,n,x,z15': 'IIIb 42:l,v:e,n,x,z15','i 42:l,v:e,n,z15': 'Coogee','iiib 42:l,v:z': 'IIIb 42:l,v:z','iiib 42:l,v:z53': 'IIIb 42:l,v:z53','ii 1,42:l,w:e,n,x': 'II [1],42:l,w:e,n,x','i 1,42:l,w:z35': 'Parakou','ii 1,42:l,z13,z28:z6': 'II [1],42:l,[z13],z28:[z6]','i 1,42:r:e,n,z15': 'Sipane','i 1,42:r:l,w': 'Brive','iiib 42:r:z': 'IIIb 42:r:z','iiib 42:r:z53': 'IIIb 42:r:z53','ii 42:r:–': 'II 42:r:–','iiib 42:r:–:z50': 'IIIb 42:r:–:[z50]','i 1,42:y:e,n,z15': 'Spalentor','i 1,42:y:z6': 'Harvestehude','ii 42:z:1,5': 'II 42:z:1,5','i 1,42:z:1,6': 'Ursenbach','ii 42:z:e,n,x,z15': 'II 42:z:e,n,x,z15','i 42:z:e,n,z15': 'Melbourne','ii 42:z:z6': 'II 42:z:z6','i 1,42:z4,z23:1,6:1,6': 'Gera','i 42:z4,z23:e,n,z15': 'Broc','iiia 42:z4,z23:–': 'IIIa 42:z4,z23:–','i 1,42:z4,z24:–': 'Toricada','iiia 42:z4,z24:–': 'IIIa 42:z4,z24:–','iv 1,42:z4,z24:–': 'IV [1],42:z4,z24:–','ii 42:z6:1,6': 'II 42:z6:1,6','ii 42:z10:1,2': 'II 42:z10:1,2','ii 42:z10:e,n,x,z15': 'II 42:z10:e,n,x,z15','iiib 42:z10:e,n,x,z15:z60': 'IIIb 42:z10:e,n,x,z15:[z60]','iiib 42:z10:z': 'IIIb 42:z10:z','i 1,42:z10:z6': 'Loenga','ii 42:z10:z6': 'II 42:z10:z6','iiib 42:z10:z35': 'IIIb 42:z10:z35','iiib 42:z10:z67': 'IIIb 42:z10:z67','i 1,42:z29:1,5': 'Djama','ii 42:z29:–': 'II 42:z29:–','i 1,42:z35:1,6': 'Kahla','i 42:z35:e,n,z15': 'Hennekamp','i 1,42:z35:z6': 'Tema','i 42:z36:–': 'Weslaco','iv 42:z36:–': 'IV 42:z36:–','i 1,42:z38:z6': 'Vogan','i 1,42:z41:–': 'Taset','iiib 42:z52:z': 'IIIb 42:z52:z','iiib 42:z57:1,5': 'IIIb 42:z57:1,5','i 43:a:1,2': 'Graz','i 43:a:1,5': 'Berkeley','ii 43:a:1,5': 'II 43:a:1,5','ii 43:a:z6': 'II 43:a:z6','i 43:b:–': 'Niederoderwitz','i 43:b:e,n,z15': 'Ede','ii 43:b:z42': 'II 43:b:z42','i 43:c:1,5': 'Montreal','i 43:d:1,5': 'Orleans','ii 43:d:e,n,x,z15': 'II 43:d:e,n,x,z15','ii 43:d:z39': 'II 43:d:z39','ii 43:d:z42': 'II 43:d:z42','ii 43:e,n,x,z15:1,5,7': 'II 43:e,n,x,z15:1,(5),7','ii 43:e,n,x,z15:1,6': 'II 43:e,n,x,z15:1,6','i 43:f,g,t:–': 'Milwaukee','ii 43:g,m,s,t:z42': 'II 43:g,m,[s],t:[z42]','ii 43:g,t:1,5': 'II 43:g,t:[1,5]','iiib 43:g,t:–': 'IIIb 43:g,t:–','iiia 43:g,z51:–': 'IIIa 43:g,z51:–','iv 43:g,z51:–': 'IV 43:g,z51:–','ii 43:g,z62:e,n,x': 'II 43:g,z62:e,n,x','i 43:i:1,2': 'Mbao','i 43:i:e,n,x': 'Voulte','i 43:k:1,2': 'Thetford','i 43:k:1,5': 'Ahuza','iiib 43:k:z': 'IIIb 43:k:z','iiib 43:l,v:z53:z56': 'IIIb 43:l,v:z53:[z56]','i 43:l,w:z44': 'Epalinges','i 43:l,z13:–': 'Sudan','ii 43:l,z13,z28:1,5': 'II 43:l,z13,z28:1,5','ii 43:m,t:–': 'II 43:m,t:–','iiib 43:r:e,n,x,z15': 'IIIb 43:r:e,n,x,z15','iiib 43:r:z': 'IIIb 43:r:z','iiib 43:r:z53': 'IIIb 43:r:z53','i 43:y:1,2': 'Farcha','i 43:y:1,5': 'Kingabwa','i 43:z:1,5': 'Ogbete','ii 43:z:1,5': 'II 43:z:1,5','i 43:z:e,n,z15': 'Arusha','i 43:z4,z23:1,2': 'Farmingdale','ii 43:z4,z23:–': 'II 43:z4,z23:–','iiia 43:z4,z23:–': 'IIIa 43:z4,z23:–','iv 43:z4,z23:–': 'IV 43:z4,z23:–','iiia 43:z4,z24:–': 'IIIa 43:z4,z24:–','iv 43:z4,z24:–': 'IV 43:z4,z24:–','iv 43:z4,z32:–': 'IV 43:z4,z32:–','i 43:z10:1,5': 'Adana','ii 43:z29:e,n,x': 'II 43:z29:e,n,x','ii 43:z29:z42': 'II 43:z29:z42','i 43:z29:–': 'Makiling','iiia 43:z29:–': 'IIIa 43:z29:–','iv 43:z29:–': 'IV 43:z29:–','i 43:z35:1,6': 'Ahepe','iiia 43:z36:–': 'IIIa 43:z36:–','iv 43:z36,z38:–': 'IV 43:z36,z38:–','i 43:z38:–': 'Irigny','i 43:z41:1,2': 'Darfur','ii 43:z42:1,5,7': 'II 43:z42:[1,5,7]','iiib 43:z52:z53': 'IIIb 43:z52:z53','iv 44:a:–': 'IV 44:a:–','i 44:a:1,5': 'Niakhar','i 44:a:e,n,x': 'Tiergarten','i 44:a:l,w': 'Niarembe','i 44:b:1,6': 'Shahalam','i 44:b:e,n,x': 'Elbeuf','i 44:b:e,n,z15': 'Sedgwick','i 44:c:1,5': 'Madigan','i 44:c:e,n,x': 'Bergerac','i 44:c:e,n,z15': 'Quebec','i 44:d:1,5': 'Bobo','i 44:d:e,n,x': 'Kermel','i 44:d:e,n,z15': 'Fischerstrasse','i 1,44:d:z35': 'Palamaner','ii 1,44:e,n,x:1,6': 'II [1],44:e,n,x:1,6','i 44:f,g:–': 'Vleuten','i 1,44:g,m,s:1,6': 'Gamaba','i 44:g,s,t:1,7': 'Splott','ii 44:g,t:z42': 'II 44:g,t:z42','iiib 1,44:g,t:1,5:z42': 'IIIb [1],44:g,t:1,5:z42','i 44:g,z51:–': 'Carswell','iv 44:g,z51:–': 'IV 44:g,z51:–','i 44:m,t:–': 'Muguga','ii 1,44:m,t:z42': 'II [1],44:m,t:z42','i 1,44:i:e,n,z15': 'Maritzburg','i 44:k:e,n,z15': 'Lawra','i 44:l,z28:1,5': 'Malika','i 44:r:1,5': 'Albertbanjul','i 44:r:e,n,z15': 'Brefet','v 44:r:–': 'V 44:r:–','i 44:z:1,5': 'Brackenridge','i 44:z:l,w': 'Uhlenhorst','i 44:z:e,n,x': 'Bolama','i 44:z:e,n,z15': 'Goodmayes','i 44:z4,z23:–': 'Kua','i 1,44:z4,z23:e,n,z15': 'Ploufragan','ii 44:z4,z23:–': 'II 44:z4,z23:–','iiia 44:z4,z23:–': 'IIIa 44:z4,z23:–','iv 44:z4,z23:–': 'IV 44:z4,z23:–','iiia 44:z4,z23,z32:–': 'IIIa 44:z4,z23,z32:–','i 44:z4,z24:–': 'Christiansborg','iiia 44:z4,z24:–': 'IIIa 44:z4,z24:–','iv 44:z4,z24:–': 'IV 44:z4,z24:–','iiia 44:z4,z32:–': 'IIIa 44:z4,z32:–','iv 1,44:z4,z32:–': 'IV [1],44:z4,z32:–','i 1,44:z10:1,7': 'Guinea','i 1,44:z10:e,n,x': 'Llobregat','ii 44:z29:e,n,x:z42': 'II 44:z29:e,n,x:z42','i 44:z29:–': 'Zinder','iv 44:z29:–': 'IV 44:z29:–','iv 44:z36,z38:–': 'IV 44:z36,[z38]:–','i 44:z38:–': 'Koketime','ii 1,44:z39:e,n,x,z15': 'II [1],44:z39:e,n,x,z15','v 44:z39:–': 'V 44:z39:–','vi 45:a:e,n,x': 'VI 45:a:e,n,x','i 45:a:e,n,z15': 'Meekatharra','ii 45:a:z10': 'II 45:a:z10','i 45:b:1,5': 'Riverside','i 45:b:1,6': 'Leinster','i 45:b:e,n,z15': 'Fomeco','i 45:c:e,n,x': 'Deversoir','i 45:d:1,6': 'Dugbe','i 45:d:e,n,x': 'Karachi','i 45:d:e,n,z15': 'Warmsen','i 45:f,g:–': 'Suelldorf','i 45:g,m,s,t:–': 'Tornow','ii 45:g,m,s,t:1,5': 'II 45:g,m,s,t:1,5','ii 45:g,m,s,t:e,n,x': 'II 45:g,m,s,t:e,n,x','ii 45:g,m,t:e,n,x,z15': 'II 45:g,m,t:e,n,x,z15','i 45:g,s,t:–': 'Binningen','iiia 45:g,z51:–': 'IIIa 45:g,z51:–','iv 45:g,z51:–': 'IV 45:g,z51:–','ii 45:m,t:1,5': 'II 45:m,t:1,5','i 45:m,t:–': 'Apapa','i 45:k:1,5': 'Verviers','i 45:k:1,7': 'Casablanca','i 45:k:e,n,z15': 'Cairns','i 45:l,v:e,n,z15': 'Imo','i 45:r:e,n,z15': 'Kofandoka','ii 45:z:1,5': 'II 45:z:1,5','i 45:z:e,n,z15': 'Yopougon','ii 45:z:z39': 'II 45:z:z39','iiia 45:z4,z23:–': 'IIIa 45:z4,z23:–','iv 45:z4,z23:–': 'IV 45:z4,z23:–','i 45:z4,z24:–': 'Transvaal','iiia 45:z4,z24:–': 'IIIa 45:z4,z24:–','iiia 45:z4,z32:–': 'IIIa 45:z4,z32:–','i 45:z10:–': 'Aprad','i 45:z29:–:z45': 'Jodhpur','ii 45:z29:1,5': 'II 45:z29:1,5','ii 45:z29:e,n,x': 'II 45:z29:e,n,x','ii 45:z29:z42': 'II 45:z29:z42','iiia 45:z29:–': 'IIIa 45:z29:–','i 45:z35:1,5': 'Lattenkamp','i 45:z36:–': 'Balcones','iiia 45:z36:–': 'IIIa 45:z36:–','iv 45:z36,z38:–': 'IV 45:z36,z38:–','ii 47:a:1,5': 'II 47:a:1,5','ii 47:a:e,n,x,z15': 'II 47:a:e,n,x,z15','i 47:b:1,2': 'Wenatchee','ii 47:b:1,5': 'II 47:b:1,5','ii 47:b:e,n,x,z15': 'II 47:b:e,n,x,z15','i 47:b:z6': 'Sya','ii 47:b:z6': 'II 47:b:z6','iiib 47:c:1,5,7': 'IIIb 47:c:1,5,7','i 47:c:1,6:z78': 'Kodjovi','iiib 47:c:e,n,x,z15:z57': 'IIIb 47:c:e,n,x,z15:[z57]','iiib 47:c:z': 'IIIb 47:c:z','i 47:c:z6': 'Braeswood','iiib 47:c:z35': 'IIIb 47:c:z35','ii 47:d:1,5': 'II 47:d:1,5','i 47:d:e,n,x:z58': 'Stellingen','ii 47:d:e,n,x,z15': 'II 47:d:e,n,x,z15','ii 47:d:z39': 'II 47:d:z39','ii 47:e,n,x,z15:1,6': 'II 47:e,n,x,z15:1,6','i 1,47:f,g:–': 'Sljeme','i 1,47:g,m:–': 'Luke','i 47:g,m,s:–': 'Carshalton','ii 47:g,t:e,n,x': 'II 47:[g,t]:e,n,x','iiia 47:g,z51:–': 'IIIa 47:g,z51:–','i 47:m,t:e,n,z15': 'Mesbit','iiib 47:i:e,n,x,z15:z50': 'IIIb 47:i:e,n,x,z15:[z50]','i 47:i:e,n,z15': 'Bergen','iiib 47:i:z': 'IIIb 47:i:z','iiib 47:i:z35': 'IIIb 47:i:z35','iiib 47:i:z53:z57,z84': 'IIIb 47:i:z53:[z57],[z84]','i 47:k:1,2': 'Staoueli','i 47:k:1,5': 'Bootle','iiib 47:k:1,5,7': 'IIIb 47:k:1,5,7','i 47:k:1,6:z58': 'Dahomey','iiib 47:k:e,n,x,z15': 'IIIb 47:k:e,n,x,z15','i 47:k:e,n,z15': 'Lyon','iiib 47:k:z': 'IIIb 47:k:z','iiib 47:k:z35': 'IIIb 47:k:z35','iiib 47:k:z53:z84': 'IIIb 47:k:z53:[z84]','iiib 47:l,v:1,5,7:z50': 'IIIb 47:l,v:1,[5],7:[z50]','i 47:l,v:e,n,x': 'Drac','iiib 47:l,v:e,n,x,z15': 'IIIb 47:l,v:e,n,x,z15','iiib 47:l,v:z': 'IIIb 47:l,v:z','iiib 47:l,v:z35': 'IIIb 47:l,v:z35','iiib 47:l,v:z53': 'IIIb 47:l,v:z53','iiib 47:l,v:z57': 'IIIb 47:l,v:z57','iv 47:l,v:–': 'IV 47:l,v:–','i 1,47:l,z13,z28:e,n,z15': 'Teshie','iiib 47:r:e,n,x,z15': 'IIIb 47:r:e,n,x,z15','i 47:r:1,2': 'Dapango','iiib 47:r:1,5,7': 'IIIb 47:r:1,5,7','iiib 47:r:z': 'IIIb 47:r:z','iiib 47:r,i:z35': 'IIIb 47:r,[i]:z35','iiib 47:r:z53:z70,z74,z77,z90': 'IIIb 47:r:z53:[z70],[z74],[z77],[z90]','i 47:y:1,6': 'Moualine','i 47:y:e,n,x': 'Blitta','i 47:z:1,5': 'Mountpleasant','i 47:z:1,6': 'Kaolack','ii 47:z:e,n,x,z15': 'II 47:z:e,n,x,z15','ii 47:z:z6': 'II 47:z:z6','i 47:z4,z23:1,2': 'Forges','i 47:z4,z23:e,n,z15': 'Tabligbo','i 47:z4,z23:1,6': 'Fehrbellin','i 47:z4,z23:z6:z45,z58': 'Bere','i 47:z4,z23:l,w': 'Binche','iiia 47:z4,z23:–': 'IIIa 47:z4,z23:–','i 47:z4,z24:–': 'Tamberma','ii 47:z6:1,6': 'II 47:z6:1,6','iiib 47:z10:1,5,7': 'IIIb 47:z10:1,5,7','i 47:z10:e,n,z15': 'Namoda','iiib 47:z10:z': 'IIIb 47:z10:z','iiib 47:z10:z35': 'IIIb 47:z10:z35','ii 47:z29:e,n,x,z15': 'II 47:z29:e,n,x,z15','i 47:z29:–': 'Ekpoui','iiia 47:z29:–': 'IIIa 47:z29:–','i 47:z35:z6': 'Yombesali','i 47:z35:e,n,z15': 'Bingerville','iv 47:z36:–': 'IV 47:z36:–','i 47:z38:–': 'Alexanderplatz','i 47:z44:–': 'Quinhon','iiib 47:z52:1,5:z54': 'IIIb 47:z52:1,5:z54','iiib 47:z52:1,5,7': 'IIIb 47:z52:1,5,7','iiib 47:z52:e,n,x,z15': 'IIIb 47:z52:e,n,x,z15','iiib 47:z52:z': 'IIIb 47:z52:z','iiib 47:z52:z35': 'IIIb 47:z52:z35','iiib 47:z53:–:z90': 'IIIb 47:z53:–:[z90]','i 48:a:1,5,7': 'Hisingen','ii 48:a:z6': 'II 48:a:z6','ii 48:a:z39': 'II 48:a:z39','ii 48:b:z6': 'II 48:b:z6','ii 48:b:e,n,x,z15': 'II 48:b:e,n,x,z15','v 48:b:–': 'V 48:b:–','iiib 48:c:z': 'IIIb 48:c:z','ii 48:d:1,2': 'II 48:d:1,2','ii 48:d:z6': 'II 48:d:z6','i 48:d:–:z58': 'Buckeye','i 48:e,h:1,5': 'Fitzroy','ii 48:e,n,x,z15:z6': 'II 48:e,n,x,z15:z6','ii 48:g,m,t:–': 'II 48:g,m,t:–','iiia 48:g,z51:–': 'IIIa 48:g,z51:–','iv 48:g,z51:–': 'IV 48:g,z51:–','iiib 48:i:z:z72': 'IIIb 48:i:z:[z72]','iiib 48:i:z35:z57': 'IIIb 48:i:z35:[z57]','iiib 48:i:z53': 'IIIb 48:i:z53','iiib 48:i:z61': 'IIIb 48:i:z61','v 48:i:–': 'V 48:i:–','iiib 48:k:1,5,7': 'IIIb 48:k:1,5,(7)','ii 48:k:e,n,x,z15': 'II 48:k:e,n,x,z15','iiib 48:k:e,n,x,z15': 'IIIb 48:k:e,n,x,z15','i 48:k:e,n,z15': 'Dahlem','iiib 48:k:z': 'IIIb 48:k:z','iiib 48:k:z35:z75': 'IIIb 48:k:z35:[z75]','ii 48:k:z39': 'II 48:k:z39','iiib 48:k:z53': 'IIIb 48:k:z53','iiib 48:k:z57': 'IIIb 48:k:z57','i 48:l,v:1,5': 'Australia','iiib 48:l,v:1,5,7:z47,z50,z89': 'IIIb 48:l,v:1,5,(7):[z47],[z50],[z89]','iiib 48:l,v:z': 'IIIb 48:l,v:z','iiib 48:l,w:1,5,7:z50': 'IIIb 48:l,w:1,5,7:[z50]','iiib 48:r:e,n,x,z15': 'IIIb 48:r:e,n,x,z15','iiib 48:r:z': 'IIIb 48:r:z','i 48:z:1,5:z58': 'Toucra','ii 48:z:1,5': 'II 48:z:1,5','iiib 48:z:1,5,7': 'IIIb 48:z:1,5,7','iiia 48:z4,z23:–': 'IIIa 48:z4,z23:–','iv 48:z4,z23:–': 'IV 48:z4,z23:–','iiia 48:z4,z23,z32:–': 'IIIa 48:z4,z23,z32:–','i 48:z4,z24:–': 'Djakarta','iiia 48:z4,z24:–': 'IIIa 48:z4,z24:–','iiib 48:z4,z24:–': 'IIIb 48:z4,z24:–','iv 48:z4,z24:–': 'IV 48:z4,z24:–','iiia 48:z4,z32:–': 'IIIa 48:z4,z32:–','iv 48:z4,z32:–': 'IV 48:z4,z32:–','ii 48:z10:1,5': 'II 48:z10:[1,5]','vi 48:z10:1,5': 'VI 48:z10:1,5','ii 48:z10:1,6': 'II 48:z10:1,6','i 48:z10:e,n,x': 'Isaszeg','iiib 48:z10:e,n,x,z15': 'IIIb 48:z10:e,n,x,z15','iiib 48:z10:z': 'IIIb 48:z10:z','ii 48:z29:–': 'II 48:z29:–','iiia 48:z29:–': 'IIIa 48:z29:–','iv 48:z29:–': 'IV 48:z29:–','iiib 48:z35:z52': 'IIIb 48:z35:z52','v 48:z35:–': 'V 48:z35:–','iiia 48:z36:–': 'IIIa 48:z36:–','iv 48:z36,z38:–': 'IV 48:z36,[z38]:–','ii 48:z39:z81': 'II 48:z39:z81','v 48:z39:–': 'V 48:z39:–','v 48:z41:–': 'V 48:z41:–','iiib 48:z52:e,n,x,z15': 'IIIb 48:z52:e,n,x,z15','iiib 48:z52:z': 'IIIb 48:z52:z','v 48:z65:–': 'V 48:z65:–','v 48:z81:–': 'V 48:z81:–','ii 48:z81:1,5,7': 'II 48:z81:1,5,7','iv 50:a:–': 'IV 50:a:–','i 50:b:e,n,x': 'Rochdale','ii 50:b:z6': 'II 50:b:z6','iv 50:b:–': 'IV 50:b:–','i 50:d:1,5:z82': 'Hemingford','iv 50:d:–': 'IV 50:d:–','ii 50:e,n,x:1,7': 'II 50:e,n,x:1,7','ii 50:g,m,s,t:1,5': 'II 50:g,[m],s,t:[1,5]','iiia 50:g,z51:–': 'IIIa 50:g,z51:–','iv 50:g,z51:–': 'IV 50:g,z51:–','ii 50:g,z62:e,n,x': 'II 50:g,z62:e,n,x','ii 50:m,t:z6:z42': 'II 50:m,t:z6:z42','iiib 50:i:1,5,7': 'IIIb 50:i:1,5,7','iiib 50:i:e,n,x,z15': 'IIIb 50:i:e,n,x,z15','iiib 50:i:z': 'IIIb 50:i:z','iiib 50:i:z53': 'IIIb 50:i:z53','iiib 50:k:1,5,7': 'IIIb 50:k:1,5,7','ii 50:k:e,n,x:z42': 'II 50:k:e,n,x:z42','iiib 50:k:e,n,x,z15': 'IIIb 50:k:e,n,x,z15','iiib 50:k:z:z50,z57,z68,z86': 'IIIb 50:k:z:[z50],[z57],[z68],[z86]','ii 50:k:z6': 'II 50:k:z6','iiib 50:k:z35': 'IIIb 50:k:z35','iiib 50:k:z53': 'IIIb 50:k:z53','i 50:l,v:1,2': 'Fass','iiib 50:l,v:e,n,x,z15': 'IIIb 50:l,v:e,n,x,z15','iiib 50:l,v:z': 'IIIb 50:l,v:z','iiib 50:l,v:z35': 'IIIb 50:l,v:z35','iiib 50:l,v:z57': 'IIIb 50:l,v:z57','vi 50:l,v:z67': 'VI 50:l,v:z67','ii 50:l,w:e,n,x,z15:z42': 'II 50:l,w:e,n,x,z15:z42','ii 50:l,z28:z42': 'II 50:l,z28:z42','iiib 50:r:1,5,7': 'IIIb 50:r:1,5,(7)','iiib 50:r:e,n,x,z15': 'IIIb 50:r:e,n,x,z15','iiib 50:r:z:z67': 'IIIb 50:r:z:[z67]','iiib 50:r:z35:z58': 'IIIb 50:r:z35:[z58]','iiib 50:r:z53': 'IIIb 50:r:z53','i 50:y:1,6': 'Dougi','ii 50:z:e,n,x': 'II 50:z:e,n,x','iiib 50:z:z52': 'IIIb 50:z:z52','iiia 50:z4,z23:–': 'IIIa 50:z4,z23:–','iv 50:z4,z23:–': 'IV 50:z4,z23:–','iiia 50:z4,z23,z32:–': 'IIIa 50:z4,z23,z32:–','iiia 50:z4,z24:–': 'IIIa 50:z4,z24:–','iv 50:z4,z24:–': 'IV 50:z4,z24:–','iiia 50:z4,z32:–': 'IIIa 50:z4,z32:–','iv 50:z4,z32:–': 'IV 50:z4,z32:–','iiib 50:z10:z:z56': 'IIIb 50:z10:z:[z56]','ii 50:z10:z6:z42': 'II 50:z10:z6:z42','iiib 50:z10:z53': 'IIIb 50:z10:z53','i 50:z29:–': 'Ivorycoast','iiia 50:z29:–': 'IIIa 50:z29:–','iiia 50:z36:–': 'IIIa 50:z36:–','ii 50:z42:1,7': 'II 50:z42:1,7','iiib 50:z52:1,5,7': 'IIIb 50:z52:1,5,7','iiib 50:z52:z35': 'IIIb 50:z52:z35','iiib 50:z52:z53': 'IIIb 50:z52:z53','iv 51:a:–': 'IV 51:a:–','i 51:a:1,2': 'Windsheim','i 51:a:e,n,x': 'Tione','i 51:b:1,5': 'Karaya','iv 51:b:–': 'IV 51:b:–','ii 51:c:–': 'II 51:c:–','i 1,51:d:1,5': 'Gokul','i 51:e,h:1,2': 'Meskin','ii 51:g,s,t:e,n,x': 'II 51:g,s,t:e,n,x','iiia 51:g,z51:–': 'IIIa 51:g,z51:–','i 51:m,t:–': 'Djinten','i 51:i:1,2': 'Elmdon','i 51:i:1,5': 'Kabete','i 51:k:e,n,z15': 'Dan','iiib 51:k:z35': 'IIIb 51:k:z35','i 51:l,v:1,2': 'Harcourt','i 51:l,v:1,5': 'Overschie','i 51:l,v:e,n,x': 'Dadzie','iiib 51:l,v:z': 'IIIb 51:l,v:z','i 51:l,z28:1,5': 'Moundou','ii 51:l,z28:z6': 'II 51:l,z28:z6','ii 51:l,z28:z39': 'II 51:l,z28:z39','i 51:r:e,n,x': 'Mali','i 51:r,i:l,z13,z28': 'Lutetia','i 51:y:1,2': 'Rosslyn','i 51:z:1,5': 'Antsalova','i 1,51:z:1,6': 'Treforest','i 51:z:e,n,z15': 'Lechler','iiia 51:z4,z23:–': 'IIIa 51:z4,z23:–','iv 51:z4,z23:–': 'IV 51:z4,z23:–','iiia 51:z4,z24:–': 'IIIa 51:z4,z24:–','iiia 51:z4,z32:–': 'IIIa 51:z4,z32:–','i 51:z10:1,5': 'Bergues','ii 51:z29:e,n,x,z15': 'II 51:z29:e,n,x,z15','iiia 51:z36:–': 'IIIa 51:z36:–','ii 51:–:1,7': 'II 51:–:1,7','i 52:a:1,5': 'Uithof','i 52:a:e,n,z15': 'Ord','i 52:b:1,5': 'Molesey','i 52:b:e,n,x': 'Flottbek','ii 52:c:k': 'II 52:c:k','i 52:d:1,5': 'Utrecht','ii 52:d:e,n,x,z15': 'II 52:d:e,n,x,z15','ii 52:d:z39': 'II 52:d:z39','i 52:e,h:1,6': 'Butare','i 52:e,h:1,7': 'Derkle','i 52:g,t:–': 'Saintemarie','ii 52:g,t:–': 'II 52:g,t:–','i 52:k:1,5': 'Bordeaux','iiib 52:k:e,n,x,z15': 'IIIb 52:k:e,n,x,z15','iiib 52:k:z35': 'IIIb 52:k:z35','iiib 52:k:z53': 'IIIb 52:k:z53','iiib 52:l,v:z53': 'IIIb 52:l,v:z53','i 52:l,w:1,5': 'Marsabit','ii 52:z:z39': 'II 52:z:z39','iiib 52:z:z52': 'IIIb 52:z:z52','ii 52:z39:1,5,7': 'II 52:z39:1,5,7','ii 52:z44:1,5,7': 'II 52:z44:1,5,7','ii 53:c:1,5:r1': 'II 53:c:1,5:[R1…]','ii 53:d:1,5': 'II 53:d:1,5','ii 1,53:d:z39': 'II [1],53:d:z39','ii 53:d:z42': 'II 53:d:z42','iiia 53:g,z51:–': 'IIIa 53:g,z51:–','iv 1,53:g,z51:–': 'IV [1],53:g,z51:–','iiib 53:i:z': 'IIIb 53:i:z','iiib 53:k:e,n,x,z15': 'IIIb 53:k:e,n,x,z15','iiib 53:k:z': 'IIIb 53:k:z','iiib 53:k:z35': 'IIIb 53:(k):z35','iiib 53:k:z53': 'IIIb 53:k:z53','iiib 53:l,v:e,n,x,z15': 'IIIb 53:l,v:e,n,x,z15','iiib 53:l,v:z': 'IIIb 53:l,v:z','iiib 53:l,v:z35': 'IIIb 53:l,v:z35','ii 53:l,z28:e,n,x': 'II 53:l,z28:e,n,x','ii 53:l,z28:z6': 'II 53:l,z28:z6','ii 53:l,z28:z39': 'II 53:l,z28:z39','iiib 53:r:z': 'IIIb 53:r:z','iiib 53:r:z35': 'IIIb 53:r:z35','iiib 53:r:z68': 'IIIb 53:r:z68','ii 53:z:1,5': 'II 53:z:1,5','iiib 53:z:1,5,7': 'IIIb 53:z:1,5,(7)','ii 53:z:z6': 'II 53:z:z6','iiia 53:z4,z23:–': 'IIIa 53:z4,z23:–','iv 53:z4,z23:–': 'IV 53:z4,z23:–','iiia 53:z4,z23,z32:–': 'IIIa 53:z4,z23,z32:–','ii 53:z4,z24:–': 'II 53:z4,z24:–','iiia 53:z4,z24:–': 'IIIa 53:z4,z24:–','iiib 53:z10:z': 'IIIb 53:z10:z','iiib 53:z10:z35': 'IIIb 53:z10:z35','iiia 53:z29:–': 'IIIa 53:z29:–','iv 1,53:z36,z38:–': 'IV [1],53:z36,z38:–','iiib 53:z52:z35': 'IIIb 53:z52:z35','iiib 53:z52:z53': 'IIIb 53:z52:z53','i 53:–:1,6': 'Leda','i 21,54:b:e,n,x': 'Tonev','i 54:e,h:1,5': 'Winnipeg','i 3,54:e,h:1,6': 'Rossleben','i 54:f,g,s:–': 'Borreze','i 3,54:g,s,t:–': 'Uccle','i 4,12,54:m,t:–': 'Newholland','i 8,20,54:i:z6': 'Poeseldorf','i 6,7,54:k:1,5': 'Ochsenwerder','i 54:r:1,5': 'Czernyring','i 3,15,54:y:1,5': 'Steinwerder','i 54:z4,z23:–': 'Yerba','i 54:z10:e,n,x': 'Canton','i 54:z10:e,n,z15': 'Barry','i 54:z29:–': 'Mundubbera','ii 55:k:z39': 'II 55:k:z39','ii 56:b:1,5': 'II 56:b:[1,5]','ii 56:d:–': 'II 56:d:–','ii 56:e,n,x:1,7': 'II 56:e,n,x:1,7','ii 56:l,v:z39': 'II 56:l,v:z39','ii 56:l,z28:–': 'II 56:l,z28:–','ii 56:z:z6': 'II 56:z:z6','iiia 56:z4,z23:–': 'IIIa 56:z4,z23:–','iiia 56:z4,z23,z32:–': 'IIIa 56:z4,z23,z32:–','ii 56:z10:e,n,x': 'II 56:z10:e,n,x','iiia 56:z29:–': 'IIIa 56:z29:–','i 57:a:z6': 'Antonio','ii 57:a:z42': 'II 57:a:z42','i 57:b:1,7': 'Maryland','i 57:b:e,n,z15': 'Batonrouge','iiib 57:c:e,n,x,z15': 'IIIb 57:c:e,n,x,z15','iiib 57:c:z:z70,z90': 'IIIb 57:c:z:[z70],[z90]','ii 57:d:1,5': 'II 57:d:1,5','ii 57:g,m,s,t:z42': 'II 57:g,[m],s,t:z42','ii 57:g,t:–': 'II 57:g,t:–','iiib 57:i:e,n,x,z15': 'IIIb 57:i:e,n,x,z15','iiib 57:i:z': 'IIIb 57:i:z','iiib 57:k:e,n,x,z15': 'IIIb 57:k:e,n,x,z15','iv 57:z4,z23:–': 'IV 57:z4,z23:–','iiib 57:z10:z': 'IIIb 57:z10:z','ii 57:z29:z42': 'II 57:z29:z42','ii 57:z39:e,n,x,z15': 'II 57:z39:e,n,x,z15','ii 57:z42:1,6:z53': 'II 57:z42:1,6:z53','ii 58:a:z6': 'II 58:a:z6','ii 58:b:1,5': 'II 58:b:1,5','ii 58:c:z6': 'II 58:c:z6','ii 58:d:z6': 'II 58:d:z6','iiib 58:i:e,n,x,z15': 'IIIb 58:i:e,n,x,z15','iiib 58:i:z53': 'IIIb 58:i:z53','iiib 58:k:z': 'IIIb 58:k:z','iiib 58:l,v:e,n,x,z15': 'IIIb 58:l,v:e,n,x,z15','iiib 58:l,v:z35': 'IIIb 58:l,v:z35','ii 58:l,z13,z28:1,5': 'II 58:l,z13,z28:1,5','ii 58:l,z13,z28:z6': 'II 58:l,z13,z28:z6','iiib 58:r:e,n,x,z15': 'IIIb 58:r:e,n,x,z15','iiib 58:r:z': 'IIIb 58:r:z','iiib 58:r:z53:z47,z57,z70,z73': 'IIIb 58:r:z53:[z47],[z57],[z70],[z73]','ii 58:z6:1,6': 'II 58:z6:1,6','ii 58:z10:1,6': 'II 58:z10:1,6','iiib 58:z10:e,n,x,z15': 'IIIb 58:z10:e,n,x,z15','ii 58:z10:z6': 'II 58:z10:z6','iiib 58:z10:z53:z50': 'IIIb 58:z10:z53:[z50]','ii 58:z39:e,n,x,z15': 'II 58:z39:e,n,x,z15','iiib 58:z52:z': 'IIIb 58:z52:z','iiib 58:z52:z35': 'IIIb 58:z52:z35','iiib 59:c:e,n,x,z15': 'IIIb 59:c:e,n,x,z15','iiib 59:i:e,n,x,z15': 'IIIb 59:i:e,n,x,z15','iiib 59:i:z': 'IIIb 59:i:z','iiib 59:i:z35:z84': 'IIIb 59:i:z35:[z84]','iiib 59:k:e,n,x,z15': 'IIIb 59:(k):e,n,x,z15','ii 59:k:z65': 'II 59:k:z65','iiib 59:k:z': 'IIIb 59:(k):z','iiib 59:k:z35': 'IIIb 59:(k):z35','iiib 59:k:z53': 'IIIb 59:k:z53','iiib 59:l,v:z': 'IIIb 59:l,v:z','iiib 59:l,v:z53': 'IIIb 59:l,v:z53','iiib 59:r:z35': 'IIIb 59:r:z35','ii 1,59:z:z6': 'II [1],59:z:z6','iiia 59:z4,z23:–': 'IIIa 59:z4,z23:–','iiib 59:z10:z53': 'IIIb 59:z10:z53','iiib 59:z10:z57': 'IIIb 59:z10:z57','iiia 59:z29:–': 'IIIa 59:z29:–','iiia 59:z36:–': 'IIIa 59:z36:–','vi 59:z36:–': 'VI 59:z36:–','iiib 59:z52:z53': 'IIIb 59:z52:z53','ii 60:b:–:r1': 'II 60:b:–:[R1…]','ii 60:g,m,t:z6': 'II 60:g,m,t:z6','iiib 60:i:e,n,x,z15:z50': 'IIIb 60:i:[e,n,x,z15]:[z50]','iiib 60:i:z:z50': 'IIIb 60:i:[z]:[z50]','iiib 60:i:z35:z50': 'IIIb 60:i:[z35]:[z50]','iiib 60:k:z': 'IIIb 60:k:z','iiib 60:k:z35': 'IIIb 60:k:z35','iiib 60:k:z53': 'IIIb 60:(k):z53','iiib 60:l,v:z': 'IIIb 60:l,v:z','iiib 60:r:e,n,x,z15': 'IIIb 60:r:e,n,x,z15','iiib 60:r:z': 'IIIb 60:r:z','iiib 60:r:z35': 'IIIb 60:r:z35','iiib 60:r:z53': 'IIIb 60:r:z53','ii 60:z:e,n,x': 'II 60:z:e,n,x','iiib 60:z10:z': 'IIIb 60:z10:z','iiib 60:z10:z35': 'IIIb 60:z10:z35','ii 60:z10:z39': 'II 60:z10:z39','iiib 60:z10:z53': 'IIIb 60:z10:z53','ii 60:z29:e,n,x': 'II 60:z29:e,n,x','v 60:z41:–': 'V 60:z41:–','iiib 60:z52:1,5,7': 'IIIb 60:z52:1,5,[7]','iiib 60:z52:z': 'IIIb 60:z52:z','iiib 60:z52:z35': 'IIIb 60:z52:z35','iiib 60:z52:z53': 'IIIb 60:z52:z53','iiib 61:c:1,5,7': 'IIIb 61:c:1,5,(7)','iiib 61:c:z35': 'IIIb 61:c:z35','iiib 61:i:e,n,x,z15': 'IIIb 61:i:e,n,x,z15','iiib 61:i:z': 'IIIb 61:i:z','iiib 61:i:z35': 'IIIb 61:i:z35','iiib 61:i:z53': 'IIIb 61:i:z53','iiib 61:k:1,5,7': 'IIIb 61:k:1,5,(7)','iiib 61:k:z35': 'IIIb 61:k:z35','iiib 61:k:z53': 'IIIb 61:(k):z53','iiib 61:l,v:1,5,7:z57': 'IIIb 61:l,v:1,5,7:[z57]','iiib 61:l,v:z': 'IIIb 61:l,v:z','iiib 61:l,v:z35': 'IIIb 61:l,v:z35','iiib 61:r:1,5,7': 'IIIb 61:r:1,5,7','iiib 61:r:z': 'IIIb 61:r:z','iiib 61:r:z35': 'IIIb 61:r:z35','iiib 61:r:z53:z47,z50': 'IIIb 61:r:z53:[z47],[z50]','iiib 61:z10:z35': 'IIIb 61:z10:z35','v 61:z35:–': 'V 61:z35:–','iiib 61:z52:1,5,7': 'IIIb 61:z52:1,5,7','iiib 61:z52:z': 'IIIb 61:z52:z','iiib 61:z52:z35': 'IIIb 61:z52:z35','iiib 61:z52:z53': 'IIIb 61:z52:z53','iiia 62:g,z51:–': 'IIIa 62:g,z51:–','iiia 62:z4,z23:–': 'IIIa 62:z4,z23:–','iiia 62:z4,z32:–': 'IIIa 62:z4,z32:–','iiia 62:z29:–': 'IIIa 62:z29:–','iiia 62:z36:–': 'IIIa 62:z36:–','iiia 63:g,z51:–': 'IIIa 63:g,z51:–','iiib 63:k:z': 'IIIb 63:(k):z','iiia 63:z4,z23:–': 'IIIa 63:z4,z23:–','iiia 63:z4,z32:–': 'IIIa 63:z4,z32:–','iiia 63:z36:–': 'IIIa 63:z36:–','iiib 65:c:1,5,7': 'IIIb 65:c:1,5,7','iiib 65:c:z': 'IIIb 65:c:z','iiib 65:c:z53': 'IIIb 65:c:z53','ii 65:g,t:–': 'II 65:g,t:–','iiib 65:i:e,n,x,z15': 'IIIb 65:i:e,n,x,z15','iiib 65:i:z': 'IIIb 65:i:z','iiib 65:k:z': 'IIIb 65:(k):z','iiib 65:k:z35': 'IIIb 65:(k):z35','iiib 65:k:z53': 'IIIb 65:(k):z53','iiib 65:l,v:e,n,x,z15': 'IIIb 65:l,v:e,n,x,z15','iiib 65:l,v:z': 'IIIb 65:l,v:z','iiib 65:l,v:z35': 'IIIb 65:l,v:z35','iiib 65:l,v:z53': 'IIIb 65:l,v:z53','iiib 65:r:z35': 'IIIb 65:r:z35','iiib 65:z10:e,n,x,z15': 'IIIb 65:z10:e,n,x,z15','iiib 65:z10:z': 'IIIb 65:z10:z','iiib 65:z52:e,n,x,z15': 'IIIb 65:z52:e,n,x,z15','iiib 65:z52:z': 'IIIb 65:z52:z','iiib 65:z52:z35': 'IIIb 65:z52:z35','iiib 65:z52:z53': 'IIIb 65:z52:z53','ii 65:–:1,6': 'II 65:–:1,6','v 66:z35:–': 'V 66:z35:–','v 66:z39:–': 'V 66:z39:–','v 66:z41:–': 'V 66:z41:–','v 66:z65:–': 'V 66:z65:–','v 66:z81:–': 'V 66:z81:–','i 67:r:1,2': 'Crossness','i 3,15:y:1,2:z45': 'Amager var. 15+','i 3,15:g,m,s:–': 'Amsterdam var. 15+','i 3,15:e,h:1,6:z64': 'Anatum var. 15+','i 3,15,34:e,h:1,6:z64': 'Anatum var. 15+,34+','i 3,15,34:b:1,5': 'Butantan var. 15+,34+','i 3,15:l,v:1,7:d,z77': 'Give var. 15+','i 3,15,34:l,v:1,7:d,z77': 'Give var. 15+,34+','i 3,15:a:1,5': 'Goelzau var. 15+','i 3,15:z10:1,5:z49': 'Lexington var. 15+','i 3,15,34:z10:1,5:z49': 'Lexington var. 15+,34+','i 3,15:l,v:1,6': 'London var. 15+','i 3,15:e,h:l,w': 'Meleagridis var. 15+','i 3,15,34:e,h:l,w': 'Meleagridis var. 15+,34+','i 3,15:e,h:1,5:z48': 'Muenster var. 15+','i 3,15,34:e,h:1,5:z48': 'Muenster var. 15+,34+','i 3,15:l,v:1,2': 'Nchanga var. 15+','i 3,15:e,h:1,7': 'Nyborg var. 15+','i 3,15:y:1,5': 'Orion var. 15+','i 3,15,34:y:1,5': 'Orion var. 15+,34+','i 3,15,34:a:1,7': 'Oxford var. 15+,34+','i 3,15:d:1,5': 'Shangani var. 15+','i 3,15:d:e,n,x': 'Souza var. 15+','i 3,15:y:z6': 'Stockholm var. 15+','i 3,15:l,z13:1,5': 'Uganda var. 15+','i 3,15:e,h:1,2': 'Vejle var. 15+','i 3,15:e,h:1,2:z27': 'Vejle var. 15+,Rz27','i 3,15:r:z6': 'Weltevreden var. 15+','i 3,15:g,s,t:–:z37': 'Westhampton var. 15+','i 3,15,34:g,s,t:–:z37': 'Westhampton var. 15+,34+'}

# Names withdrawn in the 2007 scheme [1]
# Note - This dict includes two keys for names which include a subspecies designation:
#        with subsp (e.g. II Acres) and without subsp (Acres).
wklm_old_to_new = {'ii acres': 'II 1,13,23:b:[1,5]:z42', 'ii alexander': 'II 3,10:z:1,5', 'ii alsterdorf': 'II 1,40:g,[m],[s],t:[1,5]', 'ii angola': 'II 1,9,12:z:z6', 'ardwick': 'Rissen var. 14+', 'iv argentina': 'IV 6,7:z36:–', 'arkansas': 'Muenster var. 15+,34+', 'ii artis': 'II 56:b:[1,5]', 'ii askraal': 'II 51:l,z28:z6', 'atherton': 'Waycross', 'ii atra': 'II 50:m,t:z6:z42', 'ii bacongo': 'II 6,7:z36:z42', 'v balboa': 'V 48:z41:–', 'bantam': 'Meleagridis', 'ii baragwanath': 'II 6,8:m,t:1,5', 'ii basel': 'II 58:l,z13,z28:1,5', 'batavia': 'Lexington', 'ii bechuana': '1,4,12,[27]:g,[m],t:[1,5]', 'ii bellville': 'II 16:e,n,x:1,(5),7', 'ii beloha': 'II 18:z36:–', 'iv bern': 'IV 40:z4,z32:–', 'ii betioky': 'II 59:k:z65', 'ii bilthoven': 'II 47:a:1,5', 'binza': 'Orion var. 15+', 'ii blankenese': 'II 1,9,12:b:z6', 'ii bleadon': 'II 17:g,t:[e,n,x,z15]', 'ii bloemfontein': 'II 6,7:b:e,n,x:z42', 'iv bockenheim': 'IV 1,53:z36,z38:–', 'ii boksburg': 'II 40:g,m,s,t:e,n,x', 'iv bonaire': 'IV 50:z4,z32:–', 'v bongor': 'V 48:z35:–', 'vi bornheim': 'VI 1,6,14,25:z10:1,(2),7', 'bornum': 'Lille var. 14+', 'ii boulders': 'II 1,13,23:m,t:z42', 'ii bremen': 'II 45:g,m,s,t:e,n,x', 'v brookfield': 'V 66:z41:–', 'broxbourne': 'Wien', 'buenosaires': 'Bonariensis', 'ii bulawayo': 'II 1,40:z:1,5', 'ii bunnik': 'II 43:z42:1,5,7', 'ii caledon': 'II 1,4,12,[27]:g,[m],[s],t:e,n,x', 'ii calvinia': 'II 6,7:a:z42', 'cambridge': 'Meleagridis var. 15+', 'v camdeni': 'V 44:r:–', 'ii canastel': 'II 9,12:z29:1,5', 'canoga': 'Westhampton var. 15+,34+', 'ii cape': 'II 6,7:z6:1,7', 'ii carletonville': 'II 38:d:[1,5]', 'ii ceres': 'II 28:z:z39', 'iv chameleon': 'IV 16:z4,z32:–', 'ii chersina': 'II 47:z:z6', 'ii chinovum': 'II 42:b:1,5', 'ii chudleigh': 'II 3,10:e,n,x:1,7', 'clichy': 'Goelzau var. 15+', 'ii clifton': 'II 13,22:z29:1,5', 'ii clovelly': 'II 1,44:z39:e,n,x,z15', 'ii constantia': 'II 17:z:l,w:z42', 'ii daressalaam': 'II 1,9,12:l,w:e,n,x', 'ii degania': 'II 40:z4,z24:z39', 'ii detroit': 'II 42:z:1,5', 'drypool': 'Amsterdam var. 15+', 'ii dubrovnik': 'II 41:z:1,5', 'ii duivenhoks': 'II 9,46:g,[m],[s],t:[e,n,x]', 'ii durbanville': 'II 1,4,12,[27]:z39:1,[5],7', 'ii eilbek': 'IIIb 61:i:z', 'eimsbuettel': 'Livingstone var. 14+', 'ii ejeda': 'II 45:a:z10', 'ii elsiesrivier': 'II 16:z42:1,6', 'ii emmerich': 'II 6,14:m,t:e,n,x', 'ii epping': 'II 1,13,23:e,n,x:1,[5],7', 'ii erlangen': 'II 48:g,m,t:–', 'eschersheim': 'Souza var. 15+', 'ii etosha': 'II 48:d:1,2', 'ii fandran': 'II 1,40:z35:e,n,x,z15', 'ii faure': 'II 50:z42:1,7', 'ferlac': 'VI 1,6,14,25:a:e,n,x', 'ii finchley': 'II 3,10:z:e,n,x', 'iv flint': 'IV 50:z4,z23:–', 'ii foulpointe': 'II 38:g,t:–', 'ii fremantle': 'II 42:g,t:–', 'ii fuhlsbuettel': 'II 3,10:l,v:z6', 'gelsenkirchen': 'Gdansk var. 14+', 'ii germiston': 'II 6,8:m,t:e,n,x', 'ii gilbert': 'II 6,7:z39:1,5,7', 'ii glencairn': 'II 11:a:z6:z42', 'goerlitz': 'Vejle var. 15+', 'ii gojenberg': 'II 1,13,23:g,t:1,5', 'ii goodwood': 'II 13,22:z29:e,n,x', 'ii grabouw': 'II 11:g,[m],s,t:z39', 'ii greenside': 'II 50:z:e,n,x', 'ii grunty': 'II 1,40:z39:1,6', 'ii gwaai': 'II 21:z4,z24:–', 'ii haarlem': 'II 9,46:z:e,n,x', 'ii haddon': 'II 16:z4,z23:–', 'ii hagenbeck': 'II 48:d:z6', 'halmstad': 'Westhampton var. 15+', 'hamilton': 'Vejle var. 15+,[Rz27]', 'ii hammonia': 'II 48:e,n,x,z15:z6', 'iv harmelen': 'IV 51:z4,z23:–', 'ii heilbron': 'II 6,7:l,z28:1,5:[z42]', 'ii helsinki': 'II 1,4,12:z29:e,n,x', 'heves': '6,14,[24]:d:1,5', 'ii hillbrow': 'II 17:b:e,n,x,z15', 'hirschfield': 'Paratyphi C', 'ii hooggraven': 'II 50:z10:z6:z42', 'iv houten': 'IV 43:z4,z23:–', 'ii hueningen': 'II 9,12:z:z39', 'ii huila': 'II 11:l,z28:e,n,x', 'ii humber': 'II 53:z4,z24:–', 'illinois': 'Lexington var. 15+,34+', 'ii islington': 'II 3,10:g,t:–', 'iwojima': 'Kentucky', 'ii jacksonville': 'II 16:z29:e,n,x', 'jaja': 'Stanleyville var. 27+', 'java': 'Paratyphi B var. L(+) tartrate (= d–tartrate)+', 'ii kaltenhausen': 'II 28:b:z6', 'kanda': 'Meleagridis', 'ii katesgrove': 'II 1,13,23,m,t:1,5', 'ii khami': 'II 47:b:e,n,x,z15', 'khartoum': 'Oxford var. 15+,34+', 'ii kilwa': 'II 4,12:l,w:e,n,x', 'kinshasa': 'Uganda var. 15+', 'ii klapmuts': 'II 45:z:z39', 'ii kluetjenfelde': 'II 4,12:d:e,n,x', 'ii kommetje': 'II 43:b:z42', 'iv kralendyk': 'IV 6,7:z4,z24:–', 'ii krugersdorp': 'II 50:e,n,x:1,7', 'ii kuilsrivier': 'II 1,9,12:g,m,s,t:e,n,x', 'lanka': 'Weltevreden var. 15+', 'ii lethe': 'II 41:g,t:–', 'ii lichtenberg': 'II 41:z10:z6', 'ii limbe': 'II 1,13,22:g,m,t:[1,5]', 'ii lincoln': 'II 11:m,t:e,n,x', 'ii lindrick': 'II 9,12:e,n,x:1,[5],7', 'ii llandudno': 'II 28:g,(m),[s],t:1,5', 'ii lobatsi': 'II 52:z44:1,5,7', 'ii locarno': 'II 57:z29:z42', 'iv lohbruegge': 'IV 44:z4,z32:–', 'ii louwbester': 'II 16:z:e,n,x', 'ii luanshya': 'II 1,13,23:g,m,[s],t:[e,n,x]', 'ii lundby': 'II 9,46:b:e,n,x', 'ii lurup': 'II 41:z10:e,n,x,z15', 'ii luton': 'II 60:z:e,n,x', 'ii maarssen': 'II 9,46:z4,z24:z39:z42', 'iii maartensdijk': 'IIIa 40:g,z51:–', 'ii makoma': 'II 1,4,[5],12,[27]:a:e,n,x', 'ii makumira': 'II 1,4,12,[27]:e,n,x:1,[5],7', 'v malawi': 'V 66:z65:–', 'manila': 'Lexington var. 15+', 'ii manombo': 'II 57:z39:e,n,x,z15', 'v maregrosso': 'V 66:z35:–', 'iv marina': 'IV 48:g,z51:–', 'ii matroosfontein': 'II 3,10:a:e,n,x', 'menhaden': 'Give var. 15+,34+', 'ii merseyside': 'II 16:g,t:[1,5]', 'ii midhurst': 'II 53:l,z28:z39', 'minneapolis': 'Anatum var. 15+,34+', 'ii mjimwema': 'II 1,9,12:b:e,n,x', 'ii mobeni': 'II 16:g,[m],[s],t:[e,n,x]', 'ii mondeor': 'II 39:l,z28:e,n,x', 'ii montgomery': 'II 11:a:d:e,n,z15', 'ii mosselbay': 'II 43:g,m,[s],t:[z42]', 'ii mpila': 'II 3,10:z38:z42', 'iv mundsburg': 'IV 11:g,z51:–', 'ii nachshonim': 'II 1,13,23:z:1,5', 'ii nairobi': 'II 42:r:–', 'ii namib': 'II 50:g,[m],s,t:[1,5]', 'nancy': 'Nchanga var. 15+', 'ii neasden': 'II 9,12:g,s,t:e,n,x', 'ii negev': 'II 41:z10:1,2', 'ii ngozi': 'II 48:z10:[1,5]', 'newbrunswick': 'Give var. 15+', 'newhaw': 'Muenster var. 15+', 'newington': 'Anatum var. 15+', 'nienstedten': 'Ohio var 14+', 'ii nordenham': 'II 1,4,12,27:z:e,n,x', 'ii noordhoek': 'II 16:l,w:z6', 'ii nuernberg': 'II 42:z:z6', 'iv ochsenzoll': 'IV 16:z4,z23:–', 'ii odijk': 'II 30:a:z39', 'ii oevelgoenne': 'II 28:r:e,n,z15', 'omderman': 'Amersfoort var. 14+', 'ii ottershaw': 'II 40:d:–', 'ii oysterbeds': 'II 6,7:z:z42', 'pankow': 'Shangani var. 15+', 'iv parera': 'IV 11:z4,z23:–', 'ii parow': 'II 3,10,15:g,m,s,t:–', 'ii perinet': 'II 45:g,m,t:e,n,x,z15', 'ii phoenix': 'II 47:b:1,5', 'portsmouth': 'London var. 15+', 'ii portbech': 'II 42:l,v:e,n,x,z15', 'ii quimbamba': 'II 47:d:z39', 'ii rand': 'II 42:z:e,n,x,z15', 'ii rhodesiense': 'II 9,12:d:e,n,x', 'ii roggeveld': 'II 51:–:1,7', 'ii rooikrantz': 'II 1,6,14:m,t:1,5', 'rosenthal': 'Butantan var. 15+,34+', 'iv roterberg': 'IV 6,7:z4,z23:–', 'ii rotterdam': 'II 1,13,22:g,t:1,5', 'ii rowbarton': 'II 16:m,t:[z42]', 'iv sachsenwald': 'IV 1,40:z4,z23:–', 'sakai': 'Postdam', 'ii sakaraha': 'II 48:k:z39', 'ii sarepta': 'II 16:l,z28:z42', 'schottmuelleri': 'Paratyphi B', 'ii seaforth': 'II 50:k:z6', 'selandia': 'Nyborg var. 15+', 'iv seminole': 'IV 1,40:g,z51:–', 'ii setubal': 'II 60:g,m,t:z6', 'siegburg': 'Cerro var. 14+', 'ii simonstown': 'II 1,6,14:z10:1,5', 'ii slangkop': 'II 1,6,14:z10:z6:z42', 'ii slatograd': 'II 30:g,t:–', 'iv soesterberg': 'IV 21:z4,z23:–', 'ii sofia': 'II 1,4,12,[27]:b:[e,n,x]', 'ii soutpan': 'II 11:z:z39', 'ii springs': 'II 40:a:z39', 'vi srinagar': 'VI 11:b:e,n,x', 'ii stellenbosch': 'II 1,9,12:z:1,7', 'ii stevenage': 'II 1,13,23:[z42]:1,[5],7', 'ii stikland': 'II 3,10:m,t:e,n,x', 'ii suarez': 'II 1,40:c:e,n,x,z15', 'ii suederelbe': 'II 1,9,12:b:z39', 'suez': 'Shubra', 'suipestifer': 'Choleraesuis', 'ii sullivan': 'II 6,7:z42:1,7', 'ii sunnydale': 'II 1,40:k:e,n,x,z15', 'ii tafelbaai': 'II 3,10:z:z39', 'taihoku': 'Meleagridis', 'thielallee': 'Oranienburg var. 14+', 'thomasville': 'Orion var. 15+,34+', 'ii tokai': 'II 57:z42:1,6:z53', 'ii tosamanga': 'II 6,7:z:1,5', 'tournai': 'Stockholm var. 15+', 'ii tranoroa': 'II 55:k:z39', 'tuebingen': 'Amager var. 15+', 'iv tuindorp': 'IV 43:z4,z32:–', 'ii tulear': 'II 6,8:a:z52', 'ii tygerberg': 'II 1,13,23:a:z42', 'ii uphill': 'II 42:b:e,n,x,z15', 'ii utbremen': 'II 35:z29:e,n,x', 'ii veddel': 'II 43:g,t:–', 'ii verity': 'II 17:e,n,x,z15:1,6', 'iv volksdorf': 'IV 43:z36,z38:–', 'ii vredelust': 'II 1,13,23:l,z28:z42', 'vi vrindaban': 'VI 45:a:e,n,x', 'ii wandsbek': 'II 21:z10:z6', 'iv wassenaar': 'IV 50:g,z51:–', 'ii westpark': 'II 3,10:l,z28:e,n,x', 'wildwood': 'Meleagridis var. 15+,34+', 'ii winchester': 'II 3,10:z39:1,[5],7', 'ii windhoek': 'II 45:g,m,s,t:1,5', 'ii woerden': 'II 17:c:z39', 'ii woodstock': 'II 16:z42:1,(5),7', 'ii worcester': 'II 1,13,23:m,t:e,n,x', 'ii wynberg': 'II 1,9,12:z39:1,7', 'ii zeist': 'II 18:z10:z6', 'ii zuerich': 'II 1,9,12,46,27:c:z39', 'abortusbovis': 'Abony', 'abortuscanis': 'Paratyphi B', 'anie': 'Mesbit', 'atlanta': 'Mississippi', 'bambesa': 'Miami', 'cairo': 'Stanley', 'cardiff': 'Thompson', 'congo': 'Agbeni', 'cook': 'Champaign', 'dalat': 'Ball', 'decatur': 'Choleraesuis', 'ii hamburg': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'italiana': 'Panama', 'joenkoeping': 'Kingston', 'kaposvar': 'Reading', 'ii kraaifontein': 'II 1,13,23:g,m,[s],t:[e,n,x]', 'ii manica': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'iv maritza': 'Salford', 'mexicana': 'Muenchen', 'mission': 'Isangi', 'ii muizenberg': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'nissii': 'Ohio', 'oregon': 'Muenchen', 'pikine': 'Altona', 'pueris': 'Newport', 'pullorum': 'Gallinarum', 'ruki': 'Ball', 'rutgers': 'Give', 'saka': 'Sya', 'salinatis': 'Duisburg', 'ii shomron': 'IIIa 18:z4,z32:–', 'simsbury': 'Senftenberg', 'sladun': 'Abony', 'ii sydney': 'IIIb 48:i:z', 'tim': 'Newington', 'venusberg': 'Nchanga', 'ii wilhemstrasse': 'II 52:z44:1,5,7', 'womba': 'Altendorf', 'wuerzburg': 'Miami', 'zagreb': 'Saintpaul', 'acres': 'II 1,13,23:b:[1,5]:z42', 'alexander': 'II 3,10:z:1,5', 'alsterdorf': 'II 1,40:g,[m],[s],t:[1,5]', 'angola': 'II 1,9,12:z:z6', 'argentina': 'IV 6,7:z36:–', 'artis': 'II 56:b:[1,5]', 'askraal': 'II 51:l,z28:z6', 'atra': 'II 50:m,t:z6:z42', 'bacongo': 'II 6,7:z36:z42', 'balboa': 'V 48:z41:–', 'baragwanath': 'II 6,8:m,t:1,5', 'basel': 'II 58:l,z13,z28:1,5', 'bechuana': '1,4,12,[27]:g,[m],t:[1,5]', 'bellville': 'II 16:e,n,x:1,(5),7', 'beloha': 'II 18:z36:–', 'bern': 'IV 40:z4,z32:–', 'betioky': 'II 59:k:z65', 'bilthoven': 'II 47:a:1,5', 'blankenese': 'II 1,9,12:b:z6', 'bleadon': 'II 17:g,t:[e,n,x,z15]', 'bloemfontein': 'II 6,7:b:e,n,x:z42', 'bockenheim': 'IV 1,53:z36,z38:–', 'boksburg': 'II 40:g,m,s,t:e,n,x', 'bonaire': 'IV 50:z4,z32:–', 'bongor': 'V 48:z35:–', 'bornheim': 'VI 1,6,14,25:z10:1,(2),7', 'boulders': 'II 1,13,23:m,t:z42', 'bremen': 'II 45:g,m,s,t:e,n,x', 'brookfield': 'V 66:z41:–', 'bulawayo': 'II 1,40:z:1,5', 'bunnik': 'II 43:z42:1,5,7', 'caledon': 'II 1,4,12,[27]:g,[m],[s],t:e,n,x', 'calvinia': 'II 6,7:a:z42', 'camdeni': 'V 44:r:–', 'canastel': 'II 9,12:z29:1,5', 'cape': 'II 6,7:z6:1,7', 'carletonville': 'II 38:d:[1,5]', 'ceres': 'II 28:z:z39', 'chameleon': 'IV 16:z4,z32:–', 'chersina': 'II 47:z:z6', 'chinovum': 'II 42:b:1,5', 'chudleigh': 'II 3,10:e,n,x:1,7', 'clifton': 'II 13,22:z29:1,5', 'clovelly': 'II 1,44:z39:e,n,x,z15', 'constantia': 'II 17:z:l,w:z42', 'daressalaam': 'II 1,9,12:l,w:e,n,x', 'degania': 'II 40:z4,z24:z39', 'detroit': 'II 42:z:1,5', 'dubrovnik': 'II 41:z:1,5', 'duivenhoks': 'II 9,46:g,[m],[s],t:[e,n,x]', 'durbanville': 'II 1,4,12,[27]:z39:1,[5],7', 'eilbek': 'IIIb 61:i:z', 'ejeda': 'II 45:a:z10', 'elsiesrivier': 'II 16:z42:1,6', 'emmerich': 'II 6,14:m,t:e,n,x', 'epping': 'II 1,13,23:e,n,x:1,[5],7', 'erlangen': 'II 48:g,m,t:–', 'etosha': 'II 48:d:1,2', 'fandran': 'II 1,40:z35:e,n,x,z15', 'faure': 'II 50:z42:1,7', 'finchley': 'II 3,10:z:e,n,x', 'flint': 'IV 50:z4,z23:–', 'foulpointe': 'II 38:g,t:–', 'fremantle': 'II 42:g,t:–', 'fuhlsbuettel': 'II 3,10:l,v:z6', 'germiston': 'II 6,8:m,t:e,n,x', 'gilbert': 'II 6,7:z39:1,5,7', 'glencairn': 'II 11:a:z6:z42', 'gojenberg': 'II 1,13,23:g,t:1,5', 'goodwood': 'II 13,22:z29:e,n,x', 'grabouw': 'II 11:g,[m],s,t:z39', 'greenside': 'II 50:z:e,n,x', 'grunty': 'II 1,40:z39:1,6', 'gwaai': 'II 21:z4,z24:–', 'haarlem': 'II 9,46:z:e,n,x', 'haddon': 'II 16:z4,z23:–', 'hagenbeck': 'II 48:d:z6', 'hammonia': 'II 48:e,n,x,z15:z6', 'harmelen': 'IV 51:z4,z23:–', 'heilbron': 'II 6,7:l,z28:1,5:[z42]', 'helsinki': 'II 1,4,12:z29:e,n,x', 'hillbrow': 'II 17:b:e,n,x,z15', 'hooggraven': 'II 50:z10:z6:z42', 'houten': 'IV 43:z4,z23:–', 'hueningen': 'II 9,12:z:z39', 'huila': 'II 11:l,z28:e,n,x', 'humber': 'II 53:z4,z24:–', 'islington': 'II 3,10:g,t:–', 'jacksonville': 'II 16:z29:e,n,x', 'kaltenhausen': 'II 28:b:z6', 'katesgrove': 'II 1,13,23,m,t:1,5', 'khami': 'II 47:b:e,n,x,z15', 'kilwa': 'II 4,12:l,w:e,n,x', 'klapmuts': 'II 45:z:z39', 'kluetjenfelde': 'II 4,12:d:e,n,x', 'kommetje': 'II 43:b:z42', 'kralendyk': 'IV 6,7:z4,z24:–', 'krugersdorp': 'II 50:e,n,x:1,7', 'kuilsrivier': 'II 1,9,12:g,m,s,t:e,n,x', 'lethe': 'II 41:g,t:–', 'lichtenberg': 'II 41:z10:z6', 'limbe': 'II 1,13,22:g,m,t:[1,5]', 'lincoln': 'II 11:m,t:e,n,x', 'lindrick': 'II 9,12:e,n,x:1,[5],7', 'llandudno': 'II 28:g,(m),[s],t:1,5', 'lobatsi': 'II 52:z44:1,5,7', 'locarno': 'II 57:z29:z42', 'lohbruegge': 'IV 44:z4,z32:–', 'louwbester': 'II 16:z:e,n,x', 'luanshya': 'II 1,13,23:g,m,[s],t:[e,n,x]', 'lundby': 'II 9,46:b:e,n,x', 'lurup': 'II 41:z10:e,n,x,z15', 'luton': 'II 60:z:e,n,x', 'maarssen': 'II 9,46:z4,z24:z39:z42', 'maartensdijk': 'IIIa 40:g,z51:–', 'makoma': 'II 1,4,[5],12,[27]:a:e,n,x', 'makumira': 'II 1,4,12,[27]:e,n,x:1,[5],7', 'malawi': 'V 66:z65:–', 'manombo': 'II 57:z39:e,n,x,z15', 'maregrosso': 'V 66:z35:–', 'marina': 'IV 48:g,z51:–', 'matroosfontein': 'II 3,10:a:e,n,x', 'merseyside': 'II 16:g,t:[1,5]', 'midhurst': 'II 53:l,z28:z39', 'mjimwema': 'II 1,9,12:b:e,n,x', 'mobeni': 'II 16:g,[m],[s],t:[e,n,x]', 'mondeor': 'II 39:l,z28:e,n,x', 'montgomery': 'II 11:a:d:e,n,z15', 'mosselbay': 'II 43:g,m,[s],t:[z42]', 'mpila': 'II 3,10:z38:z42', 'mundsburg': 'IV 11:g,z51:–', 'nachshonim': 'II 1,13,23:z:1,5', 'nairobi': 'II 42:r:–', 'namib': 'II 50:g,[m],s,t:[1,5]', 'neasden': 'II 9,12:g,s,t:e,n,x', 'negev': 'II 41:z10:1,2', 'ngozi': 'II 48:z10:[1,5]', 'nordenham': 'II 1,4,12,27:z:e,n,x', 'noordhoek': 'II 16:l,w:z6', 'nuernberg': 'II 42:z:z6', 'ochsenzoll': 'IV 16:z4,z23:–', 'odijk': 'II 30:a:z39', 'oevelgoenne': 'II 28:r:e,n,z15', 'ottershaw': 'II 40:d:–', 'oysterbeds': 'II 6,7:z:z42', 'parera': 'IV 11:z4,z23:–', 'parow': 'II 3,10,15:g,m,s,t:–', 'perinet': 'II 45:g,m,t:e,n,x,z15', 'phoenix': 'II 47:b:1,5', 'portbech': 'II 42:l,v:e,n,x,z15', 'quimbamba': 'II 47:d:z39', 'rand': 'II 42:z:e,n,x,z15', 'rhodesiense': 'II 9,12:d:e,n,x', 'roggeveld': 'II 51:–:1,7', 'rooikrantz': 'II 1,6,14:m,t:1,5', 'roterberg': 'IV 6,7:z4,z23:–', 'rotterdam': 'II 1,13,22:g,t:1,5', 'rowbarton': 'II 16:m,t:[z42]', 'sachsenwald': 'IV 1,40:z4,z23:–', 'sakaraha': 'II 48:k:z39', 'sarepta': 'II 16:l,z28:z42', 'seaforth': 'II 50:k:z6', 'seminole': 'IV 1,40:g,z51:–', 'setubal': 'II 60:g,m,t:z6', 'simonstown': 'II 1,6,14:z10:1,5', 'slangkop': 'II 1,6,14:z10:z6:z42', 'slatograd': 'II 30:g,t:–', 'soesterberg': 'IV 21:z4,z23:–', 'sofia': 'II 1,4,12,[27]:b:[e,n,x]', 'soutpan': 'II 11:z:z39', 'springs': 'II 40:a:z39', 'srinagar': 'VI 11:b:e,n,x', 'stellenbosch': 'II 1,9,12:z:1,7', 'stevenage': 'II 1,13,23:[z42]:1,[5],7', 'stikland': 'II 3,10:m,t:e,n,x', 'suarez': 'II 1,40:c:e,n,x,z15', 'suederelbe': 'II 1,9,12:b:z39', 'sullivan': 'II 6,7:z42:1,7', 'sunnydale': 'II 1,40:k:e,n,x,z15', 'tafelbaai': 'II 3,10:z:z39', 'tokai': 'II 57:z42:1,6:z53', 'tosamanga': 'II 6,7:z:1,5', 'tranoroa': 'II 55:k:z39', 'tuindorp': 'IV 43:z4,z32:–', 'tulear': 'II 6,8:a:z52', 'tygerberg': 'II 1,13,23:a:z42', 'uphill': 'II 42:b:e,n,x,z15', 'utbremen': 'II 35:z29:e,n,x', 'veddel': 'II 43:g,t:–', 'verity': 'II 17:e,n,x,z15:1,6', 'volksdorf': 'IV 43:z36,z38:–', 'vredelust': 'II 1,13,23:l,z28:z42', 'vrindaban': 'VI 45:a:e,n,x', 'wandsbek': 'II 21:z10:z6', 'wassenaar': 'IV 50:g,z51:–', 'westpark': 'II 3,10:l,z28:e,n,x', 'winchester': 'II 3,10:z39:1,[5],7', 'windhoek': 'II 45:g,m,s,t:1,5', 'woerden': 'II 17:c:z39', 'woodstock': 'II 16:z42:1,(5),7', 'worcester': 'II 1,13,23:m,t:e,n,x', 'wynberg': 'II 1,9,12:z39:1,7', 'zeist': 'II 18:z10:z6', 'zuerich': 'II 1,9,12,46,27:c:z39', 'hamburg': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'kraaifontein': 'II 1,13,23:g,m,[s],t:[e,n,x]', 'manica': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'maritza': 'Salford', 'muizenberg': 'II 1,9,12:g,m,[s],t:[1,5,7]:[z42]', 'shomron': 'IIIa 18:z4,z32:–', 'sydney': 'IIIb 48:i:z', 'wilhemstrasse': 'II 52:z44:1,5,7'}


####################
#### WKLM lists ####
####################

# Lists include serovars [1-4] and named subsp. enterica variants [1]
# Standardized lists are lowercase, with brackets and ellipses removed, for matching purposes
# All other lists are properly formatted for output
wklm_name = ['Paratyphi A','Nitra','Kiel','Koessen','Kisangani','Hessarek','Fulica','Arechavaleta','Bispebjerg','Tinda','II [1],4,[5],12,[27]:a:e,n,x','Huettwilen','Nakuru','II [1],4,12,[27]:a:z39','Paratyphi B','Limete','II 4,12:b:1,5','Canada','Uppsala','Abony','II [1],4,[5],12,[27]:b:[e,n,x]','Wagenia','Wien','Tripoli','Schleissheim','Legon','Abortusovis','Altendorf','Bissau','Jericho','Hallfold','Bury','Stanley','Eppendorf','Brezany','Schwarzengrund','II 4,12:d:e,n,x','Sarajane','Duisburg','Mons','Ayinde','Chennai','Saintpaul','Reading','Eko','Kaapstad','Chester','Sandiego','Chartres','II 4,12:e,n,x:1,2,7','II [1],4,12,[27]:e,n,x:1,[5],7','Derby','Agona','II [1],4,[5],12:f,g,t:z6:z42','Essen','Hato','II [1],4,12,[27]:g,[m],[s],t:e,n,x','II [1],4,12,[27]:g,[m],t:[1,5]','II 4,12:g,m,t:z39','California','Kingston','Budapest','Travis','Tennyson','II 4,12:g,z62:–','Banana','Madras','Typhimurium','Lagos','Agama','Farsta','Tsevie','Gloucester','Tumodi','II 4,12,27:i:z35','Massenya','Neumuenster','II [1],4,12,27:k:1,6','Ljubljana','Texas','Fyris','Azteca','Clackamas','Bredeney','Kimuenza','II [1],4,12,27:l,v:e,n,x','Brandenburg','II [1],4,12,27:l,v:z39','Mono','Togo','II 4,12:l,w:e,n,x','Blancmesnil','Ayton','Kunduchi','Tyresoe','Haduna','Kubacha','Kano','Vom','Reinickendorf','II [1],4,12:l,z28:[e,n,x]','Heidelberg','Bradford','Winneba','Remo','Bochum','Southampton','Drogana','Africana','Coeln','Trachau','Finaghy','Teddington','Ball','Jos','Kamoru','Shubra','Kiambu','II [1],4,12,27:z:1,5','Loubomo','Indiana','II 4,12:z:1,7','Neftenbach','II [1],4,12,27:z:e,n,x','Koenigstuhl','Preston','Entebbe','II 4,12:z:z39','Stanleyville','Perrosguirec','Vuadens','Kalamu','Haifa','Ituri','Tudu','Harrow','Albert','Tokoin','Mura','Fortune','Vellore','Brancaster','II [1],4,12:z29:e,n,x','Pasing','Tafo','Sloterdijk','Yaounde','Tejas','Yverdon','Wilhelmsburg','II [1],4,12,[27]:z39:1,[5],7','Thayngen','Maska','Abortusequi','Mygdal','Sanjuan','II 6,7,[14]:a:1,5','Umhlali','Austin','Oslo','Denver','Coleypark','Damman','II 6,7:a:z6','II 6,7:a:z42','Brazzaville','II 6,7:b:1,5','Edinburg','Adime','Koumra','Lockleaze','Georgia','II 6,7:b:[e,n,x]:z42','Ohio','Leopoldville','Kotte','II 6,7:b:z39','Hissar','Paratyphi C','Choleraesuis','Typhisuis','Birkenhead','Schwabach','Cotonou','Namibia','Kaduna','Kisii','Isangi','Kivu','Kambole','Amersfoort','Gombe','Livingstone','Wil','Nieukerk','II 6,7:d:z42','Larochelle','Lomita','Norwich','Nola','Braenderup','II 6,7:e,n,x:1,6:z42','Kastrup','Rissen','Eingedi','Afula','Montevideo','Lubbock','II 6,7:g,m,[s],t:e,n,x','II 6,7:(g),m,[s],t:1,5','II 6,7:g,m,s,t:z39','II 6,7:g,[m],s,t:[z42]','Othmarschen','Plumaugat','Menston','II 6,7:g,t:[e,n,x]:z42','Riggil','Alamo','Larose','IV 6,7:g,z51:–','Haelsingborg','Winston','Oakey','II 6,7:m,t:–','Oranienburg','Augustenborg','Oritamerin','Garoli','Lika','Athinai','Norton','Stuttgart','Galiema','Thompson','Daytona','Baiboukoum','Singapore','Escanaba','IIIb 6,7:(k):z:[z55]','II 6,7:k:[z6]','Ashford','Concord','Irumu','IIIb 6,7:l,v:1,5,7','Mkamba','Kortrijk','Bonn','Potsdam','Gdansk','Coromandel','IIIb 6,7:l,v:z53','Gabon','Colorado','II 6,7:l,w:1,5,7','Langeveld','II 6,7:l,w:z42','Nessziona','Kenya','Neukoelln','Makiso','Strathcona','II 6,7:l,z28:1,5:[z42]','II 6,7:l,z28:e,n,x','II 6,7:l,z28:z6','Virchow','Infantis','Nigeria','Colindale','Papuana','Grampian','Richmond','Bareilly','Oyonnax','Gatow','Hartford','Mikawasima','Chile','Poitiers','II 6,7:z:1,5','Oakland','Cayar','II 6,7:z:e,n,x','II 6,7:z:e,n,x,z15','Businga','Bruck','II 6,7:z:z6','II 6,7:z:z39','II 6,7:z:z42','Obogu','Planckendael','Aequatoria','Goma','II 6,7:z4,z23:–','IV 6,7:z4,z23:–','Zollikerberg','II 6,7:z4,z24:z42','Somone','IV 6,7:z4,z24:–','II 6,7:z6:1,7','Menden','Inganda','Eschweiler','Ngili','Djugu','Mbandaka','Jerusalem','Redba','Omuna','Tennessee','II 6,7:z29:[z42]','Tienba','Palime','Tampico','II 6,7:z36:z42','IV 6,7:z36:–','Rumford','II 6,7:z38:e,n,x','Lille','IIIb 6,7,[14]:z39:1,2','II 6,7:z39:1,5,7','VI 6,7:z41:1,7','Hillsborough','Tamilnadu','II 6,7:z42:1,[5],7','Bulovka','II 6,7:–:1,6','Be','Valdosta','Doncaster','Curacao','Nordufer','Narashino','II 6,8:a:e,n,x','Leith','II 6,8:a:z39','II 6,8:a:z52','Djelfa','Skansen','Korbol','Nagoya','II 6,8:b:1,5','Stourbridge','Sanga','Eboko','Konstanz','Gatuni','Shipley','Presov','Bukuru','Heistopdenberg','Tounouma','Banalia','Wingrove','Gaillac','Utah','Bronx','Belfast','Alexanderpolder','Santiago','Belem','Quiniela','Tado','Virginia','Muenchen','Yovokome','Manhattan','Portanigra','Dunkwa','Sterrenbos','Herston','Labadi','II 6,8:d:z6:z42','Bardo','Newport','Ferruch','Kottbus','Cremieu','Atakpame','Fillmore','Tshiongwe','Rechovot','Sandow','II 6,8:f,g,m,t:[e,n,x]','Emek','Chincol','II 6,8:g,m,t:1,7','Reubeuss','Alminko','Nanergou','Yokoe','II 6,8:m,t:1,5','II 6,8:m,t:e,n,x','Bassa','Lindenburg','Bargny','Takoradi','Warnow','Malmoe','Bonariensis','Aba','Magherafelt','Cyprus','Kentucky','Kallo','Haardt','Blockley','Sandbanks','Schwerin','Charlottenburg','IIIb 8:(k):z35','Pakistan','Litchfield','Loanda','Amherstiana','Manchester','Holcomb','II 6,8:l,v:e,n,x','Edmonton','Lund','Fayed','II 6,8:l,w:z6:z42','Hiduddify','Breukelen','II 6,8:l,z28:e,n,x','Bsilla','Hindmarsh','Bovismorbificans','Colchester','Noya','Akanji','Cocody','Hidalgo','Brikama','Goldcoast','IIIb 8:r:z','Altona','Giza','Lamphun','Brunei','Tananarive','Bulgaria','II 6,8:y:1,6:z42','Alagbon','Inchpark','Sunnycove','Daarle','Praha','Kralingen','Benue','Sindelfingen','Mowanjum','II 6,8:z:1,5','Marmande','Phaliron','Kalumburu','Kuru','Daula','Bellevue','Lezennes','Breda','Chailey','Dabou','Corvallis','Nuorikkala','Albany','Duesseldorf','Tallahassee','Bazenheid','Zerifin','Paris','Mapo','Cleveland','Istanbul','Hadar','Chomedey','Glostrup','Remiremont','Molade','Wippra','II 6,8:z29:1,5','II 6,8:z29:e,n,x:z42','Tamale','Uno','II 6,8:z29:e,n,x','Kolda','Yarm','Angers','Apeyeme','Diogoye','Aesch','Sendai','Miami','II 9,12:a:1,5','Os','Saarbruecken','Lomalinda','II [1],9,12:a:e,n,x','Durban','II 9,12:a:z39','II [1],9,12:a:z42','Onarimon','Frintrop','Bata','II [1],9,12:b:e,n,x','Mana','II [1],9,12:b:z6','II [1],9,12:b:z39','Goeteborg','Ipeko','Elokate','Alabama','Ridge','Ndolo','Tarshyne','Eschberg','II [1],9,12:d:e,n,x','Viikki','Bangui','Zega','Jaffna','II 9,12:d:z39','Typhi','Bournemouth','Eastbourne','Westafrica','Israel','II 9,12:e,n,x:1,[5],7','II 9,12:e,n,x:1,6','Berta','Enteritidis','Gueuletapee','Blegdam','II [1],9,12:g,m,[s],t:[1,5,7]:[z42]','II [1],9,12:g,m,s,t:e,n,x','II 9,12:g,m,t:z39','Dublin','Naestved','Rostock','Moscow','II 9,12:g,s,t:e,n,x','Newmexico','II [1],9,12:g,z62:[e,n,x]','Antarctica','Rosenberg','II 9,12:m,t:e,n,x','Pensacola','II [1],9,12:m,t:1,5','II [1],9,12:m,t:z39','Seremban','Claibornei','Goverdhan','Mendoza','Panama','Houston','Kapemba','Zaiman','II 9,12:l,v:e,n,x','Goettingen','II 9,12:l,v:z39','Victoria','II [1],9,12:l,w:e,n,x','Itami','Miyazaki','Napoli','Javiana','Kotu','II 9,12:l,z28:1,5:z42','II 9,12:l,z28:e,n,x','York','Jamaica','Camberwell','Yellowknife','Campinense','Lome','Powell','II [1],9,12:y:z39','Mulhouse','Lawndale','Kimpese','II [1],9,12:z:1,7','Aurelianis','II [1],9,12:z:z6','II 9,12:z:z39','Wangata','Natal','Franken','Portland','Treguier','Ruanda','II 9,12:z29:1,5','II [1],9,12:z29:e,n,x','Penarth','Elomrane','II [1],9,12:z39:1,7','Ottawa','II [1],9,12:z42:1,[5],7','Gallinarum','Detmold','Poole','Baildon','Doba','Montaigu','Cheltenham','Zadar','Worb','II 9,46:b:e,n,x','Bamboye','Linguere','Kolar','Argenteuil','Itutaba','Ontario','Quentin','Strasbourg','Olten','Plymouth','Sontheim','Bergedorf','Waedenswil','Guerin','II 9,46:e,n,x:1,5,7','Wernigerode','Hillingdon','Macclesfield','II 9,46:g,[m],[s],t:[e,n,x]','Gateshead','II 9,46:g,z62:–','II 9,46:m,t:e,n,x','Sangalkam','Mathura','Potto','Marylebone','Cochin','Clontarf','Ceyco','India','Geraldton','Toronto','Ackwepe','II 9,46:l,w:e,n,x','Nordrhein','Deckstein','Shoreditch','Sokode','Cork','Benin','Irchel','Nantes','Mayday','II 9,46:z:1,5','II 9,46:z:e,n,x','Bambylor','II 9,46:z:z39','Ekotedo','II 9,46:z4,z24:z39:z42','Ngaparou','Lishabi','Inglis','Mahina','Louisiana','II 9,46:z10:z6','II 9,46:z10:z39','Ouakam','Hillegersberg','Basingstoke','Trimdon','Fresno','II 9,46:z39:1,7','Wuppertal','II 1,9,12,46,27:a:z6','II 1,9,12,46,27:b:z39','II 1,9,12,46,27:c:z39','II 9,12,46,27:g,t:e,n,x','II 1,9,12,46,27:l,z13,z28:z39','II 1,9,12,46,27:y:z39','II 1,9,12,46,27:z4,z24:1,5','II 1,9,12,46,27:z10:1,5','II 1,9,12,46,27:z10:e,n,x','II 1,9,12,46,27:z10:z39','Aminatu','Goelzau','Oxford','Masembe','II 3,10:a:e,n,x','Galil','II 3,10:a:l,v','II 3,10:a:z39','II 3,10:a:z87','Kalina','Butantan','Allerton','Huvudsta','Benfica','II 3,10:b:e,n,x','Yaba','Epicrates','Wilmington','Westminster','II 3,10:b:z39','Asylanta','Gbadago','Ikayi','Pramiso','Agege','Anderlecht','Okefoko','Stormont','Shangani','Lekke','Onireke','Souza','II 3,10:d:e,n,x','Madjorio','Birmingham','Weybridge','Maron','Vejle','Muenster','Anatum','Nyborg','Newlands','Lamberhurst','Meleagridis','Sekondi','II 3,10:e,n,x:1,7','Regent','Alfort','Suberu','Amsterdam','II 3,{10}{15}:g,m,s,t:[1,5]','Westhampton','Bloomsbury','II 3,10:g,t:–','II 3,10:m,t:1,5','Southbank','II 3,10:m,t:e,n,x','Cuckmere','Amounderness','Tibati','Truro','Bessi','Falkensee','Hoboken','Yeerongpilly','Wimborne','Zanzibar','Serrekunda','Yundum','Marienthal','Newrochelle','Nchanga','Sinstorf','London','Give','II 3,10:l,v:e,n,x','Ruzizi','II 3,10:l,v:z6','Sinchew','Assinie','Freiburg','Uganda','Fallowfield','Hoghton','II 3,10:l,z28:1,5','Torbay','Joal','Lamin','II 3,10:l,z28:e,n,x','II 3,10:l,z28:z39','Ughelli','Elisabethville','Simi','Weltevreden','Seegefeld','Dumfries','Amager','Orion','Mokola','Ohlstedt','Bolton','Langensalza','Stockholm','Fufu','II 3,10:z:1,5','Harleystreet','Huddinge','II 3,10:z:e,n,x','Kingslynn','Clerkenwell','Landwasser','II 3,10:z:z39','Adabraka','Wagadugu','Florian','II 3,10:z4,z24:–','Okerara','Lexington','Harrisonburg','Coquilhatville','Podiensis','Kristianstad','Biafra','Everleigh','II 3,10:z29:[e,n,x]','Jedburgh','Santaclara','Ratchaburi','Zongo','II 3,10:z35:e,n,x,z15','Shannon','Cairina','Macallen','Sandaga','Albertslund','Bolombo','II 3,10:z38:z42','II 3,10:z39:1,[5],7','Dortmund','Pietersburg','II 3,15:z87:e,n,x,z15','Capetown','Niumi','Juba','Gwoza','Alkmaar','Gnesta','Visby','Tambacounda','Kande','Broughton','Accra','Eastglam','Bida','Madiago','Umbadah','Ahmadi','Wanatah','Liverpool','Tilburg','Niloese','Vilvoorde','Hayindogo','Sanktmarx','Sao','Calabar','Rideau','Petahtikve','Maiduguri','Kouka','Senftenberg','Cannstatt','Stratford','Ouagadougou','Chichester','Machaga','Avonmouth','Zuilen','Taksony','Oesterbro','Bethune','Ngor','Parkroyal','Svedvi','Fulda','Westerstede','Winterthur','Lokstedt','Stuivenberg','Bedford','Tomelilla','Kindia','Yalding','Fareham','Gatineau','Thies','Slade','Kinson','Krefeld','Korlebu','Kainji','Lerum','Schoeneberg','Carno','Hongkong','Miromesnil','Sambre','Yenne','Dallgow','Llandoff','Catumagos','Ochiogu','Fortbend','Chittagong','Bilu','Ilugun','Dessau','Cannonhill','Gallen','Marseille','VI 11:a:1,5','Massilia','Toowong','Luciana','II 11:a:e,n,z15:d','Epinay','II 11:a:z6:z42','Atento','Leeuwarden','Wohlen','VI 11:b:1,7','VI 11:b:e,n,x','Pharr','Erfurt','Chiredzi','Brindisi','II 11:c:e,n,z15','Woodinville','Ati','Gustavia','Chandans','Findorff','Chingola','Adamstua','Redhill','Abuja','Missouri','II 11:g,[m],s,t:z39','IV 11:g,z51:–','Moers','II 11:m,t:e,n,x','Aberdeen','Brijbhumi','Heerlen','Veneziana','IIIb 11:i:z','Pretoria','Abaetetuba','Sharon','Colobane','Kisarawe','Mannheim','Amba','IIIb 11:k:z53','Stendal','Maracaibo','Fann','Bullbay','IIIb 11:l,v:z:[z56]','IIIb 11:l,v:z53','Glidji','Tours','Connecticut','Osnabrueck','Welwyn','II 11:l,z28:e,n,x','Senegal','Rubislaw','Clanvillian','Euston','Volta','Solt','Jalisco','Herzliya','Woumbou','Crewe','Maroua','II 11:z:e,n,x','Maybush','Nyanza','II 11:z:z39','Remete','Etterbeek','IIIa 11:z4,z23:–','IV 11:z4,z23:–','Yehuda','IV 11:z4,z32:–','Wentworth','Straengnaes','Telhashomer','Lene','Maastricht','II 11:z41:e,n,x','II 11:–:1,5','Chagoua','II [1],13,23:a:1,5','Mim','II 13,22:a:e,n,x','Wyldegreen','Marshall','II [1],13,23:a:z42','Ibadan','Mississippi','Oudwijk','II [1],13,23:b:[1,5]:z42','Bracknell','Rottnest','Vaertan','Ullevi','Bahati','Durham','Sanktjohann','II [1],13,22:b:z42','Haouaria','Boston','Handen','II 13,22:d:1,5','Mishmarhaemek','Friedenau','Wichita','Grumpensis','II 13,23:d:e,n,x','Saintes','Diguel','Telelkebir','Putten','Isuge','Tschangu','Willemstad','Vridi','II [1],13,23:e,n,x:1,[5],7','Raus','Havana','Bron','IIIb 13,22:g,m,s:z','Agbeni','II [1],13,22:g,m,t:[1,5]','II 13,22:g,m,t:z42','II [1],13,23:g,m,s,t:1,5','II [1],13,23:g,m,[s],t:[e,n,x]','II [1],13,23:g,m,s,t:z42','Newyork','II 13,23:g,s,t:e,n,x','Okatie','II [1],13,22:g,t:[1,5]','II 13,22:g,t:z6','II [1],13,23:g,t:1,5','II 13,23:g,t:e,n,x','II [1],13,23:g,[s],t:z42','IIIa [1],13,23:g,z51:–','Washington','II [1],13,23:m,t:1,5','II [1],13,23:m,t:e,n,x','II 13,22:m,t:z42:z39','II [1],13,23:m,t:z42','Kintambo','V [1],13,22:i:–','Idikan','Myrria','Jukestown','Kedougou','Chapuri','II 13,22:k:1,5:z42','Noho','Marburg','Galway','II 13,23:k:z41','Lovelace','IIIb 13,22:l,v:1,5,7','Borbeck','Nanga','II 13,23:l,w:e,n,x','Taiping','II 13,22:l,z28:1,5','II 13,23:l,z28:1,5','II 13,23:l,z28:z6','II [1],13,23:l,z28:z42','V 13,22:r:–','Adjame','Linton','Tanger','Yarrabah','Ordonez','Tunis','Winslow','II [1],13,23:z:1,5','Serenli','IIIb 13,23:z:1,5','Poona','Farmsen','Bristol','Durance','Ivrysurseine','Tanzania','Worthington','II [1],13,23:z:z42','II 13,22:z:–','Ried','IIIa 13,22:z4,z23:–','Ajiobo','IIIa 13,23:z4,z23,[z32]:–','Romanby','IIIa 13,23:z4,z24:–','Roodepoort','II [1],13,22:z10:z6','Sapele','Demerara','II 13,22:z29:1,5','II 13,22:z29:e,n,x','II [1],13,23:z29:1,5','II [1],13,23:z29:e,n,x','Agoueve','Cubana','Mampong','Nimes','Picpus','Anna','Leiden','Fanti','V 13,22:z39:–','II 13,22:z39:1,7','II [1],13,23:z39:1,5,7','Pajala','II [1],13,23:[z42]:1,[5],7','II 13,23:–:1,6','Garba','VI [1],6,14:a:1,5','VI [1],6,14,[25]:a:e,n,x','Banjul','Ndjamena','Kuntair','Tucson','IIIb (6),14:b:e,n,x','Blijdorp','Kassberg','Runby','Minna','Martonos','Finkenwerder','Woodhull','Midway','Florida','Lindern','Charity','Teko','Encino','Albuquerque','Bahrenfeld','Onderstepoort','Magumeri','Beaudesert','V 6,14:e,n,z15:–','Warragul','Caracas','Sylvania','Catanzaro','II 1,6,14:m,t:1,5','II 6,14:m,t:e,n,x','Kaitaan','Mampeza','Buzu','Schalkwijk','Moussoro','Harburg','II 6,14,[24]:k:1,6','II 6,14:k:e,n,x','Trumersee','IIIb (6),14:k:z','II 1,6,14:k:z6:z42','IIIb (6),14:k:z53','Amberg','Boecker','Horsham','Alpenquai','IIIb (6),14:l,v:z','IIIb (6),14:l,v:z35','IIIb (6),14:l,v:z53','VI 6,14:l,v:z88','Owerri','Aflao','Istoria','IIIb (6),14:r:z','Surat','Carrau','Madelia','Fischerkietz','Mornington','Homosassa','Kanifing','Soahanina','Sundsvall','Royan','Stafford','Poano','Arapahoe','Bousso','IV 6,14:z4,z23:–','Chichiri','Uzaramo','Nessa','VI 1,6,14,25:z10:1,(2),7','II 1,6,14:z10:1,5','Laredo','IIIb (6),14:z10:e,n,x,z15','IIIb (6),14:z10:z:[z56],[z90]','Ulm','II 1,6,14:z10:z6:z42','IIIb 6,14:z10:z53','Potosi','II 6,14:z36:–','Sara','II 1,6,14:z42:1,6','IIIb 6,14:z52:e,n,x,z15','IIIb [1],6,14,[25]:z52:z35','Hannover','Brazil','Amunigun','Nyeko','Togba','Fischerhuette','Heron','Hull','Melaka','Wa','Glasgow','Hvittingfoss','II 16:b:e,n,x','Sangera','Vegesack','Malstatt','II 16:b:z39','II 16:b:z42','Vancouver','Gafsa','Shamba','Hithergreen','Yoruba','Oldenburg','Sculcoates','II 16:d:1,5','Sherbrooke','Gaminara','Barranquilla','II 16:d:e,n,x','Nottingham','Caen','Barmbek','Malakal','Saboya','Astridplein','Rhydyfelin','Moabit','Weston','II 16:e,n,x:1,(5),7','II 16:e,n,x:1,6:z42','Tees','Adeoyo','Nikolaifleet','II 16:g,[m],[s],t:[1,5]:[z42]','II 16:g,[m],[s],t:[e,n,x]','Cardoner','II 16:m,t:e,n,x','Morbihan','II 16:m,t:[z42]','Mpouto','Amina','Agbara','Wisbech','Frankfurt','Pisa','Abobo','IIIb 16:i:z35','Szentes','Maumee','Nuatja','Orientalis','IIIb 16:(k):e,n,x,z15','IIIb 16:k:z','IIIb 16:(k):z35','IIIb 16:k:z53','IIIb 16:l,v:1,5,7','Shanghai','Welikade','Salford','Burgas','IIIb 16:l,v:z:[z61]','Losangeles','IIIb 16:l,v:z35','IIIb 16:l,v:z53','Zigong','Westeinde','Brooklyn','Lomnava','Essingen','II 16:l,w:z6','Mandera','Enugu','Battle','Ablogame','Koblenz','II 16:l,z28:z42','Rovaniemi','Ivory','Brunflo','Lehrte','Annedal','Zwickau','Saphra','Akuafo','Kikoma','Avignon','Gerland','Fortlamy','Lingwala','Kassel','II 16:z:e,n,x','Brevik','Bouake','II 16:z:z42','Kibi','Axim','II 16:z4,z23:–','IV 16:z4,z23:–','Kaevlinge','II 16:z4,z24:–','IV 16:z4,z24:–','IV 16:z4,z32:–','II 16:z6:1,6','Badagry','IIIb 16:z10:1,7','Lisboa','IIIb 16:z10:e,n,x,z15','Redlands','Angouleme','Saloniki','II 16:z29:1,5','II 16:z29:e,n,x','Neuland','Trier','Dakota','II 16:z35:e,n,x','IV 16:z36:–','II 16:z36:e,n,z15','Naware','Grancanaria','II 16:z42:1,(5),7','II 16:z42:1,6','IIIb 16:z52:z35','Bonames','Jangwani','Kinondoni','Kirkee','Dahra','Mattenhof','II 17:b:e,n,x,z15','Bignona','II 17:b:z6','Luedinghausen','Victoriaborg','II 17:c:z39','II 17:d:–','Berlin','Karlshamn','Niamey','II 17:d:z6','Jubilee','II 17:e,n,x,z15:1,6','II 17:e,n,x,z15:1,[5],7','Wembleypark','II 17:g,m,s,t:–','Lowestoft','II 17:g,t:[e,n,x,z15]','II 17:g,t:z39','IIIa 17:g,z51:–','Bama','II 17:m,t:–','Ahanou','IIIb 17:i:z35','Irenea','Bandim','Warri','Matadi','Zaria','IIIb 17:k:z','II 17:k:–','Morotai','Michigan','Lancaster','Carmel','IIIb 17:l,v:e,n,x,z15','IIIb 17:l,v:z35','Granlo','Lode','IIIb 17:r:z','II 17:y:–','Tendeba','Hadejia','Lokomo','Gori','Warengo','Dingiri','II 17:z:1,7','Tchamba','II 17:z:l,w:z42','IIIa 17:z4,z23:–','IIIa 17:z4,z23,z32:–','IIIa 17:z4,z24:–','IIIa 17:z4,z32:–','Djibouti','IIIb 17:z10:e,n,x,z15:[z56]','IIIb 17:z10:z','II 17:z10:–','Kandla','IIIa 17:z29:–','IV 17:z29:–','Aachen','IIIa 17:z36:–','IV 17:z36:–','Brazos','Fluntern','Cochise','Rawash','Groenekan','Usumbura','Pontypridd','Eaubonne','IIIa 18:g,z51:–','IV 18:g,z51:–','II 18:m,t:1,5','Langenhorn','Memphis','IIIb 18:(k):z53','IIIb 18:(k):z54','IIIb 18:l,v:e,n,x,z15','Orlando','IIIb 18:l,v:z:[z50]','IIIb 18:l,v:z53','Toulon','Tennenlohe','IIIb 18:r:z','Troy','II 18:y:e,n,x,z15','Potengi','Cerro','Aarhus','II 18:z4,z23:–','IIIa 18:z4,z23:–','Blukwa','II 18:z4,z24:–','IIIa 18:z4,z32:–','IIIb 18:z10:e,n,x,z15','Leer','Carnac','II 18:z10:z6','II 18:z36:–','IIIa 18:z36:–','IV 18:z36,z38:–','Sinthia','Delmenhorst','Cotia','Assen','II 21:b:1,5','Ghana','Minnesota','Hydra','Rhone','II 21:c:e,n,x','IIIb 21:c:e,n,x,z15','Spartel','Magwa','Madison','Good','II 21:g,[m],[s],t:–','IIIa 21:g,z51:–','IV 21:g,z51:–','II 21:m,t:–','IV 21:m,t:–','Diourbel','IIIb 21:i:1,5,7','IIIb 21:i:e,n,x,z15','Salpetriere','IIIb 21:k:e,n,x,z15','IIIb 21:k:z','Surrey','IIIb 21:l,v:z','IIIb 21:l,v:z57','Keve','Jambur','Mountmagnet','IIIb 21:r:z','Ibaragi','Ruiru','II 21:z:–','Baguida','IIIa 21:z4,z23:–','IV 21:z4,z23:–','II 21:z4,z24:–','IIIa 21:z4,z24:–','IV 21:z4,z32:–','IIIb 21:z10:e,n,x,z15','IIIb 21:z10:z','II 21:z10:[z6]','IIIb 21:z10:z53','IIIa 21:z29:–','Gambaga','IIIa 21:z36:–','IV 21:z36:–','IV 21:z38:–','IIIb 21:z65:e,n,x,z15','Solna','Dakar','Bakau','Seattle','II 28:a:e,n,x','Honelis','Dibra','Moero','Ashanti','Bokanjac','Soumbedioune','II 28:b:e,n,x','Langford','Freefalls','II 28:b:z6','Hermannswerder','Eberswalde','Halle','Dresden','Wedding','Parabiago','Techimani','Amoutive','Hatfield','Mundonobo','Mocamedes','Patience','Cullingworth','Etobicoke','Korkeasaari','Kpeme','Gozo','II 28:e,n,x:1,7','II 28:e,n,z15:z87','Friedrichsfelde','Yardley','Abadina','II 28:g,(m),[s],t:1,5','Croft','II 28:g,m,t:e,n,x','II 28:g,m,t:z39','II 28:g,s,t:e,n,x','Ona','II 28:m,t:[e,n,x]','Vinohrady','II 28:m,t:z39','Morillons','Doorn','Cotham','Volkmarsdorf','Dieuppeul','Warnemuende','Kuessel','Douala','Guildford','Ilala','Adamstown','Ikeja','IIIb 28:k:1,7','Taunton','Ank','Leoben','Vitkin','Nashua','Ramsey','Catalunia','Penilla','II 28:l,z28:1,5','Fajara','II 28:l,z28:e,n,x','Bassadji','Kibusi','II 28:r:e,n,z15','Fairfield','Chicago','Banco','Sanktgeorg','Oskarshamn','Nima','Pomona','Kitenge','Telaviv','Shomolu','Selby','Vanier','II 28:z:1,5','Doel','Ezra','Brisbane','II 28:z:z39','Cannobio','Teltow','Babelsberg','Kethiabarny','Rogy','Farakan','Libreville','Malaysia','Umbilo','Luckenwalde','Moroto','IIIb 28:z10:z','Djermaia','II 28:z29:1,5','II 28:z29:e,n,x','Konolfingen','Babili','Santander','Aderike','Overvecht','Zehlendorf','Guarapiranga','Doulassame','II 30:a:z39','Louga','Aschersleben','Tempe','Urbana','Neudorf','Coppettswood','II 30:b:z6','Kakikoka','Zaire','Morningside','II 30:c:z39','Messina','Livulu','Torhout','Godesberg','II 30:g,m,s:e,n,x','Giessen','IV 30:g,m,t,z51:–','Sternschanze','II 30:g,t:–','Wayne','II 30:m,t:–','Landau','Morehead','Mjordan','Soerenga','Hilversum','Ramatgan','Aqua','Angoda','Odozi','II 30:k:e,n,x,z15','Scarborough','Ligeo','Donna','Ockenheim','Morocco','II 30:l,z28:z6','Grandhaven','Gege','Quincy','Matopeni','Bietri','Steinplatz','Baguirmi','Nijmegen','Hohentwiel','Stoneferry','Bodjonegoro','II 30:z6:1,6','Sada','Senneville','Kumasi','II 30:z10:e,n,x,z15','Aragua','Kokoli','Wuiti','Ago','II 30:z39:1,7','Umhlatazana','Tchad','Keurmassar','Gouloumbo','Yolo','II 35:d:1,5','Dembe','Gassi','Adelaide','Ealing','II 35:g,m,s,t:–','Ebrie','Anecho','II 35:g,t:1,5','II 35:g,t:z42','Agodi','IIIa 35:g,z51:–','II 35:g,z62:e,n,x','Monschaui','II 35:m,t:–','IIIb 35:i:e,n,x,z15','Gambia','Bandia','IIIb 35:i:z','Evry','IIIb 35:i:z35','IIIb 35:i:z53','Penzing','IIIb 35:k:e,n,x,z15','IIIb 35:k:z','IIIb 35:(k):z35','IIIb 35:k:z53:[z50]','IIIb 35:l,v:1,5,7','IIIb 35:l,v:e,n,x,z15:[z50]','IIIb 35:l,v:z35:[z67]','II 35:l,z28:–','IIIb 35:r:e,n,x,z15','Massakory','IIIb 35:r:z','IIIb 35:r:z35','IIIb 35:r:z61','Baltimore','Alachua','IIIa 35:z4,z23:–','Westphalia','IIIa 35:z4,z24:–','IIIa 35:z4,z32:–','Camberene','Enschede','Ligna','IIIb 35:z10:z35','II 35:z29:e,n,x','Widemarsh','IIIa 35:z29:–','IIIa 35:z36:–','Haga','IIIb 35:z52:1,5,7','IIIb 35:z52:e,n,x,z15','IIIb 35:z52:z','IIIb 35:z52:z35','Oran','II 38:b:1,2','Rittersbach','Sheffield','Kidderminster','Willamette','II 38:d:[1,5]','II 38:d:z39','Thiaroye','Kasenyi','Korovi','II 38:g,t:–','IIIa 38:g,z51:–','IV 38:g,z51:–','Rothenburgsort','Mgulani','Lansing','IIIb 38:i:z','IIIb 38:i:z53','Echa','Mango','Inverness','Njala','IIIb 38:k:e,n,x,z15','IIIb 38:k:z','IIIb 38:k:z53','IIIb 38:(k):1,5,7','IIIb 38:(k):z35:[z56]','IIIb 38:(k):–','IIIb 38:(k):z55','Alger','Kimberley','Taylor','Roan','IIIb 38:l,v:z','IIIb 38:l,v:z35','IIIb 38:l,v:[z53]:[z54]','Lindi','IIIb 38:r:1,5,7','Emmastad','IIIb 38:r:e,n,x,z15','IIIb 38:r:z:[z57]','IIIb 38:r:z35','Freetown','Colombo','Perth','Stachus','Yoff','IIIa 38:z4,z23:–','IV 38:z4,z23:–','Bangkok','Neunkirchen','IIIb 38:z10:z','IIIb 38:z10:z53','Carpentras','Klouto','IIIb 38:z52:z35','IIIb 38:z52:z53','IIIb 38:z53:–:[z47],[z50],[z76]','IIIb 38:z61:[z53]','II 39:a:z39','Wandsworth','Abidjan','II 39:c:e,n,x','Logone','Bruebach','Mara','II 39:e,n,x:1,7','Dietrichsdorf','II 39:[g],m,t:[e,n,x]','Hofit','Cumberland','Alma','Champaign','Newjersey','II 39:l,v:1,5','Kokomlemle','Oerlikon','II 39:l,z28:e,n,x','II 39:l,z28:z39','Frimley','Anfo','Windermere','Delan','Namur','Hegau','II 39:z10:z6','Landeskrone','II 39:–:1,7','Shikmonah','Salinas','Greiz','II 40:a:z39','Riogrande','Saugus','Johannesburg','Duval','Benguella','II 40:b:–','II [1],40:c:e,n,x,z15','II 40:c:z6','II [1],40:c:z39','Driffield','II 40:d:–','Tilene','II [1],40:e,n,x:1,[5],7','II [1],40:e,n,x,z15:1,6','Bijlmer','Athens','II [1],40:g,[m],[s],[t]:e,n,x','II [1],40:g,[m],[s],t:[1,5]','II [1],40:g,t:e,n,x,z15','II 40:g,t:z39','IV [1],40:g,t:–','II [1],40:g,[m],[s],t:z42','IIIa 40:g,z51:–','IIIb 40:g,z51:e,n,x,z15','IV [1],40:g,z51:–','II 40:m,t:e,n,x','II 40:m,t:z39','II [1],40:m,t:z42','IV 40:m,t:–','IIIb 40:i:1,5,7','Goulfey','Allandale','Hann','II [1],40:k:e,n,x,z15','IIIb 40:k:z:z57','II 40:k:z6','IIIb 40:k:z53','Millesi','Canary','II 40:l,v:e,n,x','IIIb 40:l,v:z','IIIb [1],40:l,v:z53','Overchurch','II 40:l,z28:e,n,x','Tiko','Bukavu','II [1],40:l,z28:1,5:z42','Santhiaba','II [1],40:l,z28:z39','IIIb 40:r:z53','Odienne','II [1],40:z:1,5','Casamance','Nowawes','II [1],40:z:z6','II [1],40:z:z39','II 40:z:z42','IIIa 40:z4,z23:–','IV [1],40:z4,z23:–','II 40:z4,z24:z39','IIIa 40:z4,z24:–','IV [1],40:z4,z24:–','IIIa 40:z4,z32:–','IV [1],40:z4,z32:–','II [1],40:z6:1,5','Trotha','II 40:z10:e,n,x','IIIb 40:z10:z35','Omifisan','IIIa 40:z29:–','II [1],40:z35:e,n,x,z15','Yekepa','V [1],40:z35:–','IIIa 40:z36:–','II [1],40:z39:1,5:z42','II [1],40:z39:1,6','IIIb 40:z39:1,6','II 40:z39:1,7','Karamoja','II [1],40:z42:1,6','II [1],40:[z42]:1,(5),7','II [1],40:z81:z6','V [1],40:z81:–','Burundi','II 41:b:1,5','Vaugirard','VI 41:b:1,7','Vietnam','Sica','Lonestar','IIIb 41:c:e,n,x,z15','II 41:c:z6','Egusi','II 41:d:z6','II 41:g,m,s,t:z6','II 41:g,t:–','IIIa 41:g,z51:–','Leatherhead','Samaru','Verona','Ferlo','II 41:k:1,6','II 41:k:z6','IIIb 41:(k):z35','II 41:l,z13,z28:e,n,x,z15','Lubumbashi','Konongo','Bruxelles','II 41:z:1,5','Sally','Bofflens','Waycross','IIIa 41:z4,z23:–','IV 41:z4,z23:–','IIIa 41:z4,z23,z32:–','Ipswich','IIIa 41:z4,z24:–','IIIa 41:z4,z32:–','II 41:z10:1,2','Leipzig','Landala','Inpraw','II 41:z10:e,n,x,z15','II 41:z10:z6','Lodz','IIIa 41:z29:–','IV 41:z29:–','Ahoutoue','IIIa 41:z36:–','IV 41:z36:–','Offa','IV 41:z52:–','II 41:–:1,6','Faji','II 42:b:1,5','Orbe','II 42:b:e,n,x,z15','Tomegbe','Frederiksberg','Egusitoo','II 42:b:z6','Antwerpen','Kampala','Gostenhof','II 42:d:z6','II 42:e,n,x:1,6','II 42:g,t:–','Maricopa','IIIa 42:g,z51:–','IV [1],42:g,z51:–','II 42:m,t:[e,n,x,z15]','Waral','Kaneshie','Borromea','Middlesbrough','Haferbreite','IIIb 42:k:e,n,x,z15','IIIb 42:k:z','Gwale','IIIb 42:(k):z35','IIIb 42:l,v:1,5,7:[z76]','Portedeslilas','II 42:l,v:e,n,x,z15','IIIb 42:l,v:e,n,x,z15','Coogee','IIIb 42:l,v:z','IIIb 42:l,v:z53','II [1],42:l,w:e,n,x','Parakou','II [1],42:l,[z13],z28:[z6]','Sipane','Brive','IIIb 42:r:z','IIIb 42:r:z53','II 42:r:–','IIIb 42:r:–:[z50]','Spalentor','Harvestehude','II 42:z:1,5','Ursenbach','II 42:z:e,n,x,z15','Melbourne','II 42:z:z6','Gera','Broc','IIIa 42:z4,z23:–','Toricada','IIIa 42:z4,z24:–','IV [1],42:z4,z24:–','II 42:z6:1,6','II 42:z10:1,2','II 42:z10:e,n,x,z15','IIIb 42:z10:e,n,x,z15:[z60]','IIIb 42:z10:z','Loenga','II 42:z10:z6','IIIb 42:z10:z35','IIIb 42:z10:z67','Djama','II 42:z29:–','Kahla','Hennekamp','Tema','Weslaco','IV 42:z36:–','Vogan','Taset','IIIb 42:z52:z','IIIb 42:z57:1,5','Graz','Berkeley','II 43:a:1,5','II 43:a:z6','Niederoderwitz','Ede','II 43:b:z42','Montreal','Orleans','II 43:d:e,n,x,z15','II 43:d:z39','II 43:d:z42','II 43:e,n,x,z15:1,(5),7','II 43:e,n,x,z15:1,6','Milwaukee','II 43:g,m,[s],t:[z42]','II 43:g,t:[1,5]','IIIb 43:g,t:–','IIIa 43:g,z51:–','IV 43:g,z51:–','II 43:g,z62:e,n,x','Mbao','Voulte','Thetford','Ahuza','IIIb 43:k:z','IIIb 43:l,v:z53:[z56]','Epalinges','Sudan','II 43:l,z13,z28:1,5','II 43:m,t:–','IIIb 43:r:e,n,x,z15','IIIb 43:r:z','IIIb 43:r:z53','Farcha','Kingabwa','Ogbete','II 43:z:1,5','Arusha','Farmingdale','II 43:z4,z23:–','IIIa 43:z4,z23:–','IV 43:z4,z23:–','IIIa 43:z4,z24:–','IV 43:z4,z24:–','IV 43:z4,z32:–','Adana','II 43:z29:e,n,x','II 43:z29:z42','Makiling','IIIa 43:z29:–','IV 43:z29:–','Ahepe','IIIa 43:z36:–','IV 43:z36,z38:–','Irigny','Darfur','II 43:z42:[1,5,7]','IIIb 43:z52:z53','IV 44:a:–','Niakhar','Tiergarten','Niarembe','Shahalam','Elbeuf','Sedgwick','Madigan','Bergerac','Quebec','Bobo','Kermel','Fischerstrasse','Palamaner','II [1],44:e,n,x:1,6','Vleuten','Gamaba','Splott','II 44:g,t:z42','IIIb [1],44:g,t:1,5:z42','Carswell','IV 44:g,z51:–','Muguga','II [1],44:m,t:z42','Maritzburg','Lawra','Malika','Albertbanjul','Brefet','V 44:r:–','Brackenridge','Uhlenhorst','Bolama','Goodmayes','Kua','Ploufragan','II 44:z4,z23:–','IIIa 44:z4,z23:–','IV 44:z4,z23:–','IIIa 44:z4,z23,z32:–','Christiansborg','IIIa 44:z4,z24:–','IV 44:z4,z24:–','IIIa 44:z4,z32:–','IV [1],44:z4,z32:–','Guinea','Llobregat','II 44:z29:e,n,x:z42','Zinder','IV 44:z29:–','IV 44:z36,[z38]:–','Koketime','II [1],44:z39:e,n,x,z15','V 44:z39:–','VI 45:a:e,n,x','Meekatharra','II 45:a:z10','Riverside','Leinster','Fomeco','Deversoir','Dugbe','Karachi','Warmsen','Suelldorf','Tornow','II 45:g,m,s,t:1,5','II 45:g,m,s,t:e,n,x','II 45:g,m,t:e,n,x,z15','Binningen','IIIa 45:g,z51:–','IV 45:g,z51:–','II 45:m,t:1,5','Apapa','Verviers','Casablanca','Cairns','Imo','Kofandoka','II 45:z:1,5','Yopougon','II 45:z:z39','IIIa 45:z4,z23:–','IV 45:z4,z23:–','Transvaal','IIIa 45:z4,z24:–','IIIa 45:z4,z32:–','Aprad','Jodhpur','II 45:z29:1,5','II 45:z29:e,n,x','II 45:z29:z42','IIIa 45:z29:–','Lattenkamp','Balcones','IIIa 45:z36:–','IV 45:z36,z38:–','II 47:a:1,5','II 47:a:e,n,x,z15','Wenatchee','II 47:b:1,5','II 47:b:e,n,x,z15','Sya','II 47:b:z6','IIIb 47:c:1,5,7','Kodjovi','IIIb 47:c:e,n,x,z15:[z57]','IIIb 47:c:z','Braeswood','IIIb 47:c:z35','II 47:d:1,5','Stellingen','II 47:d:e,n,x,z15','II 47:d:z39','II 47:e,n,x,z15:1,6','Sljeme','Luke','Carshalton','II 47:[g,t]:e,n,x','IIIa 47:g,z51:–','Mesbit','IIIb 47:i:e,n,x,z15:[z50]','Bergen','IIIb 47:i:z','IIIb 47:i:z35','IIIb 47:i:z53:[z57],[z84]','Staoueli','Bootle','IIIb 47:k:1,5,7','Dahomey','IIIb 47:k:e,n,x,z15','Lyon','IIIb 47:k:z','IIIb 47:k:z35','IIIb 47:k:z53:[z84]','IIIb 47:l,v:1,[5],7:[z50]','Drac','IIIb 47:l,v:e,n,x,z15','IIIb 47:l,v:z','IIIb 47:l,v:z35','IIIb 47:l,v:z53','IIIb 47:l,v:z57','IV 47:l,v:–','Teshie','IIIb 47:r:e,n,x,z15','Dapango','IIIb 47:r:1,5,7','IIIb 47:r:z','IIIb 47:r,[i]:z35','IIIb 47:r:z53:[z70],[z74],[z77],[z90]','Moualine','Blitta','Mountpleasant','Kaolack','II 47:z:e,n,x,z15','II 47:z:z6','Forges','Tabligbo','Fehrbellin','Bere','Binche','IIIa 47:z4,z23:–','Tamberma','II 47:z6:1,6','IIIb 47:z10:1,5,7','Namoda','IIIb 47:z10:z','IIIb 47:z10:z35','II 47:z29:e,n,x,z15','Ekpoui','IIIa 47:z29:–','Yombesali','Bingerville','IV 47:z36:–','Alexanderplatz','Quinhon','IIIb 47:z52:1,5:z54','IIIb 47:z52:1,5,7','IIIb 47:z52:e,n,x,z15','IIIb 47:z52:z','IIIb 47:z52:z35','IIIb 47:z53:–:[z90]','Hisingen','II 48:a:z6','II 48:a:z39','II 48:b:z6','II 48:b:e,n,x,z15','V 48:b:–','IIIb 48:c:z','II 48:d:1,2','II 48:d:z6','Buckeye','Fitzroy','II 48:e,n,x,z15:z6','II 48:g,m,t:–','IIIa 48:g,z51:–','IV 48:g,z51:–','IIIb 48:i:z:[z72]','IIIb 48:i:z35:[z57]','IIIb 48:i:z53','IIIb 48:i:z61','V 48:i:–','IIIb 48:k:1,5,(7)','II 48:k:e,n,x,z15','IIIb 48:k:e,n,x,z15','Dahlem','IIIb 48:k:z','IIIb 48:k:z35:[z75]','II 48:k:z39','IIIb 48:k:z53','IIIb 48:k:z57','Australia','IIIb 48:l,v:1,5,(7):[z47],[z50],[z89]','IIIb 48:l,v:z','IIIb 48:l,w:1,5,7:[z50]','IIIb 48:r:e,n,x,z15','IIIb 48:r:z','Toucra','II 48:z:1,5','IIIb 48:z:1,5,7','IIIa 48:z4,z23:–','IV 48:z4,z23:–','IIIa 48:z4,z23,z32:–','Djakarta','IIIa 48:z4,z24:–','IIIb 48:z4,z24:–','IV 48:z4,z24:–','IIIa 48:z4,z32:–','IV 48:z4,z32:–','II 48:z10:[1,5]','VI 48:z10:1,5','II 48:z10:1,6','Isaszeg','IIIb 48:z10:e,n,x,z15','IIIb 48:z10:z','II 48:z29:–','IIIa 48:z29:–','IV 48:z29:–','IIIb 48:z35:z52','V 48:z35:–','IIIa 48:z36:–','IV 48:z36,[z38]:–','II 48:z39:z81','V 48:z39:–','V 48:z41:–','IIIb 48:z52:e,n,x,z15','IIIb 48:z52:z','V 48:z65:–','V 48:z81:–','II 48:z81:1,5,7','IV 50:a:–','Rochdale','II 50:b:z6','IV 50:b:–','Hemingford','IV 50:d:–','II 50:e,n,x:1,7','II 50:g,[m],s,t:[1,5]','IIIa 50:g,z51:–','IV 50:g,z51:–','II 50:g,z62:e,n,x','II 50:m,t:z6:z42','IIIb 50:i:1,5,7','IIIb 50:i:e,n,x,z15','IIIb 50:i:z','IIIb 50:i:z53','IIIb 50:k:1,5,7','II 50:k:e,n,x:z42','IIIb 50:k:e,n,x,z15','IIIb 50:k:z:[z50],[z57],[z68],[z86]','II 50:k:z6','IIIb 50:k:z35','IIIb 50:k:z53','Fass','IIIb 50:l,v:e,n,x,z15','IIIb 50:l,v:z','IIIb 50:l,v:z35','IIIb 50:l,v:z57','VI 50:l,v:z67','II 50:l,w:e,n,x,z15:z42','II 50:l,z28:z42','IIIb 50:r:1,5,(7)','IIIb 50:r:e,n,x,z15','IIIb 50:r:z:[z67]','IIIb 50:r:z35:[z58]','IIIb 50:r:z53','Dougi','II 50:z:e,n,x','IIIb 50:z:z52','IIIa 50:z4,z23:–','IV 50:z4,z23:–','IIIa 50:z4,z23,z32:–','IIIa 50:z4,z24:–','IV 50:z4,z24:–','IIIa 50:z4,z32:–','IV 50:z4,z32:–','IIIb 50:z10:z:[z56]','II 50:z10:z6:z42','IIIb 50:z10:z53','Ivorycoast','IIIa 50:z29:–','IIIa 50:z36:–','II 50:z42:1,7','IIIb 50:z52:1,5,7','IIIb 50:z52:z35','IIIb 50:z52:z53','IV 51:a:–','Windsheim','Tione','Karaya','IV 51:b:–','II 51:c:–','Gokul','Meskin','II 51:g,s,t:e,n,x','IIIa 51:g,z51:–','Djinten','Elmdon','Kabete','Dan','IIIb 51:k:z35','Harcourt','Overschie','Dadzie','IIIb 51:l,v:z','Moundou','II 51:l,z28:z6','II 51:l,z28:z39','Mali','Lutetia','Rosslyn','Antsalova','Treforest','Lechler','IIIa 51:z4,z23:–','IV 51:z4,z23:–','IIIa 51:z4,z24:–','IIIa 51:z4,z32:–','Bergues','II 51:z29:e,n,x,z15','IIIa 51:z36:–','II 51:–:1,7','Uithof','Ord','Molesey','Flottbek','II 52:c:k','Utrecht','II 52:d:e,n,x,z15','II 52:d:z39','Butare','Derkle','Saintemarie','II 52:g,t:–','Bordeaux','IIIb 52:k:e,n,x,z15','IIIb 52:k:z35','IIIb 52:k:z53','IIIb 52:l,v:z53','Marsabit','II 52:z:z39','IIIb 52:z:z52','II 52:z39:1,5,7','II 52:z44:1,5,7','II 53:c:1,5:[R1…]','II 53:d:1,5','II [1],53:d:z39','II 53:d:z42','IIIa 53:g,z51:–','IV [1],53:g,z51:–','IIIb 53:i:z','IIIb 53:k:e,n,x,z15','IIIb 53:k:z','IIIb 53:(k):z35','IIIb 53:k:z53','IIIb 53:l,v:e,n,x,z15','IIIb 53:l,v:z','IIIb 53:l,v:z35','II 53:l,z28:e,n,x','II 53:l,z28:z6','II 53:l,z28:z39','IIIb 53:r:z','IIIb 53:r:z35','IIIb 53:r:z68','II 53:z:1,5','IIIb 53:z:1,5,(7)','II 53:z:z6','IIIa 53:z4,z23:–','IV 53:z4,z23:–','IIIa 53:z4,z23,z32:–','II 53:z4,z24:–','IIIa 53:z4,z24:–','IIIb 53:z10:z','IIIb 53:z10:z35','IIIa 53:z29:–','IV [1],53:z36,z38:–','IIIb 53:z52:z35','IIIb 53:z52:z53','Leda','Tonev','Winnipeg','Rossleben','Borreze','Uccle','Newholland','Poeseldorf','Ochsenwerder','Czernyring','Steinwerder','Yerba','Canton','Barry','Mundubbera','II 55:k:z39','II 56:b:[1,5]','II 56:d:–','II 56:e,n,x:1,7','II 56:l,v:z39','II 56:l,z28:–','II 56:z:z6','IIIa 56:z4,z23:–','IIIa 56:z4,z23,z32:–','II 56:z10:e,n,x','IIIa 56:z29:–','Antonio','II 57:a:z42','Maryland','Batonrouge','IIIb 57:c:e,n,x,z15','IIIb 57:c:z:[z70],[z90]','II 57:d:1,5','II 57:g,[m],s,t:z42','II 57:g,t:–','IIIb 57:i:e,n,x,z15','IIIb 57:i:z','IIIb 57:k:e,n,x,z15','IV 57:z4,z23:–','IIIb 57:z10:z','II 57:z29:z42','II 57:z39:e,n,x,z15','II 57:z42:1,6:z53','II 58:a:z6','II 58:b:1,5','II 58:c:z6','II 58:d:z6','IIIb 58:i:e,n,x,z15','IIIb 58:i:z53','IIIb 58:k:z','IIIb 58:l,v:e,n,x,z15','IIIb 58:l,v:z35','II 58:l,z13,z28:1,5','II 58:l,z13,z28:z6','IIIb 58:r:e,n,x,z15','IIIb 58:r:z','IIIb 58:r:z53:[z47],[z57],[z70],[z73]','II 58:z6:1,6','II 58:z10:1,6','IIIb 58:z10:e,n,x,z15','II 58:z10:z6','IIIb 58:z10:z53:[z50]','II 58:z39:e,n,x,z15','IIIb 58:z52:z','IIIb 58:z52:z35','IIIb 59:c:e,n,x,z15','IIIb 59:i:e,n,x,z15','IIIb 59:i:z','IIIb 59:i:z35:[z84]','IIIb 59:(k):e,n,x,z15','II 59:k:z65','IIIb 59:(k):z','IIIb 59:(k):z35','IIIb 59:k:z53','IIIb 59:l,v:z','IIIb 59:l,v:z53','IIIb 59:r:z35','II [1],59:z:z6','IIIa 59:z4,z23:–','IIIb 59:z10:z53','IIIb 59:z10:z57','IIIa 59:z29:–','IIIa 59:z36:–','VI 59:z36:–','IIIb 59:z52:z53','II 60:b:–:[R1…]','II 60:g,m,t:z6','IIIb 60:i:[e,n,x,z15]:[z50]','IIIb 60:i:[z]:[z50]','IIIb 60:i:[z35]:[z50]','IIIb 60:k:z','IIIb 60:k:z35','IIIb 60:(k):z53','IIIb 60:l,v:z','IIIb 60:r:e,n,x,z15','IIIb 60:r:z','IIIb 60:r:z35','IIIb 60:r:z53','II 60:z:e,n,x','IIIb 60:z10:z','IIIb 60:z10:z35','II 60:z10:z39','IIIb 60:z10:z53','II 60:z29:e,n,x','V 60:z41:–','IIIb 60:z52:1,5,[7]','IIIb 60:z52:z','IIIb 60:z52:z35','IIIb 60:z52:z53','IIIb 61:c:1,5,(7)','IIIb 61:c:z35','IIIb 61:i:e,n,x,z15','IIIb 61:i:z','IIIb 61:i:z35','IIIb 61:i:z53','IIIb 61:k:1,5,(7)','IIIb 61:k:z35','IIIb 61:(k):z53','IIIb 61:l,v:1,5,7:[z57]','IIIb 61:l,v:z','IIIb 61:l,v:z35','IIIb 61:r:1,5,7','IIIb 61:r:z','IIIb 61:r:z35','IIIb 61:r:z53:[z47],[z50]','IIIb 61:z10:z35','V 61:z35:–','IIIb 61:z52:1,5,7','IIIb 61:z52:z','IIIb 61:z52:z35','IIIb 61:z52:z53','IIIa 62:g,z51:–','IIIa 62:z4,z23:–','IIIa 62:z4,z32:–','IIIa 62:z29:–','IIIa 62:z36:–','IIIa 63:g,z51:–','IIIb 63:(k):z','IIIa 63:z4,z23:–','IIIa 63:z4,z32:–','IIIa 63:z36:–','IIIb 65:c:1,5,7','IIIb 65:c:z','IIIb 65:c:z53','II 65:g,t:–','IIIb 65:i:e,n,x,z15','IIIb 65:i:z','IIIb 65:(k):z','IIIb 65:(k):z35','IIIb 65:(k):z53','IIIb 65:l,v:e,n,x,z15','IIIb 65:l,v:z','IIIb 65:l,v:z35','IIIb 65:l,v:z53','IIIb 65:r:z35','IIIb 65:z10:e,n,x,z15','IIIb 65:z10:z','IIIb 65:z52:e,n,x,z15','IIIb 65:z52:z','IIIb 65:z52:z35','IIIb 65:z52:z53','II 65:–:1,6','V 66:z35:–','V 66:z39:–','V 66:z41:–','V 66:z65:–','V 66:z81:–','Choleraesuis or Typhisuis','Miami or Sendai','Crossness','Amager var. 15+', 'Amersfoort var. 14+', 'Amsterdam var. 15+', 'Anatum var. 15+', 'Anatum var. 15+,34+', 'Butantan var. 15+,34+', 'Cerro var. 14+', 'Gdansk var. 14+', 'Give var. 15+', 'Give var. 15+,34+', 'Goelzau var. 15+', 'Lexington var. 15+', 'Lexington var. 15+,34+', 'Lille var. 14+', 'Livingstone var. 14+', 'London var. 15+', 'Meleagridis var. 15+', 'Meleagridis var. 15+,34+', 'Muenster var. 15+', 'Muenster var. 15+,34+', 'Nchanga var. 15+', 'Nyborg var. 15+', 'Oranienburg var. 14+', 'Orion var. 15+', 'Orion var. 15+,34+', 'Oxford var. 15+,34+', 'Paratyphi B var. L(+) tartrate (= d–tartrate)+', 'Rissen var. 14+', 'Shangani var. 15+', 'Souza var. 15+', 'Stanleyville var. 27+', 'Stockholm var. 15+', 'Uganda var. 15+', 'Vejle var. 15+', 'Vejle var. 15+,[Rz27]', 'Weltevreden var. 15+', 'Westhampton var. 15+', 'Westhampton var. 15+,34+']
std_wklm_name = 'paratyphi a','nitra','kiel','koessen','kisangani','hessarek','fulica','arechavaleta','bispebjerg','tinda','ii 1,4,5,12,27:a:e,n,x','huettwilen','nakuru','ii 1,4,12,27:a:z39','paratyphi b','limete','ii 4,12:b:1,5','canada','uppsala','abony','ii 1,4,5,12,27:b:e,n,x','wagenia','wien','tripoli','schleissheim','legon','abortusovis','altendorf','bissau','jericho','hallfold','bury','stanley','eppendorf','brezany','schwarzengrund','ii 4,12:d:e,n,x','sarajane','duisburg','mons','ayinde','chennai','saintpaul','reading','eko','kaapstad','chester','sandiego','chartres','ii 4,12:e,n,x:1,2,7','ii 1,4,12,27:e,n,x:1,5,7','derby','agona','ii 1,4,5,12:f,g,t:z6:z42','essen','hato','ii 1,4,12,27:g,m,s,t:e,n,x','ii 1,4,12,27:g,m,t:1,5','ii 4,12:g,m,t:z39','california','kingston','budapest','travis','tennyson','ii 4,12:g,z62:–','banana','madras','typhimurium','lagos','agama','farsta','tsevie','gloucester','tumodi','ii 4,12,27:i:z35','massenya','neumuenster','ii 1,4,12,27:k:1,6','ljubljana','texas','fyris','azteca','clackamas','bredeney','kimuenza','ii 1,4,12,27:l,v:e,n,x','brandenburg','ii 1,4,12,27:l,v:z39','mono','togo','ii 4,12:l,w:e,n,x','blancmesnil','ayton','kunduchi','tyresoe','haduna','kubacha','kano','vom','reinickendorf','ii 1,4,12:l,z28:e,n,x','heidelberg','bradford','winneba','remo','bochum','southampton','drogana','africana','coeln','trachau','finaghy','teddington','ball','jos','kamoru','shubra','kiambu','ii 1,4,12,27:z:1,5','loubomo','indiana','ii 4,12:z:1,7','neftenbach','ii 1,4,12,27:z:e,n,x','koenigstuhl','preston','entebbe','ii 4,12:z:z39','stanleyville','perrosguirec','vuadens','kalamu','haifa','ituri','tudu','harrow','albert','tokoin','mura','fortune','vellore','brancaster','ii 1,4,12:z29:e,n,x','pasing','tafo','sloterdijk','yaounde','tejas','yverdon','wilhelmsburg','ii 1,4,12,27:z39:1,5,7','thayngen','maska','abortusequi','mygdal','sanjuan','ii 6,7,14:a:1,5','umhlali','austin','oslo','denver','coleypark','damman','ii 6,7:a:z6','ii 6,7:a:z42','brazzaville','ii 6,7:b:1,5','edinburg','adime','koumra','lockleaze','georgia','ii 6,7:b:e,n,x:z42','ohio','leopoldville','kotte','ii 6,7:b:z39','hissar','paratyphi c','choleraesuis','typhisuis','birkenhead','schwabach','cotonou','namibia','kaduna','kisii','isangi','kivu','kambole','amersfoort','gombe','livingstone','wil','nieukerk','ii 6,7:d:z42','larochelle','lomita','norwich','nola','braenderup','ii 6,7:e,n,x:1,6:z42','kastrup','rissen','eingedi','afula','montevideo','lubbock','ii 6,7:g,m,s,t:e,n,x','ii 6,7:g,m,s,t:1,5','ii 6,7:g,m,s,t:z39','ii 6,7:g,m,s,t:z42','othmarschen','plumaugat','menston','ii 6,7:g,t:e,n,x:z42','riggil','alamo','larose','iv 6,7:g,z51:–','haelsingborg','winston','oakey','ii 6,7:m,t:–','oranienburg','augustenborg','oritamerin','garoli','lika','athinai','norton','stuttgart','galiema','thompson','daytona','baiboukoum','singapore','escanaba','iiib 6,7:k:z:z55','ii 6,7:k:z6','ashford','concord','irumu','iiib 6,7:l,v:1,5,7','mkamba','kortrijk','bonn','potsdam','gdansk','coromandel','iiib 6,7:l,v:z53','gabon','colorado','ii 6,7:l,w:1,5,7','langeveld','ii 6,7:l,w:z42','nessziona','kenya','neukoelln','makiso','strathcona','ii 6,7:l,z28:1,5:z42','ii 6,7:l,z28:e,n,x','ii 6,7:l,z28:z6','virchow','infantis','nigeria','colindale','papuana','grampian','richmond','bareilly','oyonnax','gatow','hartford','mikawasima','chile','poitiers','ii 6,7:z:1,5','oakland','cayar','ii 6,7:z:e,n,x','ii 6,7:z:e,n,x,z15','businga','bruck','ii 6,7:z:z6','ii 6,7:z:z39','ii 6,7:z:z42','obogu','planckendael','aequatoria','goma','ii 6,7:z4,z23:–','iv 6,7:z4,z23:–','zollikerberg','ii 6,7:z4,z24:z42','somone','iv 6,7:z4,z24:–','ii 6,7:z6:1,7','menden','inganda','eschweiler','ngili','djugu','mbandaka','jerusalem','redba','omuna','tennessee','ii 6,7:z29:z42','tienba','palime','tampico','ii 6,7:z36:z42','iv 6,7:z36:–','rumford','ii 6,7:z38:e,n,x','lille','iiib 6,7,14:z39:1,2','ii 6,7:z39:1,5,7','vi 6,7:z41:1,7','hillsborough','tamilnadu','ii 6,7:z42:1,5,7','bulovka','ii 6,7:–:1,6','be','valdosta','doncaster','curacao','nordufer','narashino','ii 6,8:a:e,n,x','leith','ii 6,8:a:z39','ii 6,8:a:z52','djelfa','skansen','korbol','nagoya','ii 6,8:b:1,5','stourbridge','sanga','eboko','konstanz','gatuni','shipley','presov','bukuru','heistopdenberg','tounouma','banalia','wingrove','gaillac','utah','bronx','belfast','alexanderpolder','santiago','belem','quiniela','tado','virginia','muenchen','yovokome','manhattan','portanigra','dunkwa','sterrenbos','herston','labadi','ii 6,8:d:z6:z42','bardo','newport','ferruch','kottbus','cremieu','atakpame','fillmore','tshiongwe','rechovot','sandow','ii 6,8:f,g,m,t:e,n,x','emek','chincol','ii 6,8:g,m,t:1,7','reubeuss','alminko','nanergou','yokoe','ii 6,8:m,t:1,5','ii 6,8:m,t:e,n,x','bassa','lindenburg','bargny','takoradi','warnow','malmoe','bonariensis','aba','magherafelt','cyprus','kentucky','kallo','haardt','blockley','sandbanks','schwerin','charlottenburg','iiib 8:k:z35','pakistan','litchfield','loanda','amherstiana','manchester','holcomb','ii 6,8:l,v:e,n,x','edmonton','lund','fayed','ii 6,8:l,w:z6:z42','hiduddify','breukelen','ii 6,8:l,z28:e,n,x','bsilla','hindmarsh','bovismorbificans','colchester','noya','akanji','cocody','hidalgo','brikama','goldcoast','iiib 8:r:z','altona','giza','lamphun','brunei','tananarive','bulgaria','ii 6,8:y:1,6:z42','alagbon','inchpark','sunnycove','daarle','praha','kralingen','benue','sindelfingen','mowanjum','ii 6,8:z:1,5','marmande','phaliron','kalumburu','kuru','daula','bellevue','lezennes','breda','chailey','dabou','corvallis','nuorikkala','albany','duesseldorf','tallahassee','bazenheid','zerifin','paris','mapo','cleveland','istanbul','hadar','chomedey','glostrup','remiremont','molade','wippra','ii 6,8:z29:1,5','ii 6,8:z29:e,n,x:z42','tamale','uno','ii 6,8:z29:e,n,x','kolda','yarm','angers','apeyeme','diogoye','aesch','sendai','miami','ii 9,12:a:1,5','os','saarbruecken','lomalinda','ii 1,9,12:a:e,n,x','durban','ii 9,12:a:z39','ii 1,9,12:a:z42','onarimon','frintrop','bata','ii 1,9,12:b:e,n,x','mana','ii 1,9,12:b:z6','ii 1,9,12:b:z39','goeteborg','ipeko','elokate','alabama','ridge','ndolo','tarshyne','eschberg','ii 1,9,12:d:e,n,x','viikki','bangui','zega','jaffna','ii 9,12:d:z39','typhi','bournemouth','eastbourne','westafrica','israel','ii 9,12:e,n,x:1,5,7','ii 9,12:e,n,x:1,6','berta','enteritidis','gueuletapee','blegdam','ii 1,9,12:g,m,s,t:1,5,7:z42','ii 1,9,12:g,m,s,t:e,n,x','ii 9,12:g,m,t:z39','dublin','naestved','rostock','moscow','ii 9,12:g,s,t:e,n,x','newmexico','ii 1,9,12:g,z62:e,n,x','antarctica','rosenberg','ii 9,12:m,t:e,n,x','pensacola','ii 1,9,12:m,t:1,5','ii 1,9,12:m,t:z39','seremban','claibornei','goverdhan','mendoza','panama','houston','kapemba','zaiman','ii 9,12:l,v:e,n,x','goettingen','ii 9,12:l,v:z39','victoria','ii 1,9,12:l,w:e,n,x','itami','miyazaki','napoli','javiana','kotu','ii 9,12:l,z28:1,5:z42','ii 9,12:l,z28:e,n,x','york','jamaica','camberwell','yellowknife','campinense','lome','powell','ii 1,9,12:y:z39','mulhouse','lawndale','kimpese','ii 1,9,12:z:1,7','aurelianis','ii 1,9,12:z:z6','ii 9,12:z:z39','wangata','natal','franken','portland','treguier','ruanda','ii 9,12:z29:1,5','ii 1,9,12:z29:e,n,x','penarth','elomrane','ii 1,9,12:z39:1,7','ottawa','ii 1,9,12:z42:1,5,7','gallinarum','detmold','poole','baildon','doba','montaigu','cheltenham','zadar','worb','ii 9,46:b:e,n,x','bamboye','linguere','kolar','argenteuil','itutaba','ontario','quentin','strasbourg','olten','plymouth','sontheim','bergedorf','waedenswil','guerin','ii 9,46:e,n,x:1,5,7','wernigerode','hillingdon','macclesfield','ii 9,46:g,m,s,t:e,n,x','gateshead','ii 9,46:g,z62:–','ii 9,46:m,t:e,n,x','sangalkam','mathura','potto','marylebone','cochin','clontarf','ceyco','india','geraldton','toronto','ackwepe','ii 9,46:l,w:e,n,x','nordrhein','deckstein','shoreditch','sokode','cork','benin','irchel','nantes','mayday','ii 9,46:z:1,5','ii 9,46:z:e,n,x','bambylor','ii 9,46:z:z39','ekotedo','ii 9,46:z4,z24:z39:z42','ngaparou','lishabi','inglis','mahina','louisiana','ii 9,46:z10:z6','ii 9,46:z10:z39','ouakam','hillegersberg','basingstoke','trimdon','fresno','ii 9,46:z39:1,7','wuppertal','ii 1,9,12,46,27:a:z6','ii 1,9,12,46,27:b:z39','ii 1,9,12,46,27:c:z39','ii 9,12,46,27:g,t:e,n,x','ii 1,9,12,46,27:l,z13,z28:z39','ii 1,9,12,46,27:y:z39','ii 1,9,12,46,27:z4,z24:1,5','ii 1,9,12,46,27:z10:1,5','ii 1,9,12,46,27:z10:e,n,x','ii 1,9,12,46,27:z10:z39','aminatu','goelzau','oxford','masembe','ii 3,10:a:e,n,x','galil','ii 3,10:a:l,v','ii 3,10:a:z39','ii 3,10:a:z87','kalina','butantan','allerton','huvudsta','benfica','ii 3,10:b:e,n,x','yaba','epicrates','wilmington','westminster','ii 3,10:b:z39','asylanta','gbadago','ikayi','pramiso','agege','anderlecht','okefoko','stormont','shangani','lekke','onireke','souza','ii 3,10:d:e,n,x','madjorio','birmingham','weybridge','maron','vejle','muenster','anatum','nyborg','newlands','lamberhurst','meleagridis','sekondi','ii 3,10:e,n,x:1,7','regent','alfort','suberu','amsterdam','ii 3,10,15:g,m,s,t:1,5','westhampton','bloomsbury','ii 3,10:g,t:–','ii 3,10:m,t:1,5','southbank','ii 3,10:m,t:e,n,x','cuckmere','amounderness','tibati','truro','bessi','falkensee','hoboken','yeerongpilly','wimborne','zanzibar','serrekunda','yundum','marienthal','newrochelle','nchanga','sinstorf','london','give','ii 3,10:l,v:e,n,x','ruzizi','ii 3,10:l,v:z6','sinchew','assinie','freiburg','uganda','fallowfield','hoghton','ii 3,10:l,z28:1,5','torbay','joal','lamin','ii 3,10:l,z28:e,n,x','ii 3,10:l,z28:z39','ughelli','elisabethville','simi','weltevreden','seegefeld','dumfries','amager','orion','mokola','ohlstedt','bolton','langensalza','stockholm','fufu','ii 3,10:z:1,5','harleystreet','huddinge','ii 3,10:z:e,n,x','kingslynn','clerkenwell','landwasser','ii 3,10:z:z39','adabraka','wagadugu','florian','ii 3,10:z4,z24:–','okerara','lexington','harrisonburg','coquilhatville','podiensis','kristianstad','biafra','everleigh','ii 3,10:z29:e,n,x','jedburgh','santaclara','ratchaburi','zongo','ii 3,10:z35:e,n,x,z15','shannon','cairina','macallen','sandaga','albertslund','bolombo','ii 3,10:z38:z42','ii 3,10:z39:1,5,7','dortmund','pietersburg','ii 3,15:z87:e,n,x,z15','capetown','niumi','juba','gwoza','alkmaar','gnesta','visby','tambacounda','kande','broughton','accra','eastglam','bida','madiago','umbadah','ahmadi','wanatah','liverpool','tilburg','niloese','vilvoorde','hayindogo','sanktmarx','sao','calabar','rideau','petahtikve','maiduguri','kouka','senftenberg','cannstatt','stratford','ouagadougou','chichester','machaga','avonmouth','zuilen','taksony','oesterbro','bethune','ngor','parkroyal','svedvi','fulda','westerstede','winterthur','lokstedt','stuivenberg','bedford','tomelilla','kindia','yalding','fareham','gatineau','thies','slade','kinson','krefeld','korlebu','kainji','lerum','schoeneberg','carno','hongkong','miromesnil','sambre','yenne','dallgow','llandoff','catumagos','ochiogu','fortbend','chittagong','bilu','ilugun','dessau','cannonhill','gallen','marseille','vi 11:a:1,5','massilia','toowong','luciana','ii 11:a:e,n,z15:d','epinay','ii 11:a:z6:z42','atento','leeuwarden','wohlen','vi 11:b:1,7','vi 11:b:e,n,x','pharr','erfurt','chiredzi','brindisi','ii 11:c:e,n,z15','woodinville','ati','gustavia','chandans','findorff','chingola','adamstua','redhill','abuja','missouri','ii 11:g,m,s,t:z39','iv 11:g,z51:–','moers','ii 11:m,t:e,n,x','aberdeen','brijbhumi','heerlen','veneziana','iiib 11:i:z','pretoria','abaetetuba','sharon','colobane','kisarawe','mannheim','amba','iiib 11:k:z53','stendal','maracaibo','fann','bullbay','iiib 11:l,v:z:z56','iiib 11:l,v:z53','glidji','tours','connecticut','osnabrueck','welwyn','ii 11:l,z28:e,n,x','senegal','rubislaw','clanvillian','euston','volta','solt','jalisco','herzliya','woumbou','crewe','maroua','ii 11:z:e,n,x','maybush','nyanza','ii 11:z:z39','remete','etterbeek','iiia 11:z4,z23:–','iv 11:z4,z23:–','yehuda','iv 11:z4,z32:–','wentworth','straengnaes','telhashomer','lene','maastricht','ii 11:z41:e,n,x','ii 11:–:1,5','chagoua','ii 1,13,23:a:1,5','mim','ii 13,22:a:e,n,x','wyldegreen','marshall','ii 1,13,23:a:z42','ibadan','mississippi','oudwijk','ii 1,13,23:b:1,5:z42','bracknell','rottnest','vaertan','ullevi','bahati','durham','sanktjohann','ii 1,13,22:b:z42','haouaria','boston','handen','ii 13,22:d:1,5','mishmarhaemek','friedenau','wichita','grumpensis','ii 13,23:d:e,n,x','saintes','diguel','telelkebir','putten','isuge','tschangu','willemstad','vridi','ii 1,13,23:e,n,x:1,5,7','raus','havana','bron','iiib 13,22:g,m,s:z','agbeni','ii 1,13,22:g,m,t:1,5','ii 13,22:g,m,t:z42','ii 1,13,23:g,m,s,t:1,5','ii 1,13,23:g,m,s,t:e,n,x','ii 1,13,23:g,m,s,t:z42','newyork','ii 13,23:g,s,t:e,n,x','okatie','ii 1,13,22:g,t:1,5','ii 13,22:g,t:z6','ii 1,13,23:g,t:1,5','ii 13,23:g,t:e,n,x','ii 1,13,23:g,s,t:z42','iiia 1,13,23:g,z51:–','washington','ii 1,13,23:m,t:1,5','ii 1,13,23:m,t:e,n,x','ii 13,22:m,t:z42:z39','ii 1,13,23:m,t:z42','kintambo','v 1,13,22:i:–','idikan','myrria','jukestown','kedougou','chapuri','ii 13,22:k:1,5:z42','noho','marburg','galway','ii 13,23:k:z41','lovelace','iiib 13,22:l,v:1,5,7','borbeck','nanga','ii 13,23:l,w:e,n,x','taiping','ii 13,22:l,z28:1,5','ii 13,23:l,z28:1,5','ii 13,23:l,z28:z6','ii 1,13,23:l,z28:z42','v 13,22:r:–','adjame','linton','tanger','yarrabah','ordonez','tunis','winslow','ii 1,13,23:z:1,5','serenli','iiib 13,23:z:1,5','poona','farmsen','bristol','durance','ivrysurseine','tanzania','worthington','ii 1,13,23:z:z42','ii 13,22:z:–','ried','iiia 13,22:z4,z23:–','ajiobo','iiia 13,23:z4,z23,z32:–','romanby','iiia 13,23:z4,z24:–','roodepoort','ii 1,13,22:z10:z6','sapele','demerara','ii 13,22:z29:1,5','ii 13,22:z29:e,n,x','ii 1,13,23:z29:1,5','ii 1,13,23:z29:e,n,x','agoueve','cubana','mampong','nimes','picpus','anna','leiden','fanti','v 13,22:z39:–','ii 13,22:z39:1,7','ii 1,13,23:z39:1,5,7','pajala','ii 1,13,23:z42:1,5,7','ii 13,23:–:1,6','garba','vi 1,6,14:a:1,5','vi 1,6,14,25:a:e,n,x','banjul','ndjamena','kuntair','tucson','iiib 6,14:b:e,n,x','blijdorp','kassberg','runby','minna','martonos','finkenwerder','woodhull','midway','florida','lindern','charity','teko','encino','albuquerque','bahrenfeld','onderstepoort','magumeri','beaudesert','v 6,14:e,n,z15:–','warragul','caracas','sylvania','catanzaro','ii 1,6,14:m,t:1,5','ii 6,14:m,t:e,n,x','kaitaan','mampeza','buzu','schalkwijk','moussoro','harburg','ii 6,14,24:k:1,6','ii 6,14:k:e,n,x','trumersee','iiib 6,14:k:z','ii 1,6,14:k:z6:z42','iiib 6,14:k:z53','amberg','boecker','horsham','alpenquai','iiib 6,14:l,v:z','iiib 6,14:l,v:z35','iiib 6,14:l,v:z53','vi 6,14:l,v:z88','owerri','aflao','istoria','iiib 6,14:r:z','surat','carrau','madelia','fischerkietz','mornington','homosassa','kanifing','soahanina','sundsvall','royan','stafford','poano','arapahoe','bousso','iv 6,14:z4,z23:–','chichiri','uzaramo','nessa','vi 1,6,14,25:z10:1,2,7','ii 1,6,14:z10:1,5','laredo','iiib 6,14:z10:e,n,x,z15','iiib 6,14:z10:z:z56,z90','ulm','ii 1,6,14:z10:z6:z42','iiib 6,14:z10:z53','potosi','ii 6,14:z36:–','sara','ii 1,6,14:z42:1,6','iiib 6,14:z52:e,n,x,z15','iiib 1,6,14,25:z52:z35','hannover','brazil','amunigun','nyeko','togba','fischerhuette','heron','hull','melaka','wa','glasgow','hvittingfoss','ii 16:b:e,n,x','sangera','vegesack','malstatt','ii 16:b:z39','ii 16:b:z42','vancouver','gafsa','shamba','hithergreen','yoruba','oldenburg','sculcoates','ii 16:d:1,5','sherbrooke','gaminara','barranquilla','ii 16:d:e,n,x','nottingham','caen','barmbek','malakal','saboya','astridplein','rhydyfelin','moabit','weston','ii 16:e,n,x:1,5,7','ii 16:e,n,x:1,6:z42','tees','adeoyo','nikolaifleet','ii 16:g,m,s,t:1,5:z42','ii 16:g,m,s,t:e,n,x','cardoner','ii 16:m,t:e,n,x','morbihan','ii 16:m,t:z42','mpouto','amina','agbara','wisbech','frankfurt','pisa','abobo','iiib 16:i:z35','szentes','maumee','nuatja','orientalis','iiib 16:k:e,n,x,z15','iiib 16:k:z','iiib 16:k:z35','iiib 16:k:z53','iiib 16:l,v:1,5,7','shanghai','welikade','salford','burgas','iiib 16:l,v:z:z61','losangeles','iiib 16:l,v:z35','iiib 16:l,v:z53','zigong','westeinde','brooklyn','lomnava','essingen','ii 16:l,w:z6','mandera','enugu','battle','ablogame','koblenz','ii 16:l,z28:z42','rovaniemi','ivory','brunflo','lehrte','annedal','zwickau','saphra','akuafo','kikoma','avignon','gerland','fortlamy','lingwala','kassel','ii 16:z:e,n,x','brevik','bouake','ii 16:z:z42','kibi','axim','ii 16:z4,z23:–','iv 16:z4,z23:–','kaevlinge','ii 16:z4,z24:–','iv 16:z4,z24:–','iv 16:z4,z32:–','ii 16:z6:1,6','badagry','iiib 16:z10:1,7','lisboa','iiib 16:z10:e,n,x,z15','redlands','angouleme','saloniki','ii 16:z29:1,5','ii 16:z29:e,n,x','neuland','trier','dakota','ii 16:z35:e,n,x','iv 16:z36:–','ii 16:z36:e,n,z15','naware','grancanaria','ii 16:z42:1,5,7','ii 16:z42:1,6','iiib 16:z52:z35','bonames','jangwani','kinondoni','kirkee','dahra','mattenhof','ii 17:b:e,n,x,z15','bignona','ii 17:b:z6','luedinghausen','victoriaborg','ii 17:c:z39','ii 17:d:–','berlin','karlshamn','niamey','ii 17:d:z6','jubilee','ii 17:e,n,x,z15:1,6','ii 17:e,n,x,z15:1,5,7','wembleypark','ii 17:g,m,s,t:–','lowestoft','ii 17:g,t:e,n,x,z15','ii 17:g,t:z39','iiia 17:g,z51:–','bama','ii 17:m,t:–','ahanou','iiib 17:i:z35','irenea','bandim','warri','matadi','zaria','iiib 17:k:z','ii 17:k:–','morotai','michigan','lancaster','carmel','iiib 17:l,v:e,n,x,z15','iiib 17:l,v:z35','granlo','lode','iiib 17:r:z','ii 17:y:–','tendeba','hadejia','lokomo','gori','warengo','dingiri','ii 17:z:1,7','tchamba','ii 17:z:l,w:z42','iiia 17:z4,z23:–','iiia 17:z4,z23,z32:–','iiia 17:z4,z24:–','iiia 17:z4,z32:–','djibouti','iiib 17:z10:e,n,x,z15:z56','iiib 17:z10:z','ii 17:z10:–','kandla','iiia 17:z29:–','iv 17:z29:–','aachen','iiia 17:z36:–','iv 17:z36:–','brazos','fluntern','cochise','rawash','groenekan','usumbura','pontypridd','eaubonne','iiia 18:g,z51:–','iv 18:g,z51:–','ii 18:m,t:1,5','langenhorn','memphis','iiib 18:k:z53','iiib 18:k:z54','iiib 18:l,v:e,n,x,z15','orlando','iiib 18:l,v:z:z50','iiib 18:l,v:z53','toulon','tennenlohe','iiib 18:r:z','troy','ii 18:y:e,n,x,z15','potengi','cerro','aarhus','ii 18:z4,z23:–','iiia 18:z4,z23:–','blukwa','ii 18:z4,z24:–','iiia 18:z4,z32:–','iiib 18:z10:e,n,x,z15','leer','carnac','ii 18:z10:z6','ii 18:z36:–','iiia 18:z36:–','iv 18:z36,z38:–','sinthia','delmenhorst','cotia','assen','ii 21:b:1,5','ghana','minnesota','hydra','rhone','ii 21:c:e,n,x','iiib 21:c:e,n,x,z15','spartel','magwa','madison','good','ii 21:g,m,s,t:–','iiia 21:g,z51:–','iv 21:g,z51:–','ii 21:m,t:–','iv 21:m,t:–','diourbel','iiib 21:i:1,5,7','iiib 21:i:e,n,x,z15','salpetriere','iiib 21:k:e,n,x,z15','iiib 21:k:z','surrey','iiib 21:l,v:z','iiib 21:l,v:z57','keve','jambur','mountmagnet','iiib 21:r:z','ibaragi','ruiru','ii 21:z:–','baguida','iiia 21:z4,z23:–','iv 21:z4,z23:–','ii 21:z4,z24:–','iiia 21:z4,z24:–','iv 21:z4,z32:–','iiib 21:z10:e,n,x,z15','iiib 21:z10:z','ii 21:z10:z6','iiib 21:z10:z53','iiia 21:z29:–','gambaga','iiia 21:z36:–','iv 21:z36:–','iv 21:z38:–','iiib 21:z65:e,n,x,z15','solna','dakar','bakau','seattle','ii 28:a:e,n,x','honelis','dibra','moero','ashanti','bokanjac','soumbedioune','ii 28:b:e,n,x','langford','freefalls','ii 28:b:z6','hermannswerder','eberswalde','halle','dresden','wedding','parabiago','techimani','amoutive','hatfield','mundonobo','mocamedes','patience','cullingworth','etobicoke','korkeasaari','kpeme','gozo','ii 28:e,n,x:1,7','ii 28:e,n,z15:z87','friedrichsfelde','yardley','abadina','ii 28:g,m,s,t:1,5','croft','ii 28:g,m,t:e,n,x','ii 28:g,m,t:z39','ii 28:g,s,t:e,n,x','ona','ii 28:m,t:e,n,x','vinohrady','ii 28:m,t:z39','morillons','doorn','cotham','volkmarsdorf','dieuppeul','warnemuende','kuessel','douala','guildford','ilala','adamstown','ikeja','iiib 28:k:1,7','taunton','ank','leoben','vitkin','nashua','ramsey','catalunia','penilla','ii 28:l,z28:1,5','fajara','ii 28:l,z28:e,n,x','bassadji','kibusi','ii 28:r:e,n,z15','fairfield','chicago','banco','sanktgeorg','oskarshamn','nima','pomona','kitenge','telaviv','shomolu','selby','vanier','ii 28:z:1,5','doel','ezra','brisbane','ii 28:z:z39','cannobio','teltow','babelsberg','kethiabarny','rogy','farakan','libreville','malaysia','umbilo','luckenwalde','moroto','iiib 28:z10:z','djermaia','ii 28:z29:1,5','ii 28:z29:e,n,x','konolfingen','babili','santander','aderike','overvecht','zehlendorf','guarapiranga','doulassame','ii 30:a:z39','louga','aschersleben','tempe','urbana','neudorf','coppettswood','ii 30:b:z6','kakikoka','zaire','morningside','ii 30:c:z39','messina','livulu','torhout','godesberg','ii 30:g,m,s:e,n,x','giessen','iv 30:g,m,t,z51:–','sternschanze','ii 30:g,t:–','wayne','ii 30:m,t:–','landau','morehead','mjordan','soerenga','hilversum','ramatgan','aqua','angoda','odozi','ii 30:k:e,n,x,z15','scarborough','ligeo','donna','ockenheim','morocco','ii 30:l,z28:z6','grandhaven','gege','quincy','matopeni','bietri','steinplatz','baguirmi','nijmegen','hohentwiel','stoneferry','bodjonegoro','ii 30:z6:1,6','sada','senneville','kumasi','ii 30:z10:e,n,x,z15','aragua','kokoli','wuiti','ago','ii 30:z39:1,7','umhlatazana','tchad','keurmassar','gouloumbo','yolo','ii 35:d:1,5','dembe','gassi','adelaide','ealing','ii 35:g,m,s,t:–','ebrie','anecho','ii 35:g,t:1,5','ii 35:g,t:z42','agodi','iiia 35:g,z51:–','ii 35:g,z62:e,n,x','monschaui','ii 35:m,t:–','iiib 35:i:e,n,x,z15','gambia','bandia','iiib 35:i:z','evry','iiib 35:i:z35','iiib 35:i:z53','penzing','iiib 35:k:e,n,x,z15','iiib 35:k:z','iiib 35:k:z35','iiib 35:k:z53:z50','iiib 35:l,v:1,5,7','iiib 35:l,v:e,n,x,z15:z50','iiib 35:l,v:z35:z67','ii 35:l,z28:–','iiib 35:r:e,n,x,z15','massakory','iiib 35:r:z','iiib 35:r:z35','iiib 35:r:z61','baltimore','alachua','iiia 35:z4,z23:–','westphalia','iiia 35:z4,z24:–','iiia 35:z4,z32:–','camberene','enschede','ligna','iiib 35:z10:z35','ii 35:z29:e,n,x','widemarsh','iiia 35:z29:–','iiia 35:z36:–','haga','iiib 35:z52:1,5,7','iiib 35:z52:e,n,x,z15','iiib 35:z52:z','iiib 35:z52:z35','oran','ii 38:b:1,2','rittersbach','sheffield','kidderminster','willamette','ii 38:d:1,5','ii 38:d:z39','thiaroye','kasenyi','korovi','ii 38:g,t:–','iiia 38:g,z51:–','iv 38:g,z51:–','rothenburgsort','mgulani','lansing','iiib 38:i:z','iiib 38:i:z53','echa','mango','inverness','njala','iiib 38:k:e,n,x,z15','iiib 38:k:z','iiib 38:k:z53','iiib 38:k:1,5,7','iiib 38:k:z35:z56','iiib 38:k:–','iiib 38:k:z55','alger','kimberley','taylor','roan','iiib 38:l,v:z','iiib 38:l,v:z35','iiib 38:l,v:z53:z54','lindi','iiib 38:r:1,5,7','emmastad','iiib 38:r:e,n,x,z15','iiib 38:r:z:z57','iiib 38:r:z35','freetown','colombo','perth','stachus','yoff','iiia 38:z4,z23:–','iv 38:z4,z23:–','bangkok','neunkirchen','iiib 38:z10:z','iiib 38:z10:z53','carpentras','klouto','iiib 38:z52:z35','iiib 38:z52:z53','iiib 38:z53:–:z47,z50,z76','iiib 38:z61:z53','ii 39:a:z39','wandsworth','abidjan','ii 39:c:e,n,x','logone','bruebach','mara','ii 39:e,n,x:1,7','dietrichsdorf','ii 39:g,m,t:e,n,x','hofit','cumberland','alma','champaign','newjersey','ii 39:l,v:1,5','kokomlemle','oerlikon','ii 39:l,z28:e,n,x','ii 39:l,z28:z39','frimley','anfo','windermere','delan','namur','hegau','ii 39:z10:z6','landeskrone','ii 39:–:1,7','shikmonah','salinas','greiz','ii 40:a:z39','riogrande','saugus','johannesburg','duval','benguella','ii 40:b:–','ii 1,40:c:e,n,x,z15','ii 40:c:z6','ii 1,40:c:z39','driffield','ii 40:d:–','tilene','ii 1,40:e,n,x:1,5,7','ii 1,40:e,n,x,z15:1,6','bijlmer','athens','ii 1,40:g,m,s,t:e,n,x','ii 1,40:g,m,s,t:1,5','ii 1,40:g,t:e,n,x,z15','ii 40:g,t:z39','iv 1,40:g,t:–','ii 1,40:g,m,s,t:z42','iiia 40:g,z51:–','iiib 40:g,z51:e,n,x,z15','iv 1,40:g,z51:–','ii 40:m,t:e,n,x','ii 40:m,t:z39','ii 1,40:m,t:z42','iv 40:m,t:–','iiib 40:i:1,5,7','goulfey','allandale','hann','ii 1,40:k:e,n,x,z15','iiib 40:k:z:z57','ii 40:k:z6','iiib 40:k:z53','millesi','canary','ii 40:l,v:e,n,x','iiib 40:l,v:z','iiib 1,40:l,v:z53','overchurch','ii 40:l,z28:e,n,x','tiko','bukavu','ii 1,40:l,z28:1,5:z42','santhiaba','ii 1,40:l,z28:z39','iiib 40:r:z53','odienne','ii 1,40:z:1,5','casamance','nowawes','ii 1,40:z:z6','ii 1,40:z:z39','ii 40:z:z42','iiia 40:z4,z23:–','iv 1,40:z4,z23:–','ii 40:z4,z24:z39','iiia 40:z4,z24:–','iv 1,40:z4,z24:–','iiia 40:z4,z32:–','iv 1,40:z4,z32:–','ii 1,40:z6:1,5','trotha','ii 40:z10:e,n,x','iiib 40:z10:z35','omifisan','iiia 40:z29:–','ii 1,40:z35:e,n,x,z15','yekepa','v 1,40:z35:–','iiia 40:z36:–','ii 1,40:z39:1,5:z42','ii 1,40:z39:1,6','iiib 40:z39:1,6','ii 40:z39:1,7','karamoja','ii 1,40:z42:1,6','ii 1,40:z42:1,5,7','ii 1,40:z81:z6','v 1,40:z81:–','burundi','ii 41:b:1,5','vaugirard','vi 41:b:1,7','vietnam','sica','lonestar','iiib 41:c:e,n,x,z15','ii 41:c:z6','egusi','ii 41:d:z6','ii 41:g,m,s,t:z6','ii 41:g,t:–','iiia 41:g,z51:–','leatherhead','samaru','verona','ferlo','ii 41:k:1,6','ii 41:k:z6','iiib 41:k:z35','ii 41:l,z13,z28:e,n,x,z15','lubumbashi','konongo','bruxelles','ii 41:z:1,5','sally','bofflens','waycross','iiia 41:z4,z23:–','iv 41:z4,z23:–','iiia 41:z4,z23,z32:–','ipswich','iiia 41:z4,z24:–','iiia 41:z4,z32:–','ii 41:z10:1,2','leipzig','landala','inpraw','ii 41:z10:e,n,x,z15','ii 41:z10:z6','lodz','iiia 41:z29:–','iv 41:z29:–','ahoutoue','iiia 41:z36:–','iv 41:z36:–','offa','iv 41:z52:–','ii 41:–:1,6','faji','ii 42:b:1,5','orbe','ii 42:b:e,n,x,z15','tomegbe','frederiksberg','egusitoo','ii 42:b:z6','antwerpen','kampala','gostenhof','ii 42:d:z6','ii 42:e,n,x:1,6','ii 42:g,t:–','maricopa','iiia 42:g,z51:–','iv 1,42:g,z51:–','ii 42:m,t:e,n,x,z15','waral','kaneshie','borromea','middlesbrough','haferbreite','iiib 42:k:e,n,x,z15','iiib 42:k:z','gwale','iiib 42:k:z35','iiib 42:l,v:1,5,7:z76','portedeslilas','ii 42:l,v:e,n,x,z15','iiib 42:l,v:e,n,x,z15','coogee','iiib 42:l,v:z','iiib 42:l,v:z53','ii 1,42:l,w:e,n,x','parakou','ii 1,42:l,z13,z28:z6','sipane','brive','iiib 42:r:z','iiib 42:r:z53','ii 42:r:–','iiib 42:r:–:z50','spalentor','harvestehude','ii 42:z:1,5','ursenbach','ii 42:z:e,n,x,z15','melbourne','ii 42:z:z6','gera','broc','iiia 42:z4,z23:–','toricada','iiia 42:z4,z24:–','iv 1,42:z4,z24:–','ii 42:z6:1,6','ii 42:z10:1,2','ii 42:z10:e,n,x,z15','iiib 42:z10:e,n,x,z15:z60','iiib 42:z10:z','loenga','ii 42:z10:z6','iiib 42:z10:z35','iiib 42:z10:z67','djama','ii 42:z29:–','kahla','hennekamp','tema','weslaco','iv 42:z36:–','vogan','taset','iiib 42:z52:z','iiib 42:z57:1,5','graz','berkeley','ii 43:a:1,5','ii 43:a:z6','niederoderwitz','ede','ii 43:b:z42','montreal','orleans','ii 43:d:e,n,x,z15','ii 43:d:z39','ii 43:d:z42','ii 43:e,n,x,z15:1,5,7','ii 43:e,n,x,z15:1,6','milwaukee','ii 43:g,m,s,t:z42','ii 43:g,t:1,5','iiib 43:g,t:–','iiia 43:g,z51:–','iv 43:g,z51:–','ii 43:g,z62:e,n,x','mbao','voulte','thetford','ahuza','iiib 43:k:z','iiib 43:l,v:z53:z56','epalinges','sudan','ii 43:l,z13,z28:1,5','ii 43:m,t:–','iiib 43:r:e,n,x,z15','iiib 43:r:z','iiib 43:r:z53','farcha','kingabwa','ogbete','ii 43:z:1,5','arusha','farmingdale','ii 43:z4,z23:–','iiia 43:z4,z23:–','iv 43:z4,z23:–','iiia 43:z4,z24:–','iv 43:z4,z24:–','iv 43:z4,z32:–','adana','ii 43:z29:e,n,x','ii 43:z29:z42','makiling','iiia 43:z29:–','iv 43:z29:–','ahepe','iiia 43:z36:–','iv 43:z36,z38:–','irigny','darfur','ii 43:z42:1,5,7','iiib 43:z52:z53','iv 44:a:–','niakhar','tiergarten','niarembe','shahalam','elbeuf','sedgwick','madigan','bergerac','quebec','bobo','kermel','fischerstrasse','palamaner','ii 1,44:e,n,x:1,6','vleuten','gamaba','splott','ii 44:g,t:z42','iiib 1,44:g,t:1,5:z42','carswell','iv 44:g,z51:–','muguga','ii 1,44:m,t:z42','maritzburg','lawra','malika','albertbanjul','brefet','v 44:r:–','brackenridge','uhlenhorst','bolama','goodmayes','kua','ploufragan','ii 44:z4,z23:–','iiia 44:z4,z23:–','iv 44:z4,z23:–','iiia 44:z4,z23,z32:–','christiansborg','iiia 44:z4,z24:–','iv 44:z4,z24:–','iiia 44:z4,z32:–','iv 1,44:z4,z32:–','guinea','llobregat','ii 44:z29:e,n,x:z42','zinder','iv 44:z29:–','iv 44:z36,z38:–','koketime','ii 1,44:z39:e,n,x,z15','v 44:z39:–','vi 45:a:e,n,x','meekatharra','ii 45:a:z10','riverside','leinster','fomeco','deversoir','dugbe','karachi','warmsen','suelldorf','tornow','ii 45:g,m,s,t:1,5','ii 45:g,m,s,t:e,n,x','ii 45:g,m,t:e,n,x,z15','binningen','iiia 45:g,z51:–','iv 45:g,z51:–','ii 45:m,t:1,5','apapa','verviers','casablanca','cairns','imo','kofandoka','ii 45:z:1,5','yopougon','ii 45:z:z39','iiia 45:z4,z23:–','iv 45:z4,z23:–','transvaal','iiia 45:z4,z24:–','iiia 45:z4,z32:–','aprad','jodhpur','ii 45:z29:1,5','ii 45:z29:e,n,x','ii 45:z29:z42','iiia 45:z29:–','lattenkamp','balcones','iiia 45:z36:–','iv 45:z36,z38:–','ii 47:a:1,5','ii 47:a:e,n,x,z15','wenatchee','ii 47:b:1,5','ii 47:b:e,n,x,z15','sya','ii 47:b:z6','iiib 47:c:1,5,7','kodjovi','iiib 47:c:e,n,x,z15:z57','iiib 47:c:z','braeswood','iiib 47:c:z35','ii 47:d:1,5','stellingen','ii 47:d:e,n,x,z15','ii 47:d:z39','ii 47:e,n,x,z15:1,6','sljeme','luke','carshalton','ii 47:g,t:e,n,x','iiia 47:g,z51:–','mesbit','iiib 47:i:e,n,x,z15:z50','bergen','iiib 47:i:z','iiib 47:i:z35','iiib 47:i:z53:z57,z84','staoueli','bootle','iiib 47:k:1,5,7','dahomey','iiib 47:k:e,n,x,z15','lyon','iiib 47:k:z','iiib 47:k:z35','iiib 47:k:z53:z84','iiib 47:l,v:1,5,7:z50','drac','iiib 47:l,v:e,n,x,z15','iiib 47:l,v:z','iiib 47:l,v:z35','iiib 47:l,v:z53','iiib 47:l,v:z57','iv 47:l,v:–','teshie','iiib 47:r:e,n,x,z15','dapango','iiib 47:r:1,5,7','iiib 47:r:z','iiib 47:r,i:z35','iiib 47:r:z53:z70,z74,z77,z90','moualine','blitta','mountpleasant','kaolack','ii 47:z:e,n,x,z15','ii 47:z:z6','forges','tabligbo','fehrbellin','bere','binche','iiia 47:z4,z23:–','tamberma','ii 47:z6:1,6','iiib 47:z10:1,5,7','namoda','iiib 47:z10:z','iiib 47:z10:z35','ii 47:z29:e,n,x,z15','ekpoui','iiia 47:z29:–','yombesali','bingerville','iv 47:z36:–','alexanderplatz','quinhon','iiib 47:z52:1,5:z54','iiib 47:z52:1,5,7','iiib 47:z52:e,n,x,z15','iiib 47:z52:z','iiib 47:z52:z35','iiib 47:z53:–:z90','hisingen','ii 48:a:z6','ii 48:a:z39','ii 48:b:z6','ii 48:b:e,n,x,z15','v 48:b:–','iiib 48:c:z','ii 48:d:1,2','ii 48:d:z6','buckeye','fitzroy','ii 48:e,n,x,z15:z6','ii 48:g,m,t:–','iiia 48:g,z51:–','iv 48:g,z51:–','iiib 48:i:z:z72','iiib 48:i:z35:z57','iiib 48:i:z53','iiib 48:i:z61','v 48:i:–','iiib 48:k:1,5,7','ii 48:k:e,n,x,z15','iiib 48:k:e,n,x,z15','dahlem','iiib 48:k:z','iiib 48:k:z35:z75','ii 48:k:z39','iiib 48:k:z53','iiib 48:k:z57','australia','iiib 48:l,v:1,5,7:z47,z50,z89','iiib 48:l,v:z','iiib 48:l,w:1,5,7:z50','iiib 48:r:e,n,x,z15','iiib 48:r:z','toucra','ii 48:z:1,5','iiib 48:z:1,5,7','iiia 48:z4,z23:–','iv 48:z4,z23:–','iiia 48:z4,z23,z32:–','djakarta','iiia 48:z4,z24:–','iiib 48:z4,z24:–','iv 48:z4,z24:–','iiia 48:z4,z32:–','iv 48:z4,z32:–','ii 48:z10:1,5','vi 48:z10:1,5','ii 48:z10:1,6','isaszeg','iiib 48:z10:e,n,x,z15','iiib 48:z10:z','ii 48:z29:–','iiia 48:z29:–','iv 48:z29:–','iiib 48:z35:z52','v 48:z35:–','iiia 48:z36:–','iv 48:z36,z38:–','ii 48:z39:z81','v 48:z39:–','v 48:z41:–','iiib 48:z52:e,n,x,z15','iiib 48:z52:z','v 48:z65:–','v 48:z81:–','ii 48:z81:1,5,7','iv 50:a:–','rochdale','ii 50:b:z6','iv 50:b:–','hemingford','iv 50:d:–','ii 50:e,n,x:1,7','ii 50:g,m,s,t:1,5','iiia 50:g,z51:–','iv 50:g,z51:–','ii 50:g,z62:e,n,x','ii 50:m,t:z6:z42','iiib 50:i:1,5,7','iiib 50:i:e,n,x,z15','iiib 50:i:z','iiib 50:i:z53','iiib 50:k:1,5,7','ii 50:k:e,n,x:z42','iiib 50:k:e,n,x,z15','iiib 50:k:z:z50,z57,z68,z86','ii 50:k:z6','iiib 50:k:z35','iiib 50:k:z53','fass','iiib 50:l,v:e,n,x,z15','iiib 50:l,v:z','iiib 50:l,v:z35','iiib 50:l,v:z57','vi 50:l,v:z67','ii 50:l,w:e,n,x,z15:z42','ii 50:l,z28:z42','iiib 50:r:1,5,7','iiib 50:r:e,n,x,z15','iiib 50:r:z:z67','iiib 50:r:z35:z58','iiib 50:r:z53','dougi','ii 50:z:e,n,x','iiib 50:z:z52','iiia 50:z4,z23:–','iv 50:z4,z23:–','iiia 50:z4,z23,z32:–','iiia 50:z4,z24:–','iv 50:z4,z24:–','iiia 50:z4,z32:–','iv 50:z4,z32:–','iiib 50:z10:z:z56','ii 50:z10:z6:z42','iiib 50:z10:z53','ivorycoast','iiia 50:z29:–','iiia 50:z36:–','ii 50:z42:1,7','iiib 50:z52:1,5,7','iiib 50:z52:z35','iiib 50:z52:z53','iv 51:a:–','windsheim','tione','karaya','iv 51:b:–','ii 51:c:–','gokul','meskin','ii 51:g,s,t:e,n,x','iiia 51:g,z51:–','djinten','elmdon','kabete','dan','iiib 51:k:z35','harcourt','overschie','dadzie','iiib 51:l,v:z','moundou','ii 51:l,z28:z6','ii 51:l,z28:z39','mali','lutetia','rosslyn','antsalova','treforest','lechler','iiia 51:z4,z23:–','iv 51:z4,z23:–','iiia 51:z4,z24:–','iiia 51:z4,z32:–','bergues','ii 51:z29:e,n,x,z15','iiia 51:z36:–','ii 51:–:1,7','uithof','ord','molesey','flottbek','ii 52:c:k','utrecht','ii 52:d:e,n,x,z15','ii 52:d:z39','butare','derkle','saintemarie','ii 52:g,t:–','bordeaux','iiib 52:k:e,n,x,z15','iiib 52:k:z35','iiib 52:k:z53','iiib 52:l,v:z53','marsabit','ii 52:z:z39','iiib 52:z:z52','ii 52:z39:1,5,7','ii 52:z44:1,5,7','ii 53:c:1,5:r1','ii 53:d:1,5','ii 1,53:d:z39','ii 53:d:z42','iiia 53:g,z51:–','iv 1,53:g,z51:–','iiib 53:i:z','iiib 53:k:e,n,x,z15','iiib 53:k:z','iiib 53:k:z35','iiib 53:k:z53','iiib 53:l,v:e,n,x,z15','iiib 53:l,v:z','iiib 53:l,v:z35','ii 53:l,z28:e,n,x','ii 53:l,z28:z6','ii 53:l,z28:z39','iiib 53:r:z','iiib 53:r:z35','iiib 53:r:z68','ii 53:z:1,5','iiib 53:z:1,5,7','ii 53:z:z6','iiia 53:z4,z23:–','iv 53:z4,z23:–','iiia 53:z4,z23,z32:–','ii 53:z4,z24:–','iiia 53:z4,z24:–','iiib 53:z10:z','iiib 53:z10:z35','iiia 53:z29:–','iv 1,53:z36,z38:–','iiib 53:z52:z35','iiib 53:z52:z53','leda','tonev','winnipeg','rossleben','borreze','uccle','newholland','poeseldorf','ochsenwerder','czernyring','steinwerder','yerba','canton','barry','mundubbera','ii 55:k:z39','ii 56:b:1,5','ii 56:d:–','ii 56:e,n,x:1,7','ii 56:l,v:z39','ii 56:l,z28:–','ii 56:z:z6','iiia 56:z4,z23:–','iiia 56:z4,z23,z32:–','ii 56:z10:e,n,x','iiia 56:z29:–','antonio','ii 57:a:z42','maryland','batonrouge','iiib 57:c:e,n,x,z15','iiib 57:c:z:z70,z90','ii 57:d:1,5','ii 57:g,m,s,t:z42','ii 57:g,t:–','iiib 57:i:e,n,x,z15','iiib 57:i:z','iiib 57:k:e,n,x,z15','iv 57:z4,z23:–','iiib 57:z10:z','ii 57:z29:z42','ii 57:z39:e,n,x,z15','ii 57:z42:1,6:z53','ii 58:a:z6','ii 58:b:1,5','ii 58:c:z6','ii 58:d:z6','iiib 58:i:e,n,x,z15','iiib 58:i:z53','iiib 58:k:z','iiib 58:l,v:e,n,x,z15','iiib 58:l,v:z35','ii 58:l,z13,z28:1,5','ii 58:l,z13,z28:z6','iiib 58:r:e,n,x,z15','iiib 58:r:z','iiib 58:r:z53:z47,z57,z70,z73','ii 58:z6:1,6','ii 58:z10:1,6','iiib 58:z10:e,n,x,z15','ii 58:z10:z6','iiib 58:z10:z53:z50','ii 58:z39:e,n,x,z15','iiib 58:z52:z','iiib 58:z52:z35','iiib 59:c:e,n,x,z15','iiib 59:i:e,n,x,z15','iiib 59:i:z','iiib 59:i:z35:z84','iiib 59:k:e,n,x,z15','ii 59:k:z65','iiib 59:k:z','iiib 59:k:z35','iiib 59:k:z53','iiib 59:l,v:z','iiib 59:l,v:z53','iiib 59:r:z35','ii 1,59:z:z6','iiia 59:z4,z23:–','iiib 59:z10:z53','iiib 59:z10:z57','iiia 59:z29:–','iiia 59:z36:–','vi 59:z36:–','iiib 59:z52:z53','ii 60:b:–:r1','ii 60:g,m,t:z6','iiib 60:i:e,n,x,z15:z50','iiib 60:i:z:z50','iiib 60:i:z35:z50','iiib 60:k:z','iiib 60:k:z35','iiib 60:k:z53','iiib 60:l,v:z','iiib 60:r:e,n,x,z15','iiib 60:r:z','iiib 60:r:z35','iiib 60:r:z53','ii 60:z:e,n,x','iiib 60:z10:z','iiib 60:z10:z35','ii 60:z10:z39','iiib 60:z10:z53','ii 60:z29:e,n,x','v 60:z41:–','iiib 60:z52:1,5,7','iiib 60:z52:z','iiib 60:z52:z35','iiib 60:z52:z53','iiib 61:c:1,5,7','iiib 61:c:z35','iiib 61:i:e,n,x,z15','iiib 61:i:z','iiib 61:i:z35','iiib 61:i:z53','iiib 61:k:1,5,7','iiib 61:k:z35','iiib 61:k:z53','iiib 61:l,v:1,5,7:z57','iiib 61:l,v:z','iiib 61:l,v:z35','iiib 61:r:1,5,7','iiib 61:r:z','iiib 61:r:z35','iiib 61:r:z53:z47,z50','iiib 61:z10:z35','v 61:z35:–','iiib 61:z52:1,5,7','iiib 61:z52:z','iiib 61:z52:z35','iiib 61:z52:z53','iiia 62:g,z51:–','iiia 62:z4,z23:–','iiia 62:z4,z32:–','iiia 62:z29:–','iiia 62:z36:–','iiia 63:g,z51:–','iiib 63:k:z','iiia 63:z4,z23:–','iiia 63:z4,z32:–','iiia 63:z36:–','iiib 65:c:1,5,7','iiib 65:c:z','iiib 65:c:z53','ii 65:g,t:–','iiib 65:i:e,n,x,z15','iiib 65:i:z','iiib 65:k:z','iiib 65:k:z35','iiib 65:k:z53','iiib 65:l,v:e,n,x,z15','iiib 65:l,v:z','iiib 65:l,v:z35','iiib 65:l,v:z53','iiib 65:r:z35','iiib 65:z10:e,n,x,z15','iiib 65:z10:z','iiib 65:z52:e,n,x,z15','iiib 65:z52:z','iiib 65:z52:z35','iiib 65:z52:z53','ii 65:–:1,6','v 66:z35:–','v 66:z39:–','v 66:z41:–','v 66:z65:–','v 66:z81:–','choleraesuis or typhisuis','miami or sendai','crossness','amager var. 15+', 'amersfoort var. 14+', 'amsterdam var. 15+', 'anatum var. 15+', 'anatum var. 15+,34+', 'butantan var. 15+,34+', 'cerro var. 14+', 'gdansk var. 14+', 'give var. 15+', 'give var. 15+,34+', 'goelzau var. 15+', 'lexington var. 15+', 'lexington var. 15+,34+', 'lille var. 14+', 'livingstone var. 14+', 'london var. 15+', 'meleagridis var. 15+', 'meleagridis var. 15+,34+', 'muenster var. 15+', 'muenster var. 15+,34+', 'nchanga var. 15+', 'nyborg var. 15+', 'oranienburg var. 14+', 'orion var. 15+', 'orion var. 15+,34+', 'oxford var. 15+,34+', 'paratyphi b var. l+ tartrate = d–tartrate+', 'rissen var. 14+', 'shangani var. 15+', 'souza var. 15+', 'stanleyville var. 27+', 'stockholm var. 15+', 'uganda var. 15+', 'vejle var. 15+', 'vejle var. 15+,rz27', 'weltevreden var. 15+', 'westhampton var. 15+', 'westhampton var. 15+,34+'
wklm_formula = ['I [1],2,12:a:[1,5]','I 2,12:g,m:–','I [1],2,12:g,p:–','I 2,12:l,v:1,5','I [1],4,[5],12:a:1,2','I 4,12,[27]:a:1,5','I 4,[5],12:a:[1,5]','I 4,[5],12:a:1,7','I [1],4,[5],12:a:e,n,x','I [1],4,12,27:a:e,n,z15','II [1],4,[5],12,[27]:a:e,n,x','I [1],4,12:a:l,w','I [1],4,12,27:a:z6','II [1],4,12,[27]:a:z39','I [1],4,[5],12:b:1,2:[z5],[z33]','I [1],4,[5],12,[27]:b:1,5','II 4,12:b:1,5','I 4,12,[27]:b:1,6','I [1],4,12,27:b:1,7','I [1],4,[5],12,[27]:b:e,n,x','II [1],4,[5],12,[27]:b:[e,n,x]','I [1],4,12,27:b:e,n,z15','I [1],4,12,[27]:b:l,w','I [1],4,12,27:b:z6','I 4,12,27:b:–','I [1],4,12,[27]:c:1,5','I 4,12:c:1,6','I 4,12,[27]:c:1,7','I 4,12:c:e,n,x','I [1],4,12,27:c:e,n,z15','I [1],4,12,27:c:l,w','I 4,12,27:c:z6','I [1],4,[5],12,[27]:d:1,2','I [1],4,12,[27]:d:1,5','I [1],4,12,27:d:1,6','I [1],4,12,27:d:1,7','II 4,12:d:e,n,x','I [1],4,[5],12,[27]:d:e,n,x','I [1],4,12,[27]:d:e,n,z15:[e,h]','I [1],4,12,27:d:l,w','I [1],4,12,27:d:z6','I 4,12:d:z35','I [1],4,[5],12:e,h:1,2','I [1],4,[5],12:e,h:1,5:[R1…]','I 4,12:e,h:1,6','I 4,12:e,h:1,7','I [1],4,[5],12:e,h:e,n,x','I [1],4,[5],12:e,h:e,n,z15','I [1],4,12:e,h:l,w','II 4,12:e,n,x:1,2,7','II [1],4,12,[27]:e,n,x:1,[5],7','I [1],4,[5],12:f,g:[1,2]','I [1],4,[5],12:f,g,s:[1,2]:[z27],[z45]','II [1],4,[5],12:f,g,t:z6:z42','I 4,12:g,m:–','I [1],4,[5],12:g,m,s:[1,2]','II [1],4,12,[27]:g,[m],[s],t:e,n,x','II [1],4,12,[27]:g,[m],t:[1,5]','II 4,12:g,m,t:z39','I 4,12:g,m,t:[z67]','I [1],4,[5],12,[27]:g,s,t:[1,2]:z43','I [1],4,12,[27]:g,t:–','I 4,[5],12:g,z51:1,7','I 4,[5],12:g,z51:e,n,z15','II 4,12:g,z62:–','I [1],4,[5],12:m,t:[1,5]','I 4,[5],12:m,t:e,n,z15','I [1],4,[5],12:i:1,2','I [1],4,[5],12:i:1,5','I 4,12:i:1,6','I 4,12:i:e,n,x','I [1],4,12:i:e,n,z15','I [1],4,12,27:i:l,w','I [1],4,12:i:z6','II 4,12,27:i:z35','I [1],4,12,27:k:1,5','I [1],4,12,27:k:1,6','II [1],4,12,27:k:1,6','I 4,12,27:k:e,n,x','I 4,[5],12:k:e,n,z15','I 4,[5],12:l,v:1,2','I [1],4,[5],12,[27]:l,v:1,5','I 4,12:l,v:1,6','I [1],4,12,27:l,v:1,7:[z40]','I [1],4,12,27:l,v:e,n,x','II [1],4,12,27:l,v:e,n,x','I 4,[5],12:l,v:e,n,z15','II [1],4,12,27:l,v:z39','I 4,12:l,w:1,5','I 4,12:l,w:1,6','II 4,12:l,w:e,n,x','I 4,12:l,w:e,n,z15','I [1],4,12,27:l,w:z6','I [1],4,[5],12,[27]:l,[z13],[z28]:1,2','I [1],4,12,[27]:l,[z13],z28:1,5','I 4,12:l,z13,[z28]:1,6','I [1],4,12,27:l,z13,z28:1,7','I [1],4,12,27:l,z13,z28:e,n,x','I [1],4,12,27:l,z13,z28:e,n,z15','I 4,12:l,z28:e,n,x','II [1],4,12:l,z28:[e,n,x]','I [1],4,[5],12:r:1,2','I 4,12,[27]:r:1,5','I 4,12:r:1,6','I [1],4,12,27:r:1,7','I [1],4,[5],12:r:l,w','I 4,12,27:r:z6','I [1],4,12,27:r,[i]:e,n,z15','I 4,12:r,i:l,w','I [1],4,[5],12:y:1,2','I 4,12,27:y:1,5','I 4,12:y:1,6','I [1],4,12,27:y:1,7','I [1],4,[5],12,[27]:y:e,n,x','I [1],4,12,27:y:e,n,z15','I [1],4,12,27:y:z6','I 4,[5],12:z:1,2','I [1],4,12:z:1,5','II [1],4,12,27:z:1,5','I 4,12:z:1,6','I [1],4,12:z:1,7','II 4,12:z:1,7','I 4,12:z:e,n,x','II [1],4,12,27:z:e,n,x','I [1],4,[5],12:z:e,n,z15','I [1],4,12:z:l,w','I [1],4,12,27:z:z6','II 4,12:z:z39','I [1],4,[5],12,[27]:z4,z23:[1,2]','I 4,12:z4,z23:e,n,z15','I 4,12,27:z4,z23:z6','I [1],4,[5],12:z4,z24:[1,5]','I [1],4,[5],12:z10:1,2','I [1],4,12:z10:1,5','I 4,12:z10:1,6','I 4,12:z10:1,7','I 4,12:z10:e,n,x','I 4,12:z10:e,n,z15','I [1],4,12:z10:l,w','I [1],4,12,[27]:z10:z6','I [1],4,12,27:z10:z35','I [1],4,12,27:z29:–','II [1],4,12:z29:e,n,x','I 4,12:z35:1,5','I [1],4,12,27:z35:1,7','I [1],4,12,27:z35:z6','I [1],4,12,27:z35:e,n,z15','I 4,12:z36:–','I [1],4,12:z38:1,5','I [1],4,[5],12,[27]:z38:[e,n,z15]','II [1],4,12,[27]:z39:1,[5],7','I [1],4,12,27:z41:1,(2),5','I [1],4,12,27:z41:e,n,z15','I 4,12:–:e,n,x','I 4,12:z91:–','I 6,7:a:1,5','II 6,7,[14]:a:1,5','I 6,7:a:1,6','I 6,7:a:1,7','I 6,7,[14]:a:e,n,x','I 6,7:a:e,n,z15','I 6,7,[14]:a:l,w','I 6,7,[14]:a:z6','II 6,7:a:z6','II 6,7:a:z42','I 6,7:b:1,2','II 6,7:b:1,5','I 6,7,[14]:b:1,5','I 6,7:b:1,6','I 6,7:b:1,7','I 6,7,[14]:b:e,n,x','I 6,7:b:e,n,z15','II 6,7:b:[e,n,x]:z42','I 6,7,[14]:b:l,w:[z59]','I 6,7,[14]:b:z6','I 6,7:b:z35','II 6,7:b:z39','I 6,7,[14]:c:1,2','I 6,7,[Vi]:c:1,5','I 6,7:c:1,5','I 6,7:c:1,5','I 6,7:c:1,6','I 6,7:c:1,7','I 6,7:c:z6','I 6,7:c:e,n,x','I 6,7,[14]:c:e,n,z15','I 6,7:d:1,2','I 6,7,[14]:d:1,5','I 6,7:d:1,6','I 6,7:d:1,[2],7','I 6,7,[14]:d:e,n,x','I 6,7,[14]:d:e,n,z15','I 6,7,[14]:d:l,w','I 6,7:d:l,z13,z28','I 6,7,[14]:d:z6','II 6,7:d:z42','I 6,7:e,h:1,2','I 6,7:e,h:1,5','I 6,7:e,h:1,6','I 6,7:e,h:1,7','I 6,7,[14]:e,h:e,n,z15','II 6,7:e,n,x:1,6:z42','I 6,7:e,n,z15:1,6','I 6,7,[14]:f,g:–','I 6,7:f,g,t:1,2,7','I 6,7:f,g,t:e,n,x','I 6,7,[14],[54]:g,m,[p],s:[1,2,7]','I 6,7:g,m,s:e,n,z15','II 6,7:g,m,[s],t:e,n,x','II 6,7:(g),m,[s],t:1,5','II 6,7:g,m,s,t:z39','II 6,7:g,[m],s,t:[z42]','I 6,7,[14]:g,m,[t]:–','I 6,7:g,s,q:–','I 6,7:g,s,[t]:[1,6]','II 6,7:g,t:[e,n,x]:z42','I 6,7:g,(t):–','I 6,7:g,z51:1,5','I 6,7:g,z51:e,n,z15','IV 6,7:g,z51:–','I 6,7:m,p,t,[u]:–','I 6,7:m,t:1,6','I 6,7:m,t:z64','II 6,7:m,t:–','I 6,7,[14]:m,t:[z57]','I 6,7,[14]:i:1,2','I 6,7:i:1,5','I 6,7:i:1,6','I 6,7:i:1,7','I 6,7:i:e,n,z15','I 6,7:i:l,w','I 6,7,[14]:i:z6','I 6,7,[14]:k:1,2','I 6,7,[14]:k:1,5:[R1…]','I 6,7:k:1,6','I 6,7:k:1,7','I 6,7:k:e,n,x','I 6,7:k:e,n,z15','IIIb 6,7:(k):z:[z55]','II 6,7:k:[z6]','I 6,7:k:z35','I 6,7:l,v:1,2','I 6,7:l,v:1,5','IIIb 6,7:l,v:1,5,7','I 6,7:l,v:1,6','I 6,7:l,v:1,7','I 6,7:l,v:e,n,x','I 6,7,[14]:l,v:e,n,z15','I 6,7,[14]:l,v:z6','I 6,7:l,v:z35','IIIb 6,7:l,v:z53','I 6,7:l,w:1,2','I 6,7:l,w:1,5','II 6,7:l,w:1,5,7','I 6,7:l,w:e,n,z15','II 6,7:l,w:z42','I 6,7:l,z13:1,5','I 6,7:l,z13:e,n,x','I 6,7:l,z13,[z28]:e,n,z15','I 6,7:l,z13,z28:z6','I 6,7:l,z13,z28:1,7','II 6,7:l,z28:1,5:[z42]','II 6,7:l,z28:e,n,x','II 6,7:l,z28:z6','I 6,7,[14]:r:1,2','I 6,7,[14]:r:1,5:[R1…],[z37],[z45],[z49]','I 6,7:r:1,6','I 6,7:r:1,7','I 6,7:r:e,n,z15','I 6,7:r:l,w','I 6,7:y:1,2','I 6,7,[14]:y:1,5','I 6,7:y:1,6','I 6,7:y:1,7','I 6,7:y:e,n,x:[z67]','I 6,7,[14]:y:e,n,z15:[z47],[z50]','I 6,7:z:1,2','I 6,7:z:1,5','II 6,7:z:1,5','I 6,7:z:1,6,[7]','I 6,7:z:e,n,x','II 6,7:z:e,n,x','II 6,7:z:e,n,x,z15','I 6,7:z:e,n,z15','I 6,7:z:l,w','II 6,7:z:z6','II 6,7:z:z39','II 6,7:z:z42','I 6,7:z4,z23:1,5','I 6,7:z4,z23:1,6','I 6,7:z4,z23:e,n,z15','I 6,7:z4,z23:z6','II 6,7:z4,z23:–','IV 6,7:z4,z23:–','I 6,7:z4,z24:1,5','II 6,7:z4,z24:z42','I 6,7:z4,z24:–','IV 6,7:z4,z24:–','II 6,7:z6:1,7','I 6,7:z10:1,2','I 6,7:z10:1,5','I 6,7:z10:1,6','I 6,7:z10:1,7','I 6,7:z10:e,n,x','I 6,7,[14]:z10:e,n,z15:[z37],[z45]','I 6,7,[14]:z10:l,w','I 6,7:z10:z6','I 6,7:z10:z35','I 6,7,[14]:z29:[1,2,7]','II 6,7:z29:[z42]','I 6,7:z35:1,6','I 6,7:z35:e,n,z15','I 6,7:z36:e,n,z15','II 6,7:z36:z42','IV 6,7:z36:–','I 6,7:z38:1,2:[z82]','II 6,7:z38:e,n,x','I 6,7,[14]:z38:–:[z82]','IIIb 6,7,[14]:z39:1,2','II 6,7:z39:1,5,7','VI 6,7:z41:1,7','I 6,7:z41:l,w','I 6,7:z41:z35','II 6,7:z42:1,[5],7','I 6,7:z44:–','II 6,7:–:1,6','I 8,[20]:a:[z6]','I 6,8:a:1,2','I 6,8:a:1,5','I 6,8:a:1,6','I 6,8:a:1,7','I 6,8:a:e,n,x','II 6,8:a:e,n,x','I 6,8:a:e,n,z15','II 6,8:a:z39','II 6,8:a:z52','I 8:b:1,2','I 6,8:b:1,2','I 8,[20]:b:1,5','I 6,8:b:1,5','II 6,8:b:1,5','I 6,8:b:1,6','I 8:b:1,7','I 6,8:b:1,7','I 8:b:e,n,x','I 6,8:b:e,n,x','I 8,[20]:b:e,n,z15','I 6,8:b:e,n,z15','I 6,8:b:l,w','I 8,[20]:b:l,w','I 8,[20]:b:z6','I 6,8:b:z6','I 6,8:c:1,2','I 8,[20]:c:1,5','I 6,8:c:1,5','I 6,8:c:1,6','I 6,8:c:1,7','I 8:c:l,w','I 8,[20]:c:e,n,x','I 6,8:c:e,n,x','I 6,8:c:e,n,z15','I 8,[20]:c:z6','I 8:d:1,2','I 6,8:d:1,2:[z67]','I 8,[20]:d:1,5','I 6,8:d:1,5:[z58]','I 8,[20]:d:1,7','I 6,8:d:1,7','I 6,8:d:e,n,x','I 6,8:d:e,n,z15','I 8,[20]:d:z6','II 6,8:d:z6:z42','I 8:e,h:1,2','I 6,8,[20]:e,h:1,2:[z67],[z78]','I 8:e,h:1,5','I 6,8:e,h:1,5','I 6,8:e,h:1,6:[R1…]','I 8,[20]:e,h:1,7','I 6,8:e,h:e,n,x','I 6,8:e,h:e,n,z15','I 8,[20]:e,h:z6','I 6,8:f,g:e,n,z15','II 6,8:f,g,m,t:[e,n,x]','I 8,[20]:g,m,s:–','I 6,8:g,m,[s]:[e,n,x]','II 6,8:g,m,t:1,7','I 8,[20]:g,m,t:–','I 8,[20]:g,s,t:–','I 6,8:g,s,t:–','I 8,[20]:m,t:–','II 6,8:m,t:1,5','II 6,8:m,t:e,n,x','I 6,8:m,t:–','I 6,8:i:1,2','I 8,[20]:i:1,5','I 6,8:i:1,5','I 6,8:i:1,6','I 6,8:i:1,7','I 6,8:i:e,n,x','I 6,8:i:e,n,z15','I 8,[20]:i:l,w','I 6,8:i:l,w','I 8,[20]:i:z6','I 6,8:k:1,2','I 8:k:1,5','I 6,8:k:1,5:[z58]','I 6,8:k:1,7','I 6,8:k:e,n,x','I 6,8:k:e,n,z15','IIIb 8:(k):z35','I 8:l,v:1,2','I 6,8:l,v:1,2','I 6,8:l,v:1,5','I 8:l,v:1,6','I 6,8:l,v:1,7','I 6,8:l,v:e,n,x','II 6,8:l,v:e,n,x','I 6,8:l,v:e,n,z15','I 6,8:l,v:z6','I 6,8:l,w:1,2','II 6,8:l,w:z6:z42','I 6,8:l,z13,z28:1,5','I 6,8:l,z13,[z28]:e,n,z15','II 6,8:l,z28:e,n,x','I 6,8:r:1,2','I 8,[20]:r:1,5','I 6,8,[20]:r,[i]:1,5:[R1…]','I 8,[20]:r:1,6','I 8:r:1,7','I 6,8:r:1,7','I 8,[20]:r,i:e,n,z15','I 6,8:r,[i]:e,n,z15','I 8,[20]:r,[i]:l,w','I 6,8:r:l,w','IIIb 8:r:z','I 8,[20]:r,[i]:z6','I 8,[20]:y:1,2','I 6,8:y:1,2','I 8,[20]:y:1,5','I 6,8:y:1,5','I 6,8:y:1,6','II 6,8:y:1,6:z42','I 8,[20]:y:1,7','I 6,8:y:1,7','I 8:y:e,n,x','I 6,8:y:e,n,x','I 6,8:y:e,n,z15','I 8,[20]:y:z6','I 6,8:y:l,w','I 8,[20]:y:l,w','I 6,8:z:1,5','II 6,8:z:1,5','I 6,8:z:1,7','I 8:z:e,n,z15','I 6,8:z:e,n,z15','I 6,8:z:l,w','I 8,[20]:z:z6','I 8:z4,z23:1,7','I 6,8:z4,z23:1,7','I 6,8:z4,z23:e,n,x','I 6,8:z4,z23:[e,n,z15]','I 8,[20]:z4,z23:l,w','I 8,[20]:z4,z23:[z6]','I 8:z4,z24:e,n,x','I 8,[20]:z4,z24:–:[z45]','I 6,8:z4,z24:–','I 6,8:z4,z32:–','I 8,[20]:z10:1,2','I 6,8:z10:1,2','I 8,[20]:z10:1,5','I 6,8:z10:1,5','I 6,8:z10:1,7','I 8:z10:e,n,x','I 6,8:z10:e,n,x','I 8,[20]:z10:e,n,z15','I 6,8:z10:e,n,z15','I 8,[20]:z10:l,w','I 8,[20]:z10:z6','I 6,8:z10:z6','II 6,8:z29:1,5','II 6,8:z29:e,n,x:z42','I 8,[20]:z29:[e,n,z15]','I 6,8:z29:[e,n,z15]','II 6,8:z29:e,n,x','I 8,[20]:z35:1,2','I 6,8:z35:1,2','I 8,[20]:z35:z6','I 8,[20]:z38:–','I 8,[20]:z41:z6','I 6,8:z60:1,2','I [1],9,12:a:1,5','I [1],9,12:a:1,5','II 9,12:a:1,5','I 9,12:a:1,6','I [1],9,12:a:1,7','I [1],9,12:a:e,n,x','II [1],9,12:a:e,n,x','I [1],9,12:a:e,n,z15','II 9,12:a:z39','II [1],9,12:a:z42','I [1],9,12:b:1,2','I [1],9,12:b:1,5','I 9,12:b:1,7','II [1],9,12:b:e,n,x','I 9,12:b:e,n,z15','II [1],9,12:b:z6','II [1],9,12:b:z39','I 9,12:c:1,5','I 9,12:c:1,6','I 9,12:c:1,7','I 9,12:c:e,n,z15','I 9,12:c:z6','I [1],9,12:d:1,5','I 9,12:d:1,6','I 9,12:d:1,7','II [1],9,12:d:e,n,x','I [1],9,12:d:e,n,x','I 9,12:d:e,n,z15','I 9,12:d:z6','I [1],9,12:d:z35','II 9,12:d:z39','I 9,12,[Vi]:d:–:[z66]','I 9,12:e,h:1,2','I [1],9,12:e,h:1,5','I 9,12:e,h:1,7','I 9,12:e,h:e,n,z15','II 9,12:e,n,x:1,[5],7','II 9,12:e,n,x:1,6','I [1],9,12:[f],g,[t]:–','I [1],9,12:g,m:–','I 9,12:g,m,s:–','I 9,12:g,m,q:–','II [1],9,12:g,m,[s],t:[1,5,7]:[z42]','II [1],9,12:g,m,s,t:e,n,x','II 9,12:g,m,t:z39','I [1],9,12,[Vi]:g,p:–','I [1],9,12:g,p,s:–','I [1],9,12:g,p,u:–','I [1],9,12:g,q:–','II 9,12:g,s,t:e,n,x','I 9,12:g,z51:1,5','II [1],9,12:g,z62:[e,n,x]','I 9,12:g,z63:–','I 9,12:g,z85:–','II 9,12:m,t:e,n,x','I [1],9,12:m,t:[1,2]','II [1],9,12:m,t:1,5','II [1],9,12:m,t:z39','I 9,12:i:1,5','I [1],9,12:k:1,5','I 9,12:k:1,6','I 9,12:l,v:1,2','I [1],9,12:l,v:1,5:[R1…]','I 9,12:l,v:1,5:d','I 9,12:l,v:1,7:[z40]','I 9,12:l,v:e,n,x','II 9,12:l,v:e,n,x','I 9,12:l,v:e,n,z15','II 9,12:l,v:z39','I [1],9,12:l,w:1,5','II [1],9,12:l,w:e,n,x','I 9,12:l,z13:1,5','I 9,12:l,z13:1,7','I [1],9,12:l,z13:e,n,x','I [1],9,12:l,z28:1,5:[R1…]','I 9,12:l,z28:1,6','II 9,12:l,z28:1,5:z42','II 9,12:l,z28:e,n,x','I 9,12:l,z28:e,n,z15','I 9,12:r:1,5','I 9,12:r:1,7','I 9,12:r:e,n,x','I 9,12:r:e,n,z15','I 9,12:r:z6','I 9,12:y:1,7','II [1],9,12:y:z39','I [1],9,12:z:1,2','I [1],9,12:z:1,5','I 9,12:z:1,6','II [1],9,12:z:1,7','I 9,12:z:e,n,z15','II [1],9,12:z:z6','II 9,12:z:z39','I [1],9,12:z4,z23:[1,7]','I 9,12:z4,z24:–','I 9,12:z6:z67:[R1…]','I 9,12:z10:1,5','I 9,12:z10:z6','I 9,12:z10:e,n,z15','II 9,12:z29:1,5','II [1],9,12:z29:e,n,x','I 9,12:z35:z6','I [1],9,12:z38:–','II [1],9,12:z39:1,7','I [1],9,12:z41:1,5','II [1],9,12:z42:1,[5],7','I [1],9,12:–:–','I 9,46:a:1,2','I 9,46:a:1,6','I 9,46:a:e,n,x','I 9,46:a:e,n,z15','I 9,46:b:1,2','I 9,46:b:1,5','I 9,46:b:1,6','I 9,46:b:e,n,x','II 9,46:b:e,n,x','I 9,46:b:l,w','I 9,46:b:z6','I 9,46:b:z35','I [1],9,46:c:1,7','I 9,46:c:z6','I 9,46:d:1,5','I 9,46:d:1,6','I 9,46:d:1,7','I 9,46:d:e,n,z15','I 9,46:d:z6','I 9,46:d:z35','I 9,46:e,h:1,2','I 9,46:e,h:1,5','I 9,46:e,h:z6','II 9,46:e,n,x:1,5,7','I 9,46:f,g:–','I 9,46:g,m:–','I 9,46:g,m,s:1,2,7','II 9,46:g,[m],[s],t:[e,n,x]','I 9,46:g,s,t:–','II 9,46:g,z62:–','II 9,46:m,t:e,n,x','I 9,46:m,t:–','I 9,46:i:e,n,z15','I 9,46:i:z6','I 9,46:k:1,2','I 9,46:k:1,5','I 9,46:k:1,6','I 9,46:k:z35','I 9,46:l,v:1,5','I 9,46:l,v:1,6','I 9,46:l,v:e,n,x','I 9,46:l,w:–','II 9,46:l,w:e,n,x','I 9,46:l,z13,z28:e,n,z15','I 9,46:r:1,7','I 9,46:r:e,n,z15','I 9,46:r:z6','I 9,46:y:1,2','I 9,46:y:1,7','I 9,46:y:e,n,x','I 9,46:y:l,w','I 9,46:y:z6','II 9,46:z:1,5','II 9,46:z:e,n,x','I 9,46:z:e,n,z15','II 9,46:z:z39','I 9,46:z4,z23:–','II 9,46:z4,z24:z39:z42','I 9,46:z4,z24:–','I 9,46:z10:1,7','I 9,46:z10:e,n,x','I 9,46:z10:e,n,z15','I 9,46:z10:z6','II 9,46:z10:z6','II 9,46:z10:z39','I 9,46:z29:–:[z45]','I 9,46:z35:1,5','I 9,46:z35:e,n,z15','I 9,46:z35:z6','I 9,46:z38:–','II 9,46:z39:1,7','I 9,46:z41:–','II 1,9,12,46,27:a:z6','II 1,9,12,46,27:b:z39','II 1,9,12,46,27:c:z39','II 9,12,46,27:g,t:e,n,x','II 1,9,12,46,27:l,z13,z28:z39','II 1,9,12,46,27:y:z39','II 1,9,12,46,27:z4,z24:1,5','II 1,9,12,46,27:z10:1,5','II 1,9,12,46,27:z10:e,n,x','II 1,9,12,46,27:z10:z39','I 3,10:a:1,2','I 3,{10}{15}:a:1,5','I 3,{10}{15}{15,34}:a:1,7','I 3,10:a:e,n,x','II 3,10:a:e,n,x','I 3,10:a:e,n,z15','II 3,10:a:l,v','II 3,10:a:z39','II 3,10:a:z87','I 3,10:b:1,2','I 3,{10}{15}{15,34}:b:1,5','I 3,10:b:1,6','I 3,{10}{15,34}:b:1,7','I 3,10:b:e,n,x','II 3,10:b:e,n,x','I 3,{10}{15}:b:e,n,z15','I 3,10:b:l,w','I 3,10:b:z6','I 3,{10}{15}:b:z35','II 3,10:b:z39','I 3,10:c:1,2','I 3,{10}{15}:c:1,5','I 3,{10}{15}:c:1,6','I 3,10:c:1,7','I 3,10:c:e,n,z15','I 3,10:c:l,w','I 3,10:c:z6','I 3,10:d:1,2','I 3,{10}{15}:d:1,5','I 3,10:d:1,6','I 3,10:d:1,7','I 3,{10}{15}:d:e,n,x','II 3,10:d:e,n,x','I 3,10:d:e,n,z15','I 3,{10}{15}:d:l,w','I 3,10:d:z6','I 3,10:d:z35','I 3,{10}{15}:e,h:1,2:[z27]','I 3,{10}{15}{15,34}:e,h:1,5:[z48]','I 3,{10}{15}{15,34}:e,h:1,6:[z64]','I 3,{10}{15}:e,h:1,7','I 3,{10}{15,34}:e,h:e,n,x','I 3,10:e,h:e,n,z15','I 3,{10}{15}{15,34}:e,h:l,w','I 3,10:e,h:z6','II 3,10:e,n,x:1,7','I 3,10:f,g,[s]:[1,6]','I 3,10:f,g:e,n,x','I 3,10:g,m:–','I 3,{10}{15}{15,34}:g,m,s:–','II 3,{10}{15}:g,m,s,t:[1,5]','I 3,{10}{15}{15,34}:g,s,t:–:[z37]','I 3,10:g,t:1,5','II 3,10:g,t:–','II 3,10:m,t:1,5','I 3,{10}{15}{15,34}:m,t:[1,6]','II 3,10:m,t:e,n,x','I 3,10:i:1,2','I 3,10:i:1,5','I 3,10:i:1,6','I 3,10:i:1,7','I 3,10:i:e,n,x','I 3,{10}{15}:i:e,n,z15','I 3,10:i:l,w','I 3,10:i:z6','I 3,10:k:1,2','I 3,{10}{15}:k:1,5','I 3,10:k:1,7','I 3,10:k:e,n,x','I 3,10:k:e,n,z15','I 3,10:k:l,w','I 3,{10}{15}:l,v:1,2','I 3,10:l,v:1,5','I 3,{10}{15}:l,v:1,6','I 3,{10}{15}{15,34}:l,v:1,7:[d],[z77]','II 3,10:l,v:e,n,x','I 3,10:l,v:e,n,z15','II 3,10:l,v:z6','I 3,10:l,v:z35','I 3,10:l,w:z6:[z45]','I 3,10:l,z13:1,2','I 3,{10}{15}:l,z13:1,5','I 3,10:l,z13,z28:e,n,z15','I 3,10:l,z13,z28:z6','II 3,10:l,z28:1,5','I 3,10:l,z28:1,5','I 3,10:l,z28:1,7','I 3,10:l,z28:e,n,x','II 3,10:l,z28:e,n,x','II 3,10:l,z28:z39','I 3,10:r:1,5','I 3,{10}{15}:r:1,7','I 3,10:r:e,n,z15','I 3,{10}{15}:r:z6','I 3,10:r,i:1,2','I 3,10:r,i:1,6','I 3,{10}{15}:y:1,2:[z45]','I 3,{10}{15}{15,34}:y:1,5','I 3,10:y:1,7','I 3,{10}{15}:y:e,n,x','I 3,10:y:e,n,z15','I 3,10:y:l,w','I 3,{10}{15}:y:z6','I 3,10:z:1,5','II 3,10:z:1,5','I 3,10:z:1,6','I 3,10:z:1,7','II 3,10:z:e,n,x','I 3,10:z:e,n,z15','I 3,10:z:l,w','I 3,10:z:z6','II 3,10:z:z39','I 3,10:z4,z23:[1,7]','I 3,10:z4,z23:z6','I 3,{10}{15}:z4,z24:–','II 3,10:z4,z24:–','I 3,10:z10:1,2','I 3,{10}{15}{15,34}:z10:1,5:[z49]','I 3,{10}{15}{15,34}:z10:1,6','I 3,10:z10:1,7','I 3,10:z10:e,n,x','I 3,10:z10:e,n,z15','I 3,10:z10:z6','I 3,10:z29:e,n,x','II 3,10:z29:[e,n,x]','I 3,{10}{15}:z29:–','I 3,10:z35:1,5','I 3,10:z35:1,6','I 3,10:z35:1,7','II 3,10:z35:e,n,x,z15','I 3,10:z35:l,w','I 3,10:z35:z6','I 3,10:z36:–','I 3,10:z38:1,2','I 3,10:z38:1,6','I 3,10:z38:[z6]','II 3,10:z38:z42','II 3,10:z39:1,[5],7','I 3,10:z41:1,[2],5','I 3,{10}{15,34}:z69:1,7','II 3,15:z87:e,n,x,z15','I 3,15,34:y:1,6','I 1,3,19:a:1,5','I 1,3,19:a:1,7','I 1,3,19:a:e,n,z15','I 1,3,19:a:l,w','I 1,3,19:b:1,5:[z37]','I 1,3,19:b:1,6','I 1,3,19:b:e,n,x','I 1,3,19:b:e,n,z15','I 1,3,19:b:l,w','I 1,3,19:b:z6','I 1,3,19:c:1,5','I 1,3,19:c:1,6','I 1,3,19:c:1,7','I 1,3,19:d:1,2','I 1,3,19:d:1,5','I 1,3,19:d:1,7','I 1,3,19:d:e,n,z15','I 1,3,19:d:l,w:[z49]','I 1,3,19:d:z6','I 1,3,19:e,h:1,5','I 1,3,19:e,h:1,6','I 1,3,19:e,h:1,7','I 1,3,19:e,h:e,n,z15','I 1,3,19:e,h:l,w','I 1,3,19:f,g:–','I 1,3,19:f,g,t:1,7','I 1,3,19:f,g,t:e,n,z15','I 1,3,19:g,m,[t]:–','I 1,3,19:g,[s],t:–:[z27],[z34],[z37],[z43],[z45],[z46],[z82]','I 1,3,19:m,t:–','I 1,3,19:i:1,2','I 1,3,19:i:1,5','I 1,3,19:i:1,6','I 1,3,19:i:e,n,x','I 1,3,19:i:e,n,z15','I 1,3,19:i:l,w','I 1,3,19:i:z6','I 1,3,19:k:1,5','I 1,3,19:k:1,7','I 1,3,19:l,v:1,5','I 1,3,19:l,v:1,7','I 1,3,19:l,v:e,n,z15','I 1,3,19:l,w:1,5','I 1,3,19:l,z13:1,2','I 1,3,19:l,z13:1,6','I 1,3,19:l,z13,z28:1,2','I 1,3,19:l,[z13],z28:1,5','I 1,3,19:l,z13,z28:e,n,z15','I 1,3,19:l,z28:1,7','I 1,3,19:l,z28:e,n,x','I 1,3,19:r:e,n,z15','I 1,3,19:r,i:l,w','I 1,3,19:y:1,5','I 1,3,19:y:1,7','I 1,3,19:y:e,n,z15','I 1,3,19:y:e,n,x','I 1,3,19:y:l,w','I 1,3,19:z:1,5','I 1,3,19:z:1,6','I 1,3,19:z:1,7','I 1,3,19:z:e,n,z15','I 1,3,19:z:l,w','I 1,3,19:z:z6','I 1,3,19:z4,z23:–','I 1,3,19:z4,z24:–','I 1,3,19:z10:1,5','I 1,3,19:z10:e,n,z15','I 1,3,19:z29:[z6]:[z37]','I 1,3,19:z35:1,5','I 1,3,19:z38:[e,n,z15]','I 1,3,19:z41:e,n,z15','I 1,3,10,19:b:z35','I 1,3,10,19:f,g,t:1,(2),7','I 1,3,10,19:z4,z23:z6','I 1,3,[15],19:g,s,t:–','I 1,3,{10}{15},19:y:e,n,x','I 11:a:1,2','I 11:a:1,5','VI 11:a:1,5','I 11:a:1,6','I 11:a:1,7','I 11:a:e,n,z15','II 11:a:e,n,z15:d','I 11:a:l,z13,z28','II 11:a:z6:z42','I 11:b:1,2','I 11:b:1,5','I 11:b:1,6','VI 11:b:1,7','VI 11:b:e,n,x','I 11:b:e,n,z15','I 11:b:z6','I 11:c:1,5','I 11:c:1,6','II 11:c:e,n,z15','I 11:c:e,n,x','I 11:d:1,2','I 11:d:1,5','I 11:d:[e,n,x]:[r]','I 11:d:z6','I 11:e,h:1,2','I 11:e,h:1,6','I 11:e,h:l,z13,z28','I 11:g,m:1,5','I 11:g,s,t:–','II 11:g,[m],s,t:z39','IV 11:g,z51:–','I 11:m,t:–','II 11:m,t:e,n,x','I 11:i:1,2','I 11:i:1,5','I 11:i:1,6','I 11:i:e,n,x','IIIb 11:i:z','I 11:k:1,2','I 11:k:1,5','I 11:k:1,6','I 11:k:1,7','I 11:k:e,n,x,[z15]','I 11:k:l,w','I 11:k:l,z13,z28','IIIb 11:k:z53','I 11:l,v:1,2','I 11:l,v:1,5','I 11:l,v:e,n,x','I 11:l,v:e,n,z15','IIIb 11:l,v:z:[z56]','IIIb 11:l,v:z53','I 11:l,w:1,5','I 11:l,z13:1,2','I 11:l,z13,z28:1,5','I 11:l,z13,z28:e,n,x','I 11:l,z13,z28:e,n,z15','II 11:l,z28:e,n,x','I 11:r:1,5','I 11:r:e,n,x','I 11:r:e,n,z15','I 11:r,i:e,n,x,z15','I 11:r:l,z13,z28','I 11:y:1,5','I 11:y:1,7','I 11:y:e,n,x','I 11:y:e,n,x,z15','I 11:z:1,5','I 11:z:1,7','II 11:z:e,n,x','I 11:z:e,n,z15','I 11:z:z6:[z83]','II 11:z:z39','I 11:z4,z23:1,6','I 11:z4,z23:e,n,z15','IIIa 11:z4,z23:–','IV 11:z4,z23:–','I 11:z4,z24:–','IV 11:z4,z32:–','I 11:z10:1,2','I 11:z10:1,5','I 11:z10:e,n,x','I 11:z38:–','I 11:z41:1,2','II 11:z41:e,n,x','II 11:–:1,5','I [1],13,23:a:1,5','II [1],13,23:a:1,5','I 13,22:a:1,6','II 13,22:a:e,n,x','I [1],13,23:a:l,w','I 13,22:a:l,z13,z28','II [1],13,23:a:z42','I 13,22:b:1,5','I [1],13,23:b:1,5','I 13,22:b:1,6','II [1],13,23:b:[1,5]:z42','I 13,23:b:1,6','I [1],13,22:b:1,7','I 13,22:b:e,n,x','I [1],13,23:b:e,n,x','I 13,22:b:e,n,z15','I 13,23:b:e,n,z15','I 13,23:b:l,w','II [1],13,22:b:z42','I 13,22:c:e,n,x,z15','I 13,23:c:e,n,z15','I [1],13,23:d:1,2','II 13,22:d:1,5','I [1],13,23:d:1,5','I 13,22:d:1,6','I [1],13,23:d:1,6:[z37]','I [1],13,23:d:1,7','II 13,23:d:e,n,x','I 13,23:d:e,n,x','I [1],13,22:d:e,n,z15','I 13,23:d:e,n,z15','I [1],13,23:d:l,w','I 13,23:d:z6','I [1],13,23:e,h:1,5','I [1],13,22:e,h:1,6','I [1],13,23:e,h:l,w','II [1],13,23:e,n,x:1,[5],7','I 13,22:f,g:e,n,x','I [1],13,23:f,g,[s]:–:[z79]','I 13,22:g,m:[e,n,z15]','IIIb 13,22:g,m,s:z','I [1],13,23:g,m,[s],[t]:–','II [1],13,22:g,m,t:[1,5]','II 13,22:g,m,t:z42','II [1],13,23:g,m,s,t:1,5','II [1],13,23:g,m,[s],t:[e,n,x]','II [1],13,23:g,m,s,t:z42','I 13,22:g,s,t:–','II 13,23:g,s,t:e,n,x','I 13,23:g,[s],t:–','II [1],13,22:g,t:[1,5]','II 13,22:g,t:z6','II [1],13,23:g,t:1,5','II 13,23:g,t:e,n,x','II [1],13,23:g,[s],t:z42','IIIa [1],13,23:g,z51:–','I 13,22:m,t:–','II [1],13,23:m,t:1,5','II [1],13,23:m,t:e,n,x','II 13,22:m,t:z42:z39','II [1],13,23:m,t:z42','I [1],13,23:m,t:–','V [1],13,22:i:–','I [1],13,23:i:1,5','I 13,23:i:1,7','I 13,23:i:e,n,z15','I [1],13,23:i:l,w','I 13,23:i:z6','II 13,22:k:1,5:z42','I 13,23:k:1,6','I 13,23:k:–','I 13,23:k:e,n,z15','II 13,23:k:z41','I 13,22:l,v:1,5','IIIb 13,22:l,v:1,5,7','I 13,22:l,v:1,6','I [1],13,23:l,v:e,n,z15','II 13,23:l,w:e,n,x','I 13,22:l,z13:e,n,z15','II 13,22:l,z28:1,5','II 13,23:l,z28:1,5','II 13,23:l,z28:z6','II [1],13,23:l,z28:z42','V 13,22:r:–','I 13,23:r:1,6','I 13,23:r:e,n,z15','I [1],13,22:y:1,6','I 13,23:y:1,7','I [1],13,23:y:l,w','I [1],13,23:y:z6','I 13,22:z:1,5','II [1],13,23:z:1,5','I [1],13,23:z:1,5','IIIb 13,23:z:1,5','I [1],13,22:z:1,6:[z44],[z59]','I 13,23:z:1,6','I 13,22:z:1,7','I 13,22:z:z6','I [1],13,23:z:z6','I [1],13,22:z:e,n,z15','I [1],13,23:z:l,w:[z43]','II [1],13,23:z:z42','II 13,22:z:–','I [1],13,22:z4,z23:[e,n,z15]','IIIa 13,22:z4,z23:–','I 13,23:z4,z23:–','IIIa 13,23:z4,z23,[z32]:–','I [1],13,23:z4,z24:–','IIIa 13,23:z4,z24:–','I [1],13,22:z10:1,5','II [1],13,22:z10:z6','I 13,23:z10:e,n,z15','I 13,23:z10:l,w','II 13,22:z29:1,5','II 13,22:z29:e,n,x','II [1],13,23:z29:1,5','II [1],13,23:z29:e,n,x','I 13,22:z29:–','I [1],13,23:z29:–:[z37],[z43]','I 13,22:z35:1,6','I 13,22:z35:e,n,z15','I 13,23:z35:1,6','I 13,23:z35:e,n,z15','I 13,22:z38:–','I 13,23:z38:–','V 13,22:z39:–','II 13,22:z39:1,7','II [1],13,23:z39:1,5,7','I 13,23:z41:e,n,z15','II [1],13,23:[z42]:1,[5],7','II 13,23:–:1,6','I 1,6,14,25:a:1,5','VI [1],6,14:a:1,5','VI [1],6,14,[25]:a:e,n,x','I 1,6,14,25:a:e,n,z15','I 1,6,14,25:b:1,2','I 1,6,14,25:b:1,5','I [1],6,14,[25]:b:1,7','IIIb (6),14:b:e,n,x','I 1,6,14,25:c:1,5','I 1,6,14,25:c:1,6','I 1,6,14,25:c:e,n,x','I 1,6,14,25:c:l,w','I 6,14,24:d:1,5','I [1],6,14,[25]:d:1,5','I 1,6,14,25:d:1,6','I 6,14,24:d:1,7','I [1],6,14,[25]:d:1,7','I 6,14,[24]:d:e,n,x','I [1],6,14,[25]:d:e,n,x','I [1],6,14,[25]:d:e,n,z15','I 1,6,14,25:d:l,z13,z28','I 1,6,14,24:d:z6','I 6,14,[24]:e,h:1,5','I 1,6,14,[25]:e,h:1,5','I 1,6,14,25:e,h:1,6','I [1],6,14,[25]:e,h:1,7','V 6,14:e,n,z15:–','I [1],6,14,[25]:g,m:–','I [1],6,14,[25]:g,m,s:–','I [1],6,14,[25]:g,p:–','I 6,14:g,s,t:–','II 1,6,14:m,t:1,5','II 6,14:m,t:e,n,x','I 1,6,14,25:m,t:–','I 1,6,14,25:i:1,5','I [1],6,14,[25]:i:1,7','I 6,14,[24]:i:e,n,z15','I 1,6,14,25:i:e,n,z15','I [1],6,14,[25]:k:1,5','II 6,14,[24]:k:1,6','II 6,14:k:e,n,x','I 1,6,14,25:k:e,n,x','IIIb (6),14:k:z','II 1,6,14:k:z6:z42','IIIb (6),14:k:z53','I 6,14,24:l,v:1,7','I [1],6,14,[25]:l,v:1,7','I 1,6,14,[25]:l,v:e,n,x','I 6,14:l,v:e,n,z15','IIIb (6),14:l,v:z','IIIb (6),14:l,v:z35','IIIb (6),14:l,v:z53','VI 6,14:l,v:z88','I 6,14,25:l,z13,z28:1,6','I 1,6,14,25:l,z28:e,n,x','I 1,6,14,25:r,i:1,5','IIIb (6),14:r:z','I [1],6,14,[25]:r,[i]:e,n,z15','I 6,14,[24]:y:1,7','I 1,6,14,25:y:1,7','I 1,6,14,25:y:e,n,x','I 1,6,14,25:y:e,n,z15','I 1,6,14,25:z:1,5','I 1,6,14,25:z:1,6','I 6,14,24:z:e,n,x','I [1],6,14,[25]:z:e,n,x','I 1,6,14,25:z:e,n,z15','I 6,14,24:z:l,z13,z28','I [1],6,14,[25]:z:l,z13,z28','I 6,14:z4,z23:1,5','I 1,6,14,25:z4,z23:[e,n,z15]','IV 6,14:z4,z23:–','I 6,14,24:z4,z24:–','I 1,6,14,25:z4,z24:–','I 1,6,14,25:z10:1,2','VI 1,6,14,25:z10:1,(2),7','II 1,6,14:z10:1,5','I 1,6,14,25:z10:1,6','IIIb (6),14:z10:e,n,x,z15','IIIb (6),14:z10:z:[z56],[z90]','I 1,6,14,25:z10:z6','II 1,6,14:z10:z6:z42','IIIb 6,14:z10:z53','I 6,14:z36:1,5','II 6,14:z36:–','I 1,6,14,25:z38:[e,n,x]','II 1,6,14:z42:1,6','IIIb 6,14:z52:e,n,x,z15','IIIb [1],6,14,[25]:z52:z35','I 16:a:1,2','I 16:a:1,5','I 16:a:1,6','I 16:a:1,7','I 16:a:e,n,x','I 16:a:e,n,z15','I 16:a:z6','I 16:b:1,2','I 16:b:1,2,5','I 16:b:1,5','I 16:b:1,6','I 16:b:e,n,x','II 16:b:e,n,x','I 16:b:e,n,z15','I 16:b:l,w','I 16:b:z6','II 16:b:z39','II 16:b:z42','I 16:c:1,5','I 16:c:1,6','I 16:c:e,n,x','I 16:c:e,n,z15','I 16:c:l,w','I 16:d:1,2','I 16:d:1,5','II 16:d:1,5','I 16:d:1,6','I 16:d:1,7','I 16:d:e,n,x','II 16:d:e,n,x','I 16:d:e,n,z15','I 16:d:l,w:[z82]','I 16:d:z6','I 16:e,h:1,2','I 16:e,h:1,5','I 16:e,h:1,6','I 16:e,h:e,n,x','I 16:e,h:l,w','I 16:e,h:z6','II 16:e,n,x:1,(5),7','II 16:e,n,x:1,6:z42','I 16:f,g:–:[z37]','I 16:g,m,[t]:–','I 16:g,m,s:–','II 16:g,[m],[s],t:[1,5]:[z42]','II 16:g,[m],[s],t:[e,n,x]','I 16:g,s,t:–','II 16:m,t:e,n,x','I 16:m,t:e,n,z15','II 16:m,t:[z42]','I 16:m,t:–','I 16:i:1,5','I 16:i:1,6','I 16:i:1,7','I 16:i:e,n,z15','I 16:i:l,w','I 16:i:z6','IIIb 16:i:z35','I 16:k:1,2','I 16:k:1,6','I 16:k:e,n,x','I 16:k:e,n,z15','IIIb 16:(k):e,n,x,z15','IIIb 16:k:z','IIIb 16:(k):z35','IIIb 16:k:z53','IIIb 16:l,v:1,5,7','I 16:l,v:1,6:[z45]','I 16:l,v:1,7','I 16:l,v:e,n,x','I 16:l,v:e,n,z15','IIIb 16:l,v:z:[z61]','I 16:l,v:z6','IIIb 16:l,v:z35','IIIb 16:l,v:z53','I 16:l,w:1,5','I 16:l,w:1,6','I 16:l,w:e,n,x','I 16:l,w:e,n,z15','I 16:l,w:z6','II 16:l,w:z6','I 16:l,z13:e,n,z15','I 16:l,[z13],z28:[1,5]','I 16:l,z13,z28:1,6','I 16:l,z13,z28:z6','I 16:l,z13,z28:e,n,x','II 16:l,z28:z42','I 16:r,i:1,5','I 16:r:1,6','I 16:r:1,7','I 16:r:z6','I 16:r,i:e,n,x','I 16:r,i:e,n,z15','I 16:y:1,5','I 16:y:1,6','I 16:y:e,n,x','I 16:y:e,n,z15','I 16:z:1,5','I 16:z:1,6','I 16:z:1,7','I 16:z:e,n,x','II 16:z:e,n,x','I 16:z:e,n,[x],z15','I 16:z:z6','II 16:z:z42','I 16:z4,z23:[1,6]','I 16:z4,z23:z6','II 16:z4,z23:–','IV 16:z4,z23:–','I 16:z4,z24:–','II 16:z4,z24:–','IV 16:z4,z24:–','IV 16:z4,z32:–','II 16:z6:1,6','I 16:z10:1,5','IIIb 16:z10:1,7','I 16:z10:1,6','IIIb 16:z10:e,n,x,z15','I 16:z10:e,n,z15','I 16:z10:z6','I 16:z29:–','II 16:z29:1,5','II 16:z29:e,n,x','I 16:z35:1,5','I 16:z35:1,6','I 16:z35:e,n,z15','II 16:z35:e,n,x','IV 16:z36:–','II 16:z36:e,n,z15','I 16:z38:–','I 16:z39:[1,6]','II 16:z42:1,(5),7','II 16:z42:1,6','IIIb 16:z52:z35','I 17:a:1,2','I 17:a:1,5','I 17:a:e,n,x','I 17:b:1,2','I 17:b:1,5','I 17:b:e,n,x','II 17:b:e,n,x,z15','I 17:b:e,n,z15','II 17:b:z6','I 17:c:1,5','I 17:c:1,6','II 17:c:z39','II 17:d:–','I 17:d:1,5','I 17:d:e,n,z15','I 17:d:l,w','II 17:d:z6','I 17:e,h:1,2','II 17:e,n,x,z15:1,6','II 17:e,n,x,z15:1,[5],7','I 17:g,m:–','II 17:g,m,s,t:–','I 17:g,s,t:–','II 17:g,t:[e,n,x,z15]','II 17:g,t:z39','IIIa 17:g,z51:–','I 17:m,t:–','II 17:m,t:–','I 17:i:1,7','IIIb 17:i:z35','I 17:k:1,5','I 17:k:1,6','I 17:k:1,7','I 17:k:e,n,x','I 17:k:e,n,z15','IIIb 17:k:z','II 17:k:–','I 17:l,v:1,2','I 17:l,v:1,5','I 17:l,v:1,7','I 17:l,v:e,n,x','IIIb 17:l,v:e,n,x,z15','IIIb 17:l,v:z35','I 17:l,z28:e,n,x','I 17:r:1,2','IIIb 17:r:z','II 17:y:–','I 17:y:e,n,x','I 17:y:e,n,z15','I 17:y:l,w','I 17:z:1,2','I 17:z:1,5','I 17:z:1,6','II 17:z:1,7','I 17:z:e,n,z15','II 17:z:l,w:z42','IIIa 17:z4,z23:–','IIIa 17:z4,z23,z32:–','IIIa 17:z4,z24:–','IIIa 17:z4,z32:–','I 17:z10:e,n,x','IIIb 17:z10:e,n,x,z15:[z56]','IIIb 17:z10:z','II 17:z10:–','I 17:z29:–','IIIa 17:z29:–','IV 17:z29:–','I 17:z35:1,6','IIIa 17:z36:–','IV 17:z36:–','I [6],[14],18:a:e,n,z15','I [6],[14],18:b:1,5','I 18:b:1,7','I [6],[14],18:c:e,n,x','I 18:d:1,5','I [6],[14],18:d:1,7','I 18:g,m:–','I 18:g,s,t:–','IIIa 18:g,z51:–','IV 18:g,z51:–','II 18:m,t:1,5','I 18:m,t:–','I 18:k:1,5','IIIb 18:(k):z53','IIIb 18:(k):z54','IIIb 18:l,v:e,n,x,z15','I 18:l,v:e,n,z15','IIIb 18:l,v:z:[z50]','IIIb 18:l,v:z53','I 18:l,w:e,n,z15','I 18:r:1,5','IIIb 18:r:z','I 18:y:1,7','II 18:y:e,n,x,z15','I 18:z:–','I [6],[14],18:z4,z23:[1,5]:[z45],[z82]','I 18:z4,z23:z64','II 18:z4,z23:–','IIIa 18:z4,z23:–','I [6],[14],18:z4,z24:–','II 18:z4,z24:–','IIIa 18:z4,z32:–','IIIb 18:z10:e,n,x,z15','I 18:z10:1,5','I 18:z10:z6','II 18:z10:z6','II 18:z36:–','IIIa 18:z36:–','IV 18:z36,z38:–','I 18:z38:–','I 18:z71:–','I 18:–:1,6','I 21:a:[1,5]','II 21:b:1,5','I 21:b:1,6','I 21:b:e,n,x:[z33],[z49]','I 21:c:1,6','I 21:c:e,n,x','II 21:c:e,n,x','IIIb 21:c:e,n,x,z15','I 21:d:1,5','I 21:d:e,n,x','I 21:d:z6','I 21:f,g:e,n,x','II 21:g,[m],[s],t:–','IIIa 21:g,z51:–','IV 21:g,z51:–','II 21:m,t:–','IV 21:m,t:–','I 21:i:1,2','IIIb 21:i:1,5,7','IIIb 21:i:e,n,x,z15','I 21:k:1,6','IIIb 21:k:e,n,x,z15','IIIb 21:k:z','I 21:k:1,(2),5','IIIb 21:l,v:z','IIIb 21:l,v:z57','I 21:l,w:–','I 21:l,z28:e,n,z15','I 21:r:–','IIIb 21:r:z','I 21:y:1,2','I 21:y:e,n,x','II 21:z:–','I 21:z4,z23:–','IIIa 21:z4,z23:–','IV 21:z4,z23:–','II 21:z4,z24:–','IIIa 21:z4,z24:–','IV 21:z4,z32:–','IIIb 21:z10:e,n,x,z15','IIIb 21:z10:z','II 21:z10:[z6]','IIIb 21:z10:z53','IIIa 21:z29:–','I 21:z35:e,n,z15','IIIa 21:z36:–','IV 21:z36:–','IV 21:z38:–','IIIb 21:z65:e,n,x,z15','I 28:a:1,5','I 28:a:1,6','I 28:a:1,7','I 28:a:e,n,x','II 28:a:e,n,x','I 28:a:e,n,z15','I 28:a:z6','I 28:b:1,5','I 28:b:1,6','I 28:b:1,7','I 28:b:e,n,x','II 28:b:e,n,x','I 28:b:e,n,z15','I 28:b:l,w','II 28:b:z6','I 28:c:1,5','I 28:c:1,6','I 28:c:1,7','I 28:c:e,n,x','I 28:c:e,n,z15','I 28:c:l,w','I 28:c:z6','I 28:d:1,5','I 28:d:1,6','I 28:d:1,7','I 28:d:e,n,x','I 28:d:e,n,z15','I 28:d:l,w','I 28:d:z6','I 28:e,h:1,5','I 28:e,h:1,7','I 28:e,h:e,n,z15','II 28:e,n,x:1,7','II 28:e,n,z15:z87','I 28:f,g:–','I 28:g,m:1,6','I 28:g,m:[e,n,z15]','II 28:g,(m),[s],t:1,5','I 28:g,m,s:[e,n,z15]','II 28:g,m,t:e,n,x','II 28:g,m,t:z39','II 28:g,s,t:e,n,x','I 28:g,s,t:–','II 28:m,t:[e,n,x]','I 28:m,t:[e,n,z15]','II 28:m,t:z39','I 28:m,t:1,6','I 28:i:1,2','I 28:i:1,5','I 28:i:1,6','I 28:i:1,7','I 28:i:e,n,x','I 28:i:e,n,z15','I 28:i:l,w','I 28:k:1,2','I 28:k:1,5','I 28:k:1,6','I 28:k:1,7','IIIb 28:k:1,7','I 28:k:e,n,x','I 28:k:e,n,z15','I 28:l,v:1,5','I 28:l,v:e,n,x','I 28:l,v:e,n,z15','I 28:l,w:1,6','I 28:l,z13,z28:1,5','I 28:l,z13,z28:e,n,z15','II 28:l,z28:1,5','I 28:l,z28:e,n,x','II 28:l,z28:e,n,x','I 28:r:1,6','I 28:r:e,n,x','II 28:r:e,n,z15','I 28:r:l,w','I 28:r,[i]:1,5','I 28:r,i:1,7','I 28:r,[i]:e,n,z15','I 28:y:1,2','I 28:y:1,5','I 28:y:1,7:[z80],[z90]','I 28:y:e,n,x','I 28:y:e,n,z15','I 28:y:l,w','I 28:y:z6','I 28:z:1,5','II 28:z:1,5','I 28:z:1,6','I 28:z:1,7','I 28:z:e,n,z15','II 28:z:z39','I 28:z4,z23:1,5','I 28:z4,z23:1,6','I 28:z4,z23:[e,n,z15]','I 28:z4,z24:–','I 28:z10:1,2','I 28:z10:1,5','I 28:z10:1,6','I 28:z10:1,7','I 28:z10:e,n,x','I 28:z10:e,n,z15','I 28:z10:l,w','IIIb 28:z10:z','I 28:z29:–','II 28:z29:1,5','II 28:z29:e,n,x','I 28:z35:1,6','I 28:z35:1,7','I 28:z35:e,n,z15','I 28:z38:e,n,z15','I 30:a:1,2','I 30:a:1,5','I 30:a:e,n,x','I 30:a:e,n,z15','II 30:a:z39','I 30:b:1,2','I 30:b:1,5','I 30:b:1,7:[z33]','I 30:b:e,n,x','I 30:b:e,n,z15','I 30:b:l,z13,z28','II 30:b:z6','I 30:b:z6','I 30:c:1,7','I 30:c:e,n,z15','II 30:c:z39','I 30:d:1,5','I 30:e,h:1,2','I 30:e,h:1,5','I 30:g,m,[t]:–','II 30:g,m,s:e,n,x','I 30:g,m,s:–','IV 30:g,m,t,z51:–','I 30:g,s,t:–:[z59]','II 30:g,t:–','I 30:g,z51:–','II 30:m,t:–','I 30:i:1,2','I 30:i:1,5','I 30:i:e,n,z15','I 30:i:l,w','I 30:k:1,2','I 30:k:1,5','I 30:k:1,6','I 30:k:e,n,x','I 30:k:e,n,[x],z15','II 30:k:e,n,x,z15','I 30:k:l,z13,z28','I 30:l,v:1,2','I 30:l,v:1,5','I 30:l,z13,z28:1,6','I 30:l,z13,z28:e,n,z15','II 30:l,z28:z6','I 30:r:1,2','I 30:r:1,5','I 30:r:1,6','I 30:y:1,2','I 30:y:1,5','I 30:y:1,6','I 30:y:e,n,x','I 30:y:e,n,z15','I 30:z:e,n,x,z15','I 30:z4,z23:–','I 30:z4,z24:–','II 30:z6:1,6','I 30:z10:1,2','I 30:z10:1,5','I 30:z10:e,n,z15','II 30:z10:e,n,x,z15','I 30:z29:–','I 30:z35:1,6','I 30:z35:e,n,z15','I 30:z38:–','II 30:z39:1,7','I 35:a:e,n,z15','I 35:b:–','I 35:c:1,2','I 35:c:1,5','I 35:c:[e,n,z15]','II 35:d:1,5','I 35:d:l,w:[z58]','I 35:e,h:z6','I 35:f,g:–:[z27]','I 35:g,m,s:–','II 35:g,m,s,t:–','I 35:g,m,t:–','I 35:g,s,t:–','II 35:g,t:1,5','II 35:g,t:z42','I 35:g,t:–','IIIa 35:g,z51:–','II 35:g,z62:e,n,x','I 35:m,t:–','II 35:m,t:–','IIIb 35:i:e,n,x,z15','I 35:i:e,n,z15','I 35:i:l,w','IIIb 35:i:z','I 35:i:z6','IIIb 35:i:z35','IIIb 35:i:z53','I 35:k:e,n,z15','IIIb 35:k:e,n,x,z15','IIIb 35:k:z','IIIb 35:(k):z35','IIIb 35:k:z53:[z50]','IIIb 35:l,v:1,5,7','IIIb 35:l,v:e,n,x,z15:[z50]','IIIb 35:l,v:z35:[z67]','II 35:l,z28:–','IIIb 35:r:e,n,x,z15','I 35:r:l,w','IIIb 35:r:z','IIIb 35:r:z35','IIIb 35:r:z61','I 35:y:l,w','I 35:z4,z23:–:[z37],[z45]','IIIa 35:z4,z23:–','I 35:z4,z24:–','IIIa 35:z4,z24:–','IIIa 35:z4,z32:–','I 35:z10:1,5','I 35:z10:l,w','I 35:z10:z6','IIIb 35:z10:z35','II 35:z29:e,n,x','I 35:z29:–','IIIa 35:z29:–','IIIa 35:z36:–','I 35:z38:–','IIIb 35:z52:1,5,7','IIIb 35:z52:e,n,x,z15','IIIb 35:z52:z','IIIb 35:z52:z35','I 38:a:e,n,z15','II 38:b:1,2','I 38:b:e,n,z15','I 38:c:1,5','I 38:c:1,6:[z58]','I 38:d:1,5','II 38:d:[1,5]','II 38:d:z39','I 38:e,h:1,2','I 38:e,h:1,5','I 38:g,m,[s]:–','II 38:g,t:–','IIIa 38:g,z51:–','IV 38:g,z51:–','I 38:m,t:–','I 38:i:1,2','I 38:i:1,5','IIIb 38:i:z','IIIb 38:i:z53','I 38:k:1,2','I 38:k:1,5','I 38:k:1,6','I 38:k:e,n,x','IIIb 38:k:e,n,x,z15','IIIb 38:k:z','IIIb 38:k:z53','IIIb 38:(k):1,5,7','IIIb 38:(k):z35:[z56]','IIIb 38:(k):–','IIIb 38:(k):z55','I 38:l,v:1,2','I 38:l,v:1,5','I 38:l,v:e,n,z15','I 38:l,v:e,n,x','IIIb 38:l,v:z','IIIb 38:l,v:z35','IIIb 38:l,v:[z53]:[z54]','I 38:r:1,5','IIIb 38:r:1,5,7','I 38:r:1,6','IIIb 38:r:e,n,x,z15','IIIb 38:r:z:[z57]','IIIb 38:r:z35','I 38:y:1,5','I 38:y:1,6','I 38:y:e,n,x','I 38:z:–','I 38:z4,z23:1,2','IIIa 38:z4,z23:–','IV 38:z4,z23:–','I 38:z4,z24:–','I 38:z10:[1,5]','IIIb 38:z10:z','IIIb 38:z10:z53','I 38:z35:e,n,z15','I 38:z38:–','IIIb 38:z52:z35','IIIb 38:z52:z53','IIIb 38:z53:–:[z47],[z50],[z76]','IIIb 38:z61:[z53]','II 39:a:z39','I 39:b:1,2','I 39:b:l,w','II 39:c:e,n,x','I 39:d:1,5','I 39:e,h:1,2','I 39:e,h:1,5','II 39:e,n,x:1,7','I 39:m,t:–','II 39:[g],m,t:[e,n,x]','I 39:i:1,5','I 39:i:e,n,x','I 39:i:e,n,z15','I 39:k:1,5:[z48]','I 39:k:e,n,x','II 39:l,v:1,5','I 39:l,v:e,n,x','I 39:l,v:e,n,z15','II 39:l,z28:e,n,x','II 39:l,z28:z39','I 39:r:1,5','I 39:y:1,2','I 39:y:1,5','I 39:y:e,n,z15','I 39:z4,z23:–','I 39:z10:–','II 39:z10:z6','I 39:–:1,6','II 39:–:1,7','I 40:a:1,5','I 40:a:1,7','I 40:a:z6','II 40:a:z39','I 40:b:1,5','I 40:b:1,7','I [1],40:b:e,n,x','I [1],40:b:e,n,z15','I 40:b:z6','II 40:b:–','II [1],40:c:e,n,x,z15','II 40:c:z6','II [1],40:c:z39','I [1],40:d:1,5','II 40:d:–','I [1],40:e,h:1,2','II [1],40:e,n,x:1,[5],7','II [1],40:e,n,x,z15:1,6','I [1],40:g,m:–','I [1],40:g,m,s:e,n,x','II [1],40:g,[m],[s],[t]:e,n,x','II [1],40:g,[m],[s],t:[1,5]','II [1],40:g,t:e,n,x,z15','II 40:g,t:z39','IV [1],40:g,t:–','II [1],40:g,[m],[s],t:z42','IIIa 40:g,z51:–','IIIb 40:g,z51:e,n,x,z15','IV [1],40:g,z51:–','II 40:m,t:e,n,x','II 40:m,t:z39','II [1],40:m,t:z42','IV 40:m,t:–','IIIb 40:i:1,5,7','I [1],40:k:1,5','I [1],40:k:1,6','I 40:k:e,n,x','II [1],40:k:e,n,x,z15','IIIb 40:k:z:z57','II 40:k:z6','IIIb 40:k:z53','I [1],40:l,v:1,2','I 40:l,v:1,6','II 40:l,v:e,n,x','IIIb 40:l,v:z','IIIb [1],40:l,v:z53','I [1],40:l,w:[1,2]','II 40:l,z28:e,n,x','I [1],40:l,z13,z28:1,2','I [1],40:l,z28:1,5','II [1],40:l,z28:1,5:z42','I 40:l,z28:1,6','II [1],40:l,z28:z39','IIIb 40:r:z53','I 40:y:1,5','II [1],40:z:1,5','I 40:z:e,n,x','I 40:z:z6','II [1],40:z:z6','II [1],40:z:z39','II 40:z:z42','IIIa 40:z4,z23:–','IV [1],40:z4,z23:–','II 40:z4,z24:z39','IIIa 40:z4,z24:–','IV [1],40:z4,z24:–','IIIa 40:z4,z32:–','IV [1],40:z4,z32:–','II [1],40:z6:1,5','I 40:z10:z6','II 40:z10:e,n,x','IIIb 40:z10:z35','I [1],40:z29:–','IIIa 40:z29:–','II [1],40:z35:e,n,x,z15','I [1],40:z35:e,n,z15','V [1],40:z35:–','IIIa 40:z36:–','II [1],40:z39:1,5:z42','II [1],40:z39:1,6','IIIb 40:z39:1,6','II 40:z39:1,7','I [1],40:z41:1,2','II [1],40:z42:1,6','II [1],40:[z42]:1,(5),7','II [1],40:z81:z6','V [1],40:z81:–','I 41:a:–','II 41:b:1,5','I 41:b:1,6','VI 41:b:1,7','I 41:b:z6','I 41:b:e,n,z15','I 41:c:–','IIIb 41:c:e,n,x,z15','II 41:c:z6','I 41:d:1,5','II 41:d:z6','II 41:g,m,s,t:z6','II 41:g,t:–','IIIa 41:g,z51:–','I 41:m,t:1,6','I 41:i:1,5','I 41:i:1,6','I 41:k:1,6','II 41:k:1,6','II 41:k:z6','IIIb 41:(k):z35','II 41:l,z13,z28:e,n,x,z15','I 41:r:1,5','I 41:r:1,7','I 41:y:–','II 41:z:1,5','I 41:z:1,6','I 41:z4,z23:1,7','I 41:z4,z23:[e,n,z15]','IIIa 41:z4,z23:–','IV 41:z4,z23:–','IIIa 41:z4,z23,z32:–','I 41:z4,z24:1,5','IIIa 41:z4,z24:–','IIIa 41:z4,z32:–','II 41:z10:1,2','I 41:z10:1,5','I 41:z10:1,6','I 41:z10:e,n,x','II 41:z10:e,n,x,z15','II 41:z10:z6','I 41:z29:–','IIIa 41:z29:–','IV 41:z29:–','I 41:z35:1,6','IIIa 41:z36:–','IV 41:z36:–','I 41:z38:–','IV 41:z52:–','II 41:–:1,6','I [1],42:a:e,n,z15','II 42:b:1,5','I 42:b:1,6','II 42:b:e,n,x,z15','I [1],42:b:e,n,z15','I [1],42:b:l,w','I [1],42:b:z6','II 42:b:z6','I [1],42:c:e,n,z15','I [1],42:c:z6','I [1],42:d:z6','II 42:d:z6','II 42:e,n,x:1,6','II 42:g,t:–','I [1],42:g,z51:1,5','IIIa 42:g,z51:–','IV [1],42:g,z51:–','II 42:m,t:[e,n,x,z15]','I [1],42:m,t:–','I [1],42:i:l,w','I 42:i:1,6','I [1],42:i:z6','I 42:k:1,6','IIIb 42:k:e,n,x,z15','IIIb 42:k:z','I [1],42:k:z6','IIIb 42:(k):z35','IIIb 42:l,v:1,5,7:[z76]','I [1],42:l,v:1,6,7','II 42:l,v:e,n,x,z15','IIIb 42:l,v:e,n,x,z15','I 42:l,v:e,n,z15','IIIb 42:l,v:z','IIIb 42:l,v:z53','II [1],42:l,w:e,n,x','I [1],42:l,w:z35','II [1],42:l,[z13],z28:[z6]','I [1],42:r:e,n,z15','I [1],42:r:l,w','IIIb 42:r:z','IIIb 42:r:z53','II 42:r:–','IIIb 42:r:–:[z50]','I [1],42:y:e,n,z15','I [1],42:y:z6','II 42:z:1,5','I [1],42:z:1,6','II 42:z:e,n,x,z15','I 42:z:e,n,z15','II 42:z:z6','I [1],42:z4,z23:1,6:1,6','I 42:z4,z23:e,n,z15','IIIa 42:z4,z23:–','I [1],42:z4,z24:–','IIIa 42:z4,z24:–','IV [1],42:z4,z24:–','II 42:z6:1,6','II 42:z10:1,2','II 42:z10:e,n,x,z15','IIIb 42:z10:e,n,x,z15:[z60]','IIIb 42:z10:z','I [1],42:z10:z6','II 42:z10:z6','IIIb 42:z10:z35','IIIb 42:z10:z67','I [1],42:z29:[1,5]','II 42:z29:–','I [1],42:z35:1,6','I 42:z35:e,n,z15','I [1],42:z35:z6','I 42:z36:–','IV 42:z36:–','I [1],42:z38:z6','I [1],42:z41:–','IIIb 42:z52:z','IIIb 42:z57:1,5','I 43:a:1,2','I 43:a:1,5','II 43:a:1,5','II 43:a:z6','I 43:b:–','I 43:b:e,n,z15','II 43:b:z42','I 43:c:1,5','I 43:d:1,5','II 43:d:e,n,x,z15','II 43:d:z39','II 43:d:z42','II 43:e,n,x,z15:1,(5),7','II 43:e,n,x,z15:1,6','I 43:f,g,[t]:–','II 43:g,m,[s],t:[z42]','II 43:g,t:[1,5]','IIIb 43:g,t:–','IIIa 43:g,z51:–','IV 43:g,z51:–','II 43:g,z62:e,n,x','I 43:i:1,2','I 43:i:e,n,x','I 43:k:1,2','I 43:k:1,5','IIIb 43:k:z','IIIb 43:l,v:z53:[z56]','I 43:l,w:[z44]','I 43:l,z13:–','II 43:l,z13,z28:1,5','II 43:m,t:–','IIIb 43:r:e,n,x,z15','IIIb 43:r:z','IIIb 43:r:z53','I 43:y:1,2','I 43:y:1,5','I 43:z:1,5','II 43:z:1,5','I 43:z:e,n,z15','I 43:z4,z23:[1,2]','II 43:z4,z23:–','IIIa 43:z4,z23:–','IV 43:z4,z23:–','IIIa 43:z4,z24:–','IV 43:z4,z24:–','IV 43:z4,z32:–','I 43:z10:1,5','II 43:z29:e,n,x','II 43:z29:z42','I 43:z29:–','IIIa 43:z29:–','IV 43:z29:–','I 43:z35:1,6','IIIa 43:z36:–','IV 43:z36,z38:–','I 43:z38:–','I 43:z41:1,2','II 43:z42:[1,5,7]','IIIb 43:z52:z53','IV 44:a:–','I 44:a:1,5','I 44:a:e,n,x','I 44:a:l,w','I 44:b:1,6','I 44:b:e,n,x','I 44:b:e,n,z15','I 44:c:1,5','I 44:c:e,n,x','I 44:c:e,n,z15','I 44:d:1,5','I 44:d:e,n,x','I 44:d:e,n,z15','I [1],44:d:z35','II [1],44:e,n,x:1,6','I 44:f,g:–','I [1],44:g,m,[s]:[1,6]','I 44:g,s,t:[1,7]','II 44:g,t:z42','IIIb [1],44:g,t:1,5:z42','I 44:g,z51:–','IV 44:g,z51:–','I 44:m,t:–','II [1],44:m,t:z42','I [1],44:i:e,n,z15','I 44:k:e,n,z15','I 44:l,z28:1,5','I 44:r:1,5','I 44:r:e,n,z15','V 44:r:–','I 44:z:1,5','I 44:z:l,w','I 44:z:e,n,x','I 44:z:e,n,z15','I 44:z4,z23:–','I [1],44:z4,z23:e,n,z15','II 44:z4,z23:–','IIIa 44:z4,z23:–','IV 44:z4,z23:–','IIIa 44:z4,z23,z32:–','I 44:z4,z24:–','IIIa 44:z4,z24:–','IV 44:z4,z24:–','IIIa 44:z4,z32:–','IV [1],44:z4,z32:–','I [1],44:z10:1,7','I [1],44:z10:e,n,x','II 44:z29:e,n,x:z42','I 44:z29:–','IV 44:z29:–','IV 44:z36,[z38]:–','I 44:z38:–','II [1],44:z39:e,n,x,z15','V 44:z39:–','VI 45:a:e,n,x','I 45:a:e,n,z15','II 45:a:z10','I 45:b:1,5','I 45:b:1,6','I 45:b:e,n,z15','I 45:c:e,n,x','I 45:d:1,6','I 45:d:e,n,x','I 45:d:e,n,z15','I 45:f,g:–','I 45:g,m,[s],[t]:–','II 45:g,m,s,t:1,5','II 45:g,m,s,t:e,n,x','II 45:g,m,t:e,n,x,z15','I 45:g,s,t:–','IIIa 45:g,z51:–','IV 45:g,z51:–','II 45:m,t:1,5','I 45:m,t:–','I 45:k:1,5','I 45:k:1,7','I 45:k:e,n,z15','I 45:l,v:[e,n,z15]','I 45:r:e,n,z15','II 45:z:1,5','I 45:z:e,n,z15','II 45:z:z39','IIIa 45:z4,z23:–','IV 45:z4,z23:–','I 45:z4,z24:–','IIIa 45:z4,z24:–','IIIa 45:z4,z32:–','I 45:z10:–','I 45:z29:–:[z45]','II 45:z29:1,5','II 45:z29:e,n,x','II 45:z29:z42','IIIa 45:z29:–','I 45:z35:1,5','I 45:z36:–','IIIa 45:z36:–','IV 45:z36,z38:–','II 47:a:1,5','II 47:a:e,n,x,z15','I 47:b:1,2','II 47:b:1,5','II 47:b:e,n,x,z15','I 47:b:z6','II 47:b:z6','IIIb 47:c:1,5,7','I 47:c:1,6:[z78]','IIIb 47:c:e,n,x,z15:[z57]','IIIb 47:c:z','I 47:c:z6','IIIb 47:c:z35','II 47:d:1,5','I 47:d:e,n,x:[z58]','II 47:d:e,n,x,z15','II 47:d:z39','II 47:e,n,x,z15:1,6','I [1],47:f,g:–','I [1],47:g,m:–','I 47:g,m,s:–','II 47:[g,t]:e,n,x','IIIa 47:g,z51:–','I 47:m,t:e,n,z15','IIIb 47:i:e,n,x,z15:[z50]','I 47:i:e,n,z15','IIIb 47:i:z','IIIb 47:i:z35','IIIb 47:i:z53:[z57],[z84]','I 47:k:1,2','I 47:k:1,5','IIIb 47:k:1,5,7','I 47:k:1,6:[z58]','IIIb 47:k:e,n,x,z15','I 47:k:e,n,z15','IIIb 47:k:z','IIIb 47:k:z35','IIIb 47:k:z53:[z84]','IIIb 47:l,v:1,[5],7:[z50]','I 47:l,v:e,n,x','IIIb 47:l,v:e,n,x,z15','IIIb 47:l,v:z','IIIb 47:l,v:z35','IIIb 47:l,v:z53','IIIb 47:l,v:z57','IV 47:l,v:–','I [1],47:l,z13,z28:e,n,z15','IIIb 47:r:e,n,x,z15','I 47:r:1,2','IIIb 47:r:1,5,7','IIIb 47:r:z','IIIb 47:r,[i]:z35','IIIb 47:r:z53:[z70],[z74],[z77],[z90]','I 47:y:1,6','I 47:y:e,n,x','I 47:z:1,5','I 47:z:1,6','II 47:z:e,n,x,z15','II 47:z:z6','I 47:z4,z23:1,2','I 47:z4,z23:e,n,z15','I 47:z4,z23:1,6','I 47:z4,z23:z6:[z45],[z58]','I 47:z4,z23:l,w','IIIa 47:z4,z23:–','I 47:z4,z24:–','II 47:z6:1,6','IIIb 47:z10:1,5,7','I 47:z10:e,n,z15','IIIb 47:z10:z','IIIb 47:z10:z35','II 47:z29:e,n,x,z15','I 47:z29:–','IIIa 47:z29:–','I 47:z35:z6','I 47:z35:e,n,z15','IV 47:z36:–','I 47:z38:–','I 47:z44:–','IIIb 47:z52:1,5:z54','IIIb 47:z52:1,5,7','IIIb 47:z52:e,n,x,z15','IIIb 47:z52:z','IIIb 47:z52:z35','IIIb 47:z53:–:[z90]','I 48:a:1,5,7','II 48:a:z6','II 48:a:z39','II 48:b:z6','II 48:b:e,n,x,z15','V 48:b:–','IIIb 48:c:z','II 48:d:1,2','II 48:d:z6','I 48:d:–:[z58]','I 48:e,h:1,5','II 48:e,n,x,z15:z6','II 48:g,m,t:–','IIIa 48:g,z51:–','IV 48:g,z51:–','IIIb 48:i:z:[z72]','IIIb 48:i:z35:[z57]','IIIb 48:i:z53','IIIb 48:i:z61','V 48:i:–','IIIb 48:k:1,5,(7)','II 48:k:e,n,x,z15','IIIb 48:k:e,n,x,z15','I 48:k:e,n,z15','IIIb 48:k:z','IIIb 48:k:z35:[z75]','II 48:k:z39','IIIb 48:k:z53','IIIb 48:k:z57','I 48:l,v:1,5','IIIb 48:l,v:1,5,(7):[z47],[z50],[z89]','IIIb 48:l,v:z','IIIb 48:l,w:1,5,7:[z50]','IIIb 48:r:e,n,x,z15','IIIb 48:r:z','I 48:z:1,5:[z58]','II 48:z:1,5','IIIb 48:z:1,5,7','IIIa 48:z4,z23:–','IV 48:z4,z23:–','IIIa 48:z4,z23,z32:–','I 48:z4,z24:–','IIIa 48:z4,z24:–','IIIb 48:z4,z24:–','IV 48:z4,z24:–','IIIa 48:z4,z32:–','IV 48:z4,z32:–','II 48:z10:[1,5]','VI 48:z10:1,5','II 48:z10:1,6','I 48:z10:e,n,x','IIIb 48:z10:e,n,x,z15','IIIb 48:z10:z','II 48:z29:–','IIIa 48:z29:–','IV 48:z29:–','IIIb 48:z35:z52','V 48:z35:–','IIIa 48:z36:–','IV 48:z36,[z38]:–','II 48:z39:z81','V 48:z39:–','V 48:z41:–','IIIb 48:z52:e,n,x,z15','IIIb 48:z52:z','V 48:z65:–','V 48:z81:–','II 48:z81:1,5,7','IV 50:a:–','I 50:b:e,n,x','II 50:b:z6','IV 50:b:–','I 50:d:1,5:[z82]','IV 50:d:–','II 50:e,n,x:1,7','II 50:g,[m],s,t:[1,5]','IIIa 50:g,z51:–','IV 50:g,z51:–','II 50:g,z62:e,n,x','II 50:m,t:z6:z42','IIIb 50:i:1,5,7','IIIb 50:i:e,n,x,z15','IIIb 50:i:z','IIIb 50:i:z53','IIIb 50:k:1,5,7','II 50:k:e,n,x:z42','IIIb 50:k:e,n,x,z15','IIIb 50:k:z:[z50],[z57],[z68],[z86]','II 50:k:z6','IIIb 50:k:z35','IIIb 50:k:z53','I 50:l,v:1,2','IIIb 50:l,v:e,n,x,z15','IIIb 50:l,v:z','IIIb 50:l,v:z35','IIIb 50:l,v:z57','VI 50:l,v:z67','II 50:l,w:e,n,x,z15:z42','II 50:l,z28:z42','IIIb 50:r:1,5,(7)','IIIb 50:r:e,n,x,z15','IIIb 50:r:z:[z67]','IIIb 50:r:z35:[z58]','IIIb 50:r:z53','I 50:y:1,6','II 50:z:e,n,x','IIIb 50:z:z52','IIIa 50:z4,z23:–','IV 50:z4,z23:–','IIIa 50:z4,z23,z32:–','IIIa 50:z4,z24:–','IV 50:z4,z24:–','IIIa 50:z4,z32:–','IV 50:z4,z32:–','IIIb 50:z10:z:[z56]','II 50:z10:z6:z42','IIIb 50:z10:z53','I 50:z29:–','IIIa 50:z29:–','IIIa 50:z36:–','II 50:z42:1,7','IIIb 50:z52:1,5,7','IIIb 50:z52:z35','IIIb 50:z52:z53','IV 51:a:–','I 51:a:1,2','I 51:a:e,n,x','I 51:b:1,5','IV 51:b:–','II 51:c:–','I [1],51:d:1,5','I 51:e,h:1,2','II 51:g,s,t:e,n,x','IIIa 51:g,z51:–','I 51:m,t:–','I 51:i:1,2','I 51:i:1,5','I 51:k:e,n,z15','IIIb 51:k:z35','I 51:l,v:1,2','I 51:l,v:1,5','I 51:l,v:e,n,x','IIIb 51:l,v:z','I 51:l,z28:1,5','II 51:l,z28:z6','II 51:l,z28:z39','I 51:r:e,n,x','I 51:r,i:l,z13,z28','I 51:y:1,2','I 51:z:1,5','I [1],51:z:1,6','I 51:z:e,n,z15','IIIa 51:z4,z23:–','IV 51:z4,z23:–','IIIa 51:z4,z24:–','IIIa 51:z4,z32:–','I 51:z10:1,5','II 51:z29:e,n,x,z15','IIIa 51:z36:–','II 51:–:1,7','I 52:a:1,5','I 52:a:e,n,z15','I 52:b:1,5','I 52:b:e,n,x','II 52:c:k','I 52:d:1,5','II 52:d:e,n,x,z15','II 52:d:z39','I 52:e,h:1,6','I 52:e,h:1,7','I 52:g,t:–','II 52:g,t:–','I 52:k:1,5','IIIb 52:k:e,n,x,z15','IIIb 52:k:z35','IIIb 52:k:z53','IIIb 52:l,v:z53','I 52:l,w:1,5','II 52:z:z39','IIIb 52:z:z52','II 52:z39:1,5,7','II 52:z44:1,5,7','II 53:c:1,5:[R1…]','II 53:d:1,5','II [1],53:d:z39','II 53:d:z42','IIIa 53:g,z51:–','IV [1],53:g,z51:–','IIIb 53:i:z','IIIb 53:k:e,n,x,z15','IIIb 53:k:z','IIIb 53:(k):z35','IIIb 53:k:z53','IIIb 53:l,v:e,n,x,z15','IIIb 53:l,v:z','IIIb 53:l,v:z35','II 53:l,z28:e,n,x','II 53:l,z28:z6','II 53:l,z28:z39','IIIb 53:r:z','IIIb 53:r:z35','IIIb 53:r:z68','II 53:z:1,5','IIIb 53:z:1,5,(7)','II 53:z:z6','IIIa 53:z4,z23:–','IV 53:z4,z23:–','IIIa 53:z4,z23,z32:–','II 53:z4,z24:–','IIIa 53:z4,z24:–','IIIb 53:z10:z','IIIb 53:z10:z35','IIIa 53:z29:–','IV [1],53:z36,z38:–','IIIb 53:z52:z35','IIIb 53:z52:z53','I 53:–:1,6','I 21,54:b:e,n,x','I 54:e,h:1,5','I 3,54:e,h:1,6','I 54:f,g,s:–','I 3,54:g,s,t:–','I 4,12,54:m,t:–','I 8,[20],54:i:z6','I 6,7,54:k:1,5','I 54:r:1,5','I 3,15,54:y:1,5','I 54:z4,z23:–','I 54:z10:e,n,x','I 54:z10:e,n,z15','I 54:z29:–','II 55:k:z39','II 56:b:[1,5]','II 56:d:–','II 56:e,n,x:1,7','II 56:l,v:z39','II 56:l,z28:–','II 56:z:z6','IIIa 56:z4,z23:–','IIIa 56:z4,z23,z32:–','II 56:z10:e,n,x','IIIa 56:z29:–','I 57:a:z6','II 57:a:z42','I 57:b:1,7','I 57:b:e,n,z15','IIIb 57:c:e,n,x,z15','IIIb 57:c:z:[z70],[z90]','II 57:d:1,5','II 57:g,[m],s,t:z42','II 57:g,t:–','IIIb 57:i:e,n,x,z15','IIIb 57:i:z','IIIb 57:k:e,n,x,z15','IV 57:z4,z23:–','IIIb 57:z10:z','II 57:z29:z42','II 57:z39:e,n,x,z15','II 57:z42:1,6:z53','II 58:a:z6','II 58:b:1,5','II 58:c:z6','II 58:d:z6','IIIb 58:i:e,n,x,z15','IIIb 58:i:z53','IIIb 58:k:z','IIIb 58:l,v:e,n,x,z15','IIIb 58:l,v:z35','II 58:l,z13,z28:1,5','II 58:l,z13,z28:z6','IIIb 58:r:e,n,x,z15','IIIb 58:r:z','IIIb 58:r:z53:[z47],[z57],[z70],[z73]','II 58:z6:1,6','II 58:z10:1,6','IIIb 58:z10:e,n,x,z15','II 58:z10:z6','IIIb 58:z10:z53:[z50]','II 58:z39:e,n,x,z15','IIIb 58:z52:z','IIIb 58:z52:z35','IIIb 59:c:e,n,x,z15','IIIb 59:i:e,n,x,z15','IIIb 59:i:z','IIIb 59:i:z35:[z84]','IIIb 59:(k):e,n,x,z15','II 59:k:z65','IIIb 59:(k):z','IIIb 59:(k):z35','IIIb 59:k:z53','IIIb 59:l,v:z','IIIb 59:l,v:z53','IIIb 59:r:z35','II [1],59:z:z6','IIIa 59:z4,z23:–','IIIb 59:z10:z53','IIIb 59:z10:z57','IIIa 59:z29:–','IIIa 59:z36:–','VI 59:z36:–','IIIb 59:z52:z53','II 60:b:–:[R1…]','II 60:g,m,t:z6','IIIb 60:i:[e,n,x,z15]:[z50]','IIIb 60:i:[z]:[z50]','IIIb 60:i:[z35]:[z50]','IIIb 60:k:z','IIIb 60:k:z35','IIIb 60:(k):z53','IIIb 60:l,v:z','IIIb 60:r:e,n,x,z15','IIIb 60:r:z','IIIb 60:r:z35','IIIb 60:r:z53','II 60:z:e,n,x','IIIb 60:z10:z','IIIb 60:z10:z35','II 60:z10:z39','IIIb 60:z10:z53','II 60:z29:e,n,x','V 60:z41:–','IIIb 60:z52:1,5,[7]','IIIb 60:z52:z','IIIb 60:z52:z35','IIIb 60:z52:z53','IIIb 61:c:1,5,(7)','IIIb 61:c:z35','IIIb 61:i:e,n,x,z15','IIIb 61:i:z','IIIb 61:i:z35','IIIb 61:i:z53','IIIb 61:k:1,5,(7)','IIIb 61:k:z35','IIIb 61:(k):z53','IIIb 61:l,v:1,5,7:[z57]','IIIb 61:l,v:z','IIIb 61:l,v:z35','IIIb 61:r:1,5,7','IIIb 61:r:z','IIIb 61:r:z35','IIIb 61:r:z53:[z47],[z50]','IIIb 61:z10:z35','V 61:z35:–','IIIb 61:z52:1,5,7','IIIb 61:z52:z','IIIb 61:z52:z35','IIIb 61:z52:z53','IIIa 62:g,z51:–','IIIa 62:z4,z23:–','IIIa 62:z4,z32:–','IIIa 62:z29:–','IIIa 62:z36:–','IIIa 63:g,z51:–','IIIb 63:(k):z','IIIa 63:z4,z23:–','IIIa 63:z4,z32:–','IIIa 63:z36:–','IIIb 65:c:1,5,7','IIIb 65:c:z','IIIb 65:c:z53','II 65:g,t:–','IIIb 65:i:e,n,x,z15','IIIb 65:i:z','IIIb 65:(k):z','IIIb 65:(k):z35','IIIb 65:(k):z53','IIIb 65:l,v:e,n,x,z15','IIIb 65:l,v:z','IIIb 65:l,v:z35','IIIb 65:l,v:z53','IIIb 65:r:z35','IIIb 65:z10:e,n,x,z15','IIIb 65:z10:z','IIIb 65:z52:e,n,x,z15','IIIb 65:z52:z','IIIb 65:z52:z35','IIIb 65:z52:z53','II 65:–:1,6','V 66:z35:–','V 66:z39:–','V 66:z41:–','V 66:z65:–','V 66:z81:–','I 6,7:c:1,5','I [1],9,12:a:1,5','I 67:r:1,2','I 3,15:y:1,2:[z45]', 'I 6,7,14:d:e,n,x', 'I 3,15:g,m,s:–', 'I 3,15:e,h:1,6:[z64]', 'I 3,15,34:e,h:1,6:[z64]', 'I 3,15,34:b:1,5', 'I 6,14,18:z4,z23:[1,5]:[z45],[z82]', 'I 6,7,14:l,v:z6', 'I 3,15:l,v:1,7:[d],[z77]', 'I 3,15,34:l,v:1,7:[d],[z77]', 'I 3,15:a:1,5', 'I 3,15:z10:1,5:[z49]', 'I 3,15,34:z10:1,5:[z49]', 'I 6,7,14:z38:–:[z82]', 'I 6,7,14:d:l,w', 'I 3,15:l,v:1,6', 'I 3,15:e,h:l,w', 'I 3,15,34:e,h:l,w', 'I 3,15:e,h:1,5:[z48]', 'I 3,15,34:e,h:1,5:[z48]', 'I 3,15:l,v:1,2', 'I 3,15:e,h:1,7', 'I 6,7,14:m,t:[z57]', 'I 3,15:y:1,5', 'I 3,15,34:y:1,5', 'I 3,15,34:a:1,7', 'I [1],4,[5],12:b:1,2:[z5],[z33]', 'I 6,7,14:f,g:–', 'I 3,15:d:1,5', 'I 3,15:d:e,n,x', 'I 1,4,[5],12,27:z4,z23:[1,2]', 'I 3,15:y:z6', 'I 3,15:l,z13:1,5', 'I 3,15:e,h:1,2', 'I 3,15:e,h:1,2:z27', 'I 3,15:r:z6', 'I 3,15:g,s,t:–:[z37]', 'I 3,15,34:g,s,t:–:[z37]']
std_wklm_formula = ['i 1,2,12:a:1,5','i 2,12:g,m:–','i 1,2,12:g,p:–','i 2,12:l,v:1,5','i 1,4,5,12:a:1,2','i 4,12,27:a:1,5','i 4,5,12:a:1,5','i 4,5,12:a:1,7','i 1,4,5,12:a:e,n,x','i 1,4,12,27:a:e,n,z15','ii 1,4,5,12,27:a:e,n,x','i 1,4,12:a:l,w','i 1,4,12,27:a:z6','ii 1,4,12,27:a:z39','i 1,4,5,12:b:1,2:z5,z33','i 1,4,5,12,27:b:1,5','ii 4,12:b:1,5','i 4,12,27:b:1,6','i 1,4,12,27:b:1,7','i 1,4,5,12,27:b:e,n,x','ii 1,4,5,12,27:b:e,n,x','i 1,4,12,27:b:e,n,z15','i 1,4,12,27:b:l,w','i 1,4,12,27:b:z6','i 4,12,27:b:–','i 1,4,12,27:c:1,5','i 4,12:c:1,6','i 4,12,27:c:1,7','i 4,12:c:e,n,x','i 1,4,12,27:c:e,n,z15','i 1,4,12,27:c:l,w','i 4,12,27:c:z6','i 1,4,5,12,27:d:1,2','i 1,4,12,27:d:1,5','i 1,4,12,27:d:1,6','i 1,4,12,27:d:1,7','ii 4,12:d:e,n,x','i 1,4,5,12,27:d:e,n,x','i 1,4,12,27:d:e,n,z15:e,h','i 1,4,12,27:d:l,w','i 1,4,12,27:d:z6','i 4,12:d:z35','i 1,4,5,12:e,h:1,2','i 1,4,5,12:e,h:1,5:r1','i 4,12:e,h:1,6','i 4,12:e,h:1,7','i 1,4,5,12:e,h:e,n,x','i 1,4,5,12:e,h:e,n,z15','i 1,4,12:e,h:l,w','ii 4,12:e,n,x:1,2,7','ii 1,4,12,27:e,n,x:1,5,7','i 1,4,5,12:f,g:1,2','i 1,4,5,12:f,g,s:1,2:z27,z45','ii 1,4,5,12:f,g,t:z6:z42','i 4,12:g,m:–','i 1,4,5,12:g,m,s:1,2','ii 1,4,12,27:g,m,s,t:e,n,x','ii 1,4,12,27:g,m,t:1,5','ii 4,12:g,m,t:z39','i 4,12:g,m,t:z67','i 1,4,5,12,27:g,s,t:1,2:z43','i 1,4,12,27:g,t:–','i 4,5,12:g,z51:1,7','i 4,5,12:g,z51:e,n,z15','ii 4,12:g,z62:–','i 1,4,5,12:m,t:1,5','i 4,5,12:m,t:e,n,z15','i 1,4,5,12:i:1,2','i 1,4,5,12:i:1,5','i 4,12:i:1,6','i 4,12:i:e,n,x','i 1,4,12:i:e,n,z15','i 1,4,12,27:i:l,w','i 1,4,12:i:z6','ii 4,12,27:i:z35','i 1,4,12,27:k:1,5','i 1,4,12,27:k:1,6','ii 1,4,12,27:k:1,6','i 4,12,27:k:e,n,x','i 4,5,12:k:e,n,z15','i 4,5,12:l,v:1,2','i 1,4,5,12,27:l,v:1,5','i 4,12:l,v:1,6','i 1,4,12,27:l,v:1,7:z40','i 1,4,12,27:l,v:e,n,x','ii 1,4,12,27:l,v:e,n,x','i 4,5,12:l,v:e,n,z15','ii 1,4,12,27:l,v:z39','i 4,12:l,w:1,5','i 4,12:l,w:1,6','ii 4,12:l,w:e,n,x','i 4,12:l,w:e,n,z15','i 1,4,12,27:l,w:z6','i 1,4,5,12,27:l,z13,z28:1,2','i 1,4,12,27:l,z13,z28:1,5','i 4,12:l,z13,z28:1,6','i 1,4,12,27:l,z13,z28:1,7','i 1,4,12,27:l,z13,z28:e,n,x','i 1,4,12,27:l,z13,z28:e,n,z15','i 4,12:l,z28:e,n,x','ii 1,4,12:l,z28:e,n,x','i 1,4,5,12:r:1,2','i 4,12,27:r:1,5','i 4,12:r:1,6','i 1,4,12,27:r:1,7','i 1,4,5,12:r:l,w','i 4,12,27:r:z6','i 1,4,12,27:r,i:e,n,z15','i 4,12:r,i:l,w','i 1,4,5,12:y:1,2','i 4,12,27:y:1,5','i 4,12:y:1,6','i 1,4,12,27:y:1,7','i 1,4,5,12,27:y:e,n,x','i 1,4,12,27:y:e,n,z15','i 1,4,12,27:y:z6','i 4,5,12:z:1,2','i 1,4,12:z:1,5','ii 1,4,12,27:z:1,5','i 4,12:z:1,6','i 1,4,12:z:1,7','ii 4,12:z:1,7','i 4,12:z:e,n,x','ii 1,4,12,27:z:e,n,x','i 1,4,5,12:z:e,n,z15','i 1,4,12:z:l,w','i 1,4,12,27:z:z6','ii 4,12:z:z39','i 1,4,5,12,27:z4,z23:1,2','i 4,12:z4,z23:e,n,z15','i 4,12,27:z4,z23:z6','i 1,4,5,12:z4,z24:1,5','i 1,4,5,12:z10:1,2','i 1,4,12:z10:1,5','i 4,12:z10:1,6','i 4,12:z10:1,7','i 4,12:z10:e,n,x','i 4,12:z10:e,n,z15','i 1,4,12:z10:l,w','i 1,4,12,27:z10:z6','i 1,4,12,27:z10:z35','i 1,4,12,27:z29:–','ii 1,4,12:z29:e,n,x','i 4,12:z35:1,5','i 1,4,12,27:z35:1,7','i 1,4,12,27:z35:z6','i 1,4,12,27:z35:e,n,z15','i 4,12:z36:–','i 1,4,12:z38:1,5','i 1,4,5,12,27:z38:e,n,z15','ii 1,4,12,27:z39:1,5,7','i 1,4,12,27:z41:1,2,5','i 1,4,12,27:z41:e,n,z15','i 4,12:–:e,n,x','i 4,12:z91:–','i 6,7:a:1,5','ii 6,7,14:a:1,5','i 6,7:a:1,6','i 6,7:a:1,7','i 6,7,14:a:e,n,x','i 6,7:a:e,n,z15','i 6,7,14:a:l,w','i 6,7,14:a:z6','ii 6,7:a:z6','ii 6,7:a:z42','i 6,7:b:1,2','ii 6,7:b:1,5','i 6,7,14:b:1,5','i 6,7:b:1,6','i 6,7:b:1,7','i 6,7,14:b:e,n,x','i 6,7:b:e,n,z15','ii 6,7:b:e,n,x:z42','i 6,7,14:b:l,w:z59','i 6,7,14:b:z6','i 6,7:b:z35','ii 6,7:b:z39','i 6,7,14:c:1,2','i 6,7,vi:c:1,5','i 6,7:c:1,5','i 6,7:c:1,5','i 6,7:c:1,6','i 6,7:c:1,7','i 6,7:c:z6','i 6,7:c:e,n,x','i 6,7,14:c:e,n,z15','i 6,7:d:1,2','i 6,7,14:d:1,5','i 6,7:d:1,6','i 6,7:d:1,2,7','i 6,7,14:d:e,n,x','i 6,7,14:d:e,n,z15','i 6,7,14:d:l,w','i 6,7:d:l,z13,z28','i 6,7,14:d:z6','ii 6,7:d:z42','i 6,7:e,h:1,2','i 6,7:e,h:1,5','i 6,7:e,h:1,6','i 6,7:e,h:1,7','i 6,7,14:e,h:e,n,z15','ii 6,7:e,n,x:1,6:z42','i 6,7:e,n,z15:1,6','i 6,7,14:f,g:–','i 6,7:f,g,t:1,2,7','i 6,7:f,g,t:e,n,x','i 6,7,14,54:g,m,p,s:1,2,7','i 6,7:g,m,s:e,n,z15','ii 6,7:g,m,s,t:e,n,x','ii 6,7:g,m,s,t:1,5','ii 6,7:g,m,s,t:z39','ii 6,7:g,m,s,t:z42','i 6,7,14:g,m,t:–','i 6,7:g,s,q:–','i 6,7:g,s,t:1,6','ii 6,7:g,t:e,n,x:z42','i 6,7:g,t:–','i 6,7:g,z51:1,5','i 6,7:g,z51:e,n,z15','iv 6,7:g,z51:–','i 6,7:m,p,t,u:–','i 6,7:m,t:1,6','i 6,7:m,t:z64','ii 6,7:m,t:–','i 6,7,14:m,t:z57','i 6,7,14:i:1,2','i 6,7:i:1,5','i 6,7:i:1,6','i 6,7:i:1,7','i 6,7:i:e,n,z15','i 6,7:i:l,w','i 6,7,14:i:z6','i 6,7,14:k:1,2','i 6,7,14:k:1,5:r1','i 6,7:k:1,6','i 6,7:k:1,7','i 6,7:k:e,n,x','i 6,7:k:e,n,z15','iiib 6,7:k:z:z55','ii 6,7:k:z6','i 6,7:k:z35','i 6,7:l,v:1,2','i 6,7:l,v:1,5','iiib 6,7:l,v:1,5,7','i 6,7:l,v:1,6','i 6,7:l,v:1,7','i 6,7:l,v:e,n,x','i 6,7,14:l,v:e,n,z15','i 6,7,14:l,v:z6','i 6,7:l,v:z35','iiib 6,7:l,v:z53','i 6,7:l,w:1,2','i 6,7:l,w:1,5','ii 6,7:l,w:1,5,7','i 6,7:l,w:e,n,z15','ii 6,7:l,w:z42','i 6,7:l,z13:1,5','i 6,7:l,z13:e,n,x','i 6,7:l,z13,z28:e,n,z15','i 6,7:l,z13,z28:z6','i 6,7:l,z13,z28:1,7','ii 6,7:l,z28:1,5:z42','ii 6,7:l,z28:e,n,x','ii 6,7:l,z28:z6','i 6,7,14:r:1,2','i 6,7,14:r:1,5:r1,z37,z45,z49','i 6,7:r:1,6','i 6,7:r:1,7','i 6,7:r:e,n,z15','i 6,7:r:l,w','i 6,7:y:1,2','i 6,7,14:y:1,5','i 6,7:y:1,6','i 6,7:y:1,7','i 6,7:y:e,n,x:z67','i 6,7,14:y:e,n,z15:z47,z50','i 6,7:z:1,2','i 6,7:z:1,5','ii 6,7:z:1,5','i 6,7:z:1,6,7','i 6,7:z:e,n,x','ii 6,7:z:e,n,x','ii 6,7:z:e,n,x,z15','i 6,7:z:e,n,z15','i 6,7:z:l,w','ii 6,7:z:z6','ii 6,7:z:z39','ii 6,7:z:z42','i 6,7:z4,z23:1,5','i 6,7:z4,z23:1,6','i 6,7:z4,z23:e,n,z15','i 6,7:z4,z23:z6','ii 6,7:z4,z23:–','iv 6,7:z4,z23:–','i 6,7:z4,z24:1,5','ii 6,7:z4,z24:z42','i 6,7:z4,z24:–','iv 6,7:z4,z24:–','ii 6,7:z6:1,7','i 6,7:z10:1,2','i 6,7:z10:1,5','i 6,7:z10:1,6','i 6,7:z10:1,7','i 6,7:z10:e,n,x','i 6,7,14:z10:e,n,z15:z37,z45','i 6,7,14:z10:l,w','i 6,7:z10:z6','i 6,7:z10:z35','i 6,7,14:z29:1,2,7','ii 6,7:z29:z42','i 6,7:z35:1,6','i 6,7:z35:e,n,z15','i 6,7:z36:e,n,z15','ii 6,7:z36:z42','iv 6,7:z36:–','i 6,7:z38:1,2:z82','ii 6,7:z38:e,n,x','i 6,7,14:z38:–:z82','iiib 6,7,14:z39:1,2','ii 6,7:z39:1,5,7','vi 6,7:z41:1,7','i 6,7:z41:l,w','i 6,7:z41:z35','ii 6,7:z42:1,5,7','i 6,7:z44:–','ii 6,7:–:1,6','i 8,20:a:z6','i 6,8:a:1,2','i 6,8:a:1,5','i 6,8:a:1,6','i 6,8:a:1,7','i 6,8:a:e,n,x','ii 6,8:a:e,n,x','i 6,8:a:e,n,z15','ii 6,8:a:z39','ii 6,8:a:z52','i 8:b:1,2','i 6,8:b:1,2','i 8,20:b:1,5','i 6,8:b:1,5','ii 6,8:b:1,5','i 6,8:b:1,6','i 8:b:1,7','i 6,8:b:1,7','i 8:b:e,n,x','i 6,8:b:e,n,x','i 8,20:b:e,n,z15','i 6,8:b:e,n,z15','i 6,8:b:l,w','i 8,20:b:l,w','i 8,20:b:z6','i 6,8:b:z6','i 6,8:c:1,2','i 8,20:c:1,5','i 6,8:c:1,5','i 6,8:c:1,6','i 6,8:c:1,7','i 8:c:l,w','i 8,20:c:e,n,x','i 6,8:c:e,n,x','i 6,8:c:e,n,z15','i 8,20:c:z6','i 8:d:1,2','i 6,8:d:1,2:z67','i 8,20:d:1,5','i 6,8:d:1,5:z58','i 8,20:d:1,7','i 6,8:d:1,7','i 6,8:d:e,n,x','i 6,8:d:e,n,z15','i 8,20:d:z6','ii 6,8:d:z6:z42','i 8:e,h:1,2','i 6,8,20:e,h:1,2:z67,z78','i 8:e,h:1,5','i 6,8:e,h:1,5','i 6,8:e,h:1,6:r1','i 8,20:e,h:1,7','i 6,8:e,h:e,n,x','i 6,8:e,h:e,n,z15','i 8,20:e,h:z6','i 6,8:f,g:e,n,z15','ii 6,8:f,g,m,t:e,n,x','i 8,20:g,m,s:–','i 6,8:g,m,s:e,n,x','ii 6,8:g,m,t:1,7','i 8,20:g,m,t:–','i 8,20:g,s,t:–','i 6,8:g,s,t:–','i 8,20:m,t:–','ii 6,8:m,t:1,5','ii 6,8:m,t:e,n,x','i 6,8:m,t:–','i 6,8:i:1,2','i 8,20:i:1,5','i 6,8:i:1,5','i 6,8:i:1,6','i 6,8:i:1,7','i 6,8:i:e,n,x','i 6,8:i:e,n,z15','i 8,20:i:l,w','i 6,8:i:l,w','i 8,20:i:z6','i 6,8:k:1,2','i 8:k:1,5','i 6,8:k:1,5:z58','i 6,8:k:1,7','i 6,8:k:e,n,x','i 6,8:k:e,n,z15','iiib 8:k:z35','i 8:l,v:1,2','i 6,8:l,v:1,2','i 6,8:l,v:1,5','i 8:l,v:1,6','i 6,8:l,v:1,7','i 6,8:l,v:e,n,x','ii 6,8:l,v:e,n,x','i 6,8:l,v:e,n,z15','i 6,8:l,v:z6','i 6,8:l,w:1,2','ii 6,8:l,w:z6:z42','i 6,8:l,z13,z28:1,5','i 6,8:l,z13,z28:e,n,z15','ii 6,8:l,z28:e,n,x','i 6,8:r:1,2','i 8,20:r:1,5','i 6,8,20:r,i:1,5:r1','i 8,20:r:1,6','i 8:r:1,7','i 6,8:r:1,7','i 8,20:r,i:e,n,z15','i 6,8:r,i:e,n,z15','i 8,20:r,i:l,w','i 6,8:r:l,w','iiib 8:r:z','i 8,20:r,i:z6','i 8,20:y:1,2','i 6,8:y:1,2','i 8,20:y:1,5','i 6,8:y:1,5','i 6,8:y:1,6','ii 6,8:y:1,6:z42','i 8,20:y:1,7','i 6,8:y:1,7','i 8:y:e,n,x','i 6,8:y:e,n,x','i 6,8:y:e,n,z15','i 8,20:y:z6','i 6,8:y:l,w','i 8,20:y:l,w','i 6,8:z:1,5','ii 6,8:z:1,5','i 6,8:z:1,7','i 8:z:e,n,z15','i 6,8:z:e,n,z15','i 6,8:z:l,w','i 8,20:z:z6','i 8:z4,z23:1,7','i 6,8:z4,z23:1,7','i 6,8:z4,z23:e,n,x','i 6,8:z4,z23:e,n,z15','i 8,20:z4,z23:l,w','i 8,20:z4,z23:z6','i 8:z4,z24:e,n,x','i 8,20:z4,z24:–:z45','i 6,8:z4,z24:–','i 6,8:z4,z32:–','i 8,20:z10:1,2','i 6,8:z10:1,2','i 8,20:z10:1,5','i 6,8:z10:1,5','i 6,8:z10:1,7','i 8:z10:e,n,x','i 6,8:z10:e,n,x','i 8,20:z10:e,n,z15','i 6,8:z10:e,n,z15','i 8,20:z10:l,w','i 8,20:z10:z6','i 6,8:z10:z6','ii 6,8:z29:1,5','ii 6,8:z29:e,n,x:z42','i 8,20:z29:e,n,z15','i 6,8:z29:e,n,z15','ii 6,8:z29:e,n,x','i 8,20:z35:1,2','i 6,8:z35:1,2','i 8,20:z35:z6','i 8,20:z38:–','i 8,20:z41:z6','i 6,8:z60:1,2','i 1,9,12:a:1,5','i 1,9,12:a:1,5','ii 9,12:a:1,5','i 9,12:a:1,6','i 1,9,12:a:1,7','i 1,9,12:a:e,n,x','ii 1,9,12:a:e,n,x','i 1,9,12:a:e,n,z15','ii 9,12:a:z39','ii 1,9,12:a:z42','i 1,9,12:b:1,2','i 1,9,12:b:1,5','i 9,12:b:1,7','ii 1,9,12:b:e,n,x','i 9,12:b:e,n,z15','ii 1,9,12:b:z6','ii 1,9,12:b:z39','i 9,12:c:1,5','i 9,12:c:1,6','i 9,12:c:1,7','i 9,12:c:e,n,z15','i 9,12:c:z6','i 1,9,12:d:1,5','i 9,12:d:1,6','i 9,12:d:1,7','ii 1,9,12:d:e,n,x','i 1,9,12:d:e,n,x','i 9,12:d:e,n,z15','i 9,12:d:z6','i 1,9,12:d:z35','ii 9,12:d:z39','i 9,12,vi:d:–:z66','i 9,12:e,h:1,2','i 1,9,12:e,h:1,5','i 9,12:e,h:1,7','i 9,12:e,h:e,n,z15','ii 9,12:e,n,x:1,5,7','ii 9,12:e,n,x:1,6','i 1,9,12:f,g,t:–','i 1,9,12:g,m:–','i 9,12:g,m,s:–','i 9,12:g,m,q:–','ii 1,9,12:g,m,s,t:1,5,7:z42','ii 1,9,12:g,m,s,t:e,n,x','ii 9,12:g,m,t:z39','i 1,9,12,vi:g,p:–','i 1,9,12:g,p,s:–','i 1,9,12:g,p,u:–','i 1,9,12:g,q:–','ii 9,12:g,s,t:e,n,x','i 9,12:g,z51:1,5','ii 1,9,12:g,z62:e,n,x','i 9,12:g,z63:–','i 9,12:g,z85:–','ii 9,12:m,t:e,n,x','i 1,9,12:m,t:1,2','ii 1,9,12:m,t:1,5','ii 1,9,12:m,t:z39','i 9,12:i:1,5','i 1,9,12:k:1,5','i 9,12:k:1,6','i 9,12:l,v:1,2','i 1,9,12:l,v:1,5:r1','i 9,12:l,v:1,5:d','i 9,12:l,v:1,7:z40','i 9,12:l,v:e,n,x','ii 9,12:l,v:e,n,x','i 9,12:l,v:e,n,z15','ii 9,12:l,v:z39','i 1,9,12:l,w:1,5','ii 1,9,12:l,w:e,n,x','i 9,12:l,z13:1,5','i 9,12:l,z13:1,7','i 1,9,12:l,z13:e,n,x','i 1,9,12:l,z28:1,5:r1','i 9,12:l,z28:1,6','ii 9,12:l,z28:1,5:z42','ii 9,12:l,z28:e,n,x','i 9,12:l,z28:e,n,z15','i 9,12:r:1,5','i 9,12:r:1,7','i 9,12:r:e,n,x','i 9,12:r:e,n,z15','i 9,12:r:z6','i 9,12:y:1,7','ii 1,9,12:y:z39','i 1,9,12:z:1,2','i 1,9,12:z:1,5','i 9,12:z:1,6','ii 1,9,12:z:1,7','i 9,12:z:e,n,z15','ii 1,9,12:z:z6','ii 9,12:z:z39','i 1,9,12:z4,z23:1,7','i 9,12:z4,z24:–','i 9,12:z6:z67:r1','i 9,12:z10:1,5','i 9,12:z10:z6','i 9,12:z10:e,n,z15','ii 9,12:z29:1,5','ii 1,9,12:z29:e,n,x','i 9,12:z35:z6','i 1,9,12:z38:–','ii 1,9,12:z39:1,7','i 1,9,12:z41:1,5','ii 1,9,12:z42:1,5,7','i 1,9,12:–:–','i 9,46:a:1,2','i 9,46:a:1,6','i 9,46:a:e,n,x','i 9,46:a:e,n,z15','i 9,46:b:1,2','i 9,46:b:1,5','i 9,46:b:1,6','i 9,46:b:e,n,x','ii 9,46:b:e,n,x','i 9,46:b:l,w','i 9,46:b:z6','i 9,46:b:z35','i 1,9,46:c:1,7','i 9,46:c:z6','i 9,46:d:1,5','i 9,46:d:1,6','i 9,46:d:1,7','i 9,46:d:e,n,z15','i 9,46:d:z6','i 9,46:d:z35','i 9,46:e,h:1,2','i 9,46:e,h:1,5','i 9,46:e,h:z6','ii 9,46:e,n,x:1,5,7','i 9,46:f,g:–','i 9,46:g,m:–','i 9,46:g,m,s:1,2,7','ii 9,46:g,m,s,t:e,n,x','i 9,46:g,s,t:–','ii 9,46:g,z62:–','ii 9,46:m,t:e,n,x','i 9,46:m,t:–','i 9,46:i:e,n,z15','i 9,46:i:z6','i 9,46:k:1,2','i 9,46:k:1,5','i 9,46:k:1,6','i 9,46:k:z35','i 9,46:l,v:1,5','i 9,46:l,v:1,6','i 9,46:l,v:e,n,x','i 9,46:l,w:–','ii 9,46:l,w:e,n,x','i 9,46:l,z13,z28:e,n,z15','i 9,46:r:1,7','i 9,46:r:e,n,z15','i 9,46:r:z6','i 9,46:y:1,2','i 9,46:y:1,7','i 9,46:y:e,n,x','i 9,46:y:l,w','i 9,46:y:z6','ii 9,46:z:1,5','ii 9,46:z:e,n,x','i 9,46:z:e,n,z15','ii 9,46:z:z39','i 9,46:z4,z23:–','ii 9,46:z4,z24:z39:z42','i 9,46:z4,z24:–','i 9,46:z10:1,7','i 9,46:z10:e,n,x','i 9,46:z10:e,n,z15','i 9,46:z10:z6','ii 9,46:z10:z6','ii 9,46:z10:z39','i 9,46:z29:–:z45','i 9,46:z35:1,5','i 9,46:z35:e,n,z15','i 9,46:z35:z6','i 9,46:z38:–','ii 9,46:z39:1,7','i 9,46:z41:–','ii 1,9,12,46,27:a:z6','ii 1,9,12,46,27:b:z39','ii 1,9,12,46,27:c:z39','ii 9,12,46,27:g,t:e,n,x','ii 1,9,12,46,27:l,z13,z28:z39','ii 1,9,12,46,27:y:z39','ii 1,9,12,46,27:z4,z24:1,5','ii 1,9,12,46,27:z10:1,5','ii 1,9,12,46,27:z10:e,n,x','ii 1,9,12,46,27:z10:z39','i 3,10:a:1,2','i 3,10,15:a:1,5','i 3,10,15,15,34:a:1,7','i 3,10:a:e,n,x','ii 3,10:a:e,n,x','i 3,10:a:e,n,z15','ii 3,10:a:l,v','ii 3,10:a:z39','ii 3,10:a:z87','i 3,10:b:1,2','i 3,10,15,15,34:b:1,5','i 3,10:b:1,6','i 3,10,15,34:b:1,7','i 3,10:b:e,n,x','ii 3,10:b:e,n,x','i 3,10,15:b:e,n,z15','i 3,10:b:l,w','i 3,10:b:z6','i 3,10,15:b:z35','ii 3,10:b:z39','i 3,10:c:1,2','i 3,10,15:c:1,5','i 3,10,15:c:1,6','i 3,10:c:1,7','i 3,10:c:e,n,z15','i 3,10:c:l,w','i 3,10:c:z6','i 3,10:d:1,2','i 3,10,15:d:1,5','i 3,10:d:1,6','i 3,10:d:1,7','i 3,10,15:d:e,n,x','ii 3,10:d:e,n,x','i 3,10:d:e,n,z15','i 3,10,15:d:l,w','i 3,10:d:z6','i 3,10:d:z35','i 3,10,15:e,h:1,2:z27','i 3,10,15,15,34:e,h:1,5:z48','i 3,10,15,15,34:e,h:1,6:z64','i 3,10,15:e,h:1,7','i 3,10,15,34:e,h:e,n,x','i 3,10:e,h:e,n,z15','i 3,10,15,15,34:e,h:l,w','i 3,10:e,h:z6','ii 3,10:e,n,x:1,7','i 3,10:f,g,s:1,6','i 3,10:f,g:e,n,x','i 3,10:g,m:–','i 3,10,15,15,34:g,m,s:–','ii 3,10,15:g,m,s,t:1,5','i 3,10,15,15,34:g,s,t:–:z37','i 3,10:g,t:1,5','ii 3,10:g,t:–','ii 3,10:m,t:1,5','i 3,10,15,15,34:m,t:1,6','ii 3,10:m,t:e,n,x','i 3,10:i:1,2','i 3,10:i:1,5','i 3,10:i:1,6','i 3,10:i:1,7','i 3,10:i:e,n,x','i 3,10,15:i:e,n,z15','i 3,10:i:l,w','i 3,10:i:z6','i 3,10:k:1,2','i 3,10,15:k:1,5','i 3,10:k:1,7','i 3,10:k:e,n,x','i 3,10:k:e,n,z15','i 3,10:k:l,w','i 3,10,15:l,v:1,2','i 3,10:l,v:1,5','i 3,10,15:l,v:1,6','i 3,10,15,15,34:l,v:1,7:d,z77','ii 3,10:l,v:e,n,x','i 3,10:l,v:e,n,z15','ii 3,10:l,v:z6','i 3,10:l,v:z35','i 3,10:l,w:z6:z45','i 3,10:l,z13:1,2','i 3,10,15:l,z13:1,5','i 3,10:l,z13,z28:e,n,z15','i 3,10:l,z13,z28:z6','ii 3,10:l,z28:1,5','i 3,10:l,z28:1,5','i 3,10:l,z28:1,7','i 3,10:l,z28:e,n,x','ii 3,10:l,z28:e,n,x','ii 3,10:l,z28:z39','i 3,10:r:1,5','i 3,10,15:r:1,7','i 3,10:r:e,n,z15','i 3,10,15:r:z6','i 3,10:r,i:1,2','i 3,10:r,i:1,6','i 3,10,15:y:1,2:z45','i 3,10,15,15,34:y:1,5','i 3,10:y:1,7','i 3,10,15:y:e,n,x','i 3,10:y:e,n,z15','i 3,10:y:l,w','i 3,10,15:y:z6','i 3,10:z:1,5','ii 3,10:z:1,5','i 3,10:z:1,6','i 3,10:z:1,7','ii 3,10:z:e,n,x','i 3,10:z:e,n,z15','i 3,10:z:l,w','i 3,10:z:z6','ii 3,10:z:z39','i 3,10:z4,z23:1,7','i 3,10:z4,z23:z6','i 3,10,15:z4,z24:–','ii 3,10:z4,z24:–','i 3,10:z10:1,2','i 3,10,15,15,34:z10:1,5:z49','i 3,10,15,15,34:z10:1,6','i 3,10:z10:1,7','i 3,10:z10:e,n,x','i 3,10:z10:e,n,z15','i 3,10:z10:z6','i 3,10:z29:e,n,x','ii 3,10:z29:e,n,x','i 3,10,15:z29:–','i 3,10:z35:1,5','i 3,10:z35:1,6','i 3,10:z35:1,7','ii 3,10:z35:e,n,x,z15','i 3,10:z35:l,w','i 3,10:z35:z6','i 3,10:z36:–','i 3,10:z38:1,2','i 3,10:z38:1,6','i 3,10:z38:z6','ii 3,10:z38:z42','ii 3,10:z39:1,5,7','i 3,10:z41:1,2,5','i 3,10,15,34:z69:1,7','ii 3,15:z87:e,n,x,z15','i 3,15,34:y:1,6','i 1,3,19:a:1,5','i 1,3,19:a:1,7','i 1,3,19:a:e,n,z15','i 1,3,19:a:l,w','i 1,3,19:b:1,5:z37','i 1,3,19:b:1,6','i 1,3,19:b:e,n,x','i 1,3,19:b:e,n,z15','i 1,3,19:b:l,w','i 1,3,19:b:z6','i 1,3,19:c:1,5','i 1,3,19:c:1,6','i 1,3,19:c:1,7','i 1,3,19:d:1,2','i 1,3,19:d:1,5','i 1,3,19:d:1,7','i 1,3,19:d:e,n,z15','i 1,3,19:d:l,w:z49','i 1,3,19:d:z6','i 1,3,19:e,h:1,5','i 1,3,19:e,h:1,6','i 1,3,19:e,h:1,7','i 1,3,19:e,h:e,n,z15','i 1,3,19:e,h:l,w','i 1,3,19:f,g:–','i 1,3,19:f,g,t:1,7','i 1,3,19:f,g,t:e,n,z15','i 1,3,19:g,m,t:–','i 1,3,19:g,s,t:–:z27,z34,z37,z43,z45,z46,z82','i 1,3,19:m,t:–','i 1,3,19:i:1,2','i 1,3,19:i:1,5','i 1,3,19:i:1,6','i 1,3,19:i:e,n,x','i 1,3,19:i:e,n,z15','i 1,3,19:i:l,w','i 1,3,19:i:z6','i 1,3,19:k:1,5','i 1,3,19:k:1,7','i 1,3,19:l,v:1,5','i 1,3,19:l,v:1,7','i 1,3,19:l,v:e,n,z15','i 1,3,19:l,w:1,5','i 1,3,19:l,z13:1,2','i 1,3,19:l,z13:1,6','i 1,3,19:l,z13,z28:1,2','i 1,3,19:l,z13,z28:1,5','i 1,3,19:l,z13,z28:e,n,z15','i 1,3,19:l,z28:1,7','i 1,3,19:l,z28:e,n,x','i 1,3,19:r:e,n,z15','i 1,3,19:r,i:l,w','i 1,3,19:y:1,5','i 1,3,19:y:1,7','i 1,3,19:y:e,n,z15','i 1,3,19:y:e,n,x','i 1,3,19:y:l,w','i 1,3,19:z:1,5','i 1,3,19:z:1,6','i 1,3,19:z:1,7','i 1,3,19:z:e,n,z15','i 1,3,19:z:l,w','i 1,3,19:z:z6','i 1,3,19:z4,z23:–','i 1,3,19:z4,z24:–','i 1,3,19:z10:1,5','i 1,3,19:z10:e,n,z15','i 1,3,19:z29:z6:z37','i 1,3,19:z35:1,5','i 1,3,19:z38:e,n,z15','i 1,3,19:z41:e,n,z15','i 1,3,10,19:b:z35','i 1,3,10,19:f,g,t:1,2,7','i 1,3,10,19:z4,z23:z6','i 1,3,15,19:g,s,t:–','i 1,3,10,15,19:y:e,n,x','i 11:a:1,2','i 11:a:1,5','vi 11:a:1,5','i 11:a:1,6','i 11:a:1,7','i 11:a:e,n,z15','ii 11:a:e,n,z15:d','i 11:a:l,z13,z28','ii 11:a:z6:z42','i 11:b:1,2','i 11:b:1,5','i 11:b:1,6','vi 11:b:1,7','vi 11:b:e,n,x','i 11:b:e,n,z15','i 11:b:z6','i 11:c:1,5','i 11:c:1,6','ii 11:c:e,n,z15','i 11:c:e,n,x','i 11:d:1,2','i 11:d:1,5','i 11:d:e,n,x:r','i 11:d:z6','i 11:e,h:1,2','i 11:e,h:1,6','i 11:e,h:l,z13,z28','i 11:g,m:1,5','i 11:g,s,t:–','ii 11:g,m,s,t:z39','iv 11:g,z51:–','i 11:m,t:–','ii 11:m,t:e,n,x','i 11:i:1,2','i 11:i:1,5','i 11:i:1,6','i 11:i:e,n,x','iiib 11:i:z','i 11:k:1,2','i 11:k:1,5','i 11:k:1,6','i 11:k:1,7','i 11:k:e,n,x,z15','i 11:k:l,w','i 11:k:l,z13,z28','iiib 11:k:z53','i 11:l,v:1,2','i 11:l,v:1,5','i 11:l,v:e,n,x','i 11:l,v:e,n,z15','iiib 11:l,v:z:z56','iiib 11:l,v:z53','i 11:l,w:1,5','i 11:l,z13:1,2','i 11:l,z13,z28:1,5','i 11:l,z13,z28:e,n,x','i 11:l,z13,z28:e,n,z15','ii 11:l,z28:e,n,x','i 11:r:1,5','i 11:r:e,n,x','i 11:r:e,n,z15','i 11:r,i:e,n,x,z15','i 11:r:l,z13,z28','i 11:y:1,5','i 11:y:1,7','i 11:y:e,n,x','i 11:y:e,n,x,z15','i 11:z:1,5','i 11:z:1,7','ii 11:z:e,n,x','i 11:z:e,n,z15','i 11:z:z6:z83','ii 11:z:z39','i 11:z4,z23:1,6','i 11:z4,z23:e,n,z15','iiia 11:z4,z23:–','iv 11:z4,z23:–','i 11:z4,z24:–','iv 11:z4,z32:–','i 11:z10:1,2','i 11:z10:1,5','i 11:z10:e,n,x','i 11:z38:–','i 11:z41:1,2','ii 11:z41:e,n,x','ii 11:–:1,5','i 1,13,23:a:1,5','ii 1,13,23:a:1,5','i 13,22:a:1,6','ii 13,22:a:e,n,x','i 1,13,23:a:l,w','i 13,22:a:l,z13,z28','ii 1,13,23:a:z42','i 13,22:b:1,5','i 1,13,23:b:1,5','i 13,22:b:1,6','ii 1,13,23:b:1,5:z42','i 13,23:b:1,6','i 1,13,22:b:1,7','i 13,22:b:e,n,x','i 1,13,23:b:e,n,x','i 13,22:b:e,n,z15','i 13,23:b:e,n,z15','i 13,23:b:l,w','ii 1,13,22:b:z42','i 13,22:c:e,n,x,z15','i 13,23:c:e,n,z15','i 1,13,23:d:1,2','ii 13,22:d:1,5','i 1,13,23:d:1,5','i 13,22:d:1,6','i 1,13,23:d:1,6:z37','i 1,13,23:d:1,7','ii 13,23:d:e,n,x','i 13,23:d:e,n,x','i 1,13,22:d:e,n,z15','i 13,23:d:e,n,z15','i 1,13,23:d:l,w','i 13,23:d:z6','i 1,13,23:e,h:1,5','i 1,13,22:e,h:1,6','i 1,13,23:e,h:l,w','ii 1,13,23:e,n,x:1,5,7','i 13,22:f,g:e,n,x','i 1,13,23:f,g,s:–:z79','i 13,22:g,m:e,n,z15','iiib 13,22:g,m,s:z','i 1,13,23:g,m,s,t:–','ii 1,13,22:g,m,t:1,5','ii 13,22:g,m,t:z42','ii 1,13,23:g,m,s,t:1,5','ii 1,13,23:g,m,s,t:e,n,x','ii 1,13,23:g,m,s,t:z42','i 13,22:g,s,t:–','ii 13,23:g,s,t:e,n,x','i 13,23:g,s,t:–','ii 1,13,22:g,t:1,5','ii 13,22:g,t:z6','ii 1,13,23:g,t:1,5','ii 13,23:g,t:e,n,x','ii 1,13,23:g,s,t:z42','iiia 1,13,23:g,z51:–','i 13,22:m,t:–','ii 1,13,23:m,t:1,5','ii 1,13,23:m,t:e,n,x','ii 13,22:m,t:z42:z39','ii 1,13,23:m,t:z42','i 1,13,23:m,t:–','v 1,13,22:i:–','i 1,13,23:i:1,5','i 13,23:i:1,7','i 13,23:i:e,n,z15','i 1,13,23:i:l,w','i 13,23:i:z6','ii 13,22:k:1,5:z42','i 13,23:k:1,6','i 13,23:k:–','i 13,23:k:e,n,z15','ii 13,23:k:z41','i 13,22:l,v:1,5','iiib 13,22:l,v:1,5,7','i 13,22:l,v:1,6','i 1,13,23:l,v:e,n,z15','ii 13,23:l,w:e,n,x','i 13,22:l,z13:e,n,z15','ii 13,22:l,z28:1,5','ii 13,23:l,z28:1,5','ii 13,23:l,z28:z6','ii 1,13,23:l,z28:z42','v 13,22:r:–','i 13,23:r:1,6','i 13,23:r:e,n,z15','i 1,13,22:y:1,6','i 13,23:y:1,7','i 1,13,23:y:l,w','i 1,13,23:y:z6','i 13,22:z:1,5','ii 1,13,23:z:1,5','i 1,13,23:z:1,5','iiib 13,23:z:1,5','i 1,13,22:z:1,6:z44,z59','i 13,23:z:1,6','i 13,22:z:1,7','i 13,22:z:z6','i 1,13,23:z:z6','i 1,13,22:z:e,n,z15','i 1,13,23:z:l,w:z43','ii 1,13,23:z:z42','ii 13,22:z:–','i 1,13,22:z4,z23:e,n,z15','iiia 13,22:z4,z23:–','i 13,23:z4,z23:–','iiia 13,23:z4,z23,z32:–','i 1,13,23:z4,z24:–','iiia 13,23:z4,z24:–','i 1,13,22:z10:1,5','ii 1,13,22:z10:z6','i 13,23:z10:e,n,z15','i 13,23:z10:l,w','ii 13,22:z29:1,5','ii 13,22:z29:e,n,x','ii 1,13,23:z29:1,5','ii 1,13,23:z29:e,n,x','i 13,22:z29:–','i 1,13,23:z29:–:z37,z43','i 13,22:z35:1,6','i 13,22:z35:e,n,z15','i 13,23:z35:1,6','i 13,23:z35:e,n,z15','i 13,22:z38:–','i 13,23:z38:–','v 13,22:z39:–','ii 13,22:z39:1,7','ii 1,13,23:z39:1,5,7','i 13,23:z41:e,n,z15','ii 1,13,23:z42:1,5,7','ii 13,23:–:1,6','i 1,6,14,25:a:1,5','vi 1,6,14:a:1,5','vi 1,6,14,25:a:e,n,x','i 1,6,14,25:a:e,n,z15','i 1,6,14,25:b:1,2','i 1,6,14,25:b:1,5','i 1,6,14,25:b:1,7','iiib 6,14:b:e,n,x','i 1,6,14,25:c:1,5','i 1,6,14,25:c:1,6','i 1,6,14,25:c:e,n,x','i 1,6,14,25:c:l,w','i 6,14,24:d:1,5','i 1,6,14,25:d:1,5','i 1,6,14,25:d:1,6','i 6,14,24:d:1,7','i 1,6,14,25:d:1,7','i 6,14,24:d:e,n,x','i 1,6,14,25:d:e,n,x','i 1,6,14,25:d:e,n,z15','i 1,6,14,25:d:l,z13,z28','i 1,6,14,24:d:z6','i 6,14,24:e,h:1,5','i 1,6,14,25:e,h:1,5','i 1,6,14,25:e,h:1,6','i 1,6,14,25:e,h:1,7','v 6,14:e,n,z15:–','i 1,6,14,25:g,m:–','i 1,6,14,25:g,m,s:–','i 1,6,14,25:g,p:–','i 6,14:g,s,t:–','ii 1,6,14:m,t:1,5','ii 6,14:m,t:e,n,x','i 1,6,14,25:m,t:–','i 1,6,14,25:i:1,5','i 1,6,14,25:i:1,7','i 6,14,24:i:e,n,z15','i 1,6,14,25:i:e,n,z15','i 1,6,14,25:k:1,5','ii 6,14,24:k:1,6','ii 6,14:k:e,n,x','i 1,6,14,25:k:e,n,x','iiib 6,14:k:z','ii 1,6,14:k:z6:z42','iiib 6,14:k:z53','i 6,14,24:l,v:1,7','i 1,6,14,25:l,v:1,7','i 1,6,14,25:l,v:e,n,x','i 6,14:l,v:e,n,z15','iiib 6,14:l,v:z','iiib 6,14:l,v:z35','iiib 6,14:l,v:z53','vi 6,14:l,v:z88','i 6,14,25:l,z13,z28:1,6','i 1,6,14,25:l,z28:e,n,x','i 1,6,14,25:r,i:1,5','iiib 6,14:r:z','i 1,6,14,25:r,i:e,n,z15','i 6,14,24:y:1,7','i 1,6,14,25:y:1,7','i 1,6,14,25:y:e,n,x','i 1,6,14,25:y:e,n,z15','i 1,6,14,25:z:1,5','i 1,6,14,25:z:1,6','i 6,14,24:z:e,n,x','i 1,6,14,25:z:e,n,x','i 1,6,14,25:z:e,n,z15','i 6,14,24:z:l,z13,z28','i 1,6,14,25:z:l,z13,z28','i 6,14:z4,z23:1,5','i 1,6,14,25:z4,z23:e,n,z15','iv 6,14:z4,z23:–','i 6,14,24:z4,z24:–','i 1,6,14,25:z4,z24:–','i 1,6,14,25:z10:1,2','vi 1,6,14,25:z10:1,2,7','ii 1,6,14:z10:1,5','i 1,6,14,25:z10:1,6','iiib 6,14:z10:e,n,x,z15','iiib 6,14:z10:z:z56,z90','i 1,6,14,25:z10:z6','ii 1,6,14:z10:z6:z42','iiib 6,14:z10:z53','i 6,14:z36:1,5','ii 6,14:z36:–','i 1,6,14,25:z38:e,n,x','ii 1,6,14:z42:1,6','iiib 6,14:z52:e,n,x,z15','iiib 1,6,14,25:z52:z35','i 16:a:1,2','i 16:a:1,5','i 16:a:1,6','i 16:a:1,7','i 16:a:e,n,x','i 16:a:e,n,z15','i 16:a:z6','i 16:b:1,2','i 16:b:1,2,5','i 16:b:1,5','i 16:b:1,6','i 16:b:e,n,x','ii 16:b:e,n,x','i 16:b:e,n,z15','i 16:b:l,w','i 16:b:z6','ii 16:b:z39','ii 16:b:z42','i 16:c:1,5','i 16:c:1,6','i 16:c:e,n,x','i 16:c:e,n,z15','i 16:c:l,w','i 16:d:1,2','i 16:d:1,5','ii 16:d:1,5','i 16:d:1,6','i 16:d:1,7','i 16:d:e,n,x','ii 16:d:e,n,x','i 16:d:e,n,z15','i 16:d:l,w:z82','i 16:d:z6','i 16:e,h:1,2','i 16:e,h:1,5','i 16:e,h:1,6','i 16:e,h:e,n,x','i 16:e,h:l,w','i 16:e,h:z6','ii 16:e,n,x:1,5,7','ii 16:e,n,x:1,6:z42','i 16:f,g:–:z37','i 16:g,m,t:–','i 16:g,m,s:–','ii 16:g,m,s,t:1,5:z42','ii 16:g,m,s,t:e,n,x','i 16:g,s,t:–','ii 16:m,t:e,n,x','i 16:m,t:e,n,z15','ii 16:m,t:z42','i 16:m,t:–','i 16:i:1,5','i 16:i:1,6','i 16:i:1,7','i 16:i:e,n,z15','i 16:i:l,w','i 16:i:z6','iiib 16:i:z35','i 16:k:1,2','i 16:k:1,6','i 16:k:e,n,x','i 16:k:e,n,z15','iiib 16:k:e,n,x,z15','iiib 16:k:z','iiib 16:k:z35','iiib 16:k:z53','iiib 16:l,v:1,5,7','i 16:l,v:1,6:z45','i 16:l,v:1,7','i 16:l,v:e,n,x','i 16:l,v:e,n,z15','iiib 16:l,v:z:z61','i 16:l,v:z6','iiib 16:l,v:z35','iiib 16:l,v:z53','i 16:l,w:1,5','i 16:l,w:1,6','i 16:l,w:e,n,x','i 16:l,w:e,n,z15','i 16:l,w:z6','ii 16:l,w:z6','i 16:l,z13:e,n,z15','i 16:l,z13,z28:1,5','i 16:l,z13,z28:1,6','i 16:l,z13,z28:z6','i 16:l,z13,z28:e,n,x','ii 16:l,z28:z42','i 16:r,i:1,5','i 16:r:1,6','i 16:r:1,7','i 16:r:z6','i 16:r,i:e,n,x','i 16:r,i:e,n,z15','i 16:y:1,5','i 16:y:1,6','i 16:y:e,n,x','i 16:y:e,n,z15','i 16:z:1,5','i 16:z:1,6','i 16:z:1,7','i 16:z:e,n,x','ii 16:z:e,n,x','i 16:z:e,n,x,z15','i 16:z:z6','ii 16:z:z42','i 16:z4,z23:1,6','i 16:z4,z23:z6','ii 16:z4,z23:–','iv 16:z4,z23:–','i 16:z4,z24:–','ii 16:z4,z24:–','iv 16:z4,z24:–','iv 16:z4,z32:–','ii 16:z6:1,6','i 16:z10:1,5','iiib 16:z10:1,7','i 16:z10:1,6','iiib 16:z10:e,n,x,z15','i 16:z10:e,n,z15','i 16:z10:z6','i 16:z29:–','ii 16:z29:1,5','ii 16:z29:e,n,x','i 16:z35:1,5','i 16:z35:1,6','i 16:z35:e,n,z15','ii 16:z35:e,n,x','iv 16:z36:–','ii 16:z36:e,n,z15','i 16:z38:–','i 16:z39:1,6','ii 16:z42:1,5,7','ii 16:z42:1,6','iiib 16:z52:z35','i 17:a:1,2','i 17:a:1,5','i 17:a:e,n,x','i 17:b:1,2','i 17:b:1,5','i 17:b:e,n,x','ii 17:b:e,n,x,z15','i 17:b:e,n,z15','ii 17:b:z6','i 17:c:1,5','i 17:c:1,6','ii 17:c:z39','ii 17:d:–','i 17:d:1,5','i 17:d:e,n,z15','i 17:d:l,w','ii 17:d:z6','i 17:e,h:1,2','ii 17:e,n,x,z15:1,6','ii 17:e,n,x,z15:1,5,7','i 17:g,m:–','ii 17:g,m,s,t:–','i 17:g,s,t:–','ii 17:g,t:e,n,x,z15','ii 17:g,t:z39','iiia 17:g,z51:–','i 17:m,t:–','ii 17:m,t:–','i 17:i:1,7','iiib 17:i:z35','i 17:k:1,5','i 17:k:1,6','i 17:k:1,7','i 17:k:e,n,x','i 17:k:e,n,z15','iiib 17:k:z','ii 17:k:–','i 17:l,v:1,2','i 17:l,v:1,5','i 17:l,v:1,7','i 17:l,v:e,n,x','iiib 17:l,v:e,n,x,z15','iiib 17:l,v:z35','i 17:l,z28:e,n,x','i 17:r:1,2','iiib 17:r:z','ii 17:y:–','i 17:y:e,n,x','i 17:y:e,n,z15','i 17:y:l,w','i 17:z:1,2','i 17:z:1,5','i 17:z:1,6','ii 17:z:1,7','i 17:z:e,n,z15','ii 17:z:l,w:z42','iiia 17:z4,z23:–','iiia 17:z4,z23,z32:–','iiia 17:z4,z24:–','iiia 17:z4,z32:–','i 17:z10:e,n,x','iiib 17:z10:e,n,x,z15:z56','iiib 17:z10:z','ii 17:z10:–','i 17:z29:–','iiia 17:z29:–','iv 17:z29:–','i 17:z35:1,6','iiia 17:z36:–','iv 17:z36:–','i 6,14,18:a:e,n,z15','i 6,14,18:b:1,5','i 18:b:1,7','i 6,14,18:c:e,n,x','i 18:d:1,5','i 6,14,18:d:1,7','i 18:g,m:–','i 18:g,s,t:–','iiia 18:g,z51:–','iv 18:g,z51:–','ii 18:m,t:1,5','i 18:m,t:–','i 18:k:1,5','iiib 18:k:z53','iiib 18:k:z54','iiib 18:l,v:e,n,x,z15','i 18:l,v:e,n,z15','iiib 18:l,v:z:z50','iiib 18:l,v:z53','i 18:l,w:e,n,z15','i 18:r:1,5','iiib 18:r:z','i 18:y:1,7','ii 18:y:e,n,x,z15','i 18:z:–','i 6,14,18:z4,z23:1,5:z45,z82','i 18:z4,z23:z64','ii 18:z4,z23:–','iiia 18:z4,z23:–','i 6,14,18:z4,z24:–','ii 18:z4,z24:–','iiia 18:z4,z32:–','iiib 18:z10:e,n,x,z15','i 18:z10:1,5','i 18:z10:z6','ii 18:z10:z6','ii 18:z36:–','iiia 18:z36:–','iv 18:z36,z38:–','i 18:z38:–','i 18:z71:–','i 18:–:1,6','i 21:a:1,5','ii 21:b:1,5','i 21:b:1,6','i 21:b:e,n,x:z33,z49','i 21:c:1,6','i 21:c:e,n,x','ii 21:c:e,n,x','iiib 21:c:e,n,x,z15','i 21:d:1,5','i 21:d:e,n,x','i 21:d:z6','i 21:f,g:e,n,x','ii 21:g,m,s,t:–','iiia 21:g,z51:–','iv 21:g,z51:–','ii 21:m,t:–','iv 21:m,t:–','i 21:i:1,2','iiib 21:i:1,5,7','iiib 21:i:e,n,x,z15','i 21:k:1,6','iiib 21:k:e,n,x,z15','iiib 21:k:z','i 21:k:1,2,5','iiib 21:l,v:z','iiib 21:l,v:z57','i 21:l,w:–','i 21:l,z28:e,n,z15','i 21:r:–','iiib 21:r:z','i 21:y:1,2','i 21:y:e,n,x','ii 21:z:–','i 21:z4,z23:–','iiia 21:z4,z23:–','iv 21:z4,z23:–','ii 21:z4,z24:–','iiia 21:z4,z24:–','iv 21:z4,z32:–','iiib 21:z10:e,n,x,z15','iiib 21:z10:z','ii 21:z10:z6','iiib 21:z10:z53','iiia 21:z29:–','i 21:z35:e,n,z15','iiia 21:z36:–','iv 21:z36:–','iv 21:z38:–','iiib 21:z65:e,n,x,z15','i 28:a:1,5','i 28:a:1,6','i 28:a:1,7','i 28:a:e,n,x','ii 28:a:e,n,x','i 28:a:e,n,z15','i 28:a:z6','i 28:b:1,5','i 28:b:1,6','i 28:b:1,7','i 28:b:e,n,x','ii 28:b:e,n,x','i 28:b:e,n,z15','i 28:b:l,w','ii 28:b:z6','i 28:c:1,5','i 28:c:1,6','i 28:c:1,7','i 28:c:e,n,x','i 28:c:e,n,z15','i 28:c:l,w','i 28:c:z6','i 28:d:1,5','i 28:d:1,6','i 28:d:1,7','i 28:d:e,n,x','i 28:d:e,n,z15','i 28:d:l,w','i 28:d:z6','i 28:e,h:1,5','i 28:e,h:1,7','i 28:e,h:e,n,z15','ii 28:e,n,x:1,7','ii 28:e,n,z15:z87','i 28:f,g:–','i 28:g,m:1,6','i 28:g,m:e,n,z15','ii 28:g,m,s,t:1,5','i 28:g,m,s:e,n,z15','ii 28:g,m,t:e,n,x','ii 28:g,m,t:z39','ii 28:g,s,t:e,n,x','i 28:g,s,t:–','ii 28:m,t:e,n,x','i 28:m,t:e,n,z15','ii 28:m,t:z39','i 28:m,t:1,6','i 28:i:1,2','i 28:i:1,5','i 28:i:1,6','i 28:i:1,7','i 28:i:e,n,x','i 28:i:e,n,z15','i 28:i:l,w','i 28:k:1,2','i 28:k:1,5','i 28:k:1,6','i 28:k:1,7','iiib 28:k:1,7','i 28:k:e,n,x','i 28:k:e,n,z15','i 28:l,v:1,5','i 28:l,v:e,n,x','i 28:l,v:e,n,z15','i 28:l,w:1,6','i 28:l,z13,z28:1,5','i 28:l,z13,z28:e,n,z15','ii 28:l,z28:1,5','i 28:l,z28:e,n,x','ii 28:l,z28:e,n,x','i 28:r:1,6','i 28:r:e,n,x','ii 28:r:e,n,z15','i 28:r:l,w','i 28:r,i:1,5','i 28:r,i:1,7','i 28:r,i:e,n,z15','i 28:y:1,2','i 28:y:1,5','i 28:y:1,7:z80,z90','i 28:y:e,n,x','i 28:y:e,n,z15','i 28:y:l,w','i 28:y:z6','i 28:z:1,5','ii 28:z:1,5','i 28:z:1,6','i 28:z:1,7','i 28:z:e,n,z15','ii 28:z:z39','i 28:z4,z23:1,5','i 28:z4,z23:1,6','i 28:z4,z23:e,n,z15','i 28:z4,z24:–','i 28:z10:1,2','i 28:z10:1,5','i 28:z10:1,6','i 28:z10:1,7','i 28:z10:e,n,x','i 28:z10:e,n,z15','i 28:z10:l,w','iiib 28:z10:z','i 28:z29:–','ii 28:z29:1,5','ii 28:z29:e,n,x','i 28:z35:1,6','i 28:z35:1,7','i 28:z35:e,n,z15','i 28:z38:e,n,z15','i 30:a:1,2','i 30:a:1,5','i 30:a:e,n,x','i 30:a:e,n,z15','ii 30:a:z39','i 30:b:1,2','i 30:b:1,5','i 30:b:1,7:z33','i 30:b:e,n,x','i 30:b:e,n,z15','i 30:b:l,z13,z28','ii 30:b:z6','i 30:b:z6','i 30:c:1,7','i 30:c:e,n,z15','ii 30:c:z39','i 30:d:1,5','i 30:e,h:1,2','i 30:e,h:1,5','i 30:g,m,t:–','ii 30:g,m,s:e,n,x','i 30:g,m,s:–','iv 30:g,m,t,z51:–','i 30:g,s,t:–:z59','ii 30:g,t:–','i 30:g,z51:–','ii 30:m,t:–','i 30:i:1,2','i 30:i:1,5','i 30:i:e,n,z15','i 30:i:l,w','i 30:k:1,2','i 30:k:1,5','i 30:k:1,6','i 30:k:e,n,x','i 30:k:e,n,x,z15','ii 30:k:e,n,x,z15','i 30:k:l,z13,z28','i 30:l,v:1,2','i 30:l,v:1,5','i 30:l,z13,z28:1,6','i 30:l,z13,z28:e,n,z15','ii 30:l,z28:z6','i 30:r:1,2','i 30:r:1,5','i 30:r:1,6','i 30:y:1,2','i 30:y:1,5','i 30:y:1,6','i 30:y:e,n,x','i 30:y:e,n,z15','i 30:z:e,n,x,z15','i 30:z4,z23:–','i 30:z4,z24:–','ii 30:z6:1,6','i 30:z10:1,2','i 30:z10:1,5','i 30:z10:e,n,z15','ii 30:z10:e,n,x,z15','i 30:z29:–','i 30:z35:1,6','i 30:z35:e,n,z15','i 30:z38:–','ii 30:z39:1,7','i 35:a:e,n,z15','i 35:b:–','i 35:c:1,2','i 35:c:1,5','i 35:c:e,n,z15','ii 35:d:1,5','i 35:d:l,w:z58','i 35:e,h:z6','i 35:f,g:–:z27','i 35:g,m,s:–','ii 35:g,m,s,t:–','i 35:g,m,t:–','i 35:g,s,t:–','ii 35:g,t:1,5','ii 35:g,t:z42','i 35:g,t:–','iiia 35:g,z51:–','ii 35:g,z62:e,n,x','i 35:m,t:–','ii 35:m,t:–','iiib 35:i:e,n,x,z15','i 35:i:e,n,z15','i 35:i:l,w','iiib 35:i:z','i 35:i:z6','iiib 35:i:z35','iiib 35:i:z53','i 35:k:e,n,z15','iiib 35:k:e,n,x,z15','iiib 35:k:z','iiib 35:k:z35','iiib 35:k:z53:z50','iiib 35:l,v:1,5,7','iiib 35:l,v:e,n,x,z15:z50','iiib 35:l,v:z35:z67','ii 35:l,z28:–','iiib 35:r:e,n,x,z15','i 35:r:l,w','iiib 35:r:z','iiib 35:r:z35','iiib 35:r:z61','i 35:y:l,w','i 35:z4,z23:–:z37,z45','iiia 35:z4,z23:–','i 35:z4,z24:–','iiia 35:z4,z24:–','iiia 35:z4,z32:–','i 35:z10:1,5','i 35:z10:l,w','i 35:z10:z6','iiib 35:z10:z35','ii 35:z29:e,n,x','i 35:z29:–','iiia 35:z29:–','iiia 35:z36:–','i 35:z38:–','iiib 35:z52:1,5,7','iiib 35:z52:e,n,x,z15','iiib 35:z52:z','iiib 35:z52:z35','i 38:a:e,n,z15','ii 38:b:1,2','i 38:b:e,n,z15','i 38:c:1,5','i 38:c:1,6:z58','i 38:d:1,5','ii 38:d:1,5','ii 38:d:z39','i 38:e,h:1,2','i 38:e,h:1,5','i 38:g,m,s:–','ii 38:g,t:–','iiia 38:g,z51:–','iv 38:g,z51:–','i 38:m,t:–','i 38:i:1,2','i 38:i:1,5','iiib 38:i:z','iiib 38:i:z53','i 38:k:1,2','i 38:k:1,5','i 38:k:1,6','i 38:k:e,n,x','iiib 38:k:e,n,x,z15','iiib 38:k:z','iiib 38:k:z53','iiib 38:k:1,5,7','iiib 38:k:z35:z56','iiib 38:k:–','iiib 38:k:z55','i 38:l,v:1,2','i 38:l,v:1,5','i 38:l,v:e,n,z15','i 38:l,v:e,n,x','iiib 38:l,v:z','iiib 38:l,v:z35','iiib 38:l,v:z53:z54','i 38:r:1,5','iiib 38:r:1,5,7','i 38:r:1,6','iiib 38:r:e,n,x,z15','iiib 38:r:z:z57','iiib 38:r:z35','i 38:y:1,5','i 38:y:1,6','i 38:y:e,n,x','i 38:z:–','i 38:z4,z23:1,2','iiia 38:z4,z23:–','iv 38:z4,z23:–','i 38:z4,z24:–','i 38:z10:1,5','iiib 38:z10:z','iiib 38:z10:z53','i 38:z35:e,n,z15','i 38:z38:–','iiib 38:z52:z35','iiib 38:z52:z53','iiib 38:z53:–:z47,z50,z76','iiib 38:z61:z53','ii 39:a:z39','i 39:b:1,2','i 39:b:l,w','ii 39:c:e,n,x','i 39:d:1,5','i 39:e,h:1,2','i 39:e,h:1,5','ii 39:e,n,x:1,7','i 39:m,t:–','ii 39:g,m,t:e,n,x','i 39:i:1,5','i 39:i:e,n,x','i 39:i:e,n,z15','i 39:k:1,5:z48','i 39:k:e,n,x','ii 39:l,v:1,5','i 39:l,v:e,n,x','i 39:l,v:e,n,z15','ii 39:l,z28:e,n,x','ii 39:l,z28:z39','i 39:r:1,5','i 39:y:1,2','i 39:y:1,5','i 39:y:e,n,z15','i 39:z4,z23:–','i 39:z10:–','ii 39:z10:z6','i 39:–:1,6','ii 39:–:1,7','i 40:a:1,5','i 40:a:1,7','i 40:a:z6','ii 40:a:z39','i 40:b:1,5','i 40:b:1,7','i 1,40:b:e,n,x','i 1,40:b:e,n,z15','i 40:b:z6','ii 40:b:–','ii 1,40:c:e,n,x,z15','ii 40:c:z6','ii 1,40:c:z39','i 1,40:d:1,5','ii 40:d:–','i 1,40:e,h:1,2','ii 1,40:e,n,x:1,5,7','ii 1,40:e,n,x,z15:1,6','i 1,40:g,m:–','i 1,40:g,m,s:e,n,x','ii 1,40:g,m,s,t:e,n,x','ii 1,40:g,m,s,t:1,5','ii 1,40:g,t:e,n,x,z15','ii 40:g,t:z39','iv 1,40:g,t:–','ii 1,40:g,m,s,t:z42','iiia 40:g,z51:–','iiib 40:g,z51:e,n,x,z15','iv 1,40:g,z51:–','ii 40:m,t:e,n,x','ii 40:m,t:z39','ii 1,40:m,t:z42','iv 40:m,t:–','iiib 40:i:1,5,7','i 1,40:k:1,5','i 1,40:k:1,6','i 40:k:e,n,x','ii 1,40:k:e,n,x,z15','iiib 40:k:z:z57','ii 40:k:z6','iiib 40:k:z53','i 1,40:l,v:1,2','i 40:l,v:1,6','ii 40:l,v:e,n,x','iiib 40:l,v:z','iiib 1,40:l,v:z53','i 1,40:l,w:1,2','ii 40:l,z28:e,n,x','i 1,40:l,z13,z28:1,2','i 1,40:l,z28:1,5','ii 1,40:l,z28:1,5:z42','i 40:l,z28:1,6','ii 1,40:l,z28:z39','iiib 40:r:z53','i 40:y:1,5','ii 1,40:z:1,5','i 40:z:e,n,x','i 40:z:z6','ii 1,40:z:z6','ii 1,40:z:z39','ii 40:z:z42','iiia 40:z4,z23:–','iv 1,40:z4,z23:–','ii 40:z4,z24:z39','iiia 40:z4,z24:–','iv 1,40:z4,z24:–','iiia 40:z4,z32:–','iv 1,40:z4,z32:–','ii 1,40:z6:1,5','i 40:z10:z6','ii 40:z10:e,n,x','iiib 40:z10:z35','i 1,40:z29:–','iiia 40:z29:–','ii 1,40:z35:e,n,x,z15','i 1,40:z35:e,n,z15','v 1,40:z35:–','iiia 40:z36:–','ii 1,40:z39:1,5:z42','ii 1,40:z39:1,6','iiib 40:z39:1,6','ii 40:z39:1,7','i 1,40:z41:1,2','ii 1,40:z42:1,6','ii 1,40:z42:1,5,7','ii 1,40:z81:z6','v 1,40:z81:–','i 41:a:–','ii 41:b:1,5','i 41:b:1,6','vi 41:b:1,7','i 41:b:z6','i 41:b:e,n,z15','i 41:c:–','iiib 41:c:e,n,x,z15','ii 41:c:z6','i 41:d:1,5','ii 41:d:z6','ii 41:g,m,s,t:z6','ii 41:g,t:–','iiia 41:g,z51:–','i 41:m,t:1,6','i 41:i:1,5','i 41:i:1,6','i 41:k:1,6','ii 41:k:1,6','ii 41:k:z6','iiib 41:k:z35','ii 41:l,z13,z28:e,n,x,z15','i 41:r:1,5','i 41:r:1,7','i 41:y:–','ii 41:z:1,5','i 41:z:1,6','i 41:z4,z23:1,7','i 41:z4,z23:e,n,z15','iiia 41:z4,z23:–','iv 41:z4,z23:–','iiia 41:z4,z23,z32:–','i 41:z4,z24:1,5','iiia 41:z4,z24:–','iiia 41:z4,z32:–','ii 41:z10:1,2','i 41:z10:1,5','i 41:z10:1,6','i 41:z10:e,n,x','ii 41:z10:e,n,x,z15','ii 41:z10:z6','i 41:z29:–','iiia 41:z29:–','iv 41:z29:–','i 41:z35:1,6','iiia 41:z36:–','iv 41:z36:–','i 41:z38:–','iv 41:z52:–','ii 41:–:1,6','i 1,42:a:e,n,z15','ii 42:b:1,5','i 42:b:1,6','ii 42:b:e,n,x,z15','i 1,42:b:e,n,z15','i 1,42:b:l,w','i 1,42:b:z6','ii 42:b:z6','i 1,42:c:e,n,z15','i 1,42:c:z6','i 1,42:d:z6','ii 42:d:z6','ii 42:e,n,x:1,6','ii 42:g,t:–','i 1,42:g,z51:1,5','iiia 42:g,z51:–','iv 1,42:g,z51:–','ii 42:m,t:e,n,x,z15','i 1,42:m,t:–','i 1,42:i:l,w','i 42:i:1,6','i 1,42:i:z6','i 42:k:1,6','iiib 42:k:e,n,x,z15','iiib 42:k:z','i 1,42:k:z6','iiib 42:k:z35','iiib 42:l,v:1,5,7:z76','i 1,42:l,v:1,6,7','ii 42:l,v:e,n,x,z15','iiib 42:l,v:e,n,x,z15','i 42:l,v:e,n,z15','iiib 42:l,v:z','iiib 42:l,v:z53','ii 1,42:l,w:e,n,x','i 1,42:l,w:z35','ii 1,42:l,z13,z28:z6','i 1,42:r:e,n,z15','i 1,42:r:l,w','iiib 42:r:z','iiib 42:r:z53','ii 42:r:–','iiib 42:r:–:z50','i 1,42:y:e,n,z15','i 1,42:y:z6','ii 42:z:1,5','i 1,42:z:1,6','ii 42:z:e,n,x,z15','i 42:z:e,n,z15','ii 42:z:z6','i 1,42:z4,z23:1,6:1,6','i 42:z4,z23:e,n,z15','iiia 42:z4,z23:–','i 1,42:z4,z24:–','iiia 42:z4,z24:–','iv 1,42:z4,z24:–','ii 42:z6:1,6','ii 42:z10:1,2','ii 42:z10:e,n,x,z15','iiib 42:z10:e,n,x,z15:z60','iiib 42:z10:z','i 1,42:z10:z6','ii 42:z10:z6','iiib 42:z10:z35','iiib 42:z10:z67','i 1,42:z29:1,5','ii 42:z29:–','i 1,42:z35:1,6','i 42:z35:e,n,z15','i 1,42:z35:z6','i 42:z36:–','iv 42:z36:–','i 1,42:z38:z6','i 1,42:z41:–','iiib 42:z52:z','iiib 42:z57:1,5','i 43:a:1,2','i 43:a:1,5','ii 43:a:1,5','ii 43:a:z6','i 43:b:–','i 43:b:e,n,z15','ii 43:b:z42','i 43:c:1,5','i 43:d:1,5','ii 43:d:e,n,x,z15','ii 43:d:z39','ii 43:d:z42','ii 43:e,n,x,z15:1,5,7','ii 43:e,n,x,z15:1,6','i 43:f,g,t:–','ii 43:g,m,s,t:z42','ii 43:g,t:1,5','iiib 43:g,t:–','iiia 43:g,z51:–','iv 43:g,z51:–','ii 43:g,z62:e,n,x','i 43:i:1,2','i 43:i:e,n,x','i 43:k:1,2','i 43:k:1,5','iiib 43:k:z','iiib 43:l,v:z53:z56','i 43:l,w:z44','i 43:l,z13:–','ii 43:l,z13,z28:1,5','ii 43:m,t:–','iiib 43:r:e,n,x,z15','iiib 43:r:z','iiib 43:r:z53','i 43:y:1,2','i 43:y:1,5','i 43:z:1,5','ii 43:z:1,5','i 43:z:e,n,z15','i 43:z4,z23:1,2','ii 43:z4,z23:–','iiia 43:z4,z23:–','iv 43:z4,z23:–','iiia 43:z4,z24:–','iv 43:z4,z24:–','iv 43:z4,z32:–','i 43:z10:1,5','ii 43:z29:e,n,x','ii 43:z29:z42','i 43:z29:–','iiia 43:z29:–','iv 43:z29:–','i 43:z35:1,6','iiia 43:z36:–','iv 43:z36,z38:–','i 43:z38:–','i 43:z41:1,2','ii 43:z42:1,5,7','iiib 43:z52:z53','iv 44:a:–','i 44:a:1,5','i 44:a:e,n,x','i 44:a:l,w','i 44:b:1,6','i 44:b:e,n,x','i 44:b:e,n,z15','i 44:c:1,5','i 44:c:e,n,x','i 44:c:e,n,z15','i 44:d:1,5','i 44:d:e,n,x','i 44:d:e,n,z15','i 1,44:d:z35','ii 1,44:e,n,x:1,6','i 44:f,g:–','i 1,44:g,m,s:1,6','i 44:g,s,t:1,7','ii 44:g,t:z42','iiib 1,44:g,t:1,5:z42','i 44:g,z51:–','iv 44:g,z51:–','i 44:m,t:–','ii 1,44:m,t:z42','i 1,44:i:e,n,z15','i 44:k:e,n,z15','i 44:l,z28:1,5','i 44:r:1,5','i 44:r:e,n,z15','v 44:r:–','i 44:z:1,5','i 44:z:l,w','i 44:z:e,n,x','i 44:z:e,n,z15','i 44:z4,z23:–','i 1,44:z4,z23:e,n,z15','ii 44:z4,z23:–','iiia 44:z4,z23:–','iv 44:z4,z23:–','iiia 44:z4,z23,z32:–','i 44:z4,z24:–','iiia 44:z4,z24:–','iv 44:z4,z24:–','iiia 44:z4,z32:–','iv 1,44:z4,z32:–','i 1,44:z10:1,7','i 1,44:z10:e,n,x','ii 44:z29:e,n,x:z42','i 44:z29:–','iv 44:z29:–','iv 44:z36,z38:–','i 44:z38:–','ii 1,44:z39:e,n,x,z15','v 44:z39:–','vi 45:a:e,n,x','i 45:a:e,n,z15','ii 45:a:z10','i 45:b:1,5','i 45:b:1,6','i 45:b:e,n,z15','i 45:c:e,n,x','i 45:d:1,6','i 45:d:e,n,x','i 45:d:e,n,z15','i 45:f,g:–','i 45:g,m,s,t:–','ii 45:g,m,s,t:1,5','ii 45:g,m,s,t:e,n,x','ii 45:g,m,t:e,n,x,z15','i 45:g,s,t:–','iiia 45:g,z51:–','iv 45:g,z51:–','ii 45:m,t:1,5','i 45:m,t:–','i 45:k:1,5','i 45:k:1,7','i 45:k:e,n,z15','i 45:l,v:e,n,z15','i 45:r:e,n,z15','ii 45:z:1,5','i 45:z:e,n,z15','ii 45:z:z39','iiia 45:z4,z23:–','iv 45:z4,z23:–','i 45:z4,z24:–','iiia 45:z4,z24:–','iiia 45:z4,z32:–','i 45:z10:–','i 45:z29:–:z45','ii 45:z29:1,5','ii 45:z29:e,n,x','ii 45:z29:z42','iiia 45:z29:–','i 45:z35:1,5','i 45:z36:–','iiia 45:z36:–','iv 45:z36,z38:–','ii 47:a:1,5','ii 47:a:e,n,x,z15','i 47:b:1,2','ii 47:b:1,5','ii 47:b:e,n,x,z15','i 47:b:z6','ii 47:b:z6','iiib 47:c:1,5,7','i 47:c:1,6:z78','iiib 47:c:e,n,x,z15:z57','iiib 47:c:z','i 47:c:z6','iiib 47:c:z35','ii 47:d:1,5','i 47:d:e,n,x:z58','ii 47:d:e,n,x,z15','ii 47:d:z39','ii 47:e,n,x,z15:1,6','i 1,47:f,g:–','i 1,47:g,m:–','i 47:g,m,s:–','ii 47:g,t:e,n,x','iiia 47:g,z51:–','i 47:m,t:e,n,z15','iiib 47:i:e,n,x,z15:z50','i 47:i:e,n,z15','iiib 47:i:z','iiib 47:i:z35','iiib 47:i:z53:z57,z84','i 47:k:1,2','i 47:k:1,5','iiib 47:k:1,5,7','i 47:k:1,6:z58','iiib 47:k:e,n,x,z15','i 47:k:e,n,z15','iiib 47:k:z','iiib 47:k:z35','iiib 47:k:z53:z84','iiib 47:l,v:1,5,7:z50','i 47:l,v:e,n,x','iiib 47:l,v:e,n,x,z15','iiib 47:l,v:z','iiib 47:l,v:z35','iiib 47:l,v:z53','iiib 47:l,v:z57','iv 47:l,v:–','i 1,47:l,z13,z28:e,n,z15','iiib 47:r:e,n,x,z15','i 47:r:1,2','iiib 47:r:1,5,7','iiib 47:r:z','iiib 47:r,i:z35','iiib 47:r:z53:z70,z74,z77,z90','i 47:y:1,6','i 47:y:e,n,x','i 47:z:1,5','i 47:z:1,6','ii 47:z:e,n,x,z15','ii 47:z:z6','i 47:z4,z23:1,2','i 47:z4,z23:e,n,z15','i 47:z4,z23:1,6','i 47:z4,z23:z6:z45,z58','i 47:z4,z23:l,w','iiia 47:z4,z23:–','i 47:z4,z24:–','ii 47:z6:1,6','iiib 47:z10:1,5,7','i 47:z10:e,n,z15','iiib 47:z10:z','iiib 47:z10:z35','ii 47:z29:e,n,x,z15','i 47:z29:–','iiia 47:z29:–','i 47:z35:z6','i 47:z35:e,n,z15','iv 47:z36:–','i 47:z38:–','i 47:z44:–','iiib 47:z52:1,5:z54','iiib 47:z52:1,5,7','iiib 47:z52:e,n,x,z15','iiib 47:z52:z','iiib 47:z52:z35','iiib 47:z53:–:z90','i 48:a:1,5,7','ii 48:a:z6','ii 48:a:z39','ii 48:b:z6','ii 48:b:e,n,x,z15','v 48:b:–','iiib 48:c:z','ii 48:d:1,2','ii 48:d:z6','i 48:d:–:z58','i 48:e,h:1,5','ii 48:e,n,x,z15:z6','ii 48:g,m,t:–','iiia 48:g,z51:–','iv 48:g,z51:–','iiib 48:i:z:z72','iiib 48:i:z35:z57','iiib 48:i:z53','iiib 48:i:z61','v 48:i:–','iiib 48:k:1,5,7','ii 48:k:e,n,x,z15','iiib 48:k:e,n,x,z15','i 48:k:e,n,z15','iiib 48:k:z','iiib 48:k:z35:z75','ii 48:k:z39','iiib 48:k:z53','iiib 48:k:z57','i 48:l,v:1,5','iiib 48:l,v:1,5,7:z47,z50,z89','iiib 48:l,v:z','iiib 48:l,w:1,5,7:z50','iiib 48:r:e,n,x,z15','iiib 48:r:z','i 48:z:1,5:z58','ii 48:z:1,5','iiib 48:z:1,5,7','iiia 48:z4,z23:–','iv 48:z4,z23:–','iiia 48:z4,z23,z32:–','i 48:z4,z24:–','iiia 48:z4,z24:–','iiib 48:z4,z24:–','iv 48:z4,z24:–','iiia 48:z4,z32:–','iv 48:z4,z32:–','ii 48:z10:1,5','vi 48:z10:1,5','ii 48:z10:1,6','i 48:z10:e,n,x','iiib 48:z10:e,n,x,z15','iiib 48:z10:z','ii 48:z29:–','iiia 48:z29:–','iv 48:z29:–','iiib 48:z35:z52','v 48:z35:–','iiia 48:z36:–','iv 48:z36,z38:–','ii 48:z39:z81','v 48:z39:–','v 48:z41:–','iiib 48:z52:e,n,x,z15','iiib 48:z52:z','v 48:z65:–','v 48:z81:–','ii 48:z81:1,5,7','iv 50:a:–','i 50:b:e,n,x','ii 50:b:z6','iv 50:b:–','i 50:d:1,5:z82','iv 50:d:–','ii 50:e,n,x:1,7','ii 50:g,m,s,t:1,5','iiia 50:g,z51:–','iv 50:g,z51:–','ii 50:g,z62:e,n,x','ii 50:m,t:z6:z42','iiib 50:i:1,5,7','iiib 50:i:e,n,x,z15','iiib 50:i:z','iiib 50:i:z53','iiib 50:k:1,5,7','ii 50:k:e,n,x:z42','iiib 50:k:e,n,x,z15','iiib 50:k:z:z50,z57,z68,z86','ii 50:k:z6','iiib 50:k:z35','iiib 50:k:z53','i 50:l,v:1,2','iiib 50:l,v:e,n,x,z15','iiib 50:l,v:z','iiib 50:l,v:z35','iiib 50:l,v:z57','vi 50:l,v:z67','ii 50:l,w:e,n,x,z15:z42','ii 50:l,z28:z42','iiib 50:r:1,5,7','iiib 50:r:e,n,x,z15','iiib 50:r:z:z67','iiib 50:r:z35:z58','iiib 50:r:z53','i 50:y:1,6','ii 50:z:e,n,x','iiib 50:z:z52','iiia 50:z4,z23:–','iv 50:z4,z23:–','iiia 50:z4,z23,z32:–','iiia 50:z4,z24:–','iv 50:z4,z24:–','iiia 50:z4,z32:–','iv 50:z4,z32:–','iiib 50:z10:z:z56','ii 50:z10:z6:z42','iiib 50:z10:z53','i 50:z29:–','iiia 50:z29:–','iiia 50:z36:–','ii 50:z42:1,7','iiib 50:z52:1,5,7','iiib 50:z52:z35','iiib 50:z52:z53','iv 51:a:–','i 51:a:1,2','i 51:a:e,n,x','i 51:b:1,5','iv 51:b:–','ii 51:c:–','i 1,51:d:1,5','i 51:e,h:1,2','ii 51:g,s,t:e,n,x','iiia 51:g,z51:–','i 51:m,t:–','i 51:i:1,2','i 51:i:1,5','i 51:k:e,n,z15','iiib 51:k:z35','i 51:l,v:1,2','i 51:l,v:1,5','i 51:l,v:e,n,x','iiib 51:l,v:z','i 51:l,z28:1,5','ii 51:l,z28:z6','ii 51:l,z28:z39','i 51:r:e,n,x','i 51:r,i:l,z13,z28','i 51:y:1,2','i 51:z:1,5','i 1,51:z:1,6','i 51:z:e,n,z15','iiia 51:z4,z23:–','iv 51:z4,z23:–','iiia 51:z4,z24:–','iiia 51:z4,z32:–','i 51:z10:1,5','ii 51:z29:e,n,x,z15','iiia 51:z36:–','ii 51:–:1,7','i 52:a:1,5','i 52:a:e,n,z15','i 52:b:1,5','i 52:b:e,n,x','ii 52:c:k','i 52:d:1,5','ii 52:d:e,n,x,z15','ii 52:d:z39','i 52:e,h:1,6','i 52:e,h:1,7','i 52:g,t:–','ii 52:g,t:–','i 52:k:1,5','iiib 52:k:e,n,x,z15','iiib 52:k:z35','iiib 52:k:z53','iiib 52:l,v:z53','i 52:l,w:1,5','ii 52:z:z39','iiib 52:z:z52','ii 52:z39:1,5,7','ii 52:z44:1,5,7','ii 53:c:1,5:r1','ii 53:d:1,5','ii 1,53:d:z39','ii 53:d:z42','iiia 53:g,z51:–','iv 1,53:g,z51:–','iiib 53:i:z','iiib 53:k:e,n,x,z15','iiib 53:k:z','iiib 53:k:z35','iiib 53:k:z53','iiib 53:l,v:e,n,x,z15','iiib 53:l,v:z','iiib 53:l,v:z35','ii 53:l,z28:e,n,x','ii 53:l,z28:z6','ii 53:l,z28:z39','iiib 53:r:z','iiib 53:r:z35','iiib 53:r:z68','ii 53:z:1,5','iiib 53:z:1,5,7','ii 53:z:z6','iiia 53:z4,z23:–','iv 53:z4,z23:–','iiia 53:z4,z23,z32:–','ii 53:z4,z24:–','iiia 53:z4,z24:–','iiib 53:z10:z','iiib 53:z10:z35','iiia 53:z29:–','iv 1,53:z36,z38:–','iiib 53:z52:z35','iiib 53:z52:z53','i 53:–:1,6','i 21,54:b:e,n,x','i 54:e,h:1,5','i 3,54:e,h:1,6','i 54:f,g,s:–','i 3,54:g,s,t:–','i 4,12,54:m,t:–','i 8,20,54:i:z6','i 6,7,54:k:1,5','i 54:r:1,5','i 3,15,54:y:1,5','i 54:z4,z23:–','i 54:z10:e,n,x','i 54:z10:e,n,z15','i 54:z29:–','ii 55:k:z39','ii 56:b:1,5','ii 56:d:–','ii 56:e,n,x:1,7','ii 56:l,v:z39','ii 56:l,z28:–','ii 56:z:z6','iiia 56:z4,z23:–','iiia 56:z4,z23,z32:–','ii 56:z10:e,n,x','iiia 56:z29:–','i 57:a:z6','ii 57:a:z42','i 57:b:1,7','i 57:b:e,n,z15','iiib 57:c:e,n,x,z15','iiib 57:c:z:z70,z90','ii 57:d:1,5','ii 57:g,m,s,t:z42','ii 57:g,t:–','iiib 57:i:e,n,x,z15','iiib 57:i:z','iiib 57:k:e,n,x,z15','iv 57:z4,z23:–','iiib 57:z10:z','ii 57:z29:z42','ii 57:z39:e,n,x,z15','ii 57:z42:1,6:z53','ii 58:a:z6','ii 58:b:1,5','ii 58:c:z6','ii 58:d:z6','iiib 58:i:e,n,x,z15','iiib 58:i:z53','iiib 58:k:z','iiib 58:l,v:e,n,x,z15','iiib 58:l,v:z35','ii 58:l,z13,z28:1,5','ii 58:l,z13,z28:z6','iiib 58:r:e,n,x,z15','iiib 58:r:z','iiib 58:r:z53:z47,z57,z70,z73','ii 58:z6:1,6','ii 58:z10:1,6','iiib 58:z10:e,n,x,z15','ii 58:z10:z6','iiib 58:z10:z53:z50','ii 58:z39:e,n,x,z15','iiib 58:z52:z','iiib 58:z52:z35','iiib 59:c:e,n,x,z15','iiib 59:i:e,n,x,z15','iiib 59:i:z','iiib 59:i:z35:z84','iiib 59:k:e,n,x,z15','ii 59:k:z65','iiib 59:k:z','iiib 59:k:z35','iiib 59:k:z53','iiib 59:l,v:z','iiib 59:l,v:z53','iiib 59:r:z35','ii 1,59:z:z6','iiia 59:z4,z23:–','iiib 59:z10:z53','iiib 59:z10:z57','iiia 59:z29:–','iiia 59:z36:–','vi 59:z36:–','iiib 59:z52:z53','ii 60:b:–:r1','ii 60:g,m,t:z6','iiib 60:i:e,n,x,z15:z50','iiib 60:i:z:z50','iiib 60:i:z35:z50','iiib 60:k:z','iiib 60:k:z35','iiib 60:k:z53','iiib 60:l,v:z','iiib 60:r:e,n,x,z15','iiib 60:r:z','iiib 60:r:z35','iiib 60:r:z53','ii 60:z:e,n,x','iiib 60:z10:z','iiib 60:z10:z35','ii 60:z10:z39','iiib 60:z10:z53','ii 60:z29:e,n,x','v 60:z41:–','iiib 60:z52:1,5,7','iiib 60:z52:z','iiib 60:z52:z35','iiib 60:z52:z53','iiib 61:c:1,5,7','iiib 61:c:z35','iiib 61:i:e,n,x,z15','iiib 61:i:z','iiib 61:i:z35','iiib 61:i:z53','iiib 61:k:1,5,7','iiib 61:k:z35','iiib 61:k:z53','iiib 61:l,v:1,5,7:z57','iiib 61:l,v:z','iiib 61:l,v:z35','iiib 61:r:1,5,7','iiib 61:r:z','iiib 61:r:z35','iiib 61:r:z53:z47,z50','iiib 61:z10:z35','v 61:z35:–','iiib 61:z52:1,5,7','iiib 61:z52:z','iiib 61:z52:z35','iiib 61:z52:z53','iiia 62:g,z51:–','iiia 62:z4,z23:–','iiia 62:z4,z32:–','iiia 62:z29:–','iiia 62:z36:–','iiia 63:g,z51:–','iiib 63:k:z','iiia 63:z4,z23:–','iiia 63:z4,z32:–','iiia 63:z36:–','iiib 65:c:1,5,7','iiib 65:c:z','iiib 65:c:z53','ii 65:g,t:–','iiib 65:i:e,n,x,z15','iiib 65:i:z','iiib 65:k:z','iiib 65:k:z35','iiib 65:k:z53','iiib 65:l,v:e,n,x,z15','iiib 65:l,v:z','iiib 65:l,v:z35','iiib 65:l,v:z53','iiib 65:r:z35','iiib 65:z10:e,n,x,z15','iiib 65:z10:z','iiib 65:z52:e,n,x,z15','iiib 65:z52:z','iiib 65:z52:z35','iiib 65:z52:z53','ii 65:–:1,6','v 66:z35:–','v 66:z39:–','v 66:z41:–','v 66:z65:–','v 66:z81:–','i 6,7:c:1,5','i 1,9,12:a:1,5','i 67:r:1,2','i 3,15:y:1,2:z45', 'i 6,7,14:d:e,n,x', 'i 3,15:g,m,s:–', 'i 3,15:e,h:1,6:z64', 'i 3,15,34:e,h:1,6:z64', 'i 3,15,34:b:1,5', 'i 6,14,18:z4,z23:1,5:z45,z82', 'i 6,7,14:l,v:z6', 'i 3,15:l,v:1,7:d,z77', 'i 3,15,34:l,v:1,7:d,z77', 'i 3,15:a:1,5', 'i 3,15:z10:1,5:z49', 'i 3,15,34:z10:1,5:z49', 'i 6,7,14:z38:–:z82', 'i 6,7,14:d:l,w', 'i 3,15:l,v:1,6', 'i 3,15:e,h:l,w', 'i 3,15,34:e,h:l,w', 'i 3,15:e,h:1,5:z48', 'i 3,15,34:e,h:1,5:z48', 'i 3,15:l,v:1,2', 'i 3,15:e,h:1,7', 'i 6,7,14:m,t:z57', 'i 3,15:y:1,5', 'i 3,15,34:y:1,5', 'i 3,15,34:a:1,7', 'i 1,4,5,12:b:1,2:z5,z33', 'i 6,7,14:f,g:–', 'i 3,15:d:1,5', 'i 3,15:d:e,n,x', 'i 1,4,5,12,27:z4,z23:1,2', 'i 3,15:y:z6', 'i 3,15:l,z13:1,5', 'i 3,15:e,h:1,2', 'i 3,15:e,h:1,2:z27', 'i 3,15:r:z6', 'i 3,15:g,s,t:–:z37', 'i 3,15,34:g,s,t:–:z37']
wklm_sp = ['enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','bongori','bongori','enterica','enterica','bongori','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','bongori','bongori','bongori','bongori','bongori','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica','enterica']
wklm_subsp = ['I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','II','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','II','II','I','I','II','I','I','II','II','II','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','I','I','I','I','I','II','I','II','I','I','II','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','II','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','II','I','I','I','I','I','II','I','I','I','I','I','I','II','II','I','II','I','I','I','I','I','II','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','II','I','I','I','I','I','I','II','II','II','II','I','I','I','II','I','I','I','IV','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','IIIb','II','I','I','I','IIIb','I','I','I','I','I','I','IIIb','I','I','II','I','II','I','I','I','I','I','II','II','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','II','II','I','I','II','II','II','I','I','I','I','II','IV','I','II','I','IV','II','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','IV','I','II','I','IIIb','II','VI','I','I','II','I','II','I','I','I','I','I','I','II','I','II','II','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','I','I','II','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','IIIb','I','I','I','I','I','I','II','I','I','I','II','I','I','II','I','I','I','I','I','I','I','I','I','I','IIIb','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','II','I','I','II','I','I','I','I','I','I','I','I','II','I','I','I','II','I','II','II','I','I','I','II','I','II','II','I','I','I','I','I','I','I','I','II','I','I','I','I','II','I','I','I','I','I','II','II','I','I','I','I','II','II','II','I','I','I','I','II','I','II','I','I','II','I','II','II','I','I','I','I','I','I','I','I','II','I','II','I','II','I','I','I','I','I','II','II','I','I','I','I','I','I','I','II','I','I','I','II','I','II','II','I','I','I','I','I','I','II','II','I','I','II','I','II','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','I','II','II','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','II','II','I','II','I','II','I','I','I','I','I','II','II','I','I','I','I','I','II','I','II','II','II','II','II','II','II','II','II','II','I','I','I','I','II','I','II','II','II','I','I','I','I','I','II','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','I','II','I','I','II','II','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','II','I','I','I','I','I','I','II','I','I','I','II','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','I','II','I','I','I','II','I','I','I','I','I','I','I','I','II','I','I','I','I','II','I','I','I','I','I','I','II','II','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','VI','I','I','I','II','I','II','I','I','I','VI','VI','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','II','IV','I','II','I','I','I','I','IIIb','I','I','I','I','I','I','I','IIIb','I','I','I','I','IIIb','IIIb','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','IIIa','IV','I','IV','I','I','I','I','I','II','II','I','II','I','II','I','I','II','I','I','I','II','I','I','I','I','I','I','I','II','I','I','I','II','I','I','I','I','II','I','I','I','I','I','I','I','I','II','I','I','I','IIIb','I','II','II','II','II','II','I','II','I','II','II','II','II','II','IIIa','I','II','II','II','II','I','V','I','I','I','I','I','II','I','I','I','II','I','IIIb','I','I','II','I','II','II','II','II','V','I','I','I','I','I','I','I','II','I','IIIb','I','I','I','I','I','I','I','II','II','I','IIIa','I','IIIa','I','IIIa','I','II','I','I','II','II','II','II','I','I','I','I','I','I','I','I','V','II','II','I','II','II','I','VI','VI','I','I','I','I','IIIb','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','V','I','I','I','I','II','II','I','I','I','I','I','I','II','II','I','IIIb','II','IIIb','I','I','I','I','IIIb','IIIb','IIIb','VI','I','I','I','IIIb','I','I','I','I','I','I','I','I','I','I','I','I','I','I','IV','I','I','I','VI','II','I','IIIb','IIIb','I','II','IIIb','I','II','I','II','IIIb','IIIb','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','II','I','I','I','I','I','I','I','II','I','I','I','II','I','I','I','I','I','I','I','I','I','II','II','I','I','I','II','II','I','II','I','II','I','I','I','I','I','I','I','IIIb','I','I','I','I','IIIb','IIIb','IIIb','IIIb','IIIb','I','I','I','I','IIIb','I','IIIb','IIIb','I','I','I','I','I','II','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','II','I','I','II','IV','I','II','IV','IV','II','I','IIIb','I','IIIb','I','I','I','II','II','I','I','I','II','IV','II','I','I','II','II','IIIb','I','I','I','I','I','I','II','I','II','I','I','II','II','I','I','I','II','I','II','II','I','II','I','II','II','IIIa','I','II','I','IIIb','I','I','I','I','I','IIIb','II','I','I','I','I','IIIb','IIIb','I','I','IIIb','II','I','I','I','I','I','I','II','I','II','IIIa','IIIa','IIIa','IIIa','I','IIIb','IIIb','II','I','IIIa','IV','I','IIIa','IV','I','I','I','I','I','I','I','I','IIIa','IV','II','I','I','IIIb','IIIb','IIIb','I','IIIb','IIIb','I','I','IIIb','I','II','I','I','I','II','IIIa','I','II','IIIa','IIIb','I','I','II','II','IIIa','IV','I','I','I','I','II','I','I','I','I','II','IIIb','I','I','I','I','II','IIIa','IV','II','IV','I','IIIb','IIIb','I','IIIb','IIIb','I','IIIb','IIIb','I','I','I','IIIb','I','I','II','I','IIIa','IV','II','IIIa','IV','IIIb','IIIb','II','IIIb','IIIa','I','IIIa','IV','IV','IIIb','I','I','I','I','II','I','I','I','I','I','I','II','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','II','I','I','I','II','I','II','II','II','I','II','I','II','I','I','I','I','I','I','I','I','I','I','I','I','IIIb','I','I','I','I','I','I','I','I','II','I','II','I','I','II','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','IIIb','I','II','II','I','I','I','I','I','I','I','I','II','I','I','I','I','I','I','II','I','I','I','II','I','I','I','I','II','I','IV','I','II','I','II','I','I','I','I','I','I','I','I','I','II','I','I','I','I','I','II','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','I','I','I','I','II','I','I','I','I','I','II','I','I','I','I','II','I','I','II','II','I','IIIa','II','I','II','IIIb','I','I','IIIb','I','IIIb','IIIb','I','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','IIIb','I','IIIb','IIIb','IIIb','I','I','IIIa','I','IIIa','IIIa','I','I','I','IIIb','II','I','IIIa','IIIa','I','IIIb','IIIb','IIIb','IIIb','I','II','I','I','I','I','II','II','I','I','I','II','IIIa','IV','I','I','I','IIIb','IIIb','I','I','I','I','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','I','I','I','I','IIIb','IIIb','IIIb','I','IIIb','I','IIIb','IIIb','IIIb','I','I','I','I','I','IIIa','IV','I','I','IIIb','IIIb','I','I','IIIb','IIIb','IIIb','IIIb','II','I','I','II','I','I','I','II','I','II','I','I','I','I','I','II','I','I','II','II','I','I','I','I','I','I','II','I','II','I','I','I','II','I','I','I','I','I','II','II','II','II','I','II','I','II','II','I','I','II','II','II','II','IV','II','IIIa','IIIb','IV','II','II','II','IV','IIIb','I','I','I','II','IIIb','II','IIIb','I','I','II','IIIb','IIIb','I','II','I','I','II','I','II','IIIb','I','II','I','I','II','II','II','IIIa','IV','II','IIIa','IV','IIIa','IV','II','I','II','IIIb','I','IIIa','II','I','V','IIIa','II','II','IIIb','II','I','II','II','II','V','I','II','I','VI','I','I','I','IIIb','II','I','II','II','II','IIIa','I','I','I','I','II','II','IIIb','II','I','I','I','II','I','I','I','IIIa','IV','IIIa','I','IIIa','IIIa','II','I','I','I','II','II','I','IIIa','IV','I','IIIa','IV','I','IV','II','I','II','I','II','I','I','I','II','I','I','I','II','II','II','I','IIIa','IV','II','I','I','I','I','I','IIIb','IIIb','I','IIIb','IIIb','I','II','IIIb','I','IIIb','IIIb','II','I','II','I','I','IIIb','IIIb','II','IIIb','I','I','II','I','II','I','II','I','I','IIIa','I','IIIa','IV','II','II','II','IIIb','IIIb','I','II','IIIb','IIIb','I','II','I','I','I','I','IV','I','I','IIIb','IIIb','I','I','II','II','I','I','II','I','I','II','II','II','II','II','I','II','II','IIIb','IIIa','IV','II','I','I','I','I','IIIb','IIIb','I','I','II','II','IIIb','IIIb','IIIb','I','I','I','II','I','I','II','IIIa','IV','IIIa','IV','IV','I','II','II','I','IIIa','IV','I','IIIa','IV','I','I','II','IIIb','IV','I','I','I','I','I','I','I','I','I','I','I','I','I','II','I','I','I','II','IIIb','I','IV','I','II','I','I','I','I','I','V','I','I','I','I','I','I','II','IIIa','IV','IIIa','I','IIIa','IV','IIIa','IV','I','I','II','I','IV','IV','I','II','V','VI','I','II','I','I','I','I','I','I','I','I','I','II','II','II','I','IIIa','IV','II','I','I','I','I','I','I','II','I','II','IIIa','IV','I','IIIa','IIIa','I','I','II','II','II','IIIa','I','I','IIIa','IV','II','II','I','II','II','I','II','IIIb','I','IIIb','IIIb','I','IIIb','II','I','II','II','II','I','I','I','II','IIIa','I','IIIb','I','IIIb','IIIb','IIIb','I','I','IIIb','I','IIIb','I','IIIb','IIIb','IIIb','IIIb','I','IIIb','IIIb','IIIb','IIIb','IIIb','IV','I','IIIb','I','IIIb','IIIb','IIIb','IIIb','I','I','I','I','II','II','I','I','I','I','I','IIIa','I','II','IIIb','I','IIIb','IIIb','II','I','IIIa','I','I','IV','I','I','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','I','II','II','II','II','V','IIIb','II','II','I','I','II','II','IIIa','IV','IIIb','IIIb','IIIb','IIIb','V','IIIb','II','IIIb','I','IIIb','IIIb','II','IIIb','IIIb','I','IIIb','IIIb','IIIb','IIIb','IIIb','I','II','IIIb','IIIa','IV','IIIa','I','IIIa','IIIb','IV','IIIa','IV','II','VI','II','I','IIIb','IIIb','II','IIIa','IV','IIIb','V','IIIa','IV','II','V','V','IIIb','IIIb','V','V','II','IV','I','II','IV','I','IV','II','II','IIIa','IV','II','II','IIIb','IIIb','IIIb','IIIb','IIIb','II','IIIb','IIIb','II','IIIb','IIIb','I','IIIb','IIIb','IIIb','IIIb','VI','II','II','IIIb','IIIb','IIIb','IIIb','IIIb','I','II','IIIb','IIIa','IV','IIIa','IIIa','IV','IIIa','IV','IIIb','II','IIIb','I','IIIa','IIIa','II','IIIb','IIIb','IIIb','IV','I','I','I','IV','II','I','I','II','IIIa','I','I','I','I','IIIb','I','I','I','IIIb','I','II','II','I','I','I','I','I','I','IIIa','IV','IIIa','IIIa','I','II','IIIa','II','I','I','I','I','II','I','II','II','I','I','I','II','I','IIIb','IIIb','IIIb','IIIb','I','II','IIIb','II','II','II','II','II','II','IIIa','IV','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','II','II','IIIb','IIIb','IIIb','II','IIIb','II','IIIa','IV','IIIa','II','IIIa','IIIb','IIIb','IIIa','IV','IIIb','IIIb','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','II','II','II','II','II','II','II','IIIa','IIIa','II','IIIa','I','II','I','I','IIIb','IIIb','II','II','II','IIIb','IIIb','IIIb','IV','IIIb','II','II','II','II','II','II','II','IIIb','IIIb','IIIb','IIIb','IIIb','II','II','IIIb','IIIb','IIIb','II','II','IIIb','II','IIIb','II','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','IIIa','IIIb','IIIb','IIIa','IIIa','VI','IIIb','II','II','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','IIIb','IIIb','II','IIIb','II','V','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','V','IIIb','IIIb','IIIb','IIIb','IIIa','IIIa','IIIa','IIIa','IIIa','IIIa','IIIb','IIIa','IIIa','IIIa','IIIb','IIIb','IIIb','II','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','IIIb','II','V','V','V','V','V','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I']
wklm_O = ['[1],2,12','2,12','[1],2,12','2,12','[1],4,[5],12','4,12,[27]','4,[5],12','4,[5],12','[1],4,[5],12','[1],4,12,27','[1],4,[5],12,[27]','[1],4,12','[1],4,12,27','[1],4,12,[27]','[1],4,[5],12','[1],4,[5],12,[27]','4,12','4,12,[27]','[1],4,12,27','[1],4,[5],12,[27]','[1],4,[5],12,[27]','[1],4,12,27','[1],4,12,[27]','[1],4,12,27','4,12,27','[1],4,12,[27]','4,12','4,12,[27]','4,12','[1],4,12,27','[1],4,12,27','4,12,27','[1],4,[5],12,[27]','[1],4,12,[27]','[1],4,12,27','[1],4,12,27','4,12','[1],4,[5],12,[27]','[1],4,12,[27]','[1],4,12,27','[1],4,12,27','4,12','[1],4,[5],12','[1],4,[5],12','4,12','4,12','[1],4,[5],12','[1],4,[5],12','[1],4,12','4,12','[1],4,12,[27]','[1],4,[5],12','[1],4,[5],12','[1],4,[5],12','4,12','[1],4,[5],12','[1],4,12,[27]','[1],4,12,[27]','4,12','4,12','[1],4,[5],12,[27]','[1],4,12,[27]','4,[5],12','4,[5],12','4,12','[1],4,[5],12','4,[5],12','[1],4,[5],12','[1],4,[5],12','4,12','4,12','[1],4,12','[1],4,12,27','[1],4,12','4,12,27','[1],4,12,27','[1],4,12,27','[1],4,12,27','4,12,27','4,[5],12','4,[5],12','[1],4,[5],12,[27]','4,12','[1],4,12,27','[1],4,12,27','[1],4,12,27','4,[5],12','[1],4,12,27','4,12','4,12','4,12','4,12','[1],4,12,27','[1],4,[5],12,[27]','[1],4,12,[27]','4,12','[1],4,12,27','[1],4,12,27','[1],4,12,27','4,12','[1],4,12','[1],4,[5],12','4,12,[27]','4,12','[1],4,12,27','[1],4,[5],12','4,12,27','[1],4,12,27','4,12','[1],4,[5],12','4,12,27','4,12','[1],4,12,27','[1],4,[5],12,[27]','[1],4,12,27','[1],4,12,27','4,[5],12','[1],4,12','[1],4,12,27','4,12','[1],4,12','4,12','4,12','[1],4,12,27','[1],4,[5],12','[1],4,12','[1],4,12,27','4,12','[1],4,[5],12,[27]','4,12','4,12,27','[1],4,[5],12','[1],4,[5],12','[1],4,12','4,12','4,12','4,12','4,12','[1],4,12','[1],4,12,[27]','[1],4,12,27','[1],4,12,27','[1],4,12','4,12','[1],4,12,27','[1],4,12,27','[1],4,12,27','4,12','[1],4,12','[1],4,[5],12,[27]','[1],4,12,[27]','[1],4,12,27','[1],4,12,27','4,12','4,12','6,7','6,7,[14]','6,7','6,7','6,7,[14]','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7,[14]','6,7','6,7','6,7,[14]','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7,[14]','6,7,[Vi]','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7','6,7,[14]','6,7','6,7','6,7,[14]','6,7,[14]','6,7,[14]','6,7','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7','6,7','6,7,[14]','6,7','6,7','6,7,[14],[54]','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7','6,7','6,7','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7','6,7,[14]','6,7,[14]','6,7','6,7','6,7','6,7','6,7','6,7','6,7','8,[20]','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','8','6,8','8,[20]','6,8','6,8','6,8','8','6,8','8','6,8','8,[20]','6,8','6,8','8,[20]','8,[20]','6,8','6,8','8,[20]','6,8','6,8','6,8','8','8,[20]','6,8','6,8','8,[20]','8','6,8','8,[20]','6,8','8,[20]','6,8','6,8','6,8','8,[20]','6,8','8','6,8,[20]','8','6,8','6,8','8,[20]','6,8','6,8','8,[20]','6,8','6,8','8,[20]','6,8','6,8','8,[20]','8,[20]','6,8','8,[20]','6,8','6,8','6,8','6,8','8,[20]','6,8','6,8','6,8','6,8','6,8','8,[20]','6,8','8,[20]','6,8','8','6,8','6,8','6,8','6,8','8','8','6,8','6,8','8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','6,8','8,[20]','6,8,[20]','8,[20]','8','6,8','8,[20]','6,8','8,[20]','6,8','8','8,[20]','8,[20]','6,8','8,[20]','6,8','6,8','6,8','8,[20]','6,8','8','6,8','6,8','8,[20]','6,8','8,[20]','6,8','6,8','6,8','8','6,8','6,8','8,[20]','8','6,8','6,8','6,8','8,[20]','8,[20]','8','8,[20]','6,8','6,8','8,[20]','6,8','8,[20]','6,8','6,8','8','6,8','8,[20]','6,8','8,[20]','8,[20]','6,8','6,8','6,8','8,[20]','6,8','6,8','8,[20]','6,8','8,[20]','8,[20]','8,[20]','6,8','[1],9,12','[1],9,12','9,12','9,12','[1],9,12','[1],9,12','[1],9,12','[1],9,12','9,12','[1],9,12','[1],9,12','[1],9,12','9,12','[1],9,12','9,12','[1],9,12','[1],9,12','9,12','9,12','9,12','9,12','9,12','[1],9,12','9,12','9,12','[1],9,12','[1],9,12','9,12','9,12','[1],9,12','9,12','9,12,[Vi]','9,12','[1],9,12','9,12','9,12','9,12','9,12','[1],9,12','[1],9,12','9,12','9,12','[1],9,12','[1],9,12','9,12','[1],9,12,[Vi]','[1],9,12','[1],9,12','[1],9,12','9,12','9,12','[1],9,12','9,12','9,12','9,12','[1],9,12','[1],9,12','[1],9,12','9,12','[1],9,12','9,12','9,12','[1],9,12','9,12','9,12','9,12','9,12','9,12','9,12','[1],9,12','[1],9,12','9,12','9,12','[1],9,12','[1],9,12','9,12','9,12','9,12','9,12','9,12','9,12','9,12','9,12','9,12','9,12','[1],9,12','[1],9,12','[1],9,12','9,12','[1],9,12','9,12','[1],9,12','9,12','[1],9,12','9,12','9,12','9,12','9,12','9,12','9,12','[1],9,12','9,12','[1],9,12','[1],9,12','[1],9,12','[1],9,12','[1],9,12','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','[1],9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','9,46','1,9,12,46,27','1,9,12,46,27','1,9,12,46,27','9,12,46,27','1,9,12,46,27','1,9,12,46,27','1,9,12,46,27','1,9,12,46,27','1,9,12,46,27','1,9,12,46,27','3,10','3,{10}{15}','3,{10}{15}{15,34}','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}{15,34}','3,10','3,{10}{15,34}','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,{10}{15}','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,{10}{15}{15,34}','3,{10}{15}{15,34}','3,{10}{15}','3,{10}{15,34}','3,10','3,{10}{15}{15,34}','3,10','3,10','3,10','3,10','3,10','3,{10}{15}{15,34}','3,{10}{15}','3,{10}{15}{15,34}','3,10','3,10','3,10','3,{10}{15}{15,34}','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,{10}{15}','3,{10}{15}{15,34}','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,{10}{15}{15,34}','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,{10}{15}{15,34}','3,{10}{15}{15,34}','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15}','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,10','3,{10}{15,34}','3,15','3,15,34','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,19','1,3,10,19','1,3,10,19','1,3,10,19','1,3,[15],19','1,3,{10}{15},19','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','11','[1],13,23','[1],13,23','13,22','13,22','[1],13,23','13,22','[1],13,23','13,22','[1],13,23','13,22','[1],13,23','13,23','[1],13,22','13,22','[1],13,23','13,22','13,23','13,23','[1],13,22','13,22','13,23','[1],13,23','13,22','[1],13,23','13,22','[1],13,23','[1],13,23','13,23','13,23','[1],13,22','13,23','[1],13,23','13,23','[1],13,23','[1],13,22','[1],13,23','[1],13,23','13,22','[1],13,23','13,22','13,22','[1],13,23','[1],13,22','13,22','[1],13,23','[1],13,23','[1],13,23','13,22','13,23','13,23','[1],13,22','13,22','[1],13,23','13,23','[1],13,23','[1],13,23','13,22','[1],13,23','[1],13,23','13,22','[1],13,23','[1],13,23','[1],13,22','[1],13,23','13,23','13,23','[1],13,23','13,23','13,22','13,23','13,23','13,23','13,23','13,22','13,22','13,22','[1],13,23','13,23','13,22','13,22','13,23','13,23','[1],13,23','13,22','13,23','13,23','[1],13,22','13,23','[1],13,23','[1],13,23','13,22','[1],13,23','[1],13,23','13,23','[1],13,22','13,23','13,22','13,22','[1],13,23','[1],13,22','[1],13,23','[1],13,23','13,22','[1],13,22','13,22','13,23','13,23','[1],13,23','13,23','[1],13,22','[1],13,22','13,23','13,23','13,22','13,22','[1],13,23','[1],13,23','13,22','[1],13,23','13,22','13,22','13,23','13,23','13,22','13,23','13,22','13,22','[1],13,23','13,23','[1],13,23','13,23','1,6,14,25','[1],6,14','[1],6,14,[25]','1,6,14,25','1,6,14,25','1,6,14,25','[1],6,14,[25]','(6),14','1,6,14,25','1,6,14,25','1,6,14,25','1,6,14,25','6,14,24','[1],6,14,[25]','1,6,14,25','6,14,24','[1],6,14,[25]','6,14,[24]','[1],6,14,[25]','[1],6,14,[25]','1,6,14,25','1,6,14,24','6,14,[24]','1,6,14,[25]','1,6,14,25','[1],6,14,[25]','6,14','[1],6,14,[25]','[1],6,14,[25]','[1],6,14,[25]','6,14','1,6,14','6,14','1,6,14,25','1,6,14,25','[1],6,14,[25]','6,14,[24]','1,6,14,25','[1],6,14,[25]','6,14,[24]','6,14','6,14,25','(6),14','1,6,14','(6),14','6,14,24','[1],6,14,[25]','1,6,14,[25]','6,14','(6),14','(6),14','(6),14','6,14','6,14,25','1,6,14,25','1,6,14,25','(6),14','[1],6,14,[25]','6,14,[24]','1,6,14,25','1,6,14,25','1,6,14,25','1,6,14,25','1,6,14,25','6,14,24','[1],6,14,[25]','1,6,14,25','6,14,24','[1],6,14,[25]','6,14','1,6,14,25','6,14','6,14,24','1,6,14,25','1,6,14,25','1,6,14,25','1,6,14','1,6,14,25','(6),14','(6),14','1,6,14','1,6,14','6,14','6,14','6,14','1,6,14,25','1,6,14','6,14','[1],6,14,[25]','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','16','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','17','[6],[14],18','[6],[14],18','18','[6],[14],18','18','[6],[14],18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','18','[6],[14],18','18','18','18','[6],[14],18','18','18','18','18','18','18','18','18','18','18','18','18','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','21','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','28','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','30','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','35','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','38','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','39','40','40','40','40','40','40','[1],40','[1],40','40','40','[1],40','40','[1],40','[1],40','40','[1],40','[1],40','[1],40','[1],40','[1],40','[1],40','[1],40','[1],40','40','[1],40','[1],40','40','40','[1],40','40','40','[1],40','40','40','[1],40','[1],40','40','[1],40','40','40','40','[1],40','40','40','40','[1],40','[1],40','40','[1],40','[1],40','[1],40','40','[1],40','40','40','[1],40','40','40','[1],40','[1],40','40','40','[1],40','40','40','[1],40','40','[1],40','[1],40','40','40','40','[1],40','40','[1],40','[1],40','[1],40','40','[1],40','[1],40','40','40','[1],40','[1],40','[1],40','[1],40','[1],40','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','41','[1],42','42','42','42','[1],42','[1],42','[1],42','42','[1],42','[1],42','[1],42','42','42','42','[1],42','42','[1],42','42','[1],42','[1],42','42','[1],42','42','42','42','[1],42','42','42','[1],42','42','42','42','42','42','[1],42','[1],42','[1],42','[1],42','[1],42','42','42','42','42','[1],42','[1],42','42','[1],42','42','42','42','[1],42','42','42','[1],42','42','[1],42','42','42','42','42','42','[1],42','42','42','42','[1],42','42','[1],42','42','[1],42','42','42','[1],42','[1],42','42','42','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','43','44','44','44','44','44','44','44','44','44','44','44','44','44','[1],44','[1],44','44','[1],44','44','44','[1],44','44','44','44','[1],44','[1],44','44','44','44','44','44','44','44','44','44','44','[1],44','44','44','44','44','44','44','44','44','[1],44','[1],44','[1],44','44','44','44','44','44','[1],44','44','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','45','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','[1],47','[1],47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','[1],47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','47','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','48','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','50','51','51','51','51','51','51','[1],51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','51','[1],51','51','51','51','51','51','51','51','51','51','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','52','53','53','[1],53','53','53','[1],53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','53','[1],53','53','53','53','21,54','54','3,54','54','3,54','4,12,54','8,[20],54','6,7,54','54','3,15,54','54','54','54','54','55','56','56','56','56','56','56','56','56','56','56','57','57','57','57','57','57','57','57','57','57','57','57','57','57','57','57','57','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','58','59','59','59','59','59','59','59','59','59','59','59','59','[1],59','59','59','59','59','59','59','59','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','60','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','61','62','62','62','62','62','63','63','63','63','63','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','65','66','66','66','66','66','6,7','[1],9,12','67','3,15','6,7,14','3,15','3,15','3,15,34','3,15,34','6,14,18','6,7,14','3,15','3,15,34','3,15','3,15','3,15,34','6,7,14','6,7,14','3,15','3,15','3,15,34','3,15','3,15,34','3,15','3,15','6,7,14','3,15','3,15,34','3,15,34','[1],4,[5],12','6,7,14','3,15','3,15','1,4,[5],12,27','3,15','3,15','3,15','3,15','3,15','3,15','3,15,34']
wklm_P1 = ['a','g,m','g,p','l,v','a','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','c','c','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','e,h','e,h','e,n,x','e,n,x','f,g','f,g,s','f,g,t','g,m','g,m,s','g,[m],[s],t','g,[m],t','g,m,t','g,m,t','g,s,t','g,t','g,z51','g,z51','g,z62','m,t','m,t','i','i','i','i','i','i','i','i','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,w','l,w','l,w','l,[z13],[z28]','l,[z13],z28','l,z13,[z28]','l,z13,z28','l,z13,z28','l,z13,z28','l,z28','l,z28','r','r','r','r','r','r','r,[i]','r,i','y','y','y','y','y','y','y','z','z','z','z','z','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z24','z10','z10','z10','z10','z10','z10','z10','z10','z10','z29','z29','z35','z35','z35','z35','z36','z38','z38','z39','z41','z41','–','z91','a','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','b','c','c','c','c','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','e,n,x','e,n,z15','f,g','f,g,t','f,g,t','g,m,[p],s','g,m,s','g,m,[s],t','(g),m,[s],t','g,m,s,t','g,[m],s,t','g,m,[t]','g,s,q','g,s,[t]','g,t','g,(t)','g,z51','g,z51','g,z51','m,p,t,[u]','m,t','m,t','m,t','m,t','i','i','i','i','i','i','i','k','k','k','k','k','k','(k)','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,w','l,w','l,w','l,z13','l,z13','l,z13,[z28]','l,z13,z28','l,z13,z28','l,z28','l,z28','l,z28','r','r','r','r','r','r','y','y','y','y','y','y','z','z','z','z','z','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z24','z4,z24','z6','z10','z10','z10','z10','z10','z10','z10','z10','z10','z29','z29','z35','z35','z36','z36','z36','z38','z38','z38','z39','z39','z41','z41','z41','z42','z44','–','a','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','b','b','b','b','b','c','c','c','c','c','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','e,h','e,h','e,h','e,h','f,g','f,g,m,t','g,m,s','g,m,[s]','g,m,t','g,m,t','g,s,t','g,s,t','m,t','m,t','m,t','m,t','i','i','i','i','i','i','i','i','i','i','k','k','k','k','k','k','(k)','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,z13,z28','l,z13,[z28]','l,z28','r','r','r,[i]','r','r','r','r,i','r,[i]','r,[i]','r','r','r,[i]','y','y','y','y','y','y','y','y','y','y','y','y','y','y','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z24','z4,z32','z10','z10','z10','z10','z10','z10','z10','z10','z10','z10','z10','z10','z29','z29','z29','z29','z29','z35','z35','z35','z38','z41','z60','a','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,n,x','e,n,x','[f],g,[t]','g,m','g,m,s','g,m,q','g,m,[s],t','g,m,s,t','g,m,t','g,p','g,p,s','g,p,u','g,q','g,s,t','g,z51','g,z62','g,z63','g,z85','m,t','m,t','m,t','m,t','i','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,z13','l,z13','l,z13','l,z28','l,z28','l,z28','l,z28','l,z28','r','r','r','r','r','y','y','z','z','z','z','z','z','z','z4,z23','z4,z24','z6','z10','z10','z10','z29','z29','z35','z38','z39','z41','z42','–','a','a','a','a','b','b','b','b','b','b','b','b','c','c','d','d','d','d','d','d','e,h','e,h','e,h','e,n,x','f,g','g,m','g,m,s','g,[m],[s],t','g,s,t','g,z62','m,t','m,t','i','i','k','k','k','k','l,v','l,v','l,v','l,w','l,w','l,z13,z28','r','r','r','y','y','y','y','y','z','z','z','z','z4,z23','z4,z24','z4,z24','z10','z10','z10','z10','z10','z10','z29','z35','z35','z35','z38','z39','z41','a','b','c','g,t','l,z13,z28','y','z4,z24','z10','z10','z10','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','c','c','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','e,h','e,h','e,h','e,n,x','f,g,[s]','f,g','g,m','g,m,s','g,m,s,t','g,s,t','g,t','g,t','m,t','m,t','m,t','i','i','i','i','i','i','i','i','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,z13','l,z13','l,z13,z28','l,z13,z28','l,z28','l,z28','l,z28','l,z28','l,z28','l,z28','r','r','r','r','r,i','r,i','y','y','y','y','y','y','y','z','z','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z24','z4,z24','z10','z10','z10','z10','z10','z10','z10','z29','z29','z29','z35','z35','z35','z35','z35','z35','z36','z38','z38','z38','z38','z39','z41','z69','z87','y','a','a','a','a','b','b','b','b','b','b','c','c','c','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','f,g','f,g,t','f,g,t','g,m,[t]','g,[s],t','m,t','i','i','i','i','i','i','i','k','k','l,v','l,v','l,v','l,w','l,z13','l,z13','l,z13,z28','l,[z13],z28','l,z13,z28','l,z28','l,z28','r','r,i','y','y','y','y','y','z','z','z','z','z','z','z4,z23','z4,z24','z10','z10','z29','z35','z38','z41','b','f,g,t','z4,z23','g,s,t','y','a','a','a','a','a','a','a','a','a','b','b','b','b','b','b','b','c','c','c','c','d','d','d','d','e,h','e,h','e,h','g,m','g,s,t','g,[m],s,t','g,z51','m,t','m,t','i','i','i','i','i','k','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,z13','l,z13,z28','l,z13,z28','l,z13,z28','l,z28','r','r','r','r,i','r','y','y','y','y','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z32','z10','z10','z10','z38','z41','z41','–','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','b','c','c','d','d','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,n,x','f,g','f,g,[s]','g,m','g,m,s','g,m,[s],[t]','g,m,t','g,m,t','g,m,s,t','g,m,[s],t','g,m,s,t','g,s,t','g,s,t','g,[s],t','g,t','g,t','g,t','g,t','g,[s],t','g,z51','m,t','m,t','m,t','m,t','m,t','m,t','i','i','i','i','i','i','k','k','k','k','k','l,v','l,v','l,v','l,v','l,w','l,z13','l,z28','l,z28','l,z28','l,z28','r','r','r','y','y','y','y','z','z','z','z','z','z','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23,[z32]','z4,z24','z4,z24','z10','z10','z10','z10','z29','z29','z29','z29','z29','z29','z35','z35','z35','z35','z38','z38','z39','z39','z39','z41','[z42]','–','a','a','a','a','b','b','b','b','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,n,z15','g,m','g,m,s','g,p','g,s,t','m,t','m,t','m,t','i','i','i','i','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,z13,z28','l,z28','r,i','r','r,[i]','y','y','y','y','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z10','z10','z10','z10','z10','z10','z10','z10','z10','z36','z36','z38','z42','z52','z52','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','b','b','b','c','c','c','c','c','d','d','d','d','d','d','d','d','d','d','e,h','e,h','e,h','e,h','e,h','e,h','e,n,x','e,n,x','f,g','g,m,[t]','g,m,s','g,[m],[s],t','g,[m],[s],t','g,s,t','m,t','m,t','m,t','m,t','i','i','i','i','i','i','i','k','k','k','k','(k)','k','(k)','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,w','l,w','l,w','l,w','l,z13','l,[z13],z28','l,z13,z28','l,z13,z28','l,z13,z28','l,z28','r,i','r','r','r','r,i','r,i','y','y','y','y','z','z','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z24','z4,z32','z6','z10','z10','z10','z10','z10','z10','z29','z29','z29','z35','z35','z35','z35','z36','z36','z38','z39','z42','z42','z52','a','a','a','b','b','b','b','b','b','c','c','c','d','d','d','d','d','e,h','e,n,x,z15','e,n,x,z15','g,m','g,m,s,t','g,s,t','g,t','g,t','g,z51','m,t','m,t','i','i','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,z28','r','r','y','y','y','y','z','z','z','z','z','z','z4,z23','z4,z23,z32','z4,z24','z4,z32','z10','z10','z10','z10','z29','z29','z29','z35','z36','z36','a','b','b','c','d','d','g,m','g,s,t','g,z51','g,z51','m,t','m,t','k','(k)','(k)','l,v','l,v','l,v','l,v','l,w','r','r','y','y','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z32','z10','z10','z10','z10','z36','z36','z36,z38','z38','z71','–','a','b','b','b','c','c','c','c','d','d','d','f,g','g,[m],[s],t','g,z51','g,z51','m,t','m,t','i','i','i','k','k','k','k','l,v','l,v','l,w','l,z28','r','r','y','y','z','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z32','z10','z10','z10','z10','z29','z35','z36','z36','z38','z65','a','a','a','a','a','a','a','b','b','b','b','b','b','b','b','c','c','c','c','c','c','c','d','d','d','d','d','d','d','e,h','e,h','e,h','e,n,x','e,n,z15','f,g','g,m','g,m','g,(m),[s],t','g,m,s','g,m,t','g,m,t','g,s,t','g,s,t','m,t','m,t','m,t','m,t','i','i','i','i','i','i','i','k','k','k','k','k','k','k','l,v','l,v','l,v','l,w','l,z13,z28','l,z13,z28','l,z28','l,z28','l,z28','r','r','r','r','r,[i]','r,i','r,[i]','y','y','y','y','y','y','y','z','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z24','z10','z10','z10','z10','z10','z10','z10','z10','z29','z29','z29','z35','z35','z35','z38','a','a','a','a','a','b','b','b','b','b','b','b','b','c','c','c','d','e,h','e,h','g,m,[t]','g,m,s','g,m,s','g,m,t,z51','g,s,t','g,t','g,z51','m,t','i','i','i','i','k','k','k','k','k','k','k','l,v','l,v','l,z13,z28','l,z13,z28','l,z28','r','r','r','y','y','y','y','y','z','z4,z23','z4,z24','z6','z10','z10','z10','z10','z29','z35','z35','z38','z39','a','b','c','c','c','d','d','e,h','f,g','g,m,s','g,m,s,t','g,m,t','g,s,t','g,t','g,t','g,t','g,z51','g,z62','m,t','m,t','i','i','i','i','i','i','i','k','k','k','(k)','k','l,v','l,v','l,v','l,z28','r','r','r','r','r','y','z4,z23','z4,z23','z4,z24','z4,z24','z4,z32','z10','z10','z10','z10','z29','z29','z29','z36','z38','z52','z52','z52','z52','a','b','b','c','c','d','d','d','e,h','e,h','g,m,[s]','g,t','g,z51','g,z51','m,t','i','i','i','i','k','k','k','k','k','k','k','(k)','(k)','(k)','(k)','l,v','l,v','l,v','l,v','l,v','l,v','l,v','r','r','r','r','r','r','y','y','y','z','z4,z23','z4,z23','z4,z23','z4,z24','z10','z10','z10','z35','z38','z52','z52','z53','z61','a','b','b','c','d','e,h','e,h','e,n,x','m,t','[g],m,t','i','i','i','k','k','l,v','l,v','l,v','l,z28','l,z28','r','y','y','y','z4,z23','z10','z10','–','–','a','a','a','a','b','b','b','b','b','b','c','c','c','d','d','e,h','e,n,x','e,n,x,z15','g,m','g,m,s','g,[m],[s],[t]','g,[m],[s],t','g,t','g,t','g,t','g,[m],[s],t','g,z51','g,z51','g,z51','m,t','m,t','m,t','m,t','i','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,w','l,z28','l,z13,z28','l,z28','l,z28','l,z28','l,z28','r','y','z','z','z','z','z','z','z4,z23','z4,z23','z4,z24','z4,z24','z4,z24','z4,z32','z4,z32','z6','z10','z10','z10','z29','z29','z35','z35','z35','z36','z39','z39','z39','z39','z41','z42','[z42]','z81','z81','a','b','b','b','b','b','c','c','c','d','d','g,m,s,t','g,t','g,z51','m,t','i','i','k','k','k','(k)','l,z13,z28','r','r','y','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23,z32','z4,z24','z4,z24','z4,z32','z10','z10','z10','z10','z10','z10','z29','z29','z29','z35','z36','z36','z38','z52','–','a','b','b','b','b','b','b','b','c','c','d','d','e,n,x','g,t','g,z51','g,z51','g,z51','m,t','m,t','i','i','i','k','k','k','k','(k)','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,w','l,[z13],z28','r','r','r','r','r','r','y','y','z','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z24','z6','z10','z10','z10','z10','z10','z10','z10','z10','z29','z29','z35','z35','z35','z36','z36','z38','z41','z52','z57','a','a','a','a','b','b','b','c','d','d','d','d','e,n,x,z15','e,n,x,z15','f,g,[t]','g,m,[s],t','g,t','g,t','g,z51','g,z51','g,z62','i','i','k','k','k','l,v','l,w','l,z13','l,z13,z28','m,t','r','r','r','y','y','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z4,z24','z4,z32','z10','z29','z29','z29','z29','z29','z35','z36','z36,z38','z38','z41','z42','z52','a','a','a','a','b','b','b','c','c','c','d','d','d','d','e,n,x','f,g','g,m,[s]','g,s,t','g,t','g,t','g,z51','g,z51','m,t','m,t','i','k','l,z28','r','r','r','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23,z32','z4,z24','z4,z24','z4,z24','z4,z32','z4,z32','z10','z10','z29','z29','z29','z36,[z38]','z38','z39','z39','a','a','a','b','b','b','c','d','d','d','f,g','g,m,[s],[t]','g,m,s,t','g,m,s,t','g,m,t','g,s,t','g,z51','g,z51','m,t','m,t','k','k','k','l,v','r','z','z','z','z4,z23','z4,z23','z4,z24','z4,z24','z4,z32','z10','z29','z29','z29','z29','z29','z35','z36','z36','z36,z38','a','a','b','b','b','b','b','c','c','c','c','c','c','d','d','d','d','e,n,x,z15','f,g','g,m','g,m,s','[g,t]','g,z51','m,t','i','i','i','i','i','k','k','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,v','l,z13,z28','r','r','r','r','r,[i]','r','y','y','z','z','z','z','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z23','z4,z24','z6','z10','z10','z10','z10','z29','z29','z29','z35','z35','z36','z38','z44','z52','z52','z52','z52','z52','z53','a','a','a','b','b','b','c','d','d','d','e,h','e,n,x,z15','g,m,t','g,z51','g,z51','i','i','i','i','i','k','k','k','k','k','k','k','k','k','l,v','l,v','l,v','l,w','r','r','z','z','z','z4,z23','z4,z23','z4,z23,z32','z4,z24','z4,z24','z4,z24','z4,z24','z4,z32','z4,z32','z10','z10','z10','z10','z10','z10','z29','z29','z29','z35','z35','z36','z36,[z38]','z39','z39','z41','z52','z52','z65','z81','z81','a','b','b','b','d','d','e,n,x','g,[m],s,t','g,z51','g,z51','g,z62','m,t','i','i','i','i','k','k','k','k','k','k','k','l,v','l,v','l,v','l,v','l,v','l,v','l,w','l,z28','r','r','r','r','r','y','z','z','z4,z23','z4,z23','z4,z23,z32','z4,z24','z4,z24','z4,z32','z4,z32','z10','z10','z10','z29','z29','z36','z42','z52','z52','z52','a','a','a','b','b','c','d','e,h','g,s,t','g,z51','m,t','i','i','k','k','l,v','l,v','l,v','l,v','l,z28','l,z28','l,z28','r','r,i','y','z','z','z','z4,z23','z4,z23','z4,z24','z4,z32','z10','z29','z36','–','a','a','b','b','c','d','d','d','e,h','e,h','g,t','g,t','k','k','k','k','l,v','l,w','z','z','z39','z44','c','d','d','d','g,z51','g,z51','i','k','k','(k)','k','l,v','l,v','l,v','l,z28','l,z28','l,z28','r','r','r','z','z','z','z4,z23','z4,z23','z4,z23,z32','z4,z24','z4,z24','z10','z10','z29','z36,z38','z52','z52','–','b','e,h','e,h','f,g,s','g,s,t','m,t','i','k','r','y','z4,z23','z10','z10','z29','k','b','d','e,n,x','l,v','l,z28','z','z4,z23','z4,z23,z32','z10','z29','a','a','b','b','c','c','d','g,[m],s,t','g,t','i','i','k','z4,z23','z10','z29','z39','z42','a','b','c','d','i','i','k','l,v','l,v','l,z13,z28','l,z13,z28','r','r','r','z6','z10','z10','z10','z10','z39','z52','z52','c','i','i','i','(k)','k','(k)','(k)','k','l,v','l,v','r','z','z4,z23','z10','z10','z29','z36','z36','z52','b','g,m,t','i','i','i','k','k','(k)','l,v','r','r','r','r','z','z10','z10','z10','z10','z29','z41','z52','z52','z52','z52','c','c','i','i','i','i','k','k','(k)','l,v','l,v','l,v','r','r','r','r','z10','z35','z52','z52','z52','z52','g,z51','z4,z23','z4,z32','z29','z36','g,z51','(k)','z4,z23','z4,z32','z36','c','c','c','g,t','i','i','(k)','(k)','(k)','l,v','l,v','l,v','l,v','r','z10','z10','z52','z52','z52','z52','–','z35','z39','z41','z65','z81','c','a','r','y','d','g,m,s','e,h','e,h','b','z4,z23','l,v','l,v','l,v','a','z10','z10','z38','d','l,v','e,h','e,h','e,h','e,h','l,v','e,h','m,t','y','y','a','b','f,g','d','d','z4,z23','y','l,z13','e,h','e,h','r','g,s,t','g,s,t']
wklm_P2 = ['[1,5]','–','–','1,5','1,2','1,5','[1,5]','1,7','e,n,x','e,n,z15','e,n,x','l,w','z6','z39','1,2','1,5','1,5','1,6','1,7','e,n,x','[e,n,x]','e,n,z15','l,w','z6','–','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','z35','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','1,2,7','1,[5],7','[1,2]','[1,2]','z6','–','[1,2]','e,n,x','[1,5]','z39','[z67]','[1,2]','–','1,7','e,n,z15','–','[1,5]','e,n,z15','1,2','1,5','1,6','e,n,x','e,n,z15','l,w','z6','z35','1,5','1,6','1,6','e,n,x','e,n,z15','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z39','1,5','1,6','e,n,x','e,n,z15','z6','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','e,n,x','[e,n,x]','1,2','1,5','1,6','1,7','l,w','z6','e,n,z15','l,w','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z6','1,2','1,5','1,5','1,6','1,7','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','z39','[1,2]','e,n,z15','z6','[1,5]','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','z35','–','e,n,x','1,5','1,7','z6','e,n,z15','–','1,5','[e,n,z15]','1,[5],7','1,(2),5','e,n,z15','e,n,x','–','1,5','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','z6','z42','1,2','1,5','1,5','1,6','1,7','e,n,x','e,n,z15','[e,n,x]','l,w','z6','z35','z39','1,2','1,5','1,5','1,5','1,6','1,7','z6','e,n,x','e,n,z15','1,2','1,5','1,6','1,[2],7','e,n,x','e,n,z15','l,w','l,z13,z28','z6','z42','1,2','1,5','1,6','1,7','e,n,z15','1,6','1,6','–','1,2,7','e,n,x','[1,2,7]','e,n,z15','e,n,x','1,5','z39','[z42]','–','–','[1,6]','[e,n,x]','–','1,5','e,n,z15','–','–','1,6','z64','–','[z57]','1,2','1,5','1,6','1,7','e,n,z15','l,w','z6','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z','[z6]','z35','1,2','1,5','1,5,7','1,6','1,7','e,n,x','e,n,z15','z6','z35','z53','1,2','1,5','1,5,7','e,n,z15','z42','1,5','e,n,x','e,n,z15','z6','1,7','1,5','e,n,x','z6','1,2','1,5','1,6','1,7','e,n,z15','l,w','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','1,2','1,5','1,5','1,6,[7]','e,n,x','e,n,x','e,n,x,z15','e,n,z15','l,w','z6','z39','z42','1,5','1,6','e,n,z15','z6','–','–','1,5','z42','–','–','1,7','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','z35','[1,2,7]','[z42]','1,6','e,n,z15','e,n,z15','z42','–','1,2','e,n,x','–','1,2','1,5,7','1,7','l,w','z35','1,[5],7','–','1,6','[z6]','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z39','z52','1,2','1,2','1,5','1,5','1,5','1,6','1,7','1,7','e,n,x','e,n,x','e,n,z15','e,n,z15','l,w','l,w','z6','z6','1,2','1,5','1,5','1,6','1,7','l,w','e,n,x','e,n,x','e,n,z15','z6','1,2','1,2','1,5','1,5','1,7','1,7','e,n,x','e,n,z15','z6','z6','1,2','1,2','1,5','1,5','1,6','1,7','e,n,x','e,n,z15','z6','e,n,z15','[e,n,x]','–','[e,n,x]','1,7','–','–','–','–','1,5','e,n,x','–','1,2','1,5','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','l,w','z6','1,2','1,5','1,5','1,7','e,n,x','e,n,z15','z35','1,2','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z6','1,2','z6','1,5','e,n,z15','e,n,x','1,2','1,5','1,5','1,6','1,7','1,7','e,n,z15','e,n,z15','l,w','l,w','z','z6','1,2','1,2','1,5','1,5','1,6','1,6','1,7','1,7','e,n,x','e,n,x','e,n,z15','z6','l,w','l,w','1,5','1,5','1,7','e,n,z15','e,n,z15','l,w','z6','1,7','1,7','e,n,x','[e,n,z15]','l,w','[z6]','e,n,x','–','–','–','1,2','1,2','1,5','1,5','1,7','e,n,x','e,n,x','e,n,z15','e,n,z15','l,w','z6','z6','1,5','e,n,x','[e,n,z15]','[e,n,z15]','e,n,x','1,2','1,2','z6','–','z6','1,2','1,5','1,5','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z39','z42','1,2','1,5','1,7','e,n,x','e,n,z15','z6','z39','1,5','1,6','1,7','e,n,z15','z6','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z6','z35','z39','–','1,2','1,5','1,7','e,n,z15','1,[5],7','1,6','–','–','–','–','[1,5,7]','e,n,x','z39','–','–','–','–','e,n,x','1,5','[e,n,x]','–','–','e,n,x','[1,2]','1,5','z39','1,5','1,5','1,6','1,2','1,5','1,5','1,7','e,n,x','e,n,x','e,n,z15','z39','1,5','e,n,x','1,5','1,7','e,n,x','1,5','1,6','1,5','e,n,x','e,n,z15','1,5','1,7','e,n,x','e,n,z15','z6','1,7','z39','1,2','1,5','1,6','1,7','e,n,z15','z6','z39','[1,7]','–','z67','1,5','z6','e,n,z15','1,5','e,n,x','z6','–','1,7','1,5','1,[5],7','–','1,2','1,6','e,n,x','e,n,z15','1,2','1,5','1,6','e,n,x','e,n,x','l,w','z6','z35','1,7','z6','1,5','1,6','1,7','e,n,z15','z6','z35','1,2','1,5','z6','1,5,7','–','–','1,2,7','[e,n,x]','–','–','e,n,x','–','e,n,z15','z6','1,2','1,5','1,6','z35','1,5','1,6','e,n,x','–','e,n,x','e,n,z15','1,7','e,n,z15','z6','1,2','1,7','e,n,x','l,w','z6','1,5','e,n,x','e,n,z15','z39','–','z39','–','1,7','e,n,x','e,n,z15','z6','z6','z39','–','1,5','e,n,z15','z6','–','1,7','–','z6','z39','z39','e,n,x','z39','z39','1,5','1,5','e,n,x','z39','1,2','1,5','1,7','e,n,x','e,n,x','e,n,z15','l,v','z39','z87','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','z35','z39','1,2','1,5','1,6','1,7','e,n,z15','l,w','z6','1,2','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','z35','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','1,7','[1,6]','e,n,x','–','–','[1,5]','–','1,5','–','1,5','[1,6]','e,n,x','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','1,2','1,5','1,7','e,n,x','e,n,z15','l,w','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z6','z35','z6','1,2','1,5','e,n,z15','z6','1,5','1,5','1,7','e,n,x','e,n,x','z39','1,5','1,7','e,n,z15','z6','1,2','1,6','1,2','1,5','1,7','e,n,x','e,n,z15','l,w','z6','1,5','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','z39','[1,7]','z6','–','–','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z6','e,n,x','[e,n,x]','–','1,5','1,6','1,7','e,n,x,z15','l,w','z6','–','1,2','1,6','[z6]','z42','1,[5],7','1,[2],5','1,7','e,n,x,z15','1,6','1,5','1,7','e,n,z15','l,w','1,5','1,6','e,n,x','e,n,z15','l,w','z6','1,5','1,6','1,7','1,2','1,5','1,7','e,n,z15','l,w','z6','1,5','1,6','1,7','e,n,z15','l,w','–','1,7','e,n,z15','–','–','–','1,2','1,5','1,6','e,n,x','e,n,z15','l,w','z6','1,5','1,7','1,5','1,7','e,n,z15','1,5','1,2','1,6','1,2','1,5','e,n,z15','1,7','e,n,x','e,n,z15','l,w','1,5','1,7','e,n,z15','e,n,x','l,w','1,5','1,6','1,7','e,n,z15','l,w','z6','–','–','1,5','e,n,z15','[z6]','1,5','[e,n,z15]','e,n,z15','z35','1,(2),7','z6','–','e,n,x','1,2','1,5','1,5','1,6','1,7','e,n,z15','e,n,z15','l,z13,z28','z6','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z6','1,5','1,6','e,n,z15','e,n,x','1,2','1,5','[e,n,x]','z6','1,2','1,6','l,z13,z28','1,5','–','z39','–','–','e,n,x','1,2','1,5','1,6','e,n,x','z','1,2','1,5','1,6','1,7','e,n,x,[z15]','l,w','l,z13,z28','z53','1,2','1,5','e,n,x','e,n,z15','z','z53','1,5','1,2','1,5','e,n,x','e,n,z15','e,n,x','1,5','e,n,x','e,n,z15','e,n,x,z15','l,z13,z28','1,5','1,7','e,n,x','e,n,x,z15','1,5','1,7','e,n,x','e,n,z15','z6','z39','1,6','e,n,z15','–','–','–','–','1,2','1,5','e,n,x','–','1,2','e,n,x','1,5','1,5','1,5','1,6','e,n,x','l,w','l,z13,z28','z42','1,5','1,5','1,6','[1,5]','1,6','1,7','e,n,x','e,n,x','e,n,z15','e,n,z15','l,w','z42','e,n,x,z15','e,n,z15','1,2','1,5','1,5','1,6','1,6','1,7','e,n,x','e,n,x','e,n,z15','e,n,z15','l,w','z6','1,5','1,6','l,w','1,[5],7','e,n,x','–','[e,n,z15]','z','–','[1,5]','z42','1,5','[e,n,x]','z42','–','e,n,x','–','[1,5]','z6','1,5','e,n,x','z42','–','–','1,5','e,n,x','z42','z42','–','–','1,5','1,7','e,n,z15','l,w','z6','1,5','1,6','–','e,n,z15','z41','1,5','1,5,7','1,6','e,n,z15','e,n,x','e,n,z15','1,5','1,5','z6','z42','–','1,6','e,n,z15','1,6','1,7','l,w','z6','1,5','1,5','1,5','1,5','1,6','1,6','1,7','z6','z6','e,n,z15','l,w','z42','–','[e,n,z15]','–','–','–','–','–','1,5','z6','e,n,z15','l,w','1,5','e,n,x','1,5','e,n,x','–','–','1,6','e,n,z15','1,6','e,n,z15','–','–','–','1,7','1,5,7','e,n,z15','1,[5],7','1,6','1,5','1,5','e,n,x','e,n,z15','1,2','1,5','1,7','e,n,x','1,5','1,6','e,n,x','l,w','1,5','1,5','1,6','1,7','1,7','e,n,x','e,n,x','e,n,z15','l,z13,z28','z6','1,5','1,5','1,6','1,7','–','–','–','–','–','1,5','e,n,x','–','1,5','1,7','e,n,z15','e,n,z15','1,5','1,6','e,n,x','e,n,x','z','z6','z53','1,7','1,7','e,n,x','e,n,z15','z','z35','z53','z88','1,6','e,n,x','1,5','z','e,n,z15','1,7','1,7','e,n,x','e,n,z15','1,5','1,6','e,n,x','e,n,x','e,n,z15','l,z13,z28','l,z13,z28','1,5','[e,n,z15]','–','–','–','1,2','1,(2),7','1,5','1,6','e,n,x,z15','z','z6','z6','z53','1,5','–','[e,n,x]','1,6','e,n,x,z15','z35','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','z6','1,2','1,2,5','1,5','1,6','e,n,x','e,n,x','e,n,z15','l,w','z6','z39','z42','1,5','1,6','e,n,x','e,n,z15','l,w','1,2','1,5','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','1,2','1,5','1,6','e,n,x','l,w','z6','1,(5),7','1,6','–','–','–','[1,5]','[e,n,x]','–','e,n,x','e,n,z15','[z42]','–','1,5','1,6','1,7','e,n,z15','l,w','z6','z35','1,2','1,6','e,n,x','e,n,z15','e,n,x,z15','z','z35','z53','1,5,7','1,6','1,7','e,n,x','e,n,z15','z','z6','z35','z53','1,5','1,6','e,n,x','e,n,z15','z6','z6','e,n,z15','[1,5]','1,6','z6','e,n,x','z42','1,5','1,6','1,7','z6','e,n,x','e,n,z15','1,5','1,6','e,n,x','e,n,z15','1,5','1,6','1,7','e,n,x','e,n,x','e,n,[x],z15','z6','z42','[1,6]','z6','–','–','–','–','–','–','1,6','1,5','1,7','1,6','e,n,x,z15','e,n,z15','z6','–','1,5','e,n,x','1,5','1,6','e,n,z15','e,n,x','–','e,n,z15','–','[1,6]','1,(5),7','1,6','z35','1,2','1,5','e,n,x','1,2','1,5','e,n,x','e,n,x,z15','e,n,z15','z6','1,5','1,6','z39','–','1,5','e,n,z15','l,w','z6','1,2','1,6','1,[5],7','–','–','–','[e,n,x,z15]','z39','–','–','–','1,7','z35','1,5','1,6','1,7','e,n,x','e,n,z15','z','–','1,2','1,5','1,7','e,n,x','e,n,x,z15','z35','e,n,x','1,2','z','–','e,n,x','e,n,z15','l,w','1,2','1,5','1,6','1,7','e,n,z15','l,w','–','–','–','–','e,n,x','e,n,x,z15','z','–','–','–','–','1,6','–','–','e,n,z15','1,5','1,7','e,n,x','1,5','1,7','–','–','–','–','1,5','–','1,5','z53','z54','e,n,x,z15','e,n,z15','z','z53','e,n,z15','1,5','z','1,7','e,n,x,z15','–','[1,5]','z64','–','–','–','–','–','e,n,x,z15','1,5','z6','z6','–','–','–','–','–','1,6','[1,5]','1,5','1,6','e,n,x','1,6','e,n,x','e,n,x','e,n,x,z15','1,5','e,n,x','z6','e,n,x','–','–','–','–','–','1,2','1,5,7','e,n,x,z15','1,6','e,n,x,z15','z','1,(2),5','z','z57','–','e,n,z15','–','z','1,2','e,n,x','–','–','–','–','–','–','–','e,n,x,z15','z','[z6]','z53','–','e,n,z15','–','–','–','e,n,x,z15','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','z6','1,5','1,6','1,7','e,n,x','e,n,x','e,n,z15','l,w','z6','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z6','1,5','1,7','e,n,z15','1,7','z87','–','1,6','[e,n,z15]','1,5','[e,n,z15]','e,n,x','z39','e,n,x','–','[e,n,x]','[e,n,z15]','z39','1,6','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','1,2','1,5','1,6','1,7','1,7','e,n,x','e,n,z15','1,5','e,n,x','e,n,z15','1,6','1,5','e,n,z15','1,5','e,n,x','e,n,x','1,6','e,n,x','e,n,z15','l,w','1,5','1,7','e,n,z15','1,2','1,5','1,7','e,n,x','e,n,z15','l,w','z6','1,5','1,5','1,6','1,7','e,n,z15','z39','1,5','1,6','[e,n,z15]','–','1,2','1,5','1,6','1,7','e,n,x','e,n,z15','l,w','z','–','1,5','e,n,x','1,6','1,7','e,n,z15','e,n,z15','1,2','1,5','e,n,x','e,n,z15','z39','1,2','1,5','1,7','e,n,x','e,n,z15','l,z13,z28','z6','z6','1,7','e,n,z15','z39','1,5','1,2','1,5','–','e,n,x','–','–','–','–','–','–','1,2','1,5','e,n,z15','l,w','1,2','1,5','1,6','e,n,x','e,n,[x],z15','e,n,x,z15','l,z13,z28','1,2','1,5','1,6','e,n,z15','z6','1,2','1,5','1,6','1,2','1,5','1,6','e,n,x','e,n,z15','e,n,x,z15','–','–','1,6','1,2','1,5','e,n,z15','e,n,x,z15','–','1,6','e,n,z15','–','1,7','e,n,z15','–','1,2','1,5','[e,n,z15]','1,5','l,w','z6','–','–','–','–','–','1,5','z42','–','–','e,n,x','–','–','e,n,x,z15','e,n,z15','l,w','z','z6','z35','z53','e,n,z15','e,n,x,z15','z','z35','z53','1,5,7','e,n,x,z15','z35','–','e,n,x,z15','l,w','z','z35','z61','l,w','–','–','–','–','–','1,5','l,w','z6','z35','e,n,x','–','–','–','–','1,5,7','e,n,x,z15','z','z35','e,n,z15','1,2','e,n,z15','1,5','1,6','1,5','[1,5]','z39','1,2','1,5','–','–','–','–','–','1,2','1,5','z','z53','1,2','1,5','1,6','e,n,x','e,n,x,z15','z','z53','1,5,7','z35','–','z55','1,2','1,5','e,n,z15','e,n,x','z','z35','[z53]','1,5','1,5,7','1,6','e,n,x,z15','z','z35','1,5','1,6','e,n,x','–','1,2','–','–','–','[1,5]','z','z53','e,n,z15','–','z35','z53','–','[z53]','z39','1,2','l,w','e,n,x','1,5','1,2','1,5','1,7','–','[e,n,x]','1,5','e,n,x','e,n,z15','1,5','e,n,x','1,5','e,n,x','e,n,z15','e,n,x','z39','1,5','1,2','1,5','e,n,z15','–','–','z6','1,6','1,7','1,5','1,7','z6','z39','1,5','1,7','e,n,x','e,n,z15','z6','–','e,n,x,z15','z6','z39','1,5','–','1,2','1,[5],7','1,6','–','e,n,x','e,n,x','[1,5]','e,n,x,z15','z39','–','z42','–','e,n,x,z15','–','e,n,x','z39','z42','–','1,5,7','1,5','1,6','e,n,x','e,n,x,z15','z','z6','z53','1,2','1,6','e,n,x','z','z53','[1,2]','e,n,x','1,2','1,5','1,5','1,6','z39','z53','1,5','1,5','e,n,x','z6','z6','z39','z42','–','–','z39','–','–','–','–','1,5','z6','e,n,x','z35','–','–','e,n,x,z15','e,n,z15','–','–','1,5','1,6','1,6','1,7','1,2','1,6','1,(5),7','z6','–','–','1,5','1,6','1,7','z6','e,n,z15','–','e,n,x,z15','z6','1,5','z6','z6','–','–','1,6','1,5','1,6','1,6','1,6','z6','z35','e,n,x,z15','1,5','1,7','–','1,5','1,6','1,7','[e,n,z15]','–','–','–','1,5','–','–','1,2','1,5','1,6','e,n,x','e,n,x,z15','z6','–','–','–','1,6','–','–','–','–','1,6','e,n,z15','1,5','1,6','e,n,x,z15','e,n,z15','l,w','z6','z6','e,n,z15','z6','z6','z6','1,6','–','1,5','–','–','[e,n,x,z15]','–','l,w','1,6','z6','1,6','e,n,x,z15','z','z6','z35','1,5,7','1,6,7','e,n,x,z15','e,n,x,z15','e,n,z15','z','z53','e,n,x','z35','[z6]','e,n,z15','l,w','z','z53','–','–','e,n,z15','z6','1,5','1,6','e,n,x,z15','e,n,z15','z6','1,6','e,n,z15','–','–','–','–','1,6','1,2','e,n,x,z15','e,n,x,z15','z','z6','z6','z35','z67','[1,5]','–','1,6','e,n,z15','z6','–','–','z6','–','z','1,5','1,2','1,5','1,5','z6','–','e,n,z15','z42','1,5','1,5','e,n,x,z15','z39','z42','1,(5),7','1,6','–','[z42]','[1,5]','–','–','–','e,n,x','1,2','e,n,x','1,2','1,5','z','z53','[z44]','–','1,5','–','e,n,x,z15','z','z53','1,2','1,5','1,5','1,5','e,n,z15','[1,2]','–','–','–','–','–','–','1,5','e,n,x','z42','–','–','–','1,6','–','–','–','1,2','[1,5,7]','z53','–','1,5','e,n,x','l,w','1,6','e,n,x','e,n,z15','1,5','e,n,x','e,n,z15','1,5','e,n,x','e,n,z15','z35','1,6','–','[1,6]','[1,7]','z42','1,5','–','–','–','z42','e,n,z15','e,n,z15','1,5','1,5','e,n,z15','–','1,5','l,w','e,n,x','e,n,z15','–','e,n,z15','–','–','–','–','–','–','–','–','–','1,7','e,n,x','e,n,x','–','–','–','–','e,n,x,z15','–','e,n,x','e,n,z15','z10','1,5','1,6','e,n,z15','e,n,x','1,6','e,n,x','e,n,z15','–','–','1,5','e,n,x','e,n,x,z15','–','–','–','1,5','–','1,5','1,7','e,n,z15','[e,n,z15]','e,n,z15','1,5','e,n,z15','z39','–','–','–','–','–','–','–','1,5','e,n,x','z42','–','1,5','–','–','–','1,5','e,n,x,z15','1,2','1,5','e,n,x,z15','z6','z6','1,5,7','1,6','e,n,x,z15','z','z6','z35','1,5','e,n,x','e,n,x,z15','z39','1,6','–','–','–','e,n,x','–','e,n,z15','e,n,x,z15','e,n,z15','z','z35','z53','1,2','1,5','1,5,7','1,6','e,n,x,z15','e,n,z15','z','z35','z53','1,[5],7','e,n,x','e,n,x,z15','z','z35','z53','z57','–','e,n,z15','e,n,x,z15','1,2','1,5,7','z','z35','z53','1,6','e,n,x','1,5','1,6','e,n,x,z15','z6','1,2','e,n,z15','1,6','z6','l,w','–','–','1,6','1,5,7','e,n,z15','z','z35','e,n,x,z15','–','–','z6','e,n,z15','–','–','–','1,5','1,5,7','e,n,x,z15','z','z35','–','1,5,7','z6','z39','z6','e,n,x,z15','–','z','1,2','z6','–','1,5','z6','–','–','–','z','z35','z53','z61','–','1,5,(7)','e,n,x,z15','e,n,x,z15','e,n,z15','z','z35','z39','z53','z57','1,5','1,5,(7)','z','1,5,7','e,n,x,z15','z','1,5','1,5','1,5,7','–','–','–','–','–','–','–','–','–','[1,5]','1,5','1,6','e,n,x','e,n,x,z15','z','–','–','–','z52','–','–','–','z81','–','–','e,n,x,z15','z','–','–','1,5,7','–','e,n,x','z6','–','1,5','–','1,7','[1,5]','–','–','e,n,x','z6','1,5,7','e,n,x,z15','z','z53','1,5,7','e,n,x','e,n,x,z15','z','z6','z35','z53','1,2','e,n,x,z15','z','z35','z57','z67','e,n,x,z15','z42','1,5,(7)','e,n,x,z15','z','z35','z53','1,6','e,n,x','z52','–','–','–','–','–','–','–','z','z6','z53','–','–','–','1,7','1,5,7','z35','z53','–','1,2','e,n,x','1,5','–','–','1,5','1,2','e,n,x','–','–','1,2','1,5','e,n,z15','z35','1,2','1,5','e,n,x','z','1,5','z6','z39','e,n,x','l,z13,z28','1,2','1,5','1,6','e,n,z15','–','–','–','–','1,5','e,n,x,z15','–','1,7','1,5','e,n,z15','1,5','e,n,x','k','1,5','e,n,x,z15','z39','1,6','1,7','–','–','1,5','e,n,x,z15','z35','z53','z53','1,5','z39','z52','1,5,7','1,5,7','1,5','1,5','z39','z42','–','–','z','e,n,x,z15','z','z35','z53','e,n,x,z15','z','z35','e,n,x','z6','z39','z','z35','z68','1,5','1,5,(7)','z6','–','–','–','–','–','z','z35','–','–','z35','z53','1,6','e,n,x','1,5','1,6','–','–','–','z6','1,5','1,5','1,5','–','e,n,x','e,n,z15','–','z39','[1,5]','–','1,7','z39','–','z6','–','–','e,n,x','–','z6','z42','1,7','e,n,z15','e,n,x,z15','z','1,5','z42','–','e,n,x,z15','z','e,n,x,z15','–','z','z42','e,n,x,z15','1,6','z6','1,5','z6','z6','e,n,x,z15','z53','z','e,n,x,z15','z35','1,5','z6','e,n,x,z15','z','z53','1,6','1,6','e,n,x,z15','z6','z53','e,n,x,z15','z','z35','e,n,x,z15','e,n,x,z15','z','z35','e,n,x,z15','z65','z','z35','z53','z','z53','z35','z6','–','z53','z57','–','–','–','z53','–','z6','[e,n,x,z15]','[z]','[z35]','z','z35','z53','z','e,n,x,z15','z','z35','z53','e,n,x','z','z35','z39','z53','e,n,x','–','1,5,[7]','z','z35','z53','1,5,(7)','z35','e,n,x,z15','z','z35','z53','1,5,(7)','z35','z53','1,5,7','z','z35','1,5,7','z','z35','z53','z35','–','1,5,7','z','z35','z53','–','–','–','–','–','–','z','–','–','–','1,5,7','z','z53','–','e,n,x,z15','z','z','z35','z53','e,n,x,z15','z','z35','z53','z35','e,n,x,z15','z','e,n,x,z15','z','z35','z53','1,6','–','–','–','–','–','1,5','1,5','1,2','1,2','e,n,x','–','1,6','1,6','1,5','[1,5]','z6','1,7','1,7','1,5','1,5','1,5','–','l,w','1,6','l,w','l,w','1,5','1,5','1,2','1,7','[z57]','1,5','1,5','1,7','1,2','–','1,5','e,n,x','[1,2]','z6','1,5','1,2','1,2','z6','–','–']
wklm_other_H = ['','','','','','','','','','','','','','','[z5],[z33]','','','','','','','','','','','','','','','','','','','','','','','','[e,h]','','','','','[R1…]','','','','','','','','','[z27],[z45]','z42','','','','','','','z43','','','','','','','','','','','','','','','','','','','','','','','[z40]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','[z59]','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','[R1…]','','','','','[z55]','','','','','','','','','','','','','','','','','','','','','','','[z42]','','','','[R1…],[z37],[z45],[z49]','','','','','','','','','[z67]','[z47],[z50]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z37],[z45]','','','','','','','','','','','[z82]','','[z82]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z67]','','[z58]','','','','','','z42','','[z67],[z78]','','','[R1…]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z58]','','','','','','','','','','','','','','','z42','','','','','','[R1…]','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','[z45]','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z66]','','','','','','','','','','','[z42]','','','','','','','','','','','','','','','','','','','','[R1…]','d','[z40]','','','','','','','','','','[R1…]','','z42','','','','','','','','','','','','','','','','','','','[R1…]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','[z45]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z27]','[z48]','[z64]','','','','','','','','','','','','[z37]','','','','','','','','','','','','','','','','','','','','','','','[d],[z77]','','','','','[z45]','','','','','','','','','','','','','','','','','[z45]','','','','','','','','','','','','','','','','','','','','','[z49]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z37]','','','','','','','','','','','','','[z49]','','','','','','','','','','','[z27],[z34],[z37],[z43],[z45],[z46],[z82]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z37]','','','','','','','','','','','','','','','d','','z42','','','','','','','','','','','','','','[r]','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z56]','','','','','','','','','','','','','','','','','','','','','[z83]','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','[z37]','','','','','','','','','','','','','[z79]','','','','','','','','','','','','','','','','','','','','','z39','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','[z44],[z59]','','','','','','[z43]','','','','','','','','','','','','','','','','','','[z37],[z43]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z56],[z90]','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z82]','','','','','','','','','z42','[z37]','','','[z42]','','','','','','','','','','','','','','','','','','','','','','','[z45]','','','','[z61]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','[z56]','','','','','','','','','','','','','','','','','','','','','','','','','','[z50]','','','','','','','','[z45],[z82]','','','','','','','','','','','','','','','','','','','','[z33],[z49]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z80],[z90]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z33]','','','','','','','','','','','','','','','','[z59]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z58]','','[z27]','','','','','','','','','','','','','','','','','','','','','','','[z50]','','[z50]','[z67]','','','','','','','','[z37],[z45]','','','','','','','','','','','','','','','','','','','','','','[z58]','','','','','','','','','','','','','','','','','','','','','','','[z56]','','','','','','','','','[z54]','','','','','[z57]','','','','','','','','','','','','','','','','','[z47],[z50],[z76]','','','','','','','','','','','','','','','[z48]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z57','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z76]','','','','','','','','','','','','','','','[z50]','','','','','','','','1,6','','','','','','','','','[z60]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z56]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z45]','','','','','','','','','','','','','','','','','[z78]','[z57]','','','','','[z58]','','','','','','','','','','[z50]','','','','[z57],[z84]','','','','[z58]','','','','','[z84]','[z50]','','','','','','','','','','','','','','[z70],[z74],[z77],[z90]','','','','','','','','','','[z45],[z58]','','','','','','','','','','','','','','','','','z54','','','','','[z90]','','','','','','','','','','[z58]','','','','','','[z72]','[z57]','','','','','','','','','[z75]','','','','','[z47],[z50],[z89]','','[z50]','','','[z58]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z82]','','','','','','','z42','','','','','','z42','','[z50],[z57],[z68],[z86]','','','','','','','','','','z42','','','','[z67]','[z58]','','','','','','','','','','','','[z56]','z42','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[R1…]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z70],[z90]','','','','','','','','','','','z53','','','','','','','','','','','','','','[z47],[z57],[z70],[z73]','','','','','[z50]','','','','','','','[z84]','','','','','','','','','','','','','','','','','[R1…]','','[z50]','[z50]','[z50]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z57]','','','','','','[z47],[z50]','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','','[z45]','','','[z64]','[z64]','','[z45],[z82]','','[d],[z77]','[d],[z77]','','[z49]','[z49]','[z82]','','','','','[z48]','[z48]','','','','','','','[z5],[z33]','','','','','','','','z27','','[z37]','[z37]']
wklm_group = ['O:2','O:2','O:2','O:2','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:4','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:7','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:8','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:9,46,27','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,?','O:3,?','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:1,3,19','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:11','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:13','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:6,14','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:16','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:17','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:18','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:21','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:28','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:30','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:35','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:38','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:39','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:40','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:41','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:42','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:43','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:44','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:45','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:47','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:48','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:50','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:51','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:52','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:53','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:54','O:55','O:56','O:56','O:56','O:56','O:56','O:56','O:56','O:56','O:56','O:56','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:57','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:58','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:59','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:60','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:61','O:62','O:62','O:62','O:62','O:62','O:63','O:63','O:63','O:63','O:63','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:65','O:66','O:66','O:66','O:66','O:66','O:7','O:9','O:67','O:3,10','O:7','O:3,10','O:3,10','O:3,10','O:3,10','O:18','O:7','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:7','O:7','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:7','O:3,10','O:3,10','O:3,10','O:4','O:7','O:3,10','O:3,10','O:4','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10','O:3,10']
wklm_old_group = ['A','A','A','A','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','B','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C1','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','C2–C3','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D1','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D2','D3','D3','D3','D3','D3','D3','D3','D3','D3','D3','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','E1','–','–','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','E4','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','F','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','G','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','H','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','I','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','J','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','K','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','L','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','M','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','O','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','P','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','Q','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','R','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','S','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','T','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','U','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','V','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','W','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','X','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','Z','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','–','C1','D1','–','E1','C1','E1','E1','E1','E1','K','C1','E1','E1','E1','E1','E1','C1','C1','E1','E1','E1','E1','E1','E1','E1','C1','E1','E1','E1','B','C1','E1','E1','B','E1','E1','E1','E1','E1','E1','E1']

# Variations of Paratyphi B var. L(+) tartrate (= d-tartrate)+
paratyphi_b_var = ['l(+) tartrate (= d–tartrate)+', 'l+', 'l(+)', 'tartrate+', 'dt+', 'dt(+)'] 

current_subsp = ['i','ii','iiia','iiib','iv','v','vi']


########################
#### WKLM DataFrame ####
########################

wklm_cols = ['Name','Std_Name','Formula','Std_Formula','Species','Subspecies',
             'O','P1','P2','other_H','Group','Old_Group']
wklm_df = pd.DataFrame({
            'Name':        wklm_name,
            'Std_Name':    std_wklm_name,     # standardized name for lookups
            'Formula':     wklm_formula,
            'Std_Formula': std_wklm_formula,  # standardized formula for lookups
            'Species':     wklm_sp,
            'Subspecies':  wklm_subsp,
            'O' :          wklm_O,
            'P1' :         wklm_P1,
            'P2' :         wklm_P2,
            'other_H' :    wklm_other_H,
            'Group' :      wklm_group,
            'Old_Group' :  wklm_old_group},
            columns = wklm_cols)
            

# Regex patterns for formula formatting
subsp_pattern = '^[a-zA-Z]+\\s+.*:.*'    # I 9,12:l,v:1,7:[z40] 
no_subsp_pattern = '^\\S*:.*'            # 9,12:l,v:1,7:[z40]
antigens = ['O','P1','P2','other_H']
missing_antigen = '\u2013'


#-------------------------------------------
# Classes
#-------------------------------------------


class WKLMSerovar(object):

    def __init__(self,input):

        """Queries input against the WKLM repository and creates a custom Pandas Series.
        Args:
            input(str):   a serovar designation
        Attributes:
            The input argument is stored as an attribute.
            name(str) :   serovar name
            formula(str): antigenic formula
            meta(Series): a pandas Series formatted like the wklm_df repository 
                          with an additional index label ('Input')   
        """
        
        self.input = standardize_unicode(input)
        self.name = ''
        self.formula = ''
        
        # Create an empty Series formatted like wklm_df
        # Name, Std_Name, Formula, Std_Formula, Species, Subspecies, O, P1, P2, other_H, Group, Old_Group        
        self.meta = pd.Series(dtype='object').reindex_like(wklm_df.iloc[0].squeeze())
                        
        input = prep(standardize_input(self.input))
          
        if not wklm_df.loc[wklm_df.Std_Name == input].empty:
            self.meta = wklm_df.loc[wklm_df.Std_Name == input].squeeze()
        elif input in wklm_old_to_new:
            self.meta = wklm_df.loc[wklm_df.Std_Name == prep(wklm_old_to_new[input])].squeeze()
        elif not wklm_df.loc[wklm_df.Std_Formula == input].empty:
            self.meta = wklm_df.loc[wklm_df.Std_Formula == input].squeeze()
            # There are a couple of serovars with identical formulas for which a series should be returned instead of a dataframe
            # Miami or Sendai, Choleraesuis or Typhisuis
            if isinstance(self.meta, pd.DataFrame):
                self.meta = self.meta.iloc[-1]
        
        # Handle special named variants         
        special_cases = ['Stanleyville','Amersfoort','Livingstone','Rissen','Oranienburg',
                         'Lille','Cerro','Gdansk']
        if self.name in special_cases:
            name = 'NA'  
            if self.input == 'Stanleyville var. 27+' or (self.name == 'Stanleyville' and ',27:' in self.input):
                name = 'Stanleyville var. 27+' 
            elif self.input == 'Amersfoort var. 14+' or (self.name == 'Amersfoort' and ',14:' in self.input):
                name = 'Amersfoort var. 14+' 
            elif self.input == 'Livingstone var. 14+' or (self.name == 'Livingstone' and ',14:' in self.input):
                name = 'Livingstone var. 14+' 
            elif self.input == 'Rissen var. 14+' or (self.name == 'Rissen' and ',14:' in self.input):
                name = 'Rissen var. 14+' 
            elif self.input == 'Oranienburg var. 14+' or (self.name == 'Oranienburg' and ',14:' in self.input):
                name = 'Oranienburg var. 14+' 
            elif self.input == 'Lille var. 14+' or (self.name == 'Lille' and ',14:' in self.input):
                name = 'Lille var. 14+' 
            elif self.input == 'Cerro var. 14+' or (self.name == 'Cerro' and ',14,' in self.input):
                name = 'Cerro var. 14+' 
            elif self.input == 'Gdansk var. 14+' or (self.name == 'Gdansk' and ',14,' in self.input):
                name = 'Gdansk var. 14+'
            if name != 'NA':     
                self.meta = wklm_df.loc[wklm_df.Std_Name == prep(name)].squeeze()  
            
        if self.meta.isnull().values.all():
            if not is_name(input):
                formula = standardize_input(self.input)     
                self.meta.Formula = formula
                self.meta.Std_Formula = prep(formula)
                self.meta.Subspecies, self.meta.O, self.meta.P1, \
                    self.meta.P2, self.meta.other_H = formula_to_fields(formula)
            else:
                logging.error("The serovar '{}' was not recognized.".format(self.input))
        
        self.meta['Input'] = self.input
        if pd.isna(self.meta['Species']) and not pd.isna(self.meta['Subspecies']):
            self.meta['Species'] = 'bongori' if self.meta['Subspecies'] == 'V' else 'enterica'

        self.name = self.meta.Name
        self.formula = self.meta.Formula       


class SeroComp(object):

    def __init__(self,subj,query):
        
        """Compares two WKLMSerovar objects for congruence.
        Args:
            subj(WKLMSerovar)
            query(WKLMSerovar)
        Attributes:
            The input arguments are stored as attributes.
            comp_df(pd DataFrame): the pd Series attributes of the input objects 
                                   combined into a pd DataFrame      
            result(str):           the outcome of the comparison 
                                   ('exact','congruent','minimally congruent', 
                                    'incongruent', or 'invalid input')
        Functions:
            print_results(): print formatted comparison results
            is_exact():      
            is_congruent(): 
            is_minimally_congruent():    
        """
        
        self.subj = subj
        self.query = query        
        self.result = ''
        
        self.comp_df = pd.DataFrame([subj.meta.rename('subj'),query.meta.rename('query')])
        
        if pd.isna(self.comp_df.Formula).any():
            self.result = 'invalid input'
        elif ((pd.isna(self.comp_df.loc['subj','Subspecies']) and all_antigens_missing(self.subj)) \
            or (pd.isna(self.comp_df.loc['query','Subspecies']) and all_antigens_missing(self.query))):
            self.result = 'invalid input'   
        elif self.is_exact():
            self.result = 'exact'
        elif self.is_congruent():
            self.result = 'congruent'
        elif self.is_minimally_congruent():
            self.result = 'minimally congruent'
        else:
            self.result = 'incongruent'    

             
    def print_results(self):
        results_cols = ['Subj_Input','Subj_Name','Subj_Formula',
                        'Query_Input','Query_Name','Query_Formula','Result']
        pd.DataFrame({'Subj_Input':[self.subj.input],'Subj_Name':[self.subj.name],
                      'Subj_Formula':[self.subj.formula],'Query_Input':[self.query.input],
                      'Query_Name':[self.query.name],'Query_Formula':[self.query.formula],
                      'Result':[self.result]},columns=results_cols).to_csv(sys.stdout, 
                      index=False, header=False, sep='\t', na_rep='NA')


    def is_exact(self):
        """Compares serovar names and formulas in order to determine if there are any 
           exact matches, allowing for the following possibilities:
            1. The two inputs provided are the identical string.
            2. Every antigenic factor (required or optional) matches.
            3. There are no antigenic factors, but the subspecies designations match.   
        Returns:
            (bool) 
        """
        
        if (prep(self.subj.input) == prep(self.query.input) \
            or self.comp_df.Formula.nunique() == 1 \
            or (all_antigens_missing(self.subj) and all_antigens_missing(self.query) \
                and self.comp_df.loc['subj','Subspecies'] == self.comp_df.loc['query','Subspecies'])):
            return True
        else:
            return False    


    def is_congruent(self):
        """Compares antigenic formulas and determines whether there are congruent matches, 
           as follows:
            1. The two inputs are an exact match.
            2. All required antigenic factors must match. Note - at least one formula 
               must match a serovar in the database in order to differentiate required 
               and optional factors.
        Returns:
            (bool) 
        """

        if self.is_exact():
            return True

        result = False
                                        
        if not pd.isna(self.comp_df.Subspecies).all():
            if pd.isna(self.comp_df.Subspecies).any():  
                return result
            elif self.comp_df.loc['subj','Subspecies'] == self.comp_df.loc['query','Subspecies']:
                result = True
            else:
                return result        
        
        cols = antigens
        for col in cols:                
            if not pd.isna(self.comp_df[col]).any():
                s = self.comp_df.loc['subj',col]
                q = self.comp_df.loc['query',col]
                result = min_factors(s) == min_factors(q)
                if result == False:
                    result = is_opt_subset(s,q)
                    if result == False:
                        break
            elif pd.isna(self.comp_df[col]).any():
                result = False
                break    
                
        return result


    def is_minimally_congruent(self):
        """Compares antigenic formulas and determines whether they are minimally congruent 
           (no direct conflicts), by determining if either is a formal subset of the other.  
        Returns:
            (bool) 
        """
                
        if self.is_exact():
            return True
        
        result = False            
            
        if not pd.isna(self.comp_df.Subspecies).any():             
            if self.comp_df.loc['subj','Subspecies'] == self.comp_df.loc['query','Subspecies']:
                result = True
            else:
                return result        
         
        cols = antigens       
        # Test if i1 is a proper subset of i2
        for i1,i2 in [['subj','query'],['query','subj']]:
            if pd.isna(self.comp_df.Subspecies).any() and not pd.isna(self.comp_df.Subspecies).all():             
                # A profile missing the subspecies can be a subset of a profile with a subspecies,
                # but not vice versa
                if pd.isna(self.comp_df.loc[i1,'Subspecies']): 
                    result = True
                else: 
                    continue  
            for col in cols:
                if not pd.isna(self.comp_df[col]).any():            
                    factors = [self.comp_df.loc[i1,col],self.comp_df.loc[i2,col]]
                    result = is_min_subset(factors[0],factors[1])
                    if result == False:
                        break             
            if result == True:
                break                
        return result
    

class SeroClust(object):

    def __init__(self,clust_id,wklm_objs,sort_by=None,header=True):
        
        """Determines the most abundant serovar(s) from a list of WKLMSerovar objects
        Args:
            clust_id(str):               a cluster id
            wklm_objs(list):             a list of WKLMSerovar objects
            sort_by(str or list of str): ordered column(s) by which to sort results
                                         Options = e (exact), c (congruent), 
                                                   m (minimally congruent)
                                         Default = ['c','e','m']
            header(bool):                header parameter for print functions 
        Attributes:
            The input arguments are stored as attributes.
            clust_df(pd DataFrame):      the pd Series attributes of the wklm_objs 
                                           combined into a pd DataFrame
            metrics(pd DataFrame):       metrics for all serovars
            results(pd DataFrame):       select metrics for top serovar(s)
        Functions:
            print_serovars(): print formatted results - name and formula for top serovar(s)
            print_results():  print formatted results - select metrics for top serovar(s)
            print_metrics():  print formatted results - all metrics
                             
        """

        self.clust_id = clust_id
               
        sort_by_options = {'e':'Exact', 'c':'Congruent', 'm':'MinCon'}
        self.sort_by = sort_by
        
        if sort_by == None:
            sort_by = ['c','e','m']
        elif not all([s in sort_by_options.keys() for s in sort_by]):
            logging.error('A sort_by parameter was not recognized.')
            raise InvalidInput('A sort_by parameter was not recognized.')       
        self.sort_by = [sort_by_options[s] for s in sort_by] 
        self.header = header
        
        if wklm_objs and isinstance(wklm_objs, list):
            self.wklm_objs = wklm_objs
        else:
            raise InvalidInput('A list of WKLMSerovar objects is expected.')  
                 
        self.clust_df = pd.DataFrame([obj.meta for obj in wklm_objs], index=range(0,len(wklm_objs)))
        self.metrics = pd.DataFrame()
        self.results = pd.DataFrame()
        
        sub_df      = self.clust_df[self.clust_df.Formula.notnull()]
        uniq_df     = sub_df.drop_duplicates('Formula').reset_index(drop=True)
        init_cts    = [len(sub_df[sub_df.Formula == n]) for n in uniq_df.Formula]
        n_seros     = len(self.clust_df)
        n_sub_seros = len(sub_df)
        n_uniq      = len(uniq_df)
        
        if not n_uniq:
            logging.error('Cluster {} - No valid serovars.'.format(self.clust_id))   
           
        comps = []
        for s1 in uniq_df.Formula:
            comps.append([SeroComp(WKLMSerovar(s1),WKLMSerovar(s2)).result for s2 in uniq_df.Formula])
        
        n     = {'exact': [], 'congruent': [], 'mincon': []} # counts
        p_sub = {'exact': [], 'congruent': [], 'mincon': []} # proportion of counts to nonmissing subset
        p_all = {'exact': [], 'congruent': [], 'mincon': []} # proportion of counts to all valid input
        p_exact, p_congruent, p_mincon = [], [], []
        
        for i, r in uniq_df.iterrows():
            s_exact, s_con, s_min_con = [0] * 3
                        
            for j, comp in enumerate(comps[i]):
                if j != i:
                    if comp == 'exact':
                        s_exact += init_cts[j]
                    elif comp == 'congruent':
                        s_con += init_cts[j]
                    elif comp == 'minimally congruent':
                        s_min_con += init_cts[j] 
            
            n['exact'].append(init_cts[i] + s_exact)
            p_all['exact'].append(round(n['exact'][i] / n_seros, 4))
            p_sub['exact'].append(round(n['exact'][i] / n_sub_seros, 4))
            p_exact.append(p_sub['exact'][i] if p_sub['exact'][i] == p_all['exact'][i] \
                          else '{}({})'.format(p_sub['exact'][i],p_all['exact'][i]))
            
            n['congruent'].append(n['exact'][i] + s_con)
            p_all['congruent'].append(round(n['congruent'][i] / n_seros, 4))
            p_sub['congruent'].append(round(n['congruent'][i] / n_sub_seros, 4))
            p_congruent.append(p_sub['congruent'][i] if p_sub['congruent'][i] == p_all['congruent'][i] \
                              else '{}({})'.format(p_sub['congruent'][i],p_all['congruent'][i]))
            
            n['mincon'].append(n['congruent'][i] + s_min_con)
            p_all['mincon'].append(round(n['mincon'][i] / n_seros, 4))
            p_sub['mincon'].append(round(n['mincon'][i] / n_sub_seros, 4))
            p_mincon.append(p_sub['mincon'][i] if p_sub['mincon'][i] == p_all['mincon'][i] \
                          else '{}({})'.format(p_sub['mincon'][i],p_all['mincon'][i]))
        
        metrics_cols = ['ClusterID','ClusterSize','Input','Name','Formula','Comps',
                        'N_Exact','N_Congruent','N_MinCon','P_Exact','P_Exact_sub',
                        'P_Exact_all','P_Congruent','P_Congruent_sub','P_Congruent_all',
                        'P_MinCon','P_MinCon_sub','P_MinCon_all']
        self.metrics = pd.DataFrame({
            'ClusterID':       [self.clust_id] * n_uniq if n_uniq else self.clust_id,
            'ClusterSize':     n_seros,
            'Input':           uniq_df.Input if n_uniq else np.nan,
            'Name':            uniq_df.Name if n_uniq else np.nan,    
            'Formula':         uniq_df.Formula if n_uniq else np.nan, 
            'Comps':           comps,                
            'N_Exact':         n['exact'],           
            'N_Congruent':     n['congruent'],       
            'N_MinCon':        n['mincon'],          
            'P_Exact':         p_exact,              
            'P_Exact_sub':     p_sub['exact'],       
            'P_Exact_all':     p_all['exact'],  
            'P_Congruent':     p_congruent,              
            'P_Congruent_sub': p_sub['congruent'],   
            'P_Congruent_all': p_all['congruent'], 
            'P_MinCon':        p_mincon,              
            'P_MinCon_sub':    p_sub['mincon'],      
            'P_MinCon_all':    p_all['mincon']},
            columns = metrics_cols)     
                                        
        sort_by_cols = ['N_{}'.format(s) for s in self.sort_by]                 
        self.metrics.sort_values(by=sort_by_cols + ['Input','Name','Formula'], inplace=True, 
            ascending=([False] * len(sort_by_cols) + [True] * 3))

        
        # Specify cols for results DataFrame
        r_cols = ['ClusterID','ClusterSize','Input','Name','Formula',
                  'P_Exact','P_Congruent','P_MinCon']
                
        # Determine most abundant serovar(s)
        top_seros = self.metrics[sort_by_cols].iloc[:1].merge(
                        self.metrics[sort_by_cols],indicator=True,how='outer')            
        self.results = self.metrics[r_cols].iloc[:len(top_seros[top_seros['_merge'] == 'both'])]
        
        # If ties, attempt to condense serovar entries with duplicate serovar names    
        if len(self.results) > 1:
            if any(self.results.duplicated(['Name'],keep=False)):
                all_dups = self.results[self.results.duplicated(['Name'],keep=False)]           
                condensed_results = self.results.drop_duplicates(['Name'],keep=False)          
                for name in pd.unique(all_dups['Name']):
                    dup = all_dups.loc[all_dups['Name'] == name]
                    if not dup.loc[dup['Name'] == dup['Input']].empty:
                        condensed_results = condensed_results.append(dup.loc[dup['Name'] == dup['Input']])
                    else:
                        condensed_results = condensed_results.append(dup)     
                self.results = condensed_results       
         
        self.results.sort_values(by=['Input','Name','Formula'], inplace=True)
        
        # If serovar input consists only of missing data (NULL, NA, '')
        if self.metrics.empty: 
            self.metrics = self.metrics.append({'ClusterID':   self.clust_id,
                                                'ClusterSize': n_seros}, 
                                                ignore_index=True)
            self.metrics = self.metrics.fillna('NA') 
            self.results = self.metrics[r_cols]
           
    
    def print_serovars(self):
        serovar_df = self.results[['ClusterID','ClusterSize','Input','Name','Formula']]
        serovar_df.to_csv(sys.stdout, index=False, header=self.header, sep='\t', na_rep='NA')
 
                
    def print_results(self):
        self.results.to_csv(sys.stdout, index=False, header=self.header, sep='\t', na_rep='NA')


    def print_metrics(self):
        self.metrics.drop('Comps',1).to_csv(sys.stdout,index=False,header=self.header,sep='\t',na_rep='NA')
  
                                                   
class InvalidInput(Exception):
   """Raised when an input value is not a valid serovar name or formula."""
   pass



#-------------------------------------------
# Functions
#-------------------------------------------


def all_antigens_missing(obj):
    """Determines if all antigens are missing.
    Args:
        obj(WKLMSerovar): 
    Returns:
        (bool):
    """
        
    if (pd.isna(obj.meta[antigens[0:3]]).all() or all(obj.meta[antigens[0:3]] == missing_antigen)) \
        and (not obj.meta['other_H'] or pd.isna(obj.meta['other_H'])):
        return True
    else:
        return False 


def cluster(input_file='', sort_by=None, v=None):
    """Determines the most abundant serovar(s) for one or more clusters of isolates.
    Args:       
        in_file(str): A tab-delimited input file in which each line contains two fields: 
                      a cluster id and a serovar designation, respectively. 
        sort_by(str): One or more comma-delim options for ordered sort results.
                          Options = e (exact), c (congruent), m (min_con)
                          Default = c,e,m
        v(int):       Verbosity of output: 1 - print_serovars()
                                           2 - print_results()  # Default
                                           3 - print_metrics()
    """
    
    clusters = OrderedDict()
    if input_file:
        with open(str(input_file), 'r') as i:
            for line in i:
                if len(line.strip()):
                    cluster, serovar = line.rstrip().split('\t')
                    if cluster in clusters.keys():
                        clusters[cluster].append(serovar) 
                    else:
                        clusters[cluster] = [serovar]                       
    else:    
        raise Exception('Please provide an input file!') 

    sort_by = sort_by.split(',') if sort_by else sort_by

    cluster_objs = OrderedDict()
    for cluster in clusters:
        cluster_objs[cluster] = []
        for serovar in clusters[cluster]:
           cluster_objs[cluster].append(input_to_wklm(serovar))
    
    for i, k in enumerate(cluster_objs):
        header = True if i == 0 else False
            
        if v == 1:
            SeroClust(k, cluster_objs[k], sort_by=sort_by, header=header).print_serovars()
        elif v == 3:
            SeroClust(k, cluster_objs[k], sort_by=sort_by, header=header).print_metrics()
        else:
            SeroClust(k, cluster_objs[k], sort_by=sort_by, header=header).print_results()
            

def compare(input_file='', subj='', query='', header=False):
    """Creates a SeroComp object for comparison between two serovars and prints results 
       to STDOUT.
    Args:       
        in_file(str): A tab-delimited input file with two columns of serovars for comparison. 
        subj(str):    The first serovar for comparison.
        query(str):   The second serovar for comparison. 
        header(bool): If true, the first line is treated as a header.      
    """
    
    if input_file:
        with open(str(input_file), 'r') as i:
            if header == True:
                header_line = i.readline().rstrip().split('\t')
            sero_pairs = i.readlines()
    elif subj and query:
        sero_pairs = ['\t'.join([subj, query])]
    else:    
        raise Exception('Please provide either an input file or two serovars for comparison!')  

    if input_file and header == True:
        print('{}\t{}\t{}\t{}\t{}\t{}\t{}'.format(
            header_line[0],'Name','Formula',header_line[1],'Name','Formula','Result'))
    else:
        print('{}\t{}\t{}\t{}\t{}\t{}\t{}'.format(
            'Serovar1','Name','Formula','Serovar2','Name','Formula','Result'))
            
    for pair in sero_pairs:
        wklm_objs = []

        if len(pair.strip()):
            for s in pair.rstrip().split('\t'):
                wklm_objs.append(input_to_wklm(s))
            sero_obj = SeroComp(wklm_objs[0],wklm_objs[1])
            sero_obj.print_results()   
   

def fields_to_formula(fields):
    """Constructs an antigenic formula from a list of fields.
    Args:
        fields(list): A five element list [subsp, O, P1, P2, otherH]
                      A missing field may be designated as '–', '', or np.nan
    Returns:
        formula(str): An antigenic formula
    """
    
    subsp, O, P1, P2, other_H = [missing_antigen if not f or pd.isna(f) else f for f in fields]
    subsp = '' if subsp == missing_antigen else subsp
    other_H = '' if other_H == missing_antigen else other_H
    
    if subsp.lower() in current_subsp:
        formula = '{} {}:{}:{}'.format(subsp, O, P1, P2)
    else:
        formula = '{}:{}:{}'.format(O, P1, P2)
        
    if other_H:
        formula = '{}:{}'.format(formula, other_H)
    
    return formula
    

def find_matches(obj):
    """Finds matching serovars (exact, congruent, and minimally congruent).
    Args:
        obj(WKLMSerovar): 
    Returns:
        min_congruent_objs(list): A list of SeroComp objects with results which are not incongruent
    """
    
    if pd.isna(obj.formula): 
        return []
    
    indices = set(range(0,len(wklm_subsp))) # all serovar indices by default
    field_names = ['Subspecies'] + antigens
    input_fields = formula_to_fields(obj.input) if not is_name(obj.input) else []
    wklm_lists = [wklm_subsp, wklm_O, wklm_P1, wklm_P2, wklm_other_H]
    
    for i,f in enumerate(field_names):
        field = input_fields[i] if len(input_fields) else obj.meta[f]  
        req_factors = []    
        
        if not field or pd.isna(field) or field == missing_antigen:
            continue
        else:
            factors = prep(field).split(',')
            
            req_factors = [j for j in factors if not is_opt_factor(j,field)]
            s_indices = set()
            
            # Capture all indices which represent a subset of 'factors'
            subsets = list(itertools.chain.from_iterable(itertools.combinations(factors, r) for r in range(1,len(factors))))
            for s in subsets:
                all_matches = set()
                # sort factors appropriately (numerically, then alpha) for efficiency of permutation
                s = sorted([int(i) if i.isdigit() else i for i in s], key=lambda v: (isinstance(v, str), v))
                for p in itertools.permutations(s): # find the correct permutation of factors (not always alphanumeric)
                    if len(all_matches):
                        break
                    all_matches = set(np.flatnonzero(wklm_df[f].apply(prep) == ','.join(map(str, p))))
                    s_indices.update(all_matches) 
            # Capture all indices for which 'factors' is a proper subset
            for k in req_factors: 
                indices = set(indices.intersection(set(matching_indices(k, wklm_lists[i]))))
            # Combine indices
            indices.update(s_indices)
            
    wklm_objs = [WKLMSerovar(name) for name in [wklm_name[i] for i in indices]]
    
    # Remove duplicates (ie. Miami, Sendai, Miami or Sendai)
    dups = [wklm_obj.name.split(' or ') for wklm_obj in wklm_objs if ' or ' in wklm_obj.name]
    dups = [item for sublist in dups for item in sublist] # flatten list

    min_congruent_objs = [SeroComp(obj,wklm_obj) for wklm_obj in wklm_objs \
        if wklm_obj.name not in dups and SeroComp(obj,wklm_obj).result != 'incongruent']
    
    return min_congruent_objs        
  
            
def formula_to_fields(formula):
    """Returns the individual fields of an antigenic formula.
    Args:
        formula(str): An antigenic formula. 
    Returns:
       fields(list): A list of subsp, O, P1, P2, otherH"""
    
    subsp = np.nan
    antigens_str = None
           
    if formula.lower() in current_subsp:
        subsp = formula.upper()
    elif re.match(subsp_pattern, formula):
        subsp, antigens_str = formula.split(' ')
    elif re.match(no_subsp_pattern, formula):
        antigens_str = formula    
    else:
        logging.error("The input '{}' is not a valid formula.".format(formula))   
        raise InvalidInput("The input '{}' is not a valid formula".format(formula))
        
    fields = [subsp] + [missing_antigen] * 3 + [''] # fields = [subsp,O,P1,P2,otherH]
    
    if antigens_str:       
        antigens_lst = antigens_str.split(':')
        for i in range(0,4): 
            if i < len(antigens_lst):
                if antigens_lst[i]:
                    fields[i+1] = antigens_lst[i]
            
    return fields          


def get_factor(factor,factors):
    """Retrieves a factor from a string of factors including any bracketing - 
       optional '[]', exclusive '{}', or weakly agglutinable '()'.
    Args:
       factor(str): An antigenic factor. 
       factors(str): A string of comma-delimited factors. 
    Returns:
       f(str): An antigenic factor.
    """
    
    for f in factors.split(','):
        if prep(factor) == prep(f):
            return f


def input_to_wklm(input):
    """Converts serovar input into a WKLMSerovar object, including merging objects
       from multiple closely related serovars.
    Args:
        input(string): A string containing one or more serovars appropriate for 
                       creation of a single WKLMSerovar object.
    Returns:
        wklm_obj(WKLMSerovar): A WKLMSerovar obj.
    """

    input = split_input(input)
        
    if len(input) > 1:
        wklm_obj = merge_wklm_objs([WKLMSerovar(i) for i in input])
    else:
        wklm_obj = WKLMSerovar(input[0])
    
    return wklm_obj       
        
                          
def is_min_subset(factors1,factors2):
    """Determines if a set of minimal factors is a proper subset of a set 
       of max factors.
    Args:
       factors1(str): comma-delimited factors.
       factors2(str): comma-delimited factors.
    Returns:
       (bool)
    """
    
    factors1, factors2 = factors1.lower(), factors2.lower() 
        
    f_min = min_factors(factors1)
    f_all = max_factors(factors2)

    if not f_min or f_min == {missing_antigen} or f_min.issubset(f_all):     
        return True
    else:
        return False


def is_name(serovar):
    """Determines whether a serovar value is likely to be a name versus an antigenic profile.
    Args:
        serovar(str): A serovar designation.
    Returns:
        (bool)
    """
    
    profile_characters = [':']
    if (not any(p in serovar for p in profile_characters) 
        and serovar.lower() not in current_subsp):
        return True
    else:
        return False    


def is_opt_factor(factor,factors):
    """Compares a factor to a string of factors in order to determine if the factor is 
       optional '[]', exclusive '{}', or weakly agglutinable '()'.
    Args:
       factor(str):  A single antigenic factor. 
       factors(str): A string of comma-delimited factors. 
    Returns:
       (bool)
    """
    
    if not factors or pd.isna(factors):
        return False
    factor = prep(factor)
    optional  = r'\[{}+?\]'.format(factor)
    exclusive = r'\{{{}+?\}}|\{{{}+?|{}+?\}}'.format(factor,factor,factor)
    weak = r'\({}+?\)'.format(factor)
    any_pattern = '(' + optional + '|' + exclusive + '|' + weak + ')'
  
    if re.search(any_pattern, factors):
        return True
    else:
        return False


def is_opt_subset(factors1,factors2):
    """Determines if either set of minimal factors are a proper subset of the other set 
       of max factors and if all differences represent optional factors.
    Args:
       factors1(str): A string of comma-delimited factors.
       factors2(str): A string of comma-delimited factors.
    Returns:
       (bool)
    """
        
    factors1, factors2 = factors1.lower(), factors2.lower() 
        
    min1, min2 = min_factors(factors1), min_factors(factors2)
    all1, all2 = max_factors(factors1), max_factors(factors2)
    
    if min1 == min2: 
        return True
    if not min1 or not min2 or min1 == {missing_antigen} or min2 == {missing_antigen}:
        if (not min1 or min1 == {missing_antigen}) and (not min2 or min2 == {missing_antigen}): 
            return True 
        else:
            return False  
    if min1.issubset(all2):
        for f in all2.difference(min1):
            if not (is_opt_factor(f,factors1) or is_opt_factor(f,factors2)):
                return False
        return True    
    if min2.issubset(all1):
        for f in all1.difference(min2):
            if not (is_opt_factor(f,factors1) or is_opt_factor(f,factors2)):
                return False
        return True    
    
    return False                    


def matching_indices(factor, l=[]): 
    """For a given factor, return all indices of elements in a list which contain the factor.
    Args:
       factor(str): A single factor to match.
       l(list): A list of comma-separated factors. 
    Returns:
       (list): A list of matching indices.
    """
    
    return [i for i, x in enumerate(l) if str(factor).lower() in prep(x).split(',')]


def max_factors(factors):
    """Returns a set of all possible factors by removing parenthesis and brackets from 
       optional (or phage conversion) '[]', exclusive '{}', and weakly agglutinable 
       '()' factors. 
       For example, II 9,46:g,[m],[s],t:[e,n,x] -> II 9,46:g,m,s,t:e,n,x.
    Args:
       factors(str): A string of comma-delimited factors. 
    Returns:
       factors(set): A set of factors.
    """
    
    factors = re.sub('}{', '},{', factors)
    factors = re.sub(r'\[|\]|\(|\)|{|}', '', factors)
    factors = factors.split(',')
    factors = set(factors) if factors[0] else set()
    
    return factors


def merge_wklm_objs(objs):
    """Determines antigenic factors common among serovars and creates a merged 
       WKLMSerovar object.
    Args:
       objs(list): A list of WKLMSerovar objects from closely related serovars.  
    Returns:
       merged_obj(WKLMSerovar): A WKLMSerovar object representing the common antigenic formula.
    """
    
    df = pd.DataFrame([obj.meta for obj in objs], index=range(0,len(objs)))
    
    all_fs = pd.DataFrame()
    all_fs['O'] = df['O'].apply(lambda x: max_factors(x))
    all_fs['P1'] = df['P1'].apply(lambda x: max_factors(x))
    all_fs['P2'] = df['P2'].apply(lambda x: max_factors(x))
    all_fs['other_H'] = df['other_H'].apply(lambda x: max_factors(x))
    
    comps = [SeroComp(i,j).result for i,j in list(itertools.combinations(objs,2))]
    common_fs = {'Subspecies': '','O': [],'P1': [],'P2': [],'other_H': []} 

    if 'incongruent' in comps:
        logging.error('The input serovars are incongruent and cannot be merged.')
        formula = 'NA'   
    elif set(comps) == {'exact'}:
        formula = df.Formula[0]   
    else:    
        for f in ['Subspecies'] + antigens:
            uniq_f = pd.unique(df[f]).tolist()
            if pd.isna(uniq_f).any():
                common_fs[f] = np.nan
            elif len(uniq_f) == 1:
                common_fs[f] = uniq_f
            else: 
                common_fs[f] = sorted(list(all_fs[f][0].intersection(*all_fs[f][1:])))
                for i,factor in enumerate(common_fs[f]):
                    if all(is_opt_factor(factor,factors) for factors in df[f].to_list()):
                        common_fs[f][i] = get_factor(factor,df[f][0])
                                 
        formula = fields_to_formula([(',').join(common_fs['Subspecies']),(',').join(common_fs['O']),
                                     (',').join(common_fs['P1']),(',').join(common_fs['P2']),
                                     (',').join(common_fs['other_H'])]) 
    
    merged_obj = WKLMSerovar(formula)
    merged_obj.input = (' or ').join([obj.input for obj in objs])
    merged_obj.meta.Input = merged_obj.input
                                 
    return merged_obj


def min_factors(factors):
    """This function returns a minimal list of factors by removing optional "[]",
       exclusive "{}", and weakly agglutinable "()" factors.
       For example, II 9,46:g,[m],[s],t:[e,n,x] -> II 9,46:g,t:
    Args:
       factors(str): A string of comma-delimited factors. 
    Returns:
       factors(set): A set of factors. 
    """
    
    # Attempt to remove patterns while leaving commas as necessary
    factors = re.sub(r'\[.+?\],+?|\(.+?\),+?|{.+?},+?', '', factors)
    factors = re.sub(r',+?\[.+?\]|,+?\(.+?\)|,+?{.+?}', '', factors)
    factors = re.sub(r'\[.+?\]|\(.+?\)|{.+?}', '', factors)
    factors = factors.split(',')
    factors = set(factors) if factors[0] else set()
    
    return factors


def prep(sero):
    """Prepares a serovar string for matching by removing extra characters 
       (brackets, ellipses), and transforming to lowercase.
    Args:
        sero(str): A serovar designation.
    Returns:
        (str): A curated serovar designation. 
    """
  
    if pd.isna(sero):
        return sero
    standardize_unicode(sero)
    sero = re.sub('}{', ',', sero)
    sero = re.sub(r'\[|\]|\(|\)|{|}|\.{3}|…', '', sero)

    return sero.lower()


def query(input_file='',serovar='',exact=False):
    """Queries the WKLM repository for serovar matches.
    Args:
        in_file(str): An input file with one query (serovar or antigenic formula) per line.
        serovar(str): A query (serovar name or antigenic formula).
        exact(bool):  Find exact matches only. Default: False
    """

    sort_order = {'none': 0,'exact': 1,'congruent': 2,'minimally congruent': 3}

    if input_file:
        with open(str(input_file), 'r') as i:
            serovars = i.readlines()
    elif serovar:
        serovars = [ serovar ]
    else:    
        raise Exception('Please provide a query!')  
        
    inputs, names, formulas, levels = [], [], [], []
    
    serovars = [s.rstrip() for s in serovars if len(s.strip())]
    
    for serovar in serovars:
        wklm_obj = input_to_wklm(serovar)
            
        if exact:
            inputs.append(serovar)  
            names.append(wklm_obj.name)
            formulas.append(wklm_obj.formula if not pd.isna(wklm_obj.name) else np.nan)
            levels.append('exact' if not pd.isna(wklm_obj.name) else 'none')
        else:
            matching_objs = find_matches(wklm_obj)
                
            if len(matching_objs):
                # sort by type of match using sort_order
                matching_objs.sort(key=lambda x: sort_order[x.result])    
                for m_obj in matching_objs:
                    inputs.append(serovar)  
                    names.append(m_obj.query.name)
                    formulas.append(m_obj.query.formula)
                    levels.append(m_obj.result)
            else:
                inputs.append(serovar)  
                names.append(np.nan)
                formulas.append(np.nan)
                levels.append('none')
                                      
    results = pd.DataFrame({
                'Input':   inputs, 
                'Name':    names,
                'Formula': formulas,
                'Match':   levels},
                columns = ['Input','Name','Formula','Match']) 
    
    results.to_csv(sys.stdout, index=False, header=True, sep='\t', na_rep='NA')
                     

def split_input(input):
    """Separates multiple serovars separated by ' or ' or '/'.
    Args:
        input(str): 
    Returns:
        input_lst(list): A list of inputs. 
    """
    
    # Allow for more than one serovar prediction as input
    input = re.sub('/',' or ',input)
    input_lst = input.split(' or ')

    return input_lst
    
    
def standardize_formula(formula):
    """Standardizes an antigenic formula.
    Args:
        formula(str): An antigenic formula. 
    Returns:
        standardized_formula(str): A standardized antigenic formula. 
    """
    
    formula = re.sub(':non-motile',':-:-',formula,flags=re.IGNORECASE)
    formula = re.sub(':nonmotile',':-:-',formula,flags=re.IGNORECASE)
    formula = re.sub('Rough:','-:',formula,flags=re.IGNORECASE)
    formula = re.sub('Mucoid:','-:',formula,flags=re.IGNORECASE)
    formula = re.sub('}{','},{',formula) # ignore antigen exclusivity
#    formula = re.sub('^(IIa|IIb)\s','II ',formula)
    formula = standardize_unicode(formula)

    # Convert missing antigens to en dash
    subsp, O, P1, P2, other_H = formula_to_fields(formula)
    other_missing = '(^$|undetermined|Undetermined)'
    O = re.sub(other_missing,missing_antigen,O)
    P1 = re.sub(other_missing,missing_antigen,P1)
    P2 = re.sub(other_missing,missing_antigen,P2)

    standardized_formula = fields_to_formula([subsp, O, P1, P2, other_H])      

    invalid_formula = ['–:–:–', '–:–:–:–']
    if standardized_formula in invalid_formula:
        logging.error("The input '{}' is not a valid formula.".format(formula))
        raise InvalidInput("The input '{}' is not a valid formula".format(formula))

    return standardized_formula  


def standardize_input(input):
    """Standardizes serovar input expected to be either a name or formula.
    Args:
        input(str): 
    Returns:
        input(str): Standardized input. 
    """
    
    if re.match('paratyphi b',input.lower()) and \
        any(p in input.lower() for p in paratyphi_b_var):
        input = 'Paratyphi B var. L(+) tartrate (= d–tartrate)+'        
    # Remove content in parentheses in order to use the name as a dict key,
    # e.g. Chester (4,12:e,h:e,n,x) or Cerro var. 14+ (Siegburg) or 6,8:d:- (monophasic) or Typhimurium(O5-)
    # Remove unless adjacent to a comma or colon (II 16:e,n,x:1,(5),7 or IIIb 16:(k):e,n,x,z15)
    elif re.search(r'\(.+\)', input):
        if not (re.search(r'[:|,]+\(.+\)', input)
                or re.search(r'\(.+\)[:|,]+', input)):
            input = re.sub(r'\(.+\)', '', input)
    
    # Ignore seqsero output
    input = re.sub(r'\*|allmers','',input)

    # Remove any leading/trailing whitespace
    input = input.strip()

    # Transform subsp
    input = re.sub('^enterica','I',input,flags=re.IGNORECASE)
    input = re.sub('^salamae','II',input,flags=re.IGNORECASE)
    input = re.sub('^arizonae','IIIa',input,flags=re.IGNORECASE)
    input = re.sub('^diarizonae','IIIb',input,flags=re.IGNORECASE)
    input = re.sub('^houtenae','IV',input,flags=re.IGNORECASE)
    input = re.sub('^bongori','V',input,flags=re.IGNORECASE) # outdated, but helpful
    input = re.sub('^indica','VI',input,flags=re.IGNORECASE)
    
    # Ignore seqsero sdf for now
    input = re.sub(' Sdf prediction.+','',input,flags=re.IGNORECASE)
    input = re.sub(' sdf.+','',input,flags=re.IGNORECASE)
    
    # Transform seqsero output
    if re.match('potential monophasic variant of', input.lower()):
        if re.match('.+typhimurium',input.lower()):
            input = 'I 4:i:–'
        elif re.match('.+paratyphi b',input.lower()):
            input = 'I 4:b:–'
        elif re.match('.+heidelberg',input.lower()):
            input = 'I 4:r:–'
    
    return input


def standardize_unicode(string):
    """Standardizes unicode characters for matching purposes.
    Args:
        string(str): 
    Returns:
        standardized_string(str):  
    """
    
    other_dashes = '[\u002D\u058A\u05BE\u1400\u1806\u2010\u2011\u2012\u2014\u2015\uFE58\uFE63\uFF0D\u02D7\u207B\u208B\u2212]'
    string = re.sub(other_dashes,'\u2013',string)      #en dash = '\u2013'
    standardized_string = re.sub(r'\.{3}','…',string)   #ellipsis = '\u2026'

    return standardized_string

