"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('Renders minimal example with just a domain name', () => {
    const origin = new lib_1.HttpOrigin('www.example.com');
    const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
    expect(originBindConfig.originProperty).toEqual({
        id: 'StackOrigin029E19582',
        domainName: 'www.example.com',
        customOriginConfig: {
            originProtocolPolicy: 'https-only',
        },
    });
});
test('renders an example with all available props', () => {
    const origin = new lib_1.HttpOrigin('www.example.com', {
        originPath: '/app',
        connectionTimeout: core_1.Duration.seconds(5),
        connectionAttempts: 2,
        customHeaders: { AUTH: 'NONE' },
        protocolPolicy: cloudfront.OriginProtocolPolicy.MATCH_VIEWER,
        httpPort: 8080,
        httpsPort: 8443,
        readTimeout: core_1.Duration.seconds(45),
        keepaliveTimeout: core_1.Duration.seconds(3),
    });
    const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
    expect(originBindConfig.originProperty).toEqual({
        id: 'StackOrigin029E19582',
        domainName: 'www.example.com',
        originPath: '/app',
        connectionTimeout: 5,
        connectionAttempts: 2,
        originCustomHeaders: [{
                headerName: 'AUTH',
                headerValue: 'NONE',
            }],
        customOriginConfig: {
            originProtocolPolicy: 'match-viewer',
            httpPort: 8080,
            httpsPort: 8443,
            originReadTimeout: 45,
            originKeepaliveTimeout: 3,
        },
    });
});
test.each([
    core_1.Duration.seconds(0),
    core_1.Duration.seconds(0.5),
    core_1.Duration.seconds(60.5),
    core_1.Duration.seconds(61),
    core_1.Duration.minutes(5),
])('validates readTimeout is an integer between 1 and 60 seconds', (readTimeout) => {
    expect(() => {
        new lib_1.HttpOrigin('www.example.com', {
            readTimeout,
        });
    }).toThrow(`readTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${readTimeout.toSeconds()}.`);
});
test.each([
    core_1.Duration.seconds(0),
    core_1.Duration.seconds(0.5),
    core_1.Duration.seconds(60.5),
    core_1.Duration.seconds(61),
    core_1.Duration.minutes(5),
])('validates keepaliveTimeout is an integer between 1 and 60 seconds', (keepaliveTimeout) => {
    expect(() => {
        new lib_1.HttpOrigin('www.example.com', {
            keepaliveTimeout,
        });
    }).toThrow(`keepaliveTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${keepaliveTimeout.toSeconds()}.`);
});
//# sourceMappingURL=data:application/json;base64,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