"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @experimental
 */
class S3Origin {
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, props) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(scope, 'S3Origin', {
                comment: `Identity for ${options.originId}`,
            });
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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