"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const cdk = require("aws-cdk-lib");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new aws_servicecatalog_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: launchRole.roleArn,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(portfolio, tagOptions) {
        const portfolioStack = cdk.Stack.of(portfolio);
        for (const [key, tagOptionsList] of Object.entries(tagOptions.tagOptionsMap)) {
            validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption key', 1, 128, key);
            tagOptionsList.forEach((value) => {
                validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption value', 1, 256, value);
                const tagOptionKey = util_1.hashValues(key, value, portfolioStack.node.addr);
                const tagOptionConstructId = `TagOption${tagOptionKey}`;
                let cfnTagOption = portfolioStack.node.tryFindChild(tagOptionConstructId);
                if (!cfnTagOption) {
                    cfnTagOption = new aws_servicecatalog_1.CfnTagOption(portfolioStack, tagOptionConstructId, {
                        key: key,
                        value: value,
                        active: true,
                    });
                }
                const tagAssocationKey = util_1.hashValues(key, value, portfolio.node.addr);
                const tagAssocationConstructId = `TagOptionAssociation${tagAssocationKey}`;
                if (!portfolio.node.tryFindChild(tagAssocationConstructId)) {
                    new aws_servicecatalog_1.CfnTagOptionAssociation(portfolio, tagAssocationConstructId, {
                        resourceId: portfolio.portfolioId,
                        tagOptionId: cfnTagOption.ref,
                    });
                }
            });
        }
        ;
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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