"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * The log driver to use for the container.
 */
var LogDriver;
(function (LogDriver) {
    /**
     * Specifies the Amazon CloudWatch Logs logging driver.
     */
    LogDriver["AWSLOGS"] = "awslogs";
    /**
     * Specifies the Fluentd logging driver.
     */
    LogDriver["FLUENTD"] = "fluentd";
    /**
     * Specifies the Graylog Extended Format (GELF) logging driver.
     */
    LogDriver["GELF"] = "gelf";
    /**
     * Specifies the journald logging driver.
     */
    LogDriver["JOURNALD"] = "journald";
    /**
     * Specifies the logentries logging driver.
     */
    LogDriver["LOGENTRIES"] = "logentries";
    /**
     * Specifies the JSON file logging driver.
     */
    LogDriver["JSON_FILE"] = "json-file";
    /**
     * Specifies the Splunk logging driver.
     */
    LogDriver["SPLUNK"] = "splunk";
    /**
     * Specifies the syslog logging driver.
     */
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * Platform capabilities
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    /**
     * Specifies EC2 environment.
     */
    PlatformCapabilities["EC2"] = "EC2";
    /**
     * Specifies Fargate environment.
     */
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_JobDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!((_b = props.platformCapabilities) === null || _b === void 0 ? void 0 : _b.includes(PlatformCapabilities.FARGATE));
        const jobDef = new aws_batch_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: (_c = props.platformCapabilities) !== null && _c !== void 0 ? _c : [PlatformCapabilities.EC2],
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefName = stack.splitArn(jobDefinitionArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: jobDefinitionName,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch-alpha.JobDefinition", version: "2.23.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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