import { CustomResource } from '@aws-cdk/core';
import { AwsApiCall } from './sdk';
/**
 * Represents the "actual" results to compare
 */
export declare abstract class ActualResult {
    /**
     * Get the actual results from a CustomResource
     */
    static fromCustomResource(customResource: CustomResource, attribute: string): ActualResult;
    /**
     * Get the actual results from a AwsApiCall
     */
    static fromAwsApiCall(query: AwsApiCall, attribute: string): ActualResult;
    /**
     * The actual results as a string
     */
    abstract result: string;
}
/**
 * Represents the "expected" results to compare
 */
export declare abstract class ExpectedResult {
    /**
     * The actual results must match exactly. Missing data
     * will result in a failure
     *
     * @example
     * // actual results
     * const actual = {
     *   stringParam: 'hello',
     *   numberParam: 3,
     *   booleanParam: true,
     * };
     * // pass
     * ExpectedResult.exact({
     *   stringParam: 'hello',
     *   numberParam: 3,
     *   booleanParam: true,
     * })
     *
     * // fail
     * ExpectedResult.exact({
     *   stringParam: 'hello',
     * });
     */
    static exact(expected: any): ExpectedResult;
    /**
     * The expected results must be a subset of the
     * actual results.
     *
     * @example
     * // actual results
     * const actual = {
     *   stringParam: 'hello',
     *   numberParam: 3,
     *   booleanParam: true,
     * };
     * // pass
     * ExpectedResult.objectLike({
     *   stringParam: 'hello',
     * });
     */
    static objectLike(expected: {
        [key: string]: any;
    }): ExpectedResult;
    /**
     * The actual results must be a list and must contain
     * an item with the expected results.
     *
     * @example
     * // actual results
     * const actual = [
     *   {
     *     stringParam: 'hello',
     *   },
     *   {
     *     stringParam: 'world',
     *   },
     * ];
     * // pass
     * ExpectedResult.arrayWith([
     *   {
     *     stringParam: 'hello',
     *   },
     * ]);
     */
    static arrayWith(expected: any[]): ExpectedResult;
    /**
     * Actual results is a string that matches
     * the Expected result regex
     *
     * @example
     * // actual results
     * const actual = 'some string value';
     *
     * // pass
     * ExpectedResult.stringLikeRegexp('value');
     */
    static stringLikeRegexp(expected: string): ExpectedResult;
    /**
     * The expected results encoded as a string
     */
    abstract result: string;
}
