# rpa_suite/core/iris.py

# external imports
try:
    from docling.document_converter import DocumentConverter as IrisEngine
except ImportError as e:
    raise ImportError(
        "Iris - Error: Could not import 'docling.document_converter'. Make sure the 'docling' library is installed."
    ) from e

# third party imports
from enum import Enum
from typing import Optional, Union

# internal imports
from rpa_suite.functions._printer import success_print


class IrisError(Exception):
    """Custom exception for Iris errors."""

    def __init__(self, message):
        # Remove "Iris Error:" from message if it already exists, as it will be added by default
        clean_message = message.replace("IrisError:", "").strip()
        super().__init__(f"IrisError: {clean_message}")


class ExportFormats(Enum):
    """Supported export formats for document conversion."""

    MARKDOWN = "markdown"
    DICT = "dict"
    DOCTAGS = "doctags"
    HTML = "html"
    TEXT = "text"
    INDENTEDTEXT = "indented_text"


class Iris:
    """
    Iris (OCR-AI) - Document converter with support for multiple extensions.

    This class provides a simplified interface to convert documents in various formats (PDF, images, text) to structured formats like Markdown, HTML, plain text, among others.

    Attributes:
        engine: IrisEngine instance.
        last_result: Last processed conversion result.
        list_results: List of results generated by batch processing with: read_documents

    On Error:
        Raises IrisError if the document conversion fails.

    Example:
        >>> from rpa_suite.core import ExportFormats
        >>> iris = Iris()
        >>> content = iris.read_document("document.pdf", ExportFormats.MARKDOWN)
        >>> print(content)

    ---
    pt-br:
    Iris (OCR-AI) - Conversor de documentos com suporte a múltiplas extensões.

    Esta classe fornece uma interface simplificada para converter documentos em vários formatos (PDF, imagens, texto) para formatos estruturados como Markdown, HTML, texto simples, entre outros.

    Atributos:
        engine: Instância do IrisEngine.
        last_result: Último resultado de conversão processado.
        list_results: Lista de resultados gerados por processamento em lote com: read_documents

    Em caso de erro:
        Lança IrisError se a conversão do documento falhar.

    Exemplo:
        >>> from rpa_suite.core import ExportFormats
        >>> iris = Iris()
        >>> conteudo = iris.read_document("documento.pdf", ExportFormats.MARKDOWN)
        >>> print(conteudo)
    """

    engine: Optional[IrisEngine]
    last_result = None
    list_results = list | None

    def __init__(self) -> None:
        """
        Iris (OCR-AI) - Document converter with support for multiple extensions.

        This class provides a simplified interface to convert documents in various formats (PDF, images, text) to structured formats like Markdown, HTML, plain text, among others.

        Attributes:
            engine: IrisEngine instance.
            last_result: Last processed conversion result.
            list_results: List of results generated by batch processing with: read_documents

        On Error:
            Raises IrisError if the document conversion fails.

        Example:
            >>> from rpa_suite.core import ExportFormats
            >>> iris = Iris()
            >>> content = iris.read_document("document.pdf", ExportFormats.MARKDOWN)
            >>> print(content)

        ---
        pt-br:
        Iris (OCR-AI) - Conversor de documentos com suporte a múltiplas extensões.

        Esta classe fornece uma interface simplificada para converter documentos em vários formatos (PDF, imagens, texto) para formatos estruturados como Markdown, HTML, texto simples, entre outros.

        Atributos:
            engine: Instância do IrisEngine.
            last_result: Último resultado de conversão processado.
            list_results: Lista de resultados gerados por processamento em lote com: read_documents

        Em caso de erro:
            Lança IrisError se a conversão do documento falhar.

        Exemplo:
            >>> from rpa_suite.core import ExportFormats
            >>> iris = Iris()
            >>> conteudo = iris.read_document("documento.pdf", ExportFormats.MARKDOWN)
            >>> print(conteudo)
        """
        try:
            self.engine = IrisEngine()
            self.result_converted = None
            self.last_result = None
            self.list_results = []

        except Exception as e:
            raise IrisError(f"Failed to initialize DocumentConverter: {str(e)}.") from e

    def __convert_document(self, path_file: str = None):
        """
        Internal: Converts the document specified by the path.

        Raises:
            IrisError: If an error occurs during document conversion.
        """
        try:
            if not path_file:
                raise IrisError("Specify the file path for conversion.")
            self.result_converted = self.engine.convert(path_file)
        except Exception as e:
            raise IrisError(f"Error trying to convert document! {str(e)} ") from e

    def read_document(
        self, file_path: str = None, result_format=ExportFormats.MARKDOWN, verbose: bool = False
    ) -> Optional[Union[str, dict]]:
        """
        Reads and converts a document to the specified format.

        Args:
            file_path: Path to the document file.
            result_format: Desired export format.
            verbose: If True, displays success messages.

        Returns:
            Document converted to the specified format, or None if it fails.

        Raises:
            IrisError: If an error occurs during validation, conversion or export.

        Example:
            >>> iris = Iris()
            >>> content = iris.read_document("doc.pdf", ExportFormats.TEXT)
            >>> print(content)

        ---
        pt-br:
        Lê e converte um documento para o formato especificado.

        Parâmetros:
            file_path: Caminho para o arquivo do documento.
            result_format: Formato de exportação desejado.
            verbose: Se True, exibe mensagens de sucesso.

        Retorno:
            Documento convertido para o formato especificado, ou None se falhar.

        Exceções:
            IrisError: Se ocorrer um erro durante a validação, conversão ou exportação.

        Exemplo:
            >>> iris = Iris()
            >>> conteudo = iris.read_document("doc.pdf", ExportFormats.TEXT)
            >>> print(conteudo)
        """
        try:
            self.__convert_document(file_path)

            if not self.result_converted or not hasattr(self.result_converted, "document"):
                raise IrisError("Failed to convert file or invalid object resulted.")

            if result_format == ExportFormats.MARKDOWN:
                self.last_result = self.result_converted.document.export_to_markdown()
            elif result_format == ExportFormats.DICT:
                self.last_result = self.result_converted.document.export_to_dict()
            elif result_format == ExportFormats.DOCTAGS:
                self.last_result = self.result_converted.document.export_to_doctags()
            elif result_format == ExportFormats.HTML:
                self.last_result = self.result_converted.document.export_to_html()
            elif result_format == ExportFormats.TEXT:
                self.last_result = self.result_converted.document.export_to_text()
            elif result_format == ExportFormats.INDENTEDTEXT:
                # pylint: disable=protected-access
                self.last_result = self.result_converted.document._export_to_indented_text()
            else:
                raise IrisError(f"Not supported format: {result_format} ")
            if verbose:
                success_print("Iris: Successfully converted!")

            return self.last_result
        except Exception as e:
            raise IrisError(f"Error trying to read document: {str(e)}.") from e

    def read_documents(
        self, list_file_path: list[str] = None, result_format=ExportFormats.MARKDOWN, verbose: bool = False
    ) -> Optional[list]:
        """
        Reads and converts multiple documents to the specified format.

        Args:
            list_file_path: List of documents in path format.
            result_format: Desired export format.
            verbose: If True, displays success messages.

        Returns:
            List of documents converted to the specified format, or None if it fails.

        Raises:
            IrisError: If an error occurs during validation, conversion or export.

        Example:
            >>> iris = Iris()
            >>> contents = iris.read_documents(["doc.pdf", "doc2.docx"], ExportFormats.TEXT)
            >>> print(contents)

        ---
        pt-br:
        Lê e converte múltiplos documentos para o formato especificado.

        Parâmetros:
            list_file_path: Lista de caminhos dos documentos.
            result_format: Formato de exportação desejado.
            verbose: Se True, exibe mensagens de sucesso.

        Retorno:
            Lista de documentos convertidos para o formato especificado, ou None se falhar.

        Exceções:
            IrisError: Se ocorrer um erro durante a validação, conversão ou exportação.

        Exemplo:
            >>> iris = Iris()
            >>> conteudos = iris.read_documents(["doc.pdf", "doc2.docx"], ExportFormats.TEXT)
            >>> print(conteudos)
        """
        self.list_results = []

        for file_path in list_file_path:
            try:
                self.__convert_document(file_path)

                if not self.result_converted or not hasattr(self.result_converted, "document"):
                    raise IrisError("Failed to convert file or invalid object resulted.")

                if result_format == ExportFormats.MARKDOWN:
                    self.last_result = self.result_converted.document.export_to_markdown()
                    self.list_results.append(self.last_result)
                elif result_format == ExportFormats.DICT:
                    self.last_result = self.result_converted.document.export_to_dict()
                    self.list_results.append(self.last_result)
                elif result_format == ExportFormats.DOCTAGS:
                    self.last_result = self.result_converted.document.export_to_doctags()
                    self.list_results.append(self.last_result)
                elif result_format == ExportFormats.HTML:
                    self.last_result = self.result_converted.document.export_to_html()
                    self.list_results.append(self.last_result)
                elif result_format == ExportFormats.TEXT:
                    self.last_result = self.result_converted.document.export_to_text()
                    self.list_results.append(self.last_result)
                elif result_format == ExportFormats.INDENTEDTEXT:
                    # pylint: disable=protected-access
                    self.last_result = self.result_converted.document._export_to_indented_text()
                    self.list_results.append(self.last_result)
                else:
                    raise IrisError(f"Not supported format: {result_format} ")
                if verbose:
                    success_print("Iris: Successfully converted!")
            except IrisError as ie:
                raise ie from ie
            except Exception as e:
                raise IrisError(f"Error trying to read documents: {str(e)}.") from e

        return self.list_results
