import os
import dill
import yaml

from .._version import __version__
import guide_bot.requirements.requirement_parameters as rpars
from guide_bot.requirements.input_configuration_iterator import InputConfigurationIterator

from .Guide import Guide
from .DataCollector import DataCollector


class Project:
    """
    Main logic for performing a guide_bot run with any number of guides

    This class contains the main logic for performing a guide_bot run, and
    handles the scan over input criteria and loop over all user defined
    guides. It needs a target object describing the figure of merit for
    the guide optimization, and a moderator object describing the neutron
    source including basic restrictions. New Guide objects can be
    generated by the new_guide method, which can then be customized but are
    still connected to the Project instance. Call the run method to perform
    the optimization.
    """
    def __init__(self, name, moderator, target, analysis_moderators=None, settings=None):
        """
        Create a new Project from target and moderator specifications

        The Project object contains the main logic for performing a guide_bot
        run and serves as part of the user interface. New guide objects can be
        generated with the new_guide method, and these can then have guide
        elements added to them to describe the guide geometry. The project
        needs a name which will be used to identify the generated folders.

        Parameters
        ----------

        name : str
            Name of this project

        target : Target object
            Target object describing the figure of merit for optimization

        moderator : Object derived from BaseSource
            Description of neutron source used for optimization

        analysis_moderators : list of objects derived from BaseSource
            Description of neutron sources on which the guide will be analyzed

        settings: dict
            Dictionary of settings applicable for simulation / optimization
        """
        self.name = name
        self.base_folder = None

        self.target = target
        self.moderator = moderator

        self.guides = []

        source_names = [self.moderator.get_name()]

        if analysis_moderators is None:
            analysis_moderators = []
        elif not isinstance(analysis_moderators, list):
            analysis_moderators = [analysis_moderators]
        self.analysis_moderators = analysis_moderators

        for analysis_moderator in analysis_moderators:
            parameters = analysis_moderator.parameters
            if parameters.get_n_scanned_parameters() > 0:
                raise RuntimeError("Source instances used for analysis can not have scanned parameters!")

            analysis_moderator.make_name_unique(source_names)
            source_names.append(analysis_moderator.get_name())

        if settings is None:
            self.settings = {}
        else:
            self.settings = settings

    def add_guide(self, guide):
        """
        Adds a Guide object to the Project for subsequent optimization

        Parameters
        ----------

        guide : Guide
            Guide object to be added to the Project
        """
        for element in guide.guide_elements:
            element.set_owner("User")

        self.guides.append(guide)

    def new_guide(self, *args, **kwargs):
        """
        Generates and registers a Guide object that will be optimized

        The returned guide object can be expanded by the user to describe the
        desired guide geometry, and does not need to be added to the
        Project object as it is already registered.

        Returns
        -------

        Guide
            Guide object which is expanded by adding GuideElemenets
        """
        new_guide = Guide(*args, **kwargs)
        new_guide.set_current_owner("User")
        self.add_guide(new_guide)

        return new_guide

    def __repr__(self):
        """
        Prints status of Projectrun object

        Returns
        -------

        str
            String with description of status
        """

        string = "guide_bot Project named '" + self.name + "'\n"

        string += "Included guides: \n"
        for guide in self.guides:
            string += "  " + guide.name + "\n"

        mod_scan_shape = self.moderator.parameters.get_scan_shape()
        n_mod_configs = 1
        for dim in mod_scan_shape:
            n_mod_configs *= dim

        string += "Moderator scan configurations: " + str(n_mod_configs) + "\n"

        target_scan_shape = self.target.parameters.get_scan_shape()
        n_target_configs = 1
        for dim in target_scan_shape:
            n_target_configs *= dim

        string += "Target scan configurations: " + str(n_target_configs) + "\n"

        total_configs = n_target_configs*n_mod_configs*len(self.guides)
        string += "Total optimizations to be performed: " + str(total_configs) + "\n"

        string += "\n"

        if self.analysis_moderators is None:
            string += "No analysis moderators"
        else:
            string += "Analysis moderators: \n"
            for moderator in self.analysis_moderators:
                string += "  " + moderator.get_name() + "\n"

        return string

    def package_single(self, guide, scan_name):
        """
        Saves information needed to perform the guide optimization

        Each guide optimization is prepared as a package saved with dill ready
        to be executed by the guide_bot runner. A package contains a snapshot
        of the moderator and target used for optimization, each of which has
        their state for this step of the overall scan internalized. The guide
        provided by the user is included, along with all moderators used for
        analysis of additional sources.

        Parameters
        ----------

        guide : Guide
            Guide object which will be expanded with source and target

        scan_name : str
            Current name for this scan step
        """
        # set up name for this combination
        print(scan_name)

        guide.restore_original()

        if "ncount" not in self.settings:
            self.settings["ncount"] = 1E6

        if "swarmsize" not in self.settings:
            self.settings["swarmsize"] = 25

        if "omega" not in self.settings:
            self.settings["omega"] = 0.5

        if "phip" not in self.settings:
            self.settings["phip"] = 0.5

        if "phig" not in self.settings:
            self.settings["phig"] = 0.5

        if "maxiter" not in self.settings:
            self.settings["maxiter"] = 300

        if "minstep" not in self.settings:
            self.settings["minstep"] = 1E-4

        if "minfunc" not in self.settings:
            self.settings["minfunc"] = 1E-8

        if "logfile" not in self.settings:
            self.settings["logfile"] = True

        self.settings["optimized_monitor"] = "fom.dat"
        self.settings["foldername"] = "optimization_data"

        # Save with dill
        package = {"scan_name": scan_name, "guide": guide, "settings": self.settings,
                   "target": self.target, "moderator": self.moderator,
                   "analysis_moderators": self.analysis_moderators,
                   "required_components": guide.required_components}
        outfile = open(scan_name + ".plk", "wb")
        dill.dump(package, outfile)
        outfile.close()

    def write(self, cluster=None):
        """
        Saves optimization job packages for all input configurations

        The write method calls the package_single method for each guide and scans
        over all the input configurations from moderator and target.
        Writes an overview yaml file describing the project which will aid in
        plotting an overview of the results.
        """
        scan = InputConfigurationIterator(self.target, self.moderator)

        # Create new folder for project
        try:
            os.mkdir(self.name)
        except OSError:
            raise RuntimeError("Could not create folder for guide_bot project! \n"
                               + "Delete the folder with the project name '"
                               + self.name + "' or create a project with a different name.")
        self.original_wd = os.getcwd()
        self.base_folder = os.path.join(self.original_wd, self.name)

        # Create datafiles folder for project
        data_path = os.path.join(self.name, "input_data_folder")
        try:
            os.mkdir(data_path)
        except OSError:
            raise RuntimeError("Could not create input_data_folder in guide_bot project!")
        self.abs_data_path = os.path.abspath(data_path)

        # copy data files needed for all scan states
        data_collector = DataCollector(self.abs_data_path)
        scan.reset_configuration()
        while scan.next_state():
            data_collector.collect(self.target.parameters)
            data_collector.collect(self.moderator.parameters)

            for analysis_moderator in self.analysis_moderators:
                data_collector.collect(analysis_moderator.parameters)

        if cluster is not None:
            cluster.set_project_path(self.base_folder)
            cluster.read_configuration()
            cluster.start_launch_script()

        guide_names = []
        for guide in self.guides:
        
            guide_names.append(guide.make_name_unique(guide_names))
        
            # Create folder
            print("Now running for guide named ", guide.name)
            
            guide.save_original()

            os.chdir(self.base_folder)
            guide_folder = os.path.join(self.base_folder, guide.name)
            try:
                os.mkdir(guide_folder)
            except OSError:
                raise RuntimeError("Could not create folder for guide!" + guide_folder)

            guide.copy_components(guide_folder)
            os.chdir(guide_folder)

            scan.reset_configuration()
            while scan.next_state():
                # Run each configuration
                scan_name = guide.name + scan.state_string()
                self.package_single(guide, scan_name)

                if cluster is not None:
                    cluster.write_task(foldername=guide.name, scan_name=scan_name)

        os.chdir(self.base_folder)

        # Write overview file
        moderator = {"moderator_name": self.moderator.get_name(),
                     "moderator_fixed": scan.moderator_parameters.get_fixed_dict(),
                     "moderator_scan": scan.moderator_parameters.get_scan_dict(),
                     "moderator_units": scan.moderator_parameters.get_unit_dict()}

        target = {"target_name": type(self.target).__name__,
                  "target_fixed": scan.target_parameters.get_fixed_dict(),
                  "target_scan": scan.target_parameters.get_scan_dict(),
                  "target_units": scan.target_parameters.get_unit_dict()}

        analysis_moderators = {}
        for mod in self.analysis_moderators:
            mod_name = mod.get_name()
            analysis_moderators[mod_name] = {}
            analysis_moderators[mod_name]["parameters"] = mod.parameters.get_fixed_dict()
            analysis_moderators[mod_name]["units"] = mod.parameters.get_unit_dict()

        scan_states = []
        scan.reset_configuration()
        while scan.next_state():
            state_info = {"scan_state": scan.state_string(),
                          "target_scan": scan.get_target_state_dict(),
                          "moderator_scan": scan.get_moderator_state_dict()}

            scan_states.append(state_info)

        overview = {"guide_bot_version": __version__,
                    "guide_names": guide_names,
                    "target": target,
                    "moderator": moderator,
                    "analysis_moderators": analysis_moderators,
                    "scan_states": scan_states}

        with open("run_overview.yaml", 'w') as yaml_file:
            yaml.dump(overview, yaml_file, default_flow_style=False)

        os.chdir(self.original_wd)

