import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';
import * as constructs from 'constructs';
import { JobExecutable } from '.';
import { IConnection } from './connection';
import { ISecurityConfiguration } from './security-configuration';
/**
 * The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 */
export declare class WorkerType {
    /**
     * Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
     */
    static readonly STANDARD: WorkerType;
    /**
     * Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
     */
    static readonly G_1X: WorkerType;
    /**
     * Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
     */
    static readonly G_2X: WorkerType;
    /**
     * Custom worker type
     * @param workerType custom worker type
     */
    static of(workerType: string): WorkerType;
    /**
     * The name of this WorkerType, as expected by Job resource.
     */
    readonly name: string;
    private constructor();
}
/**
 * Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 */
export declare enum JobState {
    /**
     * State indicating job run succeeded
     */
    SUCCEEDED = "SUCCEEDED",
    /**
     * State indicating job run failed
     */
    FAILED = "FAILED",
    /**
     * State indicating job run timed out
     */
    TIMEOUT = "TIMEOUT",
    /**
     * State indicating job is starting
     */
    STARTING = "STARTING",
    /**
     * State indicating job is running
     */
    RUNNING = "RUNNING",
    /**
     * State indicating job is stopping
     */
    STOPPING = "STOPPING",
    /**
     * State indicating job stopped
     */
    STOPPED = "STOPPED"
}
/**
 * The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 */
export declare enum MetricType {
    /**
     * A value at a point in time.
     */
    GAUGE = "gauge",
    /**
     * An aggregate number.
     */
    COUNT = "count"
}
/**
 * Interface representing a created or an imported {@link Job}.
 */
export interface IJob extends cdk.IResource, iam.IGrantable {
    /**
     * The name of the job.
     * @attribute
     */
    readonly jobName: string;
    /**
     * The ARN of the job.
     * @attribute
     */
    readonly jobArn: string;
    /**
     * Defines a CloudWatch event rule triggered when something happens with this job.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when this job moves to the input jobState.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onStateChange(id: string, jobState: JobState, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when this job moves to the SUCCEEDED state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onSuccess(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when this job moves to the FAILED state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onFailure(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when this job moves to the TIMEOUT state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onTimeout(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     */
    metric(metricName: string, type: MetricType, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Create a CloudWatch Metric indicating job success.
     */
    metricSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Create a CloudWatch Metric indicating job failure.
     */
    metricFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Create a CloudWatch Metric indicating job timeout.
     */
    metricTimeout(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
declare abstract class JobBase extends cdk.Resource implements IJob {
    abstract readonly jobArn: string;
    abstract readonly jobName: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * Create a CloudWatch Event Rule for this Glue Job when it's in a given state
     *
     * @param id construct id
     * @param options event options. Note that some values are overridden if provided, these are
     *  - eventPattern.source = ['aws.glue']
     *  - eventPattern.detailType = ['Glue Job State Change', 'Glue Job Run Status']
     *  - eventPattern.detail.jobName = [this.jobName]
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     */
    onStateChange(id: string, jobState: JobState, options?: events.OnEventOptions): events.Rule;
    /**
     * Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onSuccess(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onFailure(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onTimeout(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     */
    metric(metricName: string, type: MetricType, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     */
    metricSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     */
    metricFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     */
    metricTimeout(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    private metricJobStateRule;
}
/**
 * Properties for enabling Spark UI monitoring feature for Spark-based Glue jobs.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-spark-ui-jobs.html
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 */
export interface SparkUIProps {
    /**
     * Enable Spark UI.
     */
    readonly enabled: boolean;
    /**
     * The bucket where the Glue job stores the logs.
     *
     * @default a new bucket will be created.
     */
    readonly bucket?: s3.IBucket;
    /**
     * The path inside the bucket (objects prefix) where the Glue job stores the logs.
     *
     * @default '/' - the logs will be written at the root of the bucket
     */
    readonly prefix?: string;
}
/**
 * The Spark UI logging location.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-spark-ui-jobs.html
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 */
export interface SparkUILoggingLocation {
    /**
     * The bucket where the Glue job stores the logs.
     */
    readonly bucket: s3.IBucket;
    /**
     * The path inside the bucket (objects prefix) where the Glue job stores the logs.
     *
     * @default '/' - the logs will be written at the root of the bucket
     */
    readonly prefix?: string;
}
/**
 * Properties for enabling Continuous Logging for Glue Jobs.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-continuous-logging-enable.html
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 */
export interface ContinuousLoggingProps {
    /**
     * Enable continouous logging.
     */
    readonly enabled: boolean;
    /**
     * Specify a custom CloudWatch log group name.
     *
     * @default - a log group is created with name `/aws-glue/jobs/logs-v2/`.
     */
    readonly logGroup?: logs.ILogGroup;
    /**
     * Specify a custom CloudWatch log stream prefix.
     *
     * @default - the job run ID.
     */
    readonly logStreamPrefix?: string;
    /**
     * Filter out non-useful Apache Spark driver/executor and Apache Hadoop YARN heartbeat log messages.
     *
     * @default true
     */
    readonly quiet?: boolean;
    /**
     * Apply the provided conversion pattern.
     *
     * This is a Log4j Conversion Pattern to customize driver and executor logs.
     *
     * @default `%d{yy/MM/dd HH:mm:ss} %p %c{1}: %m%n`
     */
    readonly conversionPattern?: string;
}
/**
 * Attributes for importing {@link Job}.
 */
export interface JobAttributes {
    /**
     * The name of the job.
     */
    readonly jobName: string;
    /**
     * The IAM role assumed by Glue to run this job.
     *
     * @default - undefined
     */
    readonly role?: iam.IRole;
}
/**
 * Construction properties for {@link Job}.
 */
export interface JobProps {
    /**
     * The job's executable properties.
     */
    readonly executable: JobExecutable;
    /**
     * The name of the job.
     *
     * @default - a name is automatically generated
     */
    readonly jobName?: string;
    /**
     * The description of the job.
     *
     * @default - no value
     */
    readonly description?: string;
    /**
     * The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs.
     * Cannot be used for Glue version 2.0 and later - workerType and workerCount should be used instead.
     *
     * @default - 10 when job type is Apache Spark ETL or streaming, 0.0625 when job type is Python shell
     */
    readonly maxCapacity?: number;
    /**
     * The maximum number of times to retry this job after a job run fails.
     *
     * @default 0
     */
    readonly maxRetries?: number;
    /**
     * The maximum number of concurrent runs allowed for the job.
     *
     * An error is returned when this threshold is reached. The maximum value you can specify is controlled by a service limit.
     *
     * @default 1
     */
    readonly maxConcurrentRuns?: number;
    /**
     * The number of minutes to wait after a job run starts, before sending a job run delay notification.
     *
     * @default - no delay notifications
     */
    readonly notifyDelayAfter?: cdk.Duration;
    /**
     * The maximum time that a job run can consume resources before it is terminated and enters TIMEOUT status.
     *
     * @default cdk.Duration.hours(48)
     */
    readonly timeout?: cdk.Duration;
    /**
     * The type of predefined worker that is allocated when a job runs.
     *
     * @default - differs based on specific Glue version
     */
    readonly workerType?: WorkerType;
    /**
     * The number of workers of a defined {@link WorkerType} that are allocated when a job runs.
     *
     * @default - differs based on specific Glue version/worker type
     */
    readonly workerCount?: number;
    /**
     * The {@link Connection}s used for this job.
     *
     * Connections are used to connect to other AWS Service or resources within a VPC.
     *
     * @default [] - no connections are added to the job
     */
    readonly connections?: IConnection[];
    /**
     * The {@link SecurityConfiguration} to use for this job.
     *
     * @default - no security configuration.
     */
    readonly securityConfiguration?: ISecurityConfiguration;
    /**
     * The default arguments for this job, specified as name-value pairs.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html for a list of reserved parameters
     * @default - no arguments
     */
    readonly defaultArguments?: {
        [key: string]: string;
    };
    /**
     * The tags to add to the resources on which the job runs
     *
     * @default {} - no tags
     */
    readonly tags?: {
        [key: string]: string;
    };
    /**
     * The IAM role assumed by Glue to run this job.
     *
     * If providing a custom role, it needs to trust the Glue service principal (glue.amazonaws.com) and be granted sufficient permissions.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/getting-started-access.html
     *
     * @default - a role is automatically generated
     */
    readonly role?: iam.IRole;
    /**
     * Enables the collection of metrics for job profiling.
     *
     * @default - no profiling metrics emitted.
     *
     * @see `--enable-metrics` at https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    readonly enableProfilingMetrics?: boolean;
    /**
     * Enables the Spark UI debugging and monitoring with the specified props.
     *
     * @default - Spark UI debugging and monitoring is disabled.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-spark-ui-jobs.html
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    readonly sparkUI?: SparkUIProps;
    /**
     * Enables continuous logging with the specified props.
     *
     * @default - continuous logging is disabled.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-continuous-logging-enable.html
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    readonly continuousLogging?: ContinuousLoggingProps;
}
/**
 * A Glue Job.
 */
export declare class Job extends JobBase {
    /**
     * Creates a Glue Job
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromJobAttributes(scope: constructs.Construct, id: string, attrs: JobAttributes): IJob;
    /**
     * The ARN of the job.
     */
    readonly jobArn: string;
    /**
     * The name of the job.
     */
    readonly jobName: string;
    /**
     * The IAM role Glue assumes to run this job.
     */
    readonly role: iam.IRole;
    /**
     * The principal this Glue Job is running as.
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The Spark UI logs location if Spark UI monitoring and debugging is enabled.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitor-spark-ui-jobs.html
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    readonly sparkUILoggingLocation?: SparkUILoggingLocation;
    constructor(scope: constructs.Construct, id: string, props: JobProps);
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    private checkNoReservedArgs;
    private executableArguments;
    private setupSparkUI;
    private setupContinuousLogging;
    private codeS3ObjectUrl;
}
export {};
