# coding: utf-8
# pylint: disable=no-init
u"""Реализации справочников и моделей контингента."""
from __future__ import absolute_import

from collections import namedtuple

from m3.db import BaseEnumerate
from six import python_2_unicode_compatible

from .base import BaseCatalogVirtualModel
from .base import BaseEnumerateProductSpecific
from .base import BaseModelView
from .base import load_values


class Okfs(BaseEnumerate):
    u"""Справочник ОКФС."""

    values = {
        11: u"Государственная собственность",
        49: u"Иная смешанная российская собственность",
        20: u"Иностранная собственность",
        14: u"Муниципальная собственность",
        10: u"Российская собственность",
        27: u"Смешанная иностранная собственность",
        17: u"Смешанная российская собственность",
        40: u"Смешанная российская собственность с долей "
            u"государственной собственности",
        42: u"Смешанная российская собственность с долей собственности "
            u"субъектов Российской Федерации",
        41: u"Смешанная российская собственность с долей федеральной "
            u"собственности",
        43: u"Смешанная российская собственность с долями федеральной "
            u"собственности и собственности субъектов Российской Федерации",
        50: u"Собственность благотворительных организаций",
        61: u"Собственность государственных корпораций",
        22: u"Собственность иностранных государств",
        24: u"Собственность иностранных граждан и лиц без гражданства",
        23: u"Собственность иностранных юридических лиц",
        21: u"Собственность международных организаций",
        15: u"Собственность общественных и религиозных "
            u"организаций (объединений)",
        53: u"Собственность общественных объединений",
        51: u"Собственность политических общественных объединений",
        19: u"Собственность потребительской кооперации",
        52: u"Собственность профессиональных союзов",
        54: u"Собственность религиозных объединений",
        18: u"Собственность российских граждан, постоянно "
            u"проживающих за границей",
        13: u"Собственность субъектов Российской Федерации",
        33: u"Совместная муниципальная и иностранная собственность",
        30: u"Совместная российская и иностранная собственность",
        35: u"Совместная собственность общественных и религиозных "
            u"организаций (объединений) и  иностранная собственность",
        32: u"Совместная собственность субъектов Российской Федерации "
            u"и иностранная собственность",
        31: u"Совместная федеральная и иностранная собственность",
        34: u"Совместная частная и иностранная собственность",
        12: u"Федеральная собственность",
        16: u"Частная собственность"
    }


class Citizenship(BaseEnumerate):
    u"""Гражданство."""

    RF_CITIZEN = 1
    DOUBLE_CITIZEN = 2
    EXTERNAL_CITIZEN = 3
    WITHOUT_CITIZEN = 4

    values = {
        RF_CITIZEN: u"Гражданин Российской Федерации",
        DOUBLE_CITIZEN: u"Гражданин Российской Федерации и "
                        u"иностранного государства (двойное гражданство)",
        EXTERNAL_CITIZEN: u"Иностранный гражданин",
        WITHOUT_CITIZEN: u"Лицо без гражданства"
    }

    # соотнесение значений из КО с тем, как представлено гражданство в РИС
    # основной справочник, в `values`, не соответствует тому, что
    #   указано в confluence по этому полю:
    # https://conf.bars.group/pages/viewpage.action?pageId=15292952 п. 2.1.6
    values_received_from_contingent = {
        0: RF_CITIZEN,
        1: EXTERNAL_CITIZEN,
        2: WITHOUT_CITIZEN,
        3: DOUBLE_CITIZEN
    }


class PhysicalCultureGroups(BaseEnumerate):
    u"""Физкультурные группы."""

    MAIN = 21
    PREPARATORY = 22
    SPECIAL_A = 23
    SPECIAL_B = 24

    values = {
        MAIN: u"Основная",
        PREPARATORY: u"Подготовительная",
        SPECIAL_A: u"Специальная А",
        SPECIAL_B: u"Специальная Б"

    }


class HealthGroups(BaseEnumerate):
    u"""Справочник "Группы здоровья"."""

    HEALTHY = 11
    SLIGHT_DEVIATIONS = 12
    CHRONIC_DISEASES_AND_WELLNESS = 13
    CHRONIC_DISEASES_AND_POOR_HEALTH = 14
    CHRONIC_DISEASE_AND_OBSERVED = 15

    FIRST_GROUP_18_YEARS_OLD = 16
    SECOND_GROUP_18_YEARS_OLD = 17
    THIRD_GROUP_18_YEARS_OLD = 18

    BEFORE_18_YEAR_CODES = [
        HEALTHY,
        SLIGHT_DEVIATIONS,
        CHRONIC_DISEASES_AND_WELLNESS,
        CHRONIC_DISEASES_AND_POOR_HEALTH,
        CHRONIC_DISEASE_AND_OBSERVED
    ]

    AFTER_18_YEAR_CODES = [
        FIRST_GROUP_18_YEARS_OLD,
        SECOND_GROUP_18_YEARS_OLD,
        THIRD_GROUP_18_YEARS_OLD
    ]

    values = {
        HEALTHY: u"Группа 1 - здоровые",
        SLIGHT_DEVIATIONS: u"Группа 2 - с незначительными отклонениями",
        CHRONIC_DISEASES_AND_WELLNESS: u"Группа 3 - с хроническими "
                                       u"заболеваниями и хорошим "
                                       u"самочувствием, либо с временными "
                                       u"отклонениями в состоянии здоровья",
        CHRONIC_DISEASES_AND_POOR_HEALTH: u"Группа 4 - с хроническими "
                                          u"заболеваниями и плохим "
                                          u"самочувствием",
        CHRONIC_DISEASE_AND_OBSERVED: u"Группа 5 - с хроническими "
                                      u"заболеваниями и "
                                      u"наблюдаются в специальных лечебницах",
        FIRST_GROUP_18_YEARS_OLD: u'Группа 1 (18 лет и старше)',
        SECOND_GROUP_18_YEARS_OLD: u'Группа 2 (18 лет и старше)',
        THIRD_GROUP_18_YEARS_OLD: u'Группа 3 (18 лет и старше)',
    }


class DisabilityGroups(BaseEnumerate):
    u"""Группы инвалидности."""

    FIRST_GROUP = 11
    SECOND_GROUP = 12
    THIRD_GROUP = 13
    DISABLED_CHILD = 14

    values = {
        FIRST_GROUP: u"Первая группа",
        SECOND_GROUP: u"Вторая группа",
        THIRD_GROUP: u"Третья группа",
        DISABLED_CHILD: u"Ребенок-инвалид (для лиц до 18 лет)"
    }


class CertainCategoriesOfDisability(BaseEnumerate):
    u"""Отдельные категории инвалидности."""

    DISABLED_SINCE_CHILDHOOD = 21
    ARMY_INVALID = 22

    values = {
        DISABLED_SINCE_CHILDHOOD: u"Инвалид с детства",
        ARMY_INVALID: u"Инвалид вследствие военной травмы или заболевания, "
                      u"полученного в период прохождения военной службы"
    }


class OrganizationKindByFounder(BaseEnumerate):
    u"""Вид организации по учредителю."""

    RF = 1
    RF_SUBJECT = 2
    MO = 3
    RF_COMMERCIAL_ORGANIZATION = 4
    EXTERNAL_COMMERCIAL_ORGANIZATION = 5
    RF_NON_COMMERCIAL_ORGANIZATION = 6
    EXTERNAL_NON_COMMERCIAL_ORGANIZATION = 7
    RF_RELIGION_ORGANIZATION = 8
    EXTERNAL_RELIGION_ORGANIZATION = 9
    RF_CITIZENSHIP = 10
    EXTERNAL_CITIZENSHIP = 11

    values = {
        RF: u"Российская Федерация",
        RF_SUBJECT: u"Субъект Российской Федерации",
        MO: u"Муниципальное образование",
        RF_COMMERCIAL_ORGANIZATION: u"Российские коммерческие организации",
        EXTERNAL_COMMERCIAL_ORGANIZATION: u"Иностранные коммерческие "
                                          u"организации",
        RF_NON_COMMERCIAL_ORGANIZATION: u"Российские некоммерческие "
                                        u"организации",
        EXTERNAL_NON_COMMERCIAL_ORGANIZATION: u"Иностранные некоммерческие "
                                              u"организации",
        RF_RELIGION_ORGANIZATION: u"Российские религиозные организации",
        EXTERNAL_RELIGION_ORGANIZATION: u"Иностранные религиозные "
                                        u"организации",
        RF_CITIZENSHIP: u"Граждане Российской Федерации",
        EXTERNAL_CITIZENSHIP: u"Иностранные граждане"
    }


class OrganizationStatus(BaseEnumerate):
    u"""Статусы организаций."""

    FUNCTIONS = 1
    CAPITAL_REPAIRS = 2
    RECONSTRUCTION = 3
    ACTIVITY_STOPPED = 4
    CONTINGENT_MISSING = 5
    PENDING_THE_OPENING = 6
    LIQUIDATED = 7
    CLOSED = 8
    JOINED_OTHER_ORGANIZATIONS = 9

    values = {
        FUNCTIONS: u"Функционирует",
        CAPITAL_REPAIRS: u"Капитальный ремонт",
        RECONSTRUCTION: u"Реконструкция",
        ACTIVITY_STOPPED: u"Деятельность приостановлена",
        CONTINGENT_MISSING: u"Контингент отсутствует",
        PENDING_THE_OPENING: u"Ожидает открытия",
        LIQUIDATED: u"Ликвидирована",
        CLOSED: u"Закрыта",
        JOINED_OTHER_ORGANIZATIONS: u"Присоединена к другой организации"
    }


class EduProgramKind(BaseEnumerateProductSpecific):
    u"""
    Справочник образовательных программ.

    Содержит все виды обр.программ по УФТТ
    """

    JUNIOR_COMMON = 12
    DEFAULT_COMMON = 13
    MIDDLE_COMMON = 14

    values = {
        1: u"Основная общеобразовательная программа",
        11: u"Образовательная программа дошкольного образования",
        JUNIOR_COMMON: (
            u"Образовательная программа начального общего образования"
        ),
        DEFAULT_COMMON: (
            u"Образовательная программа основного общего образования"
        ),
        MIDDLE_COMMON: (
            u"Образовательная программа среднего общего образования"
        ),
        15: u"Адаптированная основная общеобразовательная программа",
        2: u"Основная профессиональная образовательная программа",
        21: u"Образовательная программа среднего профессионального"
            u" образования",
        211: u"Программа подготовки квалифицированных рабочих, служащих",
        212: u"Программа подготовки специалистов среднего звена",
        23: u"Образовательная программа, адаптированная для обучения лиц"
            u" с ограниченными возможностями здоровья",
        3: u"Основная программа профессионального обучения",
        31: u"Программа профессиональной подготовки по профессиям рабочих,"
            u" должностям служащих",
        32: u"Программа переподготовки рабочих, служащих,",
        33: u"Программа повышения квалификации рабочих, служащих.",
        4: u"Дополнительная образовательная программа",
        41: u"Дополнительная общеобразовательная программа",
        411: u"Дополнительная общеразвивающая программа",
        412: u"Дополнительная предпрофессиональная программа",
        42: u"Дополнительная профессиональная программа",
        421: u"Программа повышения квалификации",
        422: u"Программа профессиональной переподготовки",
        5: u"Адаптированная образовательная программа",
        51: u"Адаптированная основная общеобразовательная программа",
        52: u"Образовательная программа, адаптированная для обучения лиц с"
            u" ограниченными возможностями здоровья",
    }

    # коды, соответствующие видам обр.программ в Электронной Школе
    WEBEDU_CODES = (
        1, 11, 12, 13, 14, 15, 211, 212, 4, 41, 411, 412
    )

    # коды, соответствующие видам обр.программ в Электронном Детском Саду
    KINDER_CODES = ()

    # коды, соответствующие видам обр.программ в Электронном Колледже
    SSUZ_CODES = (211, 212)

    # коды, соответствующие видам обр.программ в Электронном Доп.образовании
    EXTEDU_CODES = ()


class EduDocumentOnEducationTypes(BaseEnumerate):
    u"""Справочник 'Типы документов об образовании'."""

    values = {
        11: u'Аттестат о среднем общем образовании',
        12: u'Аттестат об основном общем образовании',
        4: u'Справка об обучении',
        41: u'Свидетельство об обучении для лиц с ограниченными возможностями '
            u'здоровья (с различными формами умственной отсталости)',
        6: u'Другой документ об образовании',
        7: u'Другой документ об обучении'
    }


class DifficultSituations(BaseEnumerateProductSpecific):
    u"""Трудные жизненные ситуации."""

    DISABILITY = 2
    LIMITED_ABILITIES = 3
    DEAF = 301
    BLINDLY = 302
    MUTE = 303
    IMMOBILIZE = 304
    PARALYSIS = 305
    INTELLIGENT_VIOLATION = 306
    AUTISTIC = 307
    IMPAIRED_MENTAL = 308
    MENTAL_ILLNESS = 309
    NERVOUS_SYSTEM_DISEASE = 310
    COMPLEX_DEFECTS = 311

    values = {
        101: u"Дети, оставшиеся без попечения родителей "
             u"по причине смерти родителей",
        102: u"Дети, оставшиеся без попечения родителей "
             u"по причине лишения родителей родительских прав",
        103: u"Дети, оставшиеся без попечения родителей по "
             u"причине ограничения родителей в родительских правах",
        104: u"Дети, оставшиеся без попечения родителей по "
             u"причине признания родителей недееспособными",
        105: u"Дети, оставшиеся без попечения родителей по "
             u"причине болезни родителей",
        106: u"Дети, оставшиеся без попечения родителей по "
             u"причине длительного отсутствия родителей",
        DISABILITY: u"Дети-инвалиды",
        LIMITED_ABILITIES: u"Дети с ограниченными возможностями здоровья",
        DEAF: u"Дети с нарушениями слуха",
        BLINDLY: u"Дети с нарушениями зрения",
        MUTE: u"Дети с тяжелыми нарушениями речи",
        IMMOBILIZE: u"Дети с нарушением опорно-двигательного аппарата",
        PARALYSIS: u"Дети с детским церебральным параличом",
        INTELLIGENT_VIOLATION: u"Дети с интеллектуальными нарушениями",
        AUTISTIC: u"Дети с расстройством аутистического спектра",
        IMPAIRED_MENTAL: u"Дети с задержкой психического развития",
        MENTAL_ILLNESS: u"Дети с психическими заболеваниями",
        NERVOUS_SYSTEM_DISEASE: u"Дети с заболеваниями нервной системы",
        COMPLEX_DEFECTS: u"Дети со сложной структурой дефекта",
        4: u"Дети - жертвы вооруженных и межнациональных конфликтов, "
           u"экологических и техногенных катастроф, стихийных бедствий",
        5: u"Дети - жертвы экологических и техногенных катастроф, "
           u"стихийных бедствий",
        6: u"Дети из семей беженцев и вынужденных переселенцев",
        7: u"Дети, оказавшиеся в экстремальных условиях",
        8: u"Дети - жертвы насилия",
        9: u"Дети, отбывающие наказание в виде лишения "
           u"свободы в воспитательных колониях",
        10: u"Дети, с девиантным (общественно опасным) поведением",
        11: u"Дети, проживающие в малоимущих семьях",
        12: u"Дети с отклонениями в поведении",
        13: u"Дети, жизнедеятельность которых объективно "
            u"нарушена в результате сложившихся обстоятельств"
    }
    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = tuple(
        code
        for code in values
        if code != 9
    )

    # Коды тяжелых жизненных ситуаций, которые требуют обучение с ОВЗ.
    NEED_OVZ_LEARNING = (
        DEAF,
        BLINDLY,
        MUTE,
        IMMOBILIZE,
        PARALYSIS,
        INTELLIGENT_VIOLATION,
        AUTISTIC,
        IMPAIRED_MENTAL,
        MENTAL_ILLNESS,
        NERVOUS_SYSTEM_DISEASE,
        COMPLEX_DEFECTS,
    )


class LegalDelegateTypes(BaseEnumerateProductSpecific):
    u"""Тип законного представителя."""

    PARENT = 1
    CURATOR = 2
    TRUSTEE = 3
    AGENCY_OF_TRUSTEE = 4
    ADOPTIVE_PARENT = 5
    EDUCATION_AGENCY_LEADER = 6
    LEGAL_REPRESENTATIVE = 7

    values = {
        PARENT: u"Родитель",
        CURATOR: u"Опекун",
        TRUSTEE: u"Попечитель",
        AGENCY_OF_TRUSTEE: u"Орган опеки и попечительства",
        ADOPTIVE_PARENT: u"Приемный родитель",
        EDUCATION_AGENCY_LEADER: u"Руководитель воспитательного, лечебного и "
                                 u"иного учреждения, в котором ребенок "
                                 u"находится на полном "
                                 u"государственном обеспечении",
        LEGAL_REPRESENTATIVE: u"Законный представитель"
    }

    # коды, соответствующие типам законного представителя в Электронной Школе
    WEBEDU_CODES = ()

    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = (2, 3, 4, 5, 6)

    # коды, соответствующие типам законного представителя в
    # Электронном Колледже
    SSUZ_CODES = (1, 2, 3, 4, 5, 6)

    # коды, соответствующие типам законного представителя в
    # Электронном Доп.образовании
    EXTEDU_CODES = ()


@python_2_unicode_compatible
class OkoguVirtualModel(BaseCatalogVirtualModel):
    u"""Виртуальная модель справочника ОКОГУ."""

    id_field = 'code'

    data = load_values("okogu.json")

    def __str__(self):
        u"""Строковое представление записи в виртуальной модели."""
        return self.full_name

    class Meta:
        verbose_name = u'ОКОГУ'


class Okogu(BaseModelView):
    u"""
    Справочник ОКОГУ.

    ОКОГУ - Общероссийский классификатор органов государственной власти и
    управления.
    """

    model = OkoguVirtualModel

    value_field = "code"
    display_field = "full_name"

    KINDER_CODES = ('2300223', '3300000', '3300100',
                    '3300200', '3300300', '3300400',
                    '3300500', '3500000', '3500100',
                    '3500200', '3500300', '3500400',
                    '3500500', '4210007', '4210008',
                    '4210009', '4210014', '4210015')


class Okopf(BaseEnumerateProductSpecific):
    u"""Справочник ОКОПФ."""

    values = {
        "1 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, "
                   u"ЯВЛЯЮЩИХСЯ КОММЕРЧЕСКИМИ КОРПОРАТИВНЫМИ ОРГАНИЗАЦИЯМИ",
        "1 10 00": u"Хозяйственные товарищества",
        "1 10 51": u"Полные товарищества",
        "1 10 64": u"Товарищества на вере (коммандитные товарищества)",
        "1 20 00": u"Хозяйственные общества",
        "1 22 00": u"Акционерные общества",
        "1 22 47": u"Публичные акционерные общества",
        "1 22 67": u"Непубличные акционерные общества",
        "1 23 00": u"Общества с ограниченной ответственностью",
        "1 30 00": u"Хозяйственные партнерства",
        "1 40 00": u"Производственные кооперативы (артели)",
        "1 41 00": u"Сельскохозяйственные производственные кооперативы",
        "1 41 53": u"Сельскохозяйственные артели (колхозы)",
        "1 41 54": u"Рыболовецкие артели (колхозы)",
        "1 41 55": u"Кооперативные хозяйства (коопхозы)",
        "1 42 00": u"Производственные кооперативы "
                   u"(кроме сельскохозяйственных "
                   u"производственных кооперативов)",
        "1 53 00": u"Крестьянские (фермерские) хозяйства",
        "1 90 00": u"Прочие юридические лица, являющиеся "
                   u"коммерческими организациями",
        "2 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, "
                   u"ЯВЛЯЮЩИХСЯ НЕКОММЕРЧЕСКИМИ КОРПОРАТИВНЫМИ ОРГАНИЗАЦИЯМИ",
        "2 01 00": u"Потребительские кооперативы",
        "2 01 01": u"Гаражные и гаражно-строительные кооперативы",
        "2 01 02": u"Жилищные или жилищно-строительные кооперативы",
        "2 01 03": u"Жилищные накопительные кооперативы",
        "2 01 04": u"Кредитные потребительские кооперативы",
        "2 01 05": u"Кредитные потребительские кооперативы граждан",
        "2 01 06": u"Кредитные кооперативы второго уровня",
        "2 01 07": u"Потребительские общества",
        "2 01 08": u"Общества взаимного страхования",
        "2 01 09": u"Сельскохозяйственные потребительские "
                   u"перерабатывающие кооперативы",
        "2 01 10": u"Сельскохозяйственные потребительские "
                   u"сбытовые (торговые) кооперативы",
        "2 01 11": u"Сельскохозяйственные потребительские "
                   u"обслуживающие кооперативы",
        "2 01 12": u"Сельскохозяйственные потребительские "
                   u"снабженческие кооперативы",
        "2 01 13": u"Сельскохозяйственные потребительские "
                   u"садоводческие кооперативы",
        "2 01 14": u"Сельскохозяйственные потребительские "
                   u"огороднические кооперативы",
        "2 01 15": u"Сельскохозяйственные потребительские "
                   u"животноводческие кооперативы",
        "2 01 20": u"Садоводческие, огороднические или дачные "
                   u"потребительские кооперативы",
        "2 01 21": u"Фонды проката",
        "2 02 00": u"Общественные организации",
        "2 02 01": u"Политические партии",
        "2 02 02": u"Профсоюзные организации",
        "2 02 10": u"Общественные движения",
        "2 02 11": u"Органы общественной самодеятельности",
        "2 02 17": u"Территориальные общественные самоуправления",
        "2 06 00": u"Ассоциации (союзы)",
        "2 06 01": u"Ассоциации (союзы) экономического взаимодействия "
                   u"субъектов Российской Федерации",
        "2 06 03": u"Советы муниципальных образований "
                   u"субъектов Российской Федерации",
        "2 06 04": u"Союзы (ассоциации) кредитных кооперативов",
        "2 06 05": u"Союзы (ассоциации) кооперативов",
        "2 06 06": u"Союзы (ассоциации) общественных объединений",
        "2 06 07": u"Союзы (ассоциации) общин малочисленных народов",
        "2 06 08": u"Союзы потребительских обществ",
        "2 06 09": u"Адвокатские палаты",
        "2 06 10": u"Нотариальные палаты",
        "2 06 11": u"Торгово-промышленные палаты",
        "2 06 12": u"Объединения работодателей",
        "2 06 13": u"Объединения фермерских хозяйств",
        "2 06 14": u"Некоммерческие партнерства",
        "2 06 15": u"Адвокатские бюро",
        "2 06 16": u"Коллегии адвокатов",
        "2 06 17": u"Садоводческие, огороднические или "
                   u"дачные некоммерческие партнерства",
        "2 06 18": u"Ассоциации (союзы) садоводческих, огороднических и "
                   u"дачных некоммерческих объединений",
        "2 06 19": u"Саморегулируемые организации",
        "2 06 20": u"Объединения (ассоциации и союзы) "
                   u"благотворительных организаций",
        "2 07 00": u"Товарищества собственников недвижимости",
        "2 07 01": u"Садоводческие, огороднические или дачные "
                   u"некоммерческие товарищества",
        "2 07 16": u"Товарищества собственников жилья",
        "2 11 00": u"Казачьи общества, внесенные в государственный "
                   u"реестр казачьих обществ в Российской Федерации",
        "2 12 00": u"Общины коренных малочисленных народов "
                   u"Российской Федерации",
        "3 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ОРГАНИЗАЦИЙ, "
                   u"СОЗДАННЫХ БЕЗ ПРАВ ЮРИДИЧЕСКОГО ЛИЦА",
        "3 00 01": u"Представительства юридических лиц",
        "3 00 02": u"Филиалы юридических лиц",
        "3 00 03": u"Обособленные подразделения юридических лиц",
        "3 00 04": u"Структурные подразделения обособленных "
                   u"подразделений юридических лиц",
        "3 00 05": u"Паевые инвестиционные фонды",
        "3 00 06": u"Простые товарищества",
        "3 00 08": u"Районные суды, городские суды, "
                   u"межрайонные суды (районные суды)",
        "4 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ "
                   u"МЕЖДУНАРОДНЫХ ОРГАНИЗАЦИЙ, ОСУЩЕСТВЛЯЮЩИХ ДЕЯТЕЛЬНОСТЬ "
                   u"НА ТЕРРИТОРИИ РОССИЙСКОЙ ФЕДЕРАЦИИ",
        "4 00 01": u"Межправительственные международные организации",
        "4 00 02": u"Неправительственные международные организации",
        "5 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ДЛЯ ДЕЯТЕЛЬНОСТИ "
                   u"ГРАЖДАН (ФИЗИЧЕСКИХ ЛИЦ)",
        "5 01 00": u"Организационно-правовые формы для коммерческой "
                   u"деятельности граждан",
        "5 01 01": u"Главы крестьянских (фермерских) хозяйств",
        "5 01 02": u"Индивидуальные предприниматели",
        "5 02 00": u"Организационно-правовые формы для деятельности граждан, "
                   u"не отнесенной к предпринимательству",
        "5 02 01": u"Адвокаты, учредившие адвокатский кабинет",
        "5 02 02": u"Нотариусы, занимающиеся частной практикой",
        "6 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, "
                   u"ЯВЛЯЮЩИХСЯ КОММЕРЧЕСКИМИ УНИТАРНЫМИ ОРГАНИЗАЦИЯМИ",
        "6 50 00": u"Унитарные предприятия",
        "6 51 00": u"Унитарные предприятия, основанные на праве оперативного "
                   u"управления (казенные предприятия)",
        "6 51 41": u"Федеральные казенные предприятия",
        "6 51 42": u"Казенные предприятия субъектов Российской Федерации",
        "6 51 43": u"Муниципальные казенные предприятия",
        "6 52 00": u"Унитарные предприятия, основанные на "
                   u"праве хозяйственного ведения",
        "6 52 41": u"Федеральные государственные унитарные предприятия",
        "6 52 42": u"Государственные унитарные предприятия субъектов "
                   u"Российской Федерации",
        "6 52 43": u"Муниципальные унитарные предприятия",
        "7 00 00": u"ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, "
                   u"ЯВЛЯЮЩИХСЯ НЕКОММЕРЧЕСКИМИ УНИТАРНЫМИ ОРГАНИЗАЦИЯМИ",
        "7 04 00": u"Фонды",
        "7 04 01": u"Благотворительные фонды",
        "7 04 02": u"Негосударственные пенсионные фонды",
        "7 04 03": u"Общественные фонды",
        "7 04 04": u"Экологические фонды",
        "7 14 00": u"Автономные некоммерческие организации",
        "7 15 00": u"Религиозные организации",
        "7 16 00": u"Публично-правовые компании",
        "7 16 01": u"Государственные корпорации",
        "7 16 02": u"Государственные компании",
        "7 16 10": u"Отделения иностранных некоммерческих "
                   u"неправительственных организаций",
        "7 50 00": u"Учреждения",
        "7 51 00": u"Учреждения, созданные Российской Федерацией",
        "7 51 01": u"Федеральные государственные автономные учреждения",
        "7 51 03": u"Федеральные государственные бюджетные учреждения",
        "7 51 04": u"Федеральные государственные казенные учреждения",
        "7 52 00": u"Учреждения, созданные субъектами Российской Федерации",
        "7 52 01": u"Государственные автономные учреждения "
                   u"субъектов Российской Федерации",
        "7 52 03": u"Государственные бюджетные учреждения субъектов "
                   u"Российской Федерации",
        "7 52 04": u"Государственные казенные учреждения субъектов "
                   u"Российской Федерации",
        "7 53 00": u"Государственные академии наук",
        "7 54 00": u"Учреждения, созданные муниципальными образованиями",
        "7 54 01": u"Муниципальные автономные учреждения",
        "7 54 03": u"Муниципальные бюджетные учреждения",
        "7 54 04": u"Муниципальные казенные учреждения",
        "7 55 00": u"Частные учреждения",
        "7 55 02": u"Благотворительные учреждения",
        "7 55 05": u"Общественные учреждения"
    }
    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = ('3 00 02', '5 01 02', '7 50 00',
                    '7 51 00', '7 51 01', '7 51 03',
                    '7 51 04', '7 52 00', '7 52 01',
                    '7 52 03', '7 52 04', '7 54 00',
                    '7 54 01', '7 54 03', '7 54 04',
                    '7 55 00', '7 55 05')


class DocumentConfirmingTypes(BaseEnumerate):
    u"""Справочник "Тип документа, подтверждающего права" """

    ADOPTION = 401
    BIRTH_CERT = 402
    TRUSTEE = 403
    OTHER_DOC = 499

    values = {
        ADOPTION: u'Решение суда об установлении усыновления',
        BIRTH_CERT: u'Свидетельство о рождении',
        TRUSTEE: u'Назначение органом опеки опекуна',
        OTHER_DOC: u'Другой документ, подтверждающий представительство',
    }


class IdentityDocumentsTypes(BaseEnumerate):

    u"""Справочник "Типы документов удостоверяющих личность".

    Ключи, это id из таблицы certificate,
    из которой перенесены данные в этот справочник.

    Значения это кортежи в которых, первое значение это значение по ключу,
    а второе (не обязательное), код УФТТ.
    """

    BIRTH_CERT = 1
    PASSPORT = 2
    DOC = 3
    TEMP_CARD_IDENT_RF = 4
    DOM_PERM_RF = 5
    PASSPORT_ALIEN = 6
    FOREIGN_PASSPORT = 7
    ARMY_TICK = 8
    DIP_PASS_RF = 11
    PASSPORT_USSR = 13
    PASSPORT_MINMOR = 15
    PASSPORT_NAUTA = 16
    TEMP_RF = 17
    REFUGEE_RF = 18
    BIRTH_CERT_INTR = 19
    FROM_PRISON = 20
    REFUGEE = 21
    OFFICER = 22
    MILLITARY_RF = 23
    TEMP_MILLITARY = 24
    WITHOUT_CITIZEN_RF = 25
    DOC_CITIZEN_RF = 26
    DOC_REFUGEE_RF = 27
    DOC_TEMP_RF = 28
    DOC_PERM_RF = 29
    CERTIFICATE_PERM_RF = 30

    # Перечень ключей словаря values, содержащие
    # типы документов, которые могут не иметь серии
    SERIESLESS_DOCS = (3, 4, 6, 15, 18, 19, 20, 21, 24, 25, 26, 27, 28)

    Value = namedtuple('Value', ['value', 'uftt_code'])

    values = {
        BIRTH_CERT: Value(
            u'Свидетельство о рождении', 11
        ),
        PASSPORT: Value(
            u'Паспорт гражданина РФ', 12
        ),
        DOC: Value(
            u'Другой документ, удостоверяющий личность', 4
        ),
        TEMP_CARD_IDENT_RF: Value(
            u'Временное удостоверение личности гражданина РФ', 17
        ),
        PASSPORT_ALIEN: Value(
            u'Паспорт иностранного гражданина', 21
        ),
        FOREIGN_PASSPORT: Value(
            u'Загранпаспорт гражданина РФ', 13
        ),
        ARMY_TICK: Value(
            u'Военный билет', 15
        ),
        DIP_PASS_RF: Value(
            u'Дипломатический паспорт гражданина Российской Федерации', None
        ),
        PASSPORT_USSR: Value(
            u'Паспорт гражданина СССР', 4
        ),
        PASSPORT_MINMOR: Value(
            u'Паспорт Минморфлота', 4
        ),
        PASSPORT_NAUTA: Value(
            u'Паспорт моряка', 4
        ),
        TEMP_RF: Value(
            u'Разрешение на временное проживание в Российской Федерации', 28
        ),
        REFUGEE_RF: Value(
            u'Свидетельство о рассмотрении ходатайства о признании  беженцем '
            u'на территории Российской Федерации', 29
        ),
        BIRTH_CERT_INTR: Value(
            u'Свидетельство о рождении, выданное уполномоченным органом '
            u'иностранного государства', 31
        ),
        FROM_PRISON: Value(
            u'Справка об освобождении из места лишения свободы', 4
        ),
        REFUGEE: Value(
            u'Удостоверение личности лица, признанного беженцем', 24
        ),
        OFFICER: Value(
            u'Удостоверение личности офицера', 4
        ),
        MILLITARY_RF: Value(
            u'Удостоверение личности военнослужащего РФ', 14
        ),
        TEMP_MILLITARY: Value(
            u'Временное удостоверение, выданное взамен военного билета', 16
        ),
        WITHOUT_CITIZEN_RF: Value(
            u'Удостоверение личности лица без гражданства в РФ', 22
        ),
        DOC_CITIZEN_RF: Value(
            u'Удостоверение личности отдельных категорий лиц, находящихся на '
            u'территории РФ, подавших заявление о признании гражданами РФ или '
            u'о приеме в гражданство РФ', 23
        ),
        DOC_REFUGEE_RF: Value(
            u'Удостоверение личности лица, ходатайствующего о признании '
            u'беженцем на территории РФ', 25
        ),
        DOC_TEMP_RF: Value(
            u'Удостоверение личности лица, получившего временное '
            u'убежище на территории РФ', 26),
        DOC_PERM_RF: Value(
            u'Вид на жительство в Российской Федерации', 27
        ),
        CERTIFICATE_PERM_RF: Value(
            u'Свидетельство о предоставлении временного убежища на территории '
            u'Российской Федерации', 30
        ),
    }

    @classmethod
    def get_full_items(cls):
        u"""Возвращает все значения справочника."""
        result = []

        for item in cls.get_items():
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_uftt_items(cls):
        u"""Возвращает значения справочника, у которых есть код УФТТ."""
        result = []

        for item in cls.get_items():
            if item[1].uftt_code is None:
                continue
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_items_without_uftt(cls):
        u"""Возвращает значения справочника, у которых нет кода УФТТ."""
        result = []

        for item in cls.get_items():
            if not item[1].uftt_code is None:
                continue
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_value_by_id(cls, key):
        u"""Возвращает значение по ключу, если ключа нет возвращает None."""
        try:
            return cls.values[key].value
        except KeyError:
            return

    @classmethod
    def __append_to_result(cls, item, result):
        u"""
        Добавляет значения к результирующему списку.

        :param item: tuple, например (ключ1, (u"значение", УФТТ код))
        :param result: [(key1, value1), (key2, value2), .. (keyN, valueN)]
        :rtype : list
        """
        key = item[0]
        value = item[1].value
        result.append((key, value))


class AdaptationType(BaseEnumerate):
    u"""Вид адаптированности."""

    HALF_BLIND = 201
    BLIND = 202
    HALF_DEAF = 203
    DEAF = 204
    HALF_DEAF_HARD = 205
    WEAK = 206
    SPEECH_PATHOLOGY = 207
    DELAYED = 208
    RETARDED = 209
    RETARDED_HARD = 210
    OTHER = 211
    AUTISTIC_DISORDER = 212

    values = {
        HALF_BLIND: u"для слабовидящих обучающихся",
        BLIND: u"для слепых обучающихся",
        HALF_DEAF: u"для слабослышащих",
        DEAF: u"для глухих",
        HALF_DEAF_HARD:
            u"для слабослышащих обучающихся, имеющих сложную структуру "
            u"дефекта (нарушение слуха и задержка психического развития)",
        WEAK:
            u"для обучающихся, имеющих нарушения опорно-двигательного "
            u"аппарата",
        SPEECH_PATHOLOGY: u"для обучающихся, имеющих тяжелые нарушения речи",
        DELAYED: u"для обучающихся с задержкой психического развития",
        RETARDED: u"для обучающихся с умственной отсталостью",
        RETARDED_HARD:
            u"для обучающихся с умственной отсталостью, имеющих сложную "
            u"структуру дефекта",
        OTHER: u"для обучающихся с иными ограничениями здоровья",
        AUTISTIC_DISORDER:
            u"для обучающихся с расстройствами аутистического спектра",
    }


class TypeTrainingLongTermTreatment(BaseEnumerate):
    u"""Справочник "Виды обучения при длительном лечении"."""

    IS_HOME_STUDY = 1
    IS_MED_ORG_STUDY = 2
    IS_HEALING_ORG_STUDY = 3

    values = {
        IS_HOME_STUDY: u'Обучение на дому',
        IS_MED_ORG_STUDY: u'Обучение в медицинской организации',
        IS_HEALING_ORG_STUDY: u'Обучение в организации, осуществляющей '
                              u'лечение, оздоровление и (или) отдых',
    }


class EducationOrganization(BaseEnumerateProductSpecific):
    u"""Образовательная организация."""

    PROFESSIONAL_ORGANIZATION_CODE = 14
    EDUCATION_ORGANIZATION_CODE = 1
    INDIVIDUAL_BUSINESS = 3
    FOR_THE_DEAF = 1301
    FOR_THE_HEARING_IMPAIRED = 1302
    FOR_LATE_DEAFENED = 1303
    FOR_THE_BLIND = 1304
    FOR_THE_VISUALLY_IMPAIRED = 1305
    WITH_SEVERE_SPEECH_DISORDERS = 1306
    WITH_DISORDERS_OF_MS = 1307
    WITH_MENTAL_RETARDATION = 1308
    WITH_AUTISM_DISORDERS = 1310
    WITH_SEVERE_DEFECTS = 1311
    FOR_OTHERS_WITH_DISORDERS = 1312

    values = {
        EDUCATION_ORGANIZATION_CODE: u"Образовательная организация",
        11: u"Дошкольная образовательная организация",
        12: u"Общеобразовательная организация",
        13: u"ОО, с деятельностью по адаптированным программам",
        FOR_THE_DEAF: (
            u"ОО, с деятельностью по адаптированным программам (для глухих)"),
        FOR_THE_HEARING_IMPAIRED: (
            u"ОО, с деятельностью по адаптированным программам "
            u"(для слабослышащих)"),
        FOR_LATE_DEAFENED: (
            u"ОО, с деятельностью по адаптированным программам "
            u"(для позднооглохших)"),
        FOR_THE_BLIND: (
            u"ОО, с деятельностью по адаптированным программам (для слепых)"),
        FOR_THE_VISUALLY_IMPAIRED: (
            u"ОО, с деятельностью по адаптированным программам "
            u"(для слабовидящих)"),
        WITH_SEVERE_SPEECH_DISORDERS: (
            u"ОО, с деятельностью по адаптированным "
            u"программам (с тяжелыми нарушениями речи)"),
        WITH_DISORDERS_OF_MS: (
            u"ОО, с деятельностью по адаптированным программам "
            u"(с нарушениями опорно-двигательного аппарата)"),
        WITH_MENTAL_RETARDATION: (
            u"ОО, с деятельностью по адаптированным"
            u" программам (с задержкой психического развития)"),
        1309: u"ОО, с деятельностью по адаптированным"
              u" программам (с умственной отсталостью)",
        WITH_AUTISM_DISORDERS: (
            u"ОО, с деятельностью по адаптированным"
            u" программам (с расстройствами аутистического спектра)"),
        WITH_SEVERE_DEFECTS: u"ОО, с деятельностью по адаптированным"
                             u" программам (со сложными дефектами)",
        FOR_OTHERS_WITH_DISORDERS: (
            u"ОО, с деятельностью по адаптированным программам "
            u"(для других обучающихся с ОВЗ)"),
        PROFESSIONAL_ORGANIZATION_CODE: u"Профессиональная образовательная"
                                        u" организация",
        15: u"Образовательная организация высшего образования",
        16: u"Организация дополнительного образования",
        17: u"Организация дополнительного профессионального образования",
        2: u"Организация, осуществляющая обучение",
        21: u"Научная организация",
        22: u"Центр психолого-педагогической, "
            u"медицинской и социальной помощи",
        23: u"Организация для детей-сирот и детей, "
            u"оставшихся без попечения родителей",
        24: u"Организация, осуществляющая лечение, "
            u"оздоровление и (или) отдых",
        25: u"Организация, осуществляющая социальное обслуживание",
        26: u"Дипломатическое представительство РФ",
        27: u"Консульское учреждение РФ",
        28: u"Представительство РФ при международных"
            u" (межгосударственных, межправительственных) организациях",
        29: u"Иные юридические лица",
        INDIVIDUAL_BUSINESS: u"Индивидуальный предприниматель, "
                             u"осуществляющий образовательную деятельность"
    }
    # коды, соответствующие типам законного представителя в Электронной Школе
    WEBEDU_CODES = (
        1, 2, 3, 12, 13,
        23, 24, 25, 26, 27, 28, 29,
        1301, 1302, 1303, 1304, 1305, 1306,
        1307, 1308, 1309, 1310, 1311, 1312
    )

    SSUZ_CODES = (14, 3)


@python_2_unicode_compatible
class OksmVirtialModel(BaseCatalogVirtualModel):

    u"""Виртуальная модель справочника ОКСМ."""

    rf_code = "643"

    id_field = 'code'
    data = load_values("oksm.json")

    fields_to_serialize = ['id', 'shortname']

    def __str__(self):
        u"""Строковое представление записи в виртуальной модели."""
        return self.shortname

    class Meta:
        verbose_name = u'ОКСМ'


class Oksm(BaseModelView):
    u"""
    Справочник ОКСМ.

    ОКСМ - Общероссийский классификатор стран мира.
    """

    model = OksmVirtialModel
    value_field = "code"
    display_field = "shortname"


class CrippleGroups(BaseEnumerate):
    u"""Справочник "Группы инвалидности"."""

    FIRST_GROUP = 11
    SECOND_GROUP = 12
    THIRD_GROUP = 13
    DISABLED_CHILD_GROUP = 14

    values = {
        FIRST_GROUP: u'1 группа',
        SECOND_GROUP: u'2 группа',
        THIRD_GROUP: u'3 группа',
        DISABLED_CHILD_GROUP: u'Ребенок-инвалид',
    }


class CrippleCategories(BaseEnumerate):
    u"""Справочник "Категории инвалидности"."""

    values = {
        21: u'Инвалид с детства',
        22: u'Инвалид вследствие военной травмы или заболевания, '
            u'полученного в период прохождения военной службы',
    }


class EducationMethods(BaseEnumerate):
    u"""Справочник "Формы образования"."""

    FULL_TIME_STUDY = 11
    CORRESPONDENCE_STUDY = 12
    FULL_TIME_AND_CORRESPONDENCE_STUDY = 13
    FAMILY_STUDY = 21
    SELF_EDUCATION_STUDY = 22

    values = {
        FULL_TIME_STUDY: u'Очная',
        CORRESPONDENCE_STUDY: u'Заочная',
        FULL_TIME_AND_CORRESPONDENCE_STUDY: u'Очно-заочная',
        FAMILY_STUDY: u'вне организации (семейное образование)',
        SELF_EDUCATION_STUDY: u'вне организации (самообразование)',
    }


class EducationTypeOVZ(BaseEnumerate):
    u"""Справочник "Вид обучения для детей с ОВЗ"."""

    COMMON = 1
    SEPARATE_CLASS = 2
    SEPARATE_UNIT = 3

    values = {
        COMMON: u"Совместно с другими обучающимися",
        SEPARATE_CLASS: u"В отдельных классах, группах",
        SEPARATE_UNIT: (
            u"В отдельных организациях, осуществляющих "
            u"образовательную деятельность")
    }


class RealizationForms(BaseEnumerateProductSpecific):

    u"""Справочник "Формы реализации образовательной программы"."""

    NETWORK = 1
    ELECTRON = 2
    REMOTE = 3
    HOME = 4
    ELECTRON_ONLY = 5
    REMOTE_ONLY = 6
    FOREIGN_NETWORK = 7

    values = {
        NETWORK: u'Сетевая форма',
        ELECTRON: u'С применением электронного обучения',
        REMOTE: u'С применением дистанционных образовательных технологий',
        HOME: u'Надомная форма получения образования',
        ELECTRON_ONLY: u'С применением исключительно электронного обучения',
        REMOTE_ONLY: (
            u'С применением исключительно дистанционных '
            u'образовательных технологий'),
        FOREIGN_NETWORK: (
            u'Сетевая форма с использованием ресурсов '
            u'иностранных организаций'),
    }

    current_kind = (1, 2, 3, 4)

    # коды, соответствующие видам форм образовательных программ
    # в Электронной Школе
    WEBEDU_CODES = (1, 2, 3, 4, 5, 6)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Детском Саду
    KINDER_CODES = (1, 2, 3, 4)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Колледже
    SSUZ_CODES = (1, 2, 3, 4, 5, 6, 7)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Доп.образовании
    EXTEDU_CODES = (1, 2, 3, 4)


class DismissalReasons(BaseEnumerate):
    u"""Справочник "Причины увольнения"."""

    TRANSFER_TO_ORG = 1
    TRANSFER_TO_IND = 2
    DISMISS = 3
    PENSION = 4
    REDUCTION = 5
    AGREEMENT_END = 6
    DEATH = 7
    RELOCATION = 8
    HEALTH = 9
    WILL = 10
    ARMY = 11
    EDUCATION = 12

    values = {
        TRANSFER_TO_ORG: u'Перевод в другую образовательную организацию',
        TRANSFER_TO_IND: u'Перевод в другую отрасль',
        DISMISS: u'Уволен по решению суда, за прогул, за нарушение дисциплины',
        PENSION: u'Уход на пенсию',
        REDUCTION: u'Сокращение штатов',
        AGREEMENT_END: u'Прекращение договора',
        DEATH: u'Смерть',
        RELOCATION: u'Смена места жительства',
        HEALTH: u'По состоянию здоровья',
        WILL: u'По собственному желанию',
        ARMY: u'Призыв в армию',
        EDUCATION: u'Дневное обучение',
    }
