"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunnerImageComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_image_builder_1 = require("./aws-image-builder");
const common_1 = require("../common");
/**
 * Components are used to build runner images. They can run commands in the image, copy files into the image, and run some Docker commands.
 */
class RunnerImageComponent {
    /**
     * Define a custom component that can run commands in the image, copy files into the image, and run some Docker commands.
     *
     * The order of operations is (1) assets (2) commands (3) docker commands.
     *
     * Use this to customize the image for the runner.
     *
     * **WARNING:** Docker commands are not guaranteed to be included before the next component
     */
    static custom(props) {
        return new class extends RunnerImageComponent {
            get name() {
                if (props.name && !props.name.match(/[a-zA-Z0-9\-]/)) {
                    throw new Error(`Invalid component name: ${props.name}. Name must only contain alphanumeric characters and dashes.`);
                }
                return `Custom-${props.name ?? 'Undefined'}`;
            }
            getCommands(_os, _architecture) {
                return props.commands ?? [];
            }
            getAssets(_os, _architecture) {
                return props.assets ?? [];
            }
            getDockerCommands(_os, _architecture) {
                return props.dockerCommands ?? [];
            }
        }();
    }
    /**
     * A component to install the required packages for the runner.
     */
    static requiredPackages() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RequiredPackages';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'amd64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
                    }
                    return [
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                        `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                        'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                        'rm /tmp/amazon-cloudwatch-agent.deb',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum update -y',
                        'yum install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://s3.amazonaws.com/amazoncloudwatch-agent/windows/amd64/latest/amazon-cloudwatch-agent.msi /qn\'',
                    ];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to prepare the required runner user.
     */
    static runnerUser() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RunnerUser';
            }
            getCommands(os, _architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'addgroup runner',
                        'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
                        'usermod -aG sudo runner',
                        'echo "%sudo   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        '/usr/sbin/groupadd runner',
                        '/usr/sbin/useradd --system --shell /usr/sbin/nologin --home-dir /home/runner --gid runner runner',
                        'mkdir -p /home/runner',
                        'chown runner /home/runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for runner user: ${os.name}`);
            }
        };
    }
    /**
     * A component to install the AWS CLI.
     */
    static awsCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'AwsCli';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
                    }
                    return [
                        `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                        'unzip -q awscliv2.zip',
                        './aws/install',
                        'rm -rf awscliv2.zip aws',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for awscli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static githubCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubCli';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                        'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                            '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'yum install -y gh',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                        '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i gh.msi /qn\'',
                        'del gh.msi',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for github cli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static git() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Git';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'add-apt-repository ppa:git-core/ppa',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y git',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                        '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                        '$GIT_REVISION = ($GIT_VERSION -Split \'.windows.\')[1]',
                        'If ($GIT_REVISION -gt 1) {$GIT_VERSION_SHORT = "$GIT_VERSION_SHORT.$GIT_REVISION"}',
                        'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                        'Start-Process git-setup.exe -Wait -ArgumentList \'/VERYSILENT\'',
                        'del git-setup.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for git: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub Actions Runner. This is the actual executable that connects to GitHub to ask for jobs and then execute them.
     *
     * @param runnerVersion The version of the runner to install. Usually you would set this to latest.
     */
    static githubRunner(runnerVersion) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubRunner';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let versionCommand;
                    if (runnerVersion.is(common_1.RunnerVersion.latest())) {
                        versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
                    }
                    else {
                        versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
                    }
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
                    }
                    let commands = [
                        versionCommand,
                        `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `tar -C /home/runner -xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                        `echo -n ${runnerVersion.version} > /home/runner/RUNNER_VERSION`,
                    ];
                    if (os.is(common_1.Os.LINUX_UBUNTU)) {
                        commands.push('/home/runner/bin/installdependencies.sh');
                    }
                    else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                        commands.push('yum install -y openssl-libs krb5-libs zlib libicu60');
                    }
                    return commands;
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    let runnerCommands;
                    if (runnerVersion.is(common_1.RunnerVersion.latest())) {
                        runnerCommands = [
                            'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                            '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                            '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        ];
                    }
                    else {
                        runnerCommands = [`$RUNNER_VERSION = '${runnerVersion.version}'`];
                    }
                    return runnerCommands.concat([
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                        'Expand-Archive actions.zip -DestinationPath C:\\actions',
                        'del actions.zip',
                        `echo ${runnerVersion.version} | Out-File -Encoding ASCII -NoNewline C:\\actions\\RUNNER_VERSION`,
                    ]);
                }
                throw new Error(`Unknown os/architecture combo for github runner: ${os.name}/${architecture.name}`);
            }
            getDockerCommands(_os, _architecture) {
                return [
                    `ENV RUNNER_VERSION=${runnerVersion.version}`,
                ];
            }
        }();
    }
    /**
     * A component to install Docker. On Windows this installs Docker Desktop.
     */
    static docker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                        'echo ' +
                            '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                            '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                        'usermod -aG docker runner',
                        'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y docker',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Invoke-WebRequest -UseBasicParsing -Uri https://desktop.docker.com/win/main/amd64/Docker%20Desktop%20Installer.exe -OutFile docker-setup.exe',
                        'Start-Process \'docker-setup.exe\' -Wait -ArgumentList \'/install --quiet --accept-license\'',
                        'del docker-setup.exe',
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/docker/compose/releases/latest > $Env:TEMP\\latest-docker-compose',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-docker-compose',
                        '$LatestDockerCompose = ($LatestUrl -Split \'/\')[-1]',
                        'Invoke-WebRequest -UseBasicParsing -Uri  "https://github.com/docker/compose/releases/download/${LatestDockerCompose}/docker-compose-Windows-x86_64.exe" -OutFile $Env:ProgramFiles\\Docker\\docker-compose.exe',
                        'copy $Env:ProgramFiles\\Docker\\docker-compose.exe $Env:ProgramFiles\\Docker\\cli-plugins\\docker-compose.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for docker: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install Docker-in-Docker.
     */
    static dockerInDocker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker-in-Docker';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for Docker-in-Docker: ${architecture.name}`);
                    }
                    return [
                        os.is(common_1.Os.LINUX_UBUNTU) ? 'DEBIAN_FRONTEND=noninteractive apt-get install -y socat' : 'yum install -y socat',
                        'DOCKER_CHANNEL="stable"',
                        'DIND_COMMIT="42b1175eda071c0e9121e1d64345928384a93df1"',
                        'DOCKER_VERSION="20.10.18"',
                        'DOCKER_COMPOSE_VERSION="2.11.0"',
                        `curl -fsSL "https://download.docker.com/linux/static/\${DOCKER_CHANNEL}/${archUrl}/docker-\${DOCKER_VERSION}.tgz" -o docker.tgz`,
                        'tar --strip-components 1 -C /usr/local/bin/ -xzf docker.tgz',
                        'rm docker.tgz',
                        '# set up subuid/subgid so that "--userns-remap=default" works out-of-the box',
                        'addgroup dockremap',
                        'useradd -g dockremap dockremap',
                        'echo \'dockremap:165536:65536\' >> /etc/subuid',
                        'echo \'dockremap:165536:65536\' >> /etc/subgid',
                        'curl -fsSL "https://raw.githubusercontent.com/docker/docker/${DIND_COMMIT}/hack/dind" -o /usr/local/bin/dind',
                        `curl -fsSL https://github.com/docker/compose/releases/download/v\${DOCKER_COMPOSE_VERSION}/docker-compose-linux-${archUrl} -o /usr/local/bin/docker-compose`,
                        'mkdir -p /home/runner/.docker/cli-plugins && ln -s /usr/local/bin/docker-compose /home/runner/.docker/cli-plugins/docker-compose',
                        'chown -R runner /home/runner/.docker',
                        'chmod +x /usr/local/bin/dind /usr/local/bin/docker-compose',
                        'addgroup docker && usermod -aG docker runner',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for Docker-in-Docker: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to add a trusted certificate authority. This can be used to support GitHub Enterprise Server with self-signed certificate.
     *
     * @param source path to certificate file in PEM format
     * @param name unique certificate name to be used on runner file system
     */
    static extraCertificates(source, name) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = `Extra-Certificates-${name}`;
            }
            getCommands(os, architecture) {
                if (!name.match(/^[a-zA-Z0-9_-]+$/)) {
                    throw new Error(`Invalid certificate name: ${name}. Name must only contain alphanumeric characters, dashes and underscores.`);
                }
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'update-ca-certificates',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'update-ca-trust',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        `Import-Certificate -FilePath C:\\${name}.crt -CertStoreLocation Cert:\\LocalMachine\\Root`,
                        `Remove-Item C:\\${name}.crt`,
                    ];
                }
                throw new Error(`Unknown os/architecture combo for extra certificates: ${os.name}/${architecture.name}`);
            }
            getAssets(os, _architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        { source, target: `/usr/local/share/ca-certificates/${name}.crt` },
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        { source, target: `/etc/pki/ca-trust/source/anchors/${name}.crt` },
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        { source, target: `C:\\${name}.crt` },
                    ];
                }
                throw new Error(`Unsupported OS for extra certificates: ${os.name}`);
            }
        }();
    }
    /**
     * A component to set up the required Lambda entrypoint for Lambda runners.
     */
    static lambdaEntrypoint() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Lambda-Entrypoint';
            }
            getCommands(os, _architecture) {
                if (!os.is(common_1.Os.LINUX_AMAZON_2) && !os.is(common_1.Os.LINUX_UBUNTU)) {
                    throw new Error(`Unsupported OS for Lambda entrypoint: ${os.name}`);
                }
                return [];
            }
            getAssets(_os, _architecture) {
                return [
                    {
                        source: path.join(__dirname, '..', 'docker-images', 'lambda', 'linux-x64', 'runner.js'),
                        target: '${LAMBDA_TASK_ROOT}/runner.js',
                    },
                    {
                        source: path.join(__dirname, '..', 'docker-images', 'lambda', 'linux-x64', 'runner.sh'),
                        target: '${LAMBDA_TASK_ROOT}/runner.sh',
                    },
                ];
            }
            getDockerCommands(_os, _architecture) {
                return [
                    'WORKDIR ${LAMBDA_TASK_ROOT}',
                    'CMD ["runner.handler"]',
                ];
            }
        };
    }
    /**
     * Returns assets to copy into the built image. Can be used to copy files into the image.
     */
    getAssets(_os, _architecture) {
        return [];
    }
    /**
     * Returns Docker commands to run to in built image. Can be used to add commands like `VOLUME`, `ENTRYPOINT`, `CMD`, etc.
     *
     * Docker commands are added after assets and normal commands.
     */
    getDockerCommands(_os, _architecture) {
        return [];
    }
    /**
     * Convert component to an AWS Image Builder component.
     *
     * @internal
     */
    _asAwsImageBuilderComponent(scope, id, os, architecture) {
        let platform;
        if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
            platform = 'Linux';
        }
        else if (os.is(common_1.Os.WINDOWS)) {
            platform = 'Windows';
        }
        else {
            throw new Error(`Unknown os/architecture combo for image builder component: ${os.name}/${architecture.name}`);
        }
        return new aws_image_builder_1.ImageBuilderComponent(scope, id, {
            platform: platform,
            commands: this.getCommands(os, architecture),
            assets: this.getAssets(os, architecture).map((asset, index) => {
                return {
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} asset ${index}`, { path: asset.source }),
                    path: asset.target,
                };
            }),
            displayName: id,
            description: id,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
RunnerImageComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerImageComponent", version: "0.9.1" };
exports.RunnerImageComponent = RunnerImageComponent;
//# sourceMappingURL=data:application/json;base64,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