"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
const delete_ami_function_1 = require("../../../../lambdas/delete-ami-function");
const utils_1 = require("../../../../utils");
const common_2 = require("../../../common");
const common_3 = require("../../common");
const ami_1 = require("../ami");
/**
 * An AMI builder that uses AWS Image Builder to build AMIs pre-baked with all the GitHub Actions runner requirements. Builders can be used with {@link Ec2Runner}.
 *
 * Each builder re-runs automatically at a set interval to make sure the AMIs contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the AMI used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the EC2 provider, use:
 *
 * ```
 * const builder = new AmiBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.addComponent(new ImageBuilderComponent(scope, id, {
 *   platform: 'Linux',
 *   displayName: 'p7zip',
 *   description: 'Install some more packages',
 *   commands: [
 *     'set -ex',
 *     'apt-get install p7zip',
 *   ],
 * }));
 * new Ec2Runner(this, 'EC2 provider', {
 *     label: 'custom-ec2',
 *     amiBuilder: builder,
 * });
 * ```
 *
 * @deprecated use RunnerImageBuilder
 */
class AmiBuilder extends common_1.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_2.Os.LINUX, common_2.Os.LINUX_UBUNTU, common_2.Os.LINUX_AMAZON_2, common_2.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_2.Architecture.X86_64, common_2.Architecture.ARM64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'AMI',
        });
        // add all basic components
        if (this.os.is(common_2.Os.WINDOWS)) {
            this.addBaseWindowsComponents(props?.installDocker ?? true);
        }
        else if (this.os.is(common_2.Os.LINUX) || this.os.is(common_2.Os.LINUX_UBUNTU)) {
            this.addBaseLinuxComponents(props?.installDocker ?? true);
        }
        else {
            throw new Error(`Unsupported OS for AMI builder: ${this.os.name}`);
        }
    }
    addBaseWindowsComponents(installDocker) {
        this.addComponent(windows_components_1.WindowsComponents.cloudwatchAgent(this, 'CloudWatch agent'));
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
        if (installDocker) {
            this.addComponent(windows_components_1.WindowsComponents.docker(this, 'Docker'));
        }
    }
    addBaseLinuxComponents(installDocker) {
        this.addComponent(linux_components_1.LinuxUbuntuComponents.requiredPackages(this, 'Upgrade packages and install basics', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.runnerUser(this, 'User', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.awsCli(this, 'AWS CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubCli(this, 'GitHub CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.git(this, 'git', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion, this.architecture));
        if (installDocker) {
            this.addComponent(linux_components_1.LinuxUbuntuComponents.docker(this, 'Docker', this.architecture));
        }
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the AMI builder.
     */
    bindAmi() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: (0, common_3.uniqueImageBuilderName)(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new ami_1.AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform,
            components: this.components,
            architecture: this.architecture,
            baseAmi: (0, ami_1.defaultBaseAmi)(this.os, this.architecture).getImage(this).imageId,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        this.imageCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    imageCleaner(launchTemplate, stackName, builderName) {
        const deleter = (0, utils_1.singletonLambda)(delete_ami_function_1.DeleteAmiFunction, this, 'delete-ami', {
            description: 'Delete old GitHub Runner AMIs',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
    bindDockerImage() {
        throw new Error('AmiBuilder cannot be used to build Docker images');
    }
}
_a = JSII_RTTI_SYMBOL_1;
AmiBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.AmiBuilder", version: "0.9.1" };
exports.AmiBuilder = AmiBuilder;
//# sourceMappingURL=data:application/json;base64,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