"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const AWS = require("aws-sdk");
const helpers_1 = require("./helpers");
const sf = new AWS.StepFunctions();
// TODO use @octokit/webhooks?
function verifyBody(event, secret) {
    const sig = Buffer.from(event.headers['x-hub-signature-256'] || '', 'utf8');
    if (!event.body) {
        throw new Error('No body');
    }
    let body;
    if (event.isBase64Encoded) {
        body = Buffer.from(event.body, 'base64');
    }
    else {
        body = Buffer.from(event.body || '', 'utf8');
    }
    const hmac = crypto.createHmac('sha256', secret);
    hmac.update(body);
    const expectedSig = Buffer.from(`sha256=${hmac.digest('hex')}`, 'utf8');
    console.log('Calculated signature: ', expectedSig.toString());
    if (sig.length !== expectedSig.length || !crypto.timingSafeEqual(sig, expectedSig)) {
        throw new Error(`Signature mismatch. Expected ${expectedSig.toString()} but got ${sig.toString()}`);
    }
    return body.toString();
}
exports.handler = async function (event) {
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.STEP_FUNCTION_ARN) {
        throw new Error('Missing environment variables');
    }
    const webhookSecret = (await (0, helpers_1.getSecretJsonValue)(process.env.WEBHOOK_SECRET_ARN)).webhookSecret;
    let body;
    try {
        body = verifyBody(event, webhookSecret);
    }
    catch (e) {
        console.error(e);
        return {
            statusCode: 403,
            body: 'Bad signature',
        };
    }
    if (event.headers['content-type'] !== 'application/json') {
        console.error(`This webhook only accepts JSON payloads, got ${event.headers['content-type']}`);
        return {
            statusCode: 400,
            body: 'Expecting JSON payload',
        };
    }
    if (event.headers['x-github-event'] === 'ping') {
        return {
            statusCode: 200,
            body: 'Pong',
        };
    }
    // if (event.headers['x-github-event'] !== 'workflow_job' && event.headers['x-github-event'] !== 'workflow_run') {
    //     console.error(`This webhook only accepts workflow_job and workflow_run, got ${event.headers['x-github-event']}`);
    if (event.headers['x-github-event'] !== 'workflow_job') {
        console.error(`This webhook only accepts workflow_job, got ${event.headers['x-github-event']}`);
        return {
            statusCode: 400,
            body: 'Expecting workflow_job',
        };
    }
    const payload = JSON.parse(body);
    if (payload.action !== 'queued') {
        console.log(`Ignoring action "${payload.action}", expecting "queued"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    if (!payload.workflow_job.labels.includes('self-hosted')) {
        console.log(`Ignoring labels "${payload.workflow_job.labels}", expecting "self-hosted"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    // it's easier to deal with maps in step functions
    let labels = {};
    payload.workflow_job.labels.forEach((l) => labels[l.toLowerCase()] = true);
    // set execution name which is also used as runner name which are limited to 64 characters
    let executionName = `${payload.repository.full_name.replace('/', '-')}-${event.headers['x-github-delivery']}`.slice(0, 64);
    // start execution
    const execution = await sf.startExecution({
        stateMachineArn: process.env.STEP_FUNCTION_ARN,
        input: JSON.stringify({
            owner: payload.repository.owner.login,
            repo: payload.repository.name,
            runId: payload.workflow_job.run_id,
            installationId: payload.installation?.id,
            labels: labels,
        }),
        // name is not random so multiple execution of this webhook won't cause multiple builders to start
        name: executionName,
    }).promise();
    console.log(`Started ${execution.executionArn}`);
    return {
        statusCode: 202,
        body: executionName,
    };
};
//# sourceMappingURL=data:application/json;base64,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