from traits.api import \
    Instance, Int, Property, Array, cached_property

from numpy import \
     zeros, dot, hstack, identity

from scipy.linalg import \
     inv

from .fets3D import FETS3D

#-----------------------------------------------------------------------------------
# FETS3D8H32U - 32 nodes subparametric volume element (3D, cubic, serendipity family)
#-----------------------------------------------------------------------------------

class FETS3D8H32U(FETS3D):
    '''
    cubic serendipity volume element
    '''
    debug_on = True

    # Dimensional mapping
    dim_slice = slice(0, 3)

    # number of nodal degrees of freedom
    # number of degrees of freedom of each element
    n_nodal_dofs = Int(3)
    n_e_dofs = Int(32 * 3)

    # Integration parameters
    #
    ngp_r = 3
    ngp_s = 3
    ngp_t = 3

    dof_r = \
                Array(value=[
                 # lower side (t=-1)
                 #
                 [-1., -1., -1.],
                 [-1. / 3., -1., -1.],
                 [  1. / 3., -1., -1.],
                 [  1., -1., -1.],
                 #
                 [-1., -1. / 3., -1.],
                 [  1., -1. / 3., -1.],
                 #
                 [-1., 1. / 3., -1.],
                 [  1., 1. / 3., -1.],
                 #
                 [-1., 1., -1.],
                 [-1. / 3., 1., -1.],
                 [  1. / 3., 1., -1.],
                 [  1., 1., -1.],

                 # sides t=-0.333/+0.333
                 #
                 [-1., -1., -1. / 3.],
                 [  1., -1., -1. / 3.],
                 #
                 [-1., 1., -1. / 3.],
                 [  1., 1., -1. / 3.],
                 #
                 [-1., -1., 1. / 3.],
                 [  1., -1., 1. / 3.],
                 #
                 [-1., 1., 1. / 3.],
                 [  1., 1., 1. / 3.],

                 # upper side (t=1)
                 #
                 [-1., -1., 1.],
                 [-1. / 3., -1., 1.],
                 [  1. / 3., -1., 1.],
                 [  1., -1., 1.],
                 #
                 [-1., -1. / 3., 1.],
                 [  1., -1. / 3., 1.],
                 #
                 [-1., 1. / 3., 1.],
                 [  1., 1. / 3., 1.],
                 #
                 [-1., 1., 1.],
                 [-1. / 3., 1., 1.],
                 [  1. / 3., 1., 1.],
                 [  1., 1., 1.]

                 ])

    geo_r = \
        Array(value=[[-1., -1., -1.],
                        [  1., -1., -1.],
                        [-1., 1., -1.],
                        [  1., 1., -1.],
                        [-1., -1., 1.],
                        [  1., -1., 1.],
                        [-1., 1., 1.],
                        [  1., 1., 1.]])

    # Used for Visualization
    vtk_cell_types = 'TriQuadraticHexahedron'
    vtk_r = Array(value=[[-1., -1., -1.],  # bottom
                           [  1., -1., -1.],
                           [  1., 1., -1.],
                           [-1., 1., -1.],
                           [-1., -1., 1.],  # top
                           [  1., -1., 1.],
                           [  1., 1., 1.],
                           [-1., 1., 1.],
                           [  0., -1., -1.],  # bottom midside
                           [  1., 0., -1.],
                           [  0., 1., -1.],
                           [-1., 0., -1.],
                           [  0., -1., 1.],  # top midside
                           [  1., 0., 1.],
                           [  0., 1., 1.],
                           [-1., 0., 1.],
                           [-1., -1., 0.],  # middle
                           [  1., -1., 0.],
                           [  1., 1., 0.],
                           [-1., 1., 0.],
                           [  0., -1., 0.],  # middle midside (different order)
                           [  1., 0., 0.],
                           [  0., 1., 0.],
                           [-1., 0., 0.],
                           [  0., 0., -1.],  # bottom centre
                           [  0., 0., 1.],  # top centre
                           [  0., 0., 0.]])  # middle centre

    vtk_cells = [[ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, \
                    10, 11, 12, 13, 14, 15, 16, 17, 18, 19, \
                    23, 21, 20, 22, 24, 25, 26]]

    #---------------------------------------------------------------------
    # Method required to represent the element geometry
    #---------------------------------------------------------------------
    def get_N_geo_mtx(self, r_pnt):
        '''
        Return the value of shape functions (derived in femple) for the
        specified local coordinate r_pnt
        '''
        N_geo_mtx = zeros((1, 8), dtype='float64')
        N_geo_mtx[0, 0] = -((-1 + r_pnt[2]) * (-1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 1] = ((-1 + r_pnt[2]) * (-1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 2] = ((-1 + r_pnt[2]) * (1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 3] = -((-1 + r_pnt[2]) * (1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 4] = ((1 + r_pnt[2]) * (-1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 5] = -((1 + r_pnt[2]) * (-1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 6] = -((1 + r_pnt[2]) * (1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        N_geo_mtx[0, 7] = ((1 + r_pnt[2]) * (1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        return N_geo_mtx

    def get_dNr_geo_mtx(self, r_pnt):
        '''
        Return the matrix of shape function derivatives (derived in femple).
        Used for the construction of the Jacobi matrix.
        '''
        dNr_geo_mtx = zeros((3, 8), dtype='float64')
        dNr_geo_mtx[0, 0] = -((-1 + r_pnt[2]) * (-1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 1] = ((-1 + r_pnt[2]) * (-1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 2] = ((-1 + r_pnt[2]) * (1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 3] = -((-1 + r_pnt[2]) * (1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 4] = ((1 + r_pnt[2]) * (-1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 5] = -((1 + r_pnt[2]) * (-1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 6] = -((1 + r_pnt[2]) * (1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[0, 7] = ((1 + r_pnt[2]) * (1 + r_pnt[1])) / 8.0
        dNr_geo_mtx[1, 0] = -((-1 + r_pnt[2]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 1] = ((-1 + r_pnt[2]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 2] = ((-1 + r_pnt[2]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 3] = -((-1 + r_pnt[2]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 4] = ((1 + r_pnt[2]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 5] = -((1 + r_pnt[2]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 6] = -((1 + r_pnt[2]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[1, 7] = ((1 + r_pnt[2]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 0] = -((-1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 1] = ((-1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 2] = ((1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 3] = -((1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 4] = ((-1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 5] = -((-1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 6] = -((1 + r_pnt[1]) * (-1 + r_pnt[0])) / 8.0
        dNr_geo_mtx[2, 7] = ((1 + r_pnt[1]) * (1 + r_pnt[0])) / 8.0
        return dNr_geo_mtx

    #---------------------------------------------------------------------
    # Method delivering the shape functions for the field variables and their
    # derivatives
    #---------------------------------------------------------------------
    def get_N_mtx(self, r_pnt):
        '''
        Returns the matrix of the shape functions used for the field approximation
        containing zero entries. The number of rows corresponds to the number of nodal
        dofs. The matrix is evaluated for the specified local coordinate r_pnt.
        '''
        r = r_pnt[0]
        s = r_pnt[1]
        t = r_pnt[2]
        N_mtx = zeros((3, 96), dtype='float64')
        N_mtx[0, 0] = 0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[0, 3] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 6] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 9] = 0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[0, 12] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 15] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 18] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 21] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 24] = 0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[0, 27] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 30] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 33] = 0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[0, 36] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 39] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 42] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 45] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 48] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 51] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 54] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 57] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[0, 60] = -0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[0, 63] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 66] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 69] = -0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[0, 72] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 75] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 78] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 81] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[0, 84] = -0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[0, 87] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 90] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[0, 93] = -0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[1, 1] = 0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[1, 4] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 7] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 10] = 0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[1, 13] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 16] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 19] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 22] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 25] = 0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[1, 28] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 31] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 34] = 0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[1, 37] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 40] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 43] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 46] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 49] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 52] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 55] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 58] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[1, 61] = -0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[1, 64] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 67] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 70] = -0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[1, 73] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 76] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 79] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 82] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[1, 85] = -0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[1, 88] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 91] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[1, 94] = -0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[2, 2] = 0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[2, 5] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 8] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 11] = 0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s - 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[2, 14] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 17] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 20] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) - 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 23] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) - 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 26] = 0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 - 0.140625 * r * s * s - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[2, 29] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 32] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 35] = 0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s - 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.296875 + 0.140625 * r * s * s - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t - 0.140625 * pow(t, 3)
        N_mtx[2, 38] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 41] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 44] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s - 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 47] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s - 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 50] = 0.140625 - 0.140625 * r - 0.140625 * s + 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 53] = 0.140625 + 0.140625 * r - 0.140625 * s - 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) + 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 56] = 0.140625 - 0.140625 * r + 0.140625 * s - 0.140625 * r * s + 0.421875 * t - 0.421875 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) + 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 59] = 0.140625 + 0.140625 * r + 0.140625 * s + 0.140625 * r * s + 0.421875 * t + 0.421875 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3) - 0.421875 * s * pow(t, 3) - 0.421875 * r * s * pow(t, 3)
        N_mtx[2, 62] = -0.296875 * t + 0.296875 * s + 0.296875 * r - 0.296875 * r * s + 0.296875 * r * t + 0.296875 * s * t - 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) - 0.140625 * r * t * t - 0.140625 * s * t * t + 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[2, 65] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t - 0.140625 * s * t + 0.421875 * r * s * t + 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 68] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t - 0.140625 * s * t - 0.421875 * r * s * t + 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 71] = -0.296875 * t + 0.296875 * s - 0.296875 * r + 0.296875 * r * s - 0.296875 * r * t + 0.296875 * s * t + 0.296875 * r * s * t - 0.140625 * r * r * s + 0.140625 * r * r * t - 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t - 0.140625 * pow(s, 3) + 0.140625 * r * t * t - 0.140625 * s * t * t - 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) - 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[2, 74] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t - 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 77] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t - 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 80] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3) + 0.140625 * t - 0.140625 * r * t + 0.421875 * s * t - 0.421875 * r * s * t - 0.140625 * s * s * t + 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t + 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 83] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3) + 0.140625 * t + 0.140625 * r * t + 0.421875 * s * t + 0.421875 * r * s * t - 0.140625 * s * s * t - 0.140625 * r * s * s * t - 0.421875 * pow(s, 3) * t - 0.421875 * r * pow(s, 3) * t
        N_mtx[2, 86] = -0.296875 * t - 0.296875 * s + 0.296875 * r + 0.296875 * r * s + 0.296875 * r * t - 0.296875 * s * t + 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r - 0.140625 * r * s * s + 0.140625 * s * s * t - 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 - 0.140625 * pow(r, 3) * s - 0.140625 * pow(r, 3) * t - 0.140625 * pow(r, 3) * s * t - 0.140625 * pow(r, 3) - 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) - 0.140625 * r * t * t + 0.140625 * s * t * t - 0.140625 * r * s * t * t - 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) - 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        N_mtx[2, 89] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t + 0.140625 * s * t - 0.421875 * r * s * t - 0.140625 * r * r * s * t + 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 92] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t + 0.140625 * s * t + 0.421875 * r * s * t - 0.140625 * r * r * s * t - 0.421875 * pow(r, 3) * s * t
        N_mtx[2, 95] = -0.296875 * t - 0.296875 * s - 0.296875 * r - 0.296875 * r * s - 0.296875 * r * t - 0.296875 * s * t - 0.296875 * r * s * t + 0.140625 * r * r * s + 0.140625 * r * r * t + 0.140625 * r * r * s * t + 0.140625 * r * r + 0.140625 * r * s * s + 0.140625 * s * s * t + 0.140625 * r * s * s * t + 0.140625 * s * s - 0.296875 + 0.140625 * pow(r, 3) * s + 0.140625 * pow(r, 3) * t + 0.140625 * pow(r, 3) * s * t + 0.140625 * pow(r, 3) + 0.140625 * r * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * r * pow(s, 3) * t + 0.140625 * pow(s, 3) + 0.140625 * r * t * t + 0.140625 * s * t * t + 0.140625 * r * s * t * t + 0.140625 * r * pow(t, 3) + 0.140625 * s * pow(t, 3) + 0.140625 * r * s * pow(t, 3) + 0.140625 * t * t + 0.140625 * pow(t, 3)
        return N_mtx

    def get_dNr_mtx(self, r_pnt):
        '''
        Return the derivatives of the shape functions used for the field approximation
        '''
        r = r_pnt[0]
        s = r_pnt[1]
        t = r_pnt[2]
        dNr_mtx = zeros((3, 32), dtype='float64')
        dNr_mtx[0, 0] = 0.296875 - 0.296875 * s - 0.296875 * t + 0.296875 * s * t - 0.281250 * r * s - 0.281250 * r * t + 0.281250 * r * s * t + 0.281250 * r - 0.140625 * s * s + 0.140625 * s * s * t + 0.421875 * r * r * s + 0.421875 * r * r * t - 0.421875 * r * r * s * t - 0.421875 * r * r + 0.140625 * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * t * t + 0.140625 * s * t * t + 0.140625 * pow(t, 3) - 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 1] = -0.421875 - 0.281250 * r + 1.265625 * r * r + 0.421875 * s + 0.281250 * r * s - 1.265625 * r * r * s + 0.421875 * t + 0.281250 * r * t - 1.265625 * r * r * t - 0.421875 * s * t - 0.281250 * r * s * t + 1.265625 * r * r * s * t
        dNr_mtx[0, 2] = 0.421875 - 0.281250 * r - 1.265625 * r * r - 0.421875 * s + 0.281250 * r * s + 1.265625 * r * r * s - 0.421875 * t + 0.281250 * r * t + 1.265625 * r * r * t + 0.421875 * s * t - 0.281250 * r * s * t - 1.265625 * r * r * s * t
        dNr_mtx[0, 3] = -0.296875 + 0.296875 * s + 0.296875 * t - 0.296875 * s * t - 0.281250 * r * s - 0.281250 * r * t + 0.281250 * r * s * t + 0.281250 * r + 0.140625 * s * s - 0.140625 * s * s * t - 0.421875 * r * r * s - 0.421875 * r * r * t + 0.421875 * r * r * s * t + 0.421875 * r * r - 0.140625 * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * t * t - 0.140625 * s * t * t - 0.140625 * pow(t, 3) + 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 4] = -0.140625 + 0.421875 * s + 0.140625 * s * s - 0.421875 * pow(s, 3) + 0.140625 * t - 0.421875 * s * t - 0.140625 * s * s * t + 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 5] = 0.140625 - 0.421875 * s - 0.140625 * s * s + 0.421875 * pow(s, 3) - 0.140625 * t + 0.421875 * s * t + 0.140625 * s * s * t - 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 6] = -0.140625 - 0.421875 * s + 0.140625 * s * s + 0.421875 * pow(s, 3) + 0.140625 * t + 0.421875 * s * t - 0.140625 * s * s * t - 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 7] = 0.140625 + 0.421875 * s - 0.140625 * s * s - 0.421875 * pow(s, 3) - 0.140625 * t - 0.421875 * s * t + 0.140625 * s * s * t + 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 8] = 0.296875 + 0.296875 * s - 0.296875 * t - 0.296875 * s * t + 0.281250 * r * s - 0.281250 * r * t - 0.281250 * r * s * t + 0.281250 * r - 0.140625 * s * s + 0.140625 * s * s * t - 0.421875 * r * r * s + 0.421875 * r * r * t + 0.421875 * r * r * s * t - 0.421875 * r * r - 0.140625 * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * t * t - 0.140625 * s * t * t + 0.140625 * pow(t, 3) + 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 9] = -0.421875 - 0.281250 * r + 1.265625 * r * r - 0.421875 * s - 0.281250 * r * s + 1.265625 * r * r * s + 0.421875 * t + 0.281250 * r * t - 1.265625 * r * r * t + 0.421875 * s * t + 0.281250 * r * s * t - 1.265625 * r * r * s * t
        dNr_mtx[0, 10] = 0.421875 - 0.281250 * r - 1.265625 * r * r + 0.421875 * s - 0.281250 * r * s - 1.265625 * r * r * s - 0.421875 * t + 0.281250 * r * t + 1.265625 * r * r * t - 0.421875 * s * t + 0.281250 * r * s * t + 1.265625 * r * r * s * t
        dNr_mtx[0, 11] = -0.296875 - 0.296875 * s + 0.296875 * t + 0.296875 * s * t + 0.281250 * r * s - 0.281250 * r * t - 0.281250 * r * s * t + 0.281250 * r + 0.140625 * s * s - 0.140625 * s * s * t + 0.421875 * r * r * s - 0.421875 * r * r * t - 0.421875 * r * r * s * t + 0.421875 * r * r + 0.140625 * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * t * t + 0.140625 * s * t * t - 0.140625 * pow(t, 3) - 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 12] = -0.140625 + 0.140625 * s + 0.421875 * t - 0.421875 * s * t + 0.140625 * t * t - 0.140625 * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 13] = 0.140625 - 0.140625 * s - 0.421875 * t + 0.421875 * s * t - 0.140625 * t * t + 0.140625 * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 14] = -0.140625 - 0.140625 * s + 0.421875 * t + 0.421875 * s * t + 0.140625 * t * t + 0.140625 * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 15] = 0.140625 + 0.140625 * s - 0.421875 * t - 0.421875 * s * t - 0.140625 * t * t - 0.140625 * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 16] = -0.140625 + 0.140625 * s - 0.421875 * t + 0.421875 * s * t + 0.140625 * t * t - 0.140625 * s * t * t + 0.421875 * pow(t, 3) - 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 17] = 0.140625 - 0.140625 * s + 0.421875 * t - 0.421875 * s * t - 0.140625 * t * t + 0.140625 * s * t * t - 0.421875 * pow(t, 3) + 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 18] = -0.140625 - 0.140625 * s - 0.421875 * t - 0.421875 * s * t + 0.140625 * t * t + 0.140625 * s * t * t + 0.421875 * pow(t, 3) + 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 19] = 0.140625 + 0.140625 * s + 0.421875 * t + 0.421875 * s * t - 0.140625 * t * t - 0.140625 * s * t * t - 0.421875 * pow(t, 3) - 0.421875 * s * pow(t, 3)
        dNr_mtx[0, 20] = 0.296875 - 0.296875 * s + 0.296875 * t - 0.296875 * s * t - 0.281250 * r * s + 0.281250 * r * t - 0.281250 * r * s * t + 0.281250 * r - 0.140625 * s * s - 0.140625 * s * s * t + 0.421875 * r * r * s - 0.421875 * r * r * t + 0.421875 * r * r * s * t - 0.421875 * r * r + 0.140625 * pow(s, 3) + 0.140625 * pow(s, 3) * t - 0.140625 * t * t + 0.140625 * s * t * t - 0.140625 * pow(t, 3) + 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 21] = -0.421875 - 0.281250 * r + 1.265625 * r * r + 0.421875 * s + 0.281250 * r * s - 1.265625 * r * r * s - 0.421875 * t - 0.281250 * r * t + 1.265625 * r * r * t + 0.421875 * s * t + 0.281250 * r * s * t - 1.265625 * r * r * s * t
        dNr_mtx[0, 22] = 0.421875 - 0.281250 * r - 1.265625 * r * r - 0.421875 * s + 0.281250 * r * s + 1.265625 * r * r * s + 0.421875 * t - 0.281250 * r * t - 1.265625 * r * r * t - 0.421875 * s * t + 0.281250 * r * s * t + 1.265625 * r * r * s * t
        dNr_mtx[0, 23] = -0.296875 + 0.296875 * s - 0.296875 * t + 0.296875 * s * t - 0.281250 * r * s + 0.281250 * r * t - 0.281250 * r * s * t + 0.281250 * r + 0.140625 * s * s + 0.140625 * s * s * t - 0.421875 * r * r * s + 0.421875 * r * r * t - 0.421875 * r * r * s * t + 0.421875 * r * r - 0.140625 * pow(s, 3) - 0.140625 * pow(s, 3) * t + 0.140625 * t * t - 0.140625 * s * t * t + 0.140625 * pow(t, 3) - 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 24] = -0.140625 + 0.421875 * s + 0.140625 * s * s - 0.421875 * pow(s, 3) - 0.140625 * t + 0.421875 * s * t + 0.140625 * s * s * t - 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 25] = 0.140625 - 0.421875 * s - 0.140625 * s * s + 0.421875 * pow(s, 3) + 0.140625 * t - 0.421875 * s * t - 0.140625 * s * s * t + 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 26] = -0.140625 - 0.421875 * s + 0.140625 * s * s + 0.421875 * pow(s, 3) - 0.140625 * t - 0.421875 * s * t + 0.140625 * s * s * t + 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 27] = 0.140625 + 0.421875 * s - 0.140625 * s * s - 0.421875 * pow(s, 3) + 0.140625 * t + 0.421875 * s * t - 0.140625 * s * s * t - 0.421875 * pow(s, 3) * t
        dNr_mtx[0, 28] = 0.296875 + 0.296875 * s + 0.296875 * t + 0.296875 * s * t + 0.281250 * r * s + 0.281250 * r * t + 0.281250 * r * s * t + 0.281250 * r - 0.140625 * s * s - 0.140625 * s * s * t - 0.421875 * r * r * s - 0.421875 * r * r * t - 0.421875 * r * r * s * t - 0.421875 * r * r - 0.140625 * pow(s, 3) - 0.140625 * pow(s, 3) * t - 0.140625 * t * t - 0.140625 * s * t * t - 0.140625 * pow(t, 3) - 0.140625 * s * pow(t, 3)
        dNr_mtx[0, 29] = -0.421875 - 0.281250 * r + 1.265625 * r * r - 0.421875 * s - 0.281250 * r * s + 1.265625 * r * r * s - 0.421875 * t - 0.281250 * r * t + 1.265625 * r * r * t - 0.421875 * s * t - 0.281250 * r * s * t + 1.265625 * r * r * s * t
        dNr_mtx[0, 30] = 0.421875 - 0.281250 * r - 1.265625 * r * r + 0.421875 * s - 0.281250 * r * s - 1.265625 * r * r * s + 0.421875 * t - 0.281250 * r * t - 1.265625 * r * r * t + 0.421875 * s * t - 0.281250 * r * s * t - 1.265625 * r * r * s * t
        dNr_mtx[0, 31] = -0.296875 - 0.296875 * s - 0.296875 * t - 0.296875 * s * t + 0.281250 * r * s + 0.281250 * r * t + 0.281250 * r * s * t + 0.281250 * r + 0.140625 * s * s + 0.140625 * s * s * t + 0.421875 * r * r * s + 0.421875 * r * r * t + 0.421875 * r * r * s * t + 0.421875 * r * r + 0.140625 * pow(s, 3) + 0.140625 * pow(s, 3) * t + 0.140625 * t * t + 0.140625 * s * t * t + 0.140625 * pow(t, 3) + 0.140625 * s * pow(t, 3)
        dNr_mtx[1, 0] = 0.296875 - 0.296875 * r - 0.296875 * t + 0.296875 * r * t - 0.140625 * r * r + 0.140625 * r * r * t - 0.281250 * r * s - 0.281250 * s * t + 0.281250 * r * s * t + 0.281250 * s + 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * t + 0.421875 * r * s * s + 0.421875 * s * s * t - 0.421875 * r * s * s * t - 0.421875 * s * s - 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * pow(t, 3) - 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 1] = -0.140625 + 0.421875 * r + 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 2] = -0.140625 - 0.421875 * r + 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 3] = 0.296875 + 0.296875 * r - 0.296875 * t - 0.296875 * r * t - 0.140625 * r * r + 0.140625 * r * r * t + 0.281250 * r * s - 0.281250 * s * t - 0.281250 * r * s * t + 0.281250 * s - 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * t - 0.421875 * r * s * s + 0.421875 * s * s * t + 0.421875 * r * s * s * t - 0.421875 * s * s - 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * pow(t, 3) + 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 4] = -0.421875 + 0.421875 * r - 0.281250 * s + 0.281250 * r * s + 1.265625 * s * s - 1.265625 * r * s * s + 0.421875 * t - 0.421875 * r * t + 0.281250 * s * t - 0.281250 * r * s * t - 1.265625 * s * s * t + 1.265625 * r * s * s * t
        dNr_mtx[1, 5] = -0.421875 - 0.421875 * r - 0.281250 * s - 0.281250 * r * s + 1.265625 * s * s + 1.265625 * r * s * s + 0.421875 * t + 0.421875 * r * t + 0.281250 * s * t + 0.281250 * r * s * t - 1.265625 * s * s * t - 1.265625 * r * s * s * t
        dNr_mtx[1, 6] = 0.421875 - 0.421875 * r - 0.281250 * s + 0.281250 * r * s - 1.265625 * s * s + 1.265625 * r * s * s - 0.421875 * t + 0.421875 * r * t + 0.281250 * s * t - 0.281250 * r * s * t + 1.265625 * s * s * t - 1.265625 * r * s * s * t
        dNr_mtx[1, 7] = 0.421875 + 0.421875 * r - 0.281250 * s - 0.281250 * r * s - 1.265625 * s * s - 1.265625 * r * s * s - 0.421875 * t - 0.421875 * r * t + 0.281250 * s * t + 0.281250 * r * s * t + 1.265625 * s * s * t + 1.265625 * r * s * s * t
        dNr_mtx[1, 8] = -0.296875 + 0.296875 * r + 0.296875 * t - 0.296875 * r * t + 0.140625 * r * r - 0.140625 * r * r * t - 0.281250 * r * s - 0.281250 * s * t + 0.281250 * r * s * t + 0.281250 * s - 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * t - 0.421875 * r * s * s - 0.421875 * s * s * t + 0.421875 * r * s * s * t + 0.421875 * s * s + 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * pow(t, 3) + 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 9] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 10] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 11] = -0.296875 - 0.296875 * r + 0.296875 * t + 0.296875 * r * t + 0.140625 * r * r - 0.140625 * r * r * t + 0.281250 * r * s - 0.281250 * s * t - 0.281250 * r * s * t + 0.281250 * s + 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * t + 0.421875 * r * s * s - 0.421875 * s * s * t - 0.421875 * r * s * s * t + 0.421875 * s * s + 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * pow(t, 3) - 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 12] = -0.140625 + 0.140625 * r + 0.421875 * t - 0.421875 * r * t + 0.140625 * t * t - 0.140625 * r * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 13] = -0.140625 - 0.140625 * r + 0.421875 * t + 0.421875 * r * t + 0.140625 * t * t + 0.140625 * r * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 14] = 0.140625 - 0.140625 * r - 0.421875 * t + 0.421875 * r * t - 0.140625 * t * t + 0.140625 * r * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 15] = 0.140625 + 0.140625 * r - 0.421875 * t - 0.421875 * r * t - 0.140625 * t * t - 0.140625 * r * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 16] = -0.140625 + 0.140625 * r - 0.421875 * t + 0.421875 * r * t + 0.140625 * t * t - 0.140625 * r * t * t + 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 17] = -0.140625 - 0.140625 * r - 0.421875 * t - 0.421875 * r * t + 0.140625 * t * t + 0.140625 * r * t * t + 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 18] = 0.140625 - 0.140625 * r + 0.421875 * t - 0.421875 * r * t - 0.140625 * t * t + 0.140625 * r * t * t - 0.421875 * pow(t, 3) + 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 19] = 0.140625 + 0.140625 * r + 0.421875 * t + 0.421875 * r * t - 0.140625 * t * t - 0.140625 * r * t * t - 0.421875 * pow(t, 3) - 0.421875 * r * pow(t, 3)
        dNr_mtx[1, 20] = 0.296875 - 0.296875 * r + 0.296875 * t - 0.296875 * r * t - 0.140625 * r * r - 0.140625 * r * r * t - 0.281250 * r * s + 0.281250 * s * t - 0.281250 * r * s * t + 0.281250 * s + 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * t + 0.421875 * r * s * s - 0.421875 * s * s * t + 0.421875 * r * s * s * t - 0.421875 * s * s - 0.140625 * t * t + 0.140625 * r * t * t - 0.140625 * pow(t, 3) + 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 21] = -0.140625 + 0.421875 * r + 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * t + 0.421875 * r * t + 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 22] = -0.140625 - 0.421875 * r + 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * t - 0.421875 * r * t + 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 23] = 0.296875 + 0.296875 * r + 0.296875 * t + 0.296875 * r * t - 0.140625 * r * r - 0.140625 * r * r * t + 0.281250 * r * s + 0.281250 * s * t + 0.281250 * r * s * t + 0.281250 * s - 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * t - 0.421875 * r * s * s - 0.421875 * s * s * t - 0.421875 * r * s * s * t - 0.421875 * s * s - 0.140625 * t * t - 0.140625 * r * t * t - 0.140625 * pow(t, 3) - 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 24] = -0.421875 + 0.421875 * r - 0.281250 * s + 0.281250 * r * s + 1.265625 * s * s - 1.265625 * r * s * s - 0.421875 * t + 0.421875 * r * t - 0.281250 * s * t + 0.281250 * r * s * t + 1.265625 * s * s * t - 1.265625 * r * s * s * t
        dNr_mtx[1, 25] = -0.421875 - 0.421875 * r - 0.281250 * s - 0.281250 * r * s + 1.265625 * s * s + 1.265625 * r * s * s - 0.421875 * t - 0.421875 * r * t - 0.281250 * s * t - 0.281250 * r * s * t + 1.265625 * s * s * t + 1.265625 * r * s * s * t
        dNr_mtx[1, 26] = 0.421875 - 0.421875 * r - 0.281250 * s + 0.281250 * r * s - 1.265625 * s * s + 1.265625 * r * s * s + 0.421875 * t - 0.421875 * r * t - 0.281250 * s * t + 0.281250 * r * s * t - 1.265625 * s * s * t + 1.265625 * r * s * s * t
        dNr_mtx[1, 27] = 0.421875 + 0.421875 * r - 0.281250 * s - 0.281250 * r * s - 1.265625 * s * s - 1.265625 * r * s * s + 0.421875 * t + 0.421875 * r * t - 0.281250 * s * t - 0.281250 * r * s * t - 1.265625 * s * s * t - 1.265625 * r * s * s * t
        dNr_mtx[1, 28] = -0.296875 + 0.296875 * r - 0.296875 * t + 0.296875 * r * t + 0.140625 * r * r + 0.140625 * r * r * t - 0.281250 * r * s + 0.281250 * s * t - 0.281250 * r * s * t + 0.281250 * s - 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * t - 0.421875 * r * s * s + 0.421875 * s * s * t - 0.421875 * r * s * s * t + 0.421875 * s * s + 0.140625 * t * t - 0.140625 * r * t * t + 0.140625 * pow(t, 3) - 0.140625 * r * pow(t, 3)
        dNr_mtx[1, 29] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * t - 0.421875 * r * t - 0.140625 * r * r * t + 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 30] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * t + 0.421875 * r * t - 0.140625 * r * r * t - 0.421875 * pow(r, 3) * t
        dNr_mtx[1, 31] = -0.296875 - 0.296875 * r - 0.296875 * t - 0.296875 * r * t + 0.140625 * r * r + 0.140625 * r * r * t + 0.281250 * r * s + 0.281250 * s * t + 0.281250 * r * s * t + 0.281250 * s + 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * t + 0.421875 * r * s * s + 0.421875 * s * s * t + 0.421875 * r * s * s * t + 0.421875 * s * s + 0.140625 * t * t + 0.140625 * r * t * t + 0.140625 * pow(t, 3) + 0.140625 * r * pow(t, 3)
        dNr_mtx[2, 0] = 0.296875 - 0.296875 * r - 0.296875 * s + 0.296875 * r * s - 0.140625 * r * r + 0.140625 * r * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * s + 0.140625 * pow(s, 3) - 0.140625 * r * pow(s, 3) - 0.281250 * r * t - 0.281250 * s * t + 0.281250 * r * s * t + 0.421875 * r * t * t + 0.421875 * s * t * t - 0.421875 * r * s * t * t + 0.281250 * t - 0.421875 * t * t
        dNr_mtx[2, 1] = -0.140625 + 0.421875 * r + 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 2] = -0.140625 - 0.421875 * r + 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 3] = 0.296875 + 0.296875 * r - 0.296875 * s - 0.296875 * r * s - 0.140625 * r * r + 0.140625 * r * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * s + 0.140625 * pow(s, 3) + 0.140625 * r * pow(s, 3) + 0.281250 * r * t - 0.281250 * s * t - 0.281250 * r * s * t - 0.421875 * r * t * t + 0.421875 * s * t * t + 0.421875 * r * s * t * t + 0.281250 * t - 0.421875 * t * t
        dNr_mtx[2, 4] = -0.140625 + 0.140625 * r + 0.421875 * s - 0.421875 * r * s + 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 5] = -0.140625 - 0.140625 * r + 0.421875 * s + 0.421875 * r * s + 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 6] = -0.140625 + 0.140625 * r - 0.421875 * s + 0.421875 * r * s + 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 7] = -0.140625 - 0.140625 * r - 0.421875 * s - 0.421875 * r * s + 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 8] = 0.296875 - 0.296875 * r + 0.296875 * s - 0.296875 * r * s - 0.140625 * r * r - 0.140625 * r * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * s - 0.140625 * pow(s, 3) + 0.140625 * r * pow(s, 3) - 0.281250 * r * t + 0.281250 * s * t - 0.281250 * r * s * t + 0.421875 * r * t * t - 0.421875 * s * t * t + 0.421875 * r * s * t * t + 0.281250 * t - 0.421875 * t * t
        dNr_mtx[2, 9] = -0.140625 + 0.421875 * r + 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 10] = -0.140625 - 0.421875 * r + 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 11] = 0.296875 + 0.296875 * r + 0.296875 * s + 0.296875 * r * s - 0.140625 * r * r - 0.140625 * r * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * s - 0.140625 * pow(s, 3) - 0.140625 * r * pow(s, 3) + 0.281250 * r * t + 0.281250 * s * t + 0.281250 * r * s * t - 0.421875 * r * t * t - 0.421875 * s * t * t - 0.421875 * r * s * t * t + 0.281250 * t - 0.421875 * t * t
        dNr_mtx[2, 12] = -0.421875 + 0.421875 * r + 0.421875 * s - 0.421875 * r * s - 0.281250 * t + 0.281250 * r * t + 0.281250 * s * t - 0.281250 * r * s * t + 1.265625 * t * t - 1.265625 * r * t * t - 1.265625 * s * t * t + 1.265625 * r * s * t * t
        dNr_mtx[2, 13] = -0.421875 - 0.421875 * r + 0.421875 * s + 0.421875 * r * s - 0.281250 * t - 0.281250 * r * t + 0.281250 * s * t + 0.281250 * r * s * t + 1.265625 * t * t + 1.265625 * r * t * t - 1.265625 * s * t * t - 1.265625 * r * s * t * t
        dNr_mtx[2, 14] = -0.421875 + 0.421875 * r - 0.421875 * s + 0.421875 * r * s - 0.281250 * t + 0.281250 * r * t - 0.281250 * s * t + 0.281250 * r * s * t + 1.265625 * t * t - 1.265625 * r * t * t + 1.265625 * s * t * t - 1.265625 * r * s * t * t
        dNr_mtx[2, 15] = -0.421875 - 0.421875 * r - 0.421875 * s - 0.421875 * r * s - 0.281250 * t - 0.281250 * r * t - 0.281250 * s * t - 0.281250 * r * s * t + 1.265625 * t * t + 1.265625 * r * t * t + 1.265625 * s * t * t + 1.265625 * r * s * t * t
        dNr_mtx[2, 16] = 0.421875 - 0.421875 * r - 0.421875 * s + 0.421875 * r * s - 0.281250 * t + 0.281250 * r * t + 0.281250 * s * t - 0.281250 * r * s * t - 1.265625 * t * t + 1.265625 * r * t * t + 1.265625 * s * t * t - 1.265625 * r * s * t * t
        dNr_mtx[2, 17] = 0.421875 + 0.421875 * r - 0.421875 * s - 0.421875 * r * s - 0.281250 * t - 0.281250 * r * t + 0.281250 * s * t + 0.281250 * r * s * t - 1.265625 * t * t - 1.265625 * r * t * t + 1.265625 * s * t * t + 1.265625 * r * s * t * t
        dNr_mtx[2, 18] = 0.421875 - 0.421875 * r + 0.421875 * s - 0.421875 * r * s - 0.281250 * t + 0.281250 * r * t - 0.281250 * s * t + 0.281250 * r * s * t - 1.265625 * t * t + 1.265625 * r * t * t - 1.265625 * s * t * t + 1.265625 * r * s * t * t
        dNr_mtx[2, 19] = 0.421875 + 0.421875 * r + 0.421875 * s + 0.421875 * r * s - 0.281250 * t - 0.281250 * r * t - 0.281250 * s * t - 0.281250 * r * s * t - 1.265625 * t * t - 1.265625 * r * t * t - 1.265625 * s * t * t - 1.265625 * r * s * t * t
        dNr_mtx[2, 20] = -0.296875 + 0.296875 * r + 0.296875 * s - 0.296875 * r * s + 0.140625 * r * r - 0.140625 * r * r * s + 0.140625 * s * s - 0.140625 * r * s * s - 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * s - 0.140625 * pow(s, 3) + 0.140625 * r * pow(s, 3) - 0.281250 * r * t - 0.281250 * s * t + 0.281250 * r * s * t - 0.421875 * r * t * t - 0.421875 * s * t * t + 0.421875 * r * s * t * t + 0.281250 * t + 0.421875 * t * t
        dNr_mtx[2, 21] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) - 0.140625 * s + 0.421875 * r * s + 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 22] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) - 0.140625 * s - 0.421875 * r * s + 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 23] = -0.296875 - 0.296875 * r + 0.296875 * s + 0.296875 * r * s + 0.140625 * r * r - 0.140625 * r * r * s + 0.140625 * s * s + 0.140625 * r * s * s + 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * s - 0.140625 * pow(s, 3) - 0.140625 * r * pow(s, 3) + 0.281250 * r * t - 0.281250 * s * t - 0.281250 * r * s * t + 0.421875 * r * t * t - 0.421875 * s * t * t - 0.421875 * r * s * t * t + 0.281250 * t + 0.421875 * t * t
        dNr_mtx[2, 24] = 0.140625 - 0.140625 * r - 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s + 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 25] = 0.140625 + 0.140625 * r - 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s + 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 26] = 0.140625 - 0.140625 * r + 0.421875 * s - 0.421875 * r * s - 0.140625 * s * s + 0.140625 * r * s * s - 0.421875 * pow(s, 3) + 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 27] = 0.140625 + 0.140625 * r + 0.421875 * s + 0.421875 * r * s - 0.140625 * s * s - 0.140625 * r * s * s - 0.421875 * pow(s, 3) - 0.421875 * r * pow(s, 3)
        dNr_mtx[2, 28] = -0.296875 + 0.296875 * r - 0.296875 * s + 0.296875 * r * s + 0.140625 * r * r + 0.140625 * r * r * s + 0.140625 * s * s - 0.140625 * r * s * s - 0.140625 * pow(r, 3) - 0.140625 * pow(r, 3) * s + 0.140625 * pow(s, 3) - 0.140625 * r * pow(s, 3) - 0.281250 * r * t + 0.281250 * s * t - 0.281250 * r * s * t - 0.421875 * r * t * t + 0.421875 * s * t * t - 0.421875 * r * s * t * t + 0.281250 * t + 0.421875 * t * t
        dNr_mtx[2, 29] = 0.140625 - 0.421875 * r - 0.140625 * r * r + 0.421875 * pow(r, 3) + 0.140625 * s - 0.421875 * r * s - 0.140625 * r * r * s + 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 30] = 0.140625 + 0.421875 * r - 0.140625 * r * r - 0.421875 * pow(r, 3) + 0.140625 * s + 0.421875 * r * s - 0.140625 * r * r * s - 0.421875 * pow(r, 3) * s
        dNr_mtx[2, 31] = -0.296875 - 0.296875 * r - 0.296875 * s - 0.296875 * r * s + 0.140625 * r * r + 0.140625 * r * r * s + 0.140625 * s * s + 0.140625 * r * s * s + 0.140625 * pow(r, 3) + 0.140625 * pow(r, 3) * s + 0.140625 * pow(s, 3) + 0.140625 * r * pow(s, 3) + 0.281250 * r * t + 0.281250 * s * t + 0.281250 * r * s * t + 0.421875 * r * t * t + 0.421875 * s * t * t + 0.421875 * r * s * t * t + 0.281250 * t + 0.421875 * t * t
        return dNr_mtx

#----------------------- example --------------------

if __name__ == '__main__':
    from ibvpy.api import \
        TStepper as TS, RTDofGraph, RTraceDomainListField, TLoop, \
        TLine, BCDofGroup, IBVPSolve as IS, DOTSEval

    # from lib.tmodel.mats2D.mats_cmdm2D.mats_mdm2d import MACMDM
    from ibvpy.tmodel.mats2D.mats2D_sdamage.mats2D_sdamage import MATS2DScalarDamage
    from ibvpy.tmodel.mats2D.mats2D_sdamage.strain_norm2d import *
    from ibvpy.tmodel.mats3D.mats3D_elastic.mats3D_elastic import MATS3DElastic

    fets_eval = FETS3D8H32U(mats_eval=MATS3DElastic())

    from ibvpy.mesh.fe_grid import FEGrid

    # Discretization
    domain = FEGrid(coord_max=(3., 3., 3.),
                           shape=(3, 3, 3),
                           fets_eval=fets_eval)

    # Put the tseval (time-stepper) into the spatial context of the
    # discretization and specify the response tracers to evaluate there.
    #

    right_dof = 2
    ts = TS(
            sdomain=domain,
             # conversion to list (square brackets) is only necessary for slicing of
             # single dofs, e.g "get_left_dofs()[0,1]" which elsewise retuns an integer only
             bcond_list=[ BCDofGroup(var='u', value=0., dims=[0],
                                  get_dof_method=domain.get_left_dofs),
                        BCDofGroup(var='u', value=0., dims=[1, 2],
                                  get_dof_method=domain.get_bottom_left_dofs),
                        BCDofGroup(var='u', value=0.002, dims=[1],
                                  get_dof_method=domain.get_right_dofs) ],
             rtrace_list=[
#                        RTDofGraph(name = 'Fi,right over u_right (iteration)' ,
#                                  var_y = 'F_int', idx_y = right_dof,
#                                  var_x = 'U_k', idx_x = right_dof,
#                                  record_on = 'update'),
#                        RTraceDomainListField(name = 'Deformation' ,
#                                       var = 'eps', idx = 0,
#                                       record_on = 'update'),
                         RTraceDomainListField(name='Displacement' ,
                                        var='u', idx=0),
#                         RTraceDomainListField(name = 'Stress' ,
#                                        var = 'sig', idx = 0,
#                                        record_on = 'update'),
#                        RTraceDomainListField(name = 'N0' ,
#                                       var = 'N_mtx', idx = 0,
#                                       record_on = 'update')
                        ]
            )

    # Add the time-loop control
    #
    global tloop
    tloop = TLoop(tstepper=ts,
         tline=TLine(min=0.0, step=0.5, max=.5))

    import cProfile
    cProfile.run('tloop.eval()', 'tloop_prof')

#    import pstats
#    p = pstats.Stats('tloop_prof')
#    p.strip_dirs()
#    print 'cumulative'
#    p.sort_stats('cumulative').print_stats(20)
#    print 'time'
#    p.sort_stats('time').print_stats(20)

    # Put the whole stuff into the simulation-framework to map the
    # individual pieces of definition into the user interface.
    #
    from ibvpy.plugins.ibvpy_app import IBVPyApp
    app = IBVPyApp(ibv_resource=tloop)
    app.main()
