# copyright 2015-2022 LOGILAB S.A. (Paris, FRANCE), all rights reserved.
# contact http://www.logilab.fr -- mailto:contact@logilab.fr
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 2.1 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License along
# with this program. If not, see <http://www.gnu.org/licenses/>.
"""cubicweb-prov views/forms/actions/components for web ui"""

from cubicweb import _, tags
from cubicweb.predicates import relation_possible, has_related_entities, is_instance
from cubicweb.view import EntityView
from cubicweb.web.views import uicfg, tableview


class ActivityGeneratedView(EntityView):
    """View for Activity, to be displayed in the context of the entity generated by the activity.
    """
    __regid__ = 'prov.activity-generated'
    __select__ = relation_possible('generated', 'object')
    empty_msg = _('no activity')
    rql = ('Any E,T,DE,DEF,D,A ORDERBY D DESC '
           'WHERE E generated X, E associated_with A?, E type T, E start D,'
           '      E description DE, E description_format DEF, X eid %(x)s')

    def entity_call(self, entity, **kwargs):
        rset = self._cw.execute(self.rql, {'x': entity.eid})
        self.w(tags.h2(self._cw._('generated_object')))
        if rset:
            self._cw.view('prov.activity-generated.table', rset, w=self.w)
        else:
            self.w(tags.div(self._cw._('no activity'), klass="text-muted"))


class ActivityGeneratedTable(tableview.EntityTableView):
    __regid__ = 'prov.activity-generated.table'
    __select__ = is_instance('Activity')
    columns = ['type', 'description', 'start', 'agent']
    column_renderers = {
        'agent': tableview.RelatedEntityColRenderer(
            getrelated=lambda x: x.associated_with and x.associated_with[0] or None),
    }


class ActivityAssociatedWithView(EntityView):
    """View for Activity, to be displayed in the context of the Agent associated with the Activity.
    """
    __regid__ = 'prov.activity-associated-with'
    __select__ = has_related_entities('associated_with', 'object')

    def entity_call(self, entity, **kwargs):
        rset = self._cw.execute(
            'Any E,T,DE,DEF,D,A ORDERBY D DESC WHERE '
            'E associated_with X, E generated A, E type T, E start D,'
            'E description DE, E description_format DEF, X eid %(x)s', {'x': entity.eid})
        self._cw.view('prov.activity-associated-with.table', rset, w=self.w)


class ActivityAssociatedWithTable(tableview.EntityTableView):
    __regid__ = 'prov.activity-associated-with.table'
    __select__ = is_instance('Activity')
    columns = ['type', 'description', 'start', 'generated']
    column_renderers = {
        'generated': tableview.RelatedEntityColRenderer(
            getrelated=lambda x: x.generated[0]),
    }


def registration_callback(vreg):
    vreg.register_all(globals().values(), __name__)

    afs = uicfg.autoform_section
    pvs = uicfg.primaryview_section
    abaa = uicfg.actionbox_appearsin_addmenu

    for rtype in ('used', 'generated', 'associated_with'):
        if rtype in vreg.schema:
            pvs.tag_object_of(('*', rtype, '*'), 'hidden')
            afs.tag_object_of(('*', rtype, '*'), 'main', 'hidden')
            abaa.tag_object_of(('*', rtype, '*'), False)

    if 'derived_from' in vreg.schema:
        afs.tag_subject_of(('*', 'derived_from', '*'), 'main', 'hidden')
        afs.tag_object_of(('*', 'derived_from', '*'), 'main', 'hidden')
