"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let child;
beforeEach(() => {
    stack = new core_1.Stack();
    child = new sfn.StateMachine(stack, 'ChildStateMachine', {
        definition: sfn.Chain.start(new sfn.Pass(stack, 'PassState')),
    });
});
test('Execute State Machine - Default - Fire and Forget', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            input: {
                foo: 'bar'
            },
            name: 'myExecutionName'
        })
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::states:startExecution",
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                foo: "bar"
            },
            Name: 'myExecutionName',
            StateMachineArn: {
                Ref: "ChildStateMachine9133117F"
            }
        },
    });
});
test('Execute State Machine - Sync', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::states:startExecution.sync",
                ],
            ],
        },
        End: true,
        Parameters: {
            StateMachineArn: {
                Ref: "ChildStateMachine9133117F"
            }
        },
    });
});
test('Execute State Machine - Wait For Task Token', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            input: {
                token: sfn.Context.taskToken
            }
        })
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::states:startExecution.waitForTaskToken",
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                "token.$": "$$.Task.Token"
            },
            StateMachineArn: {
                Ref: "ChildStateMachine9133117F"
            }
        },
    });
});
test('Execute State Machine - Wait For Task Token - Missing Task Token', () => {
    expect(() => {
        new sfn.Task(stack, 'ChildTask', {
            task: new tasks.StartExecution(child, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            })
        });
    }).toThrow('Task Token is missing in input (pass Context.taskToken somewhere in input');
});
//# sourceMappingURL=data:application/json;base64,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