"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
test('Publish literal message to SNS topic', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message')
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sns:publish",
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: 'Publish this message'
        },
    });
});
test('Publish JSON to SNS topic with task token', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            message: sfn.TaskInput.fromObject({
                Input: 'Publish this message',
                Token: sfn.Context.taskToken
            })
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sns:publish.waitForTaskToken",
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: {
                'Input': 'Publish this message',
                'Token.$': '$$.Task.Token'
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in message', () => {
    expect(() => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                message: sfn.TaskInput.fromText('Publish this message')
            }) });
        // THEN
    }).toThrow(/Task Token is missing in message/i);
});
test('Publish to topic with ARN from payload', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = sns.Topic.fromTopicArn(stack, 'Topic', sfn.Data.stringAt('$.topicArn'));
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message')
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sns:publish",
                ],
            ],
        },
        End: true,
        Parameters: {
            'TopicArn.$': '$.topicArn',
            'Message': 'Publish this message'
        },
    });
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
                message: sfn.TaskInput.fromText('Publish this message')
            }) });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call SNS./i);
});
//# sourceMappingURL=data:application/json;base64,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