"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const batch = require("@aws-cdk/aws-batch");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../lib");
/*
 * Stack verification steps:
 * * aws stepfunctions start-execution --state-machine-arn <deployed state machine arn> : should return execution arn
 * * aws batch list-jobs --job-queue <deployed job queue name or arn> --job-status RUNNABLE : should return jobs-list with size greater than 0
 * *
 * * aws batch describe-jobs --jobs <job-id returned by list-jobs> --query 'jobs[0].status': wait until the status is 'SUCCEEDED'
 * * aws stepfunctions describe-execution --execution-arn <exection-arn generated before> --query 'status': should return status as SUCCEEDED
 */
class RunBatchStack extends cdk.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        const vpc = new ec2.Vpc(this, 'vpc');
        const batchQueue = new batch.JobQueue(this, 'JobQueue', {
            computeEnvironments: [
                {
                    order: 1,
                    computeEnvironment: new batch.ComputeEnvironment(this, 'ComputeEnv', {
                        computeResources: { vpc }
                    })
                }
            ]
        });
        const batchJobDefinition = new batch.JobDefinition(this, 'JobDefinition', {
            container: {
                image: ecs.ContainerImage.fromAsset(path.resolve(__dirname, 'batchjob-image'))
            }
        });
        const submitJob = new sfn.Task(this, 'Submit Job', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'MyJob',
                jobQueue: batchQueue,
                containerOverrides: {
                    environment: { key: 'value' },
                    memory: 256,
                    vcpus: 1
                },
                payload: {
                    foo: sfn.Data.stringAt('$.bar')
                },
                attempts: 3,
                timeout: cdk.Duration.seconds(60)
            })
        });
        const definition = new sfn.Pass(this, 'Start', {
            result: sfn.Result.fromObject({ bar: 'SomeValue' })
        }).next(submitJob);
        const stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition
        });
        new cdk.CfnOutput(this, 'JobQueueArn', {
            value: batchQueue.jobQueueArn
        });
        new cdk.CfnOutput(this, 'StateMachineArn', {
            value: stateMachine.stateMachineArn
        });
    }
}
const app = new cdk.App();
new RunBatchStack(app, 'aws-stepfunctions-integ');
app.synth();
//# sourceMappingURL=data:application/json;base64,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