"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("lambda", "invoke", this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ["lambda:InvokeFunction"],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier
            }
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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