"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to run AWS Batch
 */
class RunBatchJob {
    constructor(props) {
        var _a;
        this.props = props;
        // validate integrationPattern
        this.integrationPattern =
            props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call RunBatchJob.`);
        }
        // validate arraySize limits
        if (props.arraySize !== undefined &&
            (props.arraySize < 2 || props.arraySize > 10000)) {
            throw new Error(`Invalid value of arraySize. The array size can be between 2 and 10,000.`);
        }
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`Invalid number of dependencies. A job can depend upon a maximum of 20 jobs.`);
        }
        // validate attempts
        if (props.attempts !== undefined &&
            (props.attempts < 1 || props.attempts > 10)) {
            throw new Error(`Invalid value of attempts. You may specify between 1 and 10 attempts.`);
        }
        // validate timeout
        if (props.timeout && props.timeout.toSeconds() < 60) {
            throw new Error(`Invalid value of timrout. The minimum value for the timeout is 60 seconds.`);
        }
        // This is reuqired since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_a = props.containerOverrides) === null || _a === void 0 ? void 0 : _a.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('batch', 'submitJob', this.integrationPattern),
            policyStatements: this.configurePolicyStatements(_task),
            parameters: {
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: this.props.payload,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined
            }
        };
    }
    configurePolicyStatements(task) {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*'
                    }),
                    this.props.jobQueue.jobQueueArn
                ],
                actions: ['batch:SubmitJob']
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule'
                    })
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule']
            })
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`
                }
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: containerOverrides.memory,
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus
        };
    }
}
exports.RunBatchJob = RunBatchJob;
//# sourceMappingURL=data:application/json;base64,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