"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 * A Function can be used directly as a Resource, but this class mirrors
 * integration with other AWS services via a specific class instance.
 */
class PublishToTopic {
    constructor(topic, props) {
        this.topic = topic;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SNS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is missing in message (pass Context.taskToken somewhere in message)');
            }
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("sns", "publish", this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    actions: ['sns:Publish'],
                    resources: [this.topic.topicArn]
                })],
            parameters: {
                TopicArn: this.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? "json" : undefined,
                Subject: this.props.subject,
            }
        };
    }
}
exports.PublishToTopic = PublishToTopic;
//# sourceMappingURL=data:application/json;base64,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