# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

modules = \
['pysafeguard']
install_requires = \
['requests>=2.28.1,<3.0.0']

extras_require = \
{'signalr': ['signalrcore>=0.9.5,<0.10.0']}

setup_kwargs = {
    'name': 'pysafeguard',
    'version': '1.0.2',
    'description': 'One Identity Safeguard Python Package',
    'long_description': '# PySafeguard\nOne Identity Safeguard Python SDK\n\n-----------\n\n<p align="center">\n<i>Check out our <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples\'>sample projects</a> to get started with your own custom integration to Safeguard!</i>\n</p>\n\n-----------\n\n## Support\n\nOne Identity open source projects are supported through [One Identity GitHub issues](https://github.com/OneIdentity/PySafeguard/issues) and the [One Identity Community](https://www.oneidentity.com/community/). This includes all scripts, plugins, SDKs, modules, code snippets or other solutions. For assistance with any One Identity GitHub project, please raise a new Issue on the [One Identity GitHub project](https://github.com/OneIdentity/PySafeguard/issues) page. You may also visit the [One Identity Community](https://www.oneidentity.com/community/) to ask questions.  Requests for assistance made through official One Identity Support will be referred back to GitHub and the One Identity Community forums where those requests can benefit all users.\n\n## Introduction\n\nAll functionality in Safeguard is available via the Safeguard API. There is\nnothing that can be done in the Safeguard UI that cannot also be performed\nusing the Safeguard API programmatically.\n\nPySafeguard is provided to facilitate calling the Safeguard API from Python.\nIt is meant to remove the complexity of dealing with authentication via\nSafeguard\'s embedded secure token service (STS). The basic usage is to call\none of the `connect_*()` methods to establish a connection to Safeguard, then\nyou can call `invoke()` multiple times using the same authenticated connection.\n\nPySafeguard also provides an easy way to call Safeguard A2A from Python. The A2A service requires client certificate authentication for retrieving passwords for application integration. When Safeguard A2A is properly configured, specified passwords can be retrieved with a single method call without requiring access request workflow approvals. Safeguard A2A is protected by API keys and IP restrictions in addition to client certificate authentication.\n\nPySafeguard includes an SDK for listening to Safeguard\'s powerful, real-time event notification system. Safeguard provides role-based event notifications via SignalR to subscribed clients. If a Safeguard user is an Asset Administrator events related to the creation, modification, or deletion of Assets and Asset Accounts will be sent to that user. When used with a certificate user, this provides an opportunity for reacting programmatically to any data modification in Safeguard. Events are also supported for access request workflow and for A2A password changes.\n\n## Installation\n\nThis Python module is published to the [PyPi registry](https://pypi.org/project/pysafeguard) to make it as easy as possible to install.\n\n```Bash\n> pip install pysafeguard\n```\n\n## Dependencies\npysafeguard uses the python requests module, which will need to be installed prior to using pysafeguard\n\n```Bash\n> pip install requests\n```\nIn addition if you will be using the SignalR functionality you will need to install SignalR Core client module.  SignalR Core client is only required if using the SignalR functionality\n\n```Bash\n> pip install signalrcore\n```\n\n### Communicating securely with Safeguard using the SDK\n\nWhen using the SDK to communicate with Safeguard, all communication will\nbe done using HTTPS.  To keep the communication secure, all certificates\nused to secure Safeguard\'s API should be configured on the system where\nthe SDK is used.  How this is accomplished varies on each system,\nhowever, here are some tips that can help get started.\n\nIf the system is already properly configured, the SDK should work\nwithout any errors.  If there are errors, consider using one of the\nfollowing methods to establish trust.\n\n- Environment variable providing path to certificates</li>\n\n  In Bourne Shell:\n  ```Bash\n  $ export WEBSOCKET_CLIENT_CA_BUNDLE=/etc/ssl/certs/ca-certificates.crt\n  $ export REQUESTS_CA_BUNDLE=/etc/ssl/certs/ca-certificates.crt\n  ```\n  \n  In PowerShell:\n  ```Powershell\n  > $env:WEBSOCKET_CLIENT_CA_BUNDLE="c:\\ssl\\certs\\ca-certificates.crt"\n  > $env:REQUESTS_CA_BUNDLE="c:ssl\\certs\\ca-certificates.crt"\n  ```\n  \n- Use the `verify` option when creating the `PySafeguardConnection`</li>\n\n  See examples below for utilizing this method.  While `verify` can be\n  used to disable security checking this is not recommended.\n\n> **Note**  \n> The WEBSOCKET_CLIENT_CA_BUNDLE environment variable is only necessary\n> when working with SignalR.\n\n## Getting Started\n\nA simple code example for calling the Safeguard API with username and password authentication through the local Safeguard STS:\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_password(\'Admin\',\'Admin123\')\nme = connection.invoke(HttpMethods.GET, Services.CORE, \'Me\', query=dict(fields=\'DisplayName\'))\nprint(\'Connected to Safeguard as %s\' % me.json()[\'DisplayName\'])\n```\n\nPassword authentication to an external provider is as follows:\n(Sample can be found <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples/PasswordExternalExample.py\'>here</a>.)\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_password(\'Admin\',\'Admin123\', \'myexternalprovider\')\n```\n\n\nClient certificate authentication is also available. This can be done using PEM and KEY file.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_certificate(\'ssl/pathtocertuser.pem\', \'ssl/pathtocertuser.key\')\n```\n\n> **Note**  \n> Password protected certificates are not currently supported in PySafeguard.\n\nClient certificate authentication to an external provider is also available. This can be done using PEM and KEY file.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_certificate(\'ssl/pathtocertuser.pem\', \'ssl/pathtocertuser.key\', \'myexternalprovider\')\n```\n\n\nA connection can also be made anonymously and without verifying the appliance certificate.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', False)\nsystem_time = connection.invoke(HttpMethods.GET, Services.APPLIANCE, \'SystemTime\')\n```\n\nAuthentication is also possible using an existing Safeguard API token:\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_token(myApiToken)\n```\n> **Note**  \n> Two-factor authentication is not currently supported in PySafeguard.\n\n## Getting Started With A2A\n\nOnce you have configured your A2A registration in Safeguard you can retrieve an A2A password or private key using a certificate and api key.\n\nTo retrieve a password via A2A:\n\n```Python\nfrom pysafeguard import *\n\npassword = PySafeguardConnection.a2a_get_credential(\'safeguard.sample.corp\', \'myapikey\', \'ssl/pathtocertuser.pem\', \'ssl/pathtocertuser.key\', \'ssl/pathtoca.pem\')\n```\n\nTo retrieve a private key in OpenSSH format via A2A:\n\n```Python\nfrom pysafeguard import *\n\nprivatekey = PySafeguardConnection.a2a_get_credential(\'safeguard.sample.corp\', \'myapikey\', \'ssl/pathtocertuser.pem\', \'ssl/pathtocertuser.key\', \'ssl/pathtoca.pem\', A2ATypes.PRIVATEKEY)\n```\n\n## About the Safeguard API\n\nThe Safeguard API is a REST-based Web API. Safeguard API endpoints are called\nusing HTTP operators and JSON (or XML) requests and responses. The Safeguard API\nis documented using Swagger. You may use Swagger UI to call the API directly or\nto read the documentation about URLs, parameters, and payloads.\n\nTo access the Swagger UI use a browser to navigate to:\n`https://<address>/service/<service>/swagger`\n\n- `<address>` = Safeguard network address\n- `<service>` = Safeguard service to use\n\nThe Safeguard API is made up of multiple services: core, appliance, notification,\nand a2a.\n\n|Service|Description|\n|-|-|\n|core|Most product functionality is found here. All cluster-wide operations: access request workflow, asset management, policy management, etc.|\n|appliance|Appliance specific operations, such as setting IP address, maintenance, backups, support bundles, appliance management|\n|notification|Anonymous, unauthenticated operations. This service is available even when the appliance isn\'t fully online|\n|a2a|Application integration specific operations. Fetching passwords, making access requests on behalf of users, etc.|\n\nEach of these services provides a separate Swagger endpoint.\n\nYou may use the `Authorize` button at the top of the screen to get an API token\nto call the Safeguard API directly using Swagger.\n\n### Examples\n\nMost functionality is in the core service as mentioned above.  The notification service\nprovides read-only information for status, etc.\n\n#### Anonymous Call for Safeguard Status\n\nSample can be found <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples/AnonymousExample.py\'>here</a>.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', False)\nresult = connection.invoke(HttpMethods.GET, Services.NOTIFICATION, \'Status\')\nprint(json.dumps(result.json(),indent=2,sort_keys=True))\n```\n\n#### Get remaining access token lifetime\n\nSample can be found <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples/PasswordExample.py\'>here</a>.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_password(\'username\', \'password\')\nminutes_left = connection.get_remaining_token_lifetime()\nprint(minutes_left)\n```\n\n#### Register for SignalR events\n\nTo use the SignalR functionality, you will need to install the python SignalR Core client module\n\n```Bash\n> pip install signalrcore\n```\n\nSample can be found <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples/SignalRExample.py\'>here</a>.\n\n```Python\nfrom pysafeguard import *\n\nconnection = PySafeguardConnection(hostName, caFile)\n\n# SignalR callback function to handle the signalR messages\ndef signalrcallback(results):\n    print("Received SignalR event: {0}".format(results[0][\'Message\']))\n\nprint("Connecting to SignalR via username/password")\nconnection.register_signalr_username(connection, signalrcallback, userName, password)\n\nprint("Connecting to SignalR via certifacte")\nconnection.register_signalr_certificate(connection, signalrcallback, userCertFile, userKeyFile)\n```\n> **Note**  \n> Password protected certificates are not currently supported in PySafeguard.\n\n#### Create a New User and Set the Password\n\nSample can be found <a href=\'https://github.com/OneIdentity/PySafeguard/blob/1.0.2/samples/NewUserExample.py\'>here</a>.\n\n```Python\nfrom pysafeguard import *\nimport json\n\nuser = {\n    \'PrimaryAuthenticationProvider\': { \'Id\': -1 },\n    \'Name\': \'MyNewUser\'\n}\npassword = \'MyNewUser123\'\nconnection = PySafeguardConnection(\'safeguard.sample.corp\', \'ssl/pathtoca.pem\')\nconnection.connect_password(\'username\', \'password\')\nresult = connection.invoke(HttpMethods.POST, Services.CORE, \'Users\', body=user).json()\nuserId = result.get(\'Id\')\nconnection.invoke(HttpMethods.PUT, Services.CORE, f\'Users/{userId}/Password\', body=password)\n```\n',
    'author': 'Tania Engel',
    'author_email': 'Tania.Engel@oneidentity.com',
    'maintainer': 'Rich Gagliano',
    'maintainer_email': 'Rich.Gagliano@oneidentity.com',
    'url': 'https://github.com/OneIdentity/PySafeguard',
    'package_dir': package_dir,
    'py_modules': modules,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
