"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = exports.tryGetModuleVersion = exports.exec = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const file = path.join(directory, name);
    if (fs.existsSync(file)) {
        return file;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersion(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersion = tryGetModuleVersion;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a, _b, _c, _d;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    const pkgDependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    for (const mod of modules) {
        const version = (_d = pkgDependencies[mod]) !== null && _d !== void 0 ? _d : tryGetModuleVersion(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
//# sourceMappingURL=data:application/json;base64,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