"use strict";
/* eslint-disable no-bitwise */
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
//
// Utility to encode RSA public keys (a pair of modulus (n) and exponent (e)) into DER-encoding, per ASN.1 specification.
Object.defineProperty(exports, "__esModule", { value: true });
exports.deconstructPublicKeyInDerFormat = exports.constructPublicKeyInDerFormat = void 0;
const error_1 = require("./error");
/** Enum with possible values for supported ASN.1 classes */
var Asn1Class;
(function (Asn1Class) {
    Asn1Class[Asn1Class["Universal"] = 0] = "Universal";
})(Asn1Class || (Asn1Class = {}));
/** Enum with possible values for supported ASN.1 encodings */
var Asn1Encoding;
(function (Asn1Encoding) {
    Asn1Encoding[Asn1Encoding["Primitive"] = 0] = "Primitive";
    Asn1Encoding[Asn1Encoding["Constructed"] = 1] = "Constructed";
})(Asn1Encoding || (Asn1Encoding = {}));
/** Enum with possible values for supported ASN.1 tags */
var Asn1Tag;
(function (Asn1Tag) {
    Asn1Tag[Asn1Tag["BitString"] = 3] = "BitString";
    Asn1Tag[Asn1Tag["ObjectIdentifier"] = 6] = "ObjectIdentifier";
    Asn1Tag[Asn1Tag["Sequence"] = 16] = "Sequence";
    Asn1Tag[Asn1Tag["Null"] = 5] = "Null";
    Asn1Tag[Asn1Tag["Integer"] = 2] = "Integer";
})(Asn1Tag || (Asn1Tag = {}));
/**
 * Encode an ASN.1 identifier per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.1.2
 *
 * @param identifier - The ASN.1 identifier
 * @returns The buffer
 */
function encodeIdentifier(identifier) {
    const identifierAsNumber = (identifier.class << 7) |
        (identifier.primitiveOrConstructed << 5) |
        identifier.tag;
    return Buffer.from([identifierAsNumber]);
}
/**
 * Encode the length of an ASN.1 type per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.1.3
 *
 * @param length - The length of the ASN.1 type
 * @returns The buffer
 */
function encodeLength(length) {
    if (length < 128) {
        return Buffer.from([length]);
    }
    const integers = [];
    while (length > 0) {
        integers.push(length % 256);
        length = length >> 8;
    }
    integers.reverse();
    return Buffer.from([128 | integers.length, ...integers]);
}
/**
 * Encode a buffer (that represent an integer) as integer per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.3
 *
 * @param buffer - The buffer that represent an integer to encode
 * @returns The buffer
 */
function encodeBufferAsInteger(buffer) {
    return Buffer.concat([
        encodeIdentifier({
            class: Asn1Class.Universal,
            primitiveOrConstructed: Asn1Encoding.Primitive,
            tag: Asn1Tag.Integer,
        }),
        encodeLength(buffer.length),
        buffer,
    ]);
}
/**
 * Encode an object identifier (a string such as "1.2.840.113549.1.1.1") per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.19
 *
 * @param oid - The object identifier to encode
 * @returns The buffer
 */
function encodeObjectIdentifier(oid) {
    const oidComponents = oid.split('.').map((i) => parseInt(i));
    const firstSubidentifier = oidComponents[0] * 40 + oidComponents[1];
    const subsequentSubidentifiers = oidComponents
        .slice(2)
        .reduce((expanded, component) => {
        const bytes = [];
        do {
            bytes.push(component % 128);
            component = component >> 7;
        } while (component);
        return expanded.concat(bytes.map((b, index) => (index ? b + 128 : b)).reverse());
    }, []);
    const oidBuffer = Buffer.from([
        firstSubidentifier,
        ...subsequentSubidentifiers,
    ]);
    return Buffer.concat([
        encodeIdentifier({
            class: Asn1Class.Universal,
            primitiveOrConstructed: Asn1Encoding.Primitive,
            tag: Asn1Tag.ObjectIdentifier,
        }),
        encodeLength(oidBuffer.length),
        oidBuffer,
    ]);
}
/**
 * Encode a buffer as bit string per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.6
 *
 * @param buffer - The buffer to encode
 * @returns The buffer
 */
function encodeBufferAsBitString(buffer) {
    const bitString = Buffer.concat([Buffer.from([0]), buffer]);
    return Buffer.concat([
        encodeIdentifier({
            class: Asn1Class.Universal,
            primitiveOrConstructed: Asn1Encoding.Primitive,
            tag: Asn1Tag.BitString,
        }),
        encodeLength(bitString.length),
        bitString,
    ]);
}
/**
 * Encode a sequence of DER-encoded items per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.9
 *
 * @param sequenceItems - The sequence of DER-encoded items
 * @returns The buffer
 */
function encodeSequence(sequenceItems) {
    const concatenated = Buffer.concat(sequenceItems);
    return Buffer.concat([
        encodeIdentifier({
            class: Asn1Class.Universal,
            primitiveOrConstructed: Asn1Encoding.Constructed,
            tag: Asn1Tag.Sequence,
        }),
        encodeLength(concatenated.length),
        concatenated,
    ]);
}
/**
 * Encode null per ASN.1 spec (DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.8
 *
 * @returns The buffer
 */
function encodeNull() {
    return Buffer.concat([
        encodeIdentifier({
            class: Asn1Class.Universal,
            primitiveOrConstructed: Asn1Encoding.Primitive,
            tag: Asn1Tag.Null,
        }),
        encodeLength(0),
    ]);
}
/**
 * RSA encryption object identifier constant
 *
 * From: https://tools.ietf.org/html/rfc8017
 *
 * pkcs-1    OBJECT IDENTIFIER ::= {
 *     iso(1) member-body(2) us(840) rsadsi(113549) pkcs(1) 1
 * }
 *
 * -- When rsaEncryption is used in an AlgorithmIdentifier,
 * -- the parameters MUST be present and MUST be NULL.
 * --
 * rsaEncryption    OBJECT IDENTIFIER ::= { pkcs-1 1 }
 *
 * See also: http://www.oid-info.com/get/1.2.840.113549.1.1.1
 */
const ALGORITHM_RSA_ENCRYPTION = encodeSequence([
    encodeObjectIdentifier('1.2.840.113549.1.1.1'),
    encodeNull(),
]);
/**
 * Transform an RSA public key, which is a pair of modulus (n) and exponent (e),
 *  into a buffer per ASN.1 spec (DER-encoding)
 *
 * @param n - The modulus of the public key as buffer
 * @param e - The exponent of the public key as buffer
 * @returns The buffer, which is the public key encoded per ASN.1 spec (DER-encoding)
 */
function constructPublicKeyInDerFormat(n, e) {
    return encodeSequence([
        ALGORITHM_RSA_ENCRYPTION,
        encodeBufferAsBitString(encodeSequence([encodeBufferAsInteger(n), encodeBufferAsInteger(e)])),
    ]);
}
exports.constructPublicKeyInDerFormat = constructPublicKeyInDerFormat;
/**
 * Decode an ASN.1 identifier (a number) into its parts: class, primitiveOrConstructed, tag
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.1.2
 *
 * @param identifier - The identifier
 * @returns An object with properties class, primitiveOrConstructed, tag
 */
function decodeIdentifier(identifier) {
    if (identifier >> 3 === 0b11111) {
        throw new error_1.Asn1DecodingError('Decoding of identifier with tag > 30 not implemented');
    }
    return {
        class: identifier >> 6,
        primitiveOrConstructed: (identifier >> 5) & 0b001,
        tag: identifier & 0b11111,
    };
}
/**
 * Decode an ASN.1 block of length value combinations,
 * and return the length and byte range of the first length value combination.
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.1.3 - 8.1.5
 *
 * @param blockOfLengthValues - The ASN.1 length value
 * @returns The length and byte range of the first included length value
 */
function decodeLengthValue(blockOfLengthValues) {
    if (!(blockOfLengthValues[0] & 0b10000000)) {
        return {
            length: blockOfLengthValues[0],
            firstByteOffset: 1,
            lastByteOffset: 1 + blockOfLengthValues[0],
        };
    }
    const nrLengthOctets = blockOfLengthValues[0] & 0b01111111;
    const length = Buffer.from(blockOfLengthValues.slice(1, 1 + 1 + nrLengthOctets)).readUIntBE(0, nrLengthOctets);
    return {
        length,
        firstByteOffset: 1 + nrLengthOctets,
        lastByteOffset: 1 + nrLengthOctets + length,
    };
}
/**
 * Decode an ASN.1 sequence into its constituent parts, each part being an identifier-length-value triplet
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.9
 *
 * @param sequenceValue - The ASN.1 sequence value
 * @returns Array of identifier-length-value triplets
 */
function decodeSequence(sequence) {
    const { tag } = decodeIdentifier(sequence[0]);
    if (tag !== Asn1Tag.Sequence) {
        throw new error_1.Asn1DecodingError(`Expected a sequence to decode, but got tag ${tag}`);
    }
    const { firstByteOffset, lastByteOffset } = decodeLengthValue(sequence.slice(1));
    const sequenceValue = sequence.slice(1 + firstByteOffset, 1 + 1 + lastByteOffset);
    const parts = [];
    let offset = 0;
    while (offset < sequenceValue.length) {
        // Silence false postive: accessing an octet in a Buffer at a particular index
        // is to be done with index operator: [index]
        const identifier = decodeIdentifier(sequenceValue[offset]);
        const next = decodeLengthValue(sequenceValue.slice(offset + 1));
        const value = sequenceValue.slice(offset + 1 + next.firstByteOffset, offset + 1 + next.lastByteOffset);
        parts.push({ identifier, length: next.length, value });
        offset += 1 + next.lastByteOffset;
    }
    return parts;
}
/**
 * Decode an ASN.1 sequence that is wrapped in a bit string
 * (Which is the way RSA public keys are encoded in ASN.1 DER-encoding)
 * See https://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf chapter 8.6 and 8.9
 *
 * @param bitStringValue - The ASN.1 bit string value
 * @returns Array of identifier-length-value triplets
 */
function decodeBitStringWrappedSequenceValue(bitStringValue) {
    const wrappedSequence = bitStringValue.slice(1);
    return decodeSequence(wrappedSequence);
}
/**
 * Decode an ASN.1 DER-encoded public key, into its modulus (n) and exponent (e)
 *
 * @param publicKey - The ASN.1 DER-encoded public key
 * @returns Object with modulus (n) and exponent (e)
 */
function deconstructPublicKeyInDerFormat(publicKey) {
    const [, pubkeyinfo] = decodeSequence(publicKey);
    const [n, e] = decodeBitStringWrappedSequenceValue(pubkeyinfo.value);
    return { n: n.value, e: e.value };
}
exports.deconstructPublicKeyInDerFormat = deconstructPublicKeyInDerFormat;
//# sourceMappingURL=data:application/json;base64,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